/* PARSE-PRED.H

   Type and macro definitions for code to parse box predicates.

   $Header: parse-pred.h,v 1.2 91/10/21 01:29:24 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon.
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#ifndef PARSE_PRED_H
#define PARSE_PRED_H

#include <my-types.h>

/* TYPE DEFINITIONS ======================================================== */

/* VALUE TYPE -------------------------------------------------------------- */

typedef enum {
    IntVal, IdVal, StringVal, BoolVal, BoxTypeVal
} PrimValueType;

typedef struct value_type {
    PrimValueType prim;		/* primitive, base type */
    int list_nesting;		/* number of "-list" suffixes */
} ValueType;

/* RELATION ---------------------------------------------------------------- */

/* RELATION: One of six relations, mapped from: "=", "#", "<", "<=", ">",
   and ">=", respectively.
*/
typedef enum {
    Eq=0, Neq=1, Less=2, LessEq=3, Greater=4, GreaterEq=5
} Relation;

/* VAR EXPR ---------------------------------------------------------------- */

typedef struct var_expr {
    String prefix, suffix;	/* prefix and suffix strings */
    String name;		/* name of this variable */
} VarExpr;

/* SIMPLE-PRED ------------------------------------------------------------- */

typedef enum {
    VarPredKind, StringPredKind, IntPredKind, BoolPredKind
} SimplePredKind;

typedef struct simple_pred {
    PrimValueType type;		/* type of left-hand ID of this predicate */
    String attr_name;		/* left-hand ID of this predicate */
    Relation rel;		/* relation in this comparison */
    SimplePredKind kind;	/* kind of simple predicate */
    union {
	VarExpr *var_expr;	/* when kind==VarPredKind */
	String string;		/* when kind==StringPredKind */
	int int_val;		/* when kind==IntPredKind */
	Boolean bool;		/* when kind==BoolPredKind */
    } u;
} SimplePred;

/* PREDICATE --------------------------------------------------------------- */

typedef enum {
    Unary,			/* contains only a simple-pred */
    Conjunction			/* '&' Boolean relation */
} PredOpKind;

/* PREDICATE: A predicate is combination of a predicate and a simple predicate.
*/

typedef struct pred_struct {
    PredOpKind kind;		/* binary operator */
    struct pred_struct *pred;	/* remainder of predicate */
    SimplePred *simple_pred;	/* simple predicate */
} Predicate;

/* FUNCTION DEFINITIONS ==================================================== */

Predicate *ParsePred( /* String p, int line_no */ );
/* Parse the String 'p' as a box predicate, returning a pointer to the parse
   tree for that predicate if the parse was successful; NULL otherwise. If
   errors are encountered during the parse, they should be reported as
   occurring in the box predicate on or near the line 'line_no' in the input.
*/

void FreePred( /* Predicate *p */ );
/* Deallocate all the memory allocated in the creation of 'p'.
*/

void yyerror2( /* String s */ );
/* Error function called by yacc module parsing box predicates. This function
   simply prints out the error message 's'.
*/

#ifdef DEBUG

void DisplayPred( /* Predicate *p */ );
/* Print an ASCII representation of the predicate 'p' to stderr.
*/

#endif DEBUG

#endif PARSE_PRED_H
