%{

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* $Header: pred.y,v 1.2 91/10/21 01:36:53 heydon Exp $

   Yacc source file for parsing box predicates contained in boxes in
   constraint pictures. This parsing file works in conjuction with the
   lex file "pred.l".

   GRAMMAR:

   predicate   : simple_pred | predicate PRED_OP simple_pred
   simple_pred : var_pred | string_pred | int_pred | bool_pred
   var_pred    : IDENTIFIER RELATION var_expr
   var_expr    : [STRING] variable [STRING]
   variable    : '$' IDENTIFIER
   string_pred : IDENTIFIER RELATION string_expr
   string_expr : IDENTIFIER | STRING
   int_pred    : IDENTIFIER RELATION INTEGER
   bool_pred   : IDENTIFIER | '!' IDENTIFIER

   Terminals:

   PRED_OP     : '&&'
   RELATION    : '=' | '<' | '<=' | '>' | '>='
   SET_OP      : 'in'

   CHANGES:
   o See grammar in chapter-4/bnf.tex, which can be used as a guide
   o Get rid of sets; perhaps, we should allow simple-preds to be disjunctions
     if that is necessary
   o Get rid of integers?
   o Allow variables to be concatenated with strings.

   MORE COMMENTS:
   This grammar is fairly limited, and may be extended. For example, we may
   want to add:
     o time and numeric predicates
     o disjunction and parenthisization at the top-level
     o extend the set_expr to include union, intersection, minus of sets
     o add regular expressions, concatenation, substring to string_expr's
     o numeric expressions that may involve arbitrary numbers of attr names
       and operations (+,-,*,/}
     o "Chaining" comparisons, as in "user <= type < world"
   See the file ~/miro/papers/Constraint-Language/3rd-try for more ideas
   along these lines.
*/

#include <my-types.h>
#include <lex-globals.h>
#include "mem.h"
#include <my-defs.h>

#include "parse-pred.h"
%}

%{
/* Declare a new union type that will be the type of the yacc variable yylval.
   The Lex code returns a token by filling in the appropriate field of this
   union and then returning the appropriate %token defined below, or by
   returning an actual character instead of a %token type.
*/
%}
%union {
    String string_val;		/* IDENTIFIER or STRING terminal */
    int int_val;		/* INTEGER terminal*/
    PredOpKind pred_op;		/* PRED_OP terminal */
    Relation rel;		/* RELATION terminal */
    Predicate *pred;		/* predicate */
    SimplePred *simple_pred;	/* simple_pred, var_pred, string_pred, etc. */
    VarExpr *var_expr;		/* var_expr */
    String var_name;		/* variable */
    String string_expr;		/* string_expr */
}

%{
/* Declare possible return values from the Lex code. Also declared are the
   slots to use in the %union defined above for holding the value associated
   with each %token.
*/
%}
%token <string_val> IDENTIFIER STRING
%token <int_val> INTEGER UNKNOWN_TOKEN
%token <pred_op> PRED_OP
%token <rel> RELATION

%type <pred> predicate
%type <simple_pred> simple_pred var_pred string_pred int_pred bool_pred
%type <var_expr> var_expr
%type <var_name> variable
%type <string_expr> string_expr

%start predicate

%% /* RULES SECTION -------------------------------------------------------- */

predicate   : simple_pred
        	{ $$ = AllocOne(Predicate);
		  $$->kind = Unary;
		  $$->pred = (Predicate *)NULL;
		  $$->simple_pred = $1;
	        }
            | predicate PRED_OP simple_pred
        	{ $$ = AllocOne(Predicate);
		  $$->kind = $2;
		  $$->pred = $1;
		  $$->simple_pred = $3;
		}
            ;

simple_pred : var_pred
            | string_pred
            | int_pred
	    | bool_pred
		{ $$ = $1; }
            ;

var_pred    : IDENTIFIER RELATION var_expr
                { $$ = AllocOne(SimplePred);
		  $$->attr_name = $1;
		  $$->rel = $2;
		  $$->kind = VarPredKind;
		  $$->u.var_expr = $3;
	        }
            ;

var_expr    : variable
                { $$ = AllocOne(VarExpr);
		  $$->prefix = $$->suffix = (String)NULL;
		  $$->name = $1;
	        }
            | STRING variable
                { $$ = AllocOne(VarExpr);
		  $$->prefix = $1;
		  $$->suffix = (String)NULL;
		  $$->name = $2;
	        }
            | variable STRING
                { $$ = AllocOne(VarExpr);
		  $$->prefix = (String)NULL;
		  $$->suffix = $2;
		  $$->name = $1;
	        }
            | STRING variable STRING
                { $$ = AllocOne(VarExpr);
		  $$->prefix = $1;
		  $$->suffix = $3;
		  $$->name = $2;
	        }
            ;

variable    : '$' IDENTIFIER
		{ $$ = $2; }
	    ;

string_pred : IDENTIFIER RELATION string_expr
		{ $$ = AllocOne(SimplePred);
		  $$->attr_name = $1;
		  $$->rel = $2;
		  $$->kind = StringPredKind;
		  $$->u.string = $3;
		}
            ;

string_expr : IDENTIFIER
                { $$ = $1; }
            | STRING
                { CopyString($$,$1); }
	    ;

int_pred    : IDENTIFIER RELATION INTEGER
		{ $$ = AllocOne(SimplePred);
		  $$->attr_name = $1;
		  $$->rel = $2;
		  $$->kind = IntPredKind;
		  $$->u.int_val = $3;
		}
            ;

bool_pred   : IDENTIFIER
		{ $$ = AllocOne(SimplePred);
		  $$->attr_name = $1;
		  $$->rel = Eq;
		  $$->kind = BoolPredKind;
		  $$->u.bool = True;
		}
	    | '!' IDENTIFIER
		{ $$ = AllocOne(SimplePred);
		  $$->attr_name = $2;
		  $$->rel = Eq;
		  $$->kind = BoolPredKind;
		  $$->u.bool = False;
		}
	    ;
%%
