/* RANK.C

   Module for ordering/ranking objects in a constraint picture.

   $Header: rank.c,v 1.6 92/01/05 16:34:20 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "rank.h"
#include "attr.h"
#include "hints.h"
#include "interval.h"
#include "var.h"
#include "objs.h"

/* MACROS ================================================================== */

/* values used by StringToDouble() */
#define HIGHEST_CHAR '~'	/* highest printable ASCII character */
#define LOWEST_CHAR ' '		/* lowest printable ASCII character */
#define PRECISE_CHARS 6		/* # of characters of precision */

/* minimum "distance" for 2 values to be considered equal */
#define EPSILON (1E-5)

/* LOCAL DEBUGGING FUNCTIONS =============================================== */

#ifdef DEBUG

static void DisplayEltOrder(elt,num)
  Elt *elt[];
  int num;
{
    int i;

    fputs("\nElement Orderings:\n",stderr);
    StepIndex(i,0,num) {
	fprintf(stderr," %3d: ",elt[i]->order);
	fprintf(stderr,"%5s ",(elt[i]->thickness==Thick) ? "thick" : "thin");
	if (elt[i]->kind == BoxEltKind) {
	    fputs("   box   ",stderr);
	} else {
	    switch (elt[i]->u.a->kind) {
	      case Syntax: fputs("syn",stderr); break;
	      case Semantics: fputs("sem",stderr); break;
	      case Containment: fputs("con",stderr); break;
	    }
	    fputs(" arrow",stderr);
	}
	fprintf(stderr," (sysname = %2d)\n",elt[i]->sysname);
    }
}

#endif DEBUG

/* LOCAL FUNCTIONS ========================================================= */

static double StringToDouble(s)
  String s;
/* Return a floating point number in the range [0,1] corresponding to the
   string 's' such that StringToDouble(s1) <= StringToDouble(s2) iff s1 is
   lexicographically less than s2. Note that the function is not quite exact:
   s1 may be lexicographically less than s2, but this function may compute the
   same value for both.

   IMPLEMENTATION: the function considers 's' to be a real fraction in base
   'base' (where 'base' is the number of printable characters in the ASCII
   character set (see definition below)) as if it were immediately following a
   decimal point. For example, if the base was 4 and the only 4 printable
   characters in the character set were a-d, then the string 'abcd' would give
   the value: 1/4 + 2/16 + 3/64 + 4/256. From this example, it is clear that
   the values get small very quickly, so we are limited in our precision. The
   macro PRECISE_CHARS defines how many characters we are able to process
   given the limits of the machine's representation of a double.
*/
{
    static double base = (double)(HIGHEST_CHAR - LOWEST_CHAR + 1);
    double result = 0.0;
    double mask = 1.0;
    int i;

    for (i=PRECISE_CHARS; i>0 && *s; i--,s++) {
	mask /= base;
	result += mask * (*s - LOWEST_CHAR);
    }
    return(result);
}

/* ----------------------------------------------------------------------------
 *
 * The following Ratio() functions all work on the same basic principle. The
 * routines are passed an interval and an attribute. The attribute has a min
 * and max possible value associated with it, as well as guess at the number
 * of values likely to be between min and max. The ratio functions approximate
 * the ratio:
 *
 *     dist(i->low,i->high)/dist(attr->min,attr->max).
 *
 * Several comments are in order. First, it may be that neither i->low nor
 * i->high is defined. In that case, they take on the value attr->min and
 * attr->max, respectively. Second, it may be that dist(i->low,i->high) == 0,
 * i.e., i->low == i->high. In this case, we don't want to return a ratio of
 * 0. Instead, we want to return the probability that a single point in the
 * interval chosen uniformly has the value i->low (== i->high). That
 * probability is simply the inverse of the number of values between attr->min
 * and attr->max.
 *
 * ----------------------------------------------------------------------------
*/

static float IntRatio(r,attr)
  IntrvlRange *r;
  Attr *attr;
/* Returns the ranking of the integer interval 'r' over the attribute 'attr'.
*/
{
    AttrVal low,high;
    float diff;

    low.i_val = (r->low != NULL) ? r->low->u.int_val : attr->min.i_val;
    high.i_val = (r->high != NULL) ? r->high->u.int_val : attr->max.i_val;
    if ((diff=AttrValDistance(low,high,IntKind)) < EPSILON) {
	return(1.0/((float)attr->range_size));
    } else {
	return(((float)diff)/attr->dist);
    }
}

static float StringRatio(r,attr)
  IntrvlRange *r;
  Attr *attr;
/* Returns the ranking of the string interval 'r' over the attribute 'attr'.
*/
{
    AttrVal low,high;
    float diff;

    low.s_val = (r->low != NULL) ? r->low->u.string : attr->min.s_val;
    high.s_val = (r->high != NULL) ? r->high->u.string : attr->max.s_val;
    if ((diff=AttrValDistance(low,high,StringKind)) < EPSILON) {
	return(1.0/attr->range_size);
    } else {
	return(diff/attr->dist);
    }
}

static float BoolRatio(r,attr)
  IntrvlRange *r;
  Attr *attr;
/* Assuming the interval range 'r' contains values for a BoolKind interval in
   canonical form (so both its 'low' and 'high' values are the same), return
   the corresponding floating point value associated with 'attr' ('max' for
   True and 'min' for False).
*/
{
    return(r->low->u.bool ? attr->max.f_val : attr->min.f_val);
}

static float BoxTypeRatio(r)
  IntrvlRange *r;
{
    float result;
    BoxType *curr,*top;

    if (r->low != NULL) {
	result = 0.0;
	top = (r->high != NULL) ? r->high->u.box_type : (BoxType *)NULL;
	for (curr=r->low->u.box_type; curr != top; curr=curr->u.parent) {
	    result += curr->rank;
	}
	if (curr != NULL) result += curr->rank;
    } else {
	Assert(r->high != NULL);
	top = r->high->u.box_type;
	result = top->sum_rank;
	if (r->high->kind == StrictComp) result -= top->rank;
    }
    return(result);
}

static double Prod(start,end)
  int start,end;
/* Returns the product: (start) (start+1) ... (end), assuming start <= end;
   returns 0 otherwise (i.e., if start > end).
*/
{
    double result = 1.0;
    while (start <= end) { result *= (double)start++; }
    return(result);
}

static double CountProb(cnt)
  VarCnt *cnt;
/* Computes and returns:
     (cnt->low_cnt)! (cnt->high_cnt)! / (cnt->low_cnt + cnt->high_cnt + delta)!
   where delta is 1 if cnt->eq_cnt > 0; 0 otherwise. In the event that the
   denominator of this expression is 0, return 1.0.
*/
{
    double num = Prod(1,cnt->high_cnt);
    int delta = (cnt->eq_cnt > 0) ? 1 : 0;
    double denom = Prod(1+cnt->low_cnt,cnt->low_cnt+cnt->high_cnt+delta);
    return(num/denom);
}

static void MergeCnts(c1,c2)
  INOUT VarCnt *c1;
  VarCnt *c2;
/* Merge the counts of 'c1' and 'c2' into 'c1'.
*/
{
    c1->low_cnt += c2->low_cnt;
    c1->high_cnt += c2->high_cnt;
    c1->eq_cnt += c2->eq_cnt;
}

static double VarRatio(i,var,table)
  Intrvl *i;
  Var *var;
  IdHashTable table;
/* Given the interval 'i' and the "current" counts on 'var' -- namely,
   'var->cnt' -- this routine returns a probabilistic estimate that the
   counts embodied by 'i->range' will "fit into" the cnts of 'var'. 'var' is
   assumed to be a pointer into the ID Hash Table.

   See personal notes II:24 for details.

   Note: We subtract EPSILON in the first quantity so that box patterns that
   are otherwise the same except for one including a variable comparison will
   end up favoring the one with the variable comparison.
*/
#undef EPSILON
#define EPSILON 1.0E-5
{
    IntrvlRange *r = i->range;
    VarCnt cnt;			/* temporary counts for 'r' */
    double rng_factor;
    double result;

    /* find probability embodied by 'var->cnt' */
    result = (1.0-EPSILON)/CountProb(var->cnt);

    /* factor probability of 'var->cnt' into 'cnt' for this interval */
    rng_factor = CountVarRange(r,&cnt,table,MakeBoolean(var->cnt->eq_cnt>0));
    MergeCnts(&cnt,var->cnt);
    result *= CountProb(&cnt);

    /* factor in equalities */
    result *= rng_factor;
    return(result);
}

static double RankIntrvl(i,pict)
  Intrvl *i;
  Pict *pict;
/* Returns the rank of the interval 'i', according to hint information
   embodied by 'pict'.
*/
{
    float result;
    TableEntry *tbl;		/* table entry of variable or attribute */

    /* get table value for variable or attribute */
    if (i->kind == VarKind) {
	tbl = FindTableId(pict->table,VarNameId,i->name);
    } else {
	tbl = FindTableId(pict->table,AttrNameId,i->name);
    }

    /* rank interval based on the types of its values */
    switch (i->kind) {
      case IntKind:     result = IntRatio(i->range,tbl->u.attr); break;
      case StringKind:  result = StringRatio(i->range,tbl->u.attr); break;
      case BoolKind:    result = BoolRatio(i->range,tbl->u.attr); break;
      case BoxTypeKind: result = BoxTypeRatio(i->range); break;
      case VarKind:     result = VarRatio(i,tbl->u.var,pict->table); break;
    }
    return(result);
}

static void RankBoxElt(b_elt,pict)
  BoxElt *b_elt;
  Pict *pict;
{
    int adj_cnt = 0;		/* # of marked adjacent elt's */
    EltList *adj;
    IntrvlList *curr;

    /* Count the number of marked elements adjacent on 'b_elt' */
    StepLinkedList(adj,b_elt->adj_elts) {
	if (adj->elt->marked) { adj_cnt++; }
    }

    /* process sysname equality interval if sysname fixed */
    if (adj_cnt > 0) {
	TableEntry *tbl;
	Assert((tbl=FindTableId(pict->table,AttrNameId,"sysname")) != NULL);
	b_elt->rank = 1.0/((float)tbl->u.attr->range_size);
    } else {
	/* initialize rank to highest value */
	b_elt->rank = 1.0;
    }

    /* process each interval */
    StepLinkedList(curr,b_elt->u.b->u1.intvls) {
	b_elt->rank *= RankIntrvl(curr->i,pict);
    }
}

static Boolean AnchoredBox(b_elt)
  BoxElt *b_elt;
/* Returns True iff 'b_elt' is marked, or if some arrow incident on 'b_elt' is
   marked.
*/
{
    EltList *curr;

    if (b_elt->marked) { return(True); }
    StepLinkedList(curr,b_elt->adj_elts) {
	if (curr->elt->marked) { return(True); }
    }
    return(False);
}

static void RankArrowElt(a_elt,pict)
  ArrowElt *a_elt;
  Pict *pict;
{
    int kind = (int)a_elt->u.a->kind;
    float *hint_array;
    PermList *curr;
    TableEntry *tbl;
    int side;
    int adj_cnt = 0;

    /* Count number of adjacent anchored boxes; a box is anchored if the box
       is marked or if there are any marked arrows adjacent on the box. The
       adjacent count is encoded as follows: 1 = from, 2 = to, 3 = both */
    if (AnchoredBox(a_elt->u.a->from)) { adj_cnt += 1; }
    if (AnchoredBox(a_elt->u.a->to))   { adj_cnt += 2; }

    /* assign temporary hint array */
    hint_array = (pict->ranks)[kind][adj_cnt][(int)(a_elt->u.a->parity)];

    /* rank arrow depending on its kind */
    if (kind == (int)Containment) {
	/* Containment or StarContainment arrow */
	switch (a_elt->u.a->to->u.b->side) {
	  case LeftSide: side = 0; break;
	  case NeitherSide: case RightSide: side = 1; break;
	}
	a_elt->rank = hint_array[side];
    } else {
	/* Syntax or Semantics arrow */
	if (a_elt->u.a->u.perm_list == NULL) {
	    /* empty list; use value corresponding to the sum */
	    tbl = FindTableId(pict->table,PermNameId,"AllPerms");
	    Assert(tbl != NULL);
	    a_elt->rank = hint_array[tbl->u.index];
	} else {
	    /* non-empty list; sum up the rank for each permission */
	    a_elt->rank = 0.0;
	    StepLinkedList(curr,a_elt->u.a->u.perm_list) {
		tbl = FindTableId(pict->table,PermNameId,curr->perm);
		Assert(tbl != NULL);
		a_elt->rank += hint_array[tbl->u.index];
	    }
	}
    }
}

static void RankElt(e,pict)
  Elt *e;
  Pict *pict;
/* Rank the element 'e' contained in 'pict'. This has the side-effect of
   setting the 'rank' field of 'e'.
*/
{
    /* rank 'e' depending on whether it is a box or an arrow */
    switch (e->kind) {
      case BoxEltKind: RankBoxElt((BoxElt *)e,pict); break;
      case ArrowEltKind: RankArrowElt((ArrowElt *)e,pict); break;
    }
#ifdef DEBUG
    fprintf(stderr,"Ranked element %d at %f\n",e->sysname,e->rank);
#endif DEBUG
}

static void InsertCellCopy(elt,head)
  Elt *elt;
  EltList *head[/*THICKNESSES*/];
{
    EltList *e_cell;

    e_cell = AllocOne(EltList);
    e_cell->elt = elt;
    SpliceIntoList(head[(int)(elt->thickness)],e_cell);
}

static Elt *MinRankUnmarked(head)
  EltList *head;
/* Returns a pointer to the unmarked Elt in the EltList rooted at 'head' with
   the least 'rank'; NULL if there are no unmarked Elt's in the list.

   Pre-condition: "all Elt's in the list have rank >= 0.0"
*/
{
    Elt *result = (Elt *)NULL;
    float min = 1.0e10;		/* ranks are all *much* smaller than this */

    StepInitializedLinkedList(head) {
	if (!head->elt->marked && head->elt->rank < min) {
	    result = head->elt;
	    min = result->rank;
	}
    }
    return(result);
}

static void UpdateVariableCounts(elt,t,pict)
  Elt *elt;
  Thickness t;
  Pict *pict;
/* Step through the intervals associated with 'b'. For each Variable interval,
   merge the counts of that interval with the current count registered with
   the variable in the ID Hash Table 'pict->table'.

   Also, for each variable in 'b', re-rank any unmarked 't'-boxes containing
   an interval on that variable.
*/
{
    Box *b = elt->u.b;
    IntrvlList *curr;

    Assert(b->kind == IntvlKind);
    StepLinkedList(curr,b->u1.intvls) {
	if (curr->i->kind == VarKind) {
	    VarCnt cnt;		/* temporary counts */
	    Var *v;		/* the variable in the ID Table */
	    BoxList *bl;	/* boxes for the variable 'v' */
	    /*
	     * set counts for this box's variable interval */
	    (void)CountVarRange(curr->i->range,&cnt,(IdHashTable)NULL,False);
	    /*
	     * merge those counts with the ones already in the table */
	    v = FindTableId(pict->table,VarNameId,curr->i->name)->u.var;
	    MergeCnts(v->cnt,&cnt);
	    if (v->first == NOT_BOUND) {
		VarList *vl = AllocOne(VarList);
		vl->var = v;
		SpliceIntoList(b->first_vars,vl);
		v->first = elt->order;
	    }
	    /*
	     * re-rank any unmarked 't'-boxes on this variable */
	    StepLinkedList(bl,v->boxes) {
		if (bl->elt->thickness == t && !bl->elt->marked) {
		    RankElt(bl->elt,pict);
		}
	    }
	}
    }
}

/* GLOBAL FUNCTIONS ======================================================== */

float AttrValDistance(min,max,kind)
  AttrVal min;
  AttrVal max;
  IntrvlKind kind;
/* Returns the "distance" dist(min,max), where 'min' and 'max' are interval
   values of IntrvlKind 'kind'. The only recognized IntrvlKind's are
   IntKind and StringKind.
*/
{
    float result;
    switch (kind) {
      case IntKind:
	result = (float)max.i_val-(float)min.i_val;
	break;
      case StringKind:
	result = (float)(StringToDouble(max.s_val)-StringToDouble(min.s_val));
	break;
      default:
	Assert(False);		/* error if called with any other kind */
    }
    return(result);
}

void OrderObjs(pict)
  Pict *pict;
{
    int t;			/* thickness index */
    int unmarked[THICKNESSES];	/* # of unmarked Elt's of each thickness */
    int order_num = 0;		/* order number counter */
    int elt_cnt;		/* number of boxes and arrows total */
    Elt *curr;			/* "current" element */
    EltList *el;
    EltList *elts[THICKNESSES];	/* lists of thick/thin Elt's */
    EltList *adj;		/* element adjacent on 'curr' */
    Elt *e;			/* temporary variable, e = adj->elt */

    /* build thick/thin lists */
    elts[(int)Thick] = elts[(int)Thin] = (EltList *)NULL;
    StepLinkedList(el,pict->boxes)  { InsertCellCopy(el->elt,elts); }
    StepLinkedList(el,pict->arrows) { InsertCellCopy(el->elt,elts); }

    /* unmark all elements */
    StepIndex(t,0,THICKNESSES) {
	unmarked[t] = 0;
	StepLinkedList(el,elts[t]) {
	    el->elt->marked = False;
	    unmarked[t]++;
	}
    }

    /* initialize space for pict->elts */
    elt_cnt = unmarked[(int)Thick] + unmarked[(int)Thin];
    pict->elt = AllocPtrArray(Elt,elt_cnt);

    /* main loop; consider thick Elt's first, then thin Elt's */
    StepIndex(t,0,THICKNESSES) {
	/*
	 * rank all 't' elements (they are all unmarked) */
#ifdef DEBUG
	fprintf(stderr,"\nRanking all %s elements...\n",t?"thin":"thick");
#endif DEBUG
	StepLinkedList(el,elts[t]) {
	    RankElt(el->elt,pict);
	}
	/* loop so long as there is an unmarked 't' Elt */
	while (unmarked[t] > 0) {
	    /*
	     * mark new "current" element */
	    curr = MinRankUnmarked(elts[t]);
#ifdef DEBUG
	    fprintf(stderr,"\n** Marked element %d **\n",curr->sysname);
#endif DEBUG
	    curr->marked = True;
	    unmarked[t]--;	         /* dec. unmarked count for 't' */
	    pict->elt[order_num] = curr; /* fill in array of ordered Elt's */
	    curr->order = order_num++;	 /* assign order number to 'curr' */
#ifdef DEBUG
	    fprintf(stderr,"\nRanking all effected elements...\n");
#endif DEBUG
	    /*
	     * update variable counts (if any) of BoxElt 'curr' */
	    if (curr->kind == BoxEltKind) {
		UpdateVariableCounts(curr,(Thickness)t,pict);
	    }
	    /*
	     * recompute rank of unmarked 't' elts adjacent to 'curr' */
	    StepLinkedList(adj,curr->adj_elts) {
		e = adj->elt;
		if (!e->marked &&  e->thickness == (Thickness)t) {
		    RankElt(e,pict);
		}
		/* also recompute ranks of arrows incident on incident box */
		if (!e->marked && curr->kind == ArrowEltKind) {
		    EltList *a_arrow;
		    Elt *a;
		    StepLinkedList(a_arrow,e->adj_elts) {
			a = a_arrow->elt;
			if (!a->marked && a->thickness == (Thickness)t) {
			    RankElt(a,pict);
			}
		    }
		}
	    }
	} /* end while() */
    } /* end StepIndex(t) */
#ifdef DEBUG
    DisplayEltOrder(pict->elt,elt_cnt);
#endif DEBUG
}
