;;;             -*- Mode: Lisp; Package: MIRO; -*-
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

;;; 
;;; MIRO EDITOR - COMMAND BUTTON FUNCTIONS
;;;
;;; This file contains the code for functions which are called
;;; as the notify procedures of the command buttons. 
;;; 
;;; Current functions are: 
;;; - copy-selected-objects
;;; - delete-selected-objects
;;; - display-object
;;; - unselect-all
;;; - clear-workbench
;;; - check-ambiguity
;;; - read-miro-file
;;; - save-workbench
;;; - print-workbench
;;; - exit-editor

;;; 
;;; Functions to be implemented:
;;; - undo
;;; 

#|
============================================================
Change log:
    11/04/91 ky ; new interfaces: prober, verifier, constraint checker
    06/19/91 ky ; Print the editor entry first.  Identify output files
		; as being generated by the editor.
    04/16/91 ky ; Use run-miro-tool to run programs and get output.
    04/10/91 ky ; Don't try to clean up on exit; just unmap the
		; windows.
    03/12/91 ky ; Added functions verify-workbench, do-verify.
    03/06/91 ky ; Changed arguments for iff2lisp.
    02/20/91 ky ; Print arrow type lists properly.  Use iff2lisp for
		; reading iff files.
    02/08/91 ky ; Copy the hash table from the newly-parsed aggregate
		; in function do-read.
    12/11/90 ky ; Removed gc notification for now.
    12/7/90  ky ; Changed "enter" to "enter/edit" for read/save dialog
		; box.
    12/4/90  ky ; Raise/Lower the dialog window.
    12/3/90  ky ; Destroy the dialog window when exiting.
    11/20/90 ky ; Don't change the color of work-window's background
		; if the user is copying an object.  Set :help-msg
		; when using *y-n-buttons*.
    11/12/90 ky ; Look for the new output format for the ambiguity
		; checker if *new-ambig* (defined in miro-defs.lisp)
		; is non-nil.
    11/9/90  ky ; Trap error output from ambig.
    11/6/90  ky ; Reset gc for non-CMU-common-lisp.
    11/5/90  ky ; Generate sensible output for the ambiguity checker
		; when it is run on a file other than the workbench.
    11/1/90  ky ; Use print-dbox.
    11/1/90  ky ; Use filename-dbox.
    10/30/90 ky ; Use "ambig-options" menu.
    10/30/90 ky ; Fixed typo: supercede -> supersede.
    10/25/90 ky ; Changed to use the browser gadget for ambig results.
		; Use the reset button's :selection-function to reset
		; scaling.
    10/18/90 ky ; Fixed process-output handling for Allegro.
    9/24/90 ky  ; Moved some information from ambig-menu to
		; ambig-status in an attempt to avoid display errors.
    9/21/90 ky  ; Put ambig results into ambig-menu.
		;
		; Warn the user about possibly invalid ambig results
		; when the picture is changed.
		;
    9/20/90 ky  ; Save the results of ambig in a string.
    9/18/90 ky  ; Added function destroy-undo-objects, to be called when
		; the objects in undo-objects need to be deleted.
		;
		; Delete undo-objects and fix visibility of undelete
		; button when a file is read in.
		;
    9/17/90 ky  ; Added new functions to facilitate turning buttons on
		; and off: can-ambig, can-copy, can-delete,
		; can-display, can-hide, can-undelete, can-unhide,
		; can-unselect.
		;
		; Fix visibility of copy, delete, display, and
		; unselect buttons when objects are unselected.
		;
		; Set pictype-menu's :value-obj if possible.
    9/14/90 ky  ; Don't update the display until everything to be
		; deleted has been deleted.
		; Put deleted objects in undo-agg instead of
		; destroying them immediately.
		; Changed "undo" to "undelete".
		; Implemented undelete.
    9/13/90 ky  ; Deleted function reset-scale.
		; Clear-workbench resets the scale and offsets.
		; Added declarations to get rid of compile-time
		; warnings.
    9/12/90 ky  ; Maintain a bounding box to use with scaling and with
		; the scrollpad.
		;
		; Use versions to get a unique name for temporary
		; files.  Delete temporary files when we are done with
		; them.
		;
		; Fixed bug where pictype wasn't getting set properly
		; when a picture was read in.
		;
		; Added function "check-legality" to check the
		; legality of a picture.  Not implemented yet.
		;
    8/23/90 ky  ; A few changes to work with the "test" Garnet.
		; Tell find-bounding-box to ignore the scale for
		; reading/saving.
		;
    8/10/90 ky  ; Use the bounding box of the objects being
		; saved/restored to determine the origin.
		;
		; Use :box (for boxes) or :points (for arrows) to
		; get/set size and position information, since things
		; like :height and :width may contain bogus values if
		; the object in question is not visible for any reason.
		;
		; Allow the user to specify arguments for iff2ps.
		;
    7/31/90 ky  ; Changed help messages to conform to the new size of
		; the help window.
		;
		; Use pic-sp instead of vert-sb, hor-sb.
		;
		; Use a hardwired value (675) for the origin instead
		; of the height of the work window to avoid the
		; immediate problem of pictures falling off the top of
		; the window due to its slightly reduced height.
		;
    7/23/90 ky  ; Verified that we still need to update the display
		; before destroying an arrow.
		;
		; Added functions hide-selected-objects and
		; unhide-selected-objects to hide/unhide groups of
		; boxes and/or arrows.  These functions have not been
		; implemented yet.
		;
		; Added function reset-scale to reset the scale,
		; horizontal, and vertical offsets.
		;
    7/5/90  ky  ; print-workbench allows the user to specify the
		; output file/printer.
    7/3/90  ky	; Use ambig: as the path for iff2ps.
		; Exit lisp after exiting the editor if the editor was
		; started from a core image.
    6/25/90 ky  ; Use new functions "block-interference" and
		; "allow-interference" to change button colors, turn
		; off interactors, etc.
                ;
                ; Tell the user what is happening during "save"
		; "ambig" and "print" commands.
                ;
                ; Change *package* to "MIRO" inside "with-open-file"s.
		; This solves the problem of "read"s being done
		; incorrectly if the user wasn't in the miro package
		; when they started the editor.
                ;
                ; When saving a picture, don't print the type if it is
		; an empty string.
                ;
                ; Check for the existence of the file given to the
		; "read" command and print an error message if the
		; file doesn't exist.
                ;
                ; Don't allow other commands to interfere with
		; "print", "read", and "ambig".
                ;
                ; Redirect error output from "run-program" in CMU
		; Common Lisp to the lisp window.
                ;
    6/8/90  ky  ; Changes for compatibility with Allegro Common Lisp:                ;
		;   - Don't try to warn the user about garbage
		;     collects in the help window, since there doesn't
		;     seem to be any way to find out when the gc
		;     starts, only when it ends.
                ;   - Use excl:run-shell-command instead of
		;     extensions:run-program under ACL.
    6/1/90  ky  ; Changed :interfill to :buttonfill.
    5/8/90  ky  copy-selected-objects, display-object, and undo use
                push-error-msg for their error messages.
                clear-workbench and exit-editor use *y-n-buttons*
                instead of y-or-n-p.
    4/27/90 ky  exit-editor restores the gc notification functions to
                their previous state.
    4/25/90 ky	copy-selected-objects now includes "type ^G to abort" in its
		help message, calls set-abort-inter to make copy-inter the
		interactor to abort, turns off low and medium priority
		interactors, sets *dont-interfere*, and changes the color of
		the text buttons in commands-menu (:interfill) to dark-gray.
    4/10/90 amz added prompt to help window in copy
    4/10/90 amz changed to search-list for ambig checker, 
                changed to new ambig checker syntax.
    2/21/90 amz added dialog boxes for display-object
    2/9/90 amz added new copy to deal with multiple objects
    1/16/90 amz changed to use feedback-agg and obj-agg
    1/9/90 amz modified delete-selected-objects to handle arrows
    9/11/89 amz added read-workbench
    8/24/89 amz added print-insideness and direct containment algorithms
    8/22/89 amz added copy-selected-objects
    8/11/89 amz put command functions in separate file
    8/9/89 amz  Changed menu to include constraint options.
    5/4/89 prm   Added text interactor for the boxes labels
    3/30/89 afm  added function save-workbench
                 first pass at producing output in interm file format
    Nov. 1988 Brad Myers, Philippe Marchal -- Created.

============================================================
|#

(in-package "MIRO" :use `("LISP" "KR"))

(proclaim '(function set-abort-inter)) ; defined in miro-inters.lisp
(proclaim '(function find-bounding-box)) ; defined in miro-inters.lisp
(proclaim '(function single-object-selected)) ; defined in miro-inters.lisp
(proclaim '(function get-arg-list)) ; defined in miro-toolkit.lisp
(proclaim '(function get-line-list)) ; defined in miro-toolkit.lisp
(proclaim '(function destroy-undo-objects)) ; defined in miro-cmnds.lisp

;; a list of deleted objects
(defvar undo-objects nil)
(setq undo-schema-slots nil)
(setq undo-functions nil)

(defparameter *debug-contains* NIL)

;;;============================================================
;;;                        Option settings
;;;============================================================
(defun edit-options (&rest args)
  (declare (ignore args))
  (call-schema options-menu :display-me))

(defun changing-bground ()
  (g-value *options* :change-background-for-dbox-emphasis))
(defun using-large-fonts () (g-value *options* :large))
(defun using-inverse () (> (g-value opal:black :red) 0.5))
(defun using-auto-gc () (eq excl::*global-gc-behavior* :auto))

(defun option-bground (change-bground)
  (s-value *options* :change-background-for-dbox-emphasis
	   (not (not change-bground))))

(defun option-large (large)
  (s-value *options* :large (not (not large))))

;; this doesn't work -- currently need to compile inverse video into
;; opal.
(defun option-inverse (inverse)
  (when (or (and (not (using-inverse)) inverse)
	    (and (using-inverse) (not inverse)))
	;; swap black & white
	(let ((black (g-value opal:black :red))
	      (white (g-value opal:white :red)))
	  (dolist (s '(:red :green :blue))
		  (s-value opal:black s white)
		  (s-value opal:white s black))
	  (recompute-formula opal:black :xcolor)
	  (recompute-formula opal:black :colormap-index)
	  (recompute-formula opal:white :xcolor)
	  (recompute-formula opal:white :colormap-index)
	  )

	;; fix fill-bitmaps
	(let ((white (g-value *options* :white-halftone-image))
	      (light-gray (g-value *options* :light-gray-halftone-image))
	      (dark-gray (g-value *options* :dark-gray-halftone-image))
	      (black (g-value *options* :black-halftone-image))
	      )
	  (s-value *options* :white-halftone-image black)
	  (s-value opal::white-fill-bitmap :image black)
	  (s-value *options* :light-gray-halftone-image dark-gray)
	  (s-value opal::light-gray-fill-bitmap :image dark-gray)
	  (s-value *options* :dark-gray-halftone-image light-gray)
	  (s-value opal::dark-gray-fill-bitmap :image light-gray)
	  (s-value *options* :black-halftone-image white)
	  (s-value (g-value *options* :black-fill-bitmap) :image
		   white)
	  (s-value opal:black-fill :fill-style :opaque-stippled)
	  (s-value opal:black-fill :stipple
		   (g-value *options* :black-fill-bitmap))
	  )

	;; fix our colors
	(let ((black (g-value *options* :black))
	      (really-dark-gray (g-value *options* :really-dark-gray))
	      (dark-gray (g-value *options* :dark-gray))
	      (light-gray (g-value *options* :light-gray))
	      (really-light-gray (g-value *options* :really-light-gray))
	      (white (g-value *options* :white))
	      )
	  (s-value *options* :black white)
	  (s-value *options* :really-dark-gray really-light-gray)
	  (s-value *options* :dark-gray light-gray)
	  (s-value *options* :light-gray dark-gray)
	  (s-value *options* :really-light-gray really-dark-gray)
	  (s-value *options* :white black)
	  )
	))

(defun option-gc (auto-gc)
  (setq excl::*global-gc-behavior*
	(if auto-gc :auto nil)))

;;;============================================================
;;;                        COPY
;;;============================================================

;;;------------------------------------------------------------
;;; Copy-Selected-Objects activates the copy interactor. The user then
;;; presses any mouse button in the workbench to start the interactor,
;;; which indicates the location for the new copies of the objects. 
;;;------------------------------------------------------------

(defun can-copy () (g-value obj-agg :selected))
(defun copy-selected-objects (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  ;; make copy inter active, and print prompt
  ;; turn "overlapped" interactors off
  (block-interference
   :leave-work-window-alone T
   :help-msg
   "Use mouse to specify
location for copy.
Type ^G to abort.")
  (inter:change-active copy-inter T)
  (set-abort-inter copy-inter)
  )
  

;;;============================================================
;;;                        DELETE
;;;============================================================

;;;------------------------------------------------------------
;;; Delete-selected-objects deletes the objects currently selected.
;;; Delete arrows first to avoid problems with funny arrow lists in
;;; boxes. 
;;; This is supposed to be undo-able in the future.
;;;------------------------------------------------------------
(defun can-delete () (g-value obj-agg :selected))
(defun delete-selected-objects (&rest args)
  (declare (ignore args))
  (destroy-undo-objects)

  ;; save the selected objects for later restoration
  (setq undo-objects (copy-list (g-value obj-agg :selected)))

  ;; set selected set to nil
  (s-value obj-agg :selected nil)

  ;; "delete" the selected objects
  (dolist (obj undo-objects)
	  (s-value obj :visible nil)
	  (case (g-value obj :object-type)
		(:miro-arrow (delete-arrow obj))
		(:miro-box (delete-box obj))
		))

  ;; update the bounding box
  (s-value pic-sp :bb-box
	   (find-bounding-box (get-values obj-agg :components) T))

  ;; update the work window
  (mark-as-changed work-window :aggregate)
  (opal:update work-window)

  ;; update button visibility
  ;; these buttons depend on selection: copy, delete, display,
  ;; unselect
  ;; this button depends on undo-objects: undelete
  (update-command-inactive-list)

  ;; ambig results may not be valid any more
  (if (g-value obj-agg :components)
      (when (g-value ambig-status :guaranteed-valid)
	    (s-value ambig-status :guaranteed-valid nil))
    (s-value ambig-status :visible nil)
    )
  (opal:update menu-window)
  )

;;;------------------------------------------------------------
;;; Delete-Box deletes the specified box after deleting any attached
;;; arrows.
;;;------------------------------------------------------------
(defun delete-box (box-to-delete)
  ;; delete attached arrows
  (dolist (arrow (g-value box-to-delete :from-arrows))
	  (unless (g-value arrow :selected)
		  (s-value arrow :visible nil)
		  (s-value arrow :selected T)
		  (push arrow undo-objects))
	  (delete-arrow arrow)
	  )
  (dolist (arrow (g-value box-to-delete :to-arrows))
	  (unless (g-value arrow :selected)
		  (s-value arrow :visible nil)
		  (s-value arrow :selected T)
		  (push arrow undo-objects))
	  (delete-arrow arrow)
	  )
  (opal:remove-component obj-agg box-to-delete)
  )

;;;------------------------------------------------------------
;;; Delete-Arrow deletes the specified arrow and adjusts box arrow
;;; lists appropriately.
;;;------------------------------------------------------------
(defun delete-arrow (arrow-to-delete)
  (let ((from-box (g-value arrow-to-delete :from))
	(to-box (g-value arrow-to-delete :to)))
    ;; change to-arrows and from-arrows of appropriate boxes
    (s-value from-box :from-arrows
	     (delete arrow-to-delete
		     (g-value from-box :from-arrows)))
    (s-value to-box :to-arrows
 	     (delete arrow-to-delete
		     (g-value to-box :to-arrows)))
    )
  (opal:remove-component obj-agg arrow-to-delete)
  )


;;;============================================================
;;;                        DISPLAY
;;;============================================================

(defun can-display () (g-value obj-agg :selected))
(setq *objects-to-display* nil)
(defun display-next-object (&optional clear-list)
  (dolist (obj (g-value obj-agg :selected))
	  (s-value obj :selected nil))
  (s-value obj-agg :selected nil)
  (opal:update work-window)
  (when clear-list (setq *objects-to-display* nil))
  (if *objects-to-display*
      (let* ((pair (pop *objects-to-display*))
	     (obj (if (listp pair) (car pair) pair))
	     (auto-edit (if (listp pair) T nil))
	     )
	(s-value obj :selected T)
	(s-value obj-agg :selected (list obj))
	(opal:update work-window)
	(case (g-value obj :object-type)
	      ((:miro-box)
	       (call-schema zoom-agg :display-box
			    (g-value obj :box))
	       (display-box obj auto-edit))
	      ((:miro-arrow)
	       (call-schema zoom-agg :display-box
			    (g-value obj :points))
	       (display-arrow obj auto-edit))
	      )
	T)
    (progn (update-command-inactive-list) nil)
    ))
(defun display-object (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (setq *objects-to-display* (g-value obj-agg :selected))
  (display-next-object))

;;;============================================================
;;;                 UNSELECT ALL
;;;============================================================

;;;------------------------------------------------------------
;;; Unselect-All unselects all of the currently selected objects. 
;;;------------------------------------------------------------

(defun can-unselect () (g-value obj-agg :selected))
(defun unselect-all (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  ;; reset :selected slot of each object
  (dolist (obj (g-value obj-agg :selected))
	  (s-value obj :selected NIL))
  ;; reset :selected slot of obj-agg, and update window
  (s-value obj-agg :selected NIL)
  (opal:update work-window)

  ;; update button visibility: copy, delete, display, unselect
  (update-command-inactive-list)
  )


;;;============================================================
;;;                        CLEAR
;;;============================================================

;;;------------------------------------------------------------
;;; Clear-Workbench deletes all the miro-objects in the workbench.
;;; Miro-objects reside in obj-agg. 
;;; Use remove-component instead of destroy so we can Undo (eventually).
;;;------------------------------------------------------------

(defun clear-workbench (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (s-value *y-n-buttons* :question "Do you really want to clear the workbench?")
  (s-value *y-n-buttons* :help-msg
	   (format nil "Press ~S to clear the~%workbench, ~S to~%abort."
		   "Yes" "No"))
  (s-value *y-n-buttons* :no-fn
	   #'(lambda (&rest args)
	       (declare (ignore args))
	       (funcall (g-value *y-n-buttons* :how-to-stop))))
  (s-value *y-n-buttons* :yes-fn
	   #'(lambda (&rest args)
	       (declare (ignore args))
	       ;; unselect all objects
	       (s-value obj-agg :selected nil)

	       ;; Delete the previous contents of undo-objects.
	       (destroy-undo-objects)

	       ;; Put deleted objects in undo-objects.
	       (setq undo-objects (copy-list (get-values obj-agg :components)))

	       ;; select all objects and make them invisible
	       (dolist (obj undo-objects)
		       (s-value obj :visible nil)
		       (s-value obj :selected T)
		       )

	       ;; delete everything
	       (dolist (obj undo-objects)
		       (case (g-value obj :object-type)
			     (:miro-arrow (delete-arrow obj))
			     (:miro-box (delete-box obj))
			     ))

	       (set-box-creation2-string "File:")

	       (when (get-values obj-agg :components)
		     (push-error-msg
		      "clear-workbench:
obj-agg still has
components -- this
shouldn't happen!"))

	       ;; update button visibility: copy, delete, display,
	       ;; unselect, undelete
	       (update-command-inactive-list)

	       ;; --- can remove this later ---
	       (mark-as-changed work-window :aggregate)
	       ;; -----------------------------

	       (funcall (g-value *y-n-buttons* :how-to-stop))
	       ;; reset the scale and offsets by pretending that the
	       ;; reset button has been pressed.
	       (funcall (g-value zoom-agg :reset-button :selection-function)
			(g-value zoom-agg :reset-button) nil)
	       ;; update the bounding box
	       (s-value pic-sp :bb-box nil)
	       ))
  (funcall (g-value *y-n-buttons* :how-to-start))
  nil)

;;;============================================================
;;;                        EXIT
;;;============================================================

;;;------------------------------------------------------------
;;; Exit-editor destroys the window and everything that is inside.
;;;------------------------------------------------------------

(defun exit-editor (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (s-value *y-n-buttons* :question "Do you really want to exit the editor?")
  (s-value *y-n-buttons* :help-msg
	   (format nil "Press ~S to exit the~%editor, ~S to abort."
		   "Yes" "No"))
  (s-value *y-n-buttons* :no-fn
	   #'(lambda (&rest args)
	       (declare (ignore args))
	       (funcall (g-value *y-n-buttons* :how-to-stop))))
  (s-value *y-n-buttons* :yes-fn
	   #'(lambda (&rest args)
	       (declare (ignore args))
	       ;; don't bother calling :how-to-stop; we are leaving
	       ;; anyway...

	       ;; restore gc notification functions to their previous state
	       #+cmu
	       (progn
		 (setf *gc-use-help-window* nil)
		 (setf extensions:*gc-notify-before* *default-gc-before*)
		 (setf extensions:*gc-notify-after* *default-gc-after*))

	       (opal::destroy-me miro-window)
	       (opal::destroy-me dialog-window)
	       (setf *dont-interfere* nil)
	       (unless *started-from-disksave*
		       (dolist (dbox *all-dboxes*)
			       (opal:destroy-me dbox))
		       #-cmu (inter:exit-main-event-loop)
		       #-cmu (excl:gc T)
		       )
	       (when *started-from-disksave*
		     #+cmu (extensions:quit)
		     #-cmu (excl:exit))
	       ))
  (funcall (g-value *y-n-buttons* :how-to-start))
  nil)


;;;============================================================
;;;                        SAVE
;;;============================================================

;;;------------------------------------------------------------
;;; Save-Workbench saves the objects in the workbench in IFF format.
;;; If no  file name is given, the information is output in the slave
;;; window.  
;;; Eventually, will allow option for iff format or garnet format?
;;;------------------------------------------------------------
(defun save-workbench (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (block-interference
   :help-msg
   (format nil "Enter/edit the file name.~%Press ~S to~%save, ~S to abort."
	   "Confirm" "Abort"))
  (s-value
   filename-dbox :title
   "Enter/edit the file name to write (leave blank to write to the lisp window):")
  (s-value filename-dbox :confirm-function #'do-save)
  (call-schema filename-dbox :display-me)
  )

(defun do-save (fname)
  (build-containment-lists obj-agg)
  (set-help-string "Saving file...")
  (opal:update help-window)
  (when *test-debug* (format t "filename = ~S~%" fname))
  ;; if file name is null, output to slave
  (if (null-string fname)
      ;; output to slave
      (save-objects t)
    ;; else output to file
    (progn
      (set-box-creation2-string
       (format nil "File: ~A"
	       (namestring (pathname-name (pathname fname)))))
      (opal:update box-creation2-window)
      (with-open-file (file fname :direction :output
			    :if-exists :supersede
			    :if-does-not-exist :create)
		      (format t "Saving file...~%")
		      (let ((*package* (find-package "MIRO")))
			(save-objects file))
		      (format t "...done saving~%")
		      ))
    )
  (allow-interference)
  )

;;;------------------------------------------------------------
;;; Save-Objects does the actual output to file (or slave if file is
;;; T) 
;;;------------------------------------------------------------
(defun save-objects (file)
  (let* ((bounding-box (find-bounding-box (get-values obj-agg :components) T))
	 (origin (list (first bounding-box)
		       (+ (second bounding-box) (fourth bounding-box))
		       ))
	 )
    ;; print the date
    (format file "# created by miro-editor~%")
    (print-date file)

    ;; save editor entry
    (save-editor-entry file)
    ;; save object entries
    (dolist (obj (get-values obj-agg :components))
	    (case (g-value obj :object-type)
		  ((:miro-box)
		   (save-box file obj origin))
		  ((:miro-arrow)
		   (save-arrow file obj origin))))
    ;; save insideness entries
    (save-insideness file)  
    ;; add last carriage return
    (format file "~%")
    ))


;;;------------------------------------------------------------ 
;;; Save-Box saves information about a particular box in interm file
;;; format. The position of the box in the IFF coord system is
;;; calculated from the position in the Garnet coord system. Need to
;;; add thickness info for constraint boxes 
;;;------------------------------------------------------------

(defun save-box (file obj origin)
  (format file "~%
>BOX
   sysname = ~S;
   role = ~(~A~);
   name = ~S;
   loc = {~S,~S};
   size = {~S,~S};"
	  ;; sysname
	  (g-value obj :sysname)
	  ;; role -- should come out lower case
	  (g-value obj :box-role)
	  ;; name
	  (get-object-string obj)
	  ;; loc -- convert y co-ord to bottom
	  ;; use box coord to get actual value, not scrolled location
	  (- (first (g-value obj :box)) (first origin))
	  (- (second origin) (+ (second (g-value obj :box)) 
				(fourth (g-value obj :box))
		    ))
	  ;; size
	  (third (g-value obj :box))
	  (fourth (g-value obj :box))
	  )
  (unless (equal (g-value obj :box-type) "")
	  (format file "~%   type = ~A;" (g-value obj :box-type)))
  (unless (g-value obj :contains)
	  (format file "~%   atomic = true;"))
  ; extra attributes for constraint boxes
  (when (eq (g-value pictype-menu :value) :constraint)
    (if (g-value obj :thick) 
	(format file "~%   thickness = thick;")
	(format file "~%   thickness = thin;"))
    (if (g-value obj :starred) 
	(format file "~%   starred? = true;")
	(format file "~%   starred? = false;"))))


;;;------------------------------------------------------------ 
;;; Save-Arrow saves information about a particular box in interm file
;;; format. The position of the arrow in the IFF coord system is
;;; calculated from the position in the Garnet coord system. 
;;; use ~A for type to get rid of double quotes
;;;------------------------------------------------------------ 

(defun save-arrow (file obj origin)
  (let ((points (g-value obj :points)))
    (format file "~%
>ARROW
   sysname = ~S;
   from = ~S;
   to = ~S;
   tail-loc = {~S,~S};
   head-loc = {~S,~S};"
	    ;; sysname
	    (g-value obj :sysname)
	    ;; from & to
	    (g-value obj :from :sysname)
	    (g-value obj :to :sysname)
	    ;; tail-loc (convert y to bottom)
	    (- (first points) (first origin))
	    (- (second origin) (second points))
	    ;; head-loc (convert y to bottom)
	    (- (third points) (first origin))
	    (- (second origin) (fourth points)))
    (let ((type (read-list-or-string (g-value obj :label :string))))
      (cond ((or (eq type :empty-list) (null type) (equal type "")) nil)
	    ((listp type)
	     (format file "~%   permissions = {~{~A~^,~}};" type))
	    (T (format file "~%   permissions = {~A};" type))))
    ;; parity
    (if (g-value obj :neg) (format file "~%   parity = neg;")
      (format file "~%   parity = pos;"))
    (when (eq (g-value pictype-menu :value) :constraint)
	  (let ((kind (g-value obj :arrow-type)))
	    (cond ((equal kind :syn) 
		   (format file "~%   kind  = syn;"))
		  ((equal kind :sem) 
		   (format file "~%   kind  = sem;"))
		  ((equal kind :con) 
		   (format file "~%   kind  = con;"))
		  (t ))
	    (if (g-value obj :thick) 
		(format file "~%   thickness = thick;")
	      (format file "~%   thickness = thin;"))
	    ;; only save starred for containment
	    (cond ((and (equal kind :con) (g-value obj :starred) )
		   (format file "~%   starred? = true;"))
		  ((and (equal kind :con) (not (g-value obj :starred) ))
		   (format file "~%   starred? = false;"))
		  (T nil))))
    ))

;;;------------------------------------------------------------ 
;;; Save-Insideness saves box insideness information for a picture to
;;; file 
;;;------------------------------------------------------------ 
(defun save-insideness (file)  
  ; check whether need to recompute -- eventually this will check a flag...
  (if T 
      ;then
      (build-containment-lists obj-agg)
      ;else
      (when *debug-contains* (format T "Don't have to recompute containment~%")))
  ; loop to save info for each box that directly contains something
  (dolist (obj (get-values obj-agg :components))
    (if (and (equal (g-value obj :object-type) :miro-box)
	     (g-value obj :direct-contains))
	(save-insideness-entry obj file)
	NIL )))

;;;------------------------------------------------------------ 
;;; save insideness entry for box
;;; don't know how to save list with format, so call separate function
;;;------------------------------------------------------------ 
(defun save-insideness-entry (box file)
  (format file "~%
>INSIDE parent=~S; children={" (g-value box :sysname))
  (save-insides box file)
  (format file "};"))

;;;------------------------------------------------------------ 
;;; save list of boxes directly contained in box
;;;------------------------------------------------------------ 
(defun save-insides (box file)
  (let ((contains (g-value box :direct-contains)))
  ; save first entry
  (format file "~S" (g-value (car contains) :sysname))
  (dolist (b2 (cdr contains))
    (format file ",~S" (g-value b2 :sysname)))
))


;;;------------------------------------------------------------
;;; save the editor entry for a picture to file
;;;------------------------------------------------------------
(defun save-editor-entry (file)  
  (format file "~%
>EDITOR")
  ; pictype
  (if (eq (g-value pictype-menu :value) :instance)
      (format file "~%   pictype = instance;")
      (format file "~%   pictype = constraint;"))
)

;;;------------------------------------------------------------
;;; CONTAINMENT FUNCTIONS
;;;------------------------------------------------------------

;;;------------------------------------------------------------
;;; build-containment-lists calculates the :contains, :contained-by
;;; and :direct-contains slots of each box in agg
;;;------------------------------------------------------------ 

(defun build-containment-lists (agg &optional box)
  (let ((box-list (build-box-list-for-containment agg box)))
    (dolist (b box-list)
      (build-contains-and-contained-by b box-list))
    (dolist (b box-list)
      (build-direct-contains b (g-value b :contained-by))))
  )

;;;------------------------------------------------------------ 
;;; returns box-list, a list of boxes in agg. 
;;; Initializing slots :contains, :contained-by and :direct-contains
;;; of each box to NIL
;;;------------------------------------------------------------
 
(defun build-box-list-for-containment (agg box)
  (let ((box-list NIL)
	(components (get-values agg :components))
	)
    (dolist (obj (if box (cons box components) components))
      (if (equal (g-value obj :object-type) :miro-box)
	  (push (reset-containment obj) box-list)
	  NIL))
    box-list))

;;;------------------------------------------------------------ 
;;; resets :contains, :contained-by and :direct-contains slots of box
;;; to NIL 
;;;------------------------------------------------------------ 

(defun reset-containment (box)
  (s-value box :contains NIL)
  (s-value box :contained-by NIL)
  (s-value box :direct-contains NIL)
  box)


;;;------------------------------------------------------------ 
;;; set the :contains slot of b, and for each box in boxlist that b
;;; contains, update its :contained-by slot
;;;------------------------------------------------------------ 

(defun build-contains-and-contained-by (b boxlist)
  (let ((contains NIL))
    (dolist (b1 boxlist)
      (if (box-inside-p b1 b)
	  (progn
	    ; b1 is contained by b
	    (push b (g-value b1 :contained-by))
	    ; add b1 to list of boxes that b contains
	    (push b1 contains))
	  NIL))
    ; set :contains slot of b
    (s-value b :contains contains)))

;;;------------------------------------------------------------ 
;;; for each box in boxlist, check whether it directly contains b
;;; if it does, update its :direct-contains slot.
;;; boxlist is assumed to be a list of boxes that contain b
;;; b1 directly contains b if there is not another box b2 that is
;;; "in between" (ie, b1 contains b2 and b2 contains b)
;;;------------------------------------------------------------

(defun build-direct-contains (b boxlist)
  (when *debug-contains* (format T "building direct containment for ~S~%" b))
  (dolist (b1 boxlist)
    (let ((one-in-between NIL))
      (dolist (b2 boxlist) 
	(cond 
	 ; if already know b1 doesn't directly contain b, quit
	 (one-in-between (return NIL))
	 ; if b1 contains b2, know there's one in between
	 ; could let this kick directly out, but then setting
	 ; direct containment isn't as clear (has to be exit
	 ; condition of dolist...)
	 ((member b2 (g-value b1 :contains))
	  (setq one-in-between T))
	 ; otherwise, status doesn't change; keep going
	 (T NIL)))
      (when *debug-contains* (format T "outside inner dolist, b1 is ~S~%" b1))
      ; if there's not a box b2 such that b1 contains b2 and b2 contains b,
      ; then b1 directly contains b
      (if (not one-in-between)
	  (push b (g-value b1 :direct-contains))))))


;;;------------------------------------------------------------ 
;;; returns list of boxes inside of b
;;; don't use this any more, but might be usefull for debugging...
;;;------------------------------------------------------------

(defun get-boxes-inside (box)
  (let ((contains nil))
    (dolist (obj (get-values obj-agg :components))
      (if (equal (g-value obj :object-type) :miro-box)
	  (if (box-inside-p obj box)
	      (push obj contains)
	      nil)
	  nil))
    ;return the list of boxes contained in box
    contains))

;;;------------------------------------------------------------
;;; returns T iff box1 is completely inside box2
;;;------------------------------------------------------------
(defun box-inside-p (box1 box2)
  (let* ((b1 (g-value box1 :box))
	 (b2 (g-value box2 :box))
	 (left1 (first b1))
	 (left2 (first b2))
	 (top1 (second b1))
	 (top2 (second b2))
	 (width1 (third b1))
	 (width2 (third b2))
	 (height1 (fourth b1))
	 (height2 (fourth b2))
	 )
    (and (and (< left2 left1) (< top2 top1))
	 (and (> (+ left2 width2) (+ left1 width1))
	      (> (+ top2 height2) (+ top1 height1))))
    ))


;;;============================================================
;;;                        READ
;;;============================================================

;;;------------------------------------------------------------
;;; Read-miro-file reads from a file in IFF and creates the appropriate
;;; miro objects in the obj-agg. 
;;; We're almost there -- need to set formulas for arrows, and add some
;;; error checking
;;;------------------------------------------------------------
(defun read-miro-file (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (block-interference
   :help-msg
   (format nil "Enter/edit the file name.~%Press ~S to~%read, ~S to abort."
	   "Confirm" "Abort"))
  (s-value filename-dbox :title
	   "Enter/edit the file name to read:")
  (s-value filename-dbox :confirm-function #'do-read)
  (call-schema filename-dbox :display-me)
  )

(defun agg-from-file (fname &optional no-locations)
  ;; check if file exists and give error if doesn't
  ;; parse file and get new objects
  (cond
   ((null-string fname)
    (push-error-msg "No file specified.")
    (format t "No file specified.~%")
    (force-output)
    nil)
   ((not (probe-file fname))
    (push-error-msg "File does not exist.")
    (format t "File ~S does not exist.~%" fname)
    (force-output)
    nil)
   (T
    (let* ((*package* (find-package "MIRO"))
	   (results
	    (run-miro-tool "iff2lisp"
			   (append
			    (list "-input"
				  (namestring
				   (truename
				    (pathname fname))))
			    (when no-locations (list "-noloc"))
			    )))
	   (picture-fn (first results))
	   (iff2lisp-errors (second results))
	   (exit-status (third results))
	   )

      (if (and (or (null exit-status) (/= exit-status 0))
	       (not (equal iff2lisp-errors "")))
	  (progn
	    (push-error-msg "Errors encountered in
iff2lisp (see the
lisp window for
details)")
	    (format T "Errors from iff2lisp:~%~A~%" iff2lisp-errors)
	    (force-output)
	    nil)
	(progn
	     (set-help-string (format nil "No errors -- ~A
picture..." (if no-locations "reading" "creating")))
	     (opal:update help-window)
	     (unless no-locations
		     (set-box-creation2-string
		      (format nil "File: ~A"
			      (namestring (pathname-name
					   (pathname fname)))))
		     (opal:update box-creation2-window)
		     )
	     ;; read the picture
	     (let ((parsed-agg (create-instance nil opal:aggregate)))
	       (funcall (eval (read-from-string picture-fn T)) parsed-agg)
	       parsed-agg)))
      ))
   ))


(defun do-read (fname &optional intermediate-fn)
  (let (temp-list bounding-box origin)
    (set-help-string
     "Reading file... Any
errors or warnings
will appear in the
lisp window.")
    (opal:update help-window)

    (let ((parsed-agg (agg-from-file fname)))
      (when parsed-agg
	    (let ((new-sysnames (gensym)))
	      ;; fix box sysnames
	      (dolist (b (g-value parsed-agg :boxes))
		      (let ((sname (get-new-number)))
			(setf (get new-sysnames (g-value b :sysname)) sname)
			(s-value b :sysname sname)))

	      ;; fix arrow sysnames
	      (dolist (a (g-value parsed-agg :arrows))
		      (let ((sname (get-new-number)))
			(setf (get new-sysnames (g-value a :sysname)) sname)
			(s-value a :sysname sname))
		      (s-value a :from-sysname (get new-sysnames (g-value a :from-sysname)))
		      (s-value a :to-sysname (get new-sysnames (g-value a :to-sysname))))
	      )

	    (set-help-string "Converting coordinates...")
	    (opal:update help-window)

	    (setq bounding-box
		  (or (g-value parsed-agg :bounding-box)
		      (find-bounding-box
		       (append (g-value parsed-agg :boxes)
			       (g-value parsed-agg :arrows)) T)))
	    (setq origin (list (- (first bounding-box) 5)
			       (+ (second bounding-box) (fourth bounding-box) 5)))
	    ;; convert y coords and put each box in agg into workbench
	    (dolist (obj (g-value parsed-agg :boxes))
		    ;; (format T "Moving ~s ~%" obj)
		    ;; convert y coord (2nd in :box)
		    (setq temp-list (g-value obj :box))
		    (rplaca (g-value obj :box) (- (first temp-list) (first origin)))
		    (rplaca (cdr (g-value obj :box))
			    (- (second origin)
			       (+ (second temp-list) (fourth temp-list))))
		    (opal:add-component obj-agg obj))

	    (set-help-string "Drawing...")
	    (opal:update help-window)

	    ;; copy the hash table from agg
	    (s-value obj-agg :hashtbl (g-value parsed-agg :hashtbl))
	    ;; convert y coords and put each arrow in agg into workbench
	    (dolist (obj (g-value parsed-agg :arrows))
		    ;; (format T "Moving arrow ~s ~%" obj )
		    ;; attach to boxes if can (ow convert y values)
		    ;; eventually change to print error if no boxes...
		    (set-box-info obj)
		    (opal:add-component obj-agg obj))
	    ;; check other slots of agg
	    ;;(format T "Setting pictype to ~S~%" (g-value parsed-agg :pictype))
	    (case (g-value parsed-agg :pictype)
		  (:instance (call-schema pictype-menu :set-value :instance)
			     (call-schema pictype-menu :selection-function :instance)
			     )
		  (:constraint (call-schema pictype-menu :set-value :constraint)
			       (call-schema pictype-menu :selection-function :constraint)
			       )
		  (T nil))
	    (format t "Finished reading ~%")
	    (force-output)
	    ;; update the bounding box
	    (s-value pic-sp :bb-box
		     (find-bounding-box (get-values obj-agg :components) T))
	    ;; we may have invalidated undo-objects
	    (destroy-undo-objects)
	    ;; update button visibility: undelete
	    (update-command-inactive-list)

	    ;; we may have invalidated ambig results
	    (when (g-value ambig-status :guaranteed-valid)
		  (s-value ambig-status :guaranteed-valid nil))
	    (opal:update menu-window)
	    ;; delete parsed-agg so that it doesn't hang around forever
	    (opal::destroy parsed-agg)
	    )))
	     
  (unless intermediate-fn (allow-interference))
  )

;;;------------------------------------------------------------
;;; Set the :to and :from slots of an arrow, the :to- and from-arrows
;;; of their boxes, and the formulas for :x1...:y2
;;;------------------------------------------------------------
(defun set-box-info (arrow)
  ;; set :to correctly if info is there
  (if (and (g-value arrow :to-sysname) (find-box (g-value arrow :to-sysname) obj-agg))
      (s-value arrow :to (find-box (g-value arrow :to-sysname) obj-agg))) 
  ;; set :from correctly if info is there
  (if (and (g-value arrow :from-sysname) 
	   (find-box (g-value arrow :from-sysname) obj-agg))
      (s-value arrow :from (find-box (g-value arrow :from-sysname) obj-agg)))
  ;; if know both boxes, set constraints, otherwise convert y values
  (if (and (g-value arrow :to) (g-value arrow :from))
      (progn
	;; set :from-arrows and :to-arrows of boxes
	(push arrow (g-value (g-value arrow :from) :from-arrows))
	(push arrow (g-value (g-value arrow :to) :to-arrows))
	;; set formulas
	(set-arrow-formulas arrow))
      (progn
	(s-value arrow :y1 (- (g-value obj-agg :height) (g-value arrow :y1)))
	(s-value arrow :y2 (- (g-value obj-agg :height) (g-value arrow :y2))))
      ))


;;;============================================================
;;;                        PROBER
;;;============================================================
(defun can-probe () (eq (g-value pictype-menu :value) :instance))

(defun probe-directory (obj string)
  (declare (ignore obj string))
  (block-interference
   :help-msg (format nil "Change prober options
as needed.  Press
~S to run the
prober, ~S to
abort." "Run Prober" "Abort"))
  (call-schema probe-dbox :initialize-options)
  )

(defvar *probe-debug* T)

(defun do-probe (obj string)
  (declare (ignore obj string))
  (let* ((directory (string-right-trim '(#\/) (g-value probe-dbox
						       :directory)))
	 (output-to-workbench (null-string
			       (g-value probe-dbox :output-file)))
	 (output-file
	  (if output-to-workbench (make-temporary-file "/tmp/Miro.iff")
	    (g-value probe-dbox :output-file)))
	 (save-group-file (not (null-string (g-value probe-dbox
						     :group-output))))
	 (group-output
	  (if save-group-file (g-value probe-dbox :group-output)
	    (make-temporary-file "/tmp/Miro.group")))
	 (save-user-file (not (null-string (g-value probe-dbox
						    :user-output))))
	 (user-output
	  (if save-user-file (g-value probe-dbox :user-output)
	    (make-temporary-file "/tmp/Miro.user")))

	 (perm-output (make-temporary-file "/tmp/Miro.perm"))
	 (arg-list nil)
	 (error nil)
	 )

    ;; build the argument list
    (push directory arg-list)		; directory to probe
    (progn				; user output
      (push user-output arg-list)
      (push "-uo" arg-list))
    (progn				; group output
      (push group-output arg-list)
      (push "-go" arg-list))

    (when *probe-debug* (format T "probe arg list: ~S~%" arg-list))

    ;; run the prober
    (let* ((results (run-miro-tool "probe" arg-list :stdout perm-output))
	   (stderr (second results))
	   )

      ;; print the error output
      (unless (null-string stderr)
	      (format T "probe errors:~%~A" stderr)
	      (force-output))

      ;; make sure we can access all the relevant files
      ;; (should make sure we can write output-file)
      (if (and (probe-file group-output)
	       (probe-file user-output)
	       (probe-file perm-output))
	  (apply #'write-iff-from-prober-results
		 (cons output-file
		       (process-prober-output
			perm-output group-output
			user-output)))
	(setf error "Couldn't access one of the
specified files.")
	)
      )

    (when (and output-to-workbench (not error))
	  (do-read output-file T))
    (allow-interference)
    (remove-dbox probe-dbox)
    (opal:lower-window dialog-window)

    ;; delete temporary files
    (when output-to-workbench (delete-file output-file))
    (unless save-group-file (delete-file group-output))
    (unless save-user-file (delete-file user-output))
    (delete-file perm-output)

    (when error (push-error-msg error))
    (excl:gc)
    (force-output)
    ))

;; path should be a full pathname
(defun process-path (path ftable btable itable sysname
			  &optional is-dir)
  (let* ((slash (position #\/ path :from-end T))
	 (file (when slash (subseq path (+ 1 slash))))
	 (dir (when slash (subseq path 0 slash)))
	 (parent 2)
	 )

    (assert dir (path file dir slash) "~S is not a full path" path)

    ;; don't need to do anything if path is already in the table
    (unless
     (or (not slash) (gethash path ftable))

     ;; get the parent's sysname
     (unless (null-string dir)
	     (setq sysname (process-path dir ftable btable itable
					 sysname T))
	     (setq parent (gethash dir ftable))
	     )
     (assert parent (parent path file dir ftable sysname)
	     "parent not in hash table")

     ;; create a new box
     (setf (gethash path ftable) (incf sysname))
     (push sysname (gethash parent itable))
     (setf (gethash sysname btable)
	   (list "file" "file"
		 (if is-dir (concatenate 'string file "/") file)))
     )
    )
  sysname)

(defun generate-arrows (protection-type protection-list sysname file
					user-table group-table arrow-table)
  (assert (listp protection-list) (protection-list)
	  "argument to generate-arrows should be a list")
  (cond
   ((null protection-list) T)
   ((listp (car protection-list))
    (setq sysname (generate-arrows protection-type
				   (car protection-list) sysname file
				   user-table group-table arrow-table))
    (setq sysname (generate-arrows protection-type
				   (cdr protection-list) sysname file
				   user-table group-table arrow-table)))
   (T
    (let ((arrow-type nil)
	  (parity (equal (string-downcase (car protection-list))
			 "pos"))
	  (user-list (cdr protection-list))
	  )
      ;; make the arrow label
      (dolist (type protection-type)
	      (assert
	       (or (when (equal type "R") (push "read" arrow-type) T)
		   (when (equal type "W") (push "write" arrow-type) T)
		   (when (equal type "E") (push "exec" arrow-type) T))
	       (protection-type arrow-type type)
	       "unrecognized arrow type: ~S" type))
      (setq arrow-type (sort arrow-type #'string<))

      ;; print an arrow for each user/group
      (if (null user-list)
	  (push (list (incf sysname) file arrow-type parity)
		(gethash 1 arrow-table))
	(dolist
	 (user-or-group user-list)
	 (when (listp user-or-group)
	       (format T "~%WARNING: not handling group list: ~S~%"
		       user-or-group)
	       (setq user-or-group (car user-or-group))
	       )
	 (let* ((colon (position #\: user-or-group :from-end T))
		(user (if (and colon (equal ":1"
					    (subseq user-or-group
						    colon)))
			  (gethash (subseq user-or-group 0 colon)
				   group-table)
			(gethash user-or-group user-table)))
		)
	   (assert user (user user-or-group user-table group-table)
		   "user/group not found in hash table")
	   (push (list (incf sysname) file arrow-type parity)
		 (gethash user arrow-table))
	   )))))
   )
  sysname)

;; returns (prober-comments top-user top-file user-table group-table
;;          file-table box-table arrow-table inside-table)
;; prober-comments: a string containing the first 2 lines of prober
;;                  output, which identify the directory & machine
;; top-user, top-file: sysnames of "world" and "/", respectively
;; user-table, group-table: key=string(name), value=sysname(int)
;; file-table: key=string(full path), value=sysname(int)
;; box-table: key=sysname(int),
;;            value=list of (role type name)
;;                           ^----^----^string
;; arrow-table: key=sysname(int) of user/group box,
;;              value= list of
;;                     (arrow-sysname to-sysname permissions parity)
;;                      ^int          ^int       ^ list      ^T=pos,nil=neg
;; inside-table: key=parent sysname(int),
;;               value=list of child sysnames(int)
(defun process-prober-output (probe-output group-file user-file)
  (let ((user-table (make-hash-table :test #'equal))
	(group-table (make-hash-table :test #'equal))
	(file-table (make-hash-table :test #'equal))
	(box-table (make-hash-table))
	(arrow-table (make-hash-table))
	(inside-table (make-hash-table))
	(prober-comments nil)
	(*eof* :eof)
	(sysname 2)			; increment first -- 1, 2 reserved
	(whitespace '(#\space #\newline #\tab #\page #\return))
	(world-children nil)
	)

    (with-open-file
     (probe-stream probe-output :direction :input)
     ;; the first 2 lines of prober output should be comments saying
     ;; where the file came from.  copy these lines into the iff file
     (let* ((line1 (read-line probe-stream nil *eof*))
	    (line2 (read-line probe-stream nil *eof*))
	    )
       (setq
	prober-comments
	(if (equal line2 *eof*)
	    (format
	     nil
	     "# error reading probe output -- no location/machine specified~%")
	  (format nil "~A~%~A~%" line1 line2))))

     ;; put all the groups in the hash table
     (with-open-file
      (group-stream group-file :direction :input)
      (let (group users line)
	(loop
	 ;; read a group entry
	 (when (equal (setq line
			    (read-string group-stream *eof*
					 :non-string-chars nil
					 :terminating-chars
					 '(#\.)))
		      *eof*) (return))
	 ;; skip the terminating character
	 (read-char group-stream nil *eof*)
	
	 ;; parse the line we just read
	 (let ((equalpos (position #\= line)))
	   (assert equalpos (equalpos line)
		   "Malformed prober group output: ~S" line)
	   ;; extract the group
	   (setf group (string-trim whitespace
				    (subseq line 0 equalpos)))
	   ;; extract the user list
	   (setf users nil)
	   (do* ((done nil (not end))
		 (start (+ equalpos 1) (unless done (+ end 1)))
		 (end (position #\, line :start start)
		      (unless done (position #\, line :start start)))
		 (next-user (string-trim whitespace
					 (subseq line start end))
			    (unless done
				    (string-trim whitespace
						 (subseq line start
							 end))))
		 (next-sysname (gethash next-user user-table)
			       (unless done (gethash next-user
						     user-table)))
		 )
		((or (null-string next-user) done)
		 (setf users (reverse users)))
		;; if we haven't seen this user yet, add it to the hash table
		(unless next-sysname
			(setf (gethash (incf sysname) box-table)
			      (list "user" "user" next-user))
			(setf (gethash next-user user-table) sysname)
			(setf next-sysname sysname)
			)
		(push next-sysname users)
		)
	   )

	 ;; print group box and containment information
	 (setf (gethash (incf sysname) box-table)
	       (list "user" "group" group))
	 (push sysname world-children)
	 (setf (gethash group group-table) sysname)
	 (setf (gethash sysname inside-table)
	       (append users (gethash sysname inside-table)))
	 )))

     ;; put all the users in the hash table
     (with-open-file
      (user-stream user-file :direction :input)
      (let (user)
	(loop
	 (skip-comments-and-whitespace user-stream #\#)
	 (when (equal (setq user (read-line user-stream nil *eof*))
		      *eof*) (return))

	 ;; create a new box and put it in the hash tables
	 (unless
	  (gethash user user-table)
	  (setf (gethash (incf sysname) box-table)
		(list "user" "user" user))
	  (push sysname world-children)
	  (setf (gethash user user-table) sysname)
	  )
	 )))

     ;; create the world box
     (setf (gethash 1 box-table) (list "user" "world" "world"))
     (setf (gethash 1 inside-table) world-children)

     ;; create /
     (setf (gethash 2 box-table) (list "file" "file" "/"))

     ;; generate the arrows & filenames
     (let (file)
       (loop
	;; read a filename
	(skip-comments-and-whitespace probe-stream #\#)
	(when (equal (setq file (read-line probe-stream nil *eof*))
		     *eof*) (return))
	(let ((last (- (length file) 1)))
	  (assert (equal (elt file last) #\:)
		  (file)
		  "syntax error in prober output -- ':' expected")
	  (setq file (string-trim whitespace (subseq file 0 last)))
	  )

	;; create boxes if necessary
	(setq sysname (process-path (string-trim whitespace file)
				    file-table box-table inside-table
				    sysname))

	;; read protection lists and generate arrows
	(loop
	 (let ((last-one nil)
	       line colon access-types access-list last-pos)
	   (skip-comments-and-whitespace probe-stream #\#)
	   (setq line (read-line probe-stream nil *eof*))
	   (assert (not (equal line *eof*)) (line)
		   "premature eof in prober output")
	   (setq colon (position #\: line))
	   (assert colon (line colon)
		   "syntax error in prober-output -- ':' expected")
	   (setq last-pos (- (length line) 1))
	   (when (equal (elt line last-pos) #\.)
		 (setq last-one T)
		 (setq line (subseq line 0 last-pos)))
	   (setq access-types
		 (get-arg-list (string-trim whitespace
					    (subseq line 0 colon))))
	   (setq access-list
		 (read-list-or-string
		  (string-trim whitespace
			       (subseq line (+ colon 1)))
		  T))

	   ;; generate arrows
	   (setq sysname (generate-arrows access-types access-list
					  sysname
					  (gethash file file-table)
					  user-table group-table
					  arrow-table))
	    
	   (when last-one (return))
	   ))
	))
     )
    ;; return
    (list prober-comments 1 2 user-table group-table file-table
	  box-table arrow-table inside-table)
    ))

(defun write-iff-from-prober-results
  (output-file prober-comments top-user top-file user-table
	       group-table file-table box-table arrow-table
	       inside-table)
  (declare (ignore top-user top-file file-table group-table user-table))
  (with-open-file
   (output-stream output-file :direction :output :if-exists :supersede
		  :if-does-not-exist :create)
   ;; print the date
   (format output-stream
	   "# created by miro-editor from prober output~%")
   (print-date output-stream)
   (format output-stream "~A" prober-comments)

   ;; print editor entry
   (format output-stream "~%>EDITOR pictype = instance;~%")

   ;; print the boxes
   (format output-stream "~%")
   (maphash
    #'(lambda (sysname boxinfo)
	(format output-stream
		">BOX sysname=~S; role=~(~A~); type=~(~A~); name=~S; ~A~%"
		sysname (first boxinfo) (second boxinfo)
		(third boxinfo)
		(if (gethash sysname inside-table) ""
		  "atomic=true;"))
	)
    box-table)

   ;; print the arrows
   (format output-stream "~%")
   (maphash
    #'(lambda (from arrow-list)
	(dolist
	 (arrow arrow-list)
	 (format
	  output-stream
	  ">ARROW sysname=~S; from=~S; to=~S; permissions={~{~A~^,~}}; parity=~A;~%"
	  (first arrow) from (second arrow) (third arrow)
	  (if (fourth arrow) "pos" "neg"))))
    arrow-table)

   ;; print the inside entries
   (format output-stream "~%")
   (maphash
    #'(lambda (parent children)
	(format output-stream
		">INSIDE parent=~S; children={~{~S~^,~}};~%"
		parent children))
    inside-table)
   ))

;;;============================================================
;;;                        VERIFY
;;;============================================================
(defun can-verify () (eq (g-value pictype-menu :value) :instance))

(defun verify-workbench (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (block-interference
   :help-msg (format nil "Change verifier options
as needed.  Press
~S to run
the verifier,
~S to abort." "Run Verifier" "Abort"))
  (funcall (g-value verify-dbox :initialize-options))
  )

(defvar *verify-debug* T)

;; assume all relevant boxes exist in the workbench
;;
;; user, file -- (string) user & file to check
;; wb-user-arrow-tbl, p-user-arrow-tbl -- hashtable for workbench, prober arrows
;; wb-prober-tbls, p-prober-tbls -- process-prober-output style lists

(defun perm-match (a1 a2) (member (car a1) (caddr a2) :test #'string-equal))
(defun parity-match (a1 a2) (eq (cadr a1) (cadddr a2)))
(defun arrow-changes (user file wb-user-arrow-tbl wb-prober-tbls
			   p-user-arrow-tbl p-prober-tbls)
  (let* ((add-list nil)
	 (delete-list nil)
	 (wuser-tbl (fourth wb-prober-tbls))
	 (wfile-tbl (sixth wb-prober-tbls))
	 (warrow-tbl (eighth wb-prober-tbls))
	 (puser-tbl (fourth p-prober-tbls))
	 (pfile-tbl (sixth p-prober-tbls))
	 (parrow-tbl (eighth p-prober-tbls))
	 (wb-user-sysname (gethash user wuser-tbl))
	 (p-user-sysname (gethash user puser-tbl))
	 (wb-file-sysname (gethash file wfile-tbl))
	 (p-file-sysname (gethash file pfile-tbl))
	 (pworld-arrows (gethash (second p-prober-tbls)
				 parrow-tbl))
	 (parrows
	   (mapcar #'cdr
		   (or
		    (remove-if-not #'(lambda (s) (eq s p-file-sysname))
				   (gethash p-user-sysname p-user-arrow-tbl)
				   :key #'car)
		    (remove-if-not #'(lambda (s) (eq s p-file-sysname))
				   (let ((l nil))
				     (mapc #'(lambda (a)
					       (mapc #'(lambda (p)
							 (push
							  (list (second a)
								(string-downcase p)
								(fourth a))
							  l))
						     (third a)))
					   pworld-arrows)
				     l)
				   :key #'car)
		    )))
	 (warrows
	  (mapcar #'cdr
		  (remove-if-not #'(lambda (s) (eq s wb-file-sysname))
				 (gethash wb-user-sysname wb-user-arrow-tbl)
				 :key #'car)))
	 (all-arrows (remove-if-not #'(lambda (s) (eq s wb-file-sysname))
				    (gethash wb-user-sysname warrow-tbl)
				    :key #'cadr))
	 )

    ;; arrows to add
    (dolist (a (set-difference parrows warrows :test #'equal))
	    (let ((op (find-if #'(lambda (ar) (and (perm-match a ar)
						   (not (parity-match a ar))))
			       all-arrows))
		  )
	      (when op (pushnew (car op) delete-list))
	      (pushnew a add-list :test #'equal)
	      )
	    )

    ;; arrows to delete
    (dolist (a (set-difference warrows parrows :test #'equal))
	    (let ((same (find-if #'(lambda (ar) (and (perm-match a ar)
						     (parity-match a ar)))
				 all-arrows))
		  )
	      (when same (pushnew (car same) delete-list))
	      (pushnew (list (car a) (not (cadr a))) add-list :test #'equal)
	      )
	    )
    (list add-list delete-list)))

;; fix permissions for the specified user/file pairs
(defun fix-perms (pair-list)
  (let* ((wb-prober-list (g-value verify-update-menu :workbench))
	 (p-prober-list (g-value verify-update-menu :prober-results))
	 (wb-user-arrow-tbl (g-value verify-update-menu :wb-user-arrows))
	 (p-user-arrow-tbl (g-value verify-update-menu :pr-user-arrows))
	 (arrows-to-delete nil)
	 (perm-change-list (copy-list (g-value verify-update-menu
					       :perm-change-list)))
	 )
    ;; need to update:
    ;; wb-arrow-list, wb-user-arrow-list, wb-user-file-list,
    ;; perm-change-list
    (dolist (pair pair-list)
	    (let* ((user-str (car pair))
		   (file-str (cadr pair))
		   (ac (arrow-changes user-str file-str wb-user-arrow-tbl
				      wb-prober-list p-user-arrow-tbl
				      p-prober-list))
		   (a-to-add (car ac))
		   (a-to-delete (cadr ac))
		   (user-box (find-box (gethash user-str
						(fourth wb-prober-list))))
		   (file-box (find-box (gethash file-str
						(sixth wb-prober-list))))
		   (user-arrows (when user-box
				      (g-value user-box :from-arrows)))
		   )
	      (when (and user-box file-box)
		    (dolist (a a-to-add)
			    (let ((new-arrow
				   (create-miro-arrow  0 0 0 0 user-box file-box
						       (not (cadr a)) nil :syn
						       nil)))
			      (s-value (g-value new-arrow :label) :string (car a))
			      (set-arrow-formulas new-arrow)
			      (opal:add-component obj-agg new-arrow)
			      (push new-arrow (g-value user-box :from-arrows))
			      (push new-arrow (g-value file-box :to-arrows))
			      )))
	      (dolist (a a-to-delete)
		      (let ((a-to-d (find-if #'(lambda (r)
						 (eq (g-value r :sysname) a))
					     user-arrows)))
			(when a-to-d (push a-to-d arrows-to-delete))
			))
	      )
	    (setf perm-change-list (delete pair perm-change-list :test #'equal))
	    )
    ;; delete arrows
    (when arrows-to-delete
	  (dolist (o (g-value obj-agg :selected)) (s-value o :selected nil))
	  (s-value obj-agg :selected arrows-to-delete)
	  (dolist (a (g-value obj-agg :selected)) (s-value a :selected T))
	  (delete-selected-objects)
	  )

    ;; update the change list in verify-update-menu
    (s-value verify-update-menu :perm-change-list
	     (sort perm-change-list
		   #'(lambda (p1 p2)
		       (let ((u1 (car p1))
			     (u2 (car p2)))
			 (or (string< u1 u2)
			     (and (string= u1 u2)
				  (string< (cadr p1) (cadr p2))))))
		   ))
    ))

;; user-arrows, file-arrows: key = user/file,
;; value = ((file/user perm parity(T=pos)) ...)
(defun arrow-tbls
  (users file perms parity user-arrows file-arrows box-table inside-table)
  (when users
	(let* ((user (caar users))
	       (level (cdar users))
	       (nxt-lvl (+ level 1)))
	  (cond
	   ((string-equal (cadr (gethash user box-table)) "user")
	    (let ((al (gethash user user-arrows))
		  (fl (gethash file file-arrows)))
	      (mapc #'(lambda (p)
			(let ((prev
			       (assoc (cons file p)
				      al :test
				      #'equal))
			      (fprev
			       (assoc (cons user p)
				      fl :test
				      #'equal)))
			  (cond
			   ((not prev)
			    (push (list (cons
					 file p)
					parity level)
				  al)
			    (push (list (cons
					 user p)
					parity level)
				  fl))
			   ((and (< level (third prev))
				 (not
				  (string-equal parity
						(second prev))))
			    ;; replace
			    (setf (second prev) parity)
			    (setf (second fprev) parity)
			    (setf (third prev) level)
			    (setf (third fprev) level)
			    )
			   (T T)
			   )
			  )
			)
		    perms)
	      (setf (gethash user
			     user-arrows) al)
	      (setf (gethash file
			     file-arrows) fl)
	      ))
	   (T
	    (arrow-tbls
	     (append (mapcar #'(lambda (u)
				 (cons u nxt-lvl))
			     (gethash user
				      inside-table)))
	     file perms parity user-arrows file-arrows box-table
	     inside-table)
	    )
	   )
	  (arrow-tbls (cdr users) file perms parity user-arrows file-arrows
		      box-table inside-table)
	  )
	)
  )

(defun fix-prober-arrow-tables (user-arrows file-arrows)
  (maphash #'(lambda (u a)
	       (setf (gethash u user-arrows)
		     (mapcar #'(lambda (l)
				 (list (caar l)
				       (cdar l)
				       (cadr l)))
			     a)))
	   user-arrows)
  (maphash #'(lambda (f a)
	       (setf (gethash f file-arrows)
		     (mapcar #'(lambda (l)
				 (list (caar l)
				       (cdar l)
				       (cadr l)))
			     a)))
	   file-arrows)
  )

(defun do-verify (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (let* ((checking-workbench (null-string (g-value verify-dbox :iff-file)))
	 (iff-file (if checking-workbench
		       (make-temporary-file "/tmp/Miro.iff")
		     (g-value verify-dbox :iff-file)))
	 (tmp-dir (if (null-string (g-value verify-dbox :tmp-dir))
		      "." (g-value verify-dbox :tmp-dir)))
	 (directory (string-right-trim '(#\/) (g-value verify-dbox
						       :directory)))
	 (ambig-user-file (g-value verify-dbox :ambig-user-file))
	 (ambig-perm-file (g-value verify-dbox :ambig-perm-file))
	 (temp-rels-file (null-string (g-value verify-dbox
					       :ambig-rels-file)))
	 (ambig-rels-file (if temp-rels-file
			      (make-temporary-file
			       (format nil "~A/Miro.rels" tmp-dir))
			    (g-value verify-dbox :ambig-rels-file)))
	 (temp-user-file (null-string (g-value verify-dbox
					       :probe-user-file)))
	 (probe-user-file (if temp-user-file
			      (make-temporary-file
			       (format nil "~A/Miro.users" tmp-dir))
			    (g-value verify-dbox :probe-user-file)))
	 (temp-group-file (null-string (g-value verify-dbox
						:probe-group-file)))
	 (probe-group-file (if temp-group-file
			       (make-temporary-file
				(format nil "~A/Miro.groups" tmp-dir))
			     (g-value verify-dbox :probe-group-file)))
	 (temp-perm-file (null-string (g-value verify-dbox
					       :probe-perm-file)))
	 (probe-perm-file (if temp-perm-file
			      (make-temporary-file
			       (format nil "~A/Miro.perms" tmp-dir))
			    (g-value verify-dbox :probe-perm-file)))
	 (temp-log-file (null-string (g-value verify-dbox :log-file)))
	 (log-file (if temp-log-file
		       (make-temporary-file
			(format nil "~A/Miro-vf.log" tmp-dir))
		     (g-value verify-dbox :log-file)))
	 (run-ambig (g-value verify-dbox :run-ambig))
	 (run-probe (g-value verify-dbox :run-probe))
	 (arg-list nil)
	 )

    (s-value verify-update-menu :workbench nil)
    (s-value verify-update-menu :prober-results nil)
    (excl:gc)

    (when checking-workbench
	  (set-help-string "Saving workbench...")
	  (opal:update help-window)
	  (s-value
	   verify-update-menu :workbench
	   (cons
	    (format
	     nil
	     "# (not really generated by the prober -- this is the workbench~%"
	     )
	    (compute-prober-results-from-agg obj-agg))
	   )
	  (apply #'write-iff-from-prober-results
		 (cons iff-file (g-value verify-update-menu :workbench)))
	  )

    (push iff-file arg-list)
    (push directory arg-list)

    (push log-file arg-list)
    (push "-l" arg-list)

    (push probe-perm-file arg-list)
    (push "-pp" arg-list)

    (push probe-group-file arg-list)
    (push "-pg" arg-list)

    (push probe-user-file arg-list)
    (push "-pu" arg-list)

    (push ambig-rels-file arg-list)
    (push "-ar" arg-list)

    (unless (null-string ambig-perm-file)
	    (push ambig-perm-file arg-list)
	    (push "-ap" arg-list))

    (unless (null-string ambig-user-file)
	    (push ambig-user-file arg-list)
	    (push "-au" arg-list))

    (push tmp-dir arg-list)
    (push "-t" arg-list)
    
    (unless run-probe (push "-np" arg-list))
    (unless run-ambig (push "-na" arg-list))
    (push "-v" arg-list)

    (when *verify-debug* (format T "verify arg list: ~S~%" arg-list))

    (set-help-string "Running the verifier...")
    (opal:update help-window)

    (let* ((results (run-miro-tool "verify" arg-list))
	   (stdout (first results))
	   (stderr (second results))
	   )
      (format T (format nil "Output:~%~A" stdout))
      (s-value verify-result-menu :items (get-line-list stderr))
      (s-value verify-update-menu :iff-file (if checking-workbench ""
					      iff-file))
      (s-value verify-update-menu :prober-results
	       (process-prober-output probe-perm-file probe-group-file
				      probe-user-file))
      (when checking-workbench
	    (let* ((prober-results (g-value verify-update-menu
					    :prober-results))
		   (change-list
		    (generate-change-list
		     (g-value verify-result-menu :items)
		     (g-value verify-update-menu :workbench)
		     prober-results))
		   (*eof* :eof)
		   (pr-inside-table (ninth prober-results))
		   (pr-box-table (seventh prober-results))
		   (pr-arrow-table (eighth prober-results))
		   (pr-user-arrows (make-hash-table))
		   (pr-file-arrows (make-hash-table))
		   (user-arrows (make-hash-table))
		   (file-arrows (make-hash-table))
		   )
	      
	      (maphash
	       #'(lambda (sysname alist)
		   (mapc #'(lambda (a)
			     (arrow-tbls
			      (list (cons sysname 0))
			      (second a)
			      (mapcar #'string-downcase (third a))
			      (fourth a)
			      pr-user-arrows pr-file-arrows
			      pr-box-table pr-inside-table))
			 alist))
	       pr-arrow-table)
	      (fix-prober-arrow-tables pr-user-arrows pr-file-arrows)

	      (with-open-file (afile ambig-rels-file)
			      (do ((decl (read afile nil *eof*)
					 (read afile nil *eof*)))
				  ((equal decl *eof*))
				  (let ((type (symbol-name
					       (first decl)))
					(parity (symbol-name
						 (second (fifth decl))))
					(access (string-downcase
						 (symbol-name (second
							       (fourth
								decl)))))
					(user (second decl))
					(file (third decl)))
				    (when (and (equal type "SEM")
					       (or (equal parity "POS")
						   (equal parity "NEG")))
					  (setf (gethash user user-arrows)
						(cons (list file
							    access
							    (equal
							     parity "POS"))
						      (gethash user
							       user-arrows)))
					  (setf (gethash file file-arrows)
						(cons (list user
							    access
							    parity)
						      (gethash file file-arrows)))
					  ))))

	      (s-value verify-update-menu :wb-user-arrows user-arrows)
	      (s-value verify-update-menu :wb-file-arrows file-arrows)
	      (s-value verify-update-menu :pr-user-arrows pr-user-arrows)
	      (s-value verify-update-menu :pr-file-arrows pr-file-arrows)
	      (s-value verify-update-menu :file-box-deletion-list
		       (first change-list))
	      (s-value verify-update-menu :user-box-deletion-list
		       (second change-list))
	      (s-value verify-update-menu :arrow-deletion-list
		       (third change-list))
	      (s-value verify-update-menu :file-box-addition-list
		       (fourth change-list))
	      (s-value verify-update-menu :user-box-addition-list
		       (fifth change-list))
	      (s-value verify-update-menu :arrow-addition-list
		       (sixth change-list))
	      ;;(s-value verify-group-menu :group-addition-list
	      ;;(seventh change-list))
	      ;;(s-value verify-group-menu :group-deletion-list
	      ;;(eighth change-list))
	      (s-value verify-update-menu :perm-change-list
		       (ninth change-list))
	      ))
      )

    (when checking-workbench (delete-file iff-file))
    (when temp-perm-file (delete-file probe-perm-file))
    (when temp-user-file (delete-file probe-user-file))
    (when temp-group-file (delete-file probe-group-file))
    (when temp-log-file (delete-file log-file))
    (when temp-rels-file (delete-file ambig-rels-file))
    (set-help-string "Done.")

    (remove-dbox verify-dbox)
    (call-schema verify-result-menu :display-me)
    (update-command-inactive-list)
    ))

(defun can-view-verify-results ()
  (g-value verify-update-menu :prober-results))

(defun box-creation-list-exists ()
  (or *file-boxes-to-add* *user-boxes-to-add*))

(defun show-current-box (&rest args)
  (declare (ignore args))
  (s-value box-creation-list-dbox :update-menus T)
  (call-schema box-creation-current-box-dbox :display-me)
  )

(defun show-box-creation-list (&rest args)
  (declare (ignore args))
  (call-schema box-creation-list-dbox :display-me)
  )

(defun user-box-list-exists () *user-boxes-to-add*)
(defun flush-user-box-creation-list (obj nv)
  (declare (ignore obj nv))
  (setq *user-boxes-to-add* nil)
  (when (eq *box-creation-list* :user)
	(setq *box-creation-list*
	      (when *file-boxes-to-add* :file))
	)
  (prepare-next-box-creation)
  (update-command-inactive-list)
  )

(defun file-box-list-exists () *file-boxes-to-add*)
(defun flush-file-box-creation-list (obj nv)
  (declare (ignore obj nv))
  (setq *file-boxes-to-add* nil)
  (when (eq *box-creation-list* :file)
	(setq *box-creation-list*
	      (when *user-boxes-to-add* :user))
	)
  (prepare-next-box-creation)
  (update-command-inactive-list)
  )

(defun can-postpone-box-creation () *box-creation-list*)
(defun postpone-box-creation (obj nv)
  (declare (ignore obj nv))
  (setq *box-creation-list* nil)
  (prepare-next-box-creation)
  (update-command-inactive-list))

(defun create-from-user-list (obj nv)
  (declare (ignore obj nv))
  (setq *box-creation-list* :user)
  (prepare-next-box-creation)
  (update-command-inactive-list)
  )

(defun create-from-file-list (obj nv)
  (declare (ignore obj nv))
  (setq *box-creation-list* :file)
  (prepare-next-box-creation)
  (update-command-inactive-list)
  )

(defun prepare-next-box-creation (&optional obj nv)
  (declare (ignore obj))
  (let ((parents nil)
	(children nil)
	)
    (cond
     ((eq *box-creation-list* :user)
      (if *user-boxes-to-add*
	  (let* ((first-user (first *user-boxes-to-add*))
		 (name (first first-user))
		 (type (second first-user))
		 (inside (third first-user))
		 (contains (fourth first-user))
		 )
	    (set-box-creation1-string
	     (format
	      nil
	      "Next box to create: ~S~%    Type=~S"
	      name type))
	    (setq parents inside)
	    (setq children contains)
	    )
	(progn
	  (setq *box-creation-list* (when *file-boxes-to-add* :file))
	  (prepare-next-box-creation)
	  (return-from prepare-next-box-creation)
	  )
	)
      )
     ((eq *box-creation-list* :file)
      (if *file-boxes-to-add*
	  (let* ((first-file (first *file-boxes-to-add*))
		 (name (second first-file))
		 (type (third first-file))
		 (inside (fourth first-file))
		 (contains (fifth first-file))
		 )
	    (set-box-creation1-string
	     (format
	      nil
	      "Next box to create: ~S~%    Type=~S"
	      name type))
	    (setq parents inside)
	    (setq children contains)
	    )
	(progn
	  (setq *box-creation-list* (when *user-boxes-to-add* :user))
	  (prepare-next-box-creation)
	  (return-from prepare-next-box-creation)
	  )
	)
      )
     (T (set-box-creation1-string "No boxes waiting to be created."))
     )

    ;; select parent(s)
    (unless (string-equal nv "Select Children")
	    (dolist (obj (g-value obj-agg :selected))
		    (s-value obj :selected nil))
	    (s-value obj-agg :selected nil)
	    (when (and parents (listp parents))
		  (let* ((parent (find-box (cdr (first parents))))
			 )
		    (when parent
			  ;; do the actual selection
			  (s-value parent :selected T)
			  (s-value obj-agg :selected (list parent))

			  ;; make sure the parent is displayed
			  (call-schema zoom-agg :display-box
				       (g-value parent :box))
			  ))
		  )
	    )

    ;; (in)activate buttons as appropriate
    (if (and parents (listp parents))
	(call-schema box-creation1-button :activate-item
		     "Select Parent(s)")
      (call-schema box-creation1-button :inactivate-item
		   "Select Parent(s)")
      )
    (if (and children (listp children))
	(call-schema box-creation1-button :activate-item
		     "Select Children")
      (call-schema box-creation1-button :inactivate-item
		   "Select Children")
      )

    )
  (opal:update box-creation1-window)
  (opal:update work-window)
  (opal:update sb-window)
  (update-command-inactive-list)
  )

(defun view-verify-results (obj nv)
  (declare (ignore obj nv))
  (block-interference :help-msg "Displaying verify results...")
  (cond
   ((g-value verify-update-menu :displaying-me)
    (call-schema verify-update-menu :display-me))
   ;;((g-value verify-group-menu :displaying-me)
   ;;(call-schema verify-group-menu :display-me))
   (T (call-schema verify-result-menu :display-me))
   ))


;; put files in the file table
(defun add-files-to-table (ftable btable itable parent-name
				  parent-sysname)
  (dolist (child (gethash parent-sysname itable))
	  (let* ((path (concatenate 'string parent-name
				    (third (gethash child
						    btable))))
		 (last (- (length path) 1))
		 )
	    (setf (gethash (if (eq #\/ (elt path last))
			       (subseq path 0 last) path) ftable)
		  child)
	    (add-files-to-table ftable btable itable path child)
	    )))

;; return same thing as process-prober-output without the prober
;; comments
(defun compute-prober-results-from-agg (agg)
  (let ((user-table (make-hash-table :test #'equal))
	(group-table (make-hash-table :test #'equal))
	(file-table (make-hash-table :test #'equal))
	(box-table (make-hash-table))
	(arrow-table (make-hash-table))
	(inside-table (make-hash-table))
	top-user top-file
	)
    ;; walk through the component list and put objects in the
    ;; appropriate hash tables
    (opal:do-components
     agg
     #'(lambda (obj)
	 (case (g-value obj :object-type)
	       ((:miro-box)
		(let ((sysname (g-value obj :sysname))
		      (name (g-value obj :label :string))
		      (type (g-value obj :box-type))
		      (role (g-value obj :box-role))
		      )
		  (setf (gethash sysname box-table)
			(list role type name obj))
		  (cond
		   ((string-equal role "user")
		    (unless top-user
			    (when (string-equal name "world")
				  (setq top-user sysname)))
		    (setf (gethash name (if (string-equal type "user")
					    user-table group-table))
			  sysname)
		    )
		   ((string-equal role "file")
		    (unless top-file
			    (when (string-equal name "/")
				  (setq top-file sysname)))
		    ;; can't do anything else with this until we have
		    ;; insideness information
		    )
		   (T (format T "warning: unexpected role: ~S~%"
			      role))
		   )))
	       ((:miro-arrow)
		(let ((permissions (read-list-or-string
				    (g-value obj :label :string))))
		  (push (list (g-value obj :sysname)
			      (g-value obj :to :sysname)
			      (if (listp permissions) permissions
				(list permissions))
			      (not (g-value obj :neg)))
			(gethash (g-value obj :from :sysname)
				 arrow-table))
		  ))
	       )

	 ))

    ;; compute insideness
    (build-containment-lists agg)
    (maphash
     #'(lambda (sysname info-list)
	 (let* ((obj (fourth info-list))
		(kids (mapcar #'(lambda (o) (g-value o :sysname))
			      (g-value obj :direct-contains)))
		)
	   (when kids
		 (setf (gethash sysname inside-table)
		       (append kids (gethash sysname inside-table)))
		 )
	   ))
     box-table)

    (add-files-to-table file-table box-table inside-table "/" top-file)

    ;; return
    (list top-user top-file user-table group-table file-table
	  box-table arrow-table inside-table)
    ))

;; make a list of changes needed to make the workbench compatible with
;; the prober output
;; return: (file-box-deletion-list user-box-deletion-list
;;          arrow-deletion-list file-box-addition-list
;;          user-box-addition-list arrow-addition-list
;;          group-addition-list group-deletion-list)
(defun generate-change-list (items workbench-results prober-results)
  (let ((wtop-file (third workbench-results))
	(wuser-table (fourth workbench-results))
	(puser-table (fourth prober-results))
	(wgroup-table (fifth workbench-results))
	(pgroup-table (fifth prober-results))
	(wfile-table (sixth workbench-results))
	(pfile-table (sixth prober-results))
	(wbox-table (seventh workbench-results))
	(pbox-table (seventh prober-results))
	(winside-table (ninth workbench-results))
	(pinside-table (ninth prober-results))
	(user-box-deletion-list nil)
	(file-box-deletion-list nil)
	(arrow-deletion-list nil)
	(user-box-addition-list nil)
	(file-box-addition-list nil)
	(arrow-addition-list nil)
	(group-addition-list nil)
	(group-deletion-list nil)
	(perm-change-list nil)
	)
    (dolist (item items)
	    (let ((ilist (read-list-or-string (format nil "(~A)" item)
					      T)))
	      (unless (eq ilist :empty-list)
		      (cond

		       ;; an extra file box
		       ((string-equal (car ilist) "box")
			(let* ((file (second ilist))
			       (slash (position #\/ file :from-end T))
			       (real-file (if (eq slash (- (length file) 1))
					      (subseq file 0 slash)
					    file))
			       (is-dir nil)
			       (loc (find real-file file-box-deletion-list
					  :key #'car :test #'equal))
			       )
			  (unless loc
				  (push (list file
					      (gethash real-file wfile-table)
					      nil)
					file-box-deletion-list))
			  (loop
			   (let* ((slash (position #\/ file :from-end T))
				  (dir (when slash (subseq file 0 slash)))
				  (real-file (if is-dir
						 (format nil "~A/" file)
					       file))
				  (real-dir (format nil "~A/" dir))
				  (sysname (gethash dir wfile-table))
				  )
			     (when (or (not dir) (null-string dir)) (return))
			     (if (gethash dir pfile-table) (return)
			       (let ((loc (find sysname file-box-deletion-list
						:key #'second)))
				 (if loc (pushnew file (third loc)
						  :test #'equal)
				   (push
				    (list real-dir sysname (list real-file))
				    file-box-deletion-list))
				 ))
			     (setq file dir)
			     ))
			  ))

		       ;; a missing file
		       ((string-equal (car ilist) "file")
			(let ((file (second ilist))
			      (is-dir nil)
			      )
			  (loop
			   (let* ((slash (position #\/ file :from-end T))
				  (dir (when slash (subseq file 0 slash)))
				  (sysname (gethash dir wfile-table))
				  (real-file (if is-dir
						 (format nil "~A/" file)
					       file))
				  (loc (find real-file file-box-addition-list
					     :key #'car :test #'equal))
				  (real-dir (format nil "~A/" dir))
				  )
			     (when loc (return))
			     (when (or (not dir) (null-string dir))
				   (push (list real-file
					       (list "/" wtop-file))
					    file-box-addition-list)
				   (return))
			     (when sysname
				   (push
				    (list real-file (cons real-dir sysname))
				    file-box-addition-list)
				   (return))
			     (push (list real-file (list real-dir))
				   file-box-addition-list)
			     (setq file dir)
			     (setq is-dir T)
			     ))
			  ))

		       ;; extra user
		       ((string-equal (first ilist) "miro'")
			(unless (gethash (third ilist) puser-table)
				(push (list (third ilist)
					    (gethash (third ilist)
						     wuser-table))
				      user-box-deletion-list)))

		       ;; missing user
		       ((string-equal (first ilist) "prober")
			(pushnew (third ilist) user-box-addition-list
				 :test #'equal))
		       
		       ;; permissions differ -- could be a missing
		       ;; user or extra/missing arrows
		       ((string-equal (second ilist) "permissions")
			(push (list (fifth ilist)
				    (coerce (butlast (coerce (first ilist)
							     'list))
					    'string)) perm-change-list)
			)
		       (T)
		       ))
	      ))

    ;; figure out group differences
    (let ((wgroups nil)
	  (pgroups nil)
	  )
      (maphash #'(lambda (k v) (push (list k v) wgroups)) wgroup-table)
      (maphash #'(lambda (k v) (push (list k v) pgroups)) pgroup-table)
      (setq wgroups (remove "world" wgroups :key #'car :test #'equal))
      (setq group-addition-list
	    (mapcar
	     #'(lambda (g)
		 (append
		  g
		  (list
		   (mapcar
		    #'(lambda (s)
			;; name
			(third (gethash s pbox-table)))
		    ;; inside list
		    (gethash (second g) pinside-table)))))
	     ;; list of missing groups
	     (set-difference pgroups wgroups :key #'car :test #'equal)))

      (setq group-deletion-list
	    (mapcar
	     #'(lambda (g)
		 (append
		  g
		  (list
		   (mapcar
		    #'(lambda (s)
			;; name
			(third (gethash s wbox-table)))
		    ;; inside list
		    (gethash (second g) winside-table)))))
	     ;; list of extra groups
	     (set-difference wgroups pgroups :key #'car :test #'equal)))
      )

    ;; return
    (list (sort file-box-deletion-list #'string< :key #'car)
	  (sort user-box-deletion-list #'string< :key #'car)
	  arrow-deletion-list
	  (sort file-box-addition-list #'string< :key #'car)
	  (sort user-box-addition-list #'string)
	  arrow-addition-list
	  (sort group-addition-list #'string< :key #'car)
	  (sort group-deletion-list #'string< :key #'car)
	  (sort perm-change-list
		#'(lambda (p1 p2)
		    (let ((u1 (car p1))
			  (u2 (car p2)))
		      (or (string< u1 u2)
			  (and (string= u1 u2)
			       (string< (cadr p1) (cadr p2))))))
		)
	  )
    ))

;;;============================================================
;;;                        PRINT
;;;============================================================
(defparameter *print-debug* nil)

(defun print-workbench (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (block-interference
   :help-msg
   (format
    nil
    "Change the printing
options as needed.
Press ~S to
print, ~S to
abort."
    "Print" "Abort"))
  (call-schema print-dbox :initialize-options)
  )

(defun do-print (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (let* ((printing-workbench (null-string
			      (g-value print-dbox :input-value)))
	 (fname (if printing-workbench
		    (make-temporary-file "/tmp/Miro.iff")
		  (g-value print-dbox :input-value)))
	 (iff2ps-args (g-value print-dbox :iff2ps-args-value))
	 (send-to-printer (eq (g-value print-dbox :output-where) :printer))
	 (pname (when send-to-printer (g-value print-dbox :output-value)))
	 (oname (if send-to-printer
		    (make-temporary-file "/tmp/Miro.ps")
		  (g-value print-dbox :output-value)))
	 (lpr-args
	  (let ((args (g-value print-dbox :lpr-args-value)))
	    (if (not (null-string pname))
		(format nil "~A~A ~A" Miro-Lpr-Printer-Arg pname args)
	      args)))
	 )

    (remove-dbox print-dbox)
    (opal:lower-window dialog-window)
    (when *print-debug* (format t "In print-workbench~%"))
    
    ;; if workbench, save to file /tmp/Miro.iff 
    ;; change to output a stream instead
    (when printing-workbench
	  (set-help-string "Saving workbench...")
	  (opal:update help-window)
	  (with-open-file (file fname :direction :output 
				:if-exists :supersede
				:if-does-not-exist :create)
			  (let ((*package* (find-package "MIRO")))
			    (save-objects file)))
	  )

    ;; convert to .ps file
    (set-help-string "Generating postscript
output...")
    (opal:update help-window)

    (let* ((results (run-miro-tool "iff2ps"
				   (append (get-arg-list iff2ps-args)
					   (list fname))))
	   (stdout (first results))
	   (stderr (second results))
	   )
      (if (null-string oname) (format T "Iff2ps Output:~%~A" stdout)
	(with-open-file (output-stream oname :direction :output
				       :if-exists :supersede
				       :if-does-not-exist :create)
			(format output-stream "~A" stdout))
	)
      (unless (null-string stderr)
	      (format T "Iff2ps Errors:~%~A" stderr))
      (force-output)
      )

    ;; send output to the printer
    (when send-to-printer
	  (let* ((results (run-miro-tool Miro-Lpr
					 (append (get-arg-list
						  lpr-args)
						 Miro-Lpr-Args
						 (list oname))))
		 (stdout (first results))
		 (stderr (second results))
		 )
	    (unless (null-string stdout)
		    (format T "lpr output:~%~A" stdout))
	    (unless (null-string stderr)
		    (format T "lpr errors:~%~A" stderr))
	    (force-output)
	    )
	  )

    ;; delete our temporary file(s)
    (when send-to-printer (delete-file oname))
    (when printing-workbench (delete-file fname))

    (format T "Done printing~%")
    (force-output)
    )
  (allow-interference)
  )


;;;============================================================
;;;                        AMBIG?
;;;============================================================

(defparameter *ambig-debug* nil)
(defun can-ambig () (eq (g-value pictype-menu :value) :instance))
(defun can-view-ambig-results ()
  (g-value ambig-status :have-results))

(defun view-ambig-results (obj item-string)
  (declare (ignore obj item-string))
  (block-interference :help-msg "Displaying ambig results.")
  (call-schema ambig-status :display-results)
  )

(defun check-ambiguity (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (block-interference
   :help-msg
   (format
    nil
    "Change the options
to pass to the
ambiguity checker
as needed.  Press
~S to
run the checker,
~S to abort."
    "Check Ambiguity" "Abort"))
  (call-schema ambig-options :initialize-option-menu)
  )

(defun do-ambiguity-check (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  (let* ((checking-workbench (null-string
			      (g-value ambig-options
				       :filename-value)))
	 (fname (if checking-workbench
		    (make-temporary-file "/tmp/Miro.iff")
		  (g-value ambig-options :filename-value)))
	 (user-args (g-value ambig-options :args-value))
	 )
    (if (probe-file fname)
	(progn
	  (when *ambig-debug* (format t "Checking Ambiguity~%"))
	  ;; if workbench, save to file /tmp/Miro.iff 
	  ;; change to output a stream instead
	  (when checking-workbench
		(set-help-string "Saving workbench...")
		(opal:update help-window)
		(with-open-file (file fname :direction :output 
				      :if-exists :supersede
				      :if-does-not-exist :create)
				(let ((*package* (find-package "MIRO")))
				  (save-objects file)))
		)

	  (set-help-string
	   "Checking ambiguity...")
	  (opal:update help-window)
	  (let* ((results
		  (run-miro-tool
		   "ambig"
		   (append (get-arg-list user-args)
			   (list "-l" "-M" "-a"
				 (namestring (truename (pathname
							fname)))
				 ))))
		 (stdout (first results))
		 (stderr (second results))
		 (exit-status (third results))
		 (*eof* :eof)
		 (ambig-list nil)
		 (negative-list nil)
		 (positive-list nil)
		 )

	    ;; read and handle the results
	    (when checking-workbench (delete-file fname))

	    (with-input-from-string
	     (result-string stdout)
	     (do ((decl (read result-string nil *eof*)
			(read result-string nil *eof*)))
		 ((equal decl *eof*))
		 (let ((str1 (symbol-name (first decl)))
		       (str2 (symbol-name (second (fifth decl)))))
		   (when (equal str1 "SEM")
			 (cond
			  ((equal str2 "AMBIG") (push decl ambig-list))
			  ((equal str2 "NEG") (push decl negative-list))
			  ((equal str2 "POS") (push decl positive-list))
			  ))
		   )
		 ))

	    (s-value ambig-status :workbench
		     (unless checking-workbench fname))
	    (let ((agg (if checking-workbench obj-agg
			 (agg-from-file fname T))))
	      (call-schema ambig-status :init-results
			   ambig-list negative-list positive-list
			   (if (or (null exit-status)
				   (and (/= exit-status 0)
					(/= exit-status 3))) stderr "")
			   agg)
	      (unless (or checking-workbench (not (schema-p agg)))
		      (opal:destroy agg))
	      )
	    )
	  (format T "Done checking~%")
	  (force-output)
	  (remove-dbox ambig-options)
	  (call-schema ambig-status :display-results)
	  (set-help-string "Displaying ambig
results.")
	  (opal:update dialog-window)
	  )
      (progn
	(push-error-msg "File not found.")
	(call-schema ambig-status :clear-results)
	(remove-dbox ambig-options)
	(allow-interference))
      )
    )
  )


;;;============================================================
;;;                        UNDELETE
;;;============================================================

(defun destroy-undo-objects ()
  (when undo-objects
	(push-help-string
	 (format nil "Deleting objects in
the ~S
buffer..."
		 "undelete"))
	(dolist (obj undo-objects)
		(opal::destroy-me obj))
	(setq undo-objects nil)
	(pop-help-string)
	)
  (setq undo-schema-slots nil)
  (setq undo-functions nil)
  )
(defun can-undelete () undo-objects)
(defun undelete (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  ;; unselect previously selected objects
  (dolist (obj (g-value obj-agg :selected)) (s-value obj :selected NIL))

  (let ((arrows nil)
	)
    ;; do the boxes first and keep track of the arrows for later.
    (dolist (obj undo-objects)
	    (opal:add-component obj-agg obj)
	    (case (g-value obj :object-type)
		  (:miro-arrow (push obj arrows))
		  (:miro-box (s-value obj :visible T))
		  ))
    ;; now try to restore the arrows
    (dolist (obj arrows)
	    (let* ((point-list (g-value obj :points))
		   (from-box (find-attached-box
			      obj-agg (first point-list)
			      (second point-list)))
		   (to-box (find-attached-box
			    obj-agg (third point-list)
			    (fourth point-list))))
	      (if (or (null from-box) (null to-box))
		  (push-error-msg "Couldn't restore arrow.")
		(progn
		  (push obj (g-value from-box :from-arrows))
		  (push obj (g-value to-box :to-arrows))
		  ))
	      )
	    (s-value obj :visible T)
	    )
    )

  ;; undo slots
  (dolist (slist undo-schema-slots)
	  (s-value (first slist) (second slist) (third slist)))

  ;; call undo/update functions
  (dolist (fn undo-functions) (funcall fn))

  ;; all the undeleted objects should be selected
  (s-value obj-agg :selected undo-objects)
  (setq undo-objects nil)

  ;; update button visibility: copy, delete, display, unselect,
  ;; undelete
  (update-command-inactive-list)

  ;; we may have invalidated ambig info
  (when (g-value ambig-status :guaranteed-valid)
	(s-value ambig-status :guaranteed-valid nil))

  (opal:update menu-window)

  ;; update the bounding box
  (s-value pic-sp :bb-box
	   (find-bounding-box (get-values obj-agg :components) T))

  (mark-as-changed work-window :aggregate)
  (opal:update work-window)
  )

;;;============================================================
;;; constraint checking
;;;============================================================

;; get the write date of the fasl files in the ipql directory.  we
;; only care about the most recently written one.
(defun ipql-write-date ()
  (let ((dir (directory (make-pathname :directory *ipql-dir*
				       :type "fasl" :name :wild))))
    (when dir (apply #'max (mapcar #'file-write-date dir)))))

(defvar *ipql-names* '("-box" "-syn" "-con" "-sem" ""))

;; compile the lisp files for an instance picture
(defun compile-instance-lisp (instance dir)
  (let ((ipql-date (ipql-write-date)))
    (dolist (cl (mapcar #'(lambda (f)
			    (make-pathname :directory dir
					   :name (format nil "~A~A"
							 instance f)
					   :type "cl"))
			*ipql-names*))
	    (let* ((fasl (make-pathname :directory (pathname-directory
						    (truename cl))
					:name (pathname-name cl)
					:type "fasl"))
		   (fasl-write-date (file-write-date fasl))
		   (cl-write-date (file-write-date cl))
		   )
	      (unless (and fasl-write-date
			   (< cl-write-date fasl-write-date)
			   (< ipql-date fasl-write-date))
		      (set-help-string "Compiling...")
		      (opal:update help-window)
		      (let ((*package*
			     (or (find-package "MIRO-CONSTRAINTS")
				 (make-package "MIRO-CONSTRAINTS"))))
			(with-directory dir *ipql-dir*
					#'(lambda ()
					    (proclaim '(optimize
							(speed 3)
							(safety 1)))
					    (compile-file cl :output
							  fasl)))
			))
	      ))))

;; compile the lisp file for a constraint picture
(defun compile-constraint-lisp (constraint dir)
  (let* ((ipql-date (ipql-write-date))
	 (cl (make-pathname :directory dir :name constraint :type
			    "cl"))
	 (fasl (make-pathname :directory dir :name constraint :type
			      "fasl"))
	 (cl-write-date (file-write-date cl))
	 (fasl-write-date (file-write-date fasl))
	 )
    (unless (and fasl-write-date
		 (< cl-write-date fasl-write-date)
		 (< ipql-date fasl-write-date))
	    (set-help-string "Compiling...")
	    (opal:update help-window)
	    (let ((*package*
		   (or (find-package "MIRO-CONSTRAINTS")
		       (make-package "MIRO-CONSTRAINTS"))))
	      (with-directory dir *ipql-dir*
			      #'(lambda ()
				  (proclaim '(optimize (speed 3)
						       (safety 1)))
				  (compile-file cl :output fasl)))
	      ))
    ))

;; run iff2ipql and compile the resulting lisp files
(defun run-iff2ipql (instance dir &optional force)
  (let* ((instance-file (pathname instance))
	 (instance-date (file-write-date instance-file))
	 (cl-files (mapcar #'(lambda (f)
			       (format nil "~A/~A~A.cl"
				       dir (pathname-name
					    instance-file) f))
			   *ipql-names*))
	 (cl-dates (mapcar #'file-write-date cl-files))
	 (need-to-run (or force (position nil cl-dates)
			  (> instance-date (apply #'min cl-dates))))
	 (results nil)
	 )
    (when need-to-run
	  (set-help-string "Running iff2ipql...")
	  (opal:update help-window)
	  (format T "Running iff2ipql...~%")
	  (force-output)
	  (setq results (run-miro-tool "iff2ipql"
				       (list "-f" "-q" instance dir)))
	  (format T "Finished -- exit status is ~S~%" (third results))
	  (force-output)
	  )
    (if (or (not results) (= (third results) 0))
	(compile-instance-lisp (pathname-name instance-file) dir)
      (mapc #'(lambda (f)
		(when (probe-file f) (delete-file f)))
	    cl-files))

    ;; return
    results))

;; run cpc and compile the resulting lisp files
(defun run-cpc (constraint dir &optional force)
  (let* ((constraint-file (pathname constraint))
	 (constraint-date (file-write-date constraint-file))
	 (cl-file (make-pathname :directory dir
				 :name (pathname-name constraint-file)
				 :type "cl"))
	 (cl-date (file-write-date cl-file))
	 (need-to-run (or force (null cl-date)
			  (> constraint-date cl-date)))
	 (results nil)
	 )
    (when need-to-run
	  (set-help-string "Running cpc...")
	  (opal:update help-window)
	  (format T "Running cpc...~%")
	  (force-output)
	  (setq results (run-miro-tool "cpc"
				       (list "-o" (namestring cl-file)
					     constraint)))
	  (format T "Finished -- exit status is ~S~%" (third results))
	  (force-output)
	  )
    (if (or (not results) (= (third results) 0))
	(compile-constraint-lisp (pathname-name constraint-file)
				 dir)
      (when (probe-file cl-file) (delete-file cl-file)))

    ;; return
    results))

;; check an instance picture against a constraint
(defun constraint-check
  (instance instance-fasl-dir constraint constraint-fasl-dir
	    &optional force)
  (let* ((*package* (or (find-package "MIRO-CONSTRAINTS")
			(make-package "MIRO-CONSTRAINTS")))
	 (instance-file (pathname instance))
	 (instance-fasl
	  (make-pathname :directory instance-fasl-dir
			 :name (pathname-name instance-file)
			 :type "fasl"))
	 (constraint-file (pathname constraint))
	 (constraint-fasl
	  (make-pathname :directory constraint-fasl-dir
			 :name (pathname-name constraint-file)
			 :type "fasl"))
	 (con-result nil)
	 (iff2ipql-results (run-iff2ipql instance instance-fasl-dir
					 force))
	 (iff2ipql-ok (or (not iff2ipql-results)
			  (= (third iff2ipql-results) 0)))
	 (cpc-results (run-cpc constraint constraint-fasl-dir
			       force))
	 (cpc-ok (or (not cpc-results)
		     (= (third cpc-results) 0)))
	 (con-output
	  (when (and iff2ipql-ok cpc-ok)
		(set-help-string "Checking constraint...")
		(opal:update help-window)
		(with-output-to-string
		 (results)
		 (let ((*standard-output* results))
		   (with-directory constraint-fasl-dir *ipql-dir*
				   #'(lambda ()
				       (load constraint-fasl)))
		   (with-directory
		    instance-fasl-dir *ipql-dir*
		    #'(lambda ()
			(with-input-from-string
			 (s (format nil "(progn (setq ipql:*out-stream* *standard-output*) (constraints::~A ~S) (finish-output))"
				    (pathname-name constraint-fasl)
				    (namestring instance-fasl)))
			 (setq con-result (eval (read s)))
			 )))
		   ))))
	 )
    (when (not iff2ipql-ok) (format T "iff2ipql failed!~%"))
    (when (not cpc-ok) (format T "cpc failed!~%"))
    (force-output)
    (list iff2ipql-results cpc-results con-result
	  (when con-output
		(remove-if #'(lambda (l)
			       (or (null-string l)
				   (eq (char l 0) #\;)))
			   (get-line-list con-output))))))

(defun do-constraint-check (obj nv)
  (declare (ignore obj))
  (cond
   ((string-equal nv "Abort")
    (progn
      (inter:beep)
      (call-schema constraint-check-options :undisplay-me)))
   ((not (file-write-date *ipql-dir*))
    (progn
      (format T "~%Couldn't find ipql directory '~A'!!!~%" *ipql-dir*)
      (force-output)
      (push-error-msg "Couldn't find ipql
directory!!!")
      (call-schema constraint-check-options :undisplay-me)
      ))
   (T
    (let* ((files-to-delete nil)
	   (cfile (g-value constraint-check-options :constraint-file))
	   (cdir (g-value constraint-check-options
			  :constraint-comp-dir))
	   (temp-constraint-file (and (eq (g-value pictype-menu
						   :value) :constraint)
				      (or (not cfile) (null-string
						       cfile))))
	   (constraint-file
	    (if temp-constraint-file
		(let* ((fname (make-temporary-file
			       (format nil "~A/Constraint"
				       (if (or (not cdir)
					       (null-string cdir))
					   "/tmp" cdir)))))
		  (push fname files-to-delete)
		  (set-help-string "Saving workbench...")
		  (opal:update help-window)
		  (with-open-file (file fname :direction :output
					:if-exists :supersede
					:if-does-not-exist :create)
				  (let ((*package* (find-package
						    "MIRO")))
				    (save-objects file)))
		  fname)
	      cfile))
	   (constraint-dir
	    (let* ((dir
		    (if (or (not cdir) (null-string cdir))
			(namestring
			 (make-pathname
			  :directory (pathname-directory
				      (truename constraint-file))))
		      cdir))
		   (basename (when temp-constraint-file
				   (pathname-name (pathname
						   constraint-file))))
		   (cl-file (when temp-constraint-file
				  (format nil "~A/~A.cl" dir basename)))
		   (fasl-file (when temp-constraint-file
				    (format nil "~A/~A.fasl" dir basename)))
		   )
	      (when temp-constraint-file
		    (push cl-file files-to-delete)
		    (push fasl-file files-to-delete)
		    )
	      dir))
	   (ifile (g-value constraint-check-options :instance-file))
	   (idir (g-value constraint-check-options
			  :instance-comp-dir))
	   (temp-instance-file (and (eq (g-value pictype-menu :value)
					:instance)
				    (or (not ifile) (null-string ifile))))
	   (instance-file
	    (if temp-instance-file
		(let* ((fname (make-temporary-file
			       (format nil "~A/Instance"
				       (if (or (not idir)
					       (null-string idir))
					   "/tmp" idir)))))
		  (push fname files-to-delete)
		  (set-help-string "Saving workbench...")
		  (opal:update help-window)
		  (with-open-file (file fname :direction :output
					:if-exists :supersede
					:if-does-not-exist :create)
				  (let ((*package* (find-package
						    "MIRO")))
				    (save-objects file)))
		  fname)
	      ifile))
	   (instance-dir
	    (let* ((dir
		    (if (or (not idir) (null-string idir))
			(namestring
			 (make-pathname
			  :directory (pathname-directory
				      (truename instance-file))))
		      idir))
		   (basename (when temp-instance-file
				   (pathname-name (pathname instance-file))))
		   (cl-files (when temp-instance-file
				   (mapcar #'(lambda (s)
					       (format nil "~A/~A~A.cl"
						       dir basename
						       s))
					   *ipql-names*)))
		   (fasl-files (when temp-instance-file
				     (mapcar #'(lambda (s)
						 (format nil "~A/~A~A.cl"
							 dir basename
							 s))
					     *ipql-names*)))
		   )
	      (when temp-instance-file
		    (setq files-to-delete (append files-to-delete
						  cl-files fasl-files)))
	      dir))
	   (force (g-value constraint-check-options
			   :force-compile))
	   (do-everything
	    (string-equal nv "Run the Constraint Checker"))
	   (do-cpc
	    (string-equal nv "Compile Constraint Only"))
	   (do-iff2ipql
	    (string-equal nv "Compile Instance Only"))
	   (iff2ipql-output nil)
	   (cpc-output nil)
	   (con-output nil)
	   (output-list nil)
	   )
      (cond
       ((and (eq (g-value pictype-menu :value) :instance)
	     (or (not constraint-file) (null-string constraint-file)))
	(push-error-msg "Constraint file not specified.")
	(format T "Constraint file not specified.")
	(force-output))
       ((and (eq (g-value pictype-menu :value) :constraint)
	     (or (not instance-file) (null-string instance-file)))
	(push-error-msg "Instance file not specified.")
	(format T "Instance file not specified.")
	(force-output))
       (do-iff2ipql
	(setq iff2ipql-output (run-iff2ipql instance-file instance-dir
					    force)))
       (do-cpc
	(setq cpc-output (run-cpc constraint-file constraint-dir
				  force)))
       (do-everything
	(let ((results
	       (constraint-check instance-file instance-dir
				 constraint-file constraint-dir
				 force)))
	  (setq iff2ipql-output (first results))
	  (setq cpc-output (second results))
	  (setq con-output (fourth results))
	  ))
       )

      ;; format the results
      (when iff2ipql-output
	    (setq output-list
		  (append
		   output-list
		   (when output-list '(""))
		   (list
		    (format nil "iff2ipql exited with status ~S"
			    (third iff2ipql-output)))
		   (unless (null-string (first iff2ipql-output))
			   (append
			    (list "" "iff2ipql standard output:")
			    (get-line-list (first
					    iff2ipql-output))))
		   (unless (null-string (second iff2ipql-output))
			   (append
			    (list "" "iff2ipql error output:")
			    (get-line-list (second
					    iff2ipql-output))))
		   )))
      (when cpc-output
	    (setq output-list
		  (append
		   output-list
		   (when output-list '(""))
		   (list
		    (format nil "cpc exited with status ~S"
			    (third cpc-output)))
		   (unless (null-string (first cpc-output))
			   (append
			    (list "" "cpc standard output:")
			    (get-line-list (first cpc-output))))
		   (unless (null-string (second cpc-output))
			   (append
			    (list "" "cpc error output:")
			    (get-line-list (second cpc-output))))
		   )))
      (when con-output
	    (setq output-list
		  (append output-list
			  (when output-list '(""))
			  (list "constraint-checker output:")
			  con-output)))
      (s-value constraint-check-results :result-list output-list)
      (dolist (f files-to-delete)
	      (when (probe-file f) (delete-file f)))
      (update-command-inactive-list)
      (if output-list
	  (progn
	    (set-help-string "Showing constraint
check results...")
	    (remove-dbox constraint-check-options)
	    (call-schema constraint-check-results :display-me))
	(call-schema constraint-check-options :undisplay-me))
      ))))

(defun check-legality (obj item)
  (declare (ignore obj item))
  (call-schema constraint-check-options :display-me)
  )

(defun can-show-constraint-results ()
  (not (null (g-value constraint-check-results :result-list))))

(defun show-constraint-results (obj item)
  (declare (ignore obj item))
  (block-interference
   :help-msg "Showing constraint
check results...")
  (call-schema constraint-check-results :display-me)
  )

;;; 
;;;  NOT IMPLEMENTED YET
;;; 

;; can't hide anything yet
(defun can-hide () nil)
(defun hide-selected-objects (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  )

;; can't unhide anything yet
(defun can-unhide () nil)
(defun unhide-selected-objects (gadgets-object item-string)
  (declare (ignore gadgets-object item-string))
  )

;;;============================================================
;;; HELPER FUNCTIONS
;;;============================================================

;;;------------------------------------------------------------
;;; get file name from user
;;;------------------------------------------------------------
(defun get-file-name ()
  (get-string-from-user "Please enter the file name
     [return for slave]: ")
  )

;;;------------------------------------------------------------ 
;;; returns T if string is null, NIL otherwise
;;;------------------------------------------------------------
(defun null-string (astring)
  (string= astring ""))

