;;;             -*- Mode: Lisp; Package: MIRO; -*-
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

;;; 
;;; MIRO EDITOR - FUNCTIONS FOR WORKBENCH INTERACTORS
;;;
;;; This file contains the code for the interactors used in the
;;; workbench area. 
;;; Top-level functions are
;;; -  (create-workbench-inters)  
;;;
#|
============================================================
Change log:
    11/04/91 ky ; Numerous changes -- use new menu objects, popup menu.
    02/20/91 ky ; Ignore objects with no size/location in
		; find-bounding-box.
    12/4/90  ky ; Added dialog-window to all window lists.
    12/3/90  ky ; No more dbox placement interactors.
    11/20/90 ky ; Don't change the color of work-window's background
		; if the user is editing a label.  Change the color of
		; work-window's background when the user is done
		; placing a dialog box.
    11/1/90  ky ; Make sure dialog box gets displayed.
    9/24/90 ky  ; Moved some information from ambig-menu to
		; ambig-status in an attempt to avoid display errors.
    9/21/90 ky  ; Warn the user about possibly invalid ambig results
		; when the picture is changed.
    9/20/90 ky  ; Find-bounding-box returns a list of 0's instead of
		; nil if there are no components.
    9/19/90 ky  ; Added help-inter, help-priority-level.
    9/18/90 ky  ; Get rid of undo-objects if we have done something
		; that might invalidate them (copy, move, resize, box
		; or arrow creation).
		;
		; Fix visibility of undelete when objects are
		; created/resized/moved.
		;
    9/17/90 ky  ; Fix visibility of copy, delete, display, and
		; unselect buttons when new objects are
		; selected/unselected.
    9/13/90 ky  ; A few changes to get rid of compile-time warnings.
    9/12/90 ky  ; Use :draw-function :xor and fast redraw objects.
		; Garnet seems to handle these things properly now,
		; and feedback is much faster this way :-)
		;
		; Maintain a bounding box to be used with scaling and
		; the scrollpad.
		;
		; Take scaling into account when computing the
		; minimum size of a box.
		;
		; Don't use a minimum size for sweep-select.
		;
		; Don't let Garnet handle resize-inter's feedback any
		; more; Garnet doesn't do it right (for us).
		;
		; Keep *box-db* and *arrow-db* "visible" all the time
		; so that the first display doesn't take so long.
		;
		; Try to keep the center of the work-window inside the
		; picture when scaling.
		;
    8/23/90 ky  ; A few changes to work with the "test" Garnet.
		; Added optional argument "dont-scale" to
		; find-bounding-box.
		;
    8/10/90 ky  ; Use :box (for boxes) or :points (for arrows) to
		; get/set size and position information, since slots
		; like :height and :width may contain bogus values if
		; the object is not visible for some reason.
		;
		; Don't allow the user to move/copy things outside the
		; work area.
		;
    7/31/90 ky  ; Added sb-window to the list of windows for
		; abort-inter, redraw-inter, error-inter.
		;
		; Changed help messages to conform to the new size of
		; the help window.
		;
		; Replaced hor-sb, vert-sb with pic-sp, a scrollpad.
		;
    7/23/90 ky  ; redraw-inter updates a new window, sb-window.
		;
		; *box-db* and *arrow-db* are handled differently now.
		; Dialog boxes now have another component, ":shield",
		; which makes the box invisible while allowing it to
		; be "visible" from Garnet's point of view so that
		; :height and :width will have meaningful values.
		; Just before the interactor for placing a dialog box
		; is activated, the dialog box is added to miro-agg
		; with its "shield" turned on.  The interactor is
		; responsible for removing the dialog box from
		; miro-agg, and turning its shield off and adding it
		; to work-agg, if appropriate.  The dialog box is
		; responsible for removing itself from work-agg when
		; it exits.
		;
		; The function move-selected-boxes calls
		; translate-points so that scaling information will be
		; used when determining where those boxes are *really*
		; supposed to go.
		;
		; The function translate-points uses scaling information.
		;
    7/3/90  ky	; Added :min-width, :min-height to redraw-inter.
    7/2/90  ky	; Don't use :xor as a :draw-function.  This causes
		; visibility problems in some cases.
    6/25/90 ky  ; Use new functions "block-interference" and
		; "allow-interference" to change button colors, turn
		; off interactors, etc.
                ;
                ; Deleted ":label-string", which didn't seem to update
		; or be updated by ":label :string" in all cases where
		; an update should have occurred.
    6/1/90  ky  ; Added redraw-inter to redraw all the windows.
    6/1/90  ky  ; Make *box-db* and *arrow-db* visible/invisible by
		; changing the value of :visible, rather than by
		; adding/removing them from work-agg.
                ;
                ; Replaced :interfill with :buttonfill.
    5/8/90  ky  Added error-priority-level at a higher priority than
                highest-priority-level.  error-priority-level is used
                by error-inter, which calls pop-error-msg the next
                time the user does something.  All actions are passed
                on to the other priority levels.  Added function
                create-error-inter.  Added calls to create-error-inter
                and make-y-n-buttons to create-workbench-inters.
                Combined abort-inter's into one interactor with a list
                of windows in the :window slot.

                In text-inter-2:
                  - Changed :abort-action to :how-to-abort.
                  - Call the default :start-action before aborting if
                    the number of objects selected is not one.  The
                    interactor behaves strangely (clears the label of
                    one of the selected objects) if this is not done.
    5/1/90  ky  Added :outside-action and :back-inside-action to
                draw-obj-inter, move-inter, resize-inter, copy-inter,
                box-dbox-inter,  and arrow-dbox-inter.
    4/30/90 ky  Use call-prototype-method in interactors instead of
                using the actual names of default actions.
    4/28/90 ky  Turn off selection visibility in copy-inter's, move-inter's,
                and resize-inter's :stop-action, not :start-action.
                We don't want to leave anything "half-selected" if the user decides to
                abort after pressing (but not releasing) a mouse button.  Changed
                draw-obj-inter, move-inter and resize-inter to allow
                themselves to be aborted with ^G. 
    4/26/90 ky  get-text-inter and text-inter-2 now prevent other
                commands from interfering with them.  These
                interactors also set the help string.
    4/26/90 ky	Changed macros set-abort-inter, clear-abort-inter to functions,
		since things don't seem to compile properly if they are macros.
    4/25/90 ky	Added an interactor to each window at the highest priority to
		trap ^G's and abort the appropriate interactor, if any.  The
		function create-abort-inter creates these interactors.  The
		variable *inter-to-abort* specifies the interactor to abort.
		The macro set-abort-inter sets this variable; clear-abort-inter
		clears it.  An interactor's :how-to-abort slot may contain a
		function to be called when that interactor is aborted.

		get-text-inter and text-inter-2 call set-abort-inter in their
		:start-action, clear-abort-inter in their :stop-action.  These
		interactors no longer need an :abort-event, since this is
		handled by the abort interactors.

		copy-inter, dbox-inter, and arrow-dbox-inter now specify
		:how-to-abort to make sure that the color of the text buttons,
		which is changed when they are activated, gets set back to
		normal and to turn the work-window interactors back on.  These
		interactors call clear-abort-inter in their :stop-action.

		Added three new priority levels: low-priority-level and
		medium-priority-level, which are below normal-priority-level;
		and highest-priority-level, which is above
		running-priority-level.  The first two are used for the
		interactors that are normally active in work-window; this
		allows those interactors to be easily turned off.  The third is
		used for the abort interactors, which need to be able to grab
		^G's before anyone else sees them.  These priority levels are
		created in the function create-workbench-inters.

		dbox-inter and arrow-dbox-inter now give the user some
		instruction on how to use the dialog box; hopefully this will
		prevent panic when the user discovers that the normal
		work-window interactors have been turned off.

		copy-inter sets the color of the text buttons in commands-menu
		(:interfill) to white and clears *dont-interfere* in both
		:how-to-abort and :stop-action.  dbox-inter and
		arrow-dbox-inter do this only in :how-to-abort.

    4/18/90 ky	Set :running-priority and :waiting-priority to
		inter:high-priority-level in box-dbox-inter and
		arrow-dbox-inter.
    4/17/90 ky	Set :running-priority to inter:high-priority-level in
		get-text-inter, text-inter-2, move-inter, and resize-inter
		to be consistent with :waiting-priority.  Set
		:running-priority and :waiting-priority to
		inter:high-priority-level in copy-inter.
    4/10/90 amz added resetting help prompt in copy and dbox inters
    2/9/90 amz added copy-inter
    2/6/90 amz added new text interactor to edit string of obj that
               mouse is over
    2/1/90 amz changed file name to miro-inters. 
               extracted feedback obj creation to separate function
               added sweep-select-inter
    1/24/90 amz changed resize-inter to run only if single box
                selected. 
    1/22/90 amz changed move-resize-inter to separate inter's
                changed move-inter to move multiple objects.
    1/19/90 amz changed to use object's feedback for selection. 
                removed sel-box-feedback and sel-arrow-feedback.
    1/16/90 amz changed to use feedback-agg and obj-agg
    1/4/90 amz  changed to use gadgets for menus 
============================================================
|#

(in-package "MIRO" :use `("LISP" "KR"))

(proclaim '(function destroy-undo-objects)) ; defined in miro-cmnds.lisp

;; find out exactly why this is needed
;;(proclaim '(special miro-box))


;;;============================================================
;;; DRAWING NEW OBJECTS
;;;============================================================
;;;------------------------------------------------------------
;;; Draw-New-Object actually creates an object on the workbench,
;;; according to the values in the menu. This function is called by
;;; draw-obj-inter.  
;;;
;;;------------------------------------------------------------
(defun draw-new-object (inter point-list)
  (declare (ignore inter))
  (let ((mode (g-value tool-menu :menu-items :selected))
	(pictype (g-value pictype-menu :value))
	(parity (g-value arrow-menu :buttons :value))
	(thickness (g-value constraint-menu :thickness-buttons
			    :value))
	;; have to use :value-obj instead of :value because :value
	;; never goes to nil
	(starred (g-value constraint-menu :starred-button :value)))
    (when *test-debug* (if (eq thickness :thick)
			   (format t "Thick selected~%")
			 (format t "Thin selected~%")))
    (cond ((null mode) NIL)
	  ;; draw a box
	  ((eq (g-value mode :object-type) :miro-box)
	   (create-miro-box (translate-points point-list) 
			    (and (eq pictype :constraint) (eq thickness
							      :thick))
			    (and (eq pictype :constraint) starred)
			    obj-agg get-text-inter work-window)
	   )
	  ;; draw an arrow
	  ((eq (g-value mode :object-type) :miro-arrow)
	   (draw-arrow-between-boxes 
	    (let ((p1 (translate-points
		       (list (first point-list) (second point-list) 0 0)))
		  (p2 (translate-points
		       (list (third point-list) (fourth point-list) 0 0)))
		  )
	      (list (first p1) (second p1) (first p2) (second p2)))
					; parity (:neg slot)
	    (eq parity :negative)
					; thick iff constraint and :thick
	    (and (eq thickness :thick) (eq pictype :constraint))
					; arrow type :syn for instance pictures, :arrow-type otherwise
	    (if (eq pictype :instance) 
		:syn 
	      (g-value mode :arrow-type))
					; starred - T only if containment arrow and starred is selected
	    (if (and (equal (g-value mode :arrow-type) :con) 
		     starred)
		T
					; otherwise, not starred
	      NIL)
	    obj-agg get-text-inter work-window
	    ))
	  (T NIL))
    ;; update the bounding box
    (s-value pic-sp :bb-box
	     (find-bounding-box (get-values obj-agg :components) T))
    ;; --- undo-objects deleted in individual routines, if necessary ---
    ;; update button visibility: undelete
    (update-command-inactive-list)
    (opal:update menu-window)
    )) 
;; end Draw-New-Object


;;;============================================================
;;; FEEDBACK OBJECTS
;;;============================================================

;;;------------------------------------------------------------
;;; Create-Feedback-Objects creates all of the feedback objects used
;;; by all of the interactors.
;;;------------------------------------------------------------
(defun create-feedback-objects ()
  ;; the feeback of the interactor for drawing boxes
  (create-instance 'dash-line-feedback opal:rectangle
		   (:draw-function :xor)
		   (:fast-redraw-p t)
		   (:left (o-formula (first (gvl :box))))
		   (:top (o-formula (second (gvl :box))))
		   (:width (o-formula (third (gvl :box))))
		   (:height (o-formula (fourth (gvl :box))))
		   (:visible NIL)
		   (:box '(0 0 0 0))
		   (:line-style
		    (o-formula (gv *fonts-and-styles*
				   :feedback-dash))))
  (opal:add-component feedback-agg dash-line-feedback)

  ;; the feeback of the interactor for drawing arrows
  (create-instance 'line-feedback opal:line
		   (:draw-function :xor)
		   (:fast-redraw-p t)
		   (:x1 (o-formula (first (gvl :points))))
		   (:y1 (o-formula (second (gvl :points))))
		   (:x2 (o-formula (third (gvl :points))))
		   (:y2 (o-formula (fourth (gvl :points))))
		   (:visible NIL)  
		   (:points '(0 0 0 0))
		   (:line-style
		    (o-formula (gv *fonts-and-styles*
				   :feedback-dash))))
  (opal:add-component feedback-agg line-feedback)

  ;; the feedback of the interactor to move objects
  ;; the :box slot will hold the coords of a bounding box of all
  ;; selected objects. This will be set by the start-action of the
  ;; interactor. 
  (create-instance 'move-feedback opal:rectangle
		   (:visible NIL)	; this will be changed by inter
		   (:draw-function :xor)
		   (:fast-redraw-p t)
		   (:filling-style NIL)
		   (:left (o-formula (first (gvl :box))))
		   (:top (o-formula (second (gvl :box))))
		   (:width (o-formula (third (gvl :box))))
		   (:height (o-formula (fourth (gvl :box))))
		   (:line-style
		    (o-formula (gv *fonts-and-styles*
				   :feedback-dash))))

  (opal:add-component feedback-agg move-feedback)
  
  ;; the feedback of the interactor to resize boxes
  (create-instance 'resize-box-feedback opal:rectangle
		   (:draw-function :xor)
		   (:fast-redraw-p t)
		   (:min-width (o-formula (round (* (gv zoom-agg :scale)
						    *minimum-box-width*))))
		   (:min-height (o-formula (round (* (gv zoom-agg :scale)
						     *minimum-box-height*))))
		   (:left (o-formula (first (gvl :box))))
		   (:top (o-formula (second (gvl :box))))
		   (:width (o-formula (max (third (gvl :box)) (gvl :min-width))))
		   (:height (o-formula (max (fourth (gvl :box)) (gvl :min-height))))
		   (:visible nil)
		   (:box '(0 0 0 0))
		   (:line-style
		    (o-formula (gv *fonts-and-styles*
				   :feedback-dash)))
		   (:filling-style NIL))
  (opal:add-component feedback-agg resize-box-feedback)

  ;; the feedback of the interactor to move arrows
  ;; this doesn't do anything yet.
  (create-instance 'resize-arrow-feedback opal:line
		   (:draw-function :xor)
		   (:fast-redraw-p t)
		   (:left (o-formula (first (gvl :points))))
		   (:top (o-formula (second (gvl :points))))
		   (:width (o-formula (third (gvl :points))))
		   (:height (o-formula (fourth (gvl :points))))
		   (:visible (o-formula (gvl :obj-over)))
		   (:obj-over NIL)
		   (:points '(0 0 0 0))
		   (:line-style
		    (o-formula (gv *fonts-and-styles*
				   :feedback-dash))))
  (opal:add-component feedback-agg resize-arrow-feedback)

  )


;;;============================================================
;;; WORKBENCH INTERACTORS
;;;============================================================

;;;------------------------------------------------------------
;;; Create-Workbench-Inters calls all the interactor creation
;;; functions. 
;;;------------------------------------------------------------

(defun create-priorities ()
  ;; define some new priority levels

  ;; low-priority-level -- used by draw-obj-inter, select-obj-inter,
  ;;                       text-inter-2, sweep-select-inter, move-inter,
  ;;                       resize-inter
  (create-instance 'low-priority-level inter:priority-level)

  ;; medium-priority-level -- used by get-text-inter
  (create-instance 'medium-priority-level inter:priority-level)

  ;; highest-priority-level -- used by abort-inter
  (create-instance 'highest-priority-level inter:priority-level
		   (:stop-when :if-any))

  ;; error-priority-level -- used to clear error messages from
  ;; help-window.  Higher priority than highest-priority-level, but
  ;; passes everything through...
  (create-instance 'error-priority-level inter:priority-level
		   (:stop-when nil))
  
  ;; help-priority-level -- used to clear the help message
  (create-instance 'help-priority-level inter:priority-level
		   (:stop-when :if-any))

  (s-value inter:running-priority-level :stop-when nil)
  (s-value inter:normal-priority-level :stop-when :if-any)

  (setf inter:priority-level-list
	(list error-priority-level highest-priority-level
	      help-priority-level inter:high-priority-level
	      inter:running-priority-level inter:normal-priority-level
	      medium-priority-level low-priority-level))
  )

(defun create-workbench-inters ()
  ;; create the interactors
  (create-feedback-objects)
  (create-draw-inter)
  (create-text-inters)
  (create-selection-inter)
  (create-sweep-inter)
  (create-move-inter)
  (create-resize-inter)
  (create-copy-inter)
  (create-abort-inter)
  (create-redraw-inter)
  (create-error-inter)
  (create-help-inter)
  (create-popup-inter)
  (make-y-n-buttons)
  )


;;;------------------------------------------------------------
;;; Create-Abort-Inter creates an interactor for each window that
;;; traps ^G's and aborts the appropriate interactor.
;;;
;;; Set-Abort-Inter sets the interactor to be aborted (*inter-to-abort*).
;;;
;;; Clear-Abort-Inter sets this variable to nil.
;;;------------------------------------------------------------

(defvar *inter-to-abort* nil)
(setq *all-windows* nil)

(defun set-abort-inter (interactor) (setf *inter-to-abort* interactor))

(defun clear-abort-inter () (setf *inter-to-abort* nil))

(defun create-abort-inter ()
  (create-instance 'abort-inter inter:Button-Interactor
		   (:window *all-windows*)
		   (:waiting-priority highest-priority-level)
		   (:running-priority highest-priority-level)
		   (:start-event '(:control-g :control-\g))
		   (:start-where T)
		   (:continuous nil)
		   (:active T)
		   (:stop-action
		    #'(lambda (&rest args)
			      (declare (ignore args))
			      (unless (null *inter-to-abort*)
				      (let ((abort-fn (g-value *inter-to-abort*
							       :how-to-abort)))
					   (inter:beep)
					   (inter:Abort-Interactor
					    *inter-to-abort*)
					   (unless
					    (null abort-fn)
					    (funcall abort-fn))
					   (clear-abort-inter)
					   ))
			      ))
		   )
  (clear-abort-inter)
  )

(defun create-redraw-inter ()
  (create-instance 'redraw-inter inter:Button-Interactor
		   (:window *all-windows*)
		   (:waiting-priority highest-priority-level)
		   (:running-priority highest-priority-level)
		   (:start-event '(:control-l :control-\l))
		   (:start-where T)
		   (:continuous nil)
		   (:active T)
		   (:redraw
		    #'(lambda (&rest args)
			(declare (ignore args))
			(dolist (w *all-windows*)
				(mark-as-changed w :aggregate)
				(opal:update w t)
				)
			))
		   (:stop-action (o-formula (gvl :redraw)))
		   ))

(defun create-popup-inter ()
  (create-instance 'popup-inter inter:two-point-interactor
		   (:waiting-priority highest-priority-level)
		   (:running-priority highest-priority-level)
		   (:start-event :shift-middledown)
		   (:start-where T)
		   (:window work-window)
		   (:continuous nil)
		   (:line-p nil)
		   (:active T)
		   (:stop-action
		    (dont-interfere
		     (lambda (inter points)
		       (declare (ignore inter obj))
		       (if (g-value popup-command-menu :visible)
			   (progn
			     (call-schema popup-command-menu :pop-down)
			     (allow-interference))
			 (progn
			   (block-interference
			    :menu T
			    :help-msg
			    "Select an item or press
Shift-middle-button
to resume editing.")
			   (call-schema popup-command-menu :pop-up
					(first points)
					(second points))
			   )
			 )
		       )))
		   ))

;;;------------------------------------------------------------
;;; error-inter calls pop-error-msg whenever any input is attempted in
;;; one of the windows
;;;------------------------------------------------------------
(defun create-error-inter ()
  (create-instance 'error-inter inter:Button-Interactor
		   (:window *all-windows*)
		   (:waiting-priority error-priority-level)
		   (:running-priority error-priority-level)
		   (:start-event '(:any-mousedown :any-keyboard))
		   (:start-where T)
		   (:continuous nil)
		   (:active nil)
		   (:self-deactivate T)
		   (:stop-action
		    #'(lambda (&rest args)
			(declare (ignore args))
			(pop-error-msg)
			))
		   ))

;;;------------------------------------------------------------
;;; help-inter gets rid of the help message
;;;------------------------------------------------------------
(defun create-help-inter ()
  (create-instance 'help-inter inter:Button-Interactor
		   (:window *all-windows*)
		   (:waiting-priority help-priority-level)
		   (:running-priority help-priority-level)
		   (:start-event '(:any-mousedown :any-keyboard))
		   (:start-where T)
		   (:continuous nil)
		   (:active nil)
		   (:self-deactivate T)
		   (:stop-action
		    #'(lambda (&rest args)
			(declare (ignore args))
			(remove-dbox help-display)
			(opal:lower-window dialog-window)
			(allow-interference)
			))
		   ))

;;;------------------------------------------------------------
;;; Create-Draw-Inter creates the interactor to create objects in the
;;; workbench.
;;;------------------------------------------------------------

(defun create-draw-inter ()
  (create-instance 'draw-obj-inter inter:Two-Point-Interactor
           (:window work-window)
	   (:start-event :rightdown)
	   (:start-where `(:in ,work-agg))
	   (:waiting-priority low-priority-level)
	   (:how-to-abort
	    #'(lambda ()
		(pop-help-string)))
	   (:start-action
	    #'(lambda (interactor first-points)
		(push-help-string
		 (case (g-value tool-menu :menu-items :selected :object-type)
		       ((:miro-box) "Create Box: type ^G to
abort")
		       ((:miro-arrow) "Create Arrow: type ^G
to abort")
		       (T "Type ^G to abort")))
		(set-abort-inter interactor)
		(call-prototype-method interactor first-points)))
	   (:back-inside-action
	    #'(lambda (interactor outside-control new-inside-points)
		(push-help-string
		 (case (g-value tool-menu :menu-items :selected :object-type)
		       ((:miro-box) "Create Box: type ^G to
abort")
		       ((:miro-arrow) "Create Arrow: type ^G
to abort")
		       (T "Type ^G to abort")))
		(set-abort-inter interactor)
		(call-prototype-method interactor outside-control new-inside-points)))
	   (:outside-action
	    #'(lambda (interactor outside-control)
		(pop-help-string)
		(clear-abort-inter)
		(call-prototype-method interactor outside-control)))
	   (:stop-action
	    #'(lambda (interactor final-points)
		(pop-help-string)
		(clear-abort-inter)
		(call-prototype-method interactor final-points)
		))
	   (:line-p (o-formula (if (eq (gv tool-menu :menu-items
					   :selected :object-type)
				       :miro-box) NIL T)))
	   (:feedback-obj (o-formula (case (gv tool-menu :menu-items
					       :selected :object-type) 
					  ((:miro-box) dash-line-feedback)
					  ((:miro-arrow) line-feedback)
					  (T NIL))))
	   (:min-width (o-formula (if (eq (gv tool-menu :menu-items
					      :selected :object-type)
					  :miro-box)
				      (round (* (gv zoom-agg :scale)
						*minimum-box-width*))
				    NIL)))
	   (:min-height (o-formula (if (eq (gv tool-menu :menu-items
					       :selected :object-type)
					  :miro-box)
				       (round (* (gv zoom-agg :scale)
						 *minimum-box-height*))
				     NIL)))
	   (:final-function #'draw-new-object))
  )

;;;------------------------------------------------------------
;;; Create-Text-Inters creates the interactors to enter and edit the
;;; names of the objects
;;; This creates two interactors.
;;; 1). get-text-inter is activated by mouse click and by being
;;; started by the draw-obj-inter. In the case where it is activated
;;; by a click, the cursor starts at the location of the mouse in the
;;; string. 
;;; 2). text-inter-2 is activated by a keyboard press when the mouse
;;; is over a box or arrow. This is helpful for cases where the string
;;; is currently NIL.
;;;------------------------------------------------------------

(defun create-text-inters ()
  ;; interactor for entering and editing text. 
  (create-instance 'get-text-inter inter:text-interactor
		   (:feedback-obj NIL)
		   (:window work-window)
		   ;; only want to activate on box and arrow labels, so
		   ;; change start-where to obj-agg
		   (:start-where `(:leaf-element-of ,obj-agg :type
						    (,opal:cursor-text
						     ,opal:cursor-multi-text)))
		   (:start-action
		    #'(lambda (interactor objbeingchanged event)
			(set-abort-inter interactor)
			(block-interference
			 :leave-priorities-alone T
			 :leave-inters-alone T
			 :leave-work-window-alone T
			 :help-msg
			 "Enter/edit the name of
the object.  Press
return or a mouse
button to return to
normal editing.
Type ^G to abort.")
			(call-prototype-method interactor objbeingchanged event)))
		   (:stop-event '(:any-mousedown #\RETURN))
		   (:abort-action
		    #'(lambda (interactor objbeingchanged event)
			(allow-interference :leave-inters-alone T)
			(call-prototype-method interactor objbeingchanged event)
			))
		   (:stop-action
		    #'(lambda (interactor objbeingchanged event)
			      (clear-abort-inter)
			      ;; we may have invalidated ambig results
			      (when (g-value ambig-status :guaranteed-valid)
				    (s-value ambig-status
					     :guaranteed-valid nil))
			      (allow-interference :leave-inters-alone T)
			      (call-prototype-method interactor objbeingchanged event)
			      ))
		   (:waiting-priority medium-priority-level)
		   )
  
  ;; text-inter-2 -- triggered by keyboard input
  (create-instance 'text-inter-2 inter:text-interactor
		   (:feedback-obj NIL)
		   (:window work-window)
		   ;; always start at end of string
		   (:cursor-where-press NIL)
		   (:start-event :any-keyboard)
		   ;; change to check-leaf-but-return-element because of
		   ;; problems with "internal" boxes
		   ;;	    (:start-where `(:element-of ,obj-agg))
		   ; this was for the first text inter - had to be "in" object
		   ;	    (:start-where `(:check-leaf-but-return-element
		   ;			    ,obj-agg))
		   (:start-where `(:in ,work-agg))
		   (:stop-event '(:any-mousedown #\RETURN))
		   (:abort-action
		    #'(lambda (interactor objbeingchanged event)
			(when *dont-interfere*
			      (allow-interference :leave-inters-alone T))
			(call-prototype-method interactor
					       objbeingchanged event)))
		   (:stop-action
		    #'(lambda (interactor objbeingchanged event)
			;; we may have invalidated ambig results
			(when (g-value ambig-status :guaranteed-valid)
			      (s-value ambig-status :guaranteed-valid nil))
			(allow-interference :leave-inters-alone T)
			(call-prototype-method interactor objbeingchanged event)
			))
		   (:waiting-priority low-priority-level)
		   
		   ;; both boxes and arrows have :label part
		   ; this was for the first text inter - object didn't need to
		   ; be selected to edit
		   ; (:obj-to-change (o-formula (gvl :first-obj-over :label)))
		   (:obj-to-change (o-formula (gv (car (gv obj-agg :selected))
						  :label))) 
		   (:start-action
		    #'(lambda (interactor objbeingchanged event)
			;; only start if one object selected
			(cond ((single-object-selected)
			       (set-abort-inter interactor)
			       (block-interference
				:leave-priorities-alone T
				:leave-inters-alone T
				:leave-work-window-alone T
				:help-msg
				"Enter/edit the name of
the object.  Press
return or a mouse
button to return to
normal editing.
Type ^G to abort.")
			       (call-prototype-method interactor objbeingchanged
						      event))
			      ;; otherwise, abort
			      (T
			       (call-prototype-method interactor
						      objbeingchanged event)
			       (inter:Abort-Interactor interactor)
			       ))))
		   )
  )


;;;------------------------------------------------------------
;;; Create-Selection-Inter creates the interactor to select objects in
;;; the workbench. 
;;; Select/unselect an object by left click. 
;;; The list of currently selected objects are in :selected slot of
;;; obj-agg. 
;;;------------------------------------------------------------

(defun create-selection-inter ()
  (create-instance 'select-obj-inter inter:button-interactor
	   (:window work-window)
	   (:continuous NIL)  ; selected immediately when press
	   (:waiting-priority low-priority-level)
	   (:start-where 
  	    `(:check-leaf-but-return-element ,obj-agg))
	   (:start-event :leftdown)
	   (:final-function
	    #'(lambda (&rest args)
		(declare (ignore args))
		;; update button visibility: copy, delete, display,
		;; unselect
		(update-command-inactive-list)
		))
	   ))

;;;------------------------------------------------------------
;;; Create-Sweep-Inter creates the interactor to select objects in
;;; the workbench by sweeping out a box around all of them. 
;;; Activate with middle button.
;;; The list of currently selected objects are in :selected slot of
;;; obj-agg. 
;;;------------------------------------------------------------

(defun create-sweep-inter ()
  (create-instance 'sweep-select-inter inter:Two-Point-Interactor
           (:window work-window)
	   (:waiting-priority low-priority-level)
	   (:start-event :middledown)
	   (:start-where `(:in ,work-agg))
	   (:line-p NIL)
	   ;; use same feedback-obj as draw-obj-inter. 
	   ;; should move creation of feedback obj's outside of
	   ;; create-draw-inter.
	   (:feedback-obj dash-line-feedback)
	   ;; not sure about these
	   ;; don't see a need for these -- ky
	   ;;(:min-width 20)
	   ;;(:min-height 20)
	   (:final-function #'select-swept-objects))
)


;;;------------------------------------------------------------
;;; Create-Move-Inter creates the interactor to move the selected
;;; objects. 
;;;------------------------------------------------------------

(defun create-move-inter ()
  (create-instance 'move-inter inter:Move-Grow-Interactor
	   (:window work-window)
	   (:continuous T)
	   (:waiting-priority low-priority-level)
	   (:start-event :shift-leftdown)
	   (:start-where `(:list-check-leaf-but-return-element
	   		   ,obj-agg :selected))
	   (:running-where `(:in-box ,work-agg))
	   (:outside NIL) ; goes back to original position if go
			  ; outside
	   (:feedback-obj move-feedback)
	   (:min-x (o-formula (- 1 (first (gv pic-sp :real-value)))))
	   (:max-x
	    (o-formula
	     (+ (- (gv work-window :width) (third (gvl :feedback-obj :box)) 3)
		(- (if (gv pic-sp :scroll-p) (first (gv pic-sp :val-2)) 0)
		   (first (gv pic-sp :real-value))))
	     ))
	   (:min-y (o-formula (- 1 (second (gv pic-sp :real-value)))))
	   (:max-y
	    (o-formula
	     (+ (- (gv work-window :height) (fourth (gvl :feedback-obj :box)) 3)
		(- (if (gv pic-sp :scroll-p) (second (gv pic-sp :val-2)) 0)
		   (second (gv pic-sp :real-value))))
	     ))
	   (:line-p  NIL) 
	   (:grow-p nil)  ; move only
	   (:how-to-abort
	    #'(lambda ()
		;; turn off feedback visibility and set new points of objects
		(s-value (g-value move-inter :feedback-obj) :visible nil)
		(pop-help-string)))
	   (:abort-action
	    #'(lambda (interactor objbeingchanged)
		(declare (ignore interactor objbeingchanged))
		(s-value (g-value move-inter :feedback-obj) :visible nil)))
	   (:start-action 
	    #'(lambda (interactor objbeingchanged points)
		(declare (ignore objbeingchanged points))
		(let* ((fbo (g-value interactor :feedback-obj))
		       (bounding-box (find-bounding-box
				      (g-value obj-agg :selected)))
		       (offsets (g-value pic-sp :real-value))
		       )
		  (set-abort-inter interactor)
		  ;; remember old x and y position
		  (s-value interactor :old-xy
			   (list (first bounding-box) (second bounding-box)))
		  ;; remember the offset of the mouse from the top of
		  ;; the bounding-box
		  (s-value interactor :x-offset
			   (- (first bounding-box)
			      (inter:event-x inter:*current-event*)
			      (first offsets)))
		  (s-value interactor :y-offset
			   (- (second bounding-box)
			      (inter:event-y inter:*current-event*)
			      (second offsets)))
		  ;; set :box and visibility of feedback object
		  (s-value fbo :box
			   (list (+ (g-value interactor :x-offset)
				    (inter:event-x inter:*current-event*))
				 (+ (g-value interactor :y-offset)
				    (inter:event-y inter:*current-event*))
				 (third bounding-box)
				 (fourth bounding-box)))
		  (s-value (g-value interactor :feedback-obj) :visible t)
;;		(format t "Start-Action::Set move-feedback's :box to ~A~%" 
;;			(g-value move-feedback :box))
		  (push-help-string "move: type ^G to abort")
		  )))
	   (:back-inside-action
	    #'(lambda (interactor outside-control objbeingchanged newpoints)
		(declare (ignore outside-control objbeingchanged newpoints))
		(let* ((fbo (g-value interactor :feedback-obj))
		       (box (g-value fbo :box))
		       )
		  (push-help-string "move: type ^G to abort")
		  (set-abort-inter interactor)
		  (setf (first box)
			(min-max
			 (+ (g-value interactor :x-offset)
			    (inter:event-x inter:*current-event*))
			 (g-value interactor :min-x)
			 (g-value interactor :max-x)))
		  (setf (second box)
			(min-max
			 (+ (g-value interactor :y-offset)
			    (inter:event-y inter:*current-event*))
			 (g-value interactor :min-y)
			 (g-value interactor :max-y)))
		  (mark-as-changed fbo :box)
		  (s-value fbo :visible t)
		  )))
	   (:outside-action
	    #'(lambda (interactor outside-control objbeingchanged)
		(declare (ignore objbeingchanged outside-control))
		(pop-help-string)
		(clear-abort-inter)
		(s-value (g-value interactor :feedback-obj)
			 :visible nil)
		))
	   ;; only set first two slots of :box
	   (:running-action 
	    #'(lambda (interactor objbeingchanged newpoints)
		(declare (ignore objbeingchanged newpoints))
		;; set :box of feedback-obj
		(let* ((fbo (g-value interactor :feedback-obj))
		       (box (g-value fbo :box))
		       )
		  (setf (first box)
			(min-max
			 (+ (g-value interactor :x-offset)
			    (inter:event-x inter:*current-event*))
			 (g-value interactor :min-x)
			 (g-value interactor :max-x)))
		  (setf (second box)
			(min-max
			 (+ (g-value interactor :y-offset)
			    (inter:event-y inter:*current-event*))
			 (g-value interactor :min-y)
			 (g-value interactor :max-y)))
		  ;; tell kr we've change the slot
		  (mark-as-changed fbo :box)
;;		  (format t "Running Action::Set move-feedback's :box to ~A~%" 
;;			  (g-value move-feedback :box))
		  )))

	   (:stop-action 
	    #'(lambda (interactor objbeingchanged newsize)
		(declare (ignore objbeingchanged newsize))
		(clear-abort-inter)
		;; turn off feedback visibility and set new points of objects
		(s-value (g-value interactor :feedback-obj) :visible nil)
;;		(format t "Stop Action::Final position of bounding box = ~A~%" 
;;			(g-value interactor :feedback-obj :box))
		(move-selected-boxes (- (first (g-value interactor
							:feedback-obj :box)) 
					(first (g-value interactor
							:old-xy))) 
				     (- (second (g-value interactor
							 :feedback-obj :box)) 
					(second (g-value interactor
							 :old-xy))))
		;; turn off selection _visibility_ 
		(dolist (obj (g-value obj-agg :selected))
			(s-value obj :selected NIL))
		;; unselect everything
		(s-value obj-agg :selected NIL)
		(pop-help-string)
		;; update the bounding box
		(s-value pic-sp :bb-box
			 (find-bounding-box (get-values obj-agg :components) T))

		;; we may have invalidated undo-objects
		(destroy-undo-objects)
		;; update button visibility: copy, delete, display,
		;; unselect, undelete
		(update-command-inactive-list)

		;; we may have invalidated ambig results
		(when (g-value ambig-status :guaranteed-valid)
		      (s-value ambig-status :guaranteed-valid nil))
		(opal:update menu-window)
		))
	   ))


;;;------------------------------------------------------------
;;; Create-Resize-Inter creates the interactor(s) to resize a selected
;;; object (either box or arrow). The interactor is started by
;;; pressing the Shift key and the right mouse button. If exactly one
;;; object is selected, the interactor shows the feedback for the
;;; resize, and when the mouse button is let up, it makes the
;;; appropriate changes. If no objects are selected, or if more than
;;; one is selected, it prints an error.
;;; Right now, the interactor only works for boxes.
;;;------------------------------------------------------------

(defun create-resize-inter ()
 ;; the interactor to resize boxes
  (create-instance 'resize-inter inter:Move-Grow-Interactor
	   (:window work-window)
	   (:continuous T)
	   (:waiting-priority low-priority-level)
	   (:start-event :shift-rightdown)
	   (:start-where `(:list-check-leaf-but-return-element
	   		   ,obj-agg :selected))
	   ;; this will change too 
	   ;; (:obj-to-change (o-formula (gv obj-agg :selected)))
	   (:running-where `(:in-box ,work-agg))
	   (:outside NIL) ; goes back to original position if go outside
	   (:feedback-obj resize-box-feedback)
	   ;; temporarily only resize boxes
	   (:line-p  NIL) ; boxes only
	   (:grow-p T)  ; resize only
	   (:min-width (o-formula (round (* (gv zoom-agg :scale)
					    *minimum-box-width*))))
	   (:min-height (o-formula (round (* (gv zoom-agg :scale)
					     *minimum-box-height*))))
	   (:how-to-abort
	    #'(lambda ()
		;; turn off feedback visibility
		(s-value resize-box-feedback :visible nil)
		(pop-help-string)
		))
	   (:abort-action
	    #'(lambda (interactor &rest args)
		(declare (ignore args))
		(s-value (g-value interactor :feedback-obj) :visible nil)))
	   (:start-action 
	    #'(lambda (interactor obj newsize)
		(declare (ignore newsize))
;;		(format t "Starting resize-inter~%")
		(if (single-box-selected) 
		    ;; single box selected, so do default
		    (progn
		      (let* ((fbo (g-value interactor :feedback-obj))
			     (box (g-value obj :box))
			     (threshold (or (g-value obj :hit-threshold) 0))
			     (scale (g-value zoom-agg :scale))
			     (scaled-box (list (round (* scale (first box)))
					       (round (* scale (second box)))
					       (round (* scale (third box)))
					       (round (* scale (fourth box)))
					       ))
			     (x (inter:event-x inter:*current-event*))
			     (y (inter:event-y inter:*current-event*))
			     (offsets (g-value pic-sp :real-value))
			     (left (- (first scaled-box) (first offsets)))
			     (right (+ left (third scaled-box)))
			     (top (- (second scaled-box) (second offsets)))
			     (bottom (+ top (fourth scaled-box)))
			     (on-left (<= (- left threshold) x (+ left threshold)))
			     (on-right (<= (- right threshold) x (+ right threshold)))
			     (on-top (<= (- top threshold) y (+ top threshold)))
			     (on-bottom (<= (- bottom threshold) y (+ bottom threshold)))
			     )
			;; figure out which dimensions are being changed
			(set-abort-inter interactor)
			(s-value fbo :box (list left top (- right left)
						(- bottom top)))
			(s-value fbo :old-box (g-value fbo :box))
			(s-value fbo :x-offset (- x left))
			(s-value fbo :y-offset (- y top))
			(s-value fbo :on-left on-left)
			(s-value fbo :on-right on-right)
			(s-value fbo :on-top on-top)
			(s-value fbo :on-bottom on-bottom)
			(s-value fbo :visible T)
			(push-help-string "resize: type ^G to
abort")
			;;(call-prototype-method interactor objbeingchanged newsize)
			))
		  ;; else signal error and abort
		  (progn
;		    (format T "Can't Resize -- either more than one object 
;     selected, or the object is an arrow~%")
		    (push-error-msg
		     "Can't Resize -- either
more than one object
selected, or the object
is an arrow.")
		    (inter:Abort-Interactor interactor)
		    ))))
	   (:back-inside-action
	    #'(lambda (interactor outside-control objbeingchanged newpoints)
		(declare (ignore outside-control objbeingchanged newpoints))
		(let* ((fbo (g-value interactor :feedback-obj))
		       (box (copy-list (g-value fbo :box)))
		       (obox (g-value fbo :old-box))
		       (x (inter:event-x inter:*current-event*))
		       (y (inter:event-y inter:*current-event*))
		       (on-left (g-value fbo :on-left))
		       (on-right (g-value fbo :on-right))
		       (on-top (g-value fbo :on-top))
		       (on-bottom (g-value fbo :on-bottom))
		       (new-x (- x (g-value fbo :x-offset)))
		       (new-y (- y (g-value fbo :y-offset)))
		       (min-width (g-value interactor :min-width))
		       (min-height (g-value interactor :min-height))
		       (new-width (+ (third obox)
				     (if on-right (- new-x (first obox))
				       (- (first obox) new-x))))
		       (new-height (+ (fourth obox)
				      (if on-bottom (- new-y (second obox))
					(- (second obox) new-y))))
		       )
		  (push-help-string "resize: type ^G to abort")
		  (set-abort-inter interactor)
		  ;; change :width
		  (when (or on-right on-left)
			(when (< new-width min-width)
			      (setq new-x
				    (- (+ (first box) (third box)) min-width))
			      )
			(rplaca (cddr box) (max new-width min-width))
			)
		  ;; change :height
		  (when (or on-bottom on-top)
			(when (< new-height min-height)
			      (setq new-y
				    (- (+ (second box) (fourth box)) min-height))
			      )
			(rplaca (cdddr box) (max new-height min-height))
			)
		  ;; change :left
		  (when on-left (rplaca box new-x))
		  ;; change :top
		  (when on-top (rplaca (cdr box) new-y))
		  (unless (equal (g-value fbo :box) box)
			  (s-value fbo :box box))
		  (s-value fbo :visible T)
		  ;;(call-prototype-method interactor outside-control objbeingchanged newpoints)
		  )))
	   (:running-action
	    #'(lambda (interactor obj newpoints)
		(declare (ignore obj newpoints))
		(let* ((fbo (g-value interactor :feedback-obj))
		       (box (copy-list (g-value fbo :box)))
		       (obox (g-value fbo :old-box))
		       (x (inter:event-x inter:*current-event*))
		       (y (inter:event-y inter:*current-event*))
		       (on-left (g-value fbo :on-left))
		       (on-right (g-value fbo :on-right))
		       (on-top (g-value fbo :on-top))
		       (on-bottom (g-value fbo :on-bottom))
		       (new-x (- x (g-value fbo :x-offset)))
		       (new-y (- y (g-value fbo :y-offset)))
		       (min-width (g-value interactor :min-width))
		       (min-height (g-value interactor :min-height))
		       (new-width (+ (third obox)
				     (if on-right (- new-x (first obox))
				       (- (first obox) new-x))))
		       (new-height (+ (fourth obox)
				      (if on-bottom (- new-y (second obox))
					(- (second obox) new-y))))
		       )
		  ;; change :width
		  (when (or on-right on-left)
			(when (< new-width min-width)
			      (setq new-x
				    (- (+ (first box) (third box)) min-width))
			      )
			(rplaca (cddr box) (max new-width min-width))
			)
		  ;; change :height
		  (when (or on-bottom on-top)
			(when (< new-height min-height)
			      (setq new-y
				    (- (+ (second box) (fourth box)) min-height))
			      )
			(rplaca (cdddr box) (max new-height min-height))
			)
		  ;; change :left
		  (when on-left (rplaca box new-x))
		  ;; change :top
		  (when on-top (rplaca (cdr box) new-y))
		  (unless (equal (g-value fbo :box) box)
			  (s-value fbo :box box))
		  )))
	   (:outside-action
	    #'(lambda (interactor outside-control objbeingchanged)
		(declare (ignore outside-control objbeingchanged))
		(pop-help-string)
		(clear-abort-inter)
		(s-value (g-value interactor :feedback-obj) :visible nil)
		))
	   (:stop-action 
	    #'(lambda (interactor objbeingchanged newpoints)
		(declare (ignore newpoints))
		(let* ((fbo (g-value interactor :feedback-obj))
		       (box (copy-list (g-value fbo :box)))
		       (obox (g-value fbo :old-box))
		       (x (inter:event-x inter:*current-event*))
		       (y (inter:event-y inter:*current-event*))
		       (on-left (g-value fbo :on-left))
		       (on-right (g-value fbo :on-right))
		       (on-top (g-value fbo :on-top))
		       (on-bottom (g-value fbo :on-bottom))
		       (new-x (- x (g-value fbo :x-offset)))
		       (new-y (- y (g-value fbo :y-offset)))
		       (min-width (g-value interactor :min-width))
		       (min-height (g-value interactor :min-height))
		       (new-width (+ (third obox)
				     (if on-right (- new-x (first obox))
				       (- (first obox) new-x))))
		       (new-height (+ (fourth obox)
				      (if on-bottom (- new-y (second obox))
					(- (second obox) new-y))))
		       )
;;		(format t "Stopping resize-inter~%")
		  ;; change the relevant slots.  Use rplaca since :box
		  ;; should not be shared by anything else.
		  ;; change :width
		  (when (or on-right on-left)
			(when (< new-width min-width)
			      (setq new-x
				    (- (+ (first box) (third box)) min-width))
			      )
			(rplaca (cddr box) (max new-width min-width))
			)
		  ;; change :height
		  (when (or on-bottom on-top)
			(when (< new-height min-height)
			      (setq new-y
				    (- (+ (second box) (fourth box)) min-height))
			      )
			(rplaca (cdddr box) (max new-height min-height))
			)
		  ;; change :left
		  (when on-left (rplaca box new-x))
		  ;; change :top
		  (when on-top (rplaca (cdr box) new-y))
		  (unless (equal (g-value fbo :box) box)
			  (s-value fbo :box box))
		  (clear-abort-inter)
		  (pop-help-string)
		  ;; turn off feedback visibility (by setting obj-over to
		  ;; nil) 
		  (s-value (g-value interactor :feedback-obj) :visible nil)
		  ;; set new points for box (have to adjust for scroll)
		  (s-value objbeingchanged :box
			   (translate-points (g-value interactor :feedback-obj :box)))
		  ;; turn off selection _visibility_ 
		  (s-value (car (g-value obj-agg :selected)) :selected nil)
		  ;; unselect everything
		  (s-value obj-agg :selected NIL)
		  ;; update the bounding box
		  (s-value pic-sp :bb-box
			   (find-bounding-box (get-values obj-agg :components) T))

		  ;; we may have invalidated undo-objects
		  (destroy-undo-objects)
		  ;; update button visibility: copy, delete, display,
		  ;; unselect, undelete
		  (update-command-inactive-list)

		  ;; we may have invalidated ambig results
		  (when (g-value ambig-status :guaranteed-valid)
			(s-value ambig-status :guaranteed-valid nil))
		  (opal:update menu-window)
		  )))
	   ))

  ;; the interactor to move arrows
  ;;; doesn't work yet...
#|
  (create-instance 'move-arrow-inter inter:Move-Grow-Interactor
	   (:window work-window)
	   (:continuous T)
	   (:steal-mouse T)
	   (:start-event :leftdown)
	   (:start-where `(:element-of ,sel-arrow-feedback))
	   (:running-where `(:in-box ,work-agg))
	   (:outside NIL) ; goes back to original position if go outside
	   (:feedback-obj arrow-feedback)
	   ;; the next two fields cause the object to be changed to be the object that the
	   ;; feedback boxes are defined over, rather than the feedback object itself.  The
	   ;; :first-obj-over slot is set by the interactor with the object the mouse
	   ;; pressed on, which will be one of the 8 selection boxes.
	   (:obj-to-change (o-formula (gv move-arrow-inter :first-obj-over :parent
					     :obj-over)))
	   (:attach-point :where-hit)
	   (:line-p t)
	   (:start-action
	    #'(lambda (interactor objbeingchanged newsize)
		(inter::Move-Grow-Int-Start-Action
		 interactor objbeingchanged newsize)
		(s-value (g-value interactor :first-obj-over) :visible NIL)))
	   (:abort-action
	    #'(lambda (interactor objbeingchanged)
		(let ((obj-over (g-value interactor :first-obj-over)))
		  (unless (null obj-over)
		    (s-value obj-over :visible T)))
		(inter::Move-Grow-Int-Abort-Action
		 interactor objbeingchanged)))
	   (:stop-action
	    #'(lambda (interactor objbeingchanged newsize)
		(s-value (g-value interactor :first-obj-over) :visible T)
		(s-value (g-value (g-value interactor :first-obj-over) :parent)
			 :obj-over NIL)
		(inter::Move-Grow-Int-Stop-Action
		 interactor objbeingchanged newsize)))))
 
|#

;;;------------------------------------------------------------
;;; Create-Copy-Inter creates the interactor to copy the selected
;;; objects. This interactor is initially inactive, and is only
;;; activated by pressing the Copy button in the menu. When activated,
;;; the interactor is started by any mouse press, and displays a
;;; bounding box of all objects to be copied. This bounding box can be
;;; moved around to the desired position before the mouse button is
;;; raised. When the mouse button is raised, all the objects are
;;; copied. 
;;;------------------------------------------------------------

(defun create-copy-inter ()
  (create-instance 'copy-inter inter:Move-Grow-Interactor
		   (:window work-window)
		   (:continuous T)
		   ;; initially inactive -- activated by copy command
		   (:waiting-priority inter:high-priority-level)
		   (:active NIL)
		   (:start-event :any-mousedown)
		   (:start-where `(:in ,work-agg))
		   (:running-where `(:in-box ,work-agg))
		   (:outside NIL) ; goes back to original position if go outside
		   ;; use same feedback object as for move
		   (:feedback-obj move-feedback)
		   (:max-x
		    (o-formula
		     (+ (- (gv work-window :width)
			   (third (gvl :bounding-box)) 3)
			(- (if (gv pic-sp :scroll-p)
			       (first (gv pic-sp :val-2)) 0)
			   (first (gv pic-sp :real-value))))
		     ))
		   (:max-y
		    (o-formula
		     (+ (- (gv work-window :height)
			   (fourth (gvl :bounding-box)) 3)
			(- (if (gv pic-sp :scroll-p)
			       (second (gv pic-sp :val-2)) 0)
			   (second (gv pic-sp :real-value))))
		     ))
		   (:line-p  NIL) ; box only
		   (:grow-p NIL)  ; move only
		   (:how-to-abort
		    #'(lambda ()
			      ;; turn off feedback visibility and prompt string
			      (s-value (g-value copy-inter :feedback-obj) :visible nil)
			      (allow-interference)
			      ;; deactivate interactor
			      (inter:change-active copy-inter NIL)
			      ))
		   (:abort-action
		    #'(lambda (interactor objbeingchanged)
			(declare (ignore objbeingchanged))
			(s-value (g-value interactor :feedback-obj) :visible nil)))
		   (:start-action
		    #'(lambda (interactor objbeingchanged first-points)
			(declare (ignore objbeingchanged first-points))
			(let ((bounding-box
			       (find-bounding-box (g-value obj-agg
							   :selected))))
			  (s-value interactor :bounding-box
				   bounding-box)
			  ;; feedback obj's :box takes position from
			  ;; mouse and size from bounding box 
			  (s-value (g-value interactor :feedback-obj)
				   :box
				   (list 
				    (min-max
				     (inter:event-x inter:*current-event*)
				     0 (g-value interactor :max-x))
				    (min-max
				     (inter:event-y inter:*current-event*)
				     0 (g-value interactor :max-y))
				    (third bounding-box)
				    (fourth bounding-box)))
			  )
;;		(format t "Start-Action::Set copy-feedback's :box to ~A~%" 
;;			(g-value move-feedback :box))
			;; set visibility of feedback object
			(s-value (g-value interactor :feedback-obj) :visible t)
			))
		   (:back-inside-action
		    #'(lambda (interactor outside-control
					  objbeingchanged newpoints)
			(declare (ignore outside-control objbeingchanged
					 newpoints))
			(let* ((fbo (g-value interactor :feedback-obj))
			       (box (g-value fbo :box))
			       )
			  (setf (first box)
				(min-max
				 (inter:event-x inter:*current-event*)
				 0 (g-value interactor :max-x)))
			  (setf (second box)
				(min-max
				 (inter:event-y inter:*current-event*)
				 0 (g-value interactor :max-y)))
			  (mark-as-changed fbo :box)
			  (s-value fbo :visible T))))
		   (:outside-action
		    #'(lambda (interactor outside-control objbeingchanged)
			(declare (ignore outside-control objbeingchanged))
			(s-value (g-value interactor :feedback-obj)
				 :visible nil)))
		   ;; only set first two slots of :box
		   (:running-action
		    #'(lambda (interactor objbeingchanged newsize)
			      (declare (ignore objbeingchanged newsize))
			      ;; set :box of feedback-obj
			      (let* ((fbo (g-value interactor :feedback-obj))
				     (box (g-value fbo :box)))
				    (setf (first box) 
					  (min-max
					   (inter:event-x inter:*current-event*)
					   0 (g-value interactor :max-x)))
				    (setf (second box) 
					  (min-max
					   (inter:event-y inter:*current-event*)
					   0 (g-value interactor :max-y)))
				    ;; tell kr we've change the slot
				    (mark-as-changed fbo :box)
;;		  (format t "Running Action::Set copy-feedback's :box to ~A~%" 
;;			  (g-value move-feedback :box))
				    )))
		   (:stop-action 
		    #'(lambda (interactor objbeingchanged newsize)
			      (declare (ignore objbeingchanged newsize))
			      (clear-abort-inter)
			      ;; turn off feedback visibility and prompt
			      ;; string, and set new points of objects 
			      (s-value (g-value interactor :feedback-obj) :visible nil)
;;		(format t "Stop Action::Final position of bounding box = ~A!%"
;;			(g-value interactor :feedback-obj :box))
			      (copy-objects (- (first (g-value interactor
							       :feedback-obj :box)) 
					       (first (g-value interactor
							       :bounding-box))) 
					    (- (second (g-value interactor
								:feedback-obj :box)) 
					       (second (g-value interactor
								:bounding-box))
					       ))
			      
			      ;; deactivate interactor
			      (inter:change-active interactor NIL)
			      ;; update the bounding box
			      (s-value pic-sp :bb-box
				       (find-bounding-box
					(get-values obj-agg :components) T))

			      (opal:update menu-window)
			      ))
		   ))


;;;============================================================
;;; HELPER FUNCTIONS
;;;============================================================

;;; ------------------------------------------------------------
;;; given a list of components, find their bounding box.
;;; There's probably a (more efficient?) way to do this in Garnet. 
;;; Used by move-inter.
;;; ------------------------------------------------------------
(defun find-bounding-box (component-list &optional dont-scale)
  (if (null component-list)
      (list 0 0 0 0)
    ;; otherwise, find max bounding box
    ;; min-left & min-top's initial values should be based on virtual
    ;; size of workbench, not actual size
    (let* ((first-box (g-value (first component-list) :box))
	   (first-points (g-value (first component-list) :points))
	   (min-left (first (if first-box first-box first-points)))
	   (min-top (second (if first-box first-box first-points)))
	   (max-right 0)
	   (max-bottom 0)
	   (scale (if dont-scale 1 (g-value zoom-agg :scale))))
      (dolist (obj component-list)
	      (let* ((box (g-value obj :box))
		     (points (g-value obj :points))
		     (x1 (when points (first points)))
		     (y1 (when points (second points)))
		     (x2 (when points (third points)))
		     (y2 (when points (fourth points)))
		     (obj-left (if box (first box) (when points (min x1 x2))))
		     (obj-top (if box (second box) (when points (min y1 y2))))
		     (obj-right (if box (+ obj-left (third box))
				  (when points (max x1 x2))))
		     (obj-bottom (if box (+ obj-top (fourth box))
				   (when points (max y1 y2))))
		     )
		(when (or box points)
		      (unless min-left (setq min-left obj-left))
		      (unless min-top (setq min-top obj-top))
		      (setq min-left (min obj-left min-left))
		      (setq min-top (min obj-top min-top))
		      (setq max-right (max obj-right max-right))
		      (setq max-bottom (max obj-bottom max-bottom))
		      )))
      ;; return min's and max's
      (if (and min-left min-top max-right max-bottom)
	  (list (round (* scale min-left))
		(round (* scale min-top))
		(round (* scale (- max-right min-left)))
		(round (* scale (- max-bottom min-top)))
		)
	(list 0 0 0 0))
      )))

;;; ------------------------------------------------------------
;;; given an offset, adjust the :box slot of selected boxes, and mark
;;; the positional slots of both boxes and arrows as changed.
;;; used by move-inter
;;; ------------------------------------------------------------
(defun move-selected-boxes (delta-x delta-y)
  (let* ((offsets (g-value pic-sp :real-value))
	 (scale (g-value zoom-agg :scale))
	 (dx (round (+ delta-x (first offsets)) scale))
	 (dy (round (+ delta-y (second offsets)) scale))
	 )
    (dolist (obj (g-value obj-agg :selected))
	    (if (eq (g-value obj :object-type) :miro-box)
		;; move a box
		(let* ((box-list (copy-list (g-value obj :box)))
		       (left (first box-list))
		       (top (second box-list))
		       )
		  ;; move box
		  (rplaca box-list (+ left dx))
		  (rplaca (cdr box-list) (+ top dy))
		  (s-value obj :box box-list)
		  )
	      ;; move an arrow
	      ;;(let* ((points (g-value obj :points))
		     ;;(x1 (first points))
		     ;;(y1 (second points))
		     ;;(x2 (third points))
		     ;;(y2 (fourth points))
		     ;;)
		;;(s-value obj :points (list (+ x1 dx) (+ y1 dy)
		;;(+ x2 dx) (+ y2 dy)))
		;;)
	      ))
    ))

;;;------------------------------------------------------------
;;; Copy Objects copies all selected objects with offset delta-x and
;;; delta-y .
;;; have to copy all boxes before arrows so that arrows attach to the
;;; right boxes.
;;;------------------------------------------------------------
(defun copy-objects (delta-x delta-y)
  (let* ((offsets (g-value pic-sp :real-value))
	 (scale (g-value zoom-agg :scale))
	 (dx (round (+ delta-x (first offsets)) scale))
	 (dy (round (+ delta-y (second offsets)) scale))
	 )
    (setq *objects-to-display* nil)
    (set-help-string "Copying...")
    (opal:update help-window)
    ;; copy boxes
    (dolist (obj-to-copy (g-value obj-agg :selected))
	    (when (eq (g-value obj-to-copy :object-type) :miro-box)
		  ;; (format T "Copying box ~A~%" obj-to-copy)
		  ;; have to add in scroll bar offset as well
		  (push (list (copy-miro-box obj-to-copy dx dy
					     obj-agg))
			*objects-to-display*)))
    ;; copy arrows
    (dolist (obj-to-copy (g-value obj-agg :selected))
	    (when (eq (g-value obj-to-copy :object-type) :miro-arrow)
		  ;; (format T "Copying arrow ~A~%" obj-to-copy)
		  (push (list (copy-miro-arrow obj-to-copy dx dy
					       obj-agg))
			*objects-to-display*)))
    (setq *objects-to-display* (reverse *objects-to-display*))
    (set-help-string "Displaying copied
objects...")
    (opal:update help-window)
    (display-next-object)
    ))

;;; ------------------------------------------------------------
;;; select each object that is completely inside point-list
;;; ------------------------------------------------------------
(defun select-swept-objects (inter point-list)
  (declare (ignore inter))
  (opal:do-components obj-agg 
		 #'(lambda (obj)
		     (when (obj-inside-area-p obj point-list)
			   ;; set object selection and add to selected
			   ;; list 
			   (s-value obj :selected T)
			   (pushnew obj (g-value obj-agg
						 :selected)))))

  ;; update button visibility: copy, delete, display, unselect
  (update-command-inactive-list)
  )

;;; ------------------------------------------------------------
;;; object selection functions
;;; ------------------------------------------------------------
(defun single-box-selected ()
  (let ((selected-list (g-value obj-agg :selected)))
    (and 
     ;; only one item selected
     (= (length selected-list) 1)
     ;; it's a box
     (eq (g-value (car (g-value obj-agg :selected)) :object-type)
	 :miro-box)))) 

(defun single-object-selected ()
    ;; only one item selected
    (= (length (g-value obj-agg :selected)) 1))

;;; ------------------------------------------------------------
;;; print the labels of the selected objects
;;; ------------------------------------------------------------
(defun print-selected ()
  (let ((name-list nil))
    (dolist (obj (g-value obj-agg :selected))
	    (push (g-value obj :label :string) name-list))
    (format t "Objects ~A are selected ~%" name-list)))

;;; ------------------------------------------------------------
;;; this function converts a point list of the from (x y w h), where x
;;; and y are actual points in the editor, to a new point list where x
;;; and y are adjusted based on the scroll bar positions
;;; ------------------------------------------------------------
(defun translate-points (point-list &optional (use-offsets T))
  (list
   (round (+ (first point-list)
	     (if (and use-offsets (g-value pic-sp :scroll-p))
		 (first (g-value pic-sp :real-value)) 0))
	  (g-value zoom-agg :scale))
   (round (+ (second point-list)
	     (if (and use-offsets (g-value pic-sp :scroll-p))
		 (second (g-value pic-sp :real-value)) 0))
	  (g-value zoom-agg :scale))
   (round (third point-list) (g-value zoom-agg :scale))
   (round (fourth point-list) (g-value zoom-agg :scale))
   )
  )
