;;;             -*- Mode: Lisp; Package: MIRO; -*-
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

;;; 
;;; MIRO EDITOR - MENU FUNCTIONS
;;;
;;; This file contains the code for functions which are called to 
;;; create the Miro menus. The main menu (menu-agg) has the following
;;; components:
;;; 
;;; - pictype-menu
;;; - range-menu
;;; - tool-menu
;;; - arrow-menu
;;; - constraint-menu
;;; - command-buttons (commands-agg)
;;; 
;;; the pop-up menus (dialog boxes) are controlled with the command
;;; "Display". The dboxes are *box-db* and *arrow-db* respectively.
;;; This file contains the code to create and control those dboxes.
;;; 

#|
============================================================
Change log:
    11/04/91 ky ; Use new menu objects.  Added new menus.
    04/02/91 ky ; Use call-schema, dbox-reset-button.
    04/01/91 ky ; Use dbox-labeled-text-input.
    03/21/91 ky ; Added verifier menu.
    03/11/91 ky ; Split commands into 2 menus.
    12/7/90  ky ; Fixed help message for display dbox.  Lower dialog
		; window BEFORE making ambig-menu invisible to avoid
		; an error.
    12/4/90  ky ; Raise/Lower the dialog window.
    12/3/90  ky ; Moved dialog boxes to dialog-window.
    11/26/90 ky ; Changed "Exit & Save", "Exit No Change" to "Save
		; Changes", "Abort.  Made *y-n-buttons* a dialog box
		; and moved it to work-agg.  Added :help-msg to
		; *y-n-buttons*.
    11/20/90 ky ; Get :filling-style values from *colors*.  Don't
		; change the background color of the work-window when
		; prompting the user for the location of the display
		; dialog box.
    11/19/90 ky ; Don't need to get around scrolling-input-string bug;
		; just set the value now.
    11/12/90 ky ; Look for the new output format for the ambiguity
		; checker if *new-ambig* (defined in miro-defs.lisp)
		; is non-nil.
    11/9/90  ky ; Handle ambig error output.
    11/8/90  ky ; Don't depend on multi-text for help-button for now.
    11/7/90  ky ; Fixed a problem with ambig results not being updated
		; immediately.
    11/5/90  ky ; Generate sensible output for the ambiguity checker
		; when it is run on a file other than the workbench.
    11/1/90  ky ; Added print-dbox.
    11/1/90  ky ; Added filename-dbox.
    10/30/90 ky ; Added "ambig-options" menu, declaration of function
		; "do-ambiguity-check".
    10/25/90 ky ; Convert item lists in ambig-status to lists of
		; vectors containing the original list and the string
		; representation.
    10/24/90 ky ; Use a browser-gadget to display the ambiguity
		; checker results.
    10/19/90 ky ; Added call to opal:notice-items-changed when
		; ambig-menu is changed.  Turned ambig-menu feedback
		; on.
    10/18/90 ky ; Maximum number of items visible in ambig-menu is 30.
		; Item-to-string function now returns "" instead of
		; nil if it was called with nil.
    9/24/90 ky  ; Moved some information from ambig-menu to
		; ambig-status in an attempt to avoid display errors.
		;
		; Call allow-interference when changing pictypes if
		; the ambig menu was visible.
		;
		; Added function :set-menu-type to ambig-status.
		;
    9/20/90 ky  ; Added show-ambig, ambig-status, ambig-menu.
		; Make ambig-status, ambig-menu invisible when pictype
		; changes.
    9/19/90 ky  ; Added help-display.
    9/18/90 ky  ; Added help-buttons.
    9/17/90 ky  ; Allow buttons to turn themselves off individually.
		; Added :buttonhash to commands-menu to provide an
		; easy way to find a specific button.  Added
		; :can-activate to each button.
		;
		; Update the setting of the ambig button in
		; pictype-menu's selection function.
    9/14/90 ky  ; Changed "Undo" to "Undelete".
    9/13/90 ky  ; The "reset" button for scaling resets the horizontal
		; and vertical offsets as well.
		; Made a few changes to get rid of compile-time
		; warnings.
    9/12/90 ky  ; In pictype-menu's :selection-function, don't try to
		; access :value-obj if it hasn't been created yet.
		; This situation can arise if the user reads in a file
		; without having previously changed pictype.
		;
		; Use :draw-function :xor.
		;
		; Added command button "Legal?".
		;
		; Rearranged command buttons so that "Ambig?",
		; "Legal?", and "Print" are on the top row.
		;
		; The double arrows on the scaling trill device change
		; the size by a factor of 2.
		;
		; Keep *box-db* and *arrow-db* "visible" all the time
		; so that the first "display" doesn't take so long.
		;
    8/23/90 ky  ; A few changes to work with the "test" Garnet.  Fix
		; arrow type when pictype changes from constraint to
		; instance.
		;
    7/31/90 ky  ; Changed a few sizes/positions to make room for the
		; scrollpad.
		;
		; Changed help messages to conform to the size of the
		; new help window.
		;
		; Use pic-sp instead of vert-sb, hor-sb.
		;
    7/23/90 ky  ; Moved command menu up a little bit and put the
		; buttons in 3 columns instead of 2.
		;
		; Added function create-zoom-buttons to create a trill
		; device for scaling.
		;
		; Dialog boxes (*box-db* and *arrow-db*) are handled
		; differently.  See the change log in miro-inters.lisp
		; for details.
		;
    7/2/90  ky  ; Don't use :xor as a :draw-function.  This causes
		; visibility problems in some circumstances.
    6/25/90 ky  ; Deleted ":label-string" since it didn't seem to be
		; updating/updated by ":label :string" at all the
		; proper times.
		;
		; Use new functions "block-interference" and
		; "allow-interference" to change button colors, turn
		; off interactors, etc.
		;
		; When displaying a box, update "SYSNAME:" whenever we
		; change boxes.
		;
		; "mark-as-changed" the height of dialog boxes' :lists
		; when an arrow's "kind" is changed.
		;
		; Capitalize a box's "role" and "type" before using
		; them as values in a dialog box's :button-list.
    6/8/90  ky  ; "mark-as-changed" the height of dialog boxes' :lists
		; when the value in pictype-menu changes.  We need to
		; do this to get around yet another update bug.
    6/1/90  ky  ; *box-db* and *arrow-db* try to keep themselves
		; completely in the work window.  The visibility of
		; *box-db* and *arrow-db* is now changed by changing
		; :visible rather than by adding/removing them from
		; work-agg.
                ;
                ; Replaced :interfill with :buttonfill and removed
		; :exitfill.  The exit command is no longer allowed to
		; interfere with other commands.
    5/9/90  ky  ; Allow *y-n-buttons* to be aborted with ^G.
	        ; (De)activate buttons by changing :visible -- don't
	        ; remove them from help-agg. 
    5/8/90  ky  Added function make-y-n-buttons.  Use
                call-prototype-method instead of naming default
                interactor functions explicitly.
    4/27/90 ky  The "labeled-box"es associated with *box-db* and
                *arrow-db* now set themselves up to be aborted by the abort
                interactors if the user types ^G.
    4/26/90 ky  Changed create-command-buttons to check the ordering
                of the command buttons before assigning color
                dependencies.  It is assumed that the buttons will be in order, with
               "Exit" at either the beginning or the end of the list.
    4/25/90 ky	Changes to display-box, display-arrow:
		  Added "type ^G to abort" to the help message.
		  Turn off low and medium priority interactors.
		  Set *dont-interfere*.
		  Change the color of the text buttons in commands-menu
		    (:interfill) to dark-gray.

		Changes to commands-menu:
		  All commands except exit use the dont-interfere macro.
		  Added :interfill and :exitfill to allow commands to change
		    the color of the text buttons.  The text buttons in
		    commands-menu are made to depend on :interfill (:exitfill
		    for exit) in the function create-command-buttons.

		Changes to box-db-exit-and-save, box-db-exit-no-change,
		arrow-db-exit-and-save, arrow-db-exit-no-change:
		  Clear *dont-interfere*.
		  Change :interfill to white.

		*box-db* and *arrow-db* now have a white background, so other
		  objects in work-window don't show through.

    4/17/90 ky	Use text buttons instead of radio buttons for the exit
		commands in *box-db* and *arrow-db*.  Also, use a
		command-button-panel in create-command-buttons.
    4/10/90 amz added prompt to help window in display-{box,arrow}
    2/21/90 amz added dialog boxes for changing attributes of a
              selected box or arrow
    1/4/90 amz  changed to use gadgets for menus 
    9/5/89  amz changed to o-formulas
    8/11/89 amz put menu functions in separate file
    8/9/89 amz  Changed menu to include constraint options.
    8/3/89 amz  Changed to new version of aggrestuff.
    5/4/89 prm   Added text interactor for the boxes labels
    3/30/89 afm  added function save-workbench
                 first pass at producing output in interm file format
    Nov. 1988 Brad Myers, Philippe Marchal -- Created.

============================================================
|#

(in-package "MIRO" :use `("LISP" "KR"))

(proclaim '(function set-abort-inter)) ; defined in miro-inters.lisp
(proclaim '(function clear-abort-inter)) ; defined in miro-inters.lisp
(proclaim '(function can-ambig))	; defined in miro-cmnds.lisp
(proclaim '(function can-legal))	; defined in miro-cmnds.lisp
(proclaim '(function can-copy))		; defined in miro-cmnds.lisp
(proclaim '(function can-delete))	; defined in miro-cmnds.lisp
(proclaim '(function can-display))	; defined in miro-cmnds.lisp
(proclaim '(function can-hide))		; defined in miro-cmnds.lisp
(proclaim '(function can-undelete))	; defined in miro-cmnds.lisp
(proclaim '(function can-unhide))	; defined in miro-cmnds.lisp
(proclaim '(function can-unselect))	; defined in miro-cmnds.lisp
(proclaim '(function do-ambiguity-check)) ; defined in miro-cmnds.lisp

;;;============================================================
;;; MENU PARAMETERS
;;;============================================================

(defparameter *menu-left* 10)
(defparameter *menu-column2* 100)
(defparameter *pictype-menu-top* 2)
(defparameter *range-menu-top* 40)
(defparameter *tool-menu-top* 60)
(defparameter *arrow-menu-top* 190)
(defparameter *constraint-menu-top* 245)
;(defparameter *thickness-menu-left* 85)
(defparameter *command-menu-top* 355)

; parameters for dialog boxes
(defparameter *db-top-offset* 10)
(defparameter *db-left-offset* 20)


;;;------------------------------------------------------------
;;; Create-Menus creates the menu by adding the aggregates for each
;;; menu part to the menu agg.
;;;------------------------------------------------------------

(defun create-menus (menu-agg)
  ;; add picture type menu
  (opal:add-component menu-agg (create-pictype-menu))
  (when *test-debug* (format t "added pictype menu~%"))
  ;; add range menu
  ;;  (opal:add-component menu-agg (create-range-menu))
  ;; add tool menu
  (opal:add-component menu-agg (create-tool-menu))
  (create-tool-inter)
  (when *test-debug* (format t "added tool menu~%"))
  ;; add arrow menu
  (opal:add-component menu-agg (create-arrow-menu))
  ;; add constraint menu
  (create-constraint-menu)
  (opal:add-component sb-agg (create-zoom-buttons))
  (create-help-buttons)
  (create-box-creation-buttons)
  (create-ambig-menus)
  (create-filename-dbox)
  (create-print-dbox)
  (create-probe-dbox)
  (create-verify-dbox)
  (create-box-creation-dboxes)
  (create-constraint-dboxes)
  (create-options-menu)
  )


;;;------------------------------------------------------------
;;; create-options-menu creates the menu for editing options
;;;------------------------------------------------------------
(defun create-options-menu ()
  (create-instance
   'options-menu opal:aggregadget
   (:left 0) (:top 0)
   (:draw-function :xor)
   (:fast-redraw-p T)
   (:visible T)
   (:display-me
    #'(lambda (obj)
	(block-interference
	 :help-msg
	 (format nil "Change options as
needed.  Press
~S to
save changes, ~S
to abort."
		 "Save Changes" "Abort"))
	(call-schema (g-value obj :menu :background) :set-value
		     (when (changing-bground)
			   (first (g-value obj :menu :background :items))))
	(call-schema (g-value obj :menu :large) :set-value
		     (when (using-large-fonts)
			   (first (g-value obj :menu :large :items))))
#|
        (call-schema (g-value obj :menu :inverse) :set-value
		     (when (using-inverse)
			   (first (g-value obj :menu :inverse :items))))
|#
	(call-schema (g-value obj :menu :gc) :set-value
		     (when (using-auto-gc)
			   (first (g-value obj :menu :gc :items))))
	(add-dbox obj)
	(opal:deiconify-window dialog-window)
	(opal:raise-window dialog-window)
	(opal:update dialog-window)
	))
   (:undisplay-me
    #'(lambda (obj)
	(remove-dbox obj)
	(opal:lower-window dialog-window)
	(allow-interference)
	))
   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu)))
	      )
      (:menu
       ,miro-aggregadget
       (:parts
	((:reset ,dbox-reset-button
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:top ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		 )
	 (:background ,dbox-button-entry
		      (:top ,(o-formula
			      (+ (gvl :parent :reset :top)
				 (gvl :parent :reset :height)
				 10)))
		      (:left ,(o-formula (gv *fonts-and-styles*
					     :db-frame-width)))
		      (:toggle T)
		      (:label-string "")
		      (:items
		       ("Change background when displaying dialog boxes"))
		      (:default-value nil)
		      )
#|
	 (:inverse ,dbox-button-entry
		   (:top ,(o-formula
			   (+ (gvl :parent :background :top)
			      (gvl :parent :background :height)
			      10)))
		   (:left ,(o-formula (gv *fonts-and-styles*
					  :db-frame-width)))
		   (:toggle T)
		   (:label-string "")
		   (:items ("Inverse video"))
		   (:default-value nil)
		   )
|#
	 (:gc ,dbox-button-entry
	      (:top ,(o-formula
		      (+ (gvl :parent :background :top)
			 (gvl :parent :background :height)
			 10)))
	      (:left ,(o-formula (gv *fonts-and-styles*
				     :db-frame-width)))
	      (:toggle T)
	      (:label-string "")
	      (:items ("Garbage collect automatically"))
	      (:default-value ,(o-formula (first (gvl :items))))
	      )
	 (:large ,dbox-button-entry
		 (:top ,(o-formula
			 (+ (gvl :parent :gc :top)
			    (gvl :parent :gc :height)
			    10)))
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:toggle T)
		 (:label-string "")
		 (:items ("Use large fonts and lines"))
		 (:default-value ,(o-formula (first (gvl :items))))
		 )
	 (:done ,command-button-panel
		(:top ,(o-formula
			(+ (gvl :parent :large :top)
			   (gvl :parent :large :height)
			   10)))
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:direction :horizontal)
		(:rank-margin 2)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (call-schema (g-value obj :parent :parent)
				   :undisplay-me)
		      (when (string-equal nv "Save Changes")
			    (option-bground
			     (g-value obj :parent :background
				      :value))
			    (option-large
			     (g-value obj :parent :large :value))
#|
			    (option-inverse
			     (g-value obj :parent :inverse :value))
|#
			    (option-gc
			     (g-value obj :parent :gc :value))
			    ;; we really do need to call this twice
			    (call-schema redraw-inter :redraw)
			    (call-schema redraw-inter :redraw)
			    )
		      ))
		(:items ("Save Changes" "Abort"))
		)
	 ))
       )
      ))
   )
  (push options-menu *all-dboxes*)
  )

;;;------------------------------------------------------------
;;; create-constraint-dboxes creates the dialog boxes for constraint
;;; checking.
;;;------------------------------------------------------------
(defun create-constraint-dboxes ()
  (create-instance
   'constraint-check-options opal:aggregadget
   (:left 0) (:top 0)
   (:draw-function :xor)
   (:fast-redraw-p T)
   (:visible T)
   (:instance-file (o-formula (gvl :menu :instance-file :value)))
   (:instance-comp-dir (o-formula (gvl :menu :instance-comp-dir
				       :value)))
   (:constraint-file (o-formula (gvl :menu :constraint-file :value)))
   (:constraint-comp-dir (o-formula (gvl :menu :constraint-comp-dir
					 :value)))
   (:force-compile (o-formula (gvl :menu :force-compile :value)))

   (:display-me
    #'(lambda (obj)
	(block-interference
	 :help-msg
	 (format nil "Change constraint~%checker options as~%needed."))
	(add-dbox obj)
	(opal:deiconify-window dialog-window)
	(opal:raise-window dialog-window)
	(opal:update dialog-window)
	))

   (:undisplay-me
    #'(lambda (obj)
	(remove-dbox obj)
	(opal:lower-window dialog-window)
	(allow-interference)
	))

   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu)))
	      )
      (:menu
       ,miro-aggregadget
       (:parts
	((:reset ,dbox-reset-button
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:top ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width))))
	 (:instance-file ,dbox-filename-selector
			 (:left ,(o-formula (gv *fonts-and-styles*
						:db-frame-width)))
			 (:top ,(o-formula
				 (+ (gvl :parent :reset :top)
				    (gvl :parent :reset :height)
				    10)))
			 (:title-string
			  ,(o-formula
			    (if (eq (gv pictype-menu :value)
				    :instance)
				"Instance File (leave blank for workbench):"
			      "Instance File:")))
			 (:default-value "")
			 )
	 (:instance-comp-dir ,dbox-filename-selector
			     (:left ,(o-formula (gv *fonts-and-styles*
						    :db-frame-width)))
			     (:top ,(o-formula
				     (+ (gvl :parent :instance-file
					     :top)
					(gvl :parent :instance-file
					     :height)
					10)))
			     (:title-string
			      "Directory for compiled instance file:")
			     (:default-value "")
			     )
	 (:constraint-file ,dbox-filename-selector
			   (:left ,(o-formula (gv *fonts-and-styles*
						  :db-frame-width)))
			   (:top ,(o-formula
				   (+ (gvl :parent :instance-comp-dir
					   :top)
				      (gvl :parent :instance-comp-dir
					   :height)
				      10)))
			   (:title-string
			    ,(o-formula
			      (if (eq (gv pictype-menu :value)
				      :constraint)
				  "Constraint File (leave blank for workbench):"
				"Constraint File:")))
			   (:default-value "")
			   )
	 (:constraint-comp-dir ,dbox-filename-selector
			       (:left ,(o-formula (gv *fonts-and-styles*
						      :db-frame-width)))
			       (:top ,(o-formula
				       (+ (gvl :parent :constraint-file
					       :top)
					  (gvl :parent :constraint-file
					       :height)
					  10)))
			       (:title-string
				"Directory for compiled constraint file:")
			       (:default-value "")
			       )
	 (:force-compile ,dbox-button-entry
			 (:left ,(o-formula (gv *fonts-and-styles*
						:db-frame-width)))
			 (:top ,(o-formula
				 (+ (gvl :parent :constraint-comp-dir
					 :top)
				    (gvl :parent :constraint-comp-dir
					 :height)
				    10)))
			 (:toggle T)
			 (:label-string "")
			 (:items ("Force Compilation"))
			 (:default-value nil)
			 )
	 (:done ,command-button-panel
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula
			(+ (gvl :parent :force-compile :top)
			   (gvl :parent :force-compile :height)
			   10)))
		(:direction :horizontal)
		(:rank-margin 2)
		(:fixed-width-p T)
		(:fixed-width-size 180)
		(:selection-function do-constraint-check)
		(:items ("Compile Constraint Only"
			 "Compile Instance Only"
			 "Run the Constraint Checker"
			 "Abort"))
		)
	 )))
      ))
   )
  (push constraint-check-options *all-dboxes*)
  (create-instance
   'constraint-check-results opal:aggregadget
   (:left 0) (:top 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:result-list nil)
   (:visible T)

   (:undisplay-me #'(lambda (obj)
		      (remove-dbox obj)
		      (opal:lower-window dialog-window)
		      (allow-interference)))

   (:display-me
    #'(lambda (obj)
	(remove-dbox constraint-check-options)
	(opal:notice-items-changed (g-value obj :menu :results
					    :menu-item-list))
	(add-dbox obj)
	(opal:deiconify-window dialog-window)
	(opal:raise-window dialog-window)
	(opal:update dialog-window)
	))

   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu))))
      (:menu
       ,opal:aggregadget
       (:parts
	((:done ,command-button-panel
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula (gv *fonts-and-styles*
				      :db-frame-width)))
		(:direction :horizontal)
		(:items (("Resume Editing"
			  ,#'(lambda (obj &rest args)
			       (declare (ignore args))
			       (call-schema (g-value obj :parent
						     :parent)
					    :undisplay-me)
			       ))
			 ))
		)
	 (:results ,garnet-gadgets:scrolling-menu
		   (:left ,(o-formula (gv *fonts-and-styles*
					  :db-frame-width)))
		   (:top ,(o-formula (+ (gvl :parent :done :top)
					(gvl :parent :done :height)
					10)))
		   (:num-visible 10)
		   (:item-font ,(o-formula (gv *fonts-and-styles*
					       :button-label-font)))
		   (:title "Results")
		   (:title-font ,(o-formula (gv *fonts-and-styles*
						:button-label-font)))
		   (:int-menu-feedback-p nil)
		   (:final-feedback-p nil)
		   (:items ,(o-formula (gvl :parent :parent
					    :result-list)))
		   )
	 ))
       )
      ))
   )
  (push constraint-check-results *all-dboxes*)
  )

;;;------------------------------------------------------------
;;; create-box-creation-dboxes creates the dialog boxes for examining
;;; the box creation list
;;;------------------------------------------------------------
(defun create-box-creation-dboxes ()
  (create-instance
   'box-creation-current-box-dbox opal:aggregadget
   (:visible T)
   (:draw-function :xor)
   (:left 0) (:top 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:current-box nil)
   (:current-role nil)
   (:box-description nil)
   (:my-help-string "Press \"Show Box List\"
to see the list of boxes,
\"Resume Editing\" to
continue editing.")

   (:generate-item-string
    #'(lambda (item)
	(let ((name (car item))
	      (sysname (cdr item))
	      )
	  (format nil "~A~A" name
		  (if sysname (format nil "  (sysname = ~S)" sysname)
		    "")))))

   (:display-me #'(lambda (obj)
		    (block-interference
		     :help-msg "")
		    (s-value obj :current-box
			     (if (eq *box-creation-list* :file)
				(cdar *file-boxes-to-add*)
				(car *user-boxes-to-add*)))
		    (s-value obj :current-role *box-creation-list*)
		    (s-value obj :box-description "Current Box To Create:")
		    (call-schema obj :show-me)
		    ))
   (:show-me #'(lambda (obj)
		 (set-help-string (g-value obj :my-help-string))
		 (let ((menu (g-value obj :menu :inside-list)))
		   (s-value menu :items
			    (when (listp (third (g-value obj
							 :current-box)))
				  (mapcar (g-value obj
						   :generate-item-string)
					  (third (g-value obj
							  :current-box)))))
		   (opal:notice-items-changed
		    (g-value menu :menu-item-list)))
		 (let ((menu (g-value obj :menu :contains-list)))
		   (s-value menu :items
			    (when (listp (fourth (g-value obj
							  :current-box)))
				  (mapcar (g-value obj
						   :generate-item-string)
					  (fourth (g-value obj
							   :current-box)))))
		   (opal:notice-items-changed
		    (g-value menu :menu-item-list)))
		 (add-dbox obj)
		 (opal:deiconify-window dialog-window)
		 (opal:raise-window dialog-window)
		 ))

   (:undisplay-me #'(lambda (obj)
		      (remove-dbox obj)
		      (opal:lower-window dialog-window)
		      (allow-interference)
		      ))

   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu)))
	      )
      (:menu
       ,opal:aggregadget
       (:parts
	((:done ,command-button-panel
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula (gv *fonts-and-styles*
				      :db-frame-width)))
		(:direction :horizontal)
		(:items (("Resume Editing"
			  ,#'(lambda (&rest args)
			      (declare (ignore args))
			      (call-schema
			       box-creation-current-box-dbox
			       :undisplay-me)))
			 ("Show Box List"
			  ,#'(lambda (&rest args)
			       (declare (ignore args))
			       (remove-dbox box-creation-current-box-dbox)
			       (call-schema box-creation-list-dbox
					    :show-me)))
			 ))
		)
	 (:label ,opal:text
		 (:font ,(o-formula (gv *fonts-and-styles*
					:label-font)))
		 (:top ,(o-formula (+ (gvl :parent :done :top)
				      (gvl :parent :done :height)
				      15)))
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:string ,(o-formula (gvl :parent :parent
					   :box-description)))
		 )
	 (:name ,opal:text
		(:font ,(o-formula (gv *fonts-and-styles*
				       :label-font)))
		(:top ,(o-formula (+ (gvl :parent :label :top)
				     (gvl :parent :label :height)
				     10)))
		(:left ,(o-formula (+ 5 (gv *fonts-and-styles*
					    :db-frame-width))))
		(:string ,(o-formula (format nil "Name: ~S"
					     (first (gvl :parent
							 :parent
							 :current-box)))))
		)
	 (:role ,opal:text
		(:font ,(o-formula (gv *fonts-and-styles*
				       :label-font)))
		(:top ,(o-formula (+ (gvl :parent :name :top)
				     (gvl :parent :name :height)
				     10)))
		(:left ,(o-formula (+ 5 (gv *fonts-and-styles*
					    :db-frame-width))))
		(:string ,(o-formula
			   (format nil "Role: ~S"
				   (if (eq (gvl :parent :parent
						:current-role)
					   :file)
				       "file" "user"))))
		)
	 (:type ,opal:text
		(:font ,(o-formula (gv *fonts-and-styles*
				       :label-font)))
		(:top ,(o-formula (+ (gvl :parent :role :top)
				     (gvl :parent :role :height)
				     10)))
		(:left ,(o-formula (+ 5 (gv *fonts-and-styles*
					    :db-frame-width))))
		(:string ,(o-formula
			   (format nil "Type: ~S"
				   (second (gvl :parent :parent
						:current-box)))))
		)

	 (:inside-string
	  ,opal:text
	  (:visible ,(o-formula (and (gvl :parent :parent :visible)
				     (or
				      (not (third (gvl :parent :parent
						       :current-box)))
				      (not
				       (listp (third (gvl :parent
							  :parent
							  :current-box))))
				      ))))
	  (:font ,(o-formula (gv *fonts-and-styles*
				 :label-font)))
	  (:top ,(o-formula (+ (gvl :parent :type :top)
			       (gvl :parent :type :height)
			       10)))
	  (:left ,(o-formula (+ 5 (gv *fonts-and-styles*
				      :db-frame-width))))
	  (:string
	   ,(o-formula
	     (if (gvl :visible)
		 (if (third (gvl :parent :parent :current-box))
		     (format
		      nil
		      "Box may not be inside anything except boxes of role ~S."
		      (if (eq (third (gvl :parent :parent :current-box))
			      :file) "file" "user"))
		   "Box may not be inside anything.")
	       "")))
	  )
	 (:inside-list ,garnet-gadgets:scrolling-menu
		       (:left ,(o-formula (+ 5 (gv *fonts-and-styles*
						   :db-frame-width))))
		       (:top ,(o-formula (gvl :parent :inside-string :top)))
		       (:visible ,(o-formula (not (gvl :parent
						       :inside-string
						       :visible))))
		       (:num-visible 5)
		       (:item-font ,(o-formula (gv *fonts-and-styles*
						   :label-font)))
		       (:title-font ,(o-formula (gv *fonts-and-styles*
						    :label-font)))
		       (:title "Box must be inside:")
		       (:int-menu-feedback-p nil)
		       (:final-feedback-p nil)
		       (:items nil)
		       )

	 (:contains-string
	  ,opal:text
	  (:visible ,(o-formula (and (gvl :parent :parent :visible)
				     (or
				      (not (fourth (gvl :parent
							:parent
							:current-box)))
				      (not (listp (fourth (gvl :parent
							       :parent
							       :current-box))))
				      ))))
	  (:font ,(o-formula (gv *fonts-and-styles*
				 :label-font)))
	  (:top ,(o-formula (+ (gvl :parent :inside-string :top)
			       (if (gvl :parent :inside-string
					:visible)
				   (gvl :parent :inside-string :height)
				 (gvl :parent :inside-list :height))
			       10)))
	  (:left ,(o-formula (+ 5 (gv *fonts-and-styles*
				      :db-frame-width))))
	  (:string
	   ,(o-formula
	     (if (gvl :visible)
		 (if (fourth (gvl :parent :parent :current-box))
		     (format
		      nil
		      "Box may only contain boxes of role ~S."
		      (if (eq (fourth (gvl :parent :parent
					   :current-box)) :file)
			  "file" "user"))
		   "Box may not contain anything.")
	       "")))
	  )
	 (:contains-list ,garnet-gadgets:scrolling-menu
			 (:left ,(o-formula (+ 5 (gv *fonts-and-styles*
						     :db-frame-width))))
			 (:top ,(o-formula (gvl :parent
						:contains-string :top)))
			 (:visible ,(o-formula (not (gvl :parent
							 :contains-string
							 :visible))))
			 (:num-visible 5)
			 (:item-font ,(o-formula (gv *fonts-and-styles*
						     :label-font)))
			 (:title-font ,(o-formula (gv *fonts-and-styles*
						      :label-font)))
			 (:title "Box must contain:")
			 (:int-menu-feedback-p nil)
			 (:final-feedback-p nil)
			 (:items nil)
			 )
	 ))
       )
      ))
   )
  (push box-creation-current-box-dbox *all-dboxes*)

  (create-instance
   'box-creation-list-dbox opal:aggregadget
   (:visible T)
   (:draw-function :xor)
   (:left 0) (:top 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:update-menus T)
   (:my-help-string "Select a box to display
that box.  Press
\"Resume Editing\"
when you are done.")

   (:display-me #'(lambda (obj)
		    (block-interference :help-msg "")
		    (s-value obj :update-menus T)
		    (call-schema obj :show-me)))
   (:show-me #'(lambda (obj)
		 (add-dbox obj)
		 (set-help-string (g-value obj :my-help-string))
		 (when (g-value obj :update-menus)
		       (let ((menu (g-value obj :menu :user-menu)))
			 (s-value menu :items
				  (mapcar #'car
					  *user-boxes-to-add*))
			 (s-value menu :num-visible
				  (min-max (length *user-boxes-to-add*)
					   5 10))
			 (opal:notice-items-changed
			  (g-value menu :menu-item-list)))
		       (let ((menu (g-value obj :menu :file-menu)))
			 (s-value menu :items
				  (mapcar #'car
					  *file-boxes-to-add*))
			 (s-value menu :num-visible
				  (min-max (length *file-boxes-to-add*)
					   5 10))
			 (opal:notice-items-changed
			  (g-value menu :menu-item-list))
			 )
		       (s-value (g-value obj :menu :role-string)
				:string
				(format nil
					"Currently creating ~A boxes."
					(if (eq *box-creation-list*
						:file)
					    "file" "user")))
		       )
		 (opal:deiconify-window dialog-window)
		 (opal:raise-window dialog-window)
		 ))
   (:undisplay-me #'(lambda (obj)
		      (remove-dbox obj)
		      (opal:lower-window dialog-window)
		      (allow-interference)
		      ))

   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu)))
	      )
      (:menu
       ,opal:aggregadget
       (:parts
	((:done ,command-button-panel
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula (gv *fonts-and-styles*
				      :db-frame-width)))
		(:direction :horizontal)
		(:items (("Resume Editing"
			  ,#'(lambda (&rest args)
			       (declare (ignore args))
			       (call-schema
				box-creation-list-dbox
				:undisplay-me)))
			 ))
		)
	 (:role-string ,opal:text
		       (:top ,(o-formula (+ (gvl :parent :done :top)
					    (gvl :parent :done :height)
					    15)))
		       (:left ,(o-formula (gv *fonts-and-styles*
					      :db-frame-width)))
		       (:font ,(o-formula (gv *fonts-and-styles*
					      :label-font)))
		       (:string "")
		       )
	 (:file-menu ,garnet-gadgets:scrolling-menu
		     (:left ,(o-formula (gv *fonts-and-styles*
					    :db-frame-width)))
		     (:top ,(o-formula
			     (+ (gvl :parent :role-string :top)
				(gvl :parent :role-string :height)
				10)))
		     (:visible ,(o-formula
				 (and (gvl :parent :parent :visible)
				      (gvl :items))))
		     (:num-visible 10)
		     (:item-font ,(o-formula (gv *fonts-and-styles*
						 :label-font)))
		     (:title-font ,(o-formula (gv *fonts-and-styles*
						  :label-font)))
		     (:title "File Boxes To Create:")
		     (:multiple-p nil)
		     (:items nil)
		     (:menu-selection-function
		      ,#'(lambda (obj nv)
			   (declare (ignore nv))
			   (let ((box (car (g-value obj
						    :selected-ranks))))
			     (s-value obj :selected-ranks nil)
			     (s-value box-creation-current-box-dbox
				      :current-box
				      (cdr (nth box
						*file-boxes-to-add*)))
			     (remove-dbox box-creation-list-dbox)
			     (call-schema
			      box-creation-current-box-dbox :show-me)
			     )))
		     )
	 (:user-menu ,garnet-gadgets:scrolling-menu
		     (:left ,(o-formula (gv *fonts-and-styles*
					    :db-frame-width)))
		     (:top ,(o-formula
			     (+ (gvl :parent :file-menu :top)
				(if (gvl :parent :file-menu :visible)
				    (+ (gvl :parent :file-menu
					    :height) 10)
				  0))))
		     (:visible ,(o-formula
				 (and (gvl :parent :parent :visible)
				      (gvl :items))))
		     (:num-visible 10)
		     (:item-font ,(o-formula (gv *fonts-and-styles*
						 :label-font)))
		     (:title-font ,(o-formula (gv *fonts-and-styles*
						  :label-font)))
		     (:title "User Boxes To Create:")
		     (:multiple-p nil)
		     (:items nil)
		     (:menu-selection-function
		      ,#'(lambda (obj nv)
			   (declare (ignore nv))
			   (let ((box (car (g-value obj
						    :selected-ranks))))
			     (s-value obj :selected-ranks nil)
			     (s-value box-creation-current-box-dbox
				      :current-box
				      (nth box *user-boxes-to-add*))
			     (remove-dbox box-creation-list-dbox)
			     (call-schema
			      box-creation-current-box-dbox :show-me)
			     )))
		     )
	 ))
       )
      ))
   )
  (push box-creation-list-dbox *all-dboxes*)
  )


;;;------------------------------------------------------------
;;; create-probe-dbox creates a dialog box for specifying prober
;;; options.
;;;------------------------------------------------------------
(defun create-probe-dbox ()
  (create-instance
   'probe-dbox opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:fast-redraw-p T)
   (:visible T)

   ;; the directory to probe
   (:directory (o-formula (gvl :menu :directory :value)))
   
   ;; where to save the group file
   (:group-output (o-formula (gvl :menu :group-output :value)))

   ;; where to save the user file
   (:user-output (o-formula (gvl :menu :user-output :value)))

   ;; where to send the output
   (:output-file (o-formula (gvl :menu :output-file :value)))

   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu)))
	      (:where :back)
	      )
      (:menu
       ,miro-aggregadget
       (:parts
	((:reset ,dbox-reset-button
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:top ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width))))
	 (:directory ,dbox-filename-selector
		     (:left ,(o-formula (gv *fonts-and-styles*
					    :db-frame-width)))
		     (:top ,(o-formula (+ (gvl :parent :reset :top)
					  (gvl :parent :reset
					       :height)
					  10)))
		     (:title-string "Directory to probe:")
		     (:default-value "")
		     )
	 (:output-file ,dbox-filename-selector
		       (:left ,(o-formula (gv *fonts-and-styles*
					      :db-frame-width)))
		       (:top ,(o-formula (+ (gvl :parent :directory
						 :top)
					    (gvl :parent :directory
						 :height)
					    10)))
		       (:title-string
			"Output file (leave blank for workbench):")
		       (:default-value "")
		       )
	 (:group-output ,dbox-filename-selector
			(:left ,(o-formula (gv *fonts-and-styles*
					       :db-frame-width)))
			(:top ,(o-formula (+ (gvl :parent :output-file
						  :top)
					     (gvl :parent :output-file
						  :height)
					     10)))
			(:title-string
			 "Group output file (leave blank to ignore):")
			(:default-value "")
			)
	 (:user-output ,dbox-filename-selector
		       (:left ,(o-formula (gv *fonts-and-styles*
					      :db-frame-width)))
		       (:top ,(o-formula (+ (gvl :parent :group-output
						 :top)
					    (gvl :parent :group-output
						 :height)
					    10)))
		       (:title-string
			"User output file (leave blank to ignore):")
		       (:default-value "")
		       )
	 (:done ,command-button-panel
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula (+ (gvl :parent :user-output :top)
				     (gvl :parent :user-output
					  :height)
				     10)))
		(:direction :horizontal)
		(:items (("Run Prober" ,#'do-probe)
			 ("Abort" ,#'(lambda (&rest args)
				       (declare (ignore args))
				       (remove-dbox probe-dbox)
				       (opal:lower-window dialog-window)
				       (allow-interference)
				       (inter:beep)
				       ))
			 ))
		)
	 ))
       )
      ))
   )
  (push probe-dbox *all-dboxes*)
  
  (define-method :initialize-options probe-dbox (obj)
    (add-dbox obj)
    (opal:deiconify-window dialog-window)
    (opal:raise-window dialog-window)
    )
  )

;;;------------------------------------------------------------
;;; create-verify-dbox creates a dialog box for specifying verify
;;; options.
;;;------------------------------------------------------------
(defun create-verify-dbox ()
  (create-instance
   'verify-dbox miro-aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:fast-redraw-p T)
   (:visible T)

   ;; the iff file to verify -- empty string = workbench
   (:iff-file (o-formula (gvl :main-menu :iff-name :value)))

   ;; the directory to verify
   (:directory (o-formula (gvl :main-menu :directory :value)))

   ;; the directory to use for temporary files
   (:tmp-dir (o-formula (gvl :main-menu :tmp-dir :value)))

   ;; the log file
   (:log-file (o-formula (gvl :main-menu :log-file :value)))

   ;; ambig and prober files
   (:ambig-user-file (o-formula (gvl :ambig-menu :user-name :value)))
   (:ambig-perm-file (o-formula (gvl :ambig-menu :perm-name :value)))
   (:ambig-rels-file (o-formula (gvl :ambig-menu :rels-name :value)))
   (:probe-user-file (o-formula (gvl :prober-menu :user-name :value)))
   (:probe-group-file (o-formula (gvl :prober-menu :group-name :value)))
   (:probe-perm-file (o-formula (gvl :prober-menu :perm-name :value)))

   ;; should we run ambig/probe?
   (:run-ambig (o-formula (gvl :ambig-menu :run :value)))
   (:run-probe (o-formula (gvl :prober-menu :run :value)))

   (:initialize-options
    #'(lambda ()
	(add-dbox verify-dbox)
	(s-value (g-value verify-dbox :main-menu) :visible T)
	(s-value (g-value verify-dbox :ambig-menu) :visible nil)
	(s-value (g-value verify-dbox :prober-menu) :visible nil)
	(opal:deiconify-window dialog-window)
	(opal:raise-window dialog-window)
	))

   (:parts
    `((:frame ,opal:rectangle
	      (:line-style ,(o-formula (gv *fonts-and-styles*
					   :db-frame-style)))
	      (:filling-style ,(o-formula (gv *colors* :white)))
	      (:where :back)
	      (:top ,(o-formula (gvl :parent :top)))
	      (:left ,(o-formula (gvl :parent :left)))
	      (:width ,(o-formula
			(+ (or (and (gvl :parent :main-menu :visible)
				    (gvl :parent :main-menu :width))
			       (and (gvl :parent :ambig-menu :visible)
				    (gvl :parent :ambig-menu :width))
			       (and (gvl :parent :prober-menu :visible)
				    (gvl :parent :prober-menu :width))
			       0)
			   (* 2 (gv *fonts-and-styles*
				    :db-frame-width)))))
	      (:height ,(o-formula
			 (+ (or (and (gvl :parent :main-menu :visible)
				     (gvl :parent :main-menu :height))
				(and (gvl :parent :ambig-menu :visible)
				     (gvl :parent :ambig-menu :height))
				(and (gvl :parent :prober-menu :visible)
				     (gvl :parent :prober-menu :height))
				0)
			    (* 2 (gv *fonts-and-styles*
				     :db-frame-width)))))
	      )

      (:main-menu
       ,miro-aggregadget
       (:responsible-for-parent T)
       (:visible nil)
       (:parts
	((:reset ,dbox-reset-button
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:top ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width))))
	 (:iff-name ,dbox-filename-selector
		    (:left ,(o-formula (gv *fonts-and-styles*
					   :db-frame-width)))
		    (:top ,(o-formula (+ (gvl :parent :reset :top)
					 (gvl :parent :reset :height)
					 10)))
		    (:title-string
		     "Iff file to verify (leave blank for workbench):")
		    (:default-value "")
		    )
	 (:directory ,dbox-filename-selector
		     (:left ,(o-formula (gv *fonts-and-styles*
					    :db-frame-width)))
		     (:top ,(o-formula (+ (gvl :parent :iff-name :top)
					  (gvl :parent :iff-name :height)
					  10)))
		     (:title-string "Directory to verify:")
		     (:default-value "")
		     )
	 (:tmp-dir ,dbox-filename-selector
		   (:left ,(o-formula (gv *fonts-and-styles*
					  :db-frame-width)))
		   (:top ,(o-formula (+ (gvl :parent :directory :top)
					(gvl :parent :directory :height)
					10)))
		   (:title-string "Directory to use for temporary files:")
		   (:default-value "/tmp")
		   )
	 (:log-file ,dbox-filename-selector
		    (:left ,(o-formula (gv *fonts-and-styles*
					   :db-frame-width)))
		    (:top ,(o-formula (+ (gvl :parent :tmp-dir :top)
					 (gvl :parent :tmp-dir :height)
					 10)))
		    (:title-string "Log file:")
		    (:default-value "")
		    )
	 (:other-menus ,command-button-panel
		       (:left ,(o-formula (gv *fonts-and-styles*
					      :db-frame-width)))
		       (:top ,(o-formula (+ (gvl :parent :log-file :top)
					    (gvl :parent :log-file :height)
					    10)))
		       (:direction :horizontal)
		       (:items (("Show Ambig Menu"
				 ,#'(lambda (&rest args)
				      (declare (ignore args))
				      (s-value (g-value verify-dbox
							:main-menu)
					       :visible nil)
				      (s-value (g-value verify-dbox
							:ambig-menu)
					       :visible T)
				      (opal:update dialog-window)))
				("Show Prober Menu"
				 ,#'(lambda (&rest args)
				      (declare (ignore args))
				      (s-value (g-value verify-dbox
							:main-menu)
					       :visible nil)
				      (s-value (g-value verify-dbox
							:prober-menu)
					       :visible T)
				      (opal:update dialog-window)))
				))
		       )
	 (:done ,command-button-panel
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula (+ (gvl :parent :other-menus :top)
				     (gvl :parent :other-menus :height)
				     10)))
		(:direction :horizontal)
		(:items (("Run Verifier" ,#'do-verify)
			 ("Abort" ,#'(lambda (&rest args)
				       (declare (ignore args))
				       (remove-dbox verify-dbox)
				       (opal:lower-window dialog-window)
				       (allow-interference)
				       (inter:beep)
				       ))
			 ))
		)
	 ))
       )
      
      (:ambig-menu
       ,miro-aggregadget
       (:visible nil)
       (:parts
	((:reset ,command-button-panel
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:top ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		 (:items (("Show Main Menu"
			   ,#'(lambda (&rest args)
				(declare (ignore args))
				(s-value (g-value verify-dbox
						  :ambig-menu)
					 :visible nil)
				(s-value (g-value verify-dbox
						  :main-menu) :visible
						  T)
				(opal:update dialog-window)
				))
			  ))
		 )
	 (:run ,dbox-labeled-button-input
	       (:left ,(o-formula (gv *fonts-and-styles*
				      :db-frame-width)))
	       (:top ,(o-formula (+ (gvl :parent :reset :top)
				    (gvl :parent :reset :height)
				    10)))
	       (:direction :vertical)
	       (:title-string "Ambiguity Checker files will be...")
	       ;; first item in the list is the default
	       (:items
		("... overwritten by Ambiguity Checker output"
		 "... used in place of Ambiguity Checker output"
		 ))
	       (:value-list (T nil))
	       (:default-value T)
	       )
	 (:user-name ,dbox-filename-selector
		     (:left ,(o-formula (gv *fonts-and-styles*
					    :db-frame-width)))
		     (:top ,(o-formula (+ (gvl :parent :run :top)
					  (gvl :parent :run :height)
					  10)))
		     (:title-string "Ambiguity Checker Users File:")
		     (:default-value "")
		     )
	 (:perm-name ,dbox-filename-selector
		     (:left ,(o-formula (gv *fonts-and-styles*
					    :db-frame-width)))
		     (:top ,(o-formula (+ (gvl :parent :user-name :top)
					  (gvl :parent :user-name :height)
					  10)))
		     (:title-string "Ambiguity Checker Permissions File:")
		     (:default-value "")
		     )
	 (:rels-name ,dbox-filename-selector
		     (:left ,(o-formula (gv *fonts-and-styles*
					    :db-frame-width)))
		     (:top ,(o-formula (+ (gvl :parent :perm-name :top)
					  (gvl :parent :perm-name :height)
					  10)))
		     (:title-string "Ambiguity Checker Relations File:")
		     (:default-value "")
		     )
	 ))
       )

      (:prober-menu
       ,miro-aggregadget
       (:visible nil)
       (:parts
	((:reset ,command-button-panel
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:top ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		 (:items (("Show Main Menu"
			   ,#'(lambda (&rest args)
				(declare (ignore args))
				(s-value (g-value verify-dbox
						  :prober-menu)
					 :visible nil)
				(s-value (g-value verify-dbox
						  :main-menu) :visible
						  T)
				(opal:update dialog-window)
				))
			  ))
		 )
	 (:run ,dbox-labeled-button-input
	       (:left ,(o-formula (gv *fonts-and-styles*
				      :db-frame-width)))
	       (:top ,(o-formula (+ (gvl :parent :reset :top)
				    (gvl :parent :reset :height)
				    10)))
	       (:direction :vertical)
	       (:title-string "Prober files will be...")
	       ;; first item in the list is the default
	       (:items
		("...overwritten by Prober output"
		 "...used in place of Prober output"
		 ))
	       (:value-list (T nil))
	       (:default-value T)
	       )
	 (:user-name ,dbox-filename-selector
		     (:left ,(o-formula (gv *fonts-and-styles*
					    :db-frame-width)))
		     (:top ,(o-formula (+ (gvl :parent :run :top)
					  (gvl :parent :run :height)
					  10)))
		     (:title-string "Prober Users File:")
		     (:default-value "")
		     )
	 (:group-name ,dbox-filename-selector
		      (:left ,(o-formula (gv *fonts-and-styles*
					     :db-frame-width)))
		      (:top ,(o-formula (+ (gvl :parent :user-name :top)
					   (gvl :parent :user-name :height)
					   10)))
		      (:title-string "Prober Group File:")
		      (:default-value "")
		      )
	 (:perm-name ,dbox-filename-selector
		     (:left ,(o-formula (gv *fonts-and-styles*
					    :db-frame-width)))
		     (:top ,(o-formula (+ (gvl :parent :group-name :top)
					  (gvl :parent :group-name :height)
					  10)))
		     (:title-string "Prober Permissions File:")
		     (:default-value "")
		     )
	 ))
       )
      ))
   )
  (push verify-dbox *all-dboxes*)

  (create-instance
   'verify-result-menu opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:visible T)
   (:items nil)

   (:undisplay-me #'(lambda (obj &rest args)
		      (declare (ignore args))
		      (remove-dbox obj)
		      (opal:lower-window dialog-window)
		      (allow-interference)
		      ))
   (:display-me
    #'(lambda (obj)
	(call-schema (g-value verify-update-menu :menu :iff-file)
		     :set-value (g-value obj :iff-file))
	(opal:notice-items-changed (g-value obj :menu
					    :results :menu-item-list))
	(add-dbox obj)
	(s-value verify-update-menu :displaying-me nil)
	;;(s-value verify-group-menu :displaying-me nil)
	(opal:deiconify-window dialog-window)
	(opal:raise-window dialog-window)
	(opal:update dialog-window)
	))

   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu)))
	      (:where :back)
	      )
      (:menu
       ,opal:aggregadget
       (:parts
	((:buttons ,command-button-panel
		   (:left ,(o-formula (gv *fonts-and-styles*
					  :db-frame-width)))
		   (:top ,(o-formula (gv *fonts-and-styles*
					 :db-frame-width)))
		   (:direction :horizontal)
		   (:items (("Resume Editing"
			     ,#'(lambda (obj &rest args)
				  (declare (ignore args))
				  (call-schema (g-value obj :parent
							:parent)
					       :undisplay-me)
				  ))
			    ("Update Iff File"
			     ,#'(lambda (obj &rest args)
				  (declare (ignore args))
				  (remove-dbox
				   (g-value obj :parent :parent))
				  (call-schema verify-update-menu
					       :display-me)
				  ))
			    ))
		   )
	 (:results ,garnet-gadgets:scrolling-menu
		   (:left ,(o-formula (gv *fonts-and-styles*
					  :db-frame-width)))
		   (:top ,(o-formula (+ (gvl :parent :buttons :top)
					(gvl :parent :buttons :height)
					10)))
		   (:num-visible 10)
		   (:item-font ,(o-formula (gv *fonts-and-styles*
					       :label-font)))
		   (:title "Verifier Results")
		   (:title-font ,(o-formula (gv *fonts-and-styles*
						:button-label-font)))
		   (:int-menu-feedback-p nil)
		   (:final-feedback-p nil)
		   (:items ,(o-formula (gvl :parent :parent :items)))
		   )
	       )))
      ))
   )
  (push verify-result-menu *all-dboxes*)

  (create-instance
   'verify-update-menu opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:visible T)
   (:workbench nil)
   (:prober-results nil)
   (:file-box-deletion-list nil)
   (:user-box-deletion-list nil)
   (:arrow-deletion-list nil)
   (:file-box-addition-list nil)
   (:user-box-addition-list nil)
   (:wb-user-arrows nil)
   (:wb-file-arrows nil)
   (:pr-user-arrows nil)
   (:pr-file-arrows nil)
   (:arrow-addition-list nil)
   (:perm-change-list nil)
   (:iff-file "")
   (:displaying-me nil)

   (:display-me #'(lambda (obj &rest args)
		    (declare (ignore args))
		    (s-value obj :displaying-me T)
		    ;;(s-value verify-group-menu :displaying-me nil)

		    (let ((menu (g-value obj :menu :delete-box-menu)))
		      (s-value menu :items
			       (mapcar #'car
				       (g-value obj
						:file-box-deletion-list)))
		      (opal:notice-items-changed
		       (g-value menu :menu-item-list))
		      )

		    (let ((menu (g-value obj :menu :add-box-menu)))
		      (s-value menu :items
			       (mapcar #'car
				       (g-value
					obj :file-box-addition-list)))
		      (opal:notice-items-changed
		       (g-value menu :menu-item-list))
		      )

		    (let ((menu (g-value obj :menu
					 :delete-user-menu)))
		      (s-value menu :items
			       (mapcar #'car
				       (g-value
					obj
					:user-box-deletion-list)))
		      (opal:notice-items-changed
		       (g-value menu :menu-item-list))
		      )

		    (let ((menu (g-value obj :menu :add-user-menu)))
		      (s-value menu :items
			       (g-value obj :user-box-addition-list))
		      (opal:notice-items-changed
		       (g-value menu :menu-item-list))
		      )

		    (let ((menu (g-value obj :menu :change-perms-menu)))
		      (s-value menu :items
			       (mapcar #'(lambda (l)
					   (format nil "user: ~A    file: ~A"
						   (car l) (cadr l)))
				       (g-value obj :perm-change-list)))
		      (opal:notice-items-changed
		       (g-value menu :menu-item-list))
		      )

		    (let ((menu (g-value obj :menu :change-perms-user-menu)))
		      (s-value menu :items
			       (sort (remove-duplicates
				      (mapcar #'car
					      (g-value obj :perm-change-list))
				      :test #'string=)
				     #'string<))
		      (opal:notice-items-changed
		       (g-value menu :menu-item-list))
		      )

		    (let ((menu (g-value obj :menu :change-perms-file-menu)))
		      (s-value menu :items
			       (sort (remove-duplicates
				      (mapcar #'cadr
					      (g-value obj :perm-change-list))
				      :test #'string=)
				     #'string<))
		      (opal:notice-items-changed
		       (g-value menu :menu-item-list))
		      )

		    (add-dbox obj)
		    (opal:deiconify-window dialog-window)
		    (opal:raise-window dialog-window)
		    (opal:update dialog-window)
		    ))

   (:undisplay-me #'(lambda (obj &rest args)
		      (declare (ignore args))
		      (remove-dbox obj)
		      (opal:lower-window dialog-window)
		      (allow-interference)
		      ))
   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu))))
      (:menu ,opal:aggregadget
	     (:visible ,(o-formula (gvl :parent :visible)))
	     (:parts
	      ((:resume-button
		,command-button-panel
		(:top ,(o-formula (gv *fonts-and-styles*
				      :db-frame-width)))
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:direction :horizontal)
		(:rank-margin 3)
		(:items (("Resume Editing" ,#'(lambda (obj &rest args)
						(declare (ignore args))
						(call-schema
						 (g-value obj :parent
							  :parent)
						 :undisplay-me)
						))
			 ("Show Results" ,#'(lambda (obj &rest args)
					      (declare (ignore args))
					      (remove-dbox
					       (g-value obj :parent
							:parent))
					      (call-schema
					       verify-result-menu
					       :display-me)
					      ))
#|
			 ("Show Group Differences"
			  ,#'(lambda (obj &rest args)
			       (declare (ignore args))
			       (s-value (g-value obj :parent :parent)
					:visible nil)
			       (call-schema verify-group-menu
					    :display-me)
			       ))
|#
			 ))
		)

	       (:iff-button
		,command-button-panel
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula (+ (gvl :parent :resume-button :top)
				     (gvl :parent :resume-button :height)
				     10)))
		(:items (("Generate New Iff File From Prober Output"
			  ,#'(lambda (obj &rest args)
			       (declare (ignore args))
			       (let* ((output-to-workbench
				       (null-string (g-value obj
							     :parent
							     :iff-file
							     :value)))
				      (output-file
				       (if output-to-workbench
					   (make-temporary-file
					    "/tmp/Miro.iff")
					 (g-value obj :parent
						  :iff-file :value)))
				      )
				 (apply #'write-iff-from-prober-results
					(cons output-file
					      (g-value verify-update-menu
						       :prober-results)))
				 (when output-to-workbench
				       (do-read output-file T)
				       (delete-file output-file))
				 (call-schema (g-value obj :parent
						       :parent)
					      :undisplay-me)
				 )))))
		)

	       (:iff-file ,dbox-filename-selector
			  (:left ,(o-formula (+ 10 (gv *fonts-and-styles*
						       :db-frame-width))))
			  (:top ,(o-formula (+ (gvl :parent :iff-button :top)
					       (gvl :parent :iff-button :height)
					       5)))
			  (:direction :horizontal)
			  (:title-string "Iff file (leave blank for workbench):")
			  (:default-value "")
			  )

	       (:file-box-text ,opal:text
			       (:left ,(o-formula (gv *fonts-and-styles*
						      :db-frame-width)))
			       (:top ,(o-formula (+ (gvl :parent :iff-file :top)
						    (gvl :parent :iff-file :height)
						    10)))
			       (:visible ,(o-formula
					   (and (gvl :parent :visible)
						(or
						 (gvl :parent
						      :delete-box-menu :visible)
						 (gvl :parent
						      :add-box-menu
						      :visible)))))
			       (:string "File Boxes:")
			       (:font ,(o-formula (gv *fonts-and-styles*
						      :label-font)))
			       )

	       (:delete-box-button
		,command-button-panel
		(:left ,(o-formula (+ 5 (gv *fonts-and-styles*
					    :db-frame-width))))
		(:top ,(o-formula (+ (gvl :parent :file-box-text :top)
				     (gvl :parent :file-box-text :height)
				     5)))
		(:visible ,(o-formula (gvl :parent :delete-box-menu
					   :visible)))
		(:direction :vertical)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (let* ((deleting-all
			      (string-equal nv "Delete All Extra Boxes"))
			     (selected-ranks (unless deleting-all
						     (g-value obj :parent
							      :delete-box-menu
							      :selected-ranks)))
			     (file-box-deletion-list (g-value obj :parent :parent
							      :file-box-deletion-list))
			     (boxes (if deleting-all file-box-deletion-list
				      (mapcar #'(lambda (r)
						  (nth r file-box-deletion-list))
					      selected-ranks)))
			     (sysnames (mapcar #'second boxes))
			     (parent (g-value obj :parent))
			     (grandparent (g-value parent :parent))
			     )
			(when boxes
			      ;; delete the boxes
			      (dolist (o (g-value obj-agg :selected))
				      (s-value o :selected nil))
			      (s-value obj-agg :selected
				       (mapcar #'find-box sysnames))
			      (dolist (b (g-value obj-agg :selected))
				      (s-value b :selected T))
			      (delete-selected-objects)

			      ;; remember old values
			      (setq undo-schema-slots
				    (cons
				     (list grandparent :perm-change-list
					   (copy-list
					    (g-value grandparent :perm-change-list)))
				     (cons
				      (list grandparent :file-box-deletion-list
					    (copy-list
					     (g-value grandparent
						      :file-box-deletion-list)))
				      undo-schema-slots)))
			      (setq undo-functions
				    (cons
				     #'(lambda ()
					 (opal:notice-items-changed
					  (g-value verify-update-menu
						   :menu
						   :delete-box-menu
						   :menu-item-list)))
				     undo-functions))
			      
			      (let ((box-names (mapcar #'car boxes)))
				(s-value grandparent :perm-change-list
					 (remove-if #'(lambda (f)
							(member f box-names
								:test #'string=))
						    (g-value grandparent :perm-change-list)
						    :key #'cadr))
				)

			      (s-value grandparent
				       :file-box-deletion-list
				       (unless deleting-all
					       (sort (set-difference
						      file-box-deletion-list
						      boxes :key #'second)
						     #'string< :key #'car)))
			      (s-value (g-value parent :delete-box-menu)
				       :selected-ranks nil)
			      (s-value (g-value parent :delete-box-menu)
				       :items (mapcar #'car
						      (g-value
						       grandparent
						       :file-box-deletion-list)))
			      (call-schema grandparent :display-me)
			      )
			)
		      ))
		(:items ("Delete Selected Boxes"
			 "Delete All Extra Boxes"))
		)

	       (:delete-box-menu
		,garnet-gadgets:scrolling-menu
		(:left ,(o-formula (+ 15 (gv *fonts-and-styles*
					     :db-frame-width))))
		(:top ,(o-formula (+ (gvl :parent :delete-box-button :top)
				     (if (gvl :visible)
					 (+ (gvl :parent :delete-box-button :height)
					    5)
				       0))))
		(:visible ,(o-formula (and (gvl :parent :visible)
					   (gvl :items))))
		(:num-visible 5)
		(:multiple-p T)
		(:item-font ,(o-formula (gv *fonts-and-styles*
					    :label-font)))
		(:title nil)
		(:title-font ,(o-formula (gv *fonts-and-styles*
					     :microscopic-font)))
		(:items nil)
		(:menu-selection-function
		 ,#'(lambda (obj nv)
		      (let ((selected-ranks (g-value obj :selected-ranks))
			    (new-selections nil)
			    (rank (g-value nv :rank))
			    (items (g-value obj :items))
			    (item-objs (get-values
					(g-value obj :menu-item-list)
					:components))
			    )
			(when (position rank selected-ranks)
			      (dolist
			       (file
				(third
				 (nth rank
				      (g-value obj :parent :parent
					       :file-box-deletion-list))))
			       (let* ((new-rank (position file items
							  :test #'equal))
				      (new-obj (when new-rank
						     (nth new-rank item-objs)))
				      )
				 (unless (or (not new-rank)
					     (position new-rank selected-ranks))
					 (push new-rank selected-ranks)
					 (push new-obj new-selections)
					 ))
			       )
			      (when new-selections
				    (s-value obj :selected-ranks selected-ranks)
				    (dolist (nv new-selections)
					    (call-schema obj
							 :menu-selection-function nv))
				    )
			      ))
		      ))
		)

	       (:add-box-button
		,command-button-panel
		(:left ,(o-formula
			 (if (gvl :parent :delete-box-menu :visible)
			     (+ 55 (max (gvl :parent
					     :delete-box-button
					     :width)
					(+ 10
					   (gvl :parent
						:delete-box-menu
						:width))))
			   (+ 5 (gv *fonts-and-styles*
				    :db-frame-width)))))
		(:top ,(o-formula (gvl :parent :delete-box-button :top)))
		(:visible ,(o-formula (gvl :parent :add-box-menu
					   :visible)))
		(:direction :vertical)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (let* ((adding-all
			      (string-equal nv "Add All Missing Boxes"))
			     (selected-ranks (unless adding-all
						     (g-value obj
							      :parent
							      :add-box-menu
							      :selected-ranks)))
			     (file-box-addition-list (g-value obj :parent
							      :parent
							      :file-box-addition-list))
			     (boxes (if adding-all file-box-addition-list
				      (mapcar #'(lambda (r)
						  (nth r file-box-addition-list))
					      selected-ranks)))
			     )
			(setq
			 *file-boxes-to-add*
			 (sort
			  (union
			   *file-boxes-to-add*
			   (mapcar
			    #'(lambda (b)
				(let* ((name (car b))
				       (slash
					(position #\/
						  (string-right-trim
						   (list #\/) name)
						  :from-end T))
				       )
				  (list name
					(if slash (subseq name (+ 1 slash))
					  name)
					"file"
					(list (second b)))
				  )) boxes)
			   :key #'car :test #'equal)
			  #'string< :key #'car))
			(setq *box-creation-list* :file)
			(s-value (g-value obj :parent :add-box-menu)
				 :selected-ranks nil)
			(prepare-next-box-creation)
			(call-schema verify-update-menu :undisplay-me)
			)
		      ))
		(:items ("Add Selected Boxes" "Add All Missing Boxes"))
		)

	       (:add-box-menu
		,garnet-gadgets:scrolling-menu
		(:left ,(o-formula (+ 10 (gvl :parent :add-box-button :left))))
		(:top ,(o-formula (+ (gvl :parent :add-box-button :top)
				     (if (gvl :visible)
					 (+ (gvl :parent
						 :add-box-button
						 :height) 5)
				       0))))
		(:visible ,(o-formula (and (gvl :parent :visible)
					   (gvl :items))))
		(:num-visible 5)
		(:multiple-p T)
		(:item-font ,(o-formula (gv *fonts-and-styles*
					    :label-font)))
		(:title nil)
		(:title-font ,(o-formula (gv *fonts-and-styles*
					     :microscopic-font)))
		(:items nil)
		(:menu-selection-function
		 ,#'(lambda (obj nv)
		      (let ((selected-ranks (g-value obj :selected-ranks))
			    (rank (g-value nv :rank))
			    (items (g-value obj :items))
			    (item-objs (get-values
					(g-value obj :menu-item-list)
					:components))
			    )
			(when (position rank selected-ranks)
			      (let* ((dir
				      (second
				       (nth rank
					    (g-value obj :parent
						     :parent
						     :file-box-addition-list))))
				     (new-rank (position (car dir) items
							 :test #'equal))
				     (new-obj (when new-rank
						    (nth new-rank item-objs)))
				     )
				(unless (or (cdr dir) (not new-rank)
					    (position new-rank selected-ranks))
					(push new-rank
					      (g-value obj :selected-ranks))
					(call-schema obj
						     :menu-selection-function new-obj)
					)
				)
			      )
			)
		      ))
		)

	       (:user-box-text
		,opal:text
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula
			(if (gvl :parent :file-box-text :visible)
			    (+ (max (gvl :parent
					 :delete-box-menu :top)
				    (gvl :parent
					 :add-box-menu :top))
			       (max (if (gvl :parent
					     :add-box-menu
					     :visible)
					(gvl :parent
					     :add-box-menu :height)
				      0)
				    (if (gvl :parent
					     :delete-box-menu :visible)
					(gvl :parent
					     :delete-box-menu :height)
				      0))
			       10)
			  (gvl :parent :file-box-text :top))))
		(:visible ,(o-formula
			    (and (gvl :parent :visible)
				 (or
				  (gvl :parent :delete-user-menu
				       :visible)
				  (gvl :parent :add-user-menu
				       :visible)))))
		(:string "User Boxes:")
		(:font ,(o-formula (gv *fonts-and-styles*
				       :label-font)))
		)

	       (:delete-user-button
		,command-button-panel
		(:left ,(o-formula (+ 5 (gv *fonts-and-styles*
					    :db-frame-width))))
		(:top ,(o-formula (+ (gvl :parent :user-box-text :top)
				     (gvl :parent :user-box-text
					  :height)
				     5)))
		(:visible ,(o-formula (gvl :parent :delete-user-menu
					   :visible)))
		(:direction :vertical)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (let* ((deleting-all
			      (string-equal nv "Delete All Extra Boxes"))
			     (selected-ranks
			      (unless deleting-all
				      (g-value obj :parent
					       :delete-user-menu
					       :selected-ranks)))
			     (user-box-deletion-list
			      (g-value obj :parent :parent
				       :user-box-deletion-list))
			     (boxes (if deleting-all user-box-deletion-list
				      (mapcar #'(lambda (r)
						  (nth r user-box-deletion-list)
						  ) selected-ranks)))
			     (sysnames (mapcar #'second boxes))
			     (parent (g-value obj :parent))
			     (grandparent (g-value parent :parent))
			     )
			(when boxes
			      ;; delete the boxes
			      (dolist (o (g-value obj-agg :selected))
				      (s-value o :selected nil))
			      (s-value obj-agg :selected
				       (mapcar #'find-box sysnames))
			      (dolist (b (g-value obj-agg :selected))
				      (s-value b :selected T))
			      (delete-selected-objects)

			      ;; remember old values
			      (setq undo-schema-slots
				    (cons
				     (list grandparent :perm-change-list
					   (copy-list
					    (g-value grandparent :perm-change-list)))
				     (cons
				      (list grandparent :user-box-deletion-list
					    (copy-list
					     (g-value grandparent
						      :user-box-deletion-list)))
				      undo-schema-slots)))
			      (setq undo-functions
				    (cons
				     #'(lambda ()
					 (opal:notice-items-changed
					  (g-value verify-update-menu
						   :menu
						   :delete-user-menu
						   :menu-item-list)))
				     undo-functions))

			      (let ((box-names (mapcar #'car boxes)))
				(s-value grandparent :perm-change-list
					 (remove-if #'(lambda (u)
							(member u box-names
								:test #'string=))
						    (g-value grandparent :perm-change-list)
						    :key #'car))
				)

			      (s-value grandparent :user-box-deletion-list
				       (unless deleting-all
					       (sort
						(set-difference
						 user-box-deletion-list
						 boxes :key #'second)
						#'string< :key #'car)))
			      (s-value (g-value parent
						:delete-user-menu)
				       :selected-ranks nil)
			      (s-value (g-value parent :delete-user-menu)
				       :items
				       (mapcar #'car (g-value
						      grandparent
						      :user-box-deletion-list)))
			      (call-schema grandparent :display-me)
			      )
			)))
		(:items ("Delete Selected Boxes"
			 "Delete All Extra Boxes"))
		)

	       (:delete-user-menu
		,garnet-gadgets:scrolling-menu
		(:left ,(o-formula (+ 15 (gv *fonts-and-styles*
					     :db-frame-width))))
		(:top ,(o-formula (+ (gvl :parent :delete-user-button :top)
				     (if (gvl :visible)
					 (+ (gvl :parent
						 :delete-user-button
						 :height) 5)
				       0))))
		(:visible ,(o-formula (and (gvl :parent :visible)
					   (gvl :items))))
		(:num-visible 5)
		(:multiple-p T)
		(:item-font ,(o-formula (gv *fonts-and-styles*
					    :label-font)))
		(:title nil)
		(:title-font ,(o-formula (gv *fonts-and-styles*
					     :microscopic-font)))
		(:items nil)
		;; no menu-selection-function
		)

	       (:add-user-button
		,command-button-panel
		(:left ,(o-formula
			 (if (gvl :parent :delete-user-menu :visible)
			     (+ 55 (max (gvl :parent
					     :delete-user-button
					     :width)
					(+ 10
					   (gvl :parent
						:delete-user-menu
						:width))))
			   (+ 5 (gv *fonts-and-styles*
				    :db-frame-width)))))
		(:top ,(o-formula (gvl :parent :delete-user-button :top)))
		(:visible ,(o-formula (gvl :parent :add-user-menu
					   :visible)))
		(:direction :vertical)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (let* ((adding-all
			      (string-equal nv "Add All Missing Boxes"))
			     (selected-ranks
			      (unless adding-all
				      (g-value obj :parent
					       :add-box-menu
					       :selected-ranks)))
			     (user-box-addition-list
			      (g-value obj :parent :parent
				       :user-box-addition-list))
			     (boxes (if adding-all user-box-addition-list
				      (mapcar #'(lambda (r)
						  (nth r user-box-addition-list))
					      selected-ranks)))
			     )
			(setq *user-boxes-to-add*
			      (sort
			       (union *user-boxes-to-add*
				      (mapcar #'(lambda (b)
						  (list b "user" :user))
					      boxes)
				      :key #'car :test #'equal)
			       #'string< :key #'car))
			(setq *box-creation-list* :user)
			(s-value (g-value obj :parent :add-user-menu)
				 :selected-ranks nil)
			(prepare-next-box-creation)
			(call-schema verify-update-menu :undisplay-me)
			)
		      ))
		(:items ("Add Selected Boxes"
			 "Add All Missing Boxes"))
		)

	       (:add-user-menu
		,garnet-gadgets:scrolling-menu
		(:left ,(o-formula (+ (gvl :parent :add-user-button
					   :left) 10)))
		(:top ,(o-formula (+ (gvl :parent :add-user-button :top)
				     (if (gvl :visible)
					 (+ (gvl :parent
						 :add-user-button
						 :height) 5)
				       0))))
		(:visible ,(o-formula (and (gvl :parent :visible)
					   (gvl :items))))
		(:num-visible 5)
		(:multiple-p T)
		(:item-font ,(o-formula (gv *fonts-and-styles*
					    :label-font)))
		(:title nil)
		(:title-font ,(o-formula (gv *fonts-and-styles*
					     :microscopic-font)))
		(:items nil)
		;; no menu-selection-function
		)

	       (:perms-text
		,opal:text
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula
			(if (gvl :parent :user-box-text :visible)
			    (+ (max (gvl :parent :delete-user-menu :top)
				    (gvl :parent :add-user-menu :top))
			       (max (if (gvl :parent :delete-user-menu
					     :visible)
					(gvl :parent :delete-user-menu
					     :height)
				      0)
				    (if (gvl :parent :add-user-menu
					     :visible)
					(gvl :parent :add-user-menu
					     :height)
				      0))
			       10)
			  (gvl :parent :user-box-text :top))))
		(:visible ,(o-formula
			    (and (gvl :parent :visible)
				 (gvl :parent :change-perms-menu :visible))))
		(:string "User/File pairs with different permissions:")
		(:font ,(o-formula (gv *fonts-and-styles*
				       :label-font)))
		)

	       (:change-perms-button
		,command-button-panel
		(:left ,(o-formula (+ 5 (gv *fonts-and-styles*
					    :db-frame-width))))
		(:top ,(o-formula (+ (gvl :parent :perms-text :top)
				     (gvl :parent :perms-text :height)
				     5)))
		(:visible ,(o-formula (gvl :parent :change-perms-menu :visible)))
		(:direction :vertical)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (let* ((changing-all
			      (string-equal nv "Fix All Permissions"))
			     (selected-ranks
			      (unless changing-all
				      (g-value obj :parent :change-perms-menu
					       :selected-ranks)))
			     (grandpa (g-value obj :parent :parent))
			     (perm-change-list (g-value grandpa :perm-change-list))
			     (pair-list (if changing-all perm-change-list
					  (mapcar #'(lambda (n) (nth n perm-change-list))
						  selected-ranks)))
			     )
			(fix-perms pair-list)
			(s-value (g-value obj :parent :change-perms-menu)
				 :selected-ranks nil)
			(call-schema verify-update-menu :display-me)
			)))
		(:items ("Fix Permissions for Selected User/File Pairs"
			 "Fix All Permissions"))
		)

	       (:change-perms-menu
		,garnet-gadgets:scrolling-menu
		(:left ,(o-formula (+ 15 (gv *fonts-and-styles*
					     :db-frame-width))))
		(:top ,(o-formula (+ (gvl :parent :change-perms-button :top)
				     (if (gvl :visible)
					 (+ (gvl :parent :change-perms-button
						 :height) 5)
				       0))))
		(:visible ,(o-formula (and (gvl :parent :visible)
					   (gvl :items))))
		(:num-visible 5)
		(:multiple-p T)
		(:item-font ,(o-formula (gv *fonts-and-styles*
					    :label-font)))
		(:title nil)
		(:title-font ,(o-formula (gv *fonts-and-styles*
					     :microscopic-font)))
		(:items nil)
		;; no menu-selection-function
		)

	       (:perms-user-text
		,opal:text
		(:left ,(o-formula (+ (max (+ (gvl :parent :perms-text :left)
					      (gvl :parent :perms-text :width))
					   (+ (gvl :parent :change-perms-button
						   :left)
					      (gvl :parent :change-perms-button
						   :width))
					   (+ (gvl :parent :change-perms-menu
						   :left)
					      (gvl :parent :change-perms-menu
						   :width)))
				      10)))
		(:top ,(o-formula (gvl :parent :perms-text :top)))
		(:visible ,(o-formula (gvl :parent :perms-text :visible)))
		(:string "Users with different permissions:")
		(:font ,(o-formula (gv *fonts-and-styles*
				       :label-font)))
		)

	       (:change-perms-user-button
		,command-button-panel
		(:left ,(o-formula (+ (gvl :parent :perms-user-text :left) 5)))
		(:top ,(o-formula (+ (gvl :parent :perms-user-text :top)
				     (gvl :parent :perms-user-text :height)
				     5)))
		(:visible ,(o-formula (gvl :parent :perms-text :visible)))
		(:direction :vertical)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (declare (ignore nv))
		      (let* ((menu (g-value obj :parent :change-perms-user-menu))
			     (selected-ranks
			      (g-value menu :selected-ranks))
			     (grandpa (g-value obj :parent :parent))
			     (items (g-value menu :items))
			     (perm-change-list (g-value grandpa :perm-change-list))
			     (user-list (mapcar #'(lambda (n) (nth n items))
						selected-ranks))
			     (pair-list
			      (remove-if-not #'(lambda (u)
						 (member u user-list
							 :test #'string=))
					     perm-change-list :key #'car))
			     )
			(fix-perms pair-list)
			(s-value menu :selected-ranks nil)
			(call-schema verify-update-menu :display-me)
			)))
		(:items ("Fix Permissions for Selected Users"))
		)

	       (:change-perms-user-menu
		,garnet-gadgets:scrolling-menu
		(:left ,(o-formula (+ (gvl :parent :change-perms-user-button :left)
				      10)))
		(:top ,(o-formula (+ (gvl :parent :change-perms-user-button :top)
				     (gvl :parent :change-perms-user-button :height)
				     5)))
		(:visible ,(o-formula (gvl :parent :perms-text :visible)))
		(:num-visible 5)
		(:multiple-p T)
		(:item-font ,(o-formula (gv *fonts-and-styles*
					    :label-font)))
		(:title nil)
		(:title-font ,(o-formula (gv *fonts-and-styles*
					     :microscopic-font)))
		(:items nil)
		;; no menu-selection-function
		)

	       (:perms-file-text
		,opal:text
		(:left ,(o-formula (+ (max (+ (gvl :parent :perms-user-text :left)
					      (gvl :parent :perms-user-text :width))
					   (+ (gvl :parent :change-perms-user-button
						   :left)
					      (gvl :parent :change-perms-user-button
						   :width))
					   (+ (gvl :parent :change-perms-user-menu
						   :left)
					      (gvl :parent :change-perms-user-menu
						   :width)))
				      10)))
		(:top ,(o-formula (gvl :parent :perms-text :top)))
		(:visible ,(o-formula (gvl :parent :perms-text :visible)))
		(:string "Files with different permissions:")
		(:font ,(o-formula (gv *fonts-and-styles*
				       :label-font)))
		)

	       (:change-perms-file-button
		,command-button-panel
		(:left ,(o-formula (+ (gvl :parent :perms-file-text :left) 5)))
		(:top ,(o-formula (+ (gvl :parent :perms-file-text :top)
				     (gvl :parent :perms-file-text :height)
				     5)))
		(:visible ,(o-formula (gvl :parent :perms-text :visible)))
		(:direction :vertical)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (declare (ignore nv))
		      (let* ((menu (g-value obj :parent :change-perms-file-menu))
			     (selected-ranks (g-value menu :selected-ranks))
			     (grandpa (g-value obj :parent :parent))
			     (items (g-value menu :items))
			     (perm-change-list (g-value grandpa :perm-change-list))
			     (file-list (mapcar #'(lambda (n) (nth n items))
						selected-ranks))
			     (pair-list
			      (remove-if-not #'(lambda (f)
						 (member f file-list
							 :test #'string=))
					     perm-change-list :key #'cadr))
			     )
			(fix-perms pair-list)
			(s-value menu :selected-ranks nil)
			(call-schema verify-update-menu :display-me)
			)))
		(:items ("Fix Permissions for Selected Files"))
		)

	       (:change-perms-file-menu
		,garnet-gadgets:scrolling-menu
		(:left ,(o-formula (+ (gvl :parent :change-perms-file-button :left)
				      10)))
		(:top ,(o-formula (+ (gvl :parent :change-perms-file-button :top)
				     (gvl :parent :change-perms-file-button :height)
				     5)))
		(:visible ,(o-formula (gvl :parent :perms-text :visible)))
		(:num-visible 5)
		(:multiple-p T)
		(:item-font ,(o-formula (gv *fonts-and-styles*
					    :label-font)))
		(:title nil)
		(:title-font ,(o-formula (gv *fonts-and-styles*
					     :microscopic-font)))
		(:items nil)
		;; no menu-selection-function
		)
	      ))
	     )
      )))
  (push verify-update-menu *all-dboxes*)

#|
  (create-instance
   'verify-group-menu opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:visible nil)
   (:displaying-me nil)
   (:group-addition-list nil)
   (:group-deletion-list nil)
   (:user-addition-list nil)
   (:user-deletion-list nil)

   (:display-me #'(lambda (obj &rest args)
		    (declare (ignore args))
		    (s-value obj :visible T)
		    (s-value verify-update-menu :displaying-me nil)
		    (s-value obj :displaying-me T)

		    (let ((menu (g-value obj :menu
					 :delete-group-menu)))
		      (s-value menu :items
			       (mapcar #'car
				       (g-value obj
						:group-deletion-list)))
		      (opal:notice-items-changed
		       (g-value menu :menu-item-list)))

		    (let ((menu (g-value obj :menu :add-group-menu)))
		      (s-value menu :items
			       (mapcar #'car
				       (g-value obj
						:group-addition-list)))
		      (opal:notice-items-changed
		       (g-value menu :menu-item-list)))

		    (opal:deiconify-window dialog-window)
		    (opal:raise-window dialog-window)
		    (opal:update dialog-window)
		    ))

   (:undisplay-me #'(lambda (obj &rest args)
		      (declare (ignore args))
		      (s-value obj :visible nil)
		      (opal:lower-window dialog-window)
		      (allow-interference)
		      ))

   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu))))
      (:menu ,opal:aggregadget
	     (:visible ,(o-formula (gvl :parent :visible)))
	     (:parts
	      ((:resume-button
		,command-button-panel
		(:top ,(o-formula (gv *fonts-and-styles*
                                      :db-frame-width)))
		(:left ,(o-formula (gv *fonts-and-styles*
                                       :db-frame-width)))
		(:direction :horizontal)
		(:rank-margin 3)
		(:items (("Resume Editing"
			  ,#'(lambda (obj &rest args)
			       (declare (ignore args))
			       (call-schema
				(g-value obj :parent :parent)
				:undisplay-me)
			       ))
			 ("Show Results"
			  ,#'(lambda (obj &rest args)
			       (declare (ignore args))
			       (s-value
				(g-value obj :parent :parent) :visible
				nil)
			       (call-schema verify-result-menu
					    :display-me)
			       ))
			 ("Update Iff File"
			  ,#'(lambda (obj &rest args)
			       (declare (ignore args))
			       (s-value
				(g-value obj :parent :parent) :visible
				nil)
			       (call-schema verify-update-menu
					    :display-me)
			       ))
			 )
	       ))
	       (:group-box-text ,opal:text
				(:left ,(o-formula (gv *fonts-and-styles*
                                                       :db-frame-width)))
				(:top ,(o-formula (+ (gvl :parent
							  :resume-button
							  :top)
						     (gvl :parent
							  :resume-button
							  :height)
						     10)))
				(:visible ,(o-formula
					    (and (gvl :parent :visible)
						 (or
						  (gvl :parent
						       :delete-group-menu
						       :visible)
						  (gvl :parent
						       :add-group-menu
						       :visible)))))
				(:string "Group Boxes:")
				(:font ,(o-formula (gv *fonts-and-styles*
                                                       :label-font)))
				)
	       (:delete-group-button
		,command-button-panel
		(:left ,(o-formula (+ 5 (gv *fonts-and-styles*
                                            :db-frame-width))))
		(:top ,(o-formula (+ (gvl :parent :group-box-text :top)
				     (gvl :parent :group-box-text :height)
				     5)))
		(:visible ,(o-formula (gvl :parent :delete-group-menu
					   :visible)))
		(:direction :vertical)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (let* ((deleting-all
			      (string-equal nv "Delete All Extra Boxes"))
			     (selected-ranks
			      (unless deleting-all
				      (g-value obj :parent
					       :delete-group-menu :selected-ranks)))
			     (group-deletion-list (g-value obj :parent
							   :parent
							   :group-deletion-list))
			     )
			)))
		(:items ("Delete Selected Boxes"
			 "Delete All Extra Boxes"))
		)
	       (:delete-group-menu
		,garnet-gadgets:scrolling-menu
		(:left ,(o-formula (+ 15 (gv *fonts-and-styles*
                                             :db-frame-width))))
		(:top ,(o-formula (+ (gvl :parent :delete-group-button :top)
				     (if (gvl :visible)
					 (+ (gvl :parent
						 :delete-group-button
						 :height) 5)
				       0))))
		(:visible ,(o-formula (and (gvl :parent :visible)
					   (gvl :items))))
		(:num-visible 5)
		(:multiple-p T)
		(:item-font ,(o-formula (gv *fonts-and-styles*
                                            :label-font)))
		(:title nil)
		(:title-font ,(o-formula (gv *fonts-and-styles*
                                             :microscopic-font)))
		(:items nil)
		;; no menu-selection-function
		)
	       (:add-group-button
		,command-button-panel
		(:left ,(o-formula
			 (if (gvl :parent :delete-group-menu :visible)
			     (+ 55 (max (gvl :parent
					     :delete-group-button
					     :width)
					(+ 10 (gvl :parent
						   :delete-group-menu
						   :width))))
			   ,(+ 5 (gv *fonts-and-styles*
                                     :db-frame-width)))))
		(:top ,(o-formula (gvl :parent :delete-group-button :top)))
		(:visible ,(o-formula (gvl :parent :add-group-menu
					   :visible)))
		(:direction :vertical)
		(:selection-function
		 ,#'(lambda (obj nv)
		      (let* ((adding-all
			      (string-equal nv "Add All Missing Boxes"))
			     (selected-ranks
			      (unless adding-all
				      (g-value obj :parent
					       :add-group-menu
					       :selected-ranks)))
			     (group-box-addition-list
			      (g-value obj :parent :parent
				       :group-box-addition-list))
			     (boxes (if adding-all group-box-addition-list
				      (mapcar #'(lambda (r)
						  (nth r group-box-addition-list))
					      selected-ranks)))
			     )
			)))
		(:items ("Add Selected Boxes"
			 "Add All Missing Boxes"))
		)
	       (:add-group-menu
		,garnet-gadgets:scrolling-menu
		(:left ,(o-formula (+ (gvl :parent :add-group-button
					   :left) 10)))
		(:top ,(o-formula (+ (gvl :parent :add-group-button
					  :top)
				     (if (gvl :visible)
					 (+ (gvl :parent
						 :add-group-button
						 :height) 5)
				       0))))
		(:visible ,(o-formula (and (gvl :parent :visible)
					   (gvl :items))))
		(:num-visible 5)
		(:multiple-p T)
		(:item-font ,(o-formula (gv *fonts-and-styles*
                                            :label-font)))
		(:title nil)
		(:title-font ,(o-formula (gv *fonts-and-styles*
                                             :microscopic-font)))
		(:items nil)
		;; no menu-selection-function
		)
	       )
      ))
   )))
  (push verify-group-menu *all-dboxes*)
|#
  )

;;;------------------------------------------------------------
;;; create-print-dbox creates a dialog box for specifying print
;;; options.
;;;------------------------------------------------------------
(defun create-print-dbox ()
  (create-instance
   'print-dbox opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:fast-redraw-p T)
   (:visible T)
   (:scrolling-box-width 300)

   ;; the file to print
   (:input-value (o-formula (gvl :menu :input-name :value)))

   ;; where to send the output - :file or :printer and the name
   (:output-where (o-formula (gvl :menu :output-file-or-printer :value)))
   (:output-value (o-formula (gvl :menu :output-name :value)))

   ;; additional arguments for iff2ps
   (:iff2ps-args-value (o-formula (gvl :menu :iff-args :value)))

   ;; additional arguments for lpr
   (:lpr-args-value (o-formula (gvl :menu :lpr-args :value)))

   (:initialize-options
    #'(lambda (obj)
	(add-dbox obj)
	(opal:deiconify-window dialog-window)
	(opal:raise-window dialog-window)
	))

   (:parts
    `(
      (:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu)))
	      (:where :back)
	      )
      (:menu
       ,miro-aggregadget
       (:parts
	((:reset ,dbox-reset-button
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:top ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width))))
	 (:input-name ,dbox-filename-selector
		      (:left ,(o-formula (gv *fonts-and-styles*
					     :db-frame-width)))
		      (:top ,(o-formula (+ 10 (gvl :parent :reset :top)
					   (gvl :parent :reset :height))))
		      (:title-string "File to print (leave blank for workbench):")
		      (:default-value "")
		      )
	 (:output-file-or-printer ,dbox-labeled-button-input
				  (:left ,(o-formula (gv *fonts-and-styles*
							 :db-frame-width)))
				  (:top ,(o-formula
					  (+ (gvl :parent :input-name :top)
					     (gvl :parent :input-name :height)
					     10)))
				  (:title-string "Where to send the output:")
				  (:direction :horizontal)
				  (:items (:file :printer))
				  (:value-list (:file :printer))
				  (:default-value :file)
				  )
	 (:output-name ,dbox-filename-selector
		       (:left ,(o-formula (gv *fonts-and-styles*
					      :db-frame-width)))
		       (:top ,(o-formula (+ (gvl :parent
						 :output-file-or-printer
						 :top)
					    (gvl :parent
						 :output-file-or-printer
						 :height)
					    10)))
		       (:title-string ,(o-formula
					(if (eq (gvl :parent
						     :output-file-or-printer
						     :value)
						:file)
					    "Output file (leave blank to send output to lisp window):"
					  "Printer (leave blank to use the default printer):"
					  )))
		       (:default-value "")
		       )
	 (:iff-args ,dbox-labeled-text-input
		    (:left ,(o-formula (gv *fonts-and-styles*
					   :db-frame-width)))
		    (:top ,(o-formula (+ (gvl :parent :output-name :top)
					 (gvl :parent :output-name :height)
					 10)))
		    (:title-string "Arguments for iff2ps:")
		    (:default-value "")
		    )
	 (:lpr-args ,dbox-labeled-text-input
		    (:left ,(o-formula (gv *fonts-and-styles*
					   :db-frame-width)))
		    (:top ,(o-formula (+ (gvl :parent :iff-args :top)
					 (gvl :parent :iff-args :height)
					 10)))
		    (:title-string "Arguments for lpr:")
		    (:default-value "")
		    )
	 (:done ,command-button-panel
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula (+ (gvl :parent :lpr-args :top)
				     (gvl :parent :lpr-args :height)
				     10)))
		(:direction :horizontal)
		(:items (("Print" ,#'do-print)
			 ("Abort" ,#'(lambda (&rest args)
				       (declare (ignore args))
				       (remove-dbox print-dbox)
				       (opal:lower-window dialog-window)
				       (allow-interference)
				       (inter:beep)
				       ))
			 ))
		)
	 )))
      ))
   )
  (push print-dbox *all-dboxes*)
  )


;;;------------------------------------------------------------
;;; create-filename-dbox creates a text string that can be used to
;;; get a filename from the user.
;;;------------------------------------------------------------
(defun create-filename-dbox ()
  (create-instance
   'filename-dbox opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:fast-redraw-p T)
   (:visible T)
   (:filename-value (o-formula (gvl :menu :filename :value)))
   (:title "Input a filename:")
   (:confirm-function nil)
   (:display-me #'(lambda (obj)
		    (add-dbox obj)
		    (call-schema (g-value obj :menu :filename)
				 :start-text-inter)
		    (opal:deiconify-window dialog-window)
		    (opal:raise-window dialog-window)
		    ))
   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu)))
	      )
      (:menu
       ,opal:aggregadget
       (:parts
	((:filename ,dbox-filename-selector
		    (:left ,(o-formula (gv *fonts-and-styles*
					   :db-frame-width)))
		    (:top ,(o-formula (gv *fonts-and-styles*
					  :db-frame-width)))
		    (:title-string ,(o-formula (gvl :parent :parent
						    :title)))
		    (:default-value "")
		    )
	 (:confirm ,command-button-panel
		   (:left ,(o-formula (gv *fonts-and-styles*
					  :db-frame-width)))
		   (:top ,(o-formula (+ (gvl :parent :filename :top)
					(gvl :parent :filename :height)
					10)))
		   (:direction :horizontal)
		   (:items
		    (("Confirm" ,#'(lambda (&rest args)
				     (declare (ignore args))
				     (remove-dbox filename-dbox)
				     (let ((confirm-fn (g-value
							filename-dbox
							:confirm-function)))
				       (when confirm-fn
					     (funcall confirm-fn
						      (g-value
						       filename-dbox :filename-value)
						      ))
				       )
				     (opal:lower-window dialog-window)
				     ))
		     ("Abort" ,#'(lambda (&rest args)
				   (declare (ignore args))
				   (remove-dbox filename-dbox)
				   (opal:lower-window dialog-window)
				   (allow-interference)
				   ))
		     ))
		   )
	 )))
      ))
   )
  (push filename-dbox *all-dboxes*)
  )

;;;------------------------------------------------------------
;;; create-ambig-menus creates menus and buttons for the interface to
;;; the ambiguity checker.
;;;------------------------------------------------------------
(defun create-ambig-menus ()
  (create-schema
   'ambig-status

   (:have-results nil)
   (:reset-selected-ranks T)

   (:error-list "")

   ;; set to the name of the file checked, or nil if the workbench was
   ;; used; boxes can be selected if this is the case.
   (:workbench nil)

   ;; set this when ambig is run; reset when any
   ;; change is made to the picture.
   (:guaranteed-valid nil)

   ;; no results
   (:clear-results #'(lambda (obj)
		       (s-value obj :have-results nil)
		       (s-value obj :error-list "")
		       (s-value obj :workbench nil)
		       (s-value obj :guaranteed-valid nil)
		       (s-value obj :ambig-vectors nil)
		       (s-value obj :negative-vectors nil)
		       (s-value obj :none-vectors nil)
		       (s-value obj :positive-vectors nil)
		       (update-command-inactive-list)
		       ))

   ;; convert a list of lists to a list of vectors
   (:vectorize-item-list
    #'(lambda (items agg)
	(let ((using-workbench (not (g-value ambig-status :workbench))))
	  (mapcar
	   #'(lambda (item)
	       (let ((user (find-box
			    (if *new-ambig* (second item) (third item))
			    agg))
		     (file (find-box
			    (if *new-ambig* (third item) (fourth item))
			    agg))
		     )
		 (vector item
			 (when using-workbench user)
			 (when using-workbench file)
			 (format nil "User: ~S  File: ~S  (~A)"
				 (if user (g-value user :label :string)
				   "")
				 (if file (g-value file :label :string)
				   "")
				 (if *new-ambig*
				     (string-downcase (symbol-name
						       (second
							(fourth item))))
				   (second item))
				 ))
		 )
	       )
	   items))))

   ;; initialize ambig results
   (:init-results
    #'(lambda (obj ambig-list negative-list positive-list error-list agg)
	(declare (ignore obj))
	(let ((vectorize-item-list (g-value ambig-status :vectorize-item-list)))
	  (s-value ambig-status :ambig-vectors
		   (funcall vectorize-item-list ambig-list agg))
	  (s-value ambig-status :negative-vectors
		   (funcall vectorize-item-list negative-list agg))
	  (s-value ambig-status :positive-vectors
		   (funcall vectorize-item-list positive-list agg))
	  )
	(s-value ambig-status :error-list
		 (unless (null-string error-list) error-list))
	(s-value ambig-status :guaranteed-valid T)
	(s-value ambig-status :reset-selected-ranks T)
	(s-value ambig-status :have-results T)
	(update-command-inactive-list)
	))

   ;; display the results
   (:display-results
    #'(lambda (obj)
	(when (g-value obj :reset-selected-ranks)
	      (garnet-gadgets:set-first-item
	       (g-value ambig-menu :menu) nil)
	      (garnet-gadgets:set-first-item
	       (g-value ambig-menu :menu) :title)
	      (s-value obj :reset-selected-ranks nil)
	      )
	(if (g-value obj :error-list)
	    (call-schema obj :make-error-menu)
	  (progn (add-dbox ambig-menu)
		 (opal:deiconify-window dialog-window)
		 (opal:raise-window dialog-window)))
	(set-help-string
	 "Select a box pair or
menu.")
	))

   ;; create ambig-error-menu
   (:make-error-menu
    #'(lambda (obj)
	(declare (ignore obj))
	(create-instance 'ambig-error-window garnet-gadgets:scrolling-window-with-bars
			 (:visible T)
			 (:parent-window dialog-window)
			 (:top (o-formula (gv *fonts-and-styles*
					      :db-real-frame-width)))
			 (:left (o-formula (gv *fonts-and-styles*
					       :db-real-frame-width)))
			 (:width 500) (:height 250)
			 (:total-width (o-formula (+ (gv ambig-error-list
							 :width) 10)
						  500))
			 (:total-height (o-formula (+ (gv ambig-error-list
							  :height) 10)
						   250))
			 (:border-width 1)
			 (:double-buffered-p T)
			 )
	(opal:update ambig-error-window)
	(opal:add-component (g-value ambig-error-window
				     :inner-aggregate)
			    ambig-error-list)
	(opal:update ambig-error-window)
	(add-dbox ambig-error-menu)
	))
   )

  (create-instance 'ambig-error-list opal:multi-text
		   (:left 5)
		   (:top 5)
		   (:font (o-formula (gv *fonts-and-styles*
					 :label-font)))
		   (:string (o-formula (gv ambig-status :error-list)))
		   )
  ;; this gadget should "contain" ambig-error-window, which will
  ;; display the error messages.
  (create-instance
   'ambig-error-menu opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:visible T)
   (:parts
    `((:frame ,opal:rectangle
	      (:line-style ,(o-formula (gv *fonts-and-styles*
					   :db-frame-style)))
	      (:filling-style ,(o-formula (gv *colors* :white)))
	      (:where :back)
	      (:top ,(o-formula (gvl :parent :top)))
	      (:left ,(o-formula (gvl :parent :left)))
	      (:width ,(o-formula (+ (gv ambig-error-window :width)
				     (* 2 (gv ambig-error-window :border-width))
				     (* 2 (gv *fonts-and-styles*
					      :db-real-frame-width))
				     )))
	      (:height ,(o-formula (+ (gv ambig-error-window :height)
				      (* 2 (gv ambig-error-window :border-width))
				      (* 2 (gv *fonts-and-styles*
					       :db-real-frame-width))
				      (gvl :parent :done :height)
				      15)))
	      )
      (:done ,command-button-panel
	     (:left ,(o-formula (gv *fonts-and-styles*
				    :db-frame-width)))
	     (:top ,(o-formula (+ (gv ambig-error-window :height)
				  (gv ambig-error-window :top)
				  10
				  )))
	     (:items (("Resume Editing"
		       ,#'(lambda (&rest args)
			    (declare (ignore args))
			    (opal:remove-component (g-value
						    ambig-error-window
						    :inner-aggregate)
						   ambig-error-list)
			    (opal::destroy-me ambig-error-window)
			    (allow-interference)
			    (remove-dbox ambig-error-menu)
			    (opal:lower-window dialog-window)
			    )
		       ))
		     )
	     )
      ))
   )
  (push ambig-error-menu *all-dboxes*)
  
  (create-instance
   'ambig-menu opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:fast-redraw-p T)
   (:visible T)
   (:parts
    `((:frame ,opal:rectangle
	      (:line-style ,(o-formula (gv *fonts-and-styles*
					   :db-frame-style)))
	      (:filling-style ,(o-formula (gv *colors* :white)))
	      (:where :back)
	      (:top ,(o-formula (gvl :parent :top)))
	      (:left ,(o-formula (gvl :parent :left)))
	      (:width ,(o-formula
			(+ (* 2 (gv *fonts-and-styles*
				    :db-frame-width))
			   (max
			    (gvl :parent :menu :width)
			    (gvl :parent :info-str :width)))
			))
	      (:height ,(o-formula
			 (+ (* 2 (gv *fonts-and-styles*
				     :db-frame-width))
			    (if (gvl :parent :info-str :visible)
				(+ (gvl :parent :info-str :height) 10)
			      0)
			    (gvl :parent :done :height) 5
			    (gvl :parent :menu :height))))
	      )
      (:menu ,garnet-gadgets:browser-gadget
	     (:top ,(o-formula (gv *fonts-and-styles*
				   :db-frame-width)))
	     (:left ,(o-formula (gv *fonts-and-styles*
				    :db-frame-width)))
	     (:item-font ,(o-formula (gv *fonts-and-styles* :label-font)))
	     (:title-font ,(o-formula (gv *fonts-and-styles* :label-font)))
	     (:num-menus 2)
	     (:num-rows 5)
	     (:additional-selection-p nil)
	     (:menu-items-generating-function
	      ,#'(lambda (item)
		   (cond
		    ((vectorp item)
		     (dolist (obj (g-value obj-agg :selected))
			     (s-value obj :selected nil))
		     (s-value obj-agg :selected nil)
		     (let ((user (svref item 1))
			   (file (svref item 2))
			   )
		       (when (and user (schema-p user))
			     (s-value user :selected T)
			     (push user (g-value obj-agg :selected))
			     )
		       (when (and file (schema-p file))
			     (s-value file :selected T)
			     (push file (g-value obj-agg :selected))
			     )
		       )

		     ;; update button visibility: copy, delete,
		     ;; display, unselect
		     (update-command-inactive-list)

		     (allow-interference)
		     (remove-dbox ambig-menu)
		     (opal:lower-window dialog-window)
		     nil)
		    ((eq item :ambiguous) (g-value ambig-status :ambig-vectors))
		    ((eq item :negative) (g-value ambig-status :negative-vectors))
		    ((eq item :positive) (g-value ambig-status :positive-vectors))
		    ((eq item :quit)
		     (allow-interference)
		     (remove-dbox ambig-menu)
		     (opal:lower-window dialog-window)
		     (s-value ambig-status :reset-selected-ranks T)
		     nil)
		    ((eq item :title)
		     (list :ambiguous :negative :positive))
		    )))
		    
	     (:item-to-string-function
	      ,#'(lambda (item)
		   (cond
		    ((vectorp item) (svref item 3))
		    ((eq item :ambiguous)
		     (format nil "Ambiguous (~S)"
			     (length
			      (g-value ambig-status :ambig-vectors))))
		    ((eq item :negative)
		     (format nil "Negative (~S)"
			     (length
			      (g-value ambig-status :negative-vectors))))
		    ((eq item :positive)
		     (format nil "Positive (~S)"
			     (length
			      (g-value ambig-status :positive-vectors))))
		    ((eq item :title) "Ambig? Results")
		    (T "")
		    )))
	     )
      (:info-str ,opal:text
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:top ,(o-formula (+ (gvl :parent :menu :top)
				      (gvl :parent :menu :height)
				      5)))
		 (:visible
		  ,(o-formula
		    (and (gvl :parent :visible)
			 (not (null-string (gvl :string))))))
		 (:font ,(o-formula (gv *fonts-and-styles*
					:label-font)))
		 (:string
		  ,(o-formula
		    (cond ((gv ambig-status :workbench)
			   (format nil "(Results for file ~S)"
				   (gv ambig-status :workbench)))
			  ((gv ambig-status :guaranteed-valid) "")
			  (T "WARNING: picture has changed"))))
		 )
      (:done ,command-button-panel
	     (:top ,(o-formula
		     (if (gvl :parent :info-str :visible)
			 (+ (gvl :parent :info-str :top)
			    (gvl :parent :info-str :height)
			    10)
		       (+ (gvl :parent :menu :top)
			  (gvl :parent :menu :height)
			  5))))
	     (:left ,(o-formula (gv *fonts-and-styles*
				    :db-frame-width)))
	     (:final-feedback nil)
	     (:items
	      (("Resume Editing"
		,#'(lambda (&rest args)
		     (declare (ignore args))
		     (allow-interference)
		     (remove-dbox ambig-menu)
		     (opal:lower-window dialog-window)
		     ))))
	     )
      )))
  (push ambig-menu *all-dboxes*)

  (garnet-gadgets:set-first-item (g-value ambig-menu :menu) :title)

  (create-instance
   'ambig-options opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:draw-function :xor)
   (:fast-redraw-p T)
   (:visible T)
   (:filename-value (o-formula (gvl :menu :filename :value)))
   (:args-value (o-formula (gvl :menu :args :value)))

   (:initialize-option-menu
    #'(lambda (obj)
	(add-dbox obj)
	(opal:deiconify-window dialog-window)
	(opal:raise-window dialog-window)
	))

   (:parts
    `((:frame ,miro-frame
	      (:component-to-frame ,(o-formula (gvl :parent :menu)))
	      )
      (:menu
       ,miro-aggregadget
       (:parts
	((:reset ,dbox-reset-button
		 (:left ,(o-formula (gv *fonts-and-styles*
					:db-frame-width)))
		 (:top ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width))))
	 (:filename ,dbox-filename-selector
		    (:left ,(o-formula (gv *fonts-and-styles*
					   :db-frame-width)))
		    (:top ,(o-formula (+ (gvl :parent :reset :top)
					 (gvl :parent :reset :height)
					 10)))
		    (:title-string "Filename (leave blank for workbench):")
		    (:default-value "")
		    )
	 (:args ,dbox-labeled-text-input
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula (+ (gvl :parent :filename :top)
				     (gvl :parent :filename :height)
				     10)))
		(:title-string "Arguments for ambig:")
		(:default-value "")
		)
	 (:done ,command-button-panel
		(:left ,(o-formula (gv *fonts-and-styles*
				       :db-frame-width)))
		(:top ,(o-formula (+ (gvl :parent :args :top)
				     (gvl :parent :args :height)
				     10)))
		(:direction :horizontal)
		(:items (("Check Ambiguity" ,#'do-ambiguity-check)
			 ("Abort" ,#'(lambda (&rest args)
				       (declare (ignore args))
				       (allow-interference)
				       (inter:beep)
				       (remove-dbox ambig-options)
				       (opal:lower-window dialog-window)
				       ))
			 ))
		)
	 )))
      ))
   )
  (push ambig-options *all-dboxes*)
  )
  


;;;------------------------------------------------------------
;;; Create-Pictype-Menu creates the menu to select picture type
;;; (either instance or constraint). 
;;; picture mode = (g-value pictype-menu :value) \in {:instance,
;;; :constriant} 
;;;------------------------------------------------------------
(defun create-pictype-menu ()
  (create-instance 'pictype-menu menu-button-panel
		 (:left 40)
		 (:top *pictype-menu-top*)
		 (:default-value :instance)
		 (:items '(:instance :constraint))
		 (:selection-function
		  #'(lambda (interactor nv)
		      (declare (ignore interactor))
		      ;; we need to set :value here instead of when
		      ;; :thickness-buttons are created to avoid an
		      ;; error...
		      (unless (g-value constraint-menu :thickness-buttons :value)
			      (s-value (g-value constraint-menu
						:thickness-buttons)
				       :default-value :thin)
			      (call-schema
			       (g-value constraint-menu :thickness-buttons)
			       :use-default-values))

		      ;; make constraint-menu (in)visible
		      (if (eq nv :instance)
			  (remove-menu constraint-menu)
			(add-menu constraint-menu))

		      ;; if we are changing to an instance picture, we
		      ;; may need to change the selection in tool-menu
		      (when (eq nv :instance)
			    (unless
			     (cond
			      ((g-value tool-menu :menu-items
					:sem-arrow :selected)
			       (s-value (g-value tool-menu :menu-items
						 :sem-arrow) :selected nil))
			      ((g-value tool-menu :menu-items
					:con-arrow :selected)
			       (s-value (g-value tool-menu :menu-items
						 :con-arrow) :selected nil))
			      (T T)
			      )
			     (s-value (g-value tool-menu :menu-items
					       :syn-arrow) :selected T)
			     (s-value (g-value tool-menu :menu-items)
				      :selected
				      (g-value tool-menu :menu-items
					       :syn-arrow))
			     ))

		      ;; turn off the ambig button if necessary
		      (update-command-inactive-list)

		      (s-value ambig-status :have-results nil)

		      ;; turn the ambig menu off if necessary
		      (when (g-value ambig-menu :aggregate)
			    (remove-dbox ambig-menu)
			    (opal:lower-window dialog-window)
			    (allow-interference)
			    )
		      ))
		 )
  pictype-menu)


;;;------------------------------------------------------------
;;; Create-Range-Menu creates the menu to select the range for a
;;; constraint picture. NOT IMPLEMENTED YET.
;;;------------------------------------------------------------
(defun create-range-menu ()
  (create-instance 'range-menu opal:aggregadget
		 (:overlapping T)
		 (:parts
		  `((:frame ,opal:rectangle
			   (:left 20)
			   (:top ,*range-menu-top*)
			   (:width 125)
			   (:height 25))
		    (:label ,opal:text
			    (:left ,(o-formula (+ (gvl :parent :frame :left) 5)))
			    (:top ,(o-formula (+ (gvl :parent :frame :top) 5)))
			     (:string "Range: ... "))))))


;;;------------------------------------------------------------
;;; Create-Tool-Menu creates the menu to select the kind of object
;;; that will be drawn on the workbench (boxes and arrows) 
;;; Semantic and containment arrows are visible iff pictype =
;;; constraint. The tool menu has two aggregate parts :menu-items,
;;; which are the smaller objects that act as a menu from which the
;;; user selects the desired object, and :display-items, which shows
;;; the currently selected object with the currently selected
;;; attributes. 
;;;------------------------------------------------------------

(defun create-tool-menu ()
  ;; The tool-menu aggregate.
  (create-instance 'tool-menu opal:aggregadget
		   (:top *tool-menu-top*)
		   (:overlapping T)
		   (:mode (o-formula (gvl :menu-items :selected)))
		   (:parts			  
		    ;; The aggregates that hold the icons
		    `((:menu-items ,opal:aggregadget
				   ;; default selection is box
				   (:selected
				    ,(o-formula
				      (cond
				       ((gvl :abox :selected) (gvl :abox))
				       ((gvl :syn-arrow :selected)
					(gvl :syn-arrow))
				       ((gvl :sem-arrow :selected)
					(gvl :sem-arrow))
				       ((gvl :con-arrow :selected)
					(gvl :con-arrow))
				       )))
				   (:overlapping NIL)
				   (:parts
				    ;; A  miro-box
				    ((:abox ,menu-box 
					    ;; initially selected
					    (:selected T)
					    (:box ,(o-formula (list
							       10 ;left
							       (+ *tool-menu-top* 75)
							       60 40)))) ; width, height
				     ;; A syntactic arrow
				     (:syn-arrow ,miro-arrow
						 (:select-outline-only nil)
						 (:x1 90) 
						 (:y1 ,(o-formula (+ *tool-menu-top* 105)))
						 (:x2 140) 
						 (:y2 ,(o-formula (+ *tool-menu-top* 105))))
				     ;; A semantic arrow
				     ;; should only be visible when *pictype* is constraint
				     (:sem-arrow ,miro-arrow
						 (:visible ,(o-formula (eq
									(gv pictype-menu :value)
									:constraint)))
						 (:select-outline-only nil)
						 (:arrow-type :sem)
						 (:x1 90) 
						 (:y1 ,(o-formula (+ *tool-menu-top* 80)))
						 (:x2 140) 
						 (:y2 ,(o-formula (+ *tool-menu-top* 80))))
				     ;; A containment arrow
				     ;; should only be visible when *pictype* is constraint
				     (:con-arrow ,miro-arrow
						 (:visible ,(o-formula (eq
									(gv pictype-menu :value)
									:constraint)))
						 (:select-outline-only nil)
						 (:arrow-type :con) 
						 (:x1 170) 
						 (:y1 ,(o-formula (+ *tool-menu-top* 115)))
						 (:x2 170) 
						 (:y2 ,(o-formula (+ *tool-menu-top* 75)))))))
		      ;; The aggregates that hold the displayed items. Only
		      ;; one is visible at any given time, based on the
		      ;; selection in :menu-items
		      (:display-items
		       ,opal:aggregadget
		       ;; slots to keep track of selection
		       (:box-selected ,(o-formula (gvl :parent :menu-items
						       :abox :selected)))
		       (:hor-arrow-selected ,(o-formula 
					      (or (gvl :parent :menu-items
						       :syn-arrow :selected)
						  (gvl :parent :menu-items
						       :sem-arrow
						       :selected))))
		       (:con-arrow-selected ,(o-formula (gvl :parent :menu-items
							     :con-arrow :selected)))
		       (:parts
			;; A  miro-box
			((:abox ,menu-box 
				(:visible ,(o-formula (gvl :parent
							   :box-selected)))
				;; was formula
				(:thick ,(o-formula (and (eq (gv pictype-menu
								 :value)
							     :constraint)
							 (eq (gv constraint-menu
								 :thickness-buttons
								 :value) :thick)
							 )))
				;; was formula
				(:starred ,(o-formula (and (eq (gv pictype-menu
								   :value)
							       :constraint)
							   (gv constraint-menu
							       :starred-button
							       :value)
							   T))) 
				(:box ,(o-formula (list
						   30 ;left
						   (+ *tool-menu-top* 5)
						   100 50)))) ; width, height
			 ;; A horizontal arrow
			 (:hor-arrow ,miro-arrow
				     (:visible ,(o-formula (gvl :parent
								:hor-arrow-selected)))
				     (:arrow-type ,(o-formula (gvl :parent
								   :parent :mode
								   :arrow-type)))
				     ;; was formula
				     (:neg ,(o-formula (eq (gv arrow-menu
							       :value) :negative)))
				     ;; was formula
				     (:thick ,(o-formula (and (eq (gv pictype-menu
								      :value)
								  :constraint)
							      (eq (gv constraint-menu
								      :thickness-buttons
								      :value) :thick)
							      )))
				     (:x1 40) (:y1 ,(o-formula (+ *tool-menu-top* 35)))
				     (:x2 150) 
				     (:y2 ,(o-formula (+ *tool-menu-top* 35))))
			 ;; A containment arrow
			 (:con-arrow ,miro-arrow
				     (:visible ,(o-formula (gvl :parent
								:con-arrow-selected)))
				     ;; was formula
				     (:neg ,(o-formula (eq (gv arrow-menu
							       :value) :negative)))
				     ;; was formula
				     (:thick ,(o-formula (and (eq (gv pictype-menu
								      :value)
								  :constraint)
							      (eq (gv constraint-menu
								      :thickness-buttons
								      :value) :thick)
							      )))
				     ;; was formula
				     (:starred ,(o-formula (and (eq (gv pictype-menu
									:value)
								    :constraint)
								(gv constraint-menu
								    :starred-button
								    :value)
								T))) 
				     (:arrow-type :con) 
				     (:x1 80) 
				     (:y1 ,(o-formula (+ *tool-menu-top* 70)))
				     (:x2 80) 
				     (:y2 ,*tool-menu-top*)))))
		      ;; The interim feedback (reverse-video) for the tool-menu
		      (:reverse-feedback ,opal:roundtangle
					 ;; this causes visibility problems in some cases.
					 (:draw-function :xor)
					 (:filling-style ,(o-formula
							   (gv *colors* :black)))
					 (:line-style NIL)
					 (:visible ,(o-formula (gvl :obj-over)))
					 (:left ,(o-formula (gvl :obj-over :left) 0))
					 (:top ,(o-formula (gvl :obj-over :top) 0))
					 (:width ,(o-formula (+ 1 (gvl :obj-over :width)) 0))
					 (:height ,(o-formula (+ 1 (gvl :obj-over :height)) 0)))
		      ;; The feedback (outline) for the tool-menu
		      (:outline-feedback ,opal:rectangle
					 ;; this causes visibility problems in some cases.
					 (:draw-function :xor)
					 (:line-style ,opal:line-2)
					 (:visible 
					  ,(o-formula (and (gvl :parent :menu-items
								:selected)  
							   (gvl :parent :menu-items
								:selected :visible)))) 
					 (:left 
					  ,(o-formula (- (gvl :parent :menu-items
							      :selected :left) 3) 0))
					 (:top 
					  ,(o-formula (- (gvl :parent :menu-items
							      :selected :top) 3) 0)) 
					 (:width 
					  ,(o-formula
					    (+ 8 (g-value (gvl :parent :menu-items
							       :selected) :width)) 0)) 
					 (:height 
					  ,(o-formula 
					    (+ 8 (g-value (gvl :parent :menu-items
							       :selected) :height))
					    0)))))) 
  ;; turn off visibility of arrow labels to make selection box smaller
  (s-value (g-value tool-menu :menu-items :syn-arrow :label) :visible
	   NIL)
  (s-value (g-value tool-menu :menu-items :sem-arrow :label) :visible
	   NIL)
  ;; set labels of display items
  ;; have to do s-value after create, otherwise it gets overwritten with ""
  (s-value (g-value tool-menu :display-items :abox :label) :string "Name")
  (s-value (g-value tool-menu :display-items :hor-arrow :label) :string "label")
  ;; turn off selection feedback on items in menu 
  ;; How about just deleting the feedback object itself...?
  (destroy-constraint (g-value tool-menu :menu-items :abox
			       :selected-feedback) :visible)
  (destroy-constraint (g-value tool-menu :menu-items :syn-arrow
			       :selected-feedback) :visible)
  (destroy-constraint (g-value tool-menu :menu-items :sem-arrow
			       :selected-feedback) :visible)
  (destroy-constraint (g-value tool-menu :menu-items :con-arrow
			       :selected-feedback) :visible)
  
					; return the agg as the result of the function...
  tool-menu)

;;;-------------------------------------------------------------------
;;; The interactor to chose a tool in the tool menu. 
;;;-------------------------------------------------------------------

(defun create-tool-inter ()
 ;; the tool menu interactor
  (create-instance 'tool-menu-inter inter:menu-interactor 
	   (:feedback-obj (g-value tool-menu :reverse-feedback))
	   (:start-where
	    `(:element-of ,(get-value tool-menu :menu-items)))
	   (:exception NIL)
	   (:how-set :set)
	   (:window menu-window)))


;;;------------------------------------------------------------
;;; Create-Arrow-Menu creates the menu to specify arrow parity
;;; (positive or negative)
;;;------------------------------------------------------------
(defun create-arrow-menu ()
  (create-instance 'arrow-menu opal:aggregadget
		   (:top *arrow-menu-top*)
		   (:value (o-formula (gvl :buttons :value)))
		   (:parts
		    `((:title ,opal:text
			    (:left 40)
			    (:top ,*arrow-menu-top*)
			    (:string "Arrow Parity:")
			    (:font ,(o-formula (gv *fonts-and-styles*
						   :button-label-font))))
		      (:buttons ,menu-button-panel
			    (:left 5)
			    (:top ,(+ *arrow-menu-top* 15))
			    (:direction :horizontal)
			    (:default-value :positive)
			    (:items (:positive :negative))))))
  arrow-menu)



;;;------------------------------------------------------------
;;; Create-Constraint-Menu creates the menu to specify thickness and
;;; starred for an object.
;;;------------------------------------------------------------
(defun create-constraint-menu () 
 (create-instance 'constraint-menu opal:aggregadget
		  (:visible T)
		  (:top *constraint-menu-top*)
		  (:parts
		   `((:titles ,opal:aggregadget
			      (:selected NIL) (:overlapping NIL)
					; move font to here? 
			      (:parts
			       ((:title ,opal:text
					(:left 20)
					(:top ,*constraint-menu-top*)
					(:string "Constraint Objects:")
					(:font ,(o-formula (gv *fonts-and-styles*
							       :button-label-font))))
				(:thickness-label ,opal:text
						  (:left 10)
						  (:top ,(+ *constraint-menu-top* 20))
						  (:string "- Thickness")
						  (:font ,(o-formula (gv *fonts-and-styles*
									 :button-label-font))))

				(:starred-label ,opal:text
						(:left 10)
						(:top ,(+ *constraint-menu-top* 65))
						(:string "- Containment")
						(:font ,(o-formula (gv *fonts-and-styles*
								       :button-label-font)))))))
		     (:thickness-buttons ,menu-button-panel
					 (:left 15)
					 (:top ,(+ *constraint-menu-top* 35))
					 (:direction :horizontal)
					 (:items (:thin :thick))
					 )
		     (:starred-button ,menu-button-panel
				      (:left 15)
				      (:top ,(+ *constraint-menu-top* 80))
				      (:toggle T)
				      (:direction :horizontal)
				      (:items (:starred)))
		     )))

 ;; return menu
 constraint-menu)

;;;------------------------------------------------------------
;;; Create-Popup-Menu creates the popup menu for commands.
;;;------------------------------------------------------------

;; format: (menu-entry can-we-do-this? action cant-do-this-message)
(defvar *popup-command-update-list*
  '(
    (("File -->" "Clear...") nil clear-workbench)
    (("File -->" "Read...") nil read-miro-file)
    (("File -->" "Save...") nil save-workbench)
    (("File -->" "Print...") nil print-workbench)
    (("File -->" "Exit...") nil exit-editor)
    (("Edit -->" "Copy") can-copy copy-selected-objects
     "Nothing to copy.")
    (("Edit -->" "Delete") can-delete delete-selected-objects
     "Nothing to delete.")
    (("Edit -->" "Display...") can-display display-object
     "Nothing to display.")
    (("Edit -->" "Hide") can-hide hide-selected-objects
     "Hide not implemented
yet.")
    (("Edit -->" "Undelete") can-undelete undelete "Nothing to undelete.")
    (("Edit -->" "Unhide") can-unhide unhide-selected-objects
     "Unhide not
implemented yet.")
    (("Edit -->" "Unselect") can-unselect unselect-all
     "Nothing to unselect.")
    (("Tools -->" "Ambig? -->") can-ambig nil
     "Ambig? only works on instance
pictures.")
    (("Tools -->" "Ambig? -->" "Check Ambiguity...") nil check-ambiguity)
    (("Tools -->" "Ambig? -->" "View Results...") can-view-ambig-results
     view-ambig-results "No results to view.")
    (("Tools -->" "Legal? -->" "Check Constraint...") nil check-legality)
    (("Tools -->" "Legal? -->" "View Results...")
     can-show-constraint-results show-constraint-results
     "No results to show.")
    (("Tools -->" "Probe...") can-probe probe-directory
     "Probe only works in instance
mode.")
    (("Tools -->" "Verify -->") can-verify nil
     "Verify only works on instance
pictures.")
    (("Tools -->" "Verify -->" "Run Verifier...") nil verify-workbench)
    (("Tools -->" "Verify -->" "View Results...") can-view-verify-results
     view-verify-results "No results to view.")
    (("Tools -->" "Verify -->" "Box Creation List -->")
     box-creation-list-exists nil
     "Box creation list is
empty.")
    (("Tools -->" "Verify -->" "Box Creation List -->"
      "Modify -->" "Create Files")
     file-box-list-exists create-from-file-list
     "File box creation list is empty.")
    (("Tools -->" "Verify -->" "Box Creation List -->"
      "Modify -->" "Create Users/Groups")
     user-box-list-exists create-from-user-list
     "User box creation list is empty.")
    (("Tools -->" "Verify -->" "Box Creation List -->"
      "Modify -->" "Flush Files")
     file-box-list-exists flush-file-box-creation-list
     "File box creation list is empty")
    (("Tools -->" "Verify -->" "Box Creation List -->"
      "Modify -->" "Flush Users")
     user-box-list-exists flush-user-box-creation-list
     "User box creation list is empty")
    (("Tools -->" "Verify -->" "Box Creation List -->"
      "Modify -->" "Postpone Creation")
     can-postpone-box-creation postpone-box-creation
     "Box creation already postponed.")
    (("Tools -->" "Verify -->" "Box Creation List -->"
      "Show Current Box...") nil show-current-box)
    (("Tools -->" "Verify -->" "Box Creation List -->"
      "Show List...") nil show-box-creation-list)
    (("Options...") nil edit-options)
    ))

(defun update-command-inactive-list ()
  (dolist (com *popup-command-update-list*)
	  (let ((can-run (second com))
		(item (first com)))
	    (when can-run
		  (if (funcall can-run)
		      (call-schema popup-command-menu :activate-item
				   item)
		    (call-schema popup-command-menu :inactivate-item
				 item)
		    )))))

(defun execute-command (obj nv)
  (let* ((entry (assoc nv *popup-command-update-list* :test #'equal))
	 (action (third entry)))
    (allow-interference)
    (when action (funcall action obj nv))))

(defun print-inactive-message (obj nv)
  (when (g-value obj :menu-active)
	(let* ((entry (assoc nv *popup-command-update-list* :test
			     #'equal))
	       (msg (when entry (fourth entry)))
	       )
	  (when msg (push-error-msg msg))
	  )))

(defun create-popup-menu ()
  (opal:add-component
   work-agg
   (create-instance
    'popup-command-menu nested-popup-menu
    (:left 0) (:top 0)
    (:selection-function (list (dont-interfere execute-command)
			       #'print-inactive-message))
    (:waiting-priority highest-priority-level)
    (:running-priority highest-priority-level)
    (:title-font (o-formula (gv *fonts-and-styles* :popup-title-font)))
    (:item-font (o-formula (gv *fonts-and-styles* :popup-item-font)))
    (:inactive-font (o-formula (gv *fonts-and-styles* :popup-inactive-font)))
    (:items
     `(("File -->" nil
	("Clear..." "Read..." "Save..." "Print..." "Exit..."))
       ("Edit -->" nil
	("Copy" "Delete" "Display..." "Hide" "Undelete" "Unhide"
	 "Unselect"))
       ("Tools -->" nil
	(("Ambig? -->" nil ("Check Ambiguity..." "View Results..."))
	 ("Legal? -->" nil ("Check Constraint..." "View Results..."))
	 "Probe..."
	 ("Verify -->" nil
	  ("Run Verifier..." "View Results..."
	   ("Box Creation List -->" nil
	    (("Modify -->" nil
	      ("Create Files" "Create Users/Groups"
	       "Flush Files" "Flush Users" "Postpone Creation"))
	     "Show Current Box..." "Show List..."))
	   ))
	 ))
       "Options..."
       ))
    ))
  (update-command-inactive-list)
  )

;;;------------------------------------------------------------
;;; Create-Box-Creation-Buttons creates buttons to select the parent
;;; of the currently displayed box.
;;;------------------------------------------------------------
(defun create-box-creation-buttons ()
  (create-instance 'box-creation1-button command-button-panel
		   (:fixed-width-p T)
		   (:fixed-width-size 120)
		   (:fixed-height-p T)
		   (:fixed-height-size 25)
		   (:active-fill (o-formula (gv *colors* :white)))
		   (:inactive-fill (o-formula (gv *colors*
						  :medium-gray)))
		   (:left (o-formula
			   (+ (gvl :parent :left) (gvl :parent :width)
			      (- (gvl :fixed-width-size))
			      -3)))
		   (:top 5)
		   (:items
		    `(("Select Parent(s)" prepare-next-box-creation)
		      ("Select Children" prepare-next-box-creation)
		      ))
		   )
  (opal:add-component box-creation1-agg box-creation1-button)
  )


;;;------------------------------------------------------------
;;; Create-Help-Buttons creates a button to display a help message.
;;;------------------------------------------------------------
(defun create-help-buttons ()
  (create-instance 'help-display opal:aggregadget
		   (:visible T)
		   (:draw-function :xor)
		   (:left 0) (:top 0)
		   (:width (o-formula (+ 30 (gvl :string :width))))
		   (:height (o-formula (+ 30 (gvl :string :height))))
		   (:parts
		    `((:frame ,opal:rectangle
			      (:line-style ,(o-formula (gv *fonts-and-styles*
							   :db-frame-style)))
			      (:filling-style ,(o-formula (gv *colors* :white)))
			      (:where :back)
			      (:top ,(o-formula (gvl :parent :top)))
			      (:left ,(o-formula (gvl :parent :left)))
			      (:width ,(o-formula (gvl :parent :width)))
			      (:height ,(o-formula (gvl :parent :height)))
			      )
		      (:string ,opal:multi-text
			       (:string ,*help-string*)
			       (:top ,(o-formula (+ 15 (gvl :parent :top))))
			       (:left ,(o-formula (+ 15 (gvl :parent :left))))
			       (:font ,(o-formula (gv *fonts-and-styles*
						      :label-font)))
			       )
		      ))
		   )
  (push help-display *all-dboxes*)
  (create-instance 'help-buttons command-button-panel
		   (:fixed-width-p T)
		   (:fixed-width-size 40)
		   (:fixed-height-p T)
		   (:fixed-height-size 25)
		   (:active-fill (o-formula (gv *colors* :white)))
		   (:inactive-fill (o-formula (gv *colors* :medium-gray)))
		   (:left (o-formula
			   (+ (gvl :parent :left) (gvl :parent :width)
			      (- (gvl :fixed-width-size))
			      -3)))
		   (:top (o-formula
			  (+ (gvl :parent :top) (gvl :parent :height)
			     (- (gvl :fixed-height-size))
			     -3)))
		   (:items
		    `(("Help"
		       ,#'(lambda (&rest args)
			    (declare (ignore args))
			    (unless *dont-interfere*
				    (add-dbox help-display)
				    (opal:deiconify-window dialog-window)
				    (opal:raise-window dialog-window)
				    (block-interference
				     :help-msg
				     "Press any key or mouse button to
continue."
				     )
				    (inter:change-active help-inter T)
				    )))
		      )))
  (opal:add-component help-agg help-buttons)
  )


;;;------------------------------------------------------------
;;; Create-Zoom-Buttons creates a trill device for scaling.  Scaling
;;; is done around (0,0) and the horizontal and vertical offsets are
;;; adjusted to keep the center of work-window in the same place
;;; relative to the picture.
;;;------------------------------------------------------------
(defun create-zoom-buttons ()
  ;; an aggregate to hold the "zoom" controls
  (create-instance 'zoom-agg scroll-agg
		   (:left (o-formula (+ (gvl :parent :left) 5)))
		   (:top (o-formula (gvl :parent :top)))
		   (:real-value 0)
		   (:page-incr 100)
		   (:display-box
		    #'(lambda (obj box)
			(let* ((ww-width (g-value work-window :width))
			       (ww-height (g-value work-window :height))
			       (xoff (g-value pic-sp :h-scroll :value))
			       (yoff (g-value pic-sp :v-scroll :value))
			       (scale (g-value obj :scale))
			       (x (round (* scale (first box))))
			       (y (round (* scale (second box))))
			       (width (round (* scale (third box))))
			       (height (round (* scale (fourth box))))
			       (max-coords (g-value pic-sp :val-2))
			       (max-x (first max-coords))
			       (max-y (second max-coords))
			       )
			  ;; adjust x offset if necessary
			  (cond
			   ;; box to the left of the window
			   ((< (+ x width) xoff)
			    (s-value (g-value pic-sp :h-scroll) :value
				     (min-max (+ x width 5
						 (- ww-width)) 0 max-x)))
			   ;; box to the right of the window
			   ((< (+ xoff ww-width) x)
			    (s-value (g-value pic-sp :h-scroll) :value
				     (min-max (- x 5) 0 max-x)))
			   )

			  ;; adjust y offset if necessary
			  (cond
			   ;; box above the window
			   ((< (+ y height) yoff)
			    (s-value (g-value pic-sp :v-scroll) :value
				     (min-max (+ y height 5
						 (- ww-height)) 0 max-y))) 
			   ;; box below the window
			   ((< (+ yoff ww-height) y)
			    (s-value (g-value pic-sp :v-scroll) :value
				     (min-max (- y 5) 0 max-y)))
			   )
			  (call-schema pic-sp :selection-function
				       (g-value pic-sp :value))
			  )))
		   (:set-center
		     #'(lambda (obj h-offset v-offset oldscale)
			 (let* ((ww-width (g-value work-window :width))
				(ww-height (g-value work-window :height))
				(x (/ ww-width 2))
				(y (/ ww-height 2))
				(scale (g-value obj :scale))
				(new-hoff (round (- (* (/ scale oldscale)
						       (+ h-offset x))
						    x)))
				(new-voff (round (- (* (/ scale oldscale)
						       (+ v-offset y))
						    y)))
				(max-x (first (g-value pic-sp :val-2)))
				(max-y (second (g-value pic-sp :val-2)))
				;; find the bounding box of the picture
				;; and try to keep the picture in the
				;; window.
				(bb (g-value pic-sp :bb-box))
				(bb-left (if bb (round (* scale (first bb))) 0))
				(bb-top (if bb (round (* scale (second bb))) 0))
				(bb-right (+ bb-left
					     (if bb (round (* scale (third bb)))
					       0)))
				(bb-bottom (+ bb-top
					      (if bb (round (* scale (fourth bb)))
						0)))

				;; try to keep an edge centered, at
				;; worst.
				(min-hoff (min-max (round (- bb-left x)) 0 max-x))
				(min-voff (min-max (round (- bb-top y)) 0 max-y))
				(max-hoff (min-max (round (- bb-right x)) 0 max-x))
				(max-voff (min-max (round (- bb-bottom y)) 0 max-y))

				;; try to keep the picture in the window
				(centered-hoff
				 (min-max (round (- (/ (+ bb-left bb-right) 2) x))
					  0 max-x))
				(centered-voff
				 (min-max (round (- (/ (+ bb-top bb-bottom) 2) y))
					  0 max-y))
				;;(min-hoff (min-max (- bb-left ww-width) 0 max-x))
				;;(min-voff (min-max (- bb-top ww-height) 0 max-y))
				;;(max-hoff (min bb-right max-x))
				;;(max-voff (min bb-bottom max-y))
				)
			   (s-value (g-value pic-sp :h-scroll) :value
				    (if (<= min-hoff new-hoff max-hoff)
					new-hoff centered-hoff))
			   (s-value (g-value pic-sp :v-scroll) :value
				    (if (<= min-voff new-voff max-voff)
					new-voff centered-voff))
			   (call-schema pic-sp :selection-function
					(g-value pic-sp :value))
			   )))
		   (:reset-actions
		    #'(lambda ()
			(s-value zoom-agg :real-value 0)
			(mark-as-changed (g-value pic-sp :h-scroll) :val-2)
			(mark-as-changed (g-value pic-sp :v-scroll) :val-2)
			(s-value (g-value pic-sp :h-scroll) :value 0)
			(s-value (g-value pic-sp :v-scroll) :value 0)
			(call-schema pic-sp :selection-function '(0 0))
			(opal:update sb-window)
			))
		   (:scale
		    (o-formula
		     (let ((sv (gvl :real-value)))
		       (cond ((= sv 0) 1)
			     ((< sv 0)
			      (/ 1 (+ 1 (- sv))))
			     (T ;; (> sv 0)
			      (+ 1 sv))))))
		   (:label-string (o-formula
				   (format nil "Scale: ~S"
					   (gvl :scale))))
		   (:val-1 -109) (:val-2 109)
		   (:value-feedback-p nil)
		   (:selection-function
		    #'(lambda (obj value)
			(declare (ignore obj))
			(let ((val (cond
				    ;; don't need to mess with value
				    ((<= -9 value 9) value)

				    ;; value just went out of range
				    ((= value 10) 9)
				    ((= value -10) -9)

				    ;; scale was <= 1, scaling down
				    ((< value -99)
				     (max -9 (+ (* 2 (+ value 99)) 1)))

				    ;; scale was > 1, scaling down
				    ((< value -90)
				     (- (floor (+ value 101) 2) 1))

				    ;; scale was >= 1, scaling up
				    ((> value 99)
				     (min 9 (- (* 2 (- value 99)) 1)))

				    ;; scale was < 1, scaling up
				    ((> value 90)
				     (+ (ceiling (- value 101) 2) 1))

				    ;; should never get here
				    (T
				     (cerror "" "invalid value for zoom-agg: ~S" value))
				    ))
				    
			      (oldscale (g-value zoom-agg :scale))
			      (old-hoff (first (g-value pic-sp :value)))
			      (old-voff (second (g-value pic-sp :value)))
			      )
			  (s-value (g-value zoom-agg :scroll) :value val)
			  (s-value zoom-agg :real-value val)
			  (mark-as-changed (g-value pic-sp :h-scroll) :val-2)
			  (mark-as-changed (g-value pic-sp :v-scroll) :val-2)
			  (call-schema zoom-agg :set-center
				       old-hoff old-voff oldscale)
			  (opal:update sb-window)
			  )))
		   )
  )

;;;============================================================
;;; DIALOG BOXES
;;;============================================================

;;;------------------------------------------------------------
;;; create the box and arrow dialog boxes
;;;------------------------------------------------------------
(defun create-dialog-boxes ()
  (create-popup-menu)			; create this here, after
					; priority levels have been
					; created.
  (make-box-db)
  (make-arrow-db)
  )

;;;------------------------------------------------------------
;;; create the yes-no buttons (*y-n-buttons*)
;;;
;;; The following slots need to be set before the buttons are
;;; activated:
;;;
;;;  :yes-fn is the function that will be called if/when the "yes"
;;;  button is pressed.
;;;
;;;  :no-fn is the function that will be called if/when the "no"
;;;  button is pressed.
;;;
;;;  :question is the string, to be placed in the help window when the
;;;  buttons are activated.
;;;
;;; The following slots are provided:
;;;
;;;  :how-to-start is a function that activates the buttons.
;;;
;;;  :how-to-stop is a function that deactivates the buttons.  It
;;;  should be called if :how-to-start was used to activate the
;;;  buttons.
;;;
;;;  :how-to-abort is a function that is called if the user types
;;;  "^G".
;;;------------------------------------------------------------
(defun make-y-n-buttons ()
  (create-instance
   '*y-n-buttons* opal:aggregadget
   (:top 0) (:left 0)
   (:width (o-formula (gvl :frame :width)))
   (:height (o-formula (gvl :frame :height)))
   (:visible T)
   (:yes-fn nil)
   (:no-fn nil)
   (:question "")
   (:help-msg nil)
   (:how-to-start
    #'(lambda ()
	(add-dbox *y-n-buttons*)
	(block-interference
	 :help-msg (g-value *y-n-buttons* :help-msg))
	(set-abort-inter (g-value *y-n-buttons* :text-button-press))
	(opal:deiconify-window dialog-window)
	(opal:raise-window dialog-window)
	))
   (:how-to-stop
    #'(lambda ()
	(remove-dbox *y-n-buttons*)
	(allow-interference)
	(clear-abort-inter)
	(opal:lower-window dialog-window)
	))
   (:how-to-abort
    #'(lambda ()
	(funcall (g-value *y-n-buttons* :how-to-stop))
	))
   (:parts
    `((:frame ,opal:rectangle
	      (:line-style ,(o-formula (gv *fonts-and-styles*
					   :db-frame-style)))
	      (:filling-style ,(o-formula (gv *colors* :white)))
	      (:where :back)
	      (:top ,(o-formula (gvl :parent :top)))
	      (:left ,(o-formula (gvl :parent :left)))
	      (:width ,(o-formula (+ (* 2 (gv *fonts-and-styles*
					      :db-frame-width))
				     (max (gvl :parent
					       :question-string
					       :width)
					  (gvl :parent :y-n-buttons
					       :width)))))
	      (:height ,(o-formula (+ (* 2 (gv *fonts-and-styles*
					       :db-frame-width))
				      (gvl :parent :question-string
					      :height)
				      10
				      (gvl :parent :y-n-buttons :height))))
	      )
      (:question-string ,opal:text
			(:left ,(o-formula (gv *fonts-and-styles*
					       :db-frame-width)))
			(:top ,(o-formula (gv *fonts-and-styles*
					      :db-frame-width)))
			(:string ,(o-formula (gvl :parent :question)))
			(:font ,(o-formula (gv *fonts-and-styles*
					       :label-font)))
			)
      (:y-n-buttons ,command-button-panel
		    (:top ,(o-formula (+ (gvl :parent :question-string :top)
					 (gvl :parent :question-string :height)
					 5)))
		    (:left ,(o-formula (+ 5 (gv *fonts-and-styles*
						:db-frame-width))))
		    (:h-spacing 10)
		    (:direction :horizontal)
		    (:fixed-width-p T)
		    (:items
		     (("Yes"
		       ,#'(lambda (&rest args)
			    (apply (g-value *y-n-buttons* :yes-fn) args)))
		      ("No"
		       ,#'(lambda (&rest args)
			    (apply (g-value *y-n-buttons* :no-fn) args)))
		      ))
		    )
      ))
   )
  (push *y-n-buttons* *all-dboxes*)
  (s-value (g-value *y-n-buttons* :y-n-buttons :button-panel :text-button-press)
	   :how-to-abort
	   (o-formula (gv *y-n-buttons* :how-to-abort)))
  *y-n-buttons*
  )

;;;------------------------------------------------------------
;;; create the box dialog box (*box-db*)
;;; parts = frame + label + list 
;;; list = data + name + type + role + thickness + starred + done
;;;------------------------------------------------------------

(defun make-box-db ()
  ;; main aggregate
  (create-instance '*box-db* opal:aggregadget
		   (:visible T)

		   (:draw-function :xor)

		   (:left 0) (:top 0)
		   (:width (o-formula (max (+ (* 2 (gv *fonts-and-styles*
						       :db-frame-width))
					      (gvl :label :width))
					   (+ (* 2 (gv *fonts-and-styles*
						       :db-frame-width))
					      *db-left-offset*
					      (gvl :list :width)))))
		   (:height (o-formula (+ (* 2 (gv *fonts-and-styles*
						   :db-frame-width))
					  *db-top-offset*
					  (gvl :label :height)
					  (gvl :list :height))))
		   (:parts
		    `(
		      ;; Frame
		      (:frame ,opal:rectangle
			      (:line-style ,(o-formula (gv *fonts-and-styles*
							   :db-frame-style)))
			      (:filling-style ,(o-formula (gv *colors* :white)))
			      (:where :back)
			      (:top ,(o-formula (gvl :parent :top)))
			      (:left ,(o-formula (gvl :parent :left)))
			      (:width ,(o-formula (gvl :parent :width)))
			      (:height ,(o-formula (gvl :parent :height))))
		      ;; Label
		      (:label ,opal:text
			      (:font ,(o-formula (gv *fonts-and-styles*
						     :label-font)))
			      (:top ,(o-formula (+ (gvl :parent :top)
						   (gv *fonts-and-styles*
						       :db-frame-width))))
			      (:left ,(o-formula (+ (gvl :parent :left)
						    (gv *fonts-and-styles*
							:db-frame-width))))
			      (:string "BOX   ATTRIBUTES:"))
		      ;; list for components 
		      (:list ,opal:aggrelist
			     (:top ,(o-formula (+ (gvl :parent :label :top)
						  (gvl :parent :label :height)
						  *db-top-offset*)))
			     (:left ,(o-formula (+ (gvl :parent :label :left)
						   *db-left-offset*)))
			     )
		      )))
  (push *box-db* *all-dboxes*)

  ;; create items for box-db-list

  ;; Data -- misc info for debugging. not editable
  (create-instance 'box-db-data opal:text
		   (:font (o-formula (gv *fonts-and-styles*
					 :button-label-font)))
		   (:string (o-formula
			     (let* (
				    (the-box (gv *box-db* :selected-box))
				    (name (if the-box (gv the-box :sysname)
					    "")))
			       (format nil "SYSNAME:  ~A" name)))))

  ;; Name
  (create-instance 'box-db-name dbox-labeled-multi-text
		   (:label-string "NAME:"))
  (let ((name-inter (g-value box-db-name :text-inter)))
    (s-value name-inter :start-action
	     #'(lambda (interactor objbeingchanged event)
		 (set-abort-inter name-inter)
		 (call-prototype-method interactor objbeingchanged event)))
    (s-value name-inter :stop-action
	     #'(lambda (interactor objbeingchanged event)
		 (clear-abort-inter)
		 (call-prototype-method interactor objbeingchanged event)
		 ))
    )

  ;; Type
  (create-instance 'box-db-type dbox-labeled-multi-text
		   (:visible (o-formula (eq (gv pictype-menu
						:value) :instance)))
		   (:label-string "TYPE:")
		   (:multi-text nil))
  (let ((type-inter (g-value box-db-type :text-inter)))
    (s-value type-inter :start-action
	     #'(lambda (interactor objbeingchanged event)
		 (set-abort-inter type-inter)
		 (call-prototype-method interactor objbeingchanged event)))
    (s-value type-inter :stop-action
	     #'(lambda (interactor objbeingchanged event)
		 (clear-abort-inter)
		 (call-prototype-method interactor objbeingchanged event)
		 ))
    )

  ;; type menu
  (create-instance
   'box-db-type-menu opal:aggregadget
   (:visible (o-formula (eq (gv pictype-menu :value) :instance)))
   (:displaying-menu nil)
   (:parts
    `((:show-button ,command-button-panel
		    (:visible ,(o-formula (and (gvl :parent :visible)
					       (not (gvl :parent
							 :displaying-menu)))))
		    (:top ,(o-formula (gvl :parent :top)))
		    (:left ,(o-formula (gvl :parent :left)))
		    (:items (("Show Type Menu"
			      ,#'(lambda (obj &rest args)
				   (declare (ignore args))
				   (s-value (g-value obj :parent)
					    :displaying-menu T)
				   ))))
		    )
      (:hide-button ,command-button-panel
		    (:visible ,(o-formula (and (gvl :parent :visible)
					       (gvl :parent
						    :displaying-menu))))
		    (:top ,(o-formula (gvl :parent :top)))
		    (:left ,(o-formula (gvl :parent :left)))
		    (:items (("Hide Type Menu"
			      ,#'(lambda (obj &rest args)
				   (declare (ignore args))
				   (s-value (g-value obj :parent)
					    :displaying-menu nil)
				   ))))
		    )
      (:menu ,dbox-button-entry
	     (:visible ,(o-formula
			 (let ((visible (and (gvl :parent :visible)
					     (gvl :parent :displaying-menu)))
			       (value (gv box-db-type :value))
			       (self (gv :self))
			       )
			   (when visible
				 (call-schema self :set-value
					      (string-capitalize
					       value)))
			   visible)))
	     (:top ,(o-formula (+ (gvl :parent :hide-button :top)
				  (gvl :parent :hide-button :height)
				  10)))
	     (:left ,(o-formula (gvl :parent :left)))
	     (:label-string "TYPES:")
	     (:items ("Device" "Dir" "File" "Group" "Home" "Root"
		      "Socket" "User" "World"))
	     (:selection-function
	      ,#'(lambda (obj nv)
		   (declare (ignore obj))
		   (call-schema box-db-type :set-value
				(string-downcase nv))
		   ))
	     )
      )))

  ;; Role
  (create-instance 'box-db-role dbox-button-entry
		   (:visible (o-formula (eq (gv pictype-menu
						:value) :instance)))
		   (:label-string "ROLE:")
		   (:items '("User" "File" "Unknown")))

  ;; constraint pics only
  ;; Thickness
  (create-instance 'box-db-thickness dbox-button-entry
		   (:visible (o-formula (eq (gv pictype-menu
						:value) :constraint)))
		   (:label-string "THICKNESS:")
		   (:items '("Thin" "Thick")))

  ;; Starred
  (create-instance 'box-db-starred dbox-button-entry
		   (:visible (o-formula (and (gv *box-db* :visible)
					     (eq (gv pictype-menu
						     :value) :constraint))))
		   (:toggle T)
		   (:label-string "CONTAINMENT:")
		   (:items '("Starred")))

  ;; Done
  (create-instance 'box-db-done dbox-tbutton-entry
		   (:label-string "<EXIT>:")
		   (:items '(("Save Changes" box-db-exit-and-save)
			     ("Abort" box-db-exit-no-change)
			     ("Abort All" box-db-exit-all-no-change)
			     )))

  (opal:add-components 
   (g-value *box-db* :list)
   box-db-data box-db-name box-db-type box-db-type-menu box-db-role
   box-db-thickness box-db-starred box-db-done)
  ;; return dialog box
  *box-db*
  )

;;;------------------------------------------------------------
;;; Exit and save -- save the current values of the attributes of the
;;; object, and "exit" the dialog box (remove from aggregate)
;;;------------------------------------------------------------
(defun box-db-exit-and-save (gadg-obj item-str)
  (declare (ignore gadg-obj item-str))
  (let ((box (g-value *box-db* :selected-box)))
    ;; save strings
    (s-value (g-value box :label) :string (g-value box-db-name :value))
    (s-value box :box-type (g-value box-db-type :value))
    ;; save button values
    (s-value box :box-role (g-value box-db-role :button-list :value))
    (s-value box :thick
	     (string-equal (g-value box-db-thickness :button-list :value)
			   "Thick")) 
    (s-value box :starred (g-value box-db-starred :button-list :value))
    (remove-dbox *box-db*)
    (unless (display-next-object)
	    ;; we may have invalidated ambig results
	    (when (g-value ambig-status :guaranteed-valid)
		  (s-value ambig-status :guaranteed-valid nil))
	    (allow-interference)
	    (opal:lower-window dialog-window)
	    )
    )
  )

;;;------------------------------------------------------------
;;; Exit no change -- do not save changes, just "exit" the dialog box
;;; (remove from aggregate) 
;;;------------------------------------------------------------
(defun box-db-exit-no-change (gadg-obj item-str)
  (declare (ignore gadg-obj item-str))
  (remove-dbox *box-db*)
  (unless (display-next-object)
	  (allow-interference)
	  (opal:lower-window dialog-window)
	  )
  )

(defun box-db-exit-all-no-change (gadg-obj item-str)
  (display-next-object T)
  (box-db-exit-no-change gadg-obj item-str))


;;; --------------------------------------------------
;;; create the arrow dialog box (*arrow-db*)
;;; parts = frame + label + list
;;; list = name + parity + kind + thickness + starred + done 
;;; --------------------------------------------------

(defun make-arrow-db ()
  ;; main aggregate
  (create-instance '*arrow-db* opal:aggregadget
		   (:visible T)

		   (:draw-function :xor)

		   (:left 0) (:top 0)
		   (:width (o-formula (max (+ (* 2 (gv *fonts-and-styles*
						       :db-frame-width))
					      (gvl :label :width))
					   (+ (* 2 (gv *fonts-and-styles*
						       :db-frame-width))
					      *db-left-offset*
					      (gvl :list :width)))))
		   (:height (o-formula (+ (* 2 (gv *fonts-and-styles*
						   :db-frame-width))
					  *db-top-offset*
					  (gvl :label :height)
					  (gvl :list :height))))
		   (:parts
		    `(
		      ;; Frame
		      (:frame ,opal:rectangle
			      (:line-style ,(o-formula (gv *fonts-and-styles*
							   :db-frame-style)))
			      (:filling-style ,(o-formula (gv *colors* :white)))
			      (:where :back)
			      (:top ,(o-formula (gvl :parent :top)))
			      (:left ,(o-formula (gvl :parent :left)))
			      (:width ,(o-formula (gvl :parent :width)))
			      (:height ,(o-formula (gvl :parent :height))))
		      ;; Label
		      (:label ,opal:text
			      (:font ,(o-formula (gv *fonts-and-styles*
						     :label-font)))
			      (:top ,(o-formula (+ (gvl :parent :top)
						   (gv *fonts-and-styles*
						       :db-frame-width))))
			      (:left ,(o-formula (+ (gvl :parent :left)
						    (gv *fonts-and-styles*
							:db-frame-width))))
			      (:string "ARROW   ATTRIBUTES:"))
		      ;; list for components 
		      (:list ,opal:aggrelist
			     (:top ,(o-formula (+ (gvl :parent :label :top)
						  (gvl :parent :label :height)
						  *db-top-offset*)))
			     (:left ,(o-formula (+ (gvl :parent :label :left)
						   *db-left-offset*)))
			     )
		      )))
  (push *arrow-db* *all-dboxes*)

  ;; create items for arrow-db-list

  (create-instance
   'arrow-db-to-from opal:aggregadget
   (:displaying-endpoints T)
   (:parts
    `((:show-ep-button ,command-button-panel
		       (:visible
			,(o-formula (not (gvl :parent
					      :displaying-endpoints))))
		       (:top ,(o-formula (gvl :parent :top)))
		       (:left ,(o-formula (gvl :parent :left)))
		       (:items (("Show Endpoints"
				 ,#'(lambda (obj &rest args)
				      (declare (ignore args))
				      (s-value (g-value obj :parent)
					       :displaying-endpoints
					       T)))))
		       )
      (:hide-ep-button ,command-button-panel
		       (:visible
			,(o-formula (gvl :parent
					 :displaying-endpoints)))
		       (:top ,(o-formula (gvl :parent :top)))
		       (:left ,(o-formula (gvl :parent :left)))
		       (:items (("Hide Endpoints"
				 ,#'(lambda (obj &rest args)
				      (declare (ignore args))
				      (s-value (g-value obj :parent)
					       :displaying-endpoints
					       nil)))))
		       )
      (:from-label ,opal:text
		   (:visible ,(o-formula (gvl :parent
					      :displaying-endpoints)))
		   (:top ,(o-formula (+ (gvl :parent :hide-ep-button
					     :top)
					(gvl :parent :hide-ep-button
					     :height)
					10)))
		   (:left ,(o-formula (gvl :parent :left)))
		   (:font ,(o-formula (gv *fonts-and-styles*
					  :button-label-font)))
		   (:string "FROM:")
		   )
      (:from-box ,opal:multi-text
		 (:visible ,(o-formula (gvl :parent
					    :displaying-endpoints)))
		 (:top ,(o-formula (gvl :parent :from-label :top)))
		 (:left ,(o-formula (+ (gvl :parent :from-label :left)
				       (gvl :parent :from-label :width)
				       5)))
		 (:font ,(o-formula (gv *fonts-and-styles*
					:button-label-font)))
		 (:string
		  ,(o-formula (or (gv *arrow-db*
				      :selected-arrow
				      :from :label :string)
				  "")))
		 )
      (:to-label ,opal:text
		 (:visible ,(o-formula (gvl :parent
					    :displaying-endpoints)))
		 (:top ,(o-formula (+ (gvl :parent :from-box :top)
				      (gvl :parent :from-box :height)
				      10)))
		 (:left ,(o-formula (gvl :parent :left)))
		 (:font ,(o-formula (gv *fonts-and-styles*
					:button-label-font)))
		 (:string "TO:")
		 )
      (:to-box ,opal:multi-text
	       (:visible ,(o-formula (gvl :parent
					  :displaying-endpoints)))
	       (:top ,(o-formula (gvl :parent :to-label :top)))
	       (:left ,(o-formula (+ (gvl :parent :to-label :left)
				     (gvl :parent :to-label :width)
				     5)))
	       (:font ,(o-formula (gv *fonts-and-styles*
				      :button-label-font)))
	       (:string
		,(o-formula (or (gv *arrow-db* :selected-arrow
				    :to  :label :string)
				"")))
	       )
      )))
  ;; Name
  (create-instance 'arrow-db-name dbox-labeled-multi-text
		   ;; not visible on containment arrows
		   (:visible (o-formula 
			      (and (gv *arrow-db* :visible)
				   (not (eq
					 (gv arrow-db-type :button-list :value)
					 :con)))))
		   (:label-string "LABEL:")
		   (:multi-text nil))
  (let ((name-inter (g-value arrow-db-name :text-inter)))
    (s-value name-inter :start-action
	     #'(lambda (interactor objbeingchanged event)
		 (set-abort-inter name-inter)
		 (call-prototype-method interactor objbeingchanged event)))
    (s-value name-inter :stop-action
	     #'(lambda (interactor objbeingchanged event)
		 (clear-abort-inter)
		 (call-prototype-method interactor objbeingchanged event)
		 ))
    )

  ;; Parity
  (create-instance 'arrow-db-parity dbox-button-entry
		   (:label-string "PARITY:")
		   (:items '("Pos" "Neg")))

  ;; Type
  (create-instance 'arrow-db-type dbox-button-entry
		   (:visible (o-formula (and (gv *arrow-db* :visible)
					     (eq (gv pictype-menu
						     :value) :constraint))))
		   (:label-string "TYPE:")
		   (:items '(:syn :sem :con)))


  ;; Thickness
  (create-instance 'arrow-db-thickness dbox-button-entry
		   (:visible (o-formula (and (gv *arrow-db* :visible)
					     (eq (gv pictype-menu
						     :value) :constraint))))
		   (:label-string "THICKNESS:")
		   (:items '("Thin" "Thick")))

  ;; Starred
  (create-instance 'arrow-db-starred dbox-button-entry
		   (:visible (o-formula (and (gv *arrow-db* :visible)
					     (eq (gv pictype-menu
						     :value) :constraint))))
		   (:toggle T)
		   (:label-string "CONTAINMENT:")
		   (:items '("Starred")))

  ;; Done
  (create-instance 'arrow-db-done dbox-tbutton-entry
		   (:label-string "<EXIT>:")
		   (:items '(("Save Changes" arrow-db-exit-and-save)
			     ("Abort" arrow-db-exit-no-change)
			     ("Abort All" arrow-db-exit-all-no-change)
			     )))

  (opal:add-components 
   (g-value *arrow-db* :list) 
   arrow-db-to-from arrow-db-name arrow-db-parity arrow-db-type
   arrow-db-thickness arrow-db-starred arrow-db-done) 
  ;; return dialog box
  *arrow-db*
  )

;;;------------------------------------------------------------
;;; Exit and save -- save the current values of the attributes of the
;;; object, and "exit" the dialog box (remove from aggregate)
;;;------------------------------------------------------------
(defun arrow-db-exit-and-save (gadg-obj item-str)
  (declare (ignore gadg-obj item-str))
  (let ((arrow (g-value *arrow-db* :selected-arrow)))
    ;; save strings
    ;; :value of labeled-texts not working right now
    (s-value (g-value arrow :label) :string (g-value arrow-db-name :value))
    ;; save button values
    (s-value arrow :neg 
	     (string-equal (g-value arrow-db-parity :button-list :value)
			   "Neg")) 
    (s-value arrow :arrow-type (g-value arrow-db-type :button-list :value))
    (s-value arrow :thick
	     (string-equal (g-value arrow-db-thickness :button-list
				    :value) "Thick")) 
    ;; only keep star if containment arrow
    (s-value arrow :starred
	     (and (g-value arrow-db-starred :button-list :value)
		  (equal (g-value arrow :arrow-type) :con)))
    (remove-dbox *arrow-db*)
    (unless (display-next-object)
	    ;; we may have invalidated ambig results
	    (when (g-value ambig-status :guaranteed-valid)
		  (s-value ambig-status :guaranteed-valid nil))
	    (allow-interference)
	    (opal:lower-window dialog-window)
	    )
    )
  )

;;;------------------------------------------------------------
;;; Exit no change -- do not save changes, just "exit" the dialog box
;;; (remove from aggregate) 
;;;------------------------------------------------------------
(defun arrow-db-exit-no-change (gadg-obj item-str)
  (declare (ignore gadg-obj item-str))
  (remove-dbox *arrow-db*)
  (unless (display-next-object)
	  (allow-interference)
	  (opal:lower-window dialog-window)
	  )
  )

(defun arrow-db-exit-all-no-change (gadg-obj item-str)
  (display-next-object T)
  (arrow-db-exit-no-change gadg-obj item-str))

;;;------------------------------------------------------------
;;; Display-Box (called by display-object)
;;;  set values in *box-db* based on currently selected box, then
;;;  activate interactor to place dialog box
;;;------------------------------------------------------------
(defun display-box (box &optional start-text-inter)
  (unless *dont-interfere*
	  (block-interference
	   :help-msg
	   (format
	    nil
	    "Change attributes as
needed.  Press
~S to
save changes, ~S
to abort."
	    "Save Changes" "Abort")
	   :leave-work-window-alone T))
  ;; set slots for selected box and make box-db visible
  (s-value *box-db* :selected-box box)
  ;; set strings
  (call-schema box-db-name :set-value (g-value box :label :string))
  (call-schema box-db-type :set-value (g-value box :box-type))
  (call-schema (g-value box-db-type-menu :menu) :set-value
	       (string-capitalize (g-value box :box-type)))
  ;; set buttons
  (call-schema (g-value box-db-role :button-list) :set-value
	       (string-capitalize (g-value box :box-role)))
  (call-schema (g-value box-db-thickness :button-list) :set-value
	       (if (g-value box :thick) "Thick" "Thin"))
  (call-schema (g-value box-db-starred :button-list) :set-value
	       (if (g-value box :starred) "Starred" NIL))
  ;; make the dialog box visible
  (add-dbox *box-db*)
  (opal:deiconify-window dialog-window)
  (opal:raise-window dialog-window)
  (when start-text-inter (call-schema box-db-name :start-text-inter))
  )

;;;------------------------------------------------------------
;;; Display-Arrow (called by display-object)
;;;  set values in *arrow-db* based on currently selected arrow, then
;;;  activate interactor to place dialog box
;;;------------------------------------------------------------
(defun display-arrow (arrow &optional start-text-inter)
  (unless *dont-interfere*
	  (block-interference
	   :help-msg
	   (format
	    nil
	    "Change attributes as
needed.  Press
~S to
save changes, ~S
to abort."
	    "Save Changes" "Abort"))
   :leave-work-window-alone T)
  ;; set slots for selected arrow and make arrow-db visible
  (s-value *arrow-db* :selected-arrow arrow)
  ;; set strings
  (call-schema arrow-db-name :set-value (g-value arrow :label :string))
  ;; set buttons
  (call-schema (g-value arrow-db-parity :button-list) 
	       :set-value (if (g-value arrow :neg) "Neg" "Pos")) 
  (call-schema (g-value arrow-db-type :button-list) :set-value
	       (g-value arrow :arrow-type))
  (call-schema (g-value arrow-db-thickness :button-list) 
	       :set-value (if (g-value arrow :thick) "Thick" "Thin")) 
  (call-schema (g-value arrow-db-starred :button-list) 
	       :set-value (if (g-value arrow :starred) "Starred" NIL))
  ;; make the dialog box visible
  (add-dbox *arrow-db*)
  (opal:deiconify-window dialog-window)
  (opal:raise-window dialog-window)
  (when start-text-inter (call-schema arrow-db-name :start-text-inter))
  )


;;;============================================================
;;; HELPER FUNCTIONS
;;;============================================================

;;;------------------------------------------------------------
;;; get the nth component of a list of components
;;; remember nth starts with 0
;;;------------------------------------------------------------
(defun nth-component (n agg)
  (nth n (get-values agg :components)))

;;;------------------------------------------------------------
;;; functions to check size of dboxes (used for debugging)
;;;------------------------------------------------------------
(defun w-h (obj)
    (format T "Obj ~A: [~A,~A]~%" 
	    obj (g-value obj :width) (g-value obj :height))) 

(defun check-size (dbox)
  (w-h dbox)
  (format T "   ")
  (w-h (g-value dbox :frame))
  (format T "   ")
  (w-h (g-value dbox :label))
  (format T "   ")
  (w-h (g-value dbox :list))
  (dolist (obj (get-values (g-value dbox :list) :components))
	  (format T "      ")
	  (w-h obj)))
