;;;             -*- Mode: Lisp; Package: MIRO; -*-
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

;;;
;;; MIRO EDITOR - ScrollPad
;;;

#|
============================================================
Change log:
    11/27/90 ky ; Made filling styles in the scrollbars depend on user
		; settings.
    11/20/90 ky ; Made filling-styles user-settable.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    9/13/90 ky  ; Got rid of a couple of compile-time warnings.
    9/12/90 ky  ; Added support for showing a bounding box.
    7/31/90 ky  ; Created.
============================================================
|#

(in-package 'miro :use '("LISP" "KR"))

;;;---------------------------------------------------------------------------
;;; scrollpad - a two-dimentional scrollbar
;;;
;;;
;;; The scrollpad consists of a gray rectangle with a movable cursor,
;;; a horizontal scrollbar at the bottom of the rectangle, and a
;;; vertical scrollbar on the right side.  The scrollbars work as
;;; expected.  The cursor on the pad is moved by clicking (and
;;; dragging, if desired) the left mouse button while the mouse is in
;;; the pad.
;;;
;;; The following slots may be set by the user:
;;;    top, left, height, width:
;;;      150 is the smallest recommended height or width
;;;
;;;    white-fill, light-gray-fill, gray-fill, black-fill:
;;;      The filling-styles to use for the colors white, light gray,
;;;      gray, and black, respectively.
;;;
;;;    val-1, val-2, cursor-size, scr-incr, page-incr, scr-trill-p,
;;;    page-trill-p, indicator-text-p, int-feedback-p:
;;;      These are lists (horizontal vertical) of values for use in
;;;      the corresponding slots in the scrollbars.
;;;
;;;    min-height:
;;;      The horizontal scrollbar's :min-height.
;;;
;;;    min-width:
;;;      The vertical scrollbar's :min-width.
;;;
;;;    scroll-p, indicator-font:
;;;      These are used in the corresponding slots in the scrollbars.
;;;      These are not lists; both scrollbars will have the same
;;;      value.
;;;
;;;    cursor-size:
;;;      A list (x y) that specifies the size of the cursor on the
;;;      pad.
;;;
;;;    selection-function:
;;;      A function that will be called whenever the value slot
;;;      changes.
;;;
;;; The "value" slot contains the position (list x y) of the cursor in
;;; the pad.  The user should not set this slot, as this would result
;;; in the scrollbars having the wrong value.  (A circular dependency
;;; might work here, but I have seen such things fail unexpectedly,
;;; and would rather avoid the added bug potential...)
;;;---------------------------------------------------------------------------
(create-instance 'scrollpad opal:aggregadget
		 (:top 0)
		 (:left 0)
		 (:height 150)
		 (:width 150)
		 (:white-fill opal:white-fill)
		 (:light-gray-fill opal:light-gray-fill)
		 (:gray-fill opal:gray-fill)
		 (:black-fill opal:black-fill)
		 (:bb-size nil)
		 (:show-bb nil)
		 (:val-1 '(0 0))
		 (:val-2 '(100 100))
		 (:cursor-size '(20 20))
		 (:scr-incr '(1 1))
		 (:page-incr '(5 5))
		 (:scr-trill-p '(T T))
		 (:page-trill-p '(T T))
		 (:indicator-text-p '(T T))
		 (:int-feedback-p '(T T))
		 (:min-width 20)
		 (:min-height 20)
		 (:scroll-p T)
		 (:selection-function nil)
		 (:indicator-font (create-instance NIL opal:font (:size :small)))
		 (:value (o-formula (gvl :pad :value)))
		 (:parts
		  `(
		    (:pad ,opal:rectangle
			  (:where :back)
			  (:top ,(o-formula (gvl :parent :top)))
			  (:left ,(o-formula (gvl :parent :left)))
			  (:height ,(o-formula (- (gvl :parent :height) 20)))
			  (:width ,(o-formula (- (gvl :parent :width) 20)))
			  (:filling-style ,(o-formula
					    (if (gvl :parent :scroll-p)
						(gvl :parent :gray-fill)
					      (gvl :parent :black-fill)
					      )))
			  (:value ,(o-formula
				    (list (gvl :parent :h-scroll :value)
					  (gvl :parent :v-scroll :value))))
			  )
		    (:bounding-box-pad ,opal:rectangle
				       (:where :front)
				       (:top ,(o-formula
					       (+ (gvl :parent :top) 1
						  (second (gvl :parent :bb-size)))))
				       (:left ,(o-formula
						(+ (gvl :parent :left) 1
						   (first (gvl :parent :bb-size)))))
				       (:width ,(o-formula
						 (third (gvl :parent :bb-size))))
				       (:height ,(o-formula
						  (fourth (gvl :parent :bb-size))))
				       (:visible ,(o-formula
						   (and (gvl :parent :show-bb)
							(gvl :parent :scroll-p))))
				       (:line-style ,opal:no-line)
				       (:filling-style
					,(o-formula (gvl :parent :light-gray-fill)))
				       )
		    (:pad-cursor ,opal:rectangle
				 (:where :front)
				 (:min-top ,(o-formula
					     (+ (gvl :parent :top) 1)))
				 (:max-top ,(o-formula
					     (+ (gvl :min-top)
						(- (opal:gvl-sibling :pad :height)
						   (gvl :height)
						   1))))
				 (:min-left ,(o-formula
					     (+ (gvl :parent :left) 1)))
				 (:max-left ,(o-formula
					     (+ (gvl :min-left)
						(- (opal:gvl-sibling :pad :width)
						   (gvl :width)
						   1))))
				 (:top ,(o-formula
					 (let ((val (second (gvl :parent :value)))
					       (val1 (second (gvl :parent :val-1)))
					       (val2 (second (gvl :parent :val-2)))
					       )
					   (inter:clip-and-map
					    val val1 val2
					    (gvl :min-top)
					    (gvl :max-top)
					    ))))
				 (:left ,(o-formula
					  (let ((val (first (gvl :parent :value)))
						(val1 (first (gvl :parent :val-1)))
						(val2 (first (gvl :parent :val-2)))
						)
					    (inter:clip-and-map
					     val val1 val2
					     (gvl :min-left)
					     (gvl :max-left)
					     ))))
				 (:height ,(o-formula
					    (second (gvl :parent :cursor-size))))
				 (:width ,(o-formula
					   (first (gvl :parent :cursor-size))))
				 (:filling-style
				  ,(o-formula (gvl :parent :white-fill)))
				 )
		    (:pad-feedback ,opal:rectangle
				   (:box ,(list 0 0 0 0))
				   (:line-style ,opal:line-2)
				   (:where :front)
				   (:visible nil)
				   (:left ,(o-formula
					    (let ((left (first (gvl :box)))
						  (maxleft (opal:gvl-sibling
							    :pad-cursor
							    :max-left)
							   )
						  (minleft (opal:gvl-sibling
							    :pad-cursor
							    :min-left)))
					      (cond
					       ((< left minleft) minleft)
					       ((> left maxleft) maxleft)
					       (T left))
					      )))
				   (:top ,(o-formula
					   (let ((top (second (gvl :box)))
						 (maxtop (opal:gvl-sibling
							  :pad-cursor
							  :max-top))
						 (mintop (opal:gvl-sibling
							  :pad-cursor
							  :min-top)))
					     (cond
					      ((< top mintop) mintop)
					      ((> top maxtop) maxtop)
					      (T top))
					     )))
				   (:width ,(o-formula
					     (opal:gvl-sibling :pad-cursor :width)))
				   (:height ,(o-formula
					      (opal:gvl-sibling :pad-cursor :height)))
				   )
				   
		    (:h-scroll ,garnet-gadgets:h-scroll-bar
			       (:top ,(o-formula (+ (gvl :parent :top)
						    (gvl :parent :pad :height)
						    )))
			       (:left ,(o-formula (gvl :parent :left)))
			       (:min-height ,(o-formula (gvl :parent :min-height)))
			       (:width ,(o-formula (gvl :parent :pad :width)))
			       (:val-1 ,(o-formula (first (gvl :parent :val-1))))
			       (:val-2 ,(o-formula (first (gvl :parent :val-2))))
			       (:scr-incr ,(o-formula
					    (first (gvl :parent :scr-incr))))
			       (:page-incr ,(o-formula
					     (first (gvl :parent :page-incr))))
			       (:scr-trill-p ,(o-formula
					       (first (gvl :parent :scr-trill-p))))
			       (:page-trill-p ,(o-formula
						(first (gvl :parent :page-trill-p))))
			       (:indicator-text-p ,(o-formula
						    (first
						     (gvl :parent
							  :indicator-text-p))))
			       (:int-feedback-p ,(o-formula
						  (first
						   (gvl :parent :int-feedback-p))))
			       (:scroll-p ,(o-formula (gvl :parent :scroll-p)))
			       (:selection-function
				,#'(lambda (obj val)
				     (declare (ignore val))
				     (let ((parent (g-value obj :parent))
					   (sf (g-value obj :parent
							:selection-function)))
				       (when sf (funcall
						 sf parent
						 (g-value parent :value)))
				       )))
			       (:indicator-font ,(o-formula
						  (gvl :parent :indicator-font)))
			       )
		    (:v-scroll ,garnet-gadgets:v-scroll-bar
			       (:top ,(o-formula (gvl :parent :top)))
			       (:left ,(o-formula (+ (gvl :parent :left)
						    (gvl :parent :pad :width)
						    )))
			       (:min-width ,(o-formula (gvl :parent :min-width)))
			       (:height ,(o-formula (gvl :parent :pad :height)))
			       (:val-1 ,(o-formula (second (gvl :parent :val-1))))
			       (:val-2 ,(o-formula (second (gvl :parent :val-2))))
			       (:scr-incr ,(o-formula
					    (second (gvl :parent :scr-incr))))
			       (:page-incr ,(o-formula
					     (second (gvl :parent :page-incr))))
			       (:scr-trill-p ,(o-formula
					       (second (gvl :parent :scr-trill-p))))
			       (:page-trill-p ,(o-formula
						(second (gvl :parent :page-trill-p))))
			       (:indicator-text-p ,(o-formula
						    (second
						     (gvl :parent
							  :indicator-text-p))))
			       (:int-feedback-p ,(o-formula
						  (second
						   (gvl :parent :int-feedback-p))))
			       (:scroll-p ,(o-formula (gvl :parent :scroll-p)))
			       (:selection-function
				,#'(lambda (obj val)
				     (declare (ignore val))
				     (let ((parent (g-value obj :parent))
					   (sf (g-value obj :parent
							:selection-function)))
				       (when sf (funcall
						 sf parent
						 (g-value parent :value)))
				       )))
			       (:indicator-font ,(o-formula
						  (gvl :parent :indicator-font)))
			       )
		    ))
		 (:interactors
		  `(
		    (:cursor-inter ,inter:move-grow-interactor
				   (:grow-p nil)
				   (:line-p nil)
				   (:active ,(o-formula (gvl :operates-on
							     :scroll-p)))
				   (:obj-to-change
				    ,(o-formula (gvl :operates-on
						     :pad-cursor)))
				   (:start-where
				    ,(o-formula
				      (list :in (gvl :operates-on :pad))))
				   (:running-where ,(o-formula
						     (gvl :start-where)))
				   (:window ,(o-formula (gvl :operates-on
							     :window)))
				   (:feedback-obj ,(o-formula
						    (gvl :operates-on
							 :pad-feedback)))
				   (:continuous T)
				   (:attach-point :center)
				   (:start-action
				    ,#'(lambda (an-inter obj points)
					 (s-value (g-value
						   an-inter
						   :feedback-obj)
						  :visible T)
					 (call-prototype-method
					  an-inter obj points)
					 ))
				   (:back-inside-action
				    ,#'(lambda (an-inter oc obj points)
					 (s-value (g-value
						   an-inter
						   :feedback-obj)
						  :visible T)
					 (call-prototype-method
					  an-inter oc obj points)
					 ))
				   (:outside-action
				    ,#'(lambda (an-inter oc obj)
					 (s-value (g-value
						   an-inter
						   :feedback-obj)
						  :visible nil)
					 (call-prototype-method
					  an-inter oc obj)
					 ))
				   (:stop-action
				    ,#'(lambda (an-inter obj points)
					 (let* ((parent (g-value obj :parent))
						(sf (g-value obj
							     :parent
							     :selection-function))
						(x1 (first (g-value parent :val-1)))
						(x2 (first (g-value parent :val-2)))
						(y1 (second (g-value parent :val-1)))
						(y2 (second (g-value parent :val-2)))
						)
					   (s-value (g-value
						     an-inter
						     :feedback-obj)
						    :visible nil)
					   (s-value
					    (g-value parent :h-scroll)
					    :value
					    (inter:clip-and-map
					     (first points)
					     (+ (g-value parent :left) 1)
					     (+ (g-value parent :left)
						(- (g-value parent :pad :width)
						   (g-value obj :width)
						   1))
					     x1 x2)
					    )
					   (s-value
					    (g-value parent :v-scroll)
					    :value
					    (inter:clip-and-map
					     (second points)
					     (+ (g-value parent :top) 1)
					     (+ (g-value parent :top)
						(- (g-value parent :pad :height)
						   (g-value obj :height)
						   1))
					     y1 y2)
					    )
					   (when sf (funcall
						     sf parent
						     (g-value parent :value)))
					   )))
				   )
		    ))
		 )

;; Fix the colors.
(s-value (g-value scrollpad :v-scroll :indicator) :filling-style
	 (o-formula (gvl :parent :parent :white-fill)))
(s-value (g-value scrollpad :v-scroll :bounding-area) :filling-style
	 (o-formula (if (gvl :parent :scroll-p)
			(gvl :parent :parent :gray-fill)
		      (gvl :parent :parent :white-fill))))
(s-value (g-value scrollpad :h-scroll :indicator) :filling-style
	 (o-formula (gvl :parent :parent :white-fill)))
(s-value (g-value scrollpad :h-scroll :bounding-area) :filling-style
	 (o-formula (if (gvl :parent :scroll-p)
			(gvl :parent :parent :gray-fill)
		      (gvl :parent :parent :white-fill))))

;; create a window to demonstrate the scrollpad
(defun scrollpad-demo ()
  (create-instance 'scrollwin inter:interactor-window
		   (:top 100)
		   (:left 0)
		   (:height 180)
		   (:width 180)
		   (:aggregate
		    (create-instance 'mypad scrollpad
				     (:top 10)
				     (:left 10)
				     ))
		   )
  (opal:update scrollwin)
  )
