
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* GEN.H

   Module providing functions for producing entries in the output.

   $Header: gen.h,v 1.8 91/11/08 15:29:26 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   This module provides functions for printing entries in the output and for
   saving the sysnames associated with user/group boxes indexed by name (a
   string) or uid/gid (an integer). A user and a group are allowed to have the
   same name, but no two users or groups should be registered under the same
   name.

   This module provides facilities for asserting containment relationships
   amongst boxes. The routines Gen_BeginInside(), Gen_AddInsideChild(), and
   Gen_EndInside() are used to assert containment relationships. These
   routines generate INSIDE entries in the output IFF file.

   Implementation Note: The user/group box lookups by uid/gid are performed by
   first consulting the hash table implemented by the User module, which stores
   the name of a user/group indexed by uid/gid, and then using the local hash
   table to perform the lookup by user/group name.

   WHAT THIS MODULE PROVIDES

   This module provides the following types:

     BoxSysname, ArrowSysname     sysnames for boxes and arrows
     BoxType                      types for boxes
     Parity                       arrow parity values

   It provides the following functions:

     Gen_Arrow()            print a new ARROW entry
     Gen_Box()              print a new BOX entry
     Gen_BoxSaveSysname()   register a box under a certain name & print it
     Gen_GetBoxName()       recall the table name of a box indexed by name
     Gen_GetUserName()      recall the name of a user box indexed by uid
     Gen_GetGroupName()     recall the name of a group box indexed by gid
     Gen_GetBoxSysname()    recall the sysname of a box indexed by name
     Gen_BeginInside()      initialize INSIDE entry
     Gen_AddInsideChild()   continue INSIDE entry
     Gen_EndInside()        concluded INSIDE entry
     Gen_UserRoleBoxes()    generates all user, group, world boxes
*/

#ifndef GEN_H
#define GEN_H

/* HEADER FILES =========================================================== */

#include <my-types.h>

/* TYPES ================================================================== */

/* BoxSysname/ArrowSysname -- type for sysname's assigned to a box */
typedef ULint BoxSysname;
typedef ULint ArrowSysname;

/* BoxType -- the type of each generated box.

   Note that all 3 types for user boxes have their 3rd bit set, while all
   types of object boxes do not. In this way, we can determine if a box is a
   subject or object by bitwise ANDing it with the BoxType of UserBox.
*/
typedef enum {
    FileBox =    000,		/* an object box of type 'file' */
    DeviceBox =  001,		/* an object box of type 'device' (file) */
    SocketBox =  002,		/* an object box of type 'socket' (file) */
    DirBox =     003,		/* an object box of type 'dir' */
    DirFileBox = 004,		/* dummy file box inside a directory */
    HomeBox    = 005,		/* an object box of type 'home' (directory) */
    RootBox    = 006,		/* an object box of type 'root' (directory) */
    UserBox =    010,		/* a subject box of type 'user' */
    GroupBox =   011,		/* a subject box of type 'group' */
    WorldBox =   012		/* a subject box of type 'world' */
} BoxType;

/* Parity -- parity values for arrows */
typedef enum { Pos, Neg } Parity;

/* MACRO FUNCTIONS ======================================================== */

#define MakeParity(_expr) ((_expr) ? Pos : Neg)

/* GLOBAL VARIABLES ======================================================= */

/* for use by Gen_{Begin,Add}Inside() macros */
extern Boolean first_child;

/* sysname of "world" box */
extern BoxSysname world_sysname;

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

void Gen_Arrow( /* BoxSysname from,to, String perm, Parity parity */ );
/* Generates a new arrow on stdout from the box with sysname 'from' to the box
   with sysname 'to' for permission name 'perm' of parity 'parity'.
*/

BoxSysname Gen_Box( /* String name, BoxType kind */ );
/* Generates a new box on stdout having name 'name' and BoxType 'kind'.

   RETURNS the BoxSysname assigned to the new box.
*/

BoxSysname Gen_ObjBox( /* String name, BoxType kind, struct stat *st */ );
/* Generates an object box on stdout have name 'name' and BoxType 'kind'
   (which should not have its bit 3 set). Also, depending on 'kind', checks
   are performed on 'st' to add any extra relevant attributes to the entry for
   the object. If 'st' is NULL, no such checks are performed.

   RETURNS the BoxSysname assigned to the new box.
*/

BoxSysname Gen_BoxSaveSysname( /* String name, BoxType kind */ );
/* Like Gen_Box(), this function generates a new box on stdout having name
   'name'. In addition, it registers the box as having type 'kind', so the
   BoxSysname of the new box can be retrieved with Gen_GetBoxSysname().

   RETURNS the BoxSysname assigned to the new box.
*/

String Gen_GetBoxName( /* String name, BoxType kind */ );
/* RETURNS the name associated with the box registered by a previous call to
   Gen_BoxSaveSysname() with name 'name' and type 'kind'; NULL if no such box
   exists.
*/

#ifdef MACRO_FUNCTION
String Gen_GetUserName( /* short user_id */ );
/* RETURNS the name associated with the box registered by a previous call to
   Gen_BoxSaveSysname() with type UserID and name registered under UserID
   'user_id'.
*/
#endif MACRO_FUNCTION

#define Gen_GetUserName(_user_id)\
    Gen_GetBoxName(User_GetName((_user_id),UserID),UserBox)

#ifdef MACRO_FUNCTION
String Gen_GetGroupName( /* short group_id */ );
/* RETURNS the name associated with the box registered by a previous call to
   Gen_BoxSaveSysname() with type GroupID and name registered under GroupID
   'group_id'.
*/
#endif MACRO_FUNCTION

#define Gen_GetGroupName(_group_id)\
    Gen_GetBoxName(User_GetName((_group_id),GroupID),GroupBox)

BoxSysname Gen_GetBoxSysname( /* String name, BoxType kind */ );
/* RETURNS the BoxSysname associated with the box registered by a previous
   call to Gen_BoxSaveSysname() with name 'name' and type 'kind'; 0L if no
   such box exists.
*/

#ifdef MACRO_FUNCTION
BoxSysname Gen_GetUserSysname( /* short user_id */ );
/* RETURNS the BoxSysname associated with the box registered by a previous
   call to Gen_BoxSaveSysname() with type UserID and name registered under
   UserID 'user_id'.
*/
#endif MACRO_FUNCTION

#define Gen_GetUserSysname(_user_id)\
    Gen_GetBoxSysname(User_GetName((_user_id),UserID),UserBox)

#ifdef MACRO_FUNCTION
BoxSysname Gen_GetGroupSysname( /* short group_id */ );
/* RETURNS the BoxSysname associated with the box registered by a previous
   call to Gen_BoxSaveSysname() with type GroupID and name registered under
   GroupID 'group_id'.
*/
#endif MACRO_FUNCTION

#define Gen_GetGroupSysname(_group_id)\
    Gen_GetBoxSysname(User_GetName((_group_id),GroupID),GroupBox)

void Gen_BeginInside( /* BoxSysname parent */ );
/* Generates initial segment of an INSIDE entry with parent 'parent'. To add
   children to the entry, use Gen_AddInsideChild(). To complete the entry,
   use Gen_EndInside().
*/

void Gen_AddInsideChild( /* BoxSysname child */ );
/* Adds a child 'child' to the set of boxes contained in the parent declared
   by the last call to Gen_BeginInside(). When done adding all children,
   call Gen_EndInside().
*/

void Gen_EndInside( /* void */ );
/* Completes the INSIDE entry started by Gen_BeginInside() and extended by
   Gen_AddInsideChild().
*/

void Gen_UserRoleBoxes( /* void */ );
/* Consults the /etc/passwd and /etc/group files, and generates user boxes for
   the world, all groups, and all users. Containment information for these
   boxes is also generated.

   In addition, this routine registers all user and group ID's, so that future
   calls to Gen_GetLoginID() can turn a user or group ID into the system's
   corresponding user or group name.

   If the global 'dummy_users' is True, then a dummy user box is created
   inside each group box and the world box. These boxes have names of the form
   "GROUP-USER-<groupname>" and "WORLD-USER".
*/

#endif GEN_H
