
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* PERM.C

   Module providing structures and functions for manipulating access
   permissions.

   $Header: perm.c,v 1.9 91/11/08 12:07:06 heydon Locked $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   IMPLEMENTATION:

   There are two kinds of Perm structure, denoted by the 'kind' field of the
   structure. When kind==UGW, the Perm is said to be a "three-level" Perm,
   since it distinguished the world level from the group level from the user
   level. When kind==UW, the Perm is said to be a "two-level" Perm, since it
   only distinguishes the world level from the user level.

   Two-level Perm's are more general, but using 3-level Perm's allows a saving
   of arrows, since we maintain the group abstraction. There is a function to
   convert a 3-level Perm into an equivalent 2-level Perm.

   It is important that the exact semantics of each type of Perm be spelled
   out precisely. It is assumed that all functions manipulating Perm's
   maintain the invariants implicit in these specifications.

   3-Level Perm (kind==UGW):
     BitsOf():
       W_BIT = world bit; G_BIT = group bit; U_BIT = user bit.
         A bit is on iff that level has permission. The actual user or group
	 represented by the user and group bits is stored in the OwnerIDOf()
	 and GroupIDOf() field.
       GW_BIT = group-world difference bit.
         This bit is on iff G_BIT differs from W_BIT.
       UG_BIT = user-group difference bit.
         This bit is on iff:
	   1) U_BIT differs from G_BIT, or
	   2) the GW_BIT is on *and* the user is not a member of the group.
     OwnerIDOf():
       The system ID (short) corresponding to the user who owns the file this
       Perm refers to.
     GroupIDOf():
       The system ID (short) corresponding to the group who owns the file this
       Perm refers to.

   2-level Perm (kind==UW):
     WBitOf():
       Bit 0 = world bit, representing whether the entire world has permission
       on the file corresponding to this Perm or not.
     OppUsersOf():
       List of users having permission *opposite* from the WBitOf() bit. If
       that bit is on, these are users denied permission on the file; if it is
       off, they are the only ones granted permission on the file. If this
       empty is empty, then *all* users are either denied or granted
       permission on the file, depending on the WBitOf() field.

    Notice that either a 3-level or a 2-level Perm can be used to represent
    the denial or granting of permission to all users.
*/

/* HEADER FILES =========================================================== */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>		/* for Assert() */
#include <my-defs.h>

#include "top.h"
#include "gen.h"
#include "perm.h"
#include "search.h"
#include "user.h"

/* LOCAL MACRO DEFINITIONS ================================================ */

/* Used to represent an empty Perm so no user or group arrows are drawn */
#define NO_ID (-1)

/* LOCAL VARIABLES ======================================================== */

static String access_type_strings[] = {
    "exec", "write", "read", "list", "in-del"
};

/* LOCAL FUNCTIONS ======================================================== */

#ifdef MACRO_FUNCTION
String perm_AccessType2String( /* AccessType kind */ );
/* RETURNS the name of the permission corresponding to AccessType 'kind'.
*/
#endif MACRO_FUNCTION

#define perm_AccessType2String(_kind)\
    access_type_strings[(int)(_kind)]

#ifdef MACRO_FUNCTION
int perm_GNotEqW( /* PermBits bits */ );
/* RETURNS non-zero iff the G_BIT and W_BIT of 'bits' are different.
*/
#endif MACRO_FUNCTION

#define perm_GNotEqW(_bits)\
    ((((_bits) & G_BIT) >> 3) ^ ((_bits) & W_BIT))

#ifdef MACRO_FUNCTION
int perm_UNotEqG( /* PermBits bits */ );
/* RETURNS non-zero iff the U_BIT and G_BIT of 'bits' are different.
*/
#endif MACRO_FUNCTION

#define perm_UNotEqG(_bits)\
    ((((_bits) & U_BIT) >> 3) ^ ((_bits) & G_BIT))

#ifdef MACRO_FUNCTION
int perm_UNotEqW( /* PermBits bits */ );
/* RETURNS non-zero iff the U_BIT and W_BIT of 'bits' are different.
*/
#endif MACRO_FUNCTION

#define perm_UNotEqW(_bits)\
    ((((_bits) & U_BIT) >> 6) ^ ((_bits) & W_BIT))

#ifdef MACRO_FUNCTION
static Boolean perm_HaveSameIDs( /* Perm *p1, Perm *p2 */ );
/* RETURNS True iff 'p1' and 'p2' have the same OwnerIDOf() and GroupIDOf(),
   where "same" means they actually have the same value, or the corresponding
   difference bit (UG_BIT or GW_BIT) of 'p1' is *not* set.
*/
#endif MACRO_FUNCTION

#define perm_HaveSameIDs(_p1,_p2)\
    ((OwnerIDOf(_p1)==OwnerIDOf(_p2) || !(BitsOf(_p1) & UG_BIT)) &&\
     (GroupIDOf(_p1)==GroupIDOf(_p2) || !(BitsOf(_p1) & GW_BIT)))

#ifdef MACRO_FUNCTION
static void perm_MakeNegativePerm( /* Perm *p */ );
/* Sets 'p' to be a 3-level "negative" Perm, i.e., granting no premissions and
   having special owner and group-owner fields. Only artificial, top-level
   Perm's should be made "negative" in this way.
*/
#endif MACRO_FUNCTION

#define perm_MakeNegativePerm(_p)\
    (_p)->kind = UGW;\
    Perm_Set3LevelPerm((_p),NO_BITS,NO_ID,NO_ID)

#ifdef MACRO_FUNCTION
static Boolean perm_IsEmptyPerm( /* Perm *p */ );
/* RETURNS True iff OwnerIDOf(p)==NO_ID && GroupIDOf(p)==NO_ID.
*/
#endif MACRO_FUNCTION

#define perm_IsEmptyPerm(_p)\
    (OwnerIDOf(_p)==NO_ID && GroupIDOf(_p)==NO_ID)

static void perm_SetDiffBits(p)
  INOUT Perm *p;
/* Sets UG_BIT and GW_BIT of the BitsOf(p) according to the invariant
   described for 3-level Perm's.

   PRE-CONDITION: p->kind == UGW && !(BitsOf(p) & (UG_BIT | GW_BIT))
*/
{
    PermBits bit_val = BitsOf(p);

    /* turn on GW_BIT iff G_BIT != W_BIT */
    if (perm_GNotEqW(bit_val)) { BitsOf(p) |= GW_BIT; }

    /* turn on UG_BIT in one of 2 cases */
    if (perm_UNotEqG(bit_val) || /* case 1: U_BIT and G_BIT differ */
	/* case 2: GW_BIT on and user not in group */
	((BitsOf(p) & GW_BIT) &&
	 (!User_IsInGroup(GroupIDOf(p),User_GetName(OwnerIDOf(p),UserID))))) {
	BitsOf(p) |= UG_BIT;
    }
}

static Perm *perm_StatToNewPerm(st,kind)
  struct stat *st;
  AccessType kind;
/* RETURNS a new Perm corresponding to the 'kind' bits of the stat(2)
   structure 'st'. This result Perm is always a type 3-level structure
   (kind==UGW).
*/
{
    Perm *result;

    result = Perm_New();
    result->kind = UGW;
    Perm_Set3LevelPerm(result,MakePermBits(st->st_mode,kind),
		       st->st_uid,st->st_gid);
    perm_SetDiffBits(result);
    return(result);
}

static Perm *perm_MakeNew2LevelPerm(p)
  Perm *p;
/* RETURNS a new Perm set to be a 2-level Perm (i.e., having p->kind==UW)
   semantically equivalent to the 3-level Perm '*p'.

   PRE-CONDITIONS: (p != NULL) && (p->kind == UGW)

   IMPLEMENTATION: We start by setting the WBitOf(result) to be the W_BIT of
   'p'. We must now set the OppUsersOf(result) to be a list of users having
   different permissions from this bit. We therefore first add all members of
   the group to the list if G_BIT != W_BIT (i.e., if GW_BIT). We then consider
   if U_BIT != W_BIT. If so, we also add the user to the list. However, if
   U_BIT == W_BIT, we are not done (as was the case at the group level)
   because of the extra group level between the user and world levels. We may
   need to *remove* the user from the list if the list of group members was
   added to the list. This will happen exactly when U_BIT != G_BIT.
*/
{
    String u_name;		/* user name in the gen.c hash table */
    UserNameList u_list;	/* temp list containing 'u_name' only */
    Perm *result;

    /* create and initialize a new 2-level Perm */
    result = Perm_New();
    result->kind = UW;
    WBitOf(result) = W_BIT & BitsOf(p);
    OppUsersOf(result) = NULL;

    /* set the list of opposite users to those in the group if the G_BIT and
       W_BIT are different */
    if (BitsOf(p) & GW_BIT) {
	OppUsersOf(result) =
	    User_NewCopyList(User_GetGroupMembers(GroupIDOf(p)));
    }

    /* initialize 'u_name' & 'u_list' */
    u_name = Gen_GetUserName(OwnerIDOf(p));
    u_list = User_NewList(u_name);

    /* add or remove the user name depending on U_BIT ?= W_BIT */
    if (perm_UNotEqW(BitsOf(p))) {
	/* add the user to the existing list if not there already */
	if (UG_BIT & BitsOf(p)) {
	    Assert(!User_IsInList(OppUsersOf(result),u_name));
	    SpliceIntoList(OppUsersOf(result),u_list);
	}
    } else {
	/* remove the user from the existing list if U_BIT != G_BIT */
	if (perm_UNotEqG(BitsOf(p))) {
	    User_MinusListsD(u_list,&(OppUsersOf(result)));
	    User_DestroyList(u_list); /* no longer used anywhere */
	}
    }
    return(result);
}

#define MergeWBits(_b1,_b2) (((_b1) << 4) | (_b2))
#define BOTH_OFF  0x00
#define FIRST_ON  0x10
#define SECOND_ON 0x01
#define BOTH_ON   0x11

static Perm *perm_NewAndWith2Level(p1,p2)
  Perm *p1,*p2;
/* RETURNS a new Perm representing the security AND of '*p1' and '*p2'.

   PRE-CONDITIONS: (p1 != NULL) && (p1->kind == UW) &&
                   (p1 != NULL) && (p2->kind == UGW)
*/
{
    Perm *new_p2;		/* two-level version of 'p2' */
    Perm *result;

    /* form a 2-level Perm equivalent to 'p2' */
    new_p2 = perm_MakeNew2LevelPerm(p2);

    /* fill in the 'result' */
    result = Perm_New();
    result->kind = UW;
    WBitOf(result) = WBitOf(p1) & WBitOf(new_p2);
    switch (MergeWBits(WBitOf(p1),WBitOf(new_p2))) {
      case BOTH_OFF:
	OppUsersOf(result) =
	    User_NewAndLists(OppUsersOf(p1),OppUsersOf(new_p2));
	break;
      case FIRST_ON:
	OppUsersOf(result) = User_NewCopyList(OppUsersOf(new_p2));
	User_MinusListsD(OppUsersOf(p1),&(OppUsersOf(result)));
	break;
      case SECOND_ON:
	OppUsersOf(result) = User_NewCopyList(OppUsersOf(p1));
	User_MinusListsD(OppUsersOf(new_p2),&(OppUsersOf(result)));
	break;
      case BOTH_ON:
	OppUsersOf(result) =
	    User_NewOrLists(OppUsersOf(p1),OppUsersOf(new_p2));
	break;
    }

    /* clean up */
    Perm_Destroy(new_p2);
    return(result);
}

static Perm *perm_NewAndWith3Level(p1,p2)
  Perm *p1,*p2;
/* RETURNS a new Perm representing the security AND of '*p1' and '*p2'.

   PRE-CONDITIONS: (p1 != NULL) && (p2 != NULL) &&
                   (p1->kind == UGW) && (p2->kind == UGW)
*/
{
    Perm *result;

    if (perm_HaveSameIDs(p1,p2)) {
	result = Perm_New();
	result->kind = UGW;
	Perm_Set3LevelPerm(result,(BitsOf(p1) & BitsOf(p2)),
			   OwnerIDOf(p2),GroupIDOf(p2));
	perm_SetDiffBits(result);
    } else {
	Perm *new_p1;		/* two-level version of 'p1' */

	new_p1 = perm_MakeNew2LevelPerm(p1);
	result = perm_NewAndWith2Level(new_p1,p2);
	Perm_Destroy(new_p1);
    }
    return(result);
}

static void perm_GenTypeArrows2Level(parent,child,sysname,perm_name)
  Perm *parent,*child;
  BoxSysname sysname;
  String perm_name;
/* Like Perm_GenTypeArrows(), only with the following:

   PRE-CONDITION: (parent->kind == UW) && (child->kind == UW)

   IMPLEMENTATION: There are 4 possibilities, based on possibilities for the
   WBit's of the 'parent' and 'child'. This routine generates arrows according
   to each case as follows:
     1) parent = (+W -P), child = (+W -C) => arrows = -(C-P) and +(P-C)
     2) parent = (-W +P), child = (-W +C) => arrows = +(C-P) and -(P-C)
     3) parent = (+W -P), child = (-W +C) => arrows = -W and +C
     4) parent = (-W +P), child = (+W -C) => arrows = +W and -C
   These heuristics may generate ambiguities, especially in case (4). We may
   need to test if the child permissions aren't a subset of the parent
   permissions, and generate extra arrows in that case.
*/
{
    UserName *curr_user;

    /* diverge based on whether WBits differ or not */
    if (WBitOf(parent) == WBitOf(child)) {
	UserNameList child_only;  /* users only in OppUsersOf(child) */
	UserNameList parent_only; /* users only in OppUsersOf(parent) */

	/* Set up two sets of arrows: child_only=(C-P), parent_only=(P-C) */
	child_only = User_NewCopyList(OppUsersOf(child));
	User_MinusListsD(OppUsersOf(parent),&child_only);
	parent_only = User_NewCopyList(OppUsersOf(parent));
	User_MinusListsD(OppUsersOf(child),&parent_only);

	/* generate arrows */
	StepLinkedList(curr_user,parent_only) {
	    Gen_Arrow(Gen_GetBoxSysname(curr_user->name,UserBox),sysname,
		      perm_name,MakeParity(WBitOf(child)));
	}
	StepLinkedList(curr_user,child_only) {
	    Gen_Arrow(Gen_GetBoxSysname(curr_user->name,UserBox),sysname,
		      perm_name,MakeParity(!WBitOf(child)));
	}

	/* destroy lists */
	User_DestroyList(child_only);
	User_DestroyList(parent_only);
    } else {
	/* generate world arrow and list of arrows for child list */
	Gen_Arrow(world_sysname,sysname,perm_name,MakeParity(WBitOf(child)));
	StepLinkedList(curr_user,OppUsersOf(child)) {
	    Gen_Arrow(Gen_GetBoxSysname(curr_user->name,UserBox),sysname,
		      perm_name,MakeParity(!WBitOf(child)));
	}
    }
}

#ifdef SUPERSET_TEST

static Boolean perm_SupersetEq3Bits(b1,b2)
  PermBits b1,b2;
/* RETURNS True iff the bits 'b1' represent premissions allowing at least as
   much access as those represented by 'b2'.

   IMPLEMENTATION: We start at the world bits, and work up through user and
   group bits. If the b2 bit is on and the b1 bit is off, we immediately
   return False (since then b2 allows more access then b1). If the b1 bit is
   on and the b2 bit is off, we immediately return True. If all bits are the
   same (i.e., neither of these conditions is satisfied), we return True.
*/
{
    register PermBits mask,b1_bit,b2_bit;

    for (mask=W_BIT; mask<ALL_BITS; mask<<=3) {
	if ((b2_bit=(b2&mask)) && !(b1_bit=(b1&mask))) { return(False); }
	else if (b1_bit && !b2_bit) { break; }
    }
    return(True);
}
#endif SUPERSET_TEST

static void perm_GenTypeArrows3Level(parent,child,sysname,perm_name)
  Perm *parent,*child;
  BoxSysname sysname;
  String perm_name;
/* Like Perm_GenTypeArrows(), only with the following:

   PRE-CONDITION: (parent->kind == UGW) && (child->kind == UGW)
               && (perm_HaveSameIDs(parent,child) || perm_IsEmptyPerm(parent))

   IMPLEMENTATION: The current algorithm works as follows:
     1) Starting with the "most significant" (i.e., "world") bit, find the
        first bit where the BitsOf(parent) and BitsOf(child) differ.
     2) Generate an arrow corresponding to this difference.
     3) Now, keep scanning BitsOf(child) towards the least significant bit,
        and generate an arrow whenever necessary (i.e., whenever the
	corresponding difference bit (GW_BIT or UG_BIT) is set).
   Although this algorithm does generate correct permissions for the child, it
   has the unfortunate property that it can cause an ambiguity to be
   introduced. There are two cases in which we *force* arrows to be generated
   in step (3) above, even if the ID field is NO_ID:
     1) Special case: BitsOf(parent)==0101 && BitsOf(child)==0000. In this
        case, we force a positive arrow to be generated for the owner (as well
	as the world).
#ifdef SUPERSET_TEST
     2) Child permission is a superset of Parent permission: In this case,
        we force an arrow to be generated for the owner or group-owner
	whenever the bits differ between the child and parent.
#endif SUPERSET_TEST
   These heuristics attempt to generate enough arrows to override an
   ambiguity.
*/
{
    register int i;		/* index into IDsOf() array */
    register PermBits mask;
    Boolean force[2];		/* force an arrow in step (3) */
    BoxSysname tail_name[3];	/* owner, group, and world sysnames */
    PermBits xor_bits;		/* XOR of BitsOf(parent), BitsOf(child) */

#define GET_PARITY(_perm,_mask)\
    MakeParity((BitsOf(_perm) & (_mask)))
#define SPECIAL_CASE(p,c)\
    ((BitsOf(p)&ALL_BITS)==0101 && (BitsOf(c)&ALL_BITS)==NO_BITS)

    /* set 'user_id', 'group_id', and 'tail_name[]' */
    Assert(perm_HaveSameIDs(parent,child) || perm_IsEmptyPerm(parent));
    tail_name[OWNER_ID] = Gen_GetUserSysname(OwnerIDOf(child));
    tail_name[GROUP_ID] = Gen_GetGroupSysname(GroupIDOf(child));
    tail_name[WORLD_ID] = world_sysname;

    /* (1): find the first place where 'parent' and 'child' bits differ */
    xor_bits = BitsOf(parent) ^ BitsOf(child);
    for (i=2,mask=W_BIT; i>=0; i--,mask<<=3) {
	if (xor_bits & mask) {
	    /* (2): generate the arrow to this first differing box */
	    Gen_Arrow(tail_name[i],sysname,perm_name,GET_PARITY(child,mask));
	    break;
	}
    }

    /* (3): generate arrows for any remaining differences w/in 'child' bits */
    if (SPECIAL_CASE(parent,child)
#ifdef SUPERSET_TEST
	|| (!perm_SupersetEq3Bits(BitsOf(parent),BitsOf(child)) &&
	    !perm_IsEmptyPerm(parent))
#endif SUPERSET_TEST
    ) {
	force[OWNER_ID] = MakeBoolean(xor_bits & U_BIT);
	force[GROUP_ID] = MakeBoolean(xor_bits & G_BIT);
    } else {
	force[OWNER_ID] = force[GROUP_ID] = False;
    }
    for (i--,mask<<=3; i>=0; i--,mask<<=3) {
	if (force[i] || (BitsOf(child) & (mask<<1))) {
	    Gen_Arrow(tail_name[i],sysname,perm_name,GET_PARITY(child,mask));
	}
    }
}

/* GLOBAL FUNCTIONS ======================================================= */

void Perm_Destroy(perm)
  Perm *perm;
{
    if (perm->kind == UW) {
	User_DestroyList(OppUsersOf(perm));
    }
    Dealloc(perm);
}

Perm *Perm_NewCopy(perm)
  Perm *perm;
{
    Perm *result;

    result = Perm_New();
    result->kind = perm->kind;
    if (perm->kind == UGW) {
	Perm_Set3LevelPerm(result,BitsOf(perm),OwnerIDOf(perm),
			   GroupIDOf(perm));
    } else {
	WBitOf(result) = WBitOf(perm);
	OppUsersOf(result) = User_NewCopyList(OppUsersOf(perm));
    }
    return(result);
}

PermSet Perm_NewNegativePermSet(kind)
  EntryType kind;
{
    int i,last;
    PermSet result;

    result = Perm_NewPermSet(kind);
    last = PermSetSize(kind);
    StepIndex(i,0,last) {
	result[i] = Perm_New();
	perm_MakeNegativePerm(result[i]);
    }
    return(result);
}

void Perm_DestroyPermSet(set,kind)
  PermSet set;
  EntryType kind;
{
    int i,last;

    last = PermSetSize(kind);
    StepIndex(i,0,last) { if (set[i]) { Perm_Destroy(set[i]); } }
    Dealloc(set);
}

Perm *Perm_NewAnd(perm,st,kind)
  Perm *perm;
  struct stat *st;
  AccessType kind;
{
    Perm *result;
    Perm *st_perm;		/* Perm equivalent to 'st' */

    /* check for the trivial case where 'perm' == NULL */
    st_perm = perm_StatToNewPerm(st,kind);
    if (perm == NULL) { return(st_perm); }

    /* handle 2-level and 3-level cases differently */
    switch ((int)(perm->kind)) {
      case (int)UW:
	result = perm_NewAndWith2Level(perm,st_perm);
	break;
      case (int)UGW:
	result = perm_NewAndWith3Level(perm,st_perm);
	break;
    }
    Perm_Destroy(st_perm);
    return(result);
}

Boolean Perm_IsSamePerm(p1,p2)
  Perm *p1,*p2;
/* IMPLEMENTATION: Verify both perms have the same type, and same bits. If the
   type is 3-level, make sure ID's are the same; if the type is 2-level, make
   sure the opposite users lists are the same. RETURN False if any test fails,
   and True otherwise.
*/
{
    if (p1->kind != p2->kind) { return(False); }
    if (p1->kind == UGW) {
	if ((BitsOf(p1) != BitsOf(p2)) ||
	    (OwnerIDOf(p1) != OwnerIDOf(p2)) ||
	    (GroupIDOf(p1) != GroupIDOf(p2))) {
	    return(False);
	}
    } else {
	if ((WBitOf(p1) != WBitOf(p2)) ||
	    !User_IsSameList(OppUsersOf(p1),OppUsersOf(p2))) {
	    return(False);
	}
    }
    return(True);
}

void Perm_GenTypeArrows(parent,child,sysname,kind)
  Perm *parent,*child;
  BoxSysname sysname;
  AccessType kind;
/* IMPLEMENTATION: No arrows are generated if 'parent' and 'child' are the
   same. The more advantageous perm_GenTypeArrows3Level() is called if
   'parent' and 'child' are 3-level Perm's *and* if their owner and group ID's
   are identical (or in the special case that 'parent' is the special "empty"
   root Perm). Otherwise, both 'parent' and 'child' are converted to 2-level
   Perm's (as necessary) and the routine perm_GenTypeArrows2Level() is called
   on them.
*/
{
    String perm_name;		/* String version of 'kind' */

    /* don't generate any arrows if 'parent' and 'child' are the same */
    if (Perm_IsSamePerm(parent,child)) { return; }

    /* branch based on compatability of parent and child as 3-level perms */
    perm_name = perm_AccessType2String(kind);
    if ((parent->kind == UGW) && (child->kind == UGW) &&
	(perm_HaveSameIDs(parent,child) || perm_IsEmptyPerm(parent))) {
	perm_GenTypeArrows3Level(parent,child,sysname,perm_name);
    } else {
	Perm *new_parent,*new_child;

	new_parent =
	    (parent->kind == UGW) ? perm_MakeNew2LevelPerm(parent) : parent;
	new_child =
	    (child->kind == UGW) ? perm_MakeNew2LevelPerm(child) : child;
	perm_GenTypeArrows2Level(new_parent,new_child,sysname,perm_name);
	if (new_parent != parent) { Perm_Destroy(new_parent); }
	if (new_child != child) { Perm_Destroy(new_child); }
    }
}

void Perm_GenArrows(parent,child,kind,sysname)
  PermSet parent,child;
  EntryType kind;
  BoxSysname sysname;
{
    int i,last;

    last = ((kind == FileEnt) ? LAST_FILE_ACCESS : LAST_DIR_ACCESS);
    for (i=START_ACCESS; i < last; i++) {
	Perm_GenTypeArrows(parent[i],child[i],sysname,(AccessType)i);
    }
}
