/* PERM.H

   $Header: perm.h,v 1.6 91/11/08 12:25:53 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   WHAT THIS MODULE PROVIDES

*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#ifndef PERM_H
#define PERM_H

#include <sys/types.h>

#include <my-types.h>

#include "user.h"

/* MACRO DEFINITIONS ====================================================== */

/* sizes of various PermSet's */
#define FILE_SET_SIZE ((int)Read+1)
#define DIR_SET_SIZE  MaxNumPerms

/* Access Indices for iteration */
#define START_ACCESS     0
#define LAST_FILE_ACCESS FILE_SET_SIZE
#define LAST_DIR_ACCESS  DIR_SET_SIZE

/* constants representing user, group, world bits of a PermBits value */
#define U_BIT    0100
#define G_BIT    0010
#define W_BIT    0001
#define UG_BIT   0200
#define GW_BIT   0020
#define NO_BITS  0000
#define ALL_BITS 0111

/* Indices for the 'ids[]' array in the Perm structure */
#define OWNER_ID 0
#define GROUP_ID 1
#define WORLD_ID 2

/* TYPES ================================================================== */

/* PermBits -- type for permission bits for *one* AccessType

   The st.st_mode access bits are stored in bits 0777 (octal). Since a
   PermBits represents only *one* AccessType, we store only 3 bits. They are
   stored in bits 0111. Therefore, we must shift Read bits right 2 places, and
   Write bits right 1 place to form a PermBits. See the MakePermBits() macro
   below.
*/
typedef USint PermBits;

/* Perm -- basic permission structure */
typedef struct {
    enum {UGW,UW} kind;		/* 3-level or 2-level permission */
    union {
	/* kind==UGW -- 3-level representation */
	struct {
	    PermBits bits;	/* user, group, world bits (in ALL_BITS) */
	    uid_t ids[2];	/* owner (0) and group-owner (1) ID's */
	} three;
	/* kind==UW -- 2-level representation */
	struct {
	    PermBits w_bit;	/* world bit (in W_BIT) */
	    UserNameList opp;	/* users having opposite permission to W */
	} two;
    } u;
} Perm;

/* AccessType -- types of accesses that we will generate arrows for */
typedef enum { 
    Exec=0, Write=1, Read=2,	/* r,w,x permissions on files */
    List=3,			/* read perm on a dir */
    InDel=4			/* write (insert-delete) perm on a dir */
} AccessType;

/* EntryType -- distinguishes between File and Dir objects */
typedef enum { FileEnt, DirEnt } EntryType;

/* PermSet -- a set (array) of (pointers to) Perm's */
typedef Perm **PermSet;

/* MACRO FUNCTIONS ======================================================== */

/* Form a PermBits field for a given AccessType from a st.st_mode field */
#define MakePermBits(_bits,_kind) (ALL_BITS & ((_bits) >> ((int)(_kind))))

/* Perm access functions */
#define BitsOf(_ptr)     (((_ptr)->u).three.bits)
#define IDsOf(_ptr)      (((_ptr)->u).three.ids)
#define OwnerIDOf(_ptr)  (IDsOf(_ptr)[OWNER_ID])
#define GroupIDOf(_ptr)  (IDsOf(_ptr)[GROUP_ID])
#define WBitOf(_ptr)     (((_ptr)->u).two.w_bit)
#define OppUsersOf(_ptr) (((_ptr)->u).two.opp)

/* PermSetSize(kind) -- give the size of a PermSet having type 'kind' */
#define PermSetSize(_kind)\
    (((int)(_kind)==(int)FileEnt) ? FILE_SET_SIZE : DIR_SET_SIZE)

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

#ifdef MACRO_FUNCTION
Perm *Perm_New( /* void */ );
/* RETURNS a pointer to a new, empty permision.
*/
#endif MACRO_FUNCTION

#define Perm_New()\
    AllocOne(Perm)

void Perm_Destroy( /* Perm *perm */ );
/* The Perm 'perm' is destroyed, and its memory and that used by constituent
   parts is freed.
*/

#ifdef MACRO_FUNCTION
void Perm_Set3LevelPerm( /* INOUT Perm *perm, PermBits bits, short owner_id,
			    short group_id */ );
/* Sets the 3-level Perm 'perm' to have values 'bits', 'owner_id' and
   'group_id', respectively.
*/
#endif MACRO_FUNCTION

#define Perm_Set3LevelPerm(_perm,_bits,_owner_id,_group_id)\
    BitsOf(_perm) = (_bits);\
    OwnerIDOf(_perm) = (_owner_id);\
    GroupIDOf(_perm) = (_group_id)

Perm *Perm_NewCopy( /* Perm *from */ );
/* RETURNS a newly created Perm which is copied from 'from'.
*/

#ifdef MACRO_FUNCTION
PermSet Perm_NewPermSet( /* EntryType kind */ );
/* RETURNS a new, empty PermSet of type 'kind'.
*/
#endif MACRO_FUNCTION

#define Perm_NewPermSet(_kind)\
    (PermSet)AllocPtrArray(Perm,PermSetSize(_kind))

PermSet Perm_NewNegativePermSet( /* EntryType kind */ );
/* RETURNS a new, empty PermSet of type 'kind' in which all entries are
   initialized to 3-level Perm's representing the denial of all access rights.
*/

void Perm_DestroyPermSet( /* PermSet set, EntryType kind */ );
/* The PermSet 'set' created with EntryType 'kind' is destroyed, and all its
   associated memory is freed.
*/

Perm *Perm_NewAnd( /* Perm *perm, struct stat *st, AccessType kind */ );
/* RETURNS a Perm representing the logical conjunction of the Perm 'perm' and
   the 'kind' bits of the stat(2) structure 'st'. This function allocates the
   space for the returned Perm. The 'kind' parameter must be one of: Read,
   Write, or Execute; List and InDel permissions for a directory come from the
   Read & Write bits, respectively, of the stat(2) structure 'st'.
*/

Boolean Perm_IsSamePerm( /* Perm *p1,*p2 */ );
/* RETURNS True iff 'p1' and 'p2' are the "same" perm. Two Perm's are the
   "same" if they both have the same 'kind' and if their other respective
   fields are then identical. If both have kind==UW, the OppUsersOf() list of
   users need only contain the same names (but not nec. in the same order) for
   the lists to be considered the same.
*/

void Perm_GenTypeArrows( /* Perm *parent, Perm *child, BoxSysname sysname,
			    AccessType kind */ );
/* Generates arrows to represent the differences in the permissions 'parent'
   and 'child'. The arrows generated enforce that the permission on the child
   box is *exactly* that specified by 'child'; the 'parent' permission is
   only used to reduce the number of generated arrows since we assume that the
   permission specified by 'parent' has been enforced for the single box
   directly containing the child box (hence, its permissions are inherited).
   The arrows generated are for one type of arrow, namely, the AccessType
   'kind'; 'sysname' is the BoxSysname of the box generated for 'child' that
   the heads of all generated arrows attach to.
*/

void Perm_GenArrows( /* PermSet parent, PermSet child, EntryType kind,
			BoxSysname sysname */ );
/* Generates ARROW entries to stdout for the permissions 'child' which is
   assumed to be in the directory having permissions 'parent'. The permissions
   'child' correspond to an object of type 'kind'; 'sysname' is assumed to be
   the sysname of the box generated for this object (the heads of all ARROW's
   generated by this routine will attach to this box).
*/

#endif PERM_H
