
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* SEARCH.C

   Module providing structures and functions for searching directories/files.

   $Header: search.c,v 1.14 91/11/14 22:40:52 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   IMPLEMENTATION

   The search is implemented recursively as a depth-first search, using a
   single Path array to hold the current version of the pathname. During the
   downward direction of the search, each directory receives the cd-expression
   of its parent directory and computes its own cd-expression. It then passes
   this expression on to its children. The cd-expression is the permission
   corresponding to the "permission-AND" of the x-bits of the directory and
   all its parents.

   Boxes and arrows are generated during the upward direction of the search.
   Each entry in any directory is assumed to have generated a box (and
   sysname) for itself and computed its permissions. The sysname of its box
   and its permissions are returned to its parent. The parent then uses these
   permissions to take a vote and compute its own permissions. Finally, the
   permissions of the parent are compared to those of each of the entries, and
   arrows are generated (pointing to the boxes corresponding to the entries)
   according to the *differences* between the parent and entry permissions.

   The local variable 'SearchRoot' is used to guide the initial part of the
   search. See the function search_ProcessOpenDir() for details.
*/

/* HEADER FILES =========================================================== */

#include <sys/types.h>		/* generic system types */
#include <sys/stat.h>		/* for stat(2) call */
#include <sys/param.h>		/* for MAXPATHLEN constant */
#include <sys/dir.h>		/* for opendir(3), readdir(3), etc. calls */

#include <strings.h>
#include <stdio.h>
#include <my-defs.h>

#include "top.h"
#include "gen.h"
#include "perm.h"
#include "search.h"
#include "vote.h"

/* MACRO DEFINITIONS ====================================================== */

/* define various mode types to be used after 'case' */
#define DIR_TYPE  S_IFDIR
#define FILE_TYPE S_IFREG: case S_IFCHR: case S_IFBLK: case S_IFSOCK
#define SKIP_TYPE S_IFLNK

/* LOCAL TYPES ============================================================ */

/* DirEntry -- structure for storing info for each entry in a dir */
typedef struct dir_entry {
    struct dir_entry *next;	/* pointer to next element in linked list */
    EntryType kind;		/* directory or file? */
    String name;		/* name of this dir/file */
    BoxSysname sysname;		/* sysname of the generated box */
    PermSet perm;		/* corresponding set of permissions */
} DirEntry;

/* LOCAL VARIABLES ======================================================== */

static String SearchRoot;	/* root of sub-tree to search */
static int SearchRootLen;	/* strlen(SearchRoot) */

/* LOCAL FUNCTIONS ======================================================== */

Boolean search_CheckOwnerGroup(path,st)
  String path;
  struct stat *st;
{
    if (User_GetName(st->st_uid,UserID) == NULL) {
	fprintf(stderr,"%s: unknown user ID %u on %s; skipping...\n",
		Argv0,st->st_uid,path);
	return(True);
    }
    if (st->st_gid && User_GetName(st->st_gid,GroupID) == NULL) {
	fprintf(stderr,"%s: unknown group ID %u on %s; skipping...\n",
		Argv0,st->st_gid,path);
	return(True);
    }
    return(False);
}

DirEntry *search_NewDirEntry(kind,name)
  EntryType kind;
  String name;
/* RETURNS a pointer to a new DirEntry having type 'kind' and name 'name'. The
   space for 'name' is allocated dynamically and the string is copied. The
   'perm' field of the returned DirEntry is initially NULL.
*/
{
    DirEntry *ent;

    ent = AllocOne(DirEntry);
    ent->kind = kind;
    CopyString(ent->name,name);
    return(ent);
}

void search_DestroyDirEntry(entry)
  DirEntry *entry;
/* The DirEntry 'entry' is destroyed, and all its associated memory is freed.
*/
{
    Dealloc(entry->name);
    Perm_DestroyPermSet(entry->perm,entry->kind);
    Dealloc(entry);
}

String search_TailName(path)
  String path;
/* RETURNS a pointer to the "tail" of the pathname 'path'. If 'path' is "/",
   or if 'path' contains no '/' characters, 'path' itself is returned.

   EXAMPLES: "foo" => "foo", "/" => "/", "/usr/misc" => "misc"
*/
{
    String tail;

    if ((tail=rindex(path,'/')) == NULL || !strcmp(path,"/")) {
	return(path);
    } else {
	return(tail+1);
    }
}

void search_AppendSlash(path,len_ptr,end_ptr)
  PathStr path;
  int *len_ptr;
  String *end_ptr;
/* Assuming '*end_ptr' is the tail of the string 'path' (i.e., points to a
   pointer pointing to the '\0' at the end of a string), this routine sees if
   the last character of the string is a '/', and if it is not, appends a '/'
   and increments '*len_ptr' and '*end_ptr'.
*/
{
    if (*(*end_ptr-1) != '/') {
	if (*len_ptr >= MAXPATHLEN) {
	    fprintf(stderr,"%s: pathname '%s/' too long; aborting...\n",
		    Argv0,path);
	    exit(-1);
	}
	(*len_ptr)++;
	*((*end_ptr)++) = '/';
	**end_ptr = '\0';
    }
}

static BoxType search_FileType(st)
  struct stat *st;
{
    switch (st->st_mode & S_IFMT) {
      case S_IFCHR: case S_IFBLK: return(DeviceBox);
      case S_IFSOCK: return(SocketBox);
      default: return(FileBox);
    }
}

static PermSet search_File(path,st,parent_cd,new_sysname)
  PathStr path;
  struct stat *st;		/* stat(2) struct of 'path' */
  Perm *parent_cd;
  OUT BoxSysname *new_sysname;
/* RETURNS the permissions of the file name 'path', given that 'st' is the
   stat(2) of the file and 'parent_cd' is the cd-expr permission of its parent
   directory. Furthermore, a new box is generated for 'path' that is given the
   tail component of 'path' for its name and the type corresponding to file
   boxes for its type. 'new_sysname' is set to the sysname for this generated
   box.
*/
{
    PermSet perm_expr;		/* new PermSet returned for this file */

    /* set R,W,X permissions for this file from 'parent_cd' */
    perm_expr = Perm_NewPermSet(FileEnt);
    perm_expr[(int)Read] = Perm_NewAnd(parent_cd,st,Read);
    perm_expr[(int)Write] = Perm_NewAnd(parent_cd,st,Write);
    perm_expr[(int)Exec] = Perm_NewAnd(parent_cd,st,Exec);

    /* create the new file box */
    *new_sysname = Gen_ObjBox(search_TailName(path),search_FileType(st),st);
    return(perm_expr);
}

static DirEntry *search_ProcessOpenDir(dir,path,curr_cd,vote_table)
  DIR *dir;
  PathStr path;
  Perm *curr_cd;
  INOUT VoteTable vote_table;
/* Examine the entries of the directory 'dir' (which is named 'path' and has
   cd-expr 'curr_cd') and recursively "search" those that are appropriate.
   Those that are *not* "appropriate" are:
     1) the entries named "." and ".." (current and parent directories),
     2) if 'path' is a substring of the local 'SearchRoot', any entry
        that does not extend 'path' to be a substring of 'SearchRoot', or
     3) entries not considered files or directories.
   The recursive search on each appropriate entry is done by search_File() or
   search_Dir(), depending on the type of the entry.

   The votes for each entry are additionally registered in 'vote_table'.

   RETURNS the linked list of DirEntry structures, one for each appropriate
   entry in the directory.

   IMPLEMENTATION: Since pathnames are only extended, the same space is used
   for all pathnames (namely, the space in 'path'). The end of 'path' on entry
   to this function is remembered, and then the name of each entry is simply
   appended at this point (possibly overwriting previous names). For this
   reason, 'path' is of type Path, and so must contain enough space to hold
   the system-defined maximum length pathname.
*/
{
    int path_len;		/* length of initial 'path' argument */
    int max_ent_len;		/* max length of any directory entry */
    char *old_end,*path_end;	/* pointer to end of 'path' String */
    struct direct *dir_ent;	/* structure for each directory entry */
    DirEntry *list_head=NULL;	/* head of linked list to be returned */
    DirEntry *curr_ent;		/* temporary pointer for each entry */
    struct stat st;		/* result from stat(2) */
    Boolean prefix;		/* 'path' a proper prefix of 'SearchRoot'? */
    char *path_middle;		/* pointer to middle of 'SearchRoot' */
    PermSet search_Dir();

#define ProperPrefix(s1_len,s2_len,s1,s2) \
  NotOf((s1_len) >= (s2_len) || strncmp((s1),(s2),(s1_len)))

#define BadExtension(end1,end2,len) \
  (strncmp((end1),(end2),(int)(len)) || *((end2)+(len)) != '/')

    /* find end of 'path' and extend with a '/' if necessary */
    path_len = strlen(path);
    old_end = path_end = path + path_len;
    search_AppendSlash(path,&path_len,&path_end);
    max_ent_len = MAXPATHLEN - path_len;

    /* decide if 'SearchRoot' is relevent */
    if (prefix=ProperPrefix(path_len,SearchRootLen,path,SearchRoot)) {
	/* set 'path_middle' to end of match in 'SearchRoot' */
	path_middle = SearchRoot + path_len;
    }

    /* loop over the contents of the directory 'dir' */
    while (dir_ent = readdir(dir)) {
	/* don't search "." or ".." */
	if (!strcmp(dir_ent->d_name,".") || !strcmp(dir_ent->d_name,"..")) {
	    continue;
	}

	/* make sure name is not too long and create the new path name */
	if (dir_ent->d_namlen > max_ent_len) {
	    fprintf(stderr,"%s: pathname '%s%s' too long; aborting...\n",
		    Argv0,path,dir_ent->d_name);
	    exit(-1);
	}
	(void)strcpy(path_end,dir_ent->d_name);

	/* skip entries that do not properly extend 'SearchRoot' */
	if (prefix && BadExtension(path_end,path_middle,dir_ent->d_namlen)) {
	    continue;
	}

	/* stat(2) 'path' and skip if access is denied */
	if (lstat(path,&st)) {
	    fprintf(stderr,"%s: stat(2) failed on '%s'; skipping...\n",
		    Argv0,path);
	    continue;
	}
	if (search_CheckOwnerGroup(path,&st)) continue;

	/* create a new DirEntry */
	switch (st.st_mode & S_IFMT) {
	  case FILE_TYPE:
	    curr_ent = search_NewDirEntry(FileEnt,dir_ent->d_name); break;
	  case DIR_TYPE:
	    curr_ent = search_NewDirEntry(DirEnt,dir_ent->d_name); break;
	  case SKIP_TYPE:	/* skip these types of entries */
	    continue;
	  default:
	    fprintf(stderr,"%s: unknown file type %x on file '%s'\n",
		    Argv0,st.st_mode & S_IFMT,path);
	    continue;
	}

	/* link the new DirEntry into the linked list for this dir */
	SpliceIntoList(list_head,curr_ent);

	/* recursively search this entry */
	if (curr_ent->kind == FileEnt) {
	    curr_ent->perm=search_File(path,&st,curr_cd,&(curr_ent->sysname));
	} else {
	    curr_ent->perm=search_Dir(path,&st,curr_cd,&(curr_ent->sysname));
	}
	Vote_CastVotes(vote_table,curr_ent->perm);
    }
    *old_end = '\0';		/* set 'path' to its original value */
    return(list_head);
}

static DirEntry *search_ProcessDir(path,curr_cd,vote_table)
  PathStr path;
  Perm *curr_cd;
  INOUT VoteTable vote_table;
/* Opens the directory 'path' and processes its contents by calling the
   function search_ProcessOpenDir(). 'curr_cd' should be the Perm for the
   cd-expression for the directory 'path'. Votes for the entries in the
   directory are stored in 'vote_table'.

   RETURNS a linked list of DirEntry structures, with one structure for each
   entry in the directory 'path', or NULL if 'path' could not be opened.
*/
{
    DirEntry *result;		/* linked list of entries */
    DIR *dir;			/* DIR struct assoc. w/ 'path' */

    /* open the directory */
    if (!(dir=opendir((char *)path))) {
	fprintf(stderr,"%s: opendir(3) failed on '%s'; skipping...\n",
		Argv0,path);
	/* perror(Argv0); */
	result = (DirEntry *)NULL;
    } else {
	/* read and process the contents of the directory */
	result = search_ProcessOpenDir(dir,path,curr_cd,vote_table);
	closedir(dir);
    }
    return(result);
}

static BoxType search_DirType(path)
  PathStr path;
{
    if (SameString("/",path)) {
	return(RootBox);
    }
    if (strncmp("/usr",path,4) == 0) {
	path += 4;
	while (*path >= '0' && *path <= '9') { path++; }
	if (*path == '/' && index(path+1,'/') == NULL) {
	    return(HomeBox);
	}
    }
    return(DirBox);
}

static PermSet search_Dir(path,st,parent_cd,new_sysname)
  PathStr path;
  struct stat *st;		/* stat(2) struct of 'path' */
  Perm *parent_cd;
  OUT BoxSysname *new_sysname;
/* Recursively search the directory named by 'path', where 'st' is the
   (previously computed) stat(2) structure of 'path', and 'parent_cd' is the
   "cd-permission" of the parent directory of 'path'. A box is generated for
   'path', and its permissions are compared with the permissions of each entry
   in the directory to generate arrows (pointing to the boxes for the entries)
   reflecting the difference in permissions between this directory and each
   entry. The permissions of the directory are determined by a vote as
   described below.

   By "recursively search" is meant that all entries of the directory are
   searched (by a call to search_ProcessDir()), with the permissions for each
   entry being recorded in a vote table. After all entries have been
   searched (including generating boxes and arrows for each recursively), the
   votes are correlated and used to "set" the permissions for the directory
   named by 'path'.

   'new_sysname' is set to the sysname of the box generated for the directory
   named 'path'; the box is given the tail component of 'path' for its name,
   is created with the type corresponding to directory boxes. Containment
   information is generated to reflect that the boxes corresponding to the
   entries are contained in the box corresponding to the 'path' directory.

   In addition, if the global variable 'DummyFiles' is True, a dummy file box
   is created and placed inside the box corresponding to the 'path' directory.
   If the global variable 'DirPerms' is True, 'list' and 'in-del' permission
   arrows are generated (as necessary) to the directory (or to the dummy file
   box if it is generated) to reflect the r and w bits of the directory.
*/
{
    Perm *curr_cd;		/* cd-expr for this directory */
    PermSet perm_expr;		/* new PermSet returned for this dir */
    DirEntry *entry_head;	/* head of list for this directory's entries */
    DirEntry *curr_ent;		/* temporary pointer for each entry */
    VoteTable vote_table;	/* vote table for the directory 'path' */
    String tail_name;		/* tail component of 'path' */
    Perm *list_perm,*indel_perm;/* list and in-del perms for dummy dir box */
    BoxSysname dummy;		/* sysname of dummy DIR-OBJ in each dir */

    /* set 'cd-expr' for this directory from 'parent_cd' */
    curr_cd = Perm_NewAnd(parent_cd,st,Exec);

    /* process the directory and take votes */
    perm_expr = Perm_NewPermSet(DirEnt);
    vote_table = Vote_NewTable();
    if (DirPerms) {
	/* compute the List and InDel permissions (note asymmetry) */
	list_perm = Perm_NewAnd(parent_cd,st,Read);   /* uses parent cd-perm */
	indel_perm = Perm_NewAnd(curr_cd,st,Write); /* uses current cd-perm */

	/* cast votes for these permissions for the "dummy" file box */
	Vote_CastVote(vote_table,list_perm,List);
	Vote_CastVote(vote_table,indel_perm,InDel);
    }
    entry_head = search_ProcessDir(path,curr_cd,vote_table);
    Vote_AssignWinners(perm_expr,curr_cd,vote_table);
    Vote_DestroyTable(vote_table);

    /* generate ARROW entries for each directory entry in DirEntry list */
    StepLinkedList(curr_ent,entry_head) {
	Perm_GenArrows(perm_expr,curr_ent->perm,curr_ent->kind,
		       curr_ent->sysname);
    }

    /* create boxes for this dir and a dummy file inside it */
    tail_name = search_TailName(path);
    *new_sysname = Gen_ObjBox(tail_name,search_DirType(path),st);
    if (DummyFiles) {
	String dummy_name;	/* name of dummy DIR-OBJ in each dir */
	int name_len = strlen(DUMMY_DIR_NAME)+ strlen(tail_name) + 1;

	/* create dummy dir box */
	dummy_name = malloc((unsigned)name_len);
	(void)strcat(strcpy(dummy_name,DUMMY_DIR_NAME),tail_name);
	dummy = Gen_ObjBox(dummy_name,DirFileBox,(struct stat *)NULL);
	Dealloc(dummy_name);

	/* generate arrows to the dummy file if necessary */
	if (DirPerms) {
	    Perm_GenTypeArrows(perm_expr[(int)List],list_perm,dummy,List);
	    Perm_GenTypeArrows(perm_expr[(int)InDel],indel_perm,dummy,InDel);
	    Perm_Destroy(list_perm);
	    Perm_Destroy(indel_perm);
	}
    }

    /* generate INSIDE entry and deallocate DirEntry's */
    Gen_BeginInside(*new_sysname);
    while (entry_head) {
	curr_ent = entry_head;
	entry_head = curr_ent->next;
	Gen_AddInsideChild(curr_ent->sysname);
	search_DestroyDirEntry(curr_ent);
    }
    if (DummyFiles) { Gen_AddInsideChild(dummy); }
    Gen_EndInside();

    /* clean up and return */
    Perm_Destroy(curr_cd);
    return(perm_expr);
}

/* GLOBAL FUNCTIONS ======================================================= */

void Search_TopLevelPath(path,search_root)
  PathStr path;
  String search_root;
{
    BoxSysname root_sysname;
    PermSet root_perm,empty_perm;
    struct stat st;

    /* make sure 'path' is a substring of 'search_root' */
    if (strncmp(path,search_root,strlen(path))) {
	fprintf(stderr,"%s: '%s' not substring of '%s'; aborting...\n",
		Argv0,path,search_root);
	return;
    }

    /* set SearchRoot to 'search_root' if it is not "/"; to empty o.w. */
    SearchRoot = strcmp(search_root,"/") ? search_root : "";
    SearchRootLen = strlen(SearchRoot);

    /* stat(2) the top-level 'path' */
    if (lstat(path,&st)) {
	fprintf(stderr,
		"%s: stat(2) failed on top-level path '%s'; aborting...\n",
		Argv0,path);
	/* perror(Argv0); */
	return;
    }
    if (search_CheckOwnerGroup(path,&st)) return;

    /* recursively search the top-level 'path' */
    switch (st.st_mode & S_IFMT) {
      case FILE_TYPE:
	root_perm = search_File(path,&st,(Perm *)NULL,&root_sysname);
	break;
      case DIR_TYPE:
	root_perm = search_Dir(path,&st,(Perm *)NULL,&root_sysname);
	break;
      case SKIP_TYPE:
	fprintf(stderr,"%s: path is not a file or directory; aborting...\n",
		Argv0);
	return;
      default:
	fprintf(stderr,"%s: unknown file type %x on file '%s'\n",
		Argv0,st.st_mode & S_IFMT,path);
	return;
    }

    /* generate arrows for the top-level 'path' */
    empty_perm = Perm_NewNegativePermSet(DirEnt);
    Perm_GenArrows(empty_perm,root_perm,DirEnt,root_sysname);
    Perm_DestroyPermSet(empty_perm,DirEnt);
}
