/* USER.H

   $Header: user.h,v 1.3 90/09/30 16:18:00 heydon Locked $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   WHAT THIS MODULE PROVIDES

*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

#ifndef USER_H
#define USER_H

/* HEADER FILES =========================================================== */

#include <my-types.h>

/* TYPES ================================================================== */

/* IDType -- the type of a user or group ID. */
typedef enum {
    UserID =  0x0,		/* a system id for a user */
    GroupID = 0x1		/* a system id for a group */
} IDType;

/* Linked list item for user login-id's */
typedef struct user_name {
    struct user_name *next;	/* pointer to next login-id */
    String name;		/* the login-id itself (pointer to global) */
} UserName;

/* UserNameList - pointer to linked list of UserName's */
typedef UserName *UserNameList;

/* MACRO FUNCTIONS ======================================================== */

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

#ifdef MACRO_FUNCTION
UserName *User_NewUserName( /* void */ );
/* RETURN a pointer to a new, empty UserName structure.
*/
#endif MACRO_FUNCTION

#define User_NewUserName()\
    AllocOne(UserName)

#ifdef MACRO_FUNCTION
void User_DestroyUserName( /* UserName *un */ );
/* Destroy the UserName structure pointed to by 'un'.
*/
#endif MACRO_FUNCTION

#define User_DestroyUserName(_un)\
    Dealloc(_un)

String User_RegisterName( /* short id, IDType kind, String name */ );
/* Associates login-id 'name' with the system id 'id' having IDType 'kind'.

   RETURNS 'name' if no such login-id already exists in the table; NULL
   otherwise.
*/

String User_GetName( /* short id, IDType kind */ );
/* RETURNS the user or group name (depending on 'kind') corresponding to the
   uid or gid 'id'; NULL if no such ID was registered by the function
   User_RegisterID().
*/

String User_AddUserToGroup( /* short group_id, String user_name */ );
/* Adds 'user_name' to the list of user-names associated with the group having
   ID 'group_id'. The String 'user_name' must have already been registered by
   a call to Gen_BoxSaveSysname(); results are unpredictable if this has not
   been done.

   RETURNS a pointer to the added name if such a group has been registered by
   User_RegisterName(); NULL otherwise.
*/

UserNameList User_GetGroupMembers( /* short group_id */ );
/* RETURNS the UserNameList associated with the group 'group-id' by calls to
   User_AddUserToGroup(); NULL if no such group-id has been registered by a
   call to User_RegisterName().
*/

Boolean User_IsInList( /* UserNameList list, String user_name */ );
/* RETURN True iff 'user_name' is in the list 'list'.
*/

#ifdef MACRO_FUNCTION
Boolean User_IsInGroup( /* short group_id, String user_name */ );
/* RETURN True iff 'user_name' was added to the group 'group_id' by a call to
   User_AddUserToGroup().
*/
#endif

#define User_IsInGroup(_gid,_name)\
    User_IsInList(User_GetGroupMembers(_gid),(_name))

UserNameList User_NewList( /* String name */ );
/* RETURNS a 1-item UserNameList containing the name 'name'.
*/

UserNameList User_NewCopyList( /* UserNameList list */ );
/* RETURNS a copy of the UserNameList 'list'. New memory is allocated for the
   list structures, but not the names themselves.
*/

void User_DestroyList( /* UserNameList list */ );
/* Deallocates the space allocated for the UserNameList 'list'.
*/

Boolean User_IsSameList( /* UserNameList list1,list2 */ );
/* RETURNS True iff 'list1' and 'list2' contain the same user names (not nec.
   in the same order).
*/

void User_MinusListsD( /* UserNameList list1, INOUT UserNameList *list2 */ );
/* Sets 'list2' to a list representing 'list2'-list1'. That is, all elements
   in *both* 'list1' and 'list2' are removed from 'list2'.

   SIDE-EFFECT: 'list2' is changed, and UserName's removed from it are
   destroyed.
*/

UserNameList User_NewOrLists( /* UserNameList list1,list2 */ );
/* RETURNS: a pointer to a newly created UserNameList representing the
   union of 'list1' and 'list2'.
*/

UserNameList User_NewAndLists( /* UserNameList list1,list2 */ );
/* RETURNS: a pointer to a newly created UserNameList representing the
   intersection of 'list1' and 'list2'.
*/

#endif USER_H
