/* ===========================================================================
  PRINT.C -- contains function PrintPostscript and supporting functions to
  produce information about the Miro picture in PostScript form. 

  $Header: print.c,v 1.2 91/07/18 16:49:29 heydon Exp $

  Written by Amy Moormann Zaremski for the Miro project at Carnegie Mellon 
===========================================================================*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include "iff2ps.h"

/* name of PostScript header file */
#define HeaderFile "/usr/miro/libi/ps/iff2ps_header.ps"

/* *useable* paper size in inches */
#define X_WIDTH 7.95
#define Y_WIDTH 10.69

/* translation necessary to move origin to lower left corner of printable
   space on the page (in points) */
/* these values were determined by trial and error */
#define ORIGIN_X 18.5
#define ORIGIN_Y 8.8

/* maximum postscript coordinates; 72 points/inch in PostScript */
#define PS_INCHES_TO_POINTS 72
#define PS_X (PS_INCHES_TO_POINTS * X_WIDTH)
#define PS_Y (PS_INCHES_TO_POINTS * Y_WIDTH)

void PrintPostscript(box_list, arrow_list)
     BoxNode *box_list;
     ArrowNode *arrow_list;
{
#ifdef IFF2PS_DEBUG
  printf("Printing postscript... \n");
#endif 

  PrintHeader();
  PrintFormatInfo();
  PrintBoxes(box_list);
  PrintArrows(arrow_list);
  PrintEnd();
}

PrintHeader()
{
    /* local declarations */
    FILE *fp; 

    if ((fp = fopen(HeaderFile,"r")) != NULL) {
	ffilecopy(fp,stdout);
    } else {
	halt("Couldn't open PostScript header file.");
    }
}

PrintFormatInfo()
{
    /* size of page (includes scale and rotation) */
    float page_x = PS_X, page_y = PS_Y;
    float page_center_x, page_center_y;
    float fig_center_x, fig_center_y;
    float x_width, y_width, scale_x, scale_y;
    float swap_temp;

    /* print fontsize info */
    printf("%% set font info\n");
    printf("/fontsize %d def\n", fontsize);
    printf("/Helvetica findfont\n");     /* make this variable too? */
    printf("fontsize scalefont\n");
    printf("setfont\n");

    if (! in_text) {
	/* translate first so origin corresponds to where we can print */
	printf("%% translate so origin is where we can print\n");
	printf("%.1f %.1f translate\n", ORIGIN_X, ORIGIN_Y);
    }

    /* check for best-fit mode */
    if (best_fit) {
	x_width = (max_x - min_x);
	y_width = (max_y - min_y);

	/* determine if we want to be in portrait mode or not */
	if (y_width > x_width) {
	    rotate = FALSE;
	} else {
	    /* we are rotating, so swap roles of x and y */
	    swap_temp = x_width;
	    x_width = y_width;
	    y_width = swap_temp;
	}

	/* determine scaling factor */
	scale_x = page_x/x_width;
	scale_y = page_y/y_width;
	scale_factor = (scale_x < scale_y) ? scale_x : scale_y;
	
	/* turn on centering */
	center = TRUE;
    }

    /* scale (about origin) and update page_{X,Y} */
    printf("%.5f %.5f scale\n", scale_factor, scale_factor); 
    page_x /= scale_factor;
    page_y /= scale_factor;

    /* rotation */
    if (rotate) {
	/* origin is right bottom corner */
	printf("%% translate origin to bottom right corner and rotate\n");
	printf("%.5f 0 translate\n", page_x);
	printf("90 rotate\n");
	/* swap page_{x,y} */
	swap_temp = page_x;
	page_x = page_y;
	page_y = swap_temp;
    } else {
	/* no rotation */
	printf("%% no rotation (portrait mode)\n");
    }

    /* print centering info */
    if (center) {
	printf("%% center figure\n");

	/* find centers of page and figure */
	page_center_x = page_x/2;
	page_center_y = page_y/2;
	fig_center_x = (min_x + max_x)/2;
	fig_center_y = (min_y + max_y)/2;  

	/* translate to put center of fig at center of page */
	printf("%.5f %.5f translate\n", 
	       page_center_x - fig_center_x,
	       page_center_y - fig_center_y);
    } else {
	/* no centering */
	printf("%% no centering\n");
    }

    /* figure in text -> put at bottom left corner of page */
    if (in_text) {
	printf("%% place figure at bottom left\n");
	if (rotate) {
	    printf("%.5f %.5f translate\n", -min_x, page_y - max_y);
	} else {
	    printf("%.5f %.5f translate\n", -min_x, -min_y);
	}
    }
}

PrintBoxes(box_list)
     BoxNode *box_list;
{
  /* print boxes */
  for (; box_list != NULL; Next(box_list)) {
    printf("%d %d ", box_list->loc.x, box_list->loc.y);
    printf("%d %d ", box_list->size.x, box_list->size.y);
    printf("%d %d ", box_list->labelloc.x, box_list->labelloc.y);
    printf("(%s) ", box_list->name);
    printf("box\n");
    if (box_list->highlighted || box_list->thick) printf("3 draw\n");
    else printf("1 draw\n");
  }
}

PrintArrows(arrow_list)
     ArrowNode *arrow_list;
{
  /* print arrows */
  for (; arrow_list != NULL; Next(arrow_list)) {
    printf("%d %d ", arrow_list->tailloc.x, arrow_list->tailloc.y);
    printf("%d %d ", arrow_list->headloc.x, arrow_list->headloc.y);
    printf("%d %d ", arrow_list->labelloc.x, arrow_list->labelloc.y);
    print_bool(arrow_list->parity);
    /* extra stuff for semantics arrows */
    if (strcmp(arrow_list->kind, "sem") == 0)
      /* semantic arrow */
      { 
	printf("%c ",(arrow_list->thick ? '3' : '1'));
	if (*(arrow_list->type)) printf("({ %s }) ",arrow_list->type);
	else printf("() ");
	printf("dashedarrow\n");
      }
    else /* regular arrow */
      { 
	if (*(arrow_list->type)) printf("({ %s }) ",arrow_list->type);
	else printf("() ");
	printf("arrow\n%c draw\n",
	       (arrow_list->highlighted || arrow_list->thick) ? '3' : '1');
      }
  }
}

PrintEnd()
{
  if (!in_text)
    printf("showpage\n");

}

print_bool(b)
Bool b;
/* take a boolean argument and print 1 for true, 0 for false 
followed by a space */
{
    if (b == TRUE)	
	printf("1 ");
    else printf("0 ");
}
