/* ===========================================================================
  PROCESS.C -- contains functions to process the Miro IFF parse tree. 

  $Header: process.c,v 1.7 91/07/25 09:29:37 heydon Exp $

  ExtractLists extracts a list of boxes, a list of arrows, and a list of
  editor entries.

  AddHighlighting adds highlighting information from the editor entries to the
  box and arrow lists.

  Written by Amy Moormann Zaremski for the Miro project at Carnegie Mellon 
===========================================================================*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include "iff2ps.h"

/* local functions */
void ExtractEntryLists();
void ConvertBoxList();
void SetBoxDefaults();
void UpdateMaxMin();
void ConvertArrowList();
void ProcessTypeList();
void SetArrowDefaults();
void ConvertEdList();
void ProcessHLList();
void SetCoord();

/* local variables */
static int coord_val;		/* used to hold a coordinate (x or y) value
				   temporarily */

/* box and arrow variables -- here so can reset to default */
/* used by both */
static int sysname;
static Coord labelloc;
static Bool thick;
static Bool starred;
static Bool highlighted;

/* box variables */
static String name;
static Coord loc;
static Coord size;

/* arrow variables */
static String type;
static String kind;
static Coord tailloc;
static Coord headloc;
static Bool parity;

/* use Allan's AppendToList definition */
#define AppendToList(_entry_ptr,_ptr_to_last_ptr) \
        *(_ptr_to_last_ptr) = (_entry_ptr);       \
        _ptr_to_last_ptr = &(NextOf(_entry_ptr))


/* ------------------------------------------------------------
ExtractLists traverses the list p_tree. For each node
in p_tree, it adds a node to the appropriate list, ignoring
uneccessary entries. 
This is a two-step process - first, divide p_tree into 3 lists
of entries based on entry type. Then, convert each entry list 
to a node list (entries are complex generic structures produced 
by the parser; nodes are more basic structures that contain the
information necessary for the PS translator in an easily accessible
form). 
We have to do this in two steps because the extract library only 
deals with one entry type at a time.
This is based on the ambiguity checker's ExtractBoxArrowInsideLists.
------------------------------------------------------------ */
void ExtractLists(p_tree,boxes_ptr,arrows_ptr,ed_ptr)
     Entry *p_tree;    
     BoxNode **boxes_ptr;
     ArrowNode **arrows_ptr;
     EditorNode **ed_ptr;
{
  /* local lists of each type of entry */
  Entry *box_list, *arrow_list, *ed_list;

  /* Part 1: traverse parse tree and split into 3 entry lists */
#ifdef IFF2PS_DEBUG
  printf("Traversing the entry list...\n");
#endif 

  ExtractEntryLists(p_tree,&box_list,&arrow_list,&ed_list);

  /* Part 2: convert each entry list to a node list */
#ifdef IFF2PS_DEBUG
  printf("Converting the lists to nodes...\n");
#endif 
  InitExtract(MaxNumPNames);
  ConvertBoxList(box_list,boxes_ptr);
  ConvertArrowList(arrow_list,arrows_ptr);
  ConvertEdList(ed_list,ed_ptr); 
}


/* ------------------------------------------------------------
   ExtractEntryLists: place each entry in p_tree into the 
   appropriate entry list.
------------------------------------------------------------ */
void ExtractEntryLists(p_tree,b_list_ptr,a_list_ptr,e_list_ptr)
     Entry *p_tree,**b_list_ptr,**a_list_ptr,**e_list_ptr;
{
  for (; p_tree != NULL; Next(p_tree)) {
    switch (EntryTypeOf(p_tree)) {
    case BoxEntry:    AppendToList(p_tree,b_list_ptr); break;
    case ArrowEntry:  AppendToList(p_tree,a_list_ptr); break;
    case EditorEntry: AppendToList(p_tree,e_list_ptr); break;
    }
  }
  /* terminate the lists */
  *b_list_ptr = *a_list_ptr = *e_list_ptr = NULL;
}


/* ------------------------------------------------------------
   ConvertBoxList: take the list of box entries and
   produce a list of (processed) box nodes.
------------------------------------------------------------ */
void ConvertBoxList(entry_list,node_list)
     Entry *entry_list;
     BoxNode **node_list;
{
  /* local vars */
  BoxNode *new_node;
  ListDesc loc_list;
  ListDesc size_list;
  ListDesc labelloc_list;

#ifdef IFF2PS_DEBUG
  printf("Converting boxes\n");
#endif 

  /* set up prop name list */
  StartNewPNameList();
  AddPNameDesignator("sysname",True,myConvertInt,(Generic *)(&sysname),
		     NULL_LIST);  
  AddPNameDesignator("name",False,ConvertStr,(Generic *)(&name),
		     NULL_LIST);  
  AddPNameDesignator("loc",True,myConvertInt,(Generic *)NULL,
		     &loc_list);
  AddPNameDesignator("size",True,myConvertInt,(Generic *)NULL,
		     &size_list);
  AddPNameDesignator("label-loc",False,myConvertInt,(Generic *)NULL,
		     &labelloc_list);
  AddPNameDesignator("thickness",False,ConvertThick,(Generic *)(&thick),
		     NULL_LIST);  
  AddPNameDesignator("starred?",False,ConvertBool,(Generic *)(&starred),
		     NULL_LIST);  
  
  /* cycle through entry list, convert and add to box list */
  for (; entry_list != NULL; Next(entry_list)) {
    /* generate new node */
    new_node = (BoxNode *)InitNode(sizeof(BoxNode));
    /* reset default values of slots */
    SetBoxDefaults();
    if (!MatchPNames(entry_list)) {
      /* process coordinate values - check to see if they're there first*/
      /* know loc & size will be since they're required */
      SetCoord(&loc_list,&loc);
      SetCoord(&size_list,&size);
      UpdateMaxMin(loc,size);
      /* only set labelloc if value was found */
      if (ValFoundPOf(DesigPtrOf(&labelloc_list)))
	SetCoord(&labelloc_list,&labelloc); 
      SetBoxSlots(new_node,sysname,name,loc,size,labelloc,thick,starred,
		  highlighted);
      InsertNode((GenericEntry *)new_node,(GenericEntry **)node_list); 
    }
  }
}

/* set the default values for box entry property values */
void SetBoxDefaults()
{
  sysname = 0;
  name = "";
  loc.x = loc.y = 0;  
  size.x = size.y = 0;  
  labelloc.x = labelloc.y = 0;  
  thick = FALSE;
  starred = FALSE;
  highlighted = FALSE;
}

/* update max/min coords if necessary (this info used in rotating/centering) */
void UpdateMaxMin(loc, size)
     Coord loc,size;
{
  if (loc.x + size.x > max_x)
    max_x = loc.x + size.x;
  if (loc.y + size.y > max_y)
    max_y = loc.y + size.y;
  if (loc.x < min_x)
    min_x = loc.x;
  if (loc.y < min_y)
    min_y = loc.y;
}


/* ------------------------------------------------------------
   ConvertArrowList: take the list of arrow entries and
   produce a list of (processed) arrow nodes.
------------------------------------------------------------ */
void ConvertArrowList(entry_list,node_list)
     Entry *entry_list;
     ArrowNode **node_list;
{
  /* local vars */
  ArrowNode *new_node;
  ListDesc headloc_list;
  ListDesc tailloc_list;
  ListDesc labelloc_list;
  ListDesc type_list;

#ifdef IFF2PS_DEBUG
  printf("Converting arrows\n");
#endif 

  /* set up prop name list */
  StartNewPNameList();
  AddPNameDesignator("sysname",True,myConvertInt,
		     (Generic *)(&sysname),NULL_LIST);  
  AddPNameDesignator("permissions",False,ConvertStr,
		     (Generic *)NULL,&type_list);  
  AddPNameDesignator("kind",False,ConvertStr,
		     (Generic *)(&kind),NULL_LIST);
  AddPNameDesignator("tail-loc",True,myConvertInt,
		     (Generic *)NULL,&tailloc_list);
  AddPNameDesignator("head-loc",True,myConvertInt,
		     (Generic *)NULL,&headloc_list);
  AddPNameDesignator("label-loc",False,myConvertInt,
		     (Generic *)NULL,&labelloc_list);
  AddPNameDesignator("parity",True,myConvertParity,
		     (Generic *)(&parity),NULL_LIST);  
  AddPNameDesignator("thickness",False,ConvertThick,
		     (Generic *)(&thick),NULL_LIST);  
  AddPNameDesignator("starred?",False,ConvertBool,
		     (Generic *)(&starred),NULL_LIST);  
  
  /* cycle through entry list, convert and add to arrow list */
  for (; entry_list != NULL; Next(entry_list)) {
    /* generate new node */
    new_node = (ArrowNode *)InitNode(sizeof(ArrowNode));
    /* reset default values of slots */
    SetArrowDefaults();
    if (!MatchPNames(entry_list)) {
      /* process coordinate values - check to see if they're there first*/
      /* know head & tail loc will be since they're required */
      SetCoord(&tailloc_list,&tailloc);
      SetCoord(&headloc_list,&headloc);
      /* process the list of permissions (concatenate their names) */
      if (ValFoundPOf(DesigPtrOf(&type_list)))
	  ProcessTypeList(&type_list, &type); 
      /* only set labelloc if value was found */
      if (ValFoundPOf(DesigPtrOf(&labelloc_list)))
	SetCoord(&labelloc_list,&labelloc); 
      SetArrowSlots(new_node,sysname,type,kind,tailloc,headloc,labelloc,parity,
		    thick,starred,highlighted);
      InsertNode((GenericEntry *)new_node,(GenericEntry **)node_list); 
    }
  }
}

/* take a list of types and create a string which is a concatenation */
/* of those types */
void ProcessTypeList(type_list, type_ptr)
     ListDesc *type_list;
     String *type_ptr;
{
  unsigned new_size;
  String new_type,emalloc();

  /* process first element separately to get commas right */
  if (type_list->next_list_entry_ptr != NULL) {
    if (!MatchNextListElement(type_list, &new_type)) {
      /*
       * make local copy of the type */
      CopyString(*type_ptr,new_type);
      /*
       * process rest of list. */
      while (type_list->next_list_entry_ptr != NULL) { 
	if (!MatchNextListElement(type_list,&new_type)) {
	  /* increase size of string (+3 for the ", \0") */
	  new_size = strlen(*type_ptr) + strlen(new_type) + 3;
	  *type_ptr = (String)realloc(*type_ptr,new_size);
	  strcat(*type_ptr,", ");
	  strcat(*type_ptr,new_type);
	}}
    }}
}

/* set the default values for arrow entry property values */
void SetArrowDefaults()
{
  sysname = 0;
  type = "";
  kind = "";
  headloc.x = headloc.y = 0;  
  tailloc.x = tailloc.y = 0;  
  labelloc.x = labelloc.y = 0;  
  parity = TRUE;
  thick = FALSE;
  starred = FALSE;
  highlighted = FALSE;
}


/* ------------------------------------------------------------
   ConvertEdList: take the list of editor entries and
   produce an editor node. Right now, only need one node 
   because we're only saving highlighting lists, and we
   merge all the highlighting entries into one. 
------------------------------------------------------------ */
void ConvertEdList(entry_list,ed_node)
     Entry *entry_list;
     EditorNode **ed_node;
{
  /* local vars & functions */
  ListDesc hlboxes_list;
  ListDesc hlarrows_list;

#ifdef IFF2PS_DEBUG
  printf("Converting editor nodes\n");
#endif 
  /* generate node and initialize -- 
     only need one since merging highlighting lists */
  *ed_node= (EditorNode *)InitNode(sizeof(EditorNode));
  (*ed_node)->highlightedboxes = (*ed_node)->highlightedarrows = NULL;

  /* set up prop name list */
  StartNewPNameList();
  AddPNameDesignator("highlightboxes",False,myConvertInt,(Generic *)NULL,
		     &hlboxes_list);
  AddPNameDesignator("highlightarrows",False,myConvertInt,(Generic *)NULL,
		     &hlarrows_list);
  
  /* cycle through entry list, convert and add to editor node */
  for (; entry_list != NULL; Next(entry_list)) {
    /* reset default values of slots */
    if (!MatchPNames(entry_list)) {
      /* process highlighting lists, if there */
      if (ValFoundPOf(DesigPtrOf(&hlboxes_list)))
	ProcessHLList(hlboxes_list,&((*ed_node)->highlightedboxes));
      if (ValFoundPOf(DesigPtrOf(&hlarrows_list)))
	ProcessHLList(hlarrows_list,&((*ed_node)->highlightedarrows)); 
    }
  }
}

/* "process" a highlighting list -- add the numbers in new_list 
   to the ordered list hl_list */
void ProcessHLList(new_list,hl_list)
     ListDesc new_list;
     IntNode **hl_list;
{
  IntNode *new_node;

  while (new_list.next_list_entry_ptr != NULL) {
    if (!MatchNextListElement(&new_list)) {
      /* create new node */
      new_node = (IntNode *)InitNode(sizeof(IntNode));
      new_node->value = coord_val;
      InsertOrderedList(new_node, hl_list);
    }
  }
}

/* Insert an integer node into an ordered list of nodes */
InsertOrderedList(node, list)
     IntNode *node;
     IntNode **list;
{
    int val = node->value;

    /* find place to insert list */
    for (; (*list) != NULL && val > (*list)->value; list = &((*list)->next));

    /* only insert element if it is not a duplicate */
    if ((*list) == NULL || val < (*list)->value) {
	node->next = *list;
	(*list) = node;
    }
}

/* debugging function -- print elements of list */
PrintList(list)
     IntNode *list;
{
  printf("List = ");
  for (; list != NULL; Next(list)) {  
    printf("%d, ", list->value);
  }
  printf(".\n");
}


/* ------------------------------------------------------------
   SetCoord: set coordinate values based on coord list. 
------------------------------------------------------------ */
void SetCoord(coord_list, coord)
     ListDesc *coord_list;
     Coord *coord;
{
  /* set x coord */
  if (NextListEntryPtrOf(coord_list) != NULL) {
    if (!MatchNextListElement(coord_list,&(coord->x))) {
      /* set y coord */
      if (NextListEntryPtrOf(coord_list) != NULL) {
	MatchNextListElement(coord_list,&(coord->y));
      } else {			/* no y entry */
	ParseErrorS(StartLineNumberOf(coord_list),
		"'%s' is a coordinate and requires two integers in the list",
		PNameOf(DesigPtrOf(coord_list)));
      }
    }
  } else /* no x entry */ {
    ParseErrorS(StartLineNumberOf(coord_list),
		"'%s' is a coordinate and requires two integers in the list",
		PNameOf(DesigPtrOf(coord_list)));
  }
}


/* ------------------------------------------------------------
   ProcessHighlighting: add highlighting information to the lists
   of boxes and arrows. Prints an error message if any objects
   listed in the highlighting lists that do not exist. 
------------------------------------------------------------ */
void ProcessHighlighting(box_list, arrow_list, ed_node)
     BoxNode *box_list;
     ArrowNode *arrow_list;
     EditorNode *ed_node;
{
  /* assuming that there's just one editor node and that all 
     highlighting info is in there */
  IntNode *hlb = ed_node->highlightedboxes;
  IntNode *hla = ed_node->highlightedarrows;

  /* add highlighting info for boxes */
  while (box_list != NULL && hlb != NULL) {
    if (HighlightedP(box_list->sysname,&hlb)) {
      box_list->highlighted = TRUE;
    }
    box_list = box_list->next;
  }

  /* add highlighting info for arrows */
  while (arrow_list != NULL && hla != NULL) {
    if (HighlightedP(arrow_list->sysname,&hla)) {
      arrow_list->highlighted = TRUE;
    }
    arrow_list = arrow_list->next;
  }
	
  /* check to see if any boxes/arrows still listed to highlight */
  if (hlb != NULL)
    halt("Boxes to be highlighted which were not declared\n");
  if (hla != NULL)
    halt("Arrows to be highlighted which were not declared\n");
}
