;;; BOXTYPE.CL -- Boxtype structures/functions
;;;
;;; $Header: boxtype.cl,v 1.4 91/09/17 16:31:00 heydon Locked $
;;;
;;; Written by Allan Heydon for the Miro project at Carnegie Mellon
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

(provide 'boxtype)
(in-package 'boxtype)

(export
 '(start-boxtypes end-boxtypes		;boxtype bracketting functions
   new-boxtype				;create and register a new boxtype
   box-type				;type of a box
   box-side				;side for a box's type
   type-comp))				;type comparison function

(require 'constraints)

;;; BOXTYPE OVERVIEW ==========================================================
;;;
;;; This module provides facilities for registering a boxtype tree. Each type
;;; has a name (which is an atom (or identifier)) and an optional parent.
;;; Those types not having parents are roots of the type tree.
;;;
;;; The tree is constructed by a call to START-BOXTYPES. Each type is then
;;; registered by a call to NEW-BOXTYPE. These calls define new types and
;;; their parents. It is an error to register a type before its parent type
;;; (if it has one) has been registered. When all types have been registered,
;;; a call should be made to END-BOXTYPES.
;;;
;;; This module also defines a comparison function, TYPE-COMP, on type names.
;;; TYPE-COMP can be used as the comparison function in a BST-HDS in which the
;;; keys are type names.
;;;
;;; OPERATIONS
;;;
;;; (START-BOXTYPES [size])
;;;   Begin registering box types. SIZE is the size of the initial hash table
;;;   for the types. It defaults to 51.
;;;
;;; (NEW-BOXTYPE type-name type-side parent-name attributes...)
;;;   Registers a new type named TYPE-NAME, having "side" TYPE-SIDE (which
;;;   should be either :SUBJ, :OBJ, or NIL) and having a parent type named
;;;   PARENT-NAME. It is an error for the type named PARENT-NAME not to have
;;;   been previously registered.
;;;
;;; (END-BOXTYPES)
;;;   End registration of box types. This routine must be called for the
;;;   comparison function TYPE-COMP to function correctly.
;;;
;;; (BOX-SIDE box)
;;;   Evaluates to the "side" associated with (BOXTYPE BOX).
;;;
;;; (TYPE-COMP t1-name t2-name)
;;;   Returns 2 values. The first value is -1, 0, or 1 as the type named by
;;;   T1-NAME is less than, equal to, or greater than the type named by
;;;   T2-NAME, respectively, according to an arbitrary *total* order on types.
;;;   The second value is T or NIL as the two types are related by the subtype
;;;   relation corresponding to the *partial* order on types or not.

;;; LOCAL VARIABLES ===========================================================

(defvar *roots* NIL "List of BoxTypes with no parents.")
(defvar *table* NIL "Hash table for BoxTypes.")

;;; DEFINING BOXTYPE TREE =====================================================

(deftype String-Or-Nil () '(or string null))
(deftype Side-Type () '(member :SUBJ :OBJ NIL))

(defstruct boxtype
  ;; Invariant: LOW and HIGH are such that all descendents of this type have a
  ;; LOW value that is in this type's range (LOW,HIGH].
  (name     NIL :type String-Or-Nil :read-only t) ;name of this boxtype
  (side     NIL :type Side-Type     :read-only t) ;side for this boxtype
  (children NIL :type list)                       ;list of child sub-types
  (low      0   :type fixnum)                     ;ID of type for anc. comp
  (high     0   :type fixnum))                    ;upper bound for anc. comp

;;; (START-BOXTYPES [num]) ----------------------------------------------------
;;;
(defmacro start-boxtypes (&optional (num 51))
  (declare (type fixnum num))
  `(setq *roots* NIL *table* (make-hash-table :test #'eq :size ,num)))

;;; (END-BOXTYPES) ------------------------------------------------------------
;;;
(defun end-boxtypes ()
  (mapc #'scan-root-node *roots*))
(defun scan-root-node (node)
  (scan-node node 0))
(defun scan-node (node num)
  (declare (type boxtype node)
	   (type fixnum num))
  (setf (boxtype-low node) (incf num))
  (dolist (child (boxtype-children node))
    (declare (type boxtype child))
    (setq num (scan-node child num)))
  (setf (boxtype-high node) num))

;;; (GET-TYPE type-name), (SET-TYPE type-name) --------------------------------
;;;
(defmacro get-type (type-name)
  (declare (type symbol type-name))
  `(gethash ,type-name *table*))
(defmacro set-type (type-name type-val)
  (declare (type symbol type-name)
	   (type boxtype type-val))
  `(setf (get-type ,type-name) ,type-val))

;;; (NEW-BOXTYPE name side [parent-name]) -------------------------------------
;;;
(defun register (name side parent-name)
  (declare (type symbol name parent-name)
	   (type Side-Type side))
  (let ((bt (make-boxtype :name name :side side))	;new boxtype
	pt)				                ;parent boxtype
    (declare (type boxtype bt pt))
    ;; install the new boxtype in the table
    (set-type name bt)
    ;; add it as a child if its parent is specified
    (if parent-name
      (if (setq pt (get-type parent-name))
	  (setf (boxtype-children pt) (cons bt (boxtype-children pt)))
	  (error "Box type '~A' does not exist" parent-name))
      (push bt *roots*))))

(defmacro new-boxtype (name side parent &rest attr-list)
  (declare (type symbol name parent)
	   (type Side-Type side)
	   (type list attr-list))
  (let* ((header1 `(,name (:conc-name constraints::box-)))
	 (header2 (append header1 (if parent `((:include ,parent)) nil)))
	 (result `(defstruct ,header2)))
    (declare (type cons result))
    (dolist (attr attr-list)
      (declare (type symbol attr))
      (when (not (eq attr 'type))
	(nconc result `((,attr :NONE :type t :read-only t)))))
    `(progn
      (boxtype::register ',name ,side ',parent)
      ,result)))

;;; (BOX-TYPE box) ------------------------------------------------------------
;;;
(defun box-type (box) (type-of box))

;;; (BOX-SIDE box) ------------------------------------------------------------
;;;
(defun box-side (box)
  (boxtype-side (get-type (type-of box))))

;;; (TYPE-COMP t1-name t2-name) -----------------------------------------------
;;;
(defun type-comp (t1-name t2-name)
  (declare (type symbol t1-name t2-name))
  (if (eq t1-name t2-name) (values 0 T)
    (let ((t1 (get-type t1-name))
	  (t2 (get-type t2-name)))
      (declare (type boxtype t1 t2))
      (if (sub-type t1 t2) (values -1 T)
	  (if (sub-type t2 t1) (values 1 T)
	      (if (> (boxtype-low t1) (boxtype-low t2))
		  (values -1 NIL)
		  (values 1 NIL)))))))

;;; (SUB-TYPE t1 t2) ----------------------------------------------------------
;;;
;;; Returns T iff T1 is a strict subtype of T2 according to the *partial*
;;; order on types; NIL otherwise.
;;;
(defun sub-type (t1 t2)
  (declare (type boxtype t1 t2))
  (and (> (boxtype-low t1) (boxtype-low t2))
       (<= (boxtype-low t1) (boxtype-high t2))))
