;;; FUNC-OPS.CL -- Operations on functions
;;;
;;; $Header: func-ops.cl,v 1.2 91/09/11 13:45:11 heydon Exp $
;;;
;;; Written by Allan Heydon for the Miro project at Carnegie Mellon
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

(provide 'func-ops)
(in-package 'func-ops)
(export
 '(string-comp				;string comparison function
   cross				;cross of two integeers
   fcross				;cross of two functions
   fcompose))				;composition of two functions

;;; (STRING-COMP s1 s2)
;;;
;;; Returns -1, 0, or 1 as S1 is less than, equal to, or greater than S2,
;;; respectively. S1 and/or S2 may be symbols, but remember that all
;;; alphabetic characters in a symbol's print-name are capatolized.
;;;
(defun string-comp (s1 s2)
  (declare (type string s1 s2))
  (cond ((string< s1 s2) -1) ((string> s1 s2) 1) (t 0)))

;;; (CROSS n1 n2)
;;;
;;; Cross implements an injection: Z+ x Z+ -> Z+, where Z+ is the set of
;;; positive integers. It implements this unique pairing using a slight
;;; modification (for efficiency) of the classical dovetail pattern over the
;;; grid of natural numbers. To get the exact dovetail pattern, and hence a
;;; bijection, n3 should be computed as (1- (+ n1 n2)) instead.
;;;
(defun cross (n1 n2)
  (declare (type fixnum n1))
  (declare (type fixnum n2))
  (let ((n3 (+ n1 n2)))
    (declare (type fixnum n2))
    (+ n1 (/ (* n3 (1- n3)) 2))))

;;; (FCROSS f1 f2)
;;;
;;; Returns a function object G such that (G x) = (cross (f1 x) (f2 x)).
;;;
(defun fcross (f1 f2)
  (declare (type function f1))
  (declare (type function f2))
  (function (lambda (x) (cross (funcall f1 x) (funcall f2 x)))))

;;; (FCOMPOSE f1 f2 ...)
;;;
;;; Returns a function object representing the composition of f1, f2,
;;; etc. The functions are composed in left-to-right order.
;;;
;;; EXAMPLES:
;;;
;;; (setf (symbol-function 'plus-plus-one) (fcompose #'+ #'1+))
;;; (plus-plus-one 2 3) => 6
;;; (apply 'plus-plus-one '(2 3)) => 6
;;;
;;; (setf (symbol-function 'car-not) (fcompose #'car #'not))
;;; (car-not '(t nil 2 3)) => NIL
;;; (car-not '(nil 6)) => T
;;;
(defun fcompose (&rest fns)
  (declare (type list fns))
  (function (lambda (&rest x)
    (dolist (f fns (car x))
      (setq x (list (apply f x))))))) 
