;;; HDS.CL - Hierarchical Data Structure Classes
;;;
;;; $Header: hds.cl,v 1.8 91/10/30 23:44:53 heydon Exp $
;;;
;;; Written by Allan Heydon for the Miro project at Carnegie Mellon
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

(provide 'hds)
(in-package 'hds)

(export					;Global Discriminator Tree variables
 '(*box-dt*				;HDS for boxes
   *syn-dt*				;HDS for syntax arrows
   *subj-dt*				;HDS for subjects
   *obj-dt*				;HDS for objects
   *con-dt*))				;HDS for containment arrows
(export					;HDS FUNCTIONS
 '(add-item				;add item to an HDS
   get-item				;retrieve item from HDS
   make-iter-on				;create an iterator on an HDS
   ))
(export					;LEAF-HDS FUNCTIONS
 '(make-one-item			;create a one-item-hds
   make-bag				;create a bag-hds
   ))
(export					;NON-LEAF-HDS FUNCTIONS
 '(extend				;build off of a non-leaf-hds
   ))
(export					;HT-HDS FUNCTIONS
 '(make-ht				;create an ht-hds (hash table)
   ))
(export					;XHT-HDS FUNCTIONS
 '(make-xht				;create an xht-hds (exclusive HT)
   ))
(export					;BST-HDS FUNCTIONS
 '(make-bst				;create a bst-hds (binary search tree)
   ))
(export					;types
 '(list-vector				;array type of ht-hds
   xht-entry				;xht-hds array entry
   xht-array				;array type of xht-hds
   ))

(require 'flavors)
(require 'objs)
(require 'iter)
(use-package 'flavors)
(use-package 'objs)
(use-package 'iter)

;;; HDS TYPES OVERVIEW ========================================================

;;; It is important to know the HDS type hierarchy, since methods defined for
;;; a particular type are available to all subtypes of that type. The type
;;; tree is given by:
;;;
;;; HDS				; HDS super-type
;;;   LEAF-HDS			; non-extensible HDS type
;;;     ONE-ITEM-HDS		; stores only one item
;;;     BAG-HDS			; stores an unlimited number of items
;;;   NON-LEAF-HDS		; extensible HDS type
;;;     HT-HDS			; hash table (duplicates may be in same bucket)
;;;       XHT-HDS		; exclusive hash table (no bucket duplicates)
;;;     BST-HDS			; binary search tree

;;; The NON-LEAF-HDS's use nodes to store their items. Although this
;;; information is not needed to use this module, it should help the
;;; implmentor. The nodes are arranged in the following type-tree:
;;;
;;; HDS-NODE			; simple node used by HT-HDS's
;;;   KEY-NODE			; used by XHT-HDS's
;;;     BST-NODE		; used by BST-HDS's

;;; HDS OVERVIEW ==============================================================
;;;
;;; This module defines abstract data types for constructing hierarchical data
;;; structures (HDS's), that is, data structure objects whose entries are
;;; recursive data structure objects. Abstractly, an HDS provides a (possibly
;;; unbounded) set of "slots" for storing data items. When a new item is
;;; inserted, a "key" for that item is computed (how the key is computed may
;;; be a function of the HDS), and this key is used to determine which slot
;;; the item is stored in.
;;;
;;; There are two kinds of HDS objects: leaf HDS's (implemented in
;;; lead-hds.cl) and non-leaf HDS's (implemented in non-leaf-hds.cl and
;;; bst-hds.cl, ht-hds.cl, and xht-hds.cl). The difference is that the slots
;;; of leaf HDS's contain data items, whereas the slots of the non-leaf HDS's
;;; contain nested HDS's. Each chain of nested HDS containments eventually
;;; ends in a leaf HDS. Overall, the containment relation amongst HDS's is a
;;; tree.
;;;
;;; As an HDS is built, it is extended by describing a path from the root of
;;; the tree down to some leaf, and adding on a new HDS there. Such paths take
;;; the form of a list, each entry of which is a value to use as the key for
;;; each level of the HDS-tree, with the first element being used at the top
;;; level, the second element at the second level, and so on.
;;;
;;; An element may be the keyword :OTHERS or :NEW-SLOT, in which case the
;;; extension is meant to apply to all other not explicitly installed slots of
;;; the hds at that level. This is implemented by keeping a "default-HDS" for
;;; each HDS. The difference between the two is the behavior of the HDS at
;;; run-time. If an element is inserted whose key does not match any of the
;;; keys already in the HDS, then the "default-HDS" is consulted. If it is an
;;; :OTHERS HDS, then the element is simply inserted into the default-hds. If
;;; it is a :NEW-SLOT HDS, then a *copy* of the default-HDS is made and a slot
;;; is created for this one element; the element is then inserted into the
;;; newly made "default-HDS" copy.
;;;
;;; It is also possible to iterate over items inserted into either :OTHERS or
;;; :NEW-SLOT default-HDS structures by using the :OTHERS keyword in the
;;; creation of an iterator (see MAKE-ITER-ON below).
;;;
;;; OPERATIONS
;;;
;;; (ADD-ITEM hds item)
;;;   Add ITEM to the data structure HDS.
;;;
;;; (GET-ITEM hds [path])
;;;   Return the item at the end of PATH in HDS. PATH should be a list of key
;;;   values (*not* containing the keyword :ALL), and must specify a path
;;;   through HDS ending in a leaf-hds (it is an error for PATH to end at a
;;;   non-leaf-hds, or for it to attempt to extend past a leaf-hds). PATH
;;;   defaults to NIL, in which case HDS must be a leaf-hds.
;;;
;;; (MAKE-ITER-ON hds [path])
;;;   Create and return an iterator on HDS according to PATH. PATH is a list
;;;   of key values, which is "followed" down through HDS as far as PATH goes.
;;;   The iterator returned will iterate over all items contained in the
;;;   subtree of data-structure at the end of PATH. PATH defaults to NIL.
;;;
;;;   The key item :OTHERS in PATH is matched by all items installed through
;;;   the default HDS at that level. If the default HDS was created with the
;;;   :NEW-SLOT keyword, then the iteration is instead over all items in those
;;;   slots created at run-time in response to insertions.
;;;
;;;   The key item :ALL in PATH is matched by *all* items in the data
;;;   structure at that level. Therefore, appending :ALL to a PATH not already
;;;   ending in :ALL results in an equivalent PATH specification. However, for
;;;   the sake of efficiency, :ALL should *never* be used as the last
;;;   component of PATH.
;;;
;;;   If PATH specifies a key for a leaf-hds or a key that does not exist in a
;;;   non-leaf-hds, an "empty" iterator is returned.

;;; LEAF-HDS OVERVIEW =========================================================
;;;
;;; LEAD-HDS objects are data-structures providing only the ADD-ITEM and
;;; MAKE-ITER-ON methods. They hold atomic items, not recursive HDS's, unlike
;;; their NON-LEAF-HDS counterparts. There are two kinds of leaf-hds: the
;;; "one-item-hds", which holds only one item (the most recently one put
;;; there), and the "leaf-hds", which holds a list of items.
;;;
;;; OPERATIONS
;;;
;;; Creating LEAD-HDS objects:
;;;
;;; (MAKE-ONE-ITEM)
;;;   Create and return a one-item-hds, a leaf-hds for holding one item.
;;;
;;; (MAKE-BAG)
;;;   Create and return a bag-hds, a leaf-hds for holding a collection of
;;;   items.

;;; NON-LEAF-HDS OVERVIEW =====================================================
;;;
;;; NON-LEAF-HDS objects provide *extensible* hierarchical data-structures.
;;; They store nested data-structures. In addition to the ADD-ITEM and
;;; MAKE-ITER-ON methods, they also provide an EXTEND operation for building
;;; HDS's off an existing HDS (this is equivalent to adding children to a node
;;; in the data-structure tree).
;;;
;;; OPERATIONS
;;;
;;; Extending non-leaf-hds objects:
;;;
;;; (EXTEND hds-var path new-hds)
;;;   Extend HDS-VAR by adding NEW-HDS at the end of PATH. HDS-VAR should be
;;;   one of the global variables BOX-DT, SYN-DT, SEM-DT, or CON-DT. PATH may
;;;   contain the keyword :OTHERS, which indicates that the default-hds should
;;;   be followed at that point in path. It may also contain the keyword
;;;   :NEW-SLOT, which indicates that a new slot should be created at run-time
;;;   for the key being inserted if no matching slot already exists. If PATH
;;;   is the empty list, then HDS-VAR is simply bound to NEW-HDS. The NEW-HDS
;;;   is returned.

;;; HT-HDS OVERVIEW ===========================================================
;;;
;;; This module provides a non-leaf-hds arranging its buckets in a simple hash
;;; table. The buckets do not necessarily contain items that all have the same
;;; key: items with different keys may be placed in the same bucket. This is
;;; contrasted with exclusive hash table (hxt-hds) HDS's.
;;;
;;; OPERATIONS
;;;
;;; Creating HT-HDS objects:
;;;
;;; (MAKE-HT key-func :size)
;;;   Create and return a ht-hds, a non-leaf-hds for holding a set of items
;;;   organized as a set of buckets in a hash table. In this general hash
;;;   table, items with different keys may end up in the same bucket.
;;;
;;;   KEY-FUNC is the function applied to an item as it is inserted to produce
;;;   its storage key. A hash function is then applied to the key, and the
;;;   item is inserted in the corresponding bucket.
;;;
;;;   SIZE is the size of the hash table; it defaults to 128.

;;; XHT-HDS OVERVIEW ==========================================================
;;;
;;; This module provides a non-leaf-hds arranging its buckets in an exclusive-
;;; bucket hash table. The fact that buckets are exclusive means that all
;;; items in the same bucket are guarenteed to have the same key. Since the
;;; size of the hash table (number of buckets it contains) must grow with the
;;; number of different keys of items stored in the table, provisions are made
;;; for this automatic growth.
;;;
;;; OPERATIONS
;;;
;;; Creating XHT-HDS objects:
;;;
;;; (MAKE-XHT key-func :size :equal-func :threshold :rehash-factor)
;;;   Create and return an xht-hds, a non-leaf-hds for holding a set of items
;;;   organized as a set of *exclusive* buckets in a hash table.
;;;
;;;   KEY-FUNC is the function applied to an item as it is inserted to produce
;;;   its storage key. A hash function is applied to the key to find the
;;;   bucket the item should be placed in.
;;;
;;;   SIZE is the initial size of the hash table; it defaults to 128.
;;;
;;;   EQUAL-FUNC is a function applied to a pair of keys to test for equality;
;;;   it is typically one of #'eq, #'eql, or #'equal, and defaults to #'eq.
;;;
;;;   THRESHOLD indicates how full the table should be allowed to get before
;;;   it is grown. It is  either an integer (which should be < SIZE), or a
;;;   float <= 1.0, indicating the fraction of the table that is allowed to be
;;;   filled; it defaults to 0.8.
;;;
;;;   When the table is grown, its size is expanded by a factor of
;;;   REHASH-FACTOR. This value should be a float > 1.0, and it deaults to
;;;   2.1 (the unevenness of this default should make for good table sizes).

;;; BST-HDS OVERVIEW ==========================================================
;;;
;;; This module provides a non-leaf-hds data-structure organizing buckets in a
;;; binary search tree.
;;;
;;; OPERATIONS
;;;
;;; Creating BST-HDS objects:
;;;
;;; (MAKE-BST key-func [compare-func])
;;;   Create and return a bst-hds, a non-leaf-hds for holding a set of items
;;;   organized as a binary tree. KEY-FUNC is the function applied to an item
;;;   as it is inserted to extract its storage key. COMPARE-FUNC is the
;;;   function to apply to keys to determine a total ordering on them; it
;;;   should take two arguments and return a negative number, a positive
;;;   number, or 0 as the first argument is less than, greater than, or equal
;;;   to the second argument respectively. COMPARE-FUNC defaults to #'-, which
;;;   is the comparison function for integers.

;;; TYPES =====================================================================

(deftype list-vector () '(vector list))
(deftype xht-entry () '(or null key-node))
(deftype xht-array () '(vector xht-entry))

;;; GLOBAL VARIABLES ==========================================================

(defvar *box-dt*  nil "Box Discriminator Tree.")
(defvar *syn-dt*  nil "Syntax Arrow Discriminator Tree.")
(defvar *subj-dt* nil "Subject Discriminator Tree.")
(defvar *obj-dt*  nil "Object Discriminator Tree.")
(defvar *con-dt*  nil "Containment Arrow Discriminator Tree.")

;;; HDS-OBJECT TYPES ==========================================================

;;; HDS -----------------------------------------------------------------------
;;;
(defflavor hds
  ((storage NIL))			;data structure stored here
  ()					;base class
  (:documentation "Basic class for Hierarchical Data Structures.")
  (:gettable-instance-variables storage)
  (:settable-instance-variables storage)
  (:required-methods :add-item :copy :get-item :make-iter-on))

;;; LEAF-HDS ------------------------------------------------------------------
;;;
(defflavor leaf-hds ()			;no extra state
  (hds)
  (:documentation "Basic class for leaf HDS's."))

;;; ONE-ITEM-HDS --------------------------------------------------------------
;;;
(defflavor one-item-hds ()		;no extra state
  (leaf-hds)
  (:documentation "Leaf HDS for holding a single item."))

;;; BAG-HDS -------------------------------------------------------------------
;;;
(defflavor bag-hds ()			;no extra state
  (leaf-hds)
  (:documentation "Leaf HDS for holding a collection of items."))

;;; NON-LEAF-HDS --------------------------------------------------------------
;;;
(defflavor non-leaf-hds
    (key-func			;function applied to item to produce key
     (default-hds NIL)		;default HDS in :OTHERS and :NEW-SLOT cases
     (new-slot? NIL))	        ;T iff :NEW-SLOT was specified
  (hds)
  (:documentation "Superclass for non-leaf HDS's.")
  (:initable-instance-variables key-func)
  (:gettable-instance-variables default-hds new-slot?)
  (:settable-instance-variables default-hds new-slot?)
  (:required-init-keywords :key-func)
  (:required-methods
   :extend :extend-thru-default :extend-thru-structure
   :find-key-node :make-new-node))

;;; HT-HDS --------------------------------------------------------------------
;;;
(defflavor ht-hds
    (table-size				;hash table size
     equal-func)			;equality function for keys
  (non-leaf-hds)
  (:documentation "Non-leaf HDS implementing a hash table.")
  (:initable-instance-variables table-size equal-func)
  (:gettable-instance-variables table-size equal-func)
  (:required-init-keywords :table-size :equal-func)
  (:required-methods :add-key-node :reorganize-p :new-ht-it))

;;; XHT-HDS -------------------------------------------------------------------
;;;
(defflavor xht-hds
    ((entries 0)			;number of entries
     threshold				;number of entries to rehash at
     max-entries			;real table size
     rehash-factor)			;factor for new # entries on rehash
  (ht-hds)
  (:documentation "Non-leaf HDS implementing an exclusive hash table.")
  (:initable-instance-variables threshold rehash-factor)
  (:gettable-instance-variables entries threshold rehash-factor)
  (:settable-instance-variables entries)
  (:required-init-keywords :threshold :rehash-factor)
  (:required-methods :rehash))

;;; BST-HDS -------------------------------------------------------------------
;;;
(defflavor bst-hds
    (compare-func)			;comparison function on keys
  (non-leaf-hds)
  (:documentation
   "Non-leaf HDS distinguishing data with a binary search tree.")
  (:initable-instance-variables compare-func)
  (:required-init-keywords :compare-func))

;;; HDS-OBJECT MACROS =========================================================

;;; HDS Macros ----------------------------------------------------------------

;;; Methods:
(defmacro add-item (hds item)       `(send ,hds :add-item ,item))
(defmacro copy (object)             `(send ,object :copy))
(defmacro get-item (hds &optional (path NIL))
  `(send ,hds :get-item ,path))
(defmacro make-iter-on (hds &optional (path NIL))
  `(send ,hds :make-iter-on ,path))

;;; Reader/Writers:
(defmacro storage (hds)             `(send ,hds :storage))
(defmacro set-storage (hds val)     `(send ,hds :set-storage ,val))

;;; NON-LEAF-HDS Macros -------------------------------------------------------

;;; Methods:
(defmacro extend-thru (hds path new-hds)
  `(send ,hds :extend ,path ,new-hds))
(defmacro extend (hds path new-hds)
  `(let ((.path. ,path))
    (if (null .path.)
      (setf ,hds ,new-hds)
      (extend-thru ,hds .path. ,new-hds))))
(defmacro find-key-node (hds key)
  `(send ,hds :find-key-node ,key))
(defmacro make-new-node (hds key val &optional (other? T))
  `(send ,hds :make-new-node ,key ,val ,other?))
(defmacro extend-thru-default (hds path-tail new-hds)
  `(send ,hds :extend-thru-default ,path-tail ,new-hds))
(defmacro extend-thru-structure (hds key path-tail new-hds)
  `(send ,hds :extend-thru-structure ,key ,path-tail ,new-hds))

;;; Reader/Writers:
(defmacro default-hds (hds)         `(send ,hds :default-hds))
(defmacro set-default-hds (hds val) `(send ,hds :set-default-hds ,val))
(defmacro new-slot? (hds)           `(send ,hds :new-slot?))
(defmacro set-new-slot? (hds val)   `(send ,hds :set-new-slot? ,val))

;;; HT-HDS Macros -------------------------------------------------------------

;;; Methods:
(defmacro reorganize-p (hds)        `(send ,hds :reorganize-p))
(defmacro add-key-node (hds index key new-node)
  `(send ,hds :add-key-node ,index ,key ,new-node))
(defmacro new-ht-it (hds others? rest)
  `(send ,hds :new-ht-it ,others? ,rest))

;;; Reader/Writers:
(defmacro table-size (ht)           `(send ,ht :table-size))
(defmacro equal-func (ht)           `(send ,ht :equal-func))

;;; XHT-HDS Macros ------------------------------------------------------------

;;; Methods:
(defmacro rehash (hds)              `(send ,hds    :rehash))

;;; Reader/Writers:
(defmacro entries (hds)             `(send ,hds :entries))
(defmacro set-entries (hds val)     `(send ,hds :set-entries ,val))
(defmacro threshold (hds)           `(send ,hds :threshold))
(defmacro rehash-factor (hds)       `(send ,hds :rehash-factor))

;;; NODE-OBJECT TYPES =========================================================

;;; HDS-NODE ------------------------------------------------------------------
;;;
(defflavor hds-node
    (hanging-hds			;HDS hanging off this node
     other?)				;was this node added at "add-time"?
  ()					;base class
  (:documentation "Node within a non-leaf HDS for storing another HDS.")
  (:initable-instance-variables hanging-hds other?)
  (:gettable-instance-variables hanging-hds other?)
  (:required-init-keywords :hanging-hds :other?)
  (:required-methods :copy))

;;; KEY-NODE ------------------------------------------------------------------
;;;
(defflavor key-node
    ((key-val NIL))			;key for the node
  (hds-node)
  (:documentation
   "Node within a non-leaf HDS for storing a key and its associated HDS.")
  (:initable-instance-variables key-val)
  (:gettable-instance-variables key-val)
  (:required-init-keywords :key-val))

;;; BST-NODE ------------------------------------------------------------------
;;;
(defflavor bst-node
    ((left NIL)				;left child
     (right NIL))			;right child
  (key-node)
  (:documentation "Internal key-node for a bst-hds.")
  (:gettable-instance-variables left right)
  (:settable-instance-variables left right)
  (:required-methods :expand :expand-limited))

;;; NODE-OBJECT MACROS ========================================================

;;; HDS-NODE Macros -----------------------------------------------------------

;;; Reader/Writers:
(defmacro hanging-hds (node)        `(send ,node :hanging-hds))
(defmacro other? (node)		    `(send ,node :other?))

;;; KEY-NODE Macros -----------------------------------------------------------

;;; Reader/Writers:
(defmacro key-val (node)            `(send ,node :key-val))

;;; BST-NODE Macros -----------------------------------------------------------

;;; Methods:
(defmacro expand (node not-others?) `(send ,node :expand ,not-others?))
(defmacro expand-limited (node comp do-node)
  `(send ,node :expand-limited ,comp ,do-node))

;;; Reader/Writers:
(defmacro left (node)               `(send ,node :left))
(defmacro right (node)              `(send ,node :right))
(defmacro set-left (node val)       `(send ,node :set-left ,val))
(defmacro set-right (node val)      `(send ,node :set-right ,val))

;;; INSTANTIATION FUNCTIONS ===================================================

;;; HDS-OBJECTS
;;;
(defmacro make-one-item ()
  '(make-instance 'one-item-hds))
(defmacro make-bag ()
  '(make-instance 'bag-hds))
(defmacro make-ht (key-func &key (size 128) (equal-func '#'eq))
  `(make-instance 'ht-hds :key-func ,key-func :table-size ,size
     :equal-func ,equal-func))
(defmacro make-xht (key-func &key (size 128) (equal-func '#'eq)
			      (threshold 0.8) (rehash-factor 2.1))
  `(make-instance 'xht-hds :key-func ,key-func :equal-func ,equal-func
     :table-size ,size :threshold ,threshold :rehash-factor ,rehash-factor))
(defmacro make-bst (key-func &optional (compare-func '#'-))
  `(make-instance 'bst-hds :key-func ,key-func :compare-func ,compare-func))

;;; NODE-OBJECTS
;;;
(defmacro make-hds-node (hanging-hds other?)
  `(make-instance 'hds-node :hanging-hds ,hanging-hds :other? ,other?))
(defmacro make-key-node (key-val hanging-hds other?)
  `(make-instance 'key-node :key-val ,key-val :hanging-hds ,hanging-hds
     :other? ,other?))
(defmacro make-bst-node (key-val hanging-hds other?)
  `(make-instance 'bst-node :key-val ,key-val :hanging-hds ,hanging-hds
     :other? ,other?))

;;; NODE-OBJECT-METHODS =======================================================

;;; COPY (HDS-NODE) -----------------------------------------------------------
;;;
(defmethod (hds-node :copy) ()
  (declare (type boolean other?))
  (make-hds-node (copy hanging-hds) other?))

;;; COPY (KEY-NODE) -----------------------------------------------------------
;;;
(defmethod (key-node :copy) ()
  (declare (type boolean other?))
  (make-key-node key-val (copy hanging-hds) other?))

;;; COPY (BST-NODE) -----------------------------------------------------------
;;;
(defmethod (bst-node :copy) ()
  (declare (type boolean other?))
  (let ((result (make-bst-node key-val (copy hanging-hds) other?)))
    (when left (set-left result (copy left)))
    (when right (set-right result (copy right)))
    result))

;;; EXPAND (BST-NODE) ---------------------------------------------------------
;;;
(defmethod (bst-node :expand) (not-others?)
  (declare (type boolean other?))
  (let ((result NIL))
    (declare (type list result))
    (when right (push right result))
    (when (or not-others? other?) (push hanging-hds result))
    (when left (push left result))
    result))

;;; EXPAND-LIMITED (BST-NODE) -------------------------------------------------
;;;
;;; Expand left child iff COMP >= 0, right child iff COMP <= 0, and the
;;; hanging-hds of the node itself if COMP = 0 *or* DO-NODE? == T.
;;;
(defmethod (bst-node :expand-limited) (comp do-node?)
  (declare (type fixnum comp)
	   (type boolean do-node?))
  (let ((result NIL))
    (declare (type list result))
    (when (and right (<= comp 0)) (push right result))
    (when (or (zerop comp) do-node?) (push hanging-hds result))
    (when (and left (>= comp 0)) (push left result))
    result))

;;; LEAF-HDS METHODS ==========================================================

;;; COPY (LEAF-HDS) -----------------------------------------------------------
;;;
(defmethod (leaf-hds :copy) ()
  (make-instance (type-of self)))

;;; GET-ITEM (LEAF-HDS) -------------------------------------------------------
;;;
(defmethod (leaf-hds :get-item) (path)
  (declare (type list path))
  (if path
      (error "GET-ITEM: cannot search through a leaf-hds")
      storage))

;;; ONE-ITEM-HDS METHODS ======================================================

;;; ADD-ITEM (ONE-ITEM-HDS) ---------------------------------------------------
;;;
(defmethod (one-item-hds :add-item) (item)
  (setq storage item))

;;; MAKE-ITER-ON (ONE-ITEM-HDS) -----------------------------------------------
;;;
(defmethod (one-item-hds :make-iter-on) (path)
  (declare (type list path))
  (if path *empty-iterator* (make-one-item-it storage)))

;;; BAG-HDS-METHODS ===========================================================

;;; ADD-ITEM (BAG-HDS) --------------------------------------------------------
;;;
(defmethod (bag-hds :add-item) (item)
  (declare (type list storage))
  (push item storage))

;;; MAKE-ITER-ON (BAG-HDS) ----------------------------------------------------
;;;
(defmethod (bag-hds :make-iter-on) (path)
  (declare (type list path storage))
  (if path *empty-iterator* (make-bag-it storage)))

;;; NON-LEAF-HDS METHODS ======================================================

;;; GET-ITEM (NON-LEAF-HDS) ---------------------------------------------------
;;;
(defmethod (non-leaf-hds :get-item) (path)
  (declare (type list path))
  (if path
      (let ((node (find-key-node self (car path))))
	(if node
	    (get-item (hanging-hds node) (cdr path))
	    (error "GET-ITEM: key ~A not found in non-leaf-hds ~A."
		   (car path) self)))
      (error "GET-ITEM: cannot get an item from a non-leaf-hds.")))

;;; EXTEND (NON-LEAF-HDS) -----------------------------------------------------
;;;
(defmethod (non-leaf-hds :extend) (path new-hds)
  (declare (type list path)
	   (type boolean new-slot?))
  (let* ((key (car path))
	 (others (eq key ':OTHERS))
	 (new-slot (eq key ':NEW-SLOT)))
    (declare (type boolean others new-slot))
    (if (or others new-slot)
      (if (and default-hds (not (eq new-slot? new-slot)))
	(error "EXTEND: key ~A inconsistent with new-slot? value ~A."
	       key new-slot?)
	(progn
	  (when (null default-hds) (setq new-slot? new-slot))
	  (extend-thru-default self (cdr path) new-hds)))
      (extend-thru-structure self key (cdr path) new-hds))))

;;; EXTEND-THRU-DEFAULT (NON-LEAF-HDS) ----------------------------------------
;;;
(defmethod (non-leaf-hds :extend-thru-default) (path-tail new-hds)
  (declare (type list path-tail))
  "Extend the :default-hds of SELF according to PATH-TAIL and NEW-HDS. If
PATH-TAIL is non-nil, attempt to recursively extend from the default-hds
(which should exist) by adding NEW-HDS. Otherwise, install NEW-HDS as the
default-hds (which should not exist). Errors are signalled if the default-hds
exists when it shouldn't, or vice-versa."
  (if path-tail
      ;; a data structure should exist to recursively extend
      (if default-hds
	(extend-thru default-hds path-tail new-hds)
        (error "EXTEND: no default HDS to traverse."))
      ;; no data structure should exist; we need to create a new one
      (if default-hds
	(error "EXTEND: attempt to extend over an existing default HDS.")
	(setq default-hds new-hds))))

;;; HASH TABLE COMMON FUNCTIONS/MACROS ========================================

;;; MAKE-TABLE-ARRAY [Macro] --------------------------------------------------
;;;
;;; Returns a 1-D array of size 'size, each element of which is a list (all
;;; initialized to NIL).
;;;
(defmacro make-table-array (size &optional (type 'list))
  (declare (type fixnum size))
  `(make-array (list ,size) :element-type ',type :initial-element NIL))

;;; COPY-HT [Function] --------------------------------------------------------
;;;
;;; Returns a copy of the hash table 'ht', which should be an array as created
;;; by MAKE-TABLE-ARRAY above. Each array entry is an a-list of key/node
;;; pairs; the nodes are copied recursively
;;;
(defun copy-ht (ht)
  (declare (type list-vector ht))
  (let* ((table-size (array-dimension ht 0))
	 (result (make-table-array table-size))
	 a-list)
    (declare (type fixnum table-size)
	     (type list-vector result)
	     (type list a-list))
    (dotimes (i table-size result)
      (declare (type fixnum i))
      (when (setq a-list (svref ht i))
	(dolist (pair a-list result)
	  (declare (type cons pair))
	  (setf (svref result i)
		(acons (car pair) (copy (cdr pair)) (svref result i))))))))

;;; COPY-XHT [Function] -------------------------------------------------------
;;;
;;; Returns a copy of the hash table 'ht', which should be an array as created
;;; by MAKE-TABLE-ARRAY above. Each array entry is a node; it is copied
;;; recursively.
;;;
(defun copy-xht (xht)
  (declare (type vector xht))
  (let* ((table-size (array-dimension xht 0))
	 (result (make-table-array table-size xht-entry))
	 node)
    (declare (type fixnum table-size)
	     (type vector result))
    (dotimes (i table-size result)
      (declare (type fixnum i))
      (when (setq node (svref xht i))
	(setf (svref result i) (copy node))))))

;;; HT-HDS METHODS ============================================================

;;; INIT (HT-HDS :AFTER Method) -----------------------------------------------
;;;
(defmethod (ht-hds :after :init) (init-plist)
  (declare (ignore init-plist)
	   (type fixnum table-size))
  (setq storage (make-table-array table-size)))

;;; MAKE-NEW-NODE (HT-HDS) ----------------------------------------------------
;;;
(defmethod (ht-hds :make-new-node) (key hds other?)
  (declare (ignore key)
	   (type boolean other?))
  (make-hds-node hds other?))

;;; REORGANIZE-P (HT-HDS) -----------------------------------------------------
;;;
(defmethod (ht-hds :reorganize-p) () NIL)

;;; NEW-HT-IT (HT-HDS) --------------------------------------------------------
;;;
(defmethod (ht-hds :new-ht-it) (others? rest)
  (declare (type fixnum table-size)
	   (type list-vector storage)
	   (type boolean others?)
	   (type list rest))
  (make-ht-it table-size storage others? rest))

;;; ADD-KEY-NODE (HT-HDS) -----------------------------------------------------
;;;
(defmethod (ht-hds :add-key-node) (index key new-node)
  (setf (svref storage index) (acons key new-node (svref storage index))))

;;; FIND-KEY-NODE (HT-HDS) ----------------------------------------------------
;;;
;;; Return the hds-node corresponding to KEY in the HT-HDS, or, if there is no
;;; hds-node in the hash table, return the pair of values (NIL index), where
;;; INDEX is the index in the hash table array to put a new hds-node for KEY.
;;;
(defmethod (ht-hds :find-key-node) (key)
  (declare (type fixnum table-size)
	   (type function equal-func))
  (let* ((index (mod (sxhash key) table-size))
	 (a-list (svref storage index))
	 (pair (assoc key a-list :test equal-func)))
    (declare (type fixnum index)
	     (type list a-list pair))
    (if pair (cdr pair) (values NIL index))))

;;; COPY (HT-HDS) -------------------------------------------------------------
;;;
(defmethod (ht-hds :copy) ()
  (declare (type boolean new-slot?))
  (let ((result (make-ht key-func :size table-size)))
    ;; recursively copy the default-hds if it exists
    (when default-hds
      (set-new-slot? result new-slot?)
      (set-default-hds result (copy default-hds)))
    ;; recursively copy the uninitialized hash-table instance-vars
    (set-storage result (copy-ht storage))
    result))

;;; ADD-ITEM (HT-HDS, XHT-HDS) ------------------------------------------------
;;;
(defmethod (ht-hds :add-item) (item)
  (declare (type function key-func)
	   (type boolean new-slot?))
  (let ((key (funcall key-func item)))
    (multiple-value-bind (node index) (find-key-node self key)
      (if node
	;; key match found; recursively add ITEM to the 'hanging-hds @ NODE
	(add-item (hanging-hds node) item)
	;; no node found; install the 'default-ds if one exists
	(when default-hds
	  (if new-slot?
	    ;; rehash if necessary, then copy the default-hds and make new slot
	    (if (reorganize-p self)
	      (add-item self item)
	      ;; create a new node and install it
	      (let ((new-node (make-new-node self key (copy default-hds))))
		(add-key-node self index key new-node)
		(add-item (hanging-hds new-node) item)))
	    ;; otherwise, simply install item in default-hds
	    (add-item default-hds item)))))))

;;; MAKE-ITER-ON (HT-HDS, XHT-HDS) --------------------------------------------
;;;
(defmethod (ht-hds :make-iter-on) (path)
  (declare (type list path)
	   (type boolean new-slot?))
  (let ((others? (eq ':OTHERS (car path))))
    (declare (type boolean others?))
    (if (or (null path) (eq ':ALL (car path)) (and new-slot? others?))
    ;; create a new iterator
    (new-ht-it self others? (cdr path))
    ;; otherwise, attempt to recursively find the iterator
    (if others?
      ;; recursively iterate over default-hds; new-slot? must be NIL
      (if (null default-hds)
	(error "MAKE-ITER-ON: :OTHERS key hit NIL default-hds.")
	(make-iter-on default-hds (cdr path)))
      ;; otherwise, find proper slot and iterate over it
      (let ((node (find-key-node self (car path))))
	(if node
	  ;; slot found, iterate over it
	  (make-iter-on (hanging-hds node) (cdr path))
	  ;; no slot found, give empty iterator
	  *empty-iterator*))))))

;;; EXTEND-THRU-STRUCTURE (HT-HDS, XHT-HDS) -----------------------------------
;;;
(defmethod (ht-hds :extend-thru-structure) (key path-tail new-hds)
  (declare (type list path-tail))
"Find the node with key KEY (which should *not* be :OTHERS), and extend at
that key according to PATH-TAIL and NEW-HDS. If PATH-TAIL is non-nil, attempt
to recursively extend from the node (which should have been found). Otherwise,
no node should have been found; create a new node and install NEW-HDS in that
node. Errors are signalled if a node is found when it shouldn't be, or
vice-versa."
  (multiple-value-bind (node index) (find-key-node self key)
    (if path-tail
	;; a node should have been found to recursively extend
	(if node
	    (extend-thru (hanging-hds node) path-tail new-hds)
	    (error "EXTEND: no matching key-node found to traverse."))
	;; a node should not have been found; we need to create a new one
	(if node
	    (error "EXTEND: attempt to extend over an existing HDS.")
	    (if (reorganize-p self)
		(extend-thru-structure self key path-tail new-hds)
		;; create a new node and install it
		(let ((new-node (make-new-node self key new-hds NIL)))
		  ;; install the new node in the hash table
		  (add-key-node self index key new-node)))))))

;;; XHT-HDS METHODS ===========================================================

;;; INIT (XHT-HDS :BEFORE Method) ---------------------------------------------
;;;
(defmethod (xht-hds :before :init) (init-plist)
  (declare (ignore init-plist)
	   (type fixnum table-size max-entries))
  ;; convert a float threshold to an integer
  (when (floatp threshold)
    (setq threshold (ceiling (* threshold table-size))))
  ;; make sure 0 < threshold <= table-size
  (cond
    ((<= threshold 0) (setq threshold 1))
    ((> threshold table-size) (setq threshold table-size)))
  ;; create an extra bucket in the hash table
  (setq max-entries table-size)
  (incf table-size))

;;; INIT (XHT-HDS :AFTER Method) ----------------------------------------------
;;;
(defmethod (xht-hds :after :init) (init-plist)
  (declare (ignore init-plist))
  (setq storage (make-table-array table-size xht-entry)))

;;; MAKE-NEW-NODE (XHT-HDS) ---------------------------------------------------
;;;
(defmethod (xht-hds :make-new-node) (key hds other?)
  (declare (type boolean other?))
  (make-key-node key hds other?))

;;; REORGANIZE-P (XHT-HDS) ----------------------------------------------------
;;;
;;; Return T and reorganize if the current number of entries = the threshold.
;;; Otherwise, increment the number of entries and return NIL.
;;;
(defmethod (xht-hds :reorganize-p) ()
  (declare (type fixnum entries threshold))
  (if (> (incf entries) threshold)
      (progn (decf entries) (rehash self) T)
      NIL))

;;; COPY (XHT-HDS) ------------------------------------------------------------
;;;
(defmethod (xht-hds :copy) ()
  (declare (type function key-func equal-func)
	   (type fixnum max-entries theshold)
	   (type float rehash-factor)
	   (type boolean new-slot?))
  (let ((result (make-xht key-func
		   :equal-func equal-func :size max-entries
		   :threshold threshold :rehash-factor rehash-factor)))
    ;;
    ;; recursively copy the default-hds if it exists
    (when default-hds
      (set-new-slot? result new-slot?)
      (set-default-hds result (copy default-hds)))
    ;;
    ;; recursively copy the uninitialized hash-table instance-vars
    (set-entries result entries)
    (set-storage result (copy-xht storage))
    result))

;;; NEW-HT-IT (XHT-HDS) -------------------------------------------------------
;;;
(defmethod (xht-hds :new-ht-it) (others? rest)
  (declare (type boolean others?)
	   (type list rest))
  (make-xht-it table-size storage others? rest))

;;; FIND-KEY-NODE (XHT-HDS) ---------------------------------------------------
;;;
;;; Search for an item in the hash table using linear hashing. Returns the
;;; key-node corresponding to KEY if KEY is in the table; otherwise return two
;;; values: NIL and the index of the empty slot in the hash table array to put
;;; the new key.
;;;
(defmethod (xht-hds :find-key-node) (key)
  (declare (type fixnum table-size)
	   (type xht-array storage)
	   (type function equal-func))
  (let ((i (mod (sxhash key) table-size)))
    (declare (type fixnum i))
    (do ((item (svref storage i) (svref storage i)))
	((null item) (values NIL i))
      (when (funcall equal-func key (key-val item)) (return item))
      (incf i)
      (when (eq i table-size) (setq i 0)))))

;;; ADD-KEY-NODE (XHT-HDS) ----------------------------------------------------
;;;
(defmethod (xht-hds :add-key-node) (index key new-node)
  (declare (ignore key)
	   (type fixnum index)
	   (type xht-array storage))
  (setf (svref storage index) new-node))

;;; REHASH (XHT-HDS) ----------------------------------------------------------
;;;
(defmethod (xht-hds :rehash) ()
  (declare (type fixnum max-entries table-size theshold)
	   (type float rehash-factor)
	   (type xht-array storage))
  (let ((orig-table-size table-size)
	(orig-storage storage)
	item)
    (declare (type fixnum orig-table-size)
	     (type xht-array orig-storage))
    ;;
    ;; update instance-vars for new hash-table
    (setf max-entries (ceiling (* max-entries rehash-factor)))
    (setf table-size (1+ max-entries))
    (setf threshold (ceiling (* threshold rehash-factor)))
    (setf storage (make-table-array table-size xht-entry))
    ;;
    ;; add each of the items to the new table
    (dotimes (old-i orig-table-size)
      (declare (type fixnum old-i))
      (when (setq item (svref orig-storage old-i))
	(multiple-value-bind (node new-i) (find-key-node self (key-val item))
	  (assert (null node))
	  (setf (svref storage new-i) item))))))

;;; BST-HDS COMMON FUNCTIONS/MACROS ===========================================

;;; STORE-NEW-NODE-IN-NODE [Macro] --------------------------------------------
;;;
(defmacro store-new-node-in-node (node side new-node)
  (declare (type symbol side))
"Store NEW-NODE in the SIDE (either 'left or 'right) slot of NODE (which
must be non-NIL)."
  `(if (eq ,side 'left)
       (set-left ,node ,new-node)
       (set-right ,node ,new-node)))

;;; STORE-NEW-NODE-IN-ROOT-OR-NODE [Macro] ------------------------------------
;;;
(defmacro store-new-node-in-root-or-node (hds node side new-node)
  (declare (type symbol side))
"If NODE is non-nil, store NEW-NODE in the SIDE (either 'left or 'right)
slot of NODE. If NODE is NIL, store NEW-NODE in the storage slot of HDS."
  `(let ((.node. ,node))
      (if .node.
         (store-new-node-in-node .node. ,side ,new-node) ;not at root
         (set-storage ,hds ,new-node))))                 ;install at root

;;; BST-HDS METHODS ===========================================================

;;; ADD-ITEM (BST-HDS) --------------------------------------------------------
;;;
(defmethod (bst-hds :add-item) (item)
  (declare (type boolean new-slot?)
	   (type function key-func))
  (let ((key (funcall key-func item)))
    (multiple-value-bind (node parent side) (find-key-node self key)
      (if node
	;; key match found; recursively add ITEM to the 'hanging-hds @ NODE
	(add-item (hanging-hds node) item)
	;; no node found; install the 'default-ds if one exists
	(when default-hds
	  (if new-slot?
	    ;; create a new node and install it
	    (let ((new-node (make-new-node self key (copy default-hds))))
	      (store-new-node-in-root-or-node self parent side new-node)
	      ;; recursively add the item to the 'hanging-hds of the new node
	      (add-item (hanging-hds new-node) item))
	    ;; otherwise, simply insert item into default-hds
	    (add-item default-hds item)))))))

;;; COPY (BST-HDS) ------------------------------------------------------------
;;;
(defmethod (bst-hds :copy) ()
  (declare (type boolean new-slot?)
	   (type function key-func compare-func))
  (let ((result (make-bst key-func compare-func)))
    ;; recursively copy the default-hds if it exists
    (when default-hds
      (set-new-slot? result new-slot?)
      (set-default-hds result (copy default-hds)))
    ;; recursively copy storage if it exists
    (when storage (set-storage result (copy storage)))
    result))

;;; MAKE-ITER-ON (BST-HDS) ----------------------------------------------------
;;;
(defmethod (bst-hds :make-iter-on) (path)
  (declare (type list path)
	   (type boolean new-slot?)
	   (type function compare-func))
  (let ((others? (eq ':OTHERS (car path))))
    (declare (type boolean others?))
    (cond
      ((or (null path) (eq ':ALL (car path)) (and new-slot? others?))
       ;; create a new (full) iterator
       (make-bst-it (if storage `((,storage)) NIL) others? (cdr path)))
    ((typep (car path) 'iter::bst-range)
     ;; create a new (range) iterator
     (make-bst-range-it
      (if storage `((,storage)) NIL) compare-func (car path) (cdr path)))
    (others?
     ;; recursively iterate over default-hds; new-slot? must be NIL
     (if (null default-hds)
       (error "MAKE-ITER-ON: :OTHERS key hit NIL default-hds.")
       (make-iter-on default-hds (cdr path))))
    (t
     ;; otherwise, continue down search path recursively
     (let ((node (find-key-node self (car path))))
       (if node
	 ;; slot found, iterate over it
	 (make-iter-on (hanging-hds node) (cdr path))
	 ;; no slot found, give empty iterator
	 *empty-iterator*))))))

;;; MAKE-NEW-NODE (BST-HDS) ---------------------------------------------------
;;;
(defmethod (bst-hds :make-new-node) (key hds other?)
  (declare (type boolean other?))
  (make-bst-node key hds other?))

;;; EXTEND-THRU-STRUCTURE (BST-HDS) -------------------------------------------
;;;
(defmethod (bst-hds :extend-thru-structure) (key path-tail new-hds)
  (declare (type list path-tail))
"Find the node with key KEY (which should *not* be :OTHERS), and extend at that
key according to PATH-TAIL and NEW-HDS. If PATH-TAIL is non-nil, attempt to
recursively extend from the node (which should have been found). Otherwise,
no node should have been found; create a new node and install NEW-HDS in that
node. Errors are signalled if a node is found when it shouldn't be, or
vice-versa."
  (multiple-value-bind (node parent side) (find-key-node self key)
    (declare (type symbol side))
    (if path-tail
	;; a node should have been found to recursively extend
	(if node
	    (extend-thru (hanging-hds node) path-tail new-hds)
	    (error "EXTEND: no matching key-node found to traverse."))
	;; a node should not have been found; we need to create a new one
	(if node
	    (error "EXTEND: attempt to extend over an existing HDS.")
	    ;; create and install the new node
	    (let ((new-node (make-new-node self key new-hds NIL)))
	      (store-new-node-in-root-or-node self parent side new-node))))))

;;; FIND-KEY-NODE (BST-HDS) ---------------------------------------------------
;;;
(defmethod (bst-hds :find-key-node) (key)
  (declare (type function compare-func))
  (let ((node storage)
	(old-node nil)
	(comp-result 0))
;   (declare (type bst-node node))
;   (declare (type bst-node old-node))
    (declare (type fixnum comp-result))
    (do () ((null node))
      (setq comp-result (funcall compare-func key (key-val node)))
      (setq old-node node)
      (cond
	((minusp comp-result) (setq node (left node)))
	((plusp comp-result) (setq node (right node)))
	(t (return))))
    (values node old-node (comp-to-side comp-result))))

;;; COMP-TO-SIDE
;;;
(defun comp-to-side (n)
  (declare (type fixnum n))
  "Return 'left, 'right, or nil as n is <0, >0, or =0, respectively."
  (cond ((minusp n) 'left) ((plusp n) 'right) (t nil)))

(compile-flavor-methods one-item-hds bag-hds ht-hds xht-hds bst-hds)
(compile-flavor-methods hds-node key-node bst-node)
