;;; ITER.CL - HDS Iterators
;;;
;;; $Header: iter.cl,v 1.9 91/10/30 23:45:55 heydon Exp $
;;;
;;; Written by Allan Heydon for the Miro project at Carnegie Mellon
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

(provide 'iter)
(in-package 'iter)

(export
 '(make-range				;create a bst-range for an iteration
   make-po-range			;create a bst-po-range for an iteration
   make-one-item-it			;make iterator for one-item-hds
   make-bag-it				;make iterator for bag-hds
   make-bst-it				;make iterator for bst-hds
   make-bst-range-it			;make range iterator for bst-hds
   make-ht-it				;make iterator for ht-hds
   make-xht-it				;make iterator for xht-hds
   next-item				;get next item from any iterator
   make-union-it			;form union of several iterators
   make-chaining-it			;create iterator chaining off another
   make-sem-iter-on			;iterator for semantics arrows
   *empty-iterator*))			;constant empty iterator

(require 'flavors)
(require 'objs)
(require 'hds)
(use-package 'flavors)
(use-package 'objs)
(use-package 'hds)

;;; ITER TYPES OVERVIEW =======================================================

;;; Here is a description of the ITERATOR type hierarchy:
;;;
;;; BST-RANGE                       ;top-level range for BST-RANGE-IT
;;;   BST-PO-RANGE                  ;range for Partial Orders
;;; ITERATOR                        ;top-level iterator
;;;   LEAF-IT                       ;iterator for LEAF-HDS objects
;;;     ONE-ITEM-IT                 ;iterator for ONE-ITEM-HDS objects
;;;     BAG-IT                      ;iterator for BAG-HDS objects
;;;   NON-LEAF-IT                   ;iterator for NON-LEAF-HDS objects
;;;     BST-RANGE-IT                ;iterator over BST-HDS ranges
;;;     NON-LEAF-OTHERS-IT          ;class for possible iteration over OTHERS
;;;       BST-IT                    ;iterator for entire BST-HDS objects
;;;       XHT-IT                    ;iterator for XHT-HDS objects
;;;         HT-IT                   ;iterator for HT-HDS objects
;;;   UNION-IT                      ;iterator for unions of other iterators
;;;   CHAINING-IT                   ;iterator for closures over other iter's
;;;   SEM-IT                        ;iterator for semantic arrows

;;; ITERATOR OVERVIEW =========================================================
;;;
;;; This module defines the ITERATOR abstract data type. It works hand in hand
;;; with the HDS module, which defines hierarchical data structures. The
;;; MAKE-ITER-ON operation defined in that module returns an iterator. This
;;; object provides the primary operation NEXT-ITEM, which returns the next
;;; item in the HDS according to the PATH specified in the MAKE-ITER-ON call.
;;;
;;; Functions are also defined for "composing" iterators: that is, transfoming
;;; an existing iterator or set of iterators into a new iterator whose
;;; function depends on the iterator(s) it was created from. The module
;;; currently supports "union" iterators and "chaining" iterators. See the
;;; documentation below for details.

;;; OPERATIONS
;;;
;;; Creation:
;;;
;;; (MAKE-RANGE range)
;;;   Creates and returns a new BST-RANGE. RANGE should be a cons of the form
;;;   (low-spec . high-spec), where both specs may be NIL. When non-NIL, each
;;;   spec is of the form (val . eq?), where 'val' is a value, and 'eq?' is
;;;   either T or NIL, indicating if equality is included or not. So 
;;;   for example, a range of ((5.t).NIL) would denote the range [5,infinity],
;;;   while ((3.T).(7.NIL)) would denote the range [3,7).
;;;
;;; (MAKE-PO-RANGE range)
;;;   Like MAKE-RANGE above, only the comparison function for this range is
;;;   assumed to implement only a partial order.
;;;
;;; (MAKE-ONE-ITEM-IT start-state)
;;;   Create and return a new one-item-it with initial state START-STATE,
;;;   which should be the single item to iterate over.
;;;
;;; (MAKE-BAG-IT start-state)
;;;   Create and return a new bag-it with initial state START-STATE, which
;;;   should be a list of items to iterate over.
;;;
;;; (MAKE-BST-IT start-state others? [path])
;;;   Create and return a new BST-IT iterator, i.e., an iterator for a BST-HDS.
;;;   START-STATE should be either NIL (if the BST-HDS is empty), or a list
;;;   containing the root BST-NODE of the BST-HDS to search. OTHERS? should be
;;;   T iff iteration should only include nodes with an 'other?'==T. PATH
;;;   should be the remainder of the path to iterate over; it defaults to NIL.
;;;
;;; (MAKE-BST-RANGE-IT start-state compare-func range [path])
;;;   Create and return a new BST-RANGE-IT itererator, i.e., an iterator for a
;;;   BST-HDS that iterates only over those items in the range RANGE. If RANGE
;;;   is a BST-PO-RANGE, then COMPARE-FUNC is assumed to implement a partial
;;;   order. It does this by returning 2 values. The first is -1, 0, or 1 as
;;;   its first argument is less than, equal to, or greater than its second
;;;   argument according to some arbitrary extension of the partial order to a
;;;   total order. The second value is T or NIL depending on whether the first
;;;   result is due to the partial order, or is due to the arbitrary extension.
;;;   START-STATE should be either NIL (if the BST-HDS is empty), or a list
;;;   containing the root BST-NODE of the BST-HDS to search. COMPARE-FUNC
;;;   should be the same comparison function as that of the BST-HDS. PATH
;;;   should be the remainder of the path to iterate over; it defaults to NIL.
;;;
;;; (MAKE-HT-IT size table others? [path])
;;;   Create and return a new HT-IT iterator, i.e., an iterator for a HT-HDS.
;;;   SIZE should be the size of the hash table, and TABLE should be the table
;;;   itself (an array of HDS:KEY-NODE's). OTHERS? is T or NIL, indicating if
;;;   only those nodes with an 'other?' field of T should be considered. PATH
;;;   should be the remainder of the path to iterate over; it defaults to NIL.
;;;
;;; (MAKE-XHT-IT size table others? [path])
;;;   Create and return a new XHT-IT iterator, i.e., an iterator for a XHT-HDS.
;;;   The other arguments are the same as for MAKE-HT-IT.
;;;
;;; Iteration:
;;;
;;; (NEXT-ITEM iterator)
;;;   Return the next item from ITERATOR, or NIL if there are no more items to
;;;   iterate on.

;;; Composition:
;;;
;;; (MAKE-UNION-IT it1 it2 ...)
;;;   Create and return a new iterator that returns the union of items
;;;   returned by each of the argument iterators in succession. This is not a
;;;   true set-union, as the same item may be returned more than once (if it
;;;   appears in more than one argument iterator).
;;;
;;; (MAKE-CHAINING-IT hds path [iterator])
;;;   Create and return a new iterator that "chains" off ITERATOR through HDS
;;;   according to PATH. ITERATOR defaults to the iterator over all items of
;;;   HDS (i.e. (MAKE-ITER-ON HDS)). The items returned by NEXT-ITEM on this
;;;   iterator are as follows. For each ITEM in ITERATOR, we first return
;;;   ITEM, and then recursively iterate over a new iterator that is created
;;;   from HDS and PATH, with all elements of the form '(function func) in
;;;   PATH replaced by the result of applying FUNC to ITEM.
;;;
;;;   For example, if called as (MAKE-CHAINING-IT top '(2 #'second #'car) it),
;;;   and if the last ITEM returned was '(foo bar baz), then the next iterator
;;;   would be created with (MAKE-ITER-ON top '(2 bar foo)).
;;;
;;;   Chaining thus returns items based on a pre-order depth-first-search,
;;;   where the children of a node to search are generated by calling
;;;   NEXT-ITEM on the iterator which is created as described in the previous
;;;   paragraph.
;;;
;;; (MAKE-SEM-ITER-ON parity-bit subj-hds subj-path obj-hds obj-path)
;;;   Forms an iterator for semantic arrows that implements a nested
;;;   iteration. It iterates first over the object iterator created from
;;;   (make-iter-on OBJ-HDS OBJ-PATH), and then over the subject iterator
;;;   (make-iter-on SUBJ-HDS SUBJ-PATH) for arrows having parity given by
;;;   PARITY-BIT (1 for positive, 0 for negative). For each, it fills in and
;;;   returns an appropriate SEM structure.

;;; CONSTANTS
;;;
;;; *EMPTY-ITERATOR*
;;;   Constant evaluating to an iterator guaranteed not to contain any item.
;;;   That is, this iterator returns NIL on any NEXT-ITEM calls to it.

;;; ITERATOR IMPLEMENTATION OVERVIEW ==========================================
;;;
;;; ONE-ITEM-IT
;;;   The STATE contains the single item to iterate over.
;;;
;;; BAG-IT
;;;   The STATE constains a list of the items to iterate over.
;;;
;;; BST-IT
;;;   The STATE is a list of the form: ([iterator] a_1 a_2 ... a_n), where
;;;   each a_i is a list of the form: ([left-bst-node] [hds] [right-bst-node]).
;;;   When NEXT-ITEM finds an 'iterator' at the front of the list, it simply
;;;   iterates on that iterator. If the iterator is empty, it pops the
;;;   iterator and examines a_1. When a_1 is examined, the car is considered.
;;;   If it is a BST-NODE, the node is expanded and pushed onto the overal
;;;   list. If the car is an HDS, then an iterator is created for that HDS and
;;;   pushed onto the overall list as well. The process repeats until an
;;;   iterator is found at the car of the overall list.
;;;
;;; BST-RANGE-IT
;;;   Same as for a BST-IT, only now, a BST-NODE is expanded according to the
;;;   range of the BST-RANGE-IT.
;;;
;;; HT-IT
;;;   The STATE is an integer representing an array index into the hash table.
;;;   The integer starts out at the array size, and is decremented until it
;;;   reaches 0. As it is decremented, each array entry is searched for an
;;;   existing HDS. If one is found, an iterator is created for that HDS and
;;;   stored in the CURR-IT field of the HT-IT.
;;;
;;; SEM-IT
;;;   The STATE is the iterator over the objects.

;;; ITERATOR-OBJECT TYPES =====================================================

;;; BST-RANGE -----------------------------------------------------------------
;;;
(defflavor bst-range
  ((low NIL :read-only t)		;low value (or NIL)
   (high NIL :read-only t)		;high value (or NIL)
   (low? NIL :read-only t)		;low value strict inequality?
   (high? NIL :read-only t))		;high value strict inequality?
  ()					;base class
  (:documentation "Range for iteration in a BST-HDS.")
  (:initable-instance-variables low high low? high?)
  (:required-init-keywords :low :high :low? :high?)
  (:required-methods :in-range?))

;;; BST-PO-RANGE --------------------------------------------------------------
;;;
(defflavor bst-po-range			;range for Partial Orders
    ()					;inherits all data from parent
  (bst-range))				;parent

;;; ITERATOR ------------------------------------------------------------------
;;;
(defflavor iterator
  (state)				;current state of iteration
  ()					;base class
  (:documentation "Basic class for HDS Iterators.")
  (:initable-instance-variables state)
  (:gettable-instance-variables state)
  (:settable-instance-variables state)
  (:required-init-keywords     :state)
  (:required-methods :next-item))

;;; LEAF-IT -------------------------------------------------------------------
;;;
(defflavor leaf-it ()			;no extra state
  (iterator)
  (:documentation "Iterator for leaf-hds objects."))

;;; ONE-ITEM-IT ---------------------------------------------------------------
;;;
(defflavor one-item-it ()		;no extra state
  (leaf-it)
  (:documentation "Iterator for one-item-hds objects."))

;;; BAG-IT --------------------------------------------------------------------
;;;
(defflavor bag-it ()			;no extra state
  (leaf-it)
  (:documentation "Iterator for bag-hds objects."))

;;; NON-LEAF-IT ---------------------------------------------------------------
;;;
(defflavor non-leaf-it
    (path)				;remainder path to iterate on
  (iterator)
  (:documentation "Iterator for non-leaf-hds objects.")
  (:initable-instance-variables path)
  (:required-init-keywords     :path))

;;; BST-RANGE-IT --------------------------------------------------------------
;;;
(defflavor bst-range-it
    (compare-func			;function used in node comparisons
     range)				;bst-range
  (non-leaf-it)
  (:documentation "Iterator over ranges of BST-HDS objects.")
  (:initable-instance-variables compare-func range)
  (:required-init-keywords :compare-func :range))

;;; NON-LEAF-OTHERS-IT --------------------------------------------------------
;;;
(defflavor non-leaf-others-it
    (not-others?)			;don't take nodes with 'other?'==NIL?
  (non-leaf-it)
  (:initable-instance-variables not-others?)
  (:required-init-keywords :not-others?))

;;; BST-IT --------------------------------------------------------------------
;;;
(defflavor bst-it ()			;no extra state
  (non-leaf-others-it)
  (:documentation "Iterator for BST-HDS objects."))

;;; XHT-IT --------------------------------------------------------------------
;;;
(defflavor xht-it
    ((curr-it NIL)			;current iterator
     table)				;hash table
  (non-leaf-others-it)
  (:documentation "Iterator for XHT-HDS objects.")
  (:initable-instance-variables table)
  (:required-init-keywords :table))

;;; HT-IT ---------------------------------------------------------------------
;;;
(defflavor ht-it
    ((bucket NIL))			;current bucket list
  (xht-it)
  (:documentation "Iterator for HT-HDS objects."))

;;; UNION-IT ------------------------------------------------------------------
;;;
(defflavor union-it
    ()					;no extra state
  (iterator)
  (:documentation "Iterator over a union of iterators."))

;;; CHAINING-IT ---------------------------------------------------------------
;;;
(defflavor chaining-it
    (hds				;HDS to recurse on
     path)				;PATH to use on HDS to build next iter
  (iterator)
  (:documentation "Chaining iterator.")
  (:initable-instance-variables hds path)
  (:required-init-keywords :hds :path))

;;; SEM-IT --------------------------------------------------------------------
;;;
(defflavor sem-it
    (parity-bit				;parity of sem arrow
     sem-arrow)				;semantic arrow to be filled in
  (iterator)
  (:documentation "Semantic Arrow Iterator.")
  (:initable-instance-variables parity-bit sem-arrow)
  (:required-init-keywords :parity-bit :sem-arrow))

(defflavor sem-both-it
    (subj-index)			;index for subject
  (sem-it)
  (:documentation "Semantic Arrow Iterator anchored at both ends.")
  (:initable-instance-variables subj-index)
  (:required-init-keywords :subj-index))

(defflavor sem-tail-it
    (obj-path				;path to use for iteration
     obj-it)				;iterator over objects
  (sem-it)
  (:documentation "Semantic Arrow Iterator anchored at tail.")
  (:initable-instance-variables obj-path obj-it)
  (:required-init-keywords :obj-path :obj-it))

(defflavor sem-head-it
    (subj-list)				;list of subjects for current block
  (sem-it)
  (:documentation "Semantic Arrow Iterator anchored at head.")
  (:initable-instance-variables subj-list)
  (:required-init-keywords :subj-list))

(defflavor sem-none-it
    ()					;no new fields
  (sem-head-it)				;just like a sem-head-it
  (:documentation "Semantic Arrow Iterator unanchored."))

;;; INSTANTIATION FUNCTIONS ===================================================
;;;
;;; RANGES --------------------------------------------------------------------
;;;
(defun make-some-range (range kind)
  (declare (type cons range))
  (let ((low-val (car range))
	(high-val (cdr range))
	low high low? high?)
    (declare (type fixnum low-val high-val low high)
	     (type boolean low? high?))
    (when low-val (setq low (car low-val) low? (not (cdr low-val))))
    (when high-val (setq high (car high-val) high? (not (cdr high-val))))
    (make-instance kind :low low :high high :low? low? :high? high?)))
(defun make-range (range)
  (declare (type cons range))
  (make-some-range range 'bst-range))
(defun make-po-range (range)
  (declare (type cons range))
  (make-some-range range 'bst-po-range))

;;; ITERATORS -----------------------------------------------------------------
;;;
(defun make-one-item-it (start-state)
  (make-instance 'one-item-it :state start-state))
(defun make-bag-it (start-state)
  (declare (type list start-state))
  (make-instance 'bag-it :state start-state))
(defun make-bst-it (start-state others? &optional (path NIL))
  (declare (type boolean others?)
	   (type list path))
  (make-instance 'bst-it :state start-state :not-others? (not others?)
		 :path path))
(defun make-bst-range-it (start-state compare-func range &optional (path NIL))
  (declare (type function compare-func)
	   (type cons range)
	   (type list path))
  (make-instance 'bst-range-it :state start-state :compare-func compare-func
		 :range range :path path))
(defun make-ht-it (size table others? &optional (path NIL))
  (declare (type fixnum size)
	   (type list-vector table)
	   (type list path)
	   (type boolean others?))
  (make-instance 'ht-it :state size :table table :not-others? (not others?)
		 :path path))
(defun make-xht-it (size table others? &optional (path NIL))
  (declare (type fixnum size)
	   (type xht-array table)
	   (type boolean others?)
	   (type list path))
  (make-instance 'xht-it :state size :table table :not-others? (not others?)
		 :path path))

;;; COMPOSITION OPERATORS -----------------------------------------------------
;;;
(defun make-union-it (&rest it-list)
  (declare (type list it-list))
  (make-instance 'union-it :state it-list))
(defun make-chaining-it (hds path &optional (it (make-iter-on hds)))
  (declare (type list path))
  (make-instance 'chaining-it :state (list it) :hds hds :path path))

;;; SEMANTIC ARROW ITERATORS --------------------------------------------------
;;;
(defmacro make-sem-both-it (parity-bit sem-arrow tail-index head-it)
  (declare (type bit parity-bit)
	   (type fixnum tail-index))
  `(make-instance 'sem-both-it
    :state ,head-it :parity-bit ,parity-bit :sem-arrow ,sem-arrow
    :subj-index ,tail-index))

(defun make-sem-tail-it (parity-bit sem-arrow tail-index obj-path)
  (declare (type bit parity-bit)
	   (type objs::sem sem-arrow)
	   (type fixnum tail-index)
	   (type list obj-path))
  (make-instance 'sem-tail-it
    :state (aref *col-array* tail-index parity-bit) :parity-bit parity-bit
    :sem-arrow sem-arrow :obj-path obj-path :obj-it nil))

(defmacro make-sem-head-it (parity-bit sem-arrow head-it)
  (declare (type bit parity-bit))
  `(make-instance 'sem-head-it
    :state ,head-it :parity-bit ,parity-bit :sem-arrow ,sem-arrow
    :subj-list nil))

(defmacro make-sem-none-it (parity-bit sem-arrow head-it)
  (declare (type bit parity-bit))
  `(make-instance 'sem-none-it
    :state ,head-it :parity-bit ,parity-bit :sem-arrow ,sem-arrow
    :subj-list nil))

(defun make-sem-iter-on (parity-bit subj-hds subj-path obj-hds obj-path)
  (declare (type bit parity-bit)
	   (type list subj-path obj-path))
  (let ((tail-anchored? (> (length subj-path) 0))
	(head-anchored? (> (length obj-path) 1))
	(sem-arrow (objs::make-sem :parity (eq 1 parity-bit)))
	tail-ix head-it)
    (when tail-anchored?
      (setf (sem-from sem-arrow) (car subj-path))
      (setq tail-ix (subj-index (car (get-item subj-hds subj-path)))))
    (when head-anchored? (setf (sem-to sem-arrow) (cadr obj-path)))
    (when (or head-anchored? (not tail-anchored?))
      (setq head-it (make-iter-on obj-hds obj-path)))
    (cond
      ((and tail-anchored? head-anchored?)
       (make-sem-both-it parity-bit sem-arrow tail-ix head-it))
      (tail-anchored?
       (make-sem-tail-it parity-bit sem-arrow tail-ix obj-path))
      (head-anchored?
       (make-sem-head-it parity-bit sem-arrow head-it))
      (t
       (make-sem-none-it parity-bit sem-arrow head-it)))))

;;; MACROS ====================================================================

;;; BST-RANGE method macros ---------------------------------------------------
;;;
(defmacro in-range? (r node comp-func)
  (declare (type function comp-func))
  `(send ,r :in-range? ,node ,comp-func))

;;; ITERATOR method macros ----------------------------------------------------
;;;
(defmacro next-item (it)            `(send ,it :next-item))

;;; ITERATOR reader/writer macros ---------------------------------------------
;;;
(defmacro state (it)                `(send ,it :state))
(defmacro set-state (it val)        `(send ,it :set-state ,val))
(defmacro path (it)                 `(send ,it :path))

;;; BST-RANGE METHODS =========================================================

;;; IN-RANGE? (BST-RANGE) -----------------------------------------------------
;;;
;;; Returns -1, 0, or 1 as the key of NODE is below, inside, or above the
;;; BST-RANGE according to COMP-FUNC.
;;;
(defmethod (bst-range :in-range?) (node comp-func)
  (declare (type function comp-func)
	   (type boolean low? high?))
  (let (comp				;temporary comparison result
	(key (hds::key-val node))	;key of NODE
	(below-low? NIL)		;is key below lower bound?
	(above-high? NIL))		;is key above upper bound?
    (declare (type fixnum comp)
	     (type boolean below-low? above-high?))
    (when low
      (setq comp (funcall comp-func low key))
      (setq below-low? (or (plusp comp) (and (zerop comp) low?))))
    (when high
      (setq comp (funcall comp-func key high))
      (setq above-high? (or (plusp comp) (and (zerop comp) high?))))
    (cond (below-low? -1) (above-high? 1) (t 0))))

;;; IN-RANGE? (BST-PO-RANGE) --------------------------------------------------
;;;
;;; Returns 2 values. The first is 1, -1, or 0 as the left child of NODE,
;;; right child of NODE, or both, respectively, should be expanded. The second
;;; is T or NIL as NODE itself should be visited or not. These values are
;;; determined based on the comparison function COMP-FUNC, which should
;;; implement a partial order.
;;;
(defmethod (bst-po-range :in-range?) (node comp-func)
  (declare (type function comp-func)
	   (type boolean low? high?))
  (let ((key (hds::key-val node))	;key of NODE
	(above-low? T)			;is key above lower bound?
	(below-high? T)			;is key below upper bound?
	l-comp l-valid h-comp h-valid)	;comparison results
    (declare (type boolean above-low? below-high?)
	     (type fixnum l-comp h-comp)
	     (type boolean l-valid h-valid))
    (when low
      (multiple-value-setq (l-comp l-valid) (funcall comp-func key low))
      (when (and (zerop l-comp) low?) (setq l-comp -1))
      (setq above-low? (and l-valid (>= l-comp 0))))
    (when high
      (multiple-value-setq (h-comp h-valid) (funcall comp-func key high))
      (when (and (zerop h-comp) high?) (setq h-comp 1))
      (setq below-high?	(and h-valid (<= h-comp 0))))
    ;; when low value, use it to decide which child to expand
    (when low (return (values l-comp (and above-low? below-high?))))
    ;; when high value (but no low value)...
    (when high (return (if below-high? (values 0 T) (values h-comp NIL))))
    ;; neither low nor high in range, so expand everything
    (values 0 T)))

;;; LEAF-IT METHODS ===========================================================

;;; NEXT-ITEM (ONE-ITEM-IT) ---------------------------------------------------
;;;
(defmethod (one-item-it :next-item) ()
  (let ((result state)) (setq state NIL) result))

;;; NEXT-ITEM (BAG-IT) --------------------------------------------------------
;;;
(defmethod (bag-it :next-item) ()
  (declare (type list state))
  (pop state))				;POP returns NIL if state==NIL

;;; BST-IT METHODS ============================================================
;;;
;;; SPLICE-ON-FRONT [Macro]
;;;
(defmacro splice-on-front (place new-first new-second)
  (declare (type list place))
  "If PLACE initially has the form (first second third ...), change it to
  (NEW-FIRST [NEW-SECOND] second third ...), where NEW-SECOND is included iff
  it is non-NIL."
  `(let ((.new-second. ,new-second))
    (if .new-second.
        (progn (rplaca ,place .new-second.) (push ,new-first ,place))
        (rplaca ,place ,new-first))))

;;; SPLICE-ON-FRONT2 [Macro]
;;;
(defmacro splice-on-front2 (place new-first new-second)
  (declare (type list place))
  "If PLACE initially has the form (first second third ...), change it to
  ([NEW-FIRST] [NEW-SECOND] second third ...), where each of NEW-FIRST and
  NEW-SECOND are included iff they are non-NIL."
  `(let ((.new-first. ,new-first)
	 (.new-second. ,new-second))
    (if .new-second.
        (progn
	  (rplaca ,place .new-second.)
	  (when .new-first. (push .new-first. ,place)))
	(if .new-first.
	    (rplaca ,place ,new-first)
	    (setq ,place (cdr ,place))))))

;;; NEXT-ITEM (BST-IT) --------------------------------------------------------
;;;
(defmethod (bst-it :next-item) ()
  (declare (type list state)
	   (type boolean not-others?))
  (let (curr next)
    (do () ((null state) NIL)
      (setq curr (car state))
      (typecase curr
	(iterator
	 ;; currently busy on an iterator; call it recursively
	 (let ((result (next-item curr)))
	   ;; do different things depending on whether it finished
	   (if result
	       ;; it returned a valid result, so return that for our answer
	       (return result)
	       ;; iterator was empty, so pop it and continue search
	       (pop state))))
	(list
	 ;; we have a tuple describing the current search node
	 (setq next (car curr))
	 (typecase next
	   ;; our next step is to explore either the left or right child
	   (hds::bst-node
	    ;; expand the child and put that on the front to explore next
	    (splice-on-front2 state (hds::expand next not-others?) (cdr curr)))
	   ;; our next step is to explore the current node
	   (hds::hds
	    ;; make an iterator for this node to explore next
	    (splice-on-front state (make-iter-on next path) (cdr curr)))))
	(t
	 (error "NEXT-ITEM: BST-HDS iterator state inconsistent."))))))

;;; BST-RANGE-IT METHODS ======================================================
;;;
;;; NEXT-ITEM (BST-RANGE-IT) -------------------------------------------------
;;;
(defmethod (bst-range-it :next-item) ()
  (declare (type list state)
	   (type function compare-func))
  (let (curr next expanded-node)
    (do () ((null state) NIL)
      (setq curr (car state))
      (typecase curr
	(iterator
	 ;; currently busy on an iterator; call it recursively
	 (let ((result (next-item curr)))
	   ;; do different things depending on whether it finished
	   (if result
	       ;; it returned a valid result, so return that for our answer
	       (return result)
	       ;; iterator was empty, so pop it and continue search
	       (pop state))))
	(list
	 ;; we have a tuple describing the current search node
	 (setq next (car curr))
	 (typecase next
	   ;; our next step is to explore either the left or right child
	   (hds::bst-node
	    ;; expand the child and put that on the front to explore next
	    (multiple-value-bind (range-result do-node)
		(in-range? range next compare-func)
	      (declare (type fixnum range-result)
		       (type boolean do-node))
	      (setq expanded-node
		    (hds::expand-limited next range-result do-node)))
	    ;; Note: expanded-node may be NIL here, so use SPLICE-ON-FRONT2
	    (splice-on-front2 state expanded-node (cdr curr)))
	   ;; our next step is to explore the current node
	   (hds::hds
	    ;; make an iterator for this node to explore next
	    (splice-on-front state (make-iter-on next path) (cdr curr)))))
	(t
	 (error "NEXT-ITEM: BST-RANGE-HDS iterator state inconsistent."))))))

;;; XHT-IT METHODS ============================================================
;;;
;;; NEXT-ITEM (XHT-IT) --------------------------------------------------------
;;;
(defmethod (xht-it :next-item) ()
  (declare (type fixnum state)
	   (type xht-array table)
	   (type boolean not-others?))
  (do (node result)
      ((minusp state) NIL)
    (if curr-it
      ;; evaluate the iterator, returning its result if there was one
      (if (setq result (next-item curr-it))
	  (return result)
	  (setq curr-it NIL))
      ;; otherwise, find the next non-empty bucket and make an iterator
      (when (and (<= 0 (decf state))
		 (setq node (svref table state))
		 (or not-others? (hds::other? node)))
	(setq curr-it (make-iter-on (hds::hanging-hds node) path))))))

;;; HT-IT METHODS =============================================================
;;;
;;; NEXT-ITEM (HT-IT) ---------------------------------------------------------
;;;
(defmethod (ht-it :next-item) ()
  (declare (type fixnum state)
	   (type list-vector table)
	   (type boolean not-others?)
	   (type list bucket))
  (do (node result)
      ((minusp state) NIL)
    (cond
      (curr-it
       ;; evaluate the iterator, returning its result if there was one
       (if (setq result (next-item curr-it))
	   (return result)
	   (setq curr-it NIL)))
      (bucket
       ;; make an iterator out of the next node in the bucket
       (setq node (cdr (pop bucket)))	;get the next node
       (when (or not-others? (hds::other? node))
	 (setq curr-it (make-iter-on (hds::hanging-hds node) path))))
      ;; otherwise, find the next non-empty bucket and set the bucket field
      ((<= 0 (decf state))
       (setq bucket (svref table state))))))

;;; UNION-IT METHODS ==========================================================

;;; NEXT-ITEM (UNION-IT) ------------------------------------------------------
;;;
(defmethod (union-it :next-item) ()
  (declare (type list state))
  (do () ((null state) NIL)
    (let ((result (next-item (car state))))
      (if result (return result) (pop state)))))

;;; CHAINING-IT METHODS =======================================================

;;; NEXT-ITEM (CHAINING-IT) ---------------------------------------------------
;;;
(defmethod (chaining-it :next-item) ()
  (declare (type list state path))
  (let (curr)
    (do () ((null state) NIL)
      (setq curr (car state))
      (typecase curr
	(iterator
	 ;; currently on an iterator; get the item if possible
	 (let ((result (next-item curr)))
	   (if result
	       ;; it returned a valid result; save the item
	       (progn (push result state) (return result))
	       ;; iterator was empty; pop it and continue search
	       (pop state))))
	(t
	 ;; currently just returned an item; chain off it
	 (rplaca state
		 (make-iter-on hds (apply-funcs-in-path path curr))))))))

;;; APPLY-FUNCS-IN-PATH [Function]
;;;
(defun apply-funcs-in-path (path item)
  (declare (type list path))
  "PATH is assumed to be a list of keys (integers or symbols or strings) or of
elements of the form '(function name). It returns a list with the keys
unchanged and with the '(function name) entries replaced by the result of
called the function NAME on ITEM."
  ;; (cadr x) in next line was (eval x)
  (mapcar #'(lambda (x) (if (consp x) (funcall (cadr x) item) x)) path))

;;; *EMPTY-ITERATOR* ==========================================================
;;;
(defconstant *empty-iterator* (make-one-item-it NIL)
"Iterator guaranteed to return NIL on the first invocation of NEXT-ITEM.")

(compile-flavor-methods bst-range bst-po-range)
(compile-flavor-methods one-item-it bag-it)
(compile-flavor-methods bst-range-it bst-it xht-it ht-it union-it chaining-it)

;;; NEXT-ITEM (SEM-IT) ========================================================
;;;
(defmethod (sem-both-it :next-item) ()
  (declare (type bit parity-bit)
	   (type objs::sem sem-arrow)
	   (type fixnum subj-index))
  (do ((obj (next-item state) (next-item state)))
      ((null obj) nil)
    (declare (type objs::obj obj))
    (let ((block (svref *block-array* (obj-index obj))))
      (when (eq parity-bit (aref block subj-index))
	(setf (sem-perm sem-arrow) (obj-perm obj))
	(return sem-arrow)))))

(defmethod (sem-tail-it :next-item) ()
  (declare (type bit parity-bit)
	   (type objs::sem sem-arrow)
	   (type list state obj-path))
  (block this-method
    (loop
     (unless obj-it
       (let ((blk-ix (pop state)))
	 (declare (type (or fixnum null) blk-ix))
	 (when (null blk-ix) (return-from this-method nil))
	 (setq obj-it (make-iter-on (svref *obj-array* blk-ix) obj-path))))
     (let ((obj (next-item obj-it)))
       (declare (type objs::obj obj))
       (when obj
	 (setf (sem-perm sem-arrow) (obj-perm obj))
	 (setf (sem-to sem-arrow) (obj-to obj))
	 (return-from this-method sem-arrow)))
     (setq obj-it nil))))

(defmethod (sem-head-it :next-item) ()
  (declare (type bit parity-bit)
	   (type objs::sem sem-arrow)
	   (type list subj-list))
  (block this-method
    (loop
     (unless subj-list
       (let ((obj (next-item state)))
	 (declare (type objs::obj obj))
	 (when (null obj) (return-from this-method nil))
	 (setf (sem-perm sem-arrow) (obj-perm obj))
	 (setq subj-list (aref *subj-array* (obj-index obj) parity-bit))))
     (let ((subj-sysname (pop subj-list)))
       (when subj-sysname
	 (setf (sem-from sem-arrow) subj-sysname)
	 (return-from this-method sem-arrow))))))

(defmethod (sem-none-it :next-item) ()
  (declare (type bit parity-bit)
	   (type objs::sem sem-arrow)
	   (type list subj-list))
  (block this-method
    (loop
     (unless subj-list
       (let ((obj (next-item state)))
	 (declare (type objs::obj obj))
	 (when (null obj) (return-from this-method nil))
	 (setf (sem-perm sem-arrow) (obj-perm obj))
	 (setf (sem-to sem-arrow) (obj-to obj))
	 (setq subj-list (aref *subj-array* (obj-index obj) parity-bit))))
     (let ((subj-sysname (pop subj-list)))
       (when subj-sysname
	 (setf (sem-from sem-arrow) subj-sysname)
	 (return-from this-method sem-arrow))))))
