/* prober.c

$Header: prober.c,v 1.5 91/07/24 12:54:44 ky Exp $

$log$

*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <sys/dir.h>		/* dir.h, types.h, stat.h   are needed for
				   the   opendir   and   lstat   routines */
#include <sys/types.h>		/* for uid_t, gid_t, struct stat symbols */
#include <sys/stat.h>		/* for struct stat */
#include <string.h>		/* for   strchr()   */
#include <list_type.h>
#include "prober.h"
#include "util.h"		/* for   squish(), StackMemoryT   */

/* Macro definitions for   PermT.   These values will be matched against
   permissions returned by the   stat(1)   routine in UNIX */

#define EXECUTE 0111
#define WRITE 0222
#define READ 0444

/* Accessor functions for PermT.  When invoked, each of these routines
   checks the status of a single permission bit */

#define world_bit(mode,value) (((value) & (mode) & 07) == ((mode) & 07))
#define group_bit(mode,value) (((value) & (mode) & 070) == ((mode) & 070))
#define user_bit(mode,value)  (((value) & (mode) & 0700) == ((mode) & 0700))



/* EXTERNAL GLOBAL VARIABLES */

extern char *this_program;	/* name of the program invoked on the
				   command line */
extern int bad_id;		/* illegal value for both user and group
				   numeric IDs */
extern int compress;		/* BOOLEAN -- true iff compressed output
				   is desired */
extern int do_dirs;		/* BOOLEAN -- true iff acl's for
				   directories should also be output */
extern int max_subdirs;		/* Max depth of subdirectories to probe */
PermT dir_perm = EXECUTE;	/* type of the permission a user must have
				   to get through directories in a
				   pathname */
HashTableT *us2n, *un2s, *gn2i, *gs2n;
				/* Symbol tables for both users and
				   groups, these implement the
				   number<->name invertible mappings */



/* STATIC VARIABLES */

/*   idstore and idlen   define a bounded queue used to store lists of
   identifiers (each identifier represented by an integer).  These are the
   POS and NEG lists of USERS and GROUPS associated with each directory
   and file in the filesystem.  We use a queue to avoid mallocing each
   list individually.  However, much of the queue storage is likely to be
   wasted since we allocate for worst-case usage, which is unlikely to
   arise in practise.

   We'll be needing at most   max_perm_stack_depth   ids for any given
   directory, but since the next directory in the path might change the
   lists, we'll make a copy for each directory in the path.  Hence, we
   need room for O(mpsd^2) uids.  (4 sets of explicitly terminated lists,
   each set may have at most 2+3+4+....+(n+1) == n(n+3)/2 values)  Note
   that once symlinks are added, we'll need room for O(mpsd^3) uids (since
   each directory might be linked to a too-long filename).  When that
   happens, you're better off going to a more memory-conservative allocation
   strategy */

static int *idstore = NULL;	/* Pointer to storage for lists of user
				   and group numbers.  This is where the
				   {pos,neg}{g,u} lists are kept. */
static int idlen;		/* Number of bytes allocated in   idstore */

static StackMemoryT *filename_stack = NULL;
				/* Holds storage for names of parent
				   directories on the permission stack */



/* LOCAL FUNCTION DECLARATIONS */

int strptrcmp ();


/* EXTERNAL FUNCTION DECLARATIONS */

char *int_print (), *info_print (), *str_print ();



/* HISTORY
        24 Jul 91 (ky)  -- Be careful about using permissions from
			   parent directories -- this can cause
			   extra groups to be added to posg when
			   a file's group is different from the
			   containing directory's group.

	09 Oct 90 (mwm) -- cleaned up code and comments while looking for
   known bugs.

	xx Sep 90 (mwm) -- Various bug fixes

	13 Aug 90 (mwm) -- Fixed bug found by heydon in updating of group
   permissions.  Consider the case when a group is denied permission at
   one level, then granted permission (with no world access) at a deeper
   level.  I had forgotten to wipe out the posg list
*/



/* ProbeUNIX -- Probes the part of the UNIX filesystem rooted at   dir,
   focusing attention on those users and groups found in   instance_users
   and instance_groups.  Output is written to   files, users, groups, and
   missing_users.   */

ProbeUNIX (dir, instance_users, instance_groups, sys_users, sys_groups)
char *dir, *instance_users, *instance_groups, *sys_users, *sys_groups;
{
    DirPermT *stack, *tail;
    char *squish (), *whats_today (), *what_host ();
    char store[MAXPATHLEN + 1], *squished_dir;
    extern char cwd[];

/* Allocate storage for the stack of permissions.   max_perm_stack_depth
   reflects the maximum number of directories allowed in a pathname.  A new
   permission stack is allocated (and freed below) for each call of
   ProbeUNIX */

    stack = (DirPermT *) must_malloc (max_perm_stack_depth *
	    sizeof (DirPermT), "Directory Permission Stack");

/* Initialize the filename-sorting storage.  The built-in UNIX directory
   reading functions do not guarantee that filenames will be returned in
   sorted order, so we must explicitly sort them. */

    if (filename_stack == NULL)
	filename_stack = st_init (STRING_BLOCK_SIZE);

/* Allocate storage for user lists used in write_acl.  For an explanation
   of the storage's size, see the comment where idstore is declared. */

    idlen = max_perm_stack_depth * (max_perm_stack_depth + 3) * 2 *
	    sizeof (int);
    idstore = (int *) must_malloc (idlen, "Pos and Neg user list storage");

/* Get rid of extraneous characters in the directory name */

    squished_dir = squish (dir);
    store[MAXPATHLEN] = '\0';
    if (*squished_dir != '/')
	sprintf (store, "%s/%s", cwd, squished_dir);
    else
	strcpy (store, squished_dir);
    if (store[MAXPATHLEN] != '\0') {
	fprintf (stderr,
		"%s:  FATAL ERROR squished name exceeds %d chars, quitting\n",
		this_program, MAXPATHLEN);
	exit (3);
    } /* if store[MAXPATHLEN] != '\0' */

    if (strncmp (store, "/afs/", 5) == 0) {
	fprintf (stderr, "%s:  Can't probe /afs, skipping\n\t'%s'\n",
		this_program, store);
	return;
    } /* if strncmp == 0 */

/* Initialize the user and group tables */

    LoadPasswordFile (sys_users, &us2n, &un2s);
    LoadGroupFile (sys_groups, &gs2n, &gn2i);
    LoadIDFile (0, instance_users);
    LoadIDFile (1, instance_groups);

/* Initialize the stack with permissions of each directory in the path */

    if (InitPermStack (store, stack, &tail, dir_perm) == 0) {
	fprintf (stderr, "%s:  Can't probe '%s', skipping it\n",
		this_program, store);
	return;
    } /* if InitPermStack == 0 */

    printf ("# Probing '%s' (squished from '%s')\n", store, dir);
    printf ("#    on the %s\t%s\n", what_host (), whats_today ());

    probe (store, store + strlen (store), stack, tail, 0);

    free ((char *) stack);
} /* ProbeUNIX */



/* InitPermStack -- Determines the access lists for all directories in the
   root pathname.   dir   holds the root pathname,   stack   points an
   initialized permission stack,   tail   will be initialized here, and
   perm   is the kind of permission required to get access through a
   directory.  Returns 0 on failure.  */

int InitPermStack (dir, stack, tail, perm)
char *dir;
DirPermT *stack, **tail;
PermT perm;
{
    char *last, *this;
    int retval = 1;

    *tail = stack;
    last = dir;

    for (this = strchr (dir, '/'); this; last = this + 1, this =
	    strchr (this + 1, '/')) {
	char temp;

	temp = *(this + 1);
	*(this + 1) = '\0';
	retval = add_dir_to_stack (dir, stack, tail, perm, 1);
	*(this + 1) = temp;
	if (!retval)
	    break;
    } /* for this = dir */

/* Make sure we don't add another permission when the pathname ends with
   a slash.  The pathname is assumed to be squished, so we needn't check
   for double slashes at the end */

    if (retval && last && *last)
	retval = add_dir_to_stack (dir, stack, tail, perm, 0);

    return retval;
} /* InitPermStack */



/* add_dir_to_stack --   Adds the permissions associated with full pathname
   dir   onto the permission stack represented by   stack and tail.   perm
   is the kind of permission required for access to this   dir,   and
   must_be_dir   is set when you don't want to add a file to the stack.
   Returns 0 on failure.  */


int add_dir_to_stack (dir, stack, tail, perm, must_be_dir)
char *dir;
DirPermT *stack, **tail;
PermT perm;
int must_be_dir;
{
    int retval = 1, err;
    struct stat buf;

    if (err = lstat (dir, &buf)) {
	fprintf (stderr, "%s:  No such file or directory '%s' (err=%d)\n",
		this_program, dir, err);
	retval = 0;
    } else if ((buf.st_mode & S_IFMT) == S_IFLNK) {
	fprintf (stderr, "%s:  Can't probe a symlink!  Quitting at '%s'\n",
		this_program, dir);
	retval = 0;
    } else if ((buf.st_mode & S_IFMT) != S_IFDIR) {
	if (must_be_dir) {
	    fprintf (stderr, "%s:  Can't probe, '%s' is not a directory!\n",
		    this_program, dir);
	    retval = 0;
	} /* if must_be_dir */
    } else {
/*fprintf (stderr, "===> Adding dir \"%s\" uid=%d, gid=%d\n", dir,
   buf.st_uid, buf.st_gid);*/
	push_perm (&buf, stack, tail);
	UpdateParams (stack, *tail, perm);
    } /* else */
    return retval;
} /* add_dir_to_stack */



/* MACRO DEFINITIONS for the functions below */

#define top_val(tail) ((tail) -> perms)
#define InGroup(uidp,group) (!Empty (Lookup (group, (char *) uidp), \
	(char *) &bad_id, (unsigned) sizeof (int)))



/* Although push_perm has loaded the new uid and gid onto the stack, the
   two uid and gid lists have not yet been updated.  In this routine we
   update those lists as required.  A list might be zeroed, get one new
   id, or several of the posusers might be moved over to the negusers
   list.  If symlinks were handled, each list might get as many as
   max_perm_stack_depth   new ids, but for now we'll just assume there
   will be at most one extra space needed.

   tail -- pointer to the next available slot; the top value is in location
	tail - 1
*/

UpdateParams (stack, tail, perm)
DirPermT *stack, *tail;
PermT perm;
{
    register DirPermT *top;
    int sw, user, group;
    char *look;
    GroupInfoT *group_info;
    HashTableT *group_mems;

    if (tail == stack) {
	fprintf (stderr, "UpdateParams:  ERROR empty stack!\n");
	return;
    } /* if tail == stack */
    top = tail - 1;

/* Do some quick preprocessing, checks that don't require hashtable
   lookups.  The   any_*   macros check the whos_valid flag in the
   permission data structure to see if such access has been previously
   prohibited (by a negative permission at a higher level directory) */

    if (any_world (top))
	if (!world_bit (perm, top_val (top)))
	    top -> whos_valid = VALID_GROUP;
    if (any_group (top))
	if (!world_bit (perm, top_val (top)) &&
		!group_bit (perm, top_val (top))) {
	    top -> whos_valid = VALID_OWNER;
	    *(top -> posg) = *(top -> negg) = bad_id;
	} /* if !world_bit && !group_bit */
    if (any_owner (top))
	if (!world_bit (perm, top_val (top)) &&
		!group_bit (perm, top_val (top)) &&
		!user_bit (perm, top_val (top))) {
	    top -> whos_valid = 0;
	    *(top -> posg) = *(top -> negg) = bad_id;
	    *(top -> posu) = *(top -> negu) = bad_id;
	} /* if !world_bit && !group_bit && !user_bit */
    if (!any_owner (top))
	return;

/* Answers aren't obvious, use the permissions stack to find them */

/* ky: don't consider parent's permissions when computing sw */
    sw  = ( user_bit (perm, top_val (top)) /* && any_owner (top)*/) ? 04 : 0;
    sw += (group_bit (perm, top_val (top)) /* && any_group (top)*/) ? 02 : 0;
    sw += (world_bit (perm, top_val (top)) /* && any_world (top)*/) ? 01 : 0;
    user = top -> user;
    group = top -> group;

    if ((look = Lookup (gn2i, (char *) &group)) == NULL) {
	fprintf (stderr, "%s: FATAL ERROR!  Group id %d not found in ",
		this_program, group);
	fprintf (stderr, "system group file!\n");
	exit (17);
    } else if (!Empty (look, (char *) &bad_id, sizeof (int)))
	group_info = hash2groupinfo (look);
    else {
	char *name = itoa (group, NULL), *ptr;

	fprintf (stderr, "%s:  No group entry for gid %d\n",
		this_program, group);

/* Insert this group number with an empty hashtable */

	(void) Insert (gn2i, &group, &name);
	(void) Insert (gs2n, &name, &group);
	if ((ptr = Lookup (gn2i, (char *) &group)) != NULL &&
		*(int *) ptr == group) {
	    extern int users_per_group, intcmp ();

	    group_info = hash2groupinfo (ptr);
	    group_info -> flags = 0;
	    MarkCreated (group_info);
	    (void) InitHashTable (&(group_info -> users), (char *) &bad_id,
		    2 * users_per_group, sizeof (int), sizeof (int), intcmp,
		    NULL, NULL);
	} /* if ptr */
    } /* if Empty */

    group_mems = group_info -> users;

    switch (sw) {
	register int *ptr;

/*rrr*/ case 07:
	    break;
/*---*/ case 00:
clean:
	    top -> whos_valid = 0;
	    *(top -> posu) = *(top -> negu) = bad_id;
	    *(top -> posg) = *(top -> negg) = bad_id;
	    break;
/*rr-*/	case 06:
	    for (ptr = top -> posu; *ptr != bad_id; ptr++)
		if (*ptr != user && !InGroup (ptr, group_mems)) {
		    pos_to_neg (ptr, top);
		} /* if *ptr != user && */
	    if (any_world (top))
		top -> whos_valid = VALID_GROUP;
	    if (!perm_member (user, top -> posu) && any_group (top))
		if (check_all_levels (user, dir_perm, stack, top))
		    perm_insert (user, top -> posu);
		else
		    perm_insert (user, top -> negu);
	    if (!perm_member (group, top -> negg))
		perm_insert (group, top -> posg);
	    else
		*(top -> posg) = bad_id;
	    break;
/*r-r*/	case 05:
	    for (ptr = top -> posu; *ptr != bad_id; ptr++)
		if (*ptr != user && InGroup (ptr, group_mems))
		    pos_to_neg (ptr, top);
	    if (!perm_member (user, top -> posu))
		if (check_all_levels (user, dir_perm, stack, top))
		    perm_insert (user, top -> posu);
		else
		    perm_insert (user, top -> negu);
	    if (any_group (top)) {
		if (perm_member (group, top -> posg))
		    perm_delete (group, top -> posg);
		perm_insert (group, top -> negg);
	    } /* if any_group (top) */
	    break;
/*r--*/	case 04:
	    if (perm_member (user, top -> negu) ||
		  (!perm_member (user, top -> posu) &&
		    (in_any_group (user, top -> negg) ||
		       (!in_all_groups (user, top -> posg) &&
			!any_world (top)))))
		goto clean;
	    else {
		top -> whos_valid = VALID_OWNER;
		*(top -> posu) = user;
		*(top -> posu + 1) = bad_id;
		*(top -> negu) = *(top -> posg) = *(top -> negg) = bad_id;
	    } /* else */
	    break;
/*-rr*/	case 03:
	    if (perm_member (user, top -> posu))
		perm_delete (user, top -> posu);
	    perm_insert (user, top -> negu);
/* ky: "world" has access here, so we shouldn't add group to
	posg.
	    if (!perm_member (group, top -> negg))
	      perm_insert (group, top -> posg);
 */
	    break;
/*-r-*/	case 02:
	    for (ptr = top -> posu; *ptr != bad_id; ptr++)
		if (*ptr == user || !InGroup (ptr, group_mems))
		    pos_to_neg (ptr, top);
	    if (any_world (top))
		top -> whos_valid = VALID_GROUP;
	    if (!perm_member (group, top -> negg))
		perm_insert (group, top -> posg);
	    else
		*(top -> posg) = bad_id;
	    perm_insert (user, top -> negu);
	    break;
/*--r*/	case 01:
	    for (ptr = top -> posu; *ptr != bad_id; ptr++)
		if (*ptr == user || InGroup (ptr, group_mems))
		    pos_to_neg (ptr, top);
	    perm_insert (user, top -> negu);
	    if (perm_member (group, top -> posg))
		perm_delete (group, top -> posg);
	    perm_insert (group, top -> negg);
	    break;
    } /* switch */
} /* UpdateParams */



/* check_all_levels -- verifies that   user   has access through all
   permissions stored in the permissions stack, checking only the   perm
   part of the intermediate permissions.  If SymLinks are added
   this could get complicated. */

int check_all_levels (user, perm, stack, tail)
int user;
PermT perm;
DirPermT *stack, *tail;
{
    GroupInfoT *g;
    int retval = 1;

    for (; retval && stack < tail; stack++) {
	char *entry;
	if (user == stack -> user) {
	    if (!user_bit (perm, stack -> perms))
		retval = 0;
	} else if (Empty (entry = Lookup (gn2i, (char *) &(stack -> group)),
		(char *) &bad_id, sizeof (int))) {

/*fprintf (stderr, "stack = %x, tail = %x, tail - stack = %d, uid = %d,
   gid = %d\n", stack, tail,tail-stack, stack -> user, stack -> group),*/

	    fprintf (stderr, "check_all_levels:  couldn't find gid %d\n",
		    stack -> group);
	} else {
	    g = hash2groupinfo (entry);
	    if (InGroup (&user, g -> users)) {
		if (!group_bit (perm, stack -> perms))
		    retval = 0;
	    } else if (!world_bit (perm, stack -> perms))
		retval = 0;
	} /* else */
    } /* for */

    return retval;
} /* check_all_levels */



/* in_any_group -- returns TRUE iff   user   is a member of at least one
   of the groups represented (by their numeric IDs) in   glist.   */

int in_any_group (user, glist)
int user, *glist;
{
    char *entry;
    GroupInfoT *g;

    for (; *glist != bad_id; glist++)
	if (Empty (entry = Lookup (gn2i, (char *) glist), (char *) &bad_id,
		sizeof (int)))
	    fprintf (stderr, "in_any_group:  couldn't find gid %d\n", *glist);
	else {
	    g = hash2groupinfo (entry);
	    if (InGroup (&user, g -> users))
		return 1;
	} /* else */
    return 0;
} /* in_any_group */



/* in_all_groups -- returns TRUE iff   user is a member of *all* groups
   mentioned (by numeric ID) in   glist.   */

int in_all_groups (user, glist)
int user, *glist;
{
    char *entry;
    GroupInfoT *g;

    for (; *glist != bad_id; glist++)
	if (Empty (entry = Lookup (gn2i, (char *) glist), (char *) &bad_id,
		sizeof (int)))
	    fprintf (stderr, "in_all_groups:  couldn't find gid %d\n", *glist);
	else {
	    g = hash2groupinfo (entry);
	    if (!InGroup (&user, g -> users))
		return 0;
	} /* else */
    return 1;
} /* in_all_groups */



/* pos_to_neg -- Moves a numeric *user* ID from its position in the positive
   user list   (posu)   onto the beginning of the list of negative users.
   This is cute because it does a little pointer shuffling in order to
   avoid dealing with memory reallocation.  */

pos_to_neg (posp, perm)
int *posp;
DirPermT *perm;
{
    int *endp, temp;

    for (endp = posp; *endp != bad_id; endp++)
	;
    temp = *posp;			/* save the original ID */
    *posp = *(endp - 1);		/* move the last pos ID to take
					   its place */
    *(endp - 1) = bad_id;		/* move the end of posu in by one */
    *--(perm -> negu) = temp;		/* add original ID to negu list,
					   decrement pointer to negu list */
} /* pos_to_neg */



/* perm_member -- checks to see if   id   is a member of the list of
   permissions   list.   */

int perm_member (id, list)
int id, *list;
{
    for (; *list != bad_id; list++)
	if (*list == id)
	    return 1;
    return 0;
} /* perm_member */



/* perm_delete -- delete an id from the list by copying the last id over
   it, then moving the end pointer down. */

perm_delete (id, list)
int id, *list;
{
    int *ptr;

    for (; *list != bad_id && *list != id; list++)
	;

    if (*list == bad_id)

/* id   wasn't found in the list, so do nothing */

	return;

    for (ptr = list; *ptr; ptr++)
	;
    *list = *(ptr - 1);
    *(ptr - 1) = bad_id;
} /* perm_delete */



/* perm_insert -- add a new   id   into the   list,   but don't allow
   duplicate entries. */

perm_insert (id, list)
int id, *list;
{
    int *endp;

    for (endp = list; *endp != bad_id && *endp != id; endp++)
	;

    if (*endp == id)
	return;

    *endp++ = id;
    *endp = bad_id;
} /* perm_insert */



/* probe -- main body of the prober

	This routine is called recursively on the subdirectories in
   dirstring.   Since all of the lists might be modified, local copies are
   made before modifications occur.  However, the directory permissions of
   all the ancestors might be needed at any time, so these are keep in an
   explicit stack, the   perm_stack   parameter.

	dirstring -- string naming the current directory/file
	string_tail -- pointer to the '\0' character in   dirstring
	perm_stack -- pointer to the base of the stack
	perm_tail -- pointer to the head of the stack, where new
	   permissions are enqueued
*/

static probe (dirstring, string_tail, perm_stack, perm_tail, depth)
char *dirstring, *string_tail;
DirPermT *perm_stack, *perm_tail;
int depth;
{
    DIR *opendir ();
    DIR *dp;

    if (depth > max_subdirs)
	return;

    if ((dp = opendir (dirstring)) == (DIR *) NULL) {
	struct stat buf;
	DirPermT thisdir;

/* We've been asked to probe a file */

	lstat (dirstring, &buf);
	if ((buf.st_mode & S_IFMT) == S_IFLNK)
	    fprintf (stderr, "%s:  Can't probe a symlink!  Skipping '%s'\n",
		    this_program, dirstring);
	else if ((buf.st_mode & S_IFMT) != S_IFDIR) {
	    while (string_tail > dirstring && *string_tail != '/')
		string_tail--;
	    if (*string_tail == '/')
		string_tail++;

/* Save the original pointers, *not* list elements, in a local copy */

	    dir_perm_cpy (&thisdir, perm_tail - 1);

/* compress_lists   might zero a pointer, but will not affect the contents
   of the lists.  That's why we need the local copy of pointers, but don't
   need to copy the lists themselves. */

	    compress_lists (&thisdir);
	    write_header (stdout, dirstring, string_tail, &thisdir);
	    write_file (stdout, dirstring, string_tail, &buf,
		    perm_stack, perm_tail, &thisdir);
	} else
	    fprintf (stderr, "%s:  Can't open directory '%s' for probing.\n",
		    this_program, dirstring);
    } else {

/* Adjust the parameters to reflect the current directory's permissions */

	UpdateParams (perm_stack, perm_tail, dir_perm);
	if (*string_tail != '/')
	    *string_tail++ = '/';

/* Scan through the files/subdirectories in this directory */

	do_this_directory (dp, dirstring, string_tail, perm_stack, perm_tail,
		depth);
	closedir (dp);
    } /* else */
} /* probe */



#define JUST_FILES 1
#define JUST_DIRS 2
#define BOTH 3

/* do_this_directory -- Scan all the files and subdirectories in this
   directory, and generate output for each of them.  Files will be written
   out in sorted order, followed by all subdirectories in sorted order.
   Unfortunately, this approach means the total output will *not* be
   sorted, because it's quite possible that some subdirectories appear
   lexicographically between some files. */

do_this_directory (dp, dirstring, string_tail, perm_stack, perm_tail, depth)
DIR *dp;
char *dirstring, *string_tail;
DirPermT *perm_stack, *perm_tail;
int depth;
{
    int total, files;

    count_files_in_this_dir (dp, dirstring, string_tail, &total, &files);
    rewinddir (dp);
    if (!compress)
	do_things_in_this_dir (BOTH, 0, dp, dirstring, string_tail,
		perm_stack, perm_tail, total, depth);
    else {
	do_things_in_this_dir (JUST_FILES, 0, dp, dirstring, string_tail,
		perm_stack, perm_tail, files, depth);
	rewinddir (dp);
	do_things_in_this_dir (JUST_DIRS, 1, dp, dirstring, string_tail,
		perm_stack, perm_tail, total - files, depth);
    } /* else */

} /* do_this_directory */



/* count_files_in_this_dir -- determine the total number of files and
   subdirectories in this directory.  Results will be returned in
   (uninitialized) parameters   totalptr, filesptr.   We want to
   precompute these counts so that we can store the names (and pointers
   to the names) in an array for sorting.
*/

count_files_in_this_dir (dp, dirstring, string_tail, totalptr, filesptr)
DIR *dp;
char *dirstring, *string_tail;
int *totalptr, *filesptr;
{
    struct direct *readdir ();
    struct direct *fp;

    *totalptr = 0;
    *filesptr = 0;

    while (fp = readdir (dp)) {
	char *this_file = fp -> d_name;
	struct stat buf;

	strcpy (string_tail, this_file);
	lstat (dirstring, &buf);
	if ((buf.st_mode & S_IFMT) != S_IFLNK) {
	    (*totalptr)++;
	    if ((buf.st_mode & S_IFMT) != S_IFDIR)
	        (*filesptr)++;
	} /* if buf.st_mode & S_IFMT != S_IFLNK */
    } /* while fp = readdir (dp) */
    (*totalptr) -= 2;		/* Get rid of the "." and ".." counts */
} /* count_files_in_this_dir */



/* do_things_in_this_dir -- Write the access lists for all of the files
   and/or subdirectories in this directory.  The files are read in an
   arbitrary order, so the names must be sorted in core before writing out
   their info.

	PARAMETERS

	which -- one of JUST_FILES, JUST_DIRS, BOTH
	ignore_syms -- if TRUE, quietly skip symlinks, else output warning
	dp -- pointer to a *rewound* directory, ready to be read.
	count -- number of files/directories to be processed (precomputed
	   from the same data)
	depth -- current subdirectory depth

	EXTERNAL REFERENCES

	filename_stack -- memory for pointers and text of filenames
	this_program -- name of the prober as given on the command line
	strptrcmp -- compare function for qsort
*/


do_things_in_this_dir (which, ignore_syms, dp, dirstring, string_tail,
	perm_stack, perm_tail, count, depth)
DIR *dp;
char *dirstring, *string_tail;
DirPermT *perm_stack, *perm_tail;
int count, which, ignore_syms, depth;
{
    struct direct *readdir ();
    struct direct *fp;
    DirPermT thisdir;
    int did_one = 0;
    char **mem, **start, **ptr;

    if ((mem = (char **) st_alloc (&filename_stack, (int) (sizeof (long) +
	    (count + 1) * sizeof (char *)))) == NULL) {
	fprintf (stderr, "do_things_in_this_dir:  Out of memory!!\n");
	return;
    } /* if mem = st_alloc */
    *((long *) mem) = count;
    start = mem = (char **) (((char *) mem) + sizeof (long));

/* We need to sort the filenames (and/or directories) before writing them
   out, so we'll collect them in the memory pointed to by   mem.   That
   memory has the following example structure:

	3	file1ptr   file2ptr   file3ptr   NULL   file1text   file2text

   The first word holds a long int, which is the number of filenames that
   follow.  After the int we have an array of string pointers, which is
   NULL-terminated.  Finally the text of the filenames themselves, each of
   which is '\0'-terminated.

   All of this memory is allocated in the   filename_stack   space, and is
   freed up on completion.
*/

    while (fp = readdir (dp)) {
	char *this_file = fp -> d_name;
	struct stat buf;

	strcpy (string_tail, this_file);
	lstat (dirstring, &buf);
	if ((buf.st_mode & S_IFMT) == S_IFLNK) {

/* Pass over symlinks */

	    if (!ignore_syms)
		fprintf (stderr,
			"%s: Can't probe a symlink!  Skipping '%s'\n",
			this_program, dirstring);
	} else if (which != JUST_FILES && (buf.st_mode & S_IFMT) == S_IFDIR) {

/* Remember subdirectories, skipping the hard-coded '.' and '..' */

	    if (*this_file != '.' || (*(this_file + 1) != '\0' &&
		    (*(this_file + 1) != '.' || *(this_file + 2) !=
		    '\0'))) {
		*mem = st_alloc (&filename_stack, strlen (this_file) + 1);
		strcpy (*mem++, this_file);
	    } /* if *this_file != '.' */
	} else if (which != JUST_DIRS && (buf.st_mode & S_IFMT) != S_IFDIR) {

/* Remember filenames */

	    *mem = st_alloc (&filename_stack, strlen (this_file) + 1);
	    strcpy (*mem++, this_file);
	} /* if which != JUST_DIRS */
    } /* while fp = readdir */
    *mem++ = (char *) NULL;

    qsort ((char *) start, count, sizeof (char *), strptrcmp);
/*print_strs (start);*/

#define WRITE_HEADER_IF_NEEDED if (compress && !did_one) { \
		dir_perm_cpy (&thisdir, perm_tail - 1); \
		compress_lists (&thisdir); \
		write_header (stdout, dirstring, string_tail, &thisdir); \
		did_one = 1; \
	    } /* if compress && !did_one */


    if (do_dirs) {
	struct stat buf;
	char c = *string_tail;

	*string_tail = '\0';
	if (lstat (dirstring, &buf) != 0)
	    fprintf (stderr,
		    "%s: ERROR Couldn't stat '%s', skipping its ACLs\n", 
		    this_program, dirstring);
	else {
	    WRITE_HEADER_IF_NEEDED;
	    write_file (stdout, dirstring, string_tail, &buf,
		    perm_stack, perm_tail, &thisdir);
	} /* else */
	*string_tail = c;
    } /* if do_dirs */

    for (ptr = start; *ptr; ptr++) {
	struct stat buf;

	strcpy (string_tail, *ptr);
	if (lstat (dirstring, &buf) != 0)

/* Could not get "x" access on this file/subdirectory */

	    continue;
	else if ((buf.st_mode & S_IFMT) == S_IFLNK) {
	    fprintf (stderr, "%s:  '%s' just became a symlink??? Skipping\n",
		    this_program, dirstring);
	    continue;
	} else if (which != JUST_DIRS && (buf.st_mode & S_IFMT) != S_IFDIR) {
	    WRITE_HEADER_IF_NEEDED;
	    write_file (stdout, dirstring, string_tail, &buf,
		    perm_stack, perm_tail, &thisdir);
	} else if (which != JUST_FILES && (buf.st_mode & S_IFMT) == S_IFDIR) {

/* Save these directory permissions and recurse on the next subdirectory */

	    push_perm (&buf, perm_stack, &perm_tail);
	    probe (dirstring, string_tail + strlen (string_tail),
		    perm_stack, perm_tail, depth + 1);
	    pop_perm (perm_stack, &perm_tail);
	} /* if (which != JUST_FILES && */
    } /* for */

    if (st_free (&filename_stack, (char *) start) == 0)
	fprintf (stderr, "do_things_in_this_dir: WARNING!  Couldn't free %x\n",
		mem);
} /* do_things_in_this_dir*/



/* pop_perm -- remove a directory's permissions from the stack, once all
   of its subdirectories have been processed */

pop_perm (perm_stack, perm_tail)
DirPermT *perm_stack, **perm_tail;
{
    if (*perm_tail <= perm_stack) {
	fprintf (stderr, "%s:  FATAL ERROR  Stack underflow\n", this_program);
	exit (3);
    } /* if perm_tail <= perm_stack */

    (*perm_tail)--;
} /* pop_perm */



/* push_perm -- add a new subdirectory onto the permission stack.  We need
   to keep all this info around because sometimes it's impossible to
   decide whether a user has access to a file without checking group
   membership all the way back up the directory path.  We automatically
   update the access list from the parent directory, by copying the old
   list and leaving room for one more identifier.  Only one more is needed
   (so long as we ignore SymLinks) because each directory in the path has
   only one user and one group name associated with it. */

push_perm (statbuf, perm_stack, perm_tail)
struct stat *statbuf;
DirPermT *perm_stack, **perm_tail;
{
    if (*perm_tail >= perm_stack + max_perm_stack_depth) {
	fprintf (stderr, "%s:  FATAL ERROR  Stack Size too deep, rerun with ",
		this_program);
	fprintf (stderr, "size %d\n", max_perm_stack_depth * 2);
	fprintf (stderr, "stack %x, tail %x\n", perm_stack, *perm_tail);
	exit (2);
    } /* if perm_tail - perm_stack */

    (*perm_tail) -> user = statbuf -> st_uid;
    (*perm_tail) -> group = statbuf -> st_gid;
    (*perm_tail) -> perms = statbuf -> st_mode;
    if (*perm_tail == perm_stack) {
	(*perm_tail) -> posu = idstore;
	(*perm_tail) -> negu = idstore + 2;
	(*perm_tail) -> posg = idstore + 4;
	(*perm_tail) -> negg = idstore + 6;
	(*perm_tail) -> realtail = idstore + 8;
	(*perm_tail) -> whos_valid = VALID_WORLD;
	*((*perm_tail) -> posu) = *((*perm_tail) -> negu) = bad_id;
	*((*perm_tail) -> posg) = *((*perm_tail) -> negg) = bad_id;
    } else {
	int *permcpy ();

	(*perm_tail) -> posu = (*perm_tail - 1) -> realtail;
	(*perm_tail) -> negu = permcpy ((*perm_tail) -> posu, (*perm_tail - 1)
		-> posu);
	(*perm_tail) -> posg = permcpy ((*perm_tail) -> negu, (*perm_tail - 1)
		-> negu);
	(*perm_tail) -> negg = permcpy ((*perm_tail) -> posg, (*perm_tail - 1)
		-> posg);
	(*perm_tail) -> realtail = permcpy ((*perm_tail) -> negg,
		(*perm_tail - 1) -> negg);
	(*perm_tail) -> whos_valid = (*perm_tail - 1) -> whos_valid;
	if ((*perm_tail) -> realtail >= idstore + idlen) {
	    fprintf (stderr, "%s:  FATAL ERROR  Identifier store exceeded!\n",
		    this_program);
	    fprintf (stderr, "Try running with a larger stack than %d\n",
		    max_perm_stack_depth);
	    exit (2);
	} /* if */
    } /* else */
    (*perm_tail)++;
} /* push_perm */



/* permcpy -- copies a list of integers (terminated by   bad_id)   from
   one location to another. */

int *permcpy (to, from)
int *to, *from;
{
    while (*from != bad_id)
	*to++ = *from++;
    *to++ = bad_id;

    return to + 1;	/* leave room for one additional int */
} /* permcpy */



/* Static variables used for writing out the access lists.   perms   holds
   the masks for all permissions we care about,   perm_str   holds the
   strings used to distinguish those permissions in the output file, and
   perm_store   is used to cache all access lists for this file, so we can
   see if any overlap.  When two permissions have the same access lists,
   we need only output the list once with multiple names. */

static PermT perms[] = {READ, WRITE, EXECUTE, 0};
static char *perm_str[] = {"R", "W", "E", 0};
#define PCNT (sizeof(perms) / sizeof (PermT) - 1)
static DirPermT perm_store[PCNT];



/* write_header -- Writes the full pathname of the current directory (named in
   dirstring)   to the output file.  In compressed mode, the access list
   associated with this directory is also written, under the assumption
   that many of the files in this directory will have similar lists. */

write_header (outfp, dirstring, string_tail, thisdir)
FILE *outfp;
char *dirstring, *string_tail;
DirPermT *thisdir;
{
    char c;

    if (*(string_tail - 1) != '/') {
	fprintf (stderr,
		"%s [write_header]:  Error, couldn't find slash in '%s'\n",
		this_program, dirstring);
	return;
    } /* if */
    c = *string_tail;
    *string_tail = '\0';
    fprintf (outfp, "%s  ", dirstring);
    *string_tail = c;

    write_acl (outfp, thisdir);
    putc ('\n', outfp);

} /* write_header */



/* write_acl -- uses the   whos_valid, posu, negu, posg, and negg   fields
   of   top   to generate a simple access control list */

write_acl (fp, top)
FILE *fp;
DirPermT *top;
{
    int world = any_world (top);
    register int *posu = top -> posu, *negu = top -> negu;
    register int *posg = top -> posg, *negg = top -> negg;

/* Everything must be enclosed within parentheses */

    fprintf (fp, "(");

/* See if everyone has positive (then negative) access */

    if (world && (negu == NULL || *negu == bad_id) &&
		 (negg == NULL || *negg == bad_id))
	fprintf (fp, "POS");
    else if (!world && (posu == NULL || *posu == bad_id) &&
		       (posg == NULL || *posg == bad_id))
	fprintf (fp, "NEG");
    else {
	int did_one = 0, polarity;

/* So the ACL has at least one exception */

/* The ACL will be constructed from these lists in the following order:

	POSU	NEGU	NEGG	POSG	world_bit

   The NEGG ids *must* appear before the POSG ids, because the access list
   will be read from left to right.  The semantics of user access are as
   follows:

	     if user is in POSU then user has access.
	else if user is in NEGU then user does *not* have access.
	else if user is in at least one of NEGG then *no* access.
	else if user is in *all* of POSG then user has access.

   The   polarity   flag is used to determine the type of access just
   granted (1 means we're writing a POS list, 0 means we're writing a NEG
   list).  Of course,   polarity   is only valid when   did_one   is true.
*/

	putc ('(', fp);
	if (*posu != bad_id) {
	    fprintf (fp, "POS");
	    write_uids (fp, posu, 1);
	    did_one = 1;
	    polarity = 1;
	} /* if *posu */

	if (negu && *negu != bad_id) {
	    if (did_one)
		fprintf (fp, ") (");
	    fprintf (fp, "NEG");
	    write_uids (fp, negu, 1);
	    did_one = 1;
	    polarity = 0;
	} /* if *negu */

	if (negg && *negg != bad_id) {
	    if (did_one) {
		if (polarity == 1)
		    fprintf (fp, ") (NEG");
	    } else
		fprintf (fp, "NEG");
	    write_gids (fp, negg, 1);
	    did_one = 1;
	    polarity = 0;
	} /* if *negg */

	if (posg && *posg != bad_id) {
	    if (did_one) {
		if (polarity == 0)
		    fprintf (fp, ") (POS");
	    } else
		fprintf (fp, "POS");

/* Unlike the other lists, if there is more than one group in   posg,
   then positive access is only granted is a user is in *all* groups in
   posg,   not just one.  The way we communicate that notion is to enclose
   the group ids in another level of parentheses. */

	    if (*(posg + 1) != bad_id)
		fprintf (fp, " (");
	    write_gids (fp, posg, (*(posg + 1) == bad_id));
	    if (*(posg + 1) != bad_id)
		putc (')', fp);
	    did_one = 1;
	    polarity = 1;
	} /* if *posg */

	fprintf (fp, ") %s", world ? "POS" : "NEG");
    } /* else */
    fprintf (fp, ")");
} /* write_acl */



/* compress_lists -- update the pointers in the current directory to
   reflect a more concise representation.  For example, if we have lists
   of users and groups with positive permission, none with negative
   permission, *and* the world has positive permission, we may as well
   zero the positive lists since they're covered by the global access
   anyway. */

compress_lists (top)
DirPermT *top;
{
    int world = any_world (top);

/* See if everyone has positive (then negative) access */

/*    fprintf (stderr, "\n(POSU:");
    write_uids (stderr, top -> posu, 1);
    fprintf (stderr, ") (NEGU:");
    write_uids (stderr, top -> negu, 1);
    fprintf (stderr, ") (NEGG:");
    write_gids (stderr, top -> negg, 1);
    fprintf (stderr, ") (POSG:");
    write_gids (stderr, top -> posg, 1);
    fprintf (stderr, ")\n");
*/

    if (world && *(top -> negu) == bad_id && *(top -> negg) == bad_id)
	*(top -> posu) = *(top -> posg) = bad_id;
    else if (!world && *(top -> posu) == bad_id && *(top -> posg) == bad_id)
	*(top -> negu) = *(top -> negg) = bad_id;
    else {

/* So the ACL has at least one exception */

/* The ACL will be constructed from these lists in the following order:

	POSU	NEGU	NEGG	POSG	world_bit

   The NEGG ids *must* appear before the POSG ids, because the access list
   will be read from left to right.  The semantics of user access are as
   follows:

	     if user is in POSU then user has access.
	else if user is in NEGU then user does *not* have access.
	else if user is in at least one of NEGG then *no* access.
	else if user is in *all* of POSG then user has access.

*/

/* Lose any redundant POS's or NEG's */

	if (world)
	    (top -> posg) = NULL;
	else if (top -> posg == NULL || *(top -> posg) == bad_id)
	    (top -> negg) = (top -> negu) = NULL;
    } /* else */

/*    fprintf (stderr, "\n(POSU:");
    write_uids (stderr, top -> posu, 1);
    fprintf (stderr, ") (NEGU:");
    write_uids (stderr, top -> negu, 1);
    fprintf (stderr, ") (NEGG:");
    write_gids (stderr, top -> negg, 1);
    fprintf (stderr, ") (POSG:");
    write_gids (stderr, top -> posg, 1);
    fprintf (stderr, ")\n");
*/


} /* compress_lists */



/* write_uids -- write the list of names associated with the numeric user
   ids in   uids.   There will be a space after each name, and if   space
   is true then a leading space will be output.  This is *not* a generic
   routine, for it assumes that any user id's that get written must be
   important, so it tags their symbol table flags with markers to indicate
   that this ID is *needed* to represent this filesystem.
*/

write_uids (fp, uids, space)
FILE *fp;
int *uids, space;
{
    if (*uids) {
	if (space)
	    putc (' ', fp);
	fprintf (fp, "%s", unum2name (uids));
	MarkNeed (hash2strinfo (Lookup (un2s, (char *) uids)));
	uids++;
    } /* if *uids */

    for (; *uids != bad_id; uids++) {
	fprintf (fp, " %s", unum2name (uids));
	MarkNeed (hash2strinfo (Lookup (un2s, (char *) uids)));
    } /* for */
} /* write_uids */



/* write_gids -- write the list of names associated with the numeric group
   ids in   gids.   There will be a space after each name, and if   space
   is true then a leading space will also be output.  In order to
   distinguish group names from user names, the string ":1" will be
   appended to all group names. This is *not* a generic routine, for it
   assumes that any group id's that get written must be important, so it
   tags their symbol table flags with markers to indicate that this ID is
   *needed* to represent this filesystem.
*/

write_gids (fp, gids, space)
FILE *fp;
int *gids, space;
{
    if (*gids) {
	GroupInfoT *g = gnum2info(gids);

	if (space)
	    putc (' ', fp);
	fprintf (fp, "%s:1", g -> name);
	MarkNeed (g);
	gids++;
    } /* if (*gids) */

    for (; *gids != bad_id; gids++) {
	GroupInfoT *g = gnum2info(gids);

	fprintf (fp, " %s:1", g -> name);
	MarkNeed (g);
    } /* for */
} /* write_gids */



/* write_file -- write the access lists for one file into the output
   stream.  This is complicated by the compressed mode of output, which
   requires that we check every opportunity to compact the information
   produced.

	PARAMETERS

	dirstring -- full pathname of this file
	filename -- pointer into   dirstring,   pointing to the filename
   without preceeding full pathname
	thisdir -- Used in the compressed mode; if this file's access list
   matches that for the directory, nothing will be output
*/

write_file (outfp, dirstring, filename, statbuf, perm_stack, perm_tail,
	thisdir)
FILE *outfp;
char *dirstring, *filename;
struct stat *statbuf;
DirPermT *perm_stack, *perm_tail, *thisdir;
{
    char *str_print (), *int_print (), *info_print ();
    int i, *realtail, printed[PCNT], did_one = 0;

    fprintf (outfp, "%s%s:\n", compress ? "\t" : "", compress ? filename
    		: dirstring);
    realtail = (perm_tail - 1) -> realtail;

/* This routine is a little complex, because we must handle dynamic list
   allocation explicitly.  Also, we want to keep the access lists for
   *all* permissions around, to see which (if any) may be omitted from the
   output because they are redundant.

   UpdateParams generates the access lists, but does so only on the
   permission stack.  Because it's a stack, we must clear the previous
   permission before computing the next one.  We therefore copy those
   previous pointers (only the pointers!) into the temporary storage array
   perm_store.   The actual list contents are stored in the global list
   memory, and *not* deallocated when the stack is popped.  We do this by
   fudging the   realtail   pointer in the parent directory's stack frame.
   So the high level approach is:

	for each permission P:
		generate access list w.r.t P on the stack
		copy the stack pointers into temp storage
		make sure the id's in the acl don't get overwritten
*/

    for (i = 0; i < PCNT; i++) {

	printed[i] = 0;
	push_perm (statbuf, perm_stack, &perm_tail);
	UpdateParams (perm_stack, perm_tail, perms[i]);
	compress_lists (perm_tail - 1);
	dir_perm_cpy (&perm_store[i], perm_tail - 1);
	(perm_tail - 2) -> realtail = (perm_tail - 1) -> realtail;
	pop_perm (perm_stack, &perm_tail);
    } /* for i = 0 */

/* Check here for equality of id lists. */

    for (i = 0; i < PCNT; i++) {
	int j;

	if (printed[i])
	    continue;

/* Use the same access list as that for the directory */

	if (compress && dir_perm_eq (&perm_store[i], thisdir))
	    continue;
	if (did_one)
	    putc ('\n', outfp);
	did_one = 1;

	fprintf (outfp, "\t   %s", perm_str[i]);
	for (j = i + 1; j < PCNT; j++)
	    if (!printed[j] && dir_perm_eq (&perm_store[i],
		    &perm_store[j])) {
		fprintf (outfp, " %s", perm_str[j]);
		printed[j] = 1;
	    } /* if !printed[j] && */
	fprintf (outfp, ": ");
	write_acl (outfp, &perm_store[i]);
    } /* for i = 0 */

    fprintf (outfp, ".\n");
    (perm_tail - 1) -> realtail = realtail;
} /* write_file */



/* hexdump -- debugging routine, outputs   len   bytes starting at   mem
   in hex */

hexdump (fp, mem, len)
FILE *fp;
char *mem;
int len;
{
    int i;

    for (i = 0; i < len; i++)
	fprintf (fp, "%2x ", mem[i]);
} /* hexdump */



/* dir_perm_eq -- Compares the access lists in the permission structure */

int dir_perm_eq (d1, d2)
DirPermT *d1, *d2;
{
/*    fprintf (stderr, "dir_perm_eq:  D1 is ");
    print_dir_perm (stderr, d1);
    fprintf (stderr, "              D2 is ");
    print_dir_perm (stderr, d2);
*/
    return d1 != NULL && d2 != NULL &&
	   any_world (d1) == any_world (d2) &&
	   perm_eq (d1 -> posu, d2 -> posu) &&
	   perm_eq (d1 -> negu, d2 -> negu) &&
	   perm_eq (d1 -> posg, d2 -> posg) &&
	   perm_eq (d1 -> negg, d2 -> negg);
} /* dir_perm_eq */

print_dir_perm (fp, d)
FILE *fp;
DirPermT *d;
{
    static char *strs[] = {"noone", "user", "group", "world"};
    fprintf (fp, "U:%d G:%d p:%o w:%s ", d -> user, d -> group,
	    d -> perms, strs[d -> whos_valid]);
    if (d -> posu) {
	fprintf (fp, "(PU:");
	write_uids (fp, d -> posu, 1);
    }
    if (d -> negu) {
	fprintf (fp, ") (NU:");
	write_uids (fp, d -> negu, 1);
    }
    if (d -> negg) {
	fprintf (fp, ") (NG:");
	write_gids (fp, d -> negg, 1);
    }
    if (d -> posg) {
	fprintf (fp, ") (PG:");
	write_gids (fp, d -> posg, 1);
    }
    fprintf (fp, ")\n");

    
} /* print_dir_perm */

int perm_eq (p1, p2)
int *p1, *p2;
{
    if (p1 == p2)
	return 1;
    if (p1 == NULL || p2 == NULL)
	return 0;

    while (*p1 != bad_id && *p1 == *p2)
	p1++, p2++;

    return *p1 == bad_id && *p2 == bad_id;
} /* perm_eq */


static char *memcpy(s1, s2, n)
register char *s1, *s2;
int n;
{
    register char *s0 = s1, *se = s1 + n;

    while(s1 < se)
	*s1++ = *s2++;
    return s0;
} /* memcpy */


dir_perm_cpy (d1, d2)
DirPermT *d1, *d2;
{
    memcpy ((char *) d1, (char *) d2, (int) sizeof (DirPermT));
} /* dir_perm_cpy */


print_strs (start)
char **start;
{
    char *str;

    while (*start)
	str = *start++,	printf ("%x=%s ", str, str);
    printf ("\n");
} /* print_strs */

