/* util.c -- utility routines used by the prober/verifier

	Source file:	$Source: /usr/miro/src/bin/probe/RCS/util.c,v $
	State:		$State: Exp $

$Header: util.c,v 1.2 91/01/25 01:34:08 mwm Exp $

*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

#include <stdio.h>
#include <sys/param.h>		/* For MAXPATHLEN */
#include <sys/time.h>		/* for gettimeofday(2) */
#include <time.h>		/* for ctime(3) */
#include <strings.h>		/* for strcpy(3) */
#include "util.h"

#define SLASH '/'		/* change this to '\\' for MSDOS */

char cwd[MAXPATHLEN + 1];
extern char *this_program;

init_cwd ()
{
    char *getwd ();
    int len;

    if (getwd (cwd) == (char *) 0) {
	fprintf (stderr, "%s:  couldn't find current directory, quitting\n",
		this_program);
	fprintf (stderr, "%s:  '%s'\n", this_program, cwd);
	exit (1);
    } /* if getwd == 0 */
    if (*cwd != SLASH) {
	fprintf (stderr, "%s:  current directory doesn't start with a '%c',",
		this_program, SLASH);
	fprintf (stderr, "quitting\n");
	exit (1);
    } /* if *cwd != SLASH */

/* Now we have the full pathname of the current directory in global
   variable   cwd.   Now remove the trailing slash if one exists. */

    len = strlen (cwd);
    if (len == 0) {
	fprintf (stderr, "%s:  Zero length current directory, quitting\n",
		this_program);
	exit (1);
    } /* if len == 0 */

    if (cwd[len - 1] == SLASH) {
	cwd[len - 1] = '\0';
    } /* if cwd[len - 1] == SLASH */

} /* init_cwd */


/* squish -- remove double slashes and parent pointers from a pathname.
	The result is returned in static storage, so it must be copied out if
	it needs to be saved.

	BUG BUG  doesn't handle symlinks
 */

char *squish (pathname)
char *pathname;
{
    static char store[MAXPATHLEN + 1];
    char *result, *input;

    if (pathname == (char *) NULL)
	return NULL;

    result = store;

/* The "squished" pathname may be longer than the input string, if there
   are any parent pointers ("../") that move up past the current
   directory. */

    for (input = pathname; *input; input++) {
	switch (*input) {
	    case '.':
	        if (input == pathname || *(input - 1) == SLASH)

/* Only worry about periods at the beginning of directory names */

		    if (*(input + 1) == SLASH || *(input + 1) == '\0') {

/* Strip out any "/./" strings, since they are no-ops */

			if (result == store)
/* case "." */		    if (*(input + 1) == '\0') {
				strcpy (store, cwd);
				result = store + strlen (cwd);
/* case "./" */		    } else
				input++;
/* case "/./" or "/." */else
			    result--;
			continue;
		    } else if (*(input + 1) == '.' && (*(input + 2) == SLASH ||
			    *(input + 2) == '\0')) {

/* Found a parent pointer (".."), so move the result pointer back up one
   level.  Don't forget to handle pointers above the current directory */

			if (result == store) {
			    strcpy (store, cwd);
			    result = store + strlen (cwd);
			} /* if result == store */

			if (result == store + 1 && store[0] == SLASH) {
			    fprintf (stderr,
    "%s:  Warning!  Attempting to access super-root in pathname\n",
				    this_program);
			    fprintf (stderr, "%s:  '%s'\n", this_program,
				    pathname);
			    *result++ = '.';
			    *result++ = '.';
			} else {
			    char *ptr;

/* Find prior slash */

			    for (ptr = result - 2; ptr > store && *ptr != SLASH;)
				ptr--;
			    if (*ptr == SLASH)
				result = ptr;
			    else {
				strcpy (store, cwd);
				result = store + strlen (cwd);
			    } /* else */
			} /* else */
			input++;
			continue;
		    } /* if *input+1 == '.' */
	        break;
	    case SLASH:
	        if (*(input + 1) == SLASH) {
		    input++;
		    if (*(input + 1) == SLASH) {
			fprintf (stderr, "%s:  Too many slashes in '%s'\n",
				this_program, pathname);
			result = store;
			goto end;
		    } /* if *input+1 */
		} /* if *input+1 */
	        break;
	} /* switch */

	*result++ = *input;
    } /* for input = pathname */

    if (result == store)
	if (*pathname == SLASH)
	    *result++ = SLASH;
	else {
	    strcpy (store, cwd);
	    result = store + strlen (cwd);
	} /* else */

end:
    *result = '\0';
    return store;
} /* squish */


char *whats_today ()
{
    struct timeval tp;
    struct timezone tzp;
    extern char *ctime ();
    char *ptr;

    gettimeofday (&tp, &tzp);
    ptr = ctime (&(tp.tv_sec));
    if (ptr == NULL)
	ptr = "[date lookup failed]";

    return ptr;
} /* whats_today */

char *what_host ()
{
    static char hostname[HOSTNAMELEN + 1];
    extern int errno;

    if (gethostname (hostname, HOSTNAMELEN) != 0)
	sprintf (hostname, "[gethostname error %d]", errno);
    return hostname;
} /* what_host */



char *must_malloc (bytes, desc)
int bytes;
char *desc;
{
    char *malloc ();
    char *retval = NULL;

    retval = malloc ((unsigned) bytes);
    if (retval == NULL) {
	fprintf (stderr,
		"%s:  Failed to allocate %d bytes for %s, exiting\n",
		this_program, bytes, desc);
	exit (1);
    } /* if retval == NULL */

    return retval;
} /* must_malloc */



int strptrcmp (s1, s2)
char **s1, **s2;
{
/*fprintf (stderr, "Check '%s' = '%s'\n", *s1, *s2);*/
    return strcmp (*s1, *s2);
} /* strptrcmp */

int strptrptrcmp (s1, s2)
char ***s1, ***s2;
{
    return strcmp (**s1, **s2);
} /* strptrptrcmp */

/* Stack Memory Abstract Data Type

	-- Implements stack memory using a non-circular, doubly linked list
	-- Can allocate any amount of memory, but the largest expected
	      block is the input parameter   block_len.   Larger blocks
	      will be allocated if needed, but a warning message will be
	      output.
*/

StackMemoryT *st_init (block_len)
int block_len;
{
    StackMemoryT *mem;
    char *malloc ();

    mem = (StackMemoryT *) malloc (sizeof (StackMemoryT));
    if (mem == NULL) {
	fprintf (stderr,
	    "st_init:  Couldn't allocate stack memory table pointer\n");
	return NULL;
    } /* if mem == NULL */
    mem -> prev = NULL;
    mem -> next = NULL;
    mem -> block_len = block_len;
    if ((mem -> start = malloc ((unsigned) block_len)) == NULL) {
	fprintf (stderr, "st_init:  Couldn't allocate table with %d bytes\n",
		block_len);
	return NULL;
    } /* if mem -> start == NULL */
    mem -> top = mem -> start;
    return mem;
} /* st_init */



char *st_alloc (memptr, bytes)
StackMemoryT **memptr;
int bytes;
{
    char *malloc ();
    char *retval;
    int new_block_len;
    StackMemoryT *mem;

    if (memptr == NULL || *memptr == NULL) {
	fprintf (stderr, "st_alloc:  NULL StackMemory pointer\n");
	return NULL;
    } /* if mem == NULL */
    mem = *memptr;

/* Need to round off to a word size */

    bytes = ((bytes + sizeof (long) - 1) / sizeof (long)) * sizeof (long);
    if (bytes > (new_block_len = mem -> block_len)) {
	fprintf (stderr,
		"st_alloc:  WARNING, block length increasing from %d to %d\n",
		mem -> block_len, bytes);
	new_block_len = bytes;
    } /* if bytes > mem -> block_len */

    if (bytes + (mem -> top - mem -> start) > mem -> block_len) {
	if (mem -> next == NULL || bytes > mem -> next -> block_len) {
	    free_st_chain (mem -> next);  /* if the already-allocated
					     block isn't big enough, lose
					     it and its successors */
	    mem -> next = st_init (new_block_len);
	    if (mem -> next == NULL) {
		fprintf (stderr,
			"st_alloc:  out of memory, no room for %d bytes\n",
			bytes);
		return NULL;
	    } /* if mem -> next == NULL */
	    mem -> next -> prev = mem;
	} /* if mem -> next == NULL || */
	mem = mem -> next;
	mem -> top = mem -> start;
    } /* if bytes + top - start > block_len */

    retval = mem -> top;
    mem -> top += bytes;
    *memptr = mem;
    return retval;
} /* st_alloc */


/* st_in -- check to see if a memory address lies within the stack memory.
   */

int st_in (memptr, loc)
StackMemoryT **memptr;
char *loc;
{
    StackMemoryT *mem;

    if (memptr == NULL) {
	fprintf (stderr, "st_free: NULL StackMemory pointer, loc=%x\n", loc);
	return 0;
    } /* if memptr == NULL */

    for (mem = *memptr; mem; mem = mem -> prev)
	if (mem -> start <= loc && loc < mem -> top)
	    return 1;
    return 0;
} /* st_in */


/* st_free -- free up all stack frames above a certain address.  Note that
   only the current pointer gets adjusted, no memory is actually
   deallocated.  We do this to prevent thrashing on malloc calls.  If
   you need to conserve memory, add a call to free_st_chain as indicated.
   */

int st_free (memptr, loc)
StackMemoryT **memptr;
char *loc;
{
    StackMemoryT *mem;

    if (memptr == NULL) {
	fprintf (stderr, "st_free: NULL StackMemory pointer, loc=%x\n", loc);
	return 0;
    } /* if memptr == NULL */

    for (mem = *memptr; mem; mem = mem -> prev)
	if (mem -> start <= loc && mem -> start + mem -> block_len > loc) {
	    *memptr = mem;
	    mem -> top = loc;

/* DO NOT ADD THIS LINE, unless memory starts getting *really* tight. */
/*	    free_st_chain (mem -> next); mem -> next = NULL; */

	    return 1;
	} /* if mem -> start <= loc */
    return 0;
} /* st_free */



/* free_st_chain -- frees all stack frames from the current one to the
   end, following   next   pointers. */

static free_st_chain (mem)
StackMemoryT *mem;
{
    if (mem != NULL) {
	while (mem -> next != NULL) {
	    free (mem -> start);
	    mem = mem -> next;
	    free (mem -> prev);
	} /* while mem != NULL */

	free (mem);
    } /* if mem != NULL */
} /* free_st_chain */




/* new_string -- Allocate a non-reclaimable copy of   x.   */

char *new_string (x)
char *x;
{
    char *malloc ();
    char *retval;
    static char *bufend = NULL, *ptr = NULL;
    int len = strlen (x);

    if (ptr == NULL || ptr + len > bufend) {
	unsigned size = STRING_BLOCK_SIZE;

	if (len > size)
		size = len + 1;
	ptr = must_malloc ((int) size, "new_string string buffer");
	bufend = ptr + size - 1;
    } /* if ptr + len <= bufend */

/* bufend   points to the last *good* byte in the current block.   ptr
   points to the next *good* byte (i.e. past the previous '\0') */

    strcpy (ptr, x);
    retval = ptr;
    ptr += len + 1;

    return retval;
} /* new_string */

