
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* ARROW.H -- Declarations for types/functions related to data-structures
              for instance arrows.

   $Header: arrow.h,v 1.4 91/02/18 17:31:24 heydon Locked $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   This module implements functions for managing the data structures
   associated with instance arrows. An arrow in this implementation has the
   following parts (types are shown in parentheses): 1) sysname (SysName),
   2) head box (Box *), 3) tail box (Box *), 4) parity (ArrowParity). These
   parts are grouped together in an Arrow structure.

   Arrows are grouped by type (permission). For each Permission, the arrows
   having that permission are stored in a linked list. At the head of each of
   these linked list is an ArrowHead structure. The ArrowHead points to the
   first Arrow of the list, and also contains: 1) the permission (Permission),
   2) the number of positive arrows in the list (int), 3) the number of
   negative arrows in the list (int).

   The ArrowHeads themselves are organized as a linked list, to facilitate
   stepping through all arrows or all arrows of a particular type. The global
   variable "arrow_head_head" defined in arrow.g is a pointer to the first
   ArrowHead in this linked list of ArrowHead's. To quickly find the ArrowHead
   for a particular permission, use the routine FindArrowHeadId(perm) in the
   parse library module id-hash.c.

   WHAT THIS MODULE PROVIDES

   This module defines the following TYPES:
     ArrowParity - possible parity values
     Permission  - access right names
     Arrow       - arrow structure
     ArrowHead   - head of arrow linked list structure

   It provides MACRO DEFINITIONS for accessing the Arrow and ArrowHead
   structures; see the structures and macro definitions below for details.

   It also provides the following FUNCTIONS:
     InitArrows()     - initialize this module; must be called before any
                        other routines in the module
     AddArrow()       - add a new arrow to the data structure

   It provides the following CONVERSION FUNCTIONS related to ARROW
   attributes:
     ConvertParity()  - converts "pos" or "neg" to type (ArrowParity)
*/

#ifndef ARROW_H
#define ARROW_H

#include "box.h"

/* TYPES ================================================================== */

#define NumParities 2		/* number of valid ArrowParity values */
typedef enum { PosParity, NegParity, UnknownParity } ArrowParity;

typedef struct arrow_tag {
    struct arrow_tag *next;	/* pointer to next Arrow in linked list */
    SysName a_sys_name;		/* arrow's sysname (for identification) */
    Box *head,*tail;		/* pointers to boxes at ends of this arrow */
    ArrowParity parity;		/* parity of arrow */
} Arrow;

typedef String Permission;	/* arrow permission types are strings */

typedef struct arrow_head_tag {
    struct arrow_head_tag *next;/* pointer to next ArrowHead in linked list */
    Permission perm;		/* name of permission type, e.g. "read" */
    Arrow *first_arrow;		/* points to first element of linked list */
    int pos_cnt,neg_cnt;	/* number of pos/neg arrows of this type */
} ArrowHead;

/* MACRO FUNCTIONS ======================================================== */

/* Arrow access functions */
#define ASysNameOf(_arrow_ptr)        ((_arrow_ptr)->a_sys_name)
#define HeadOf(_arrow_ptr)            ((_arrow_ptr)->head)
#define TailOf(_arrow_ptr)            ((_arrow_ptr)->tail)
#define ParityOf(_arrow_ptr)          ((_arrow_ptr)->parity)

/* ArrowHead access functions */
#define PermOf(_arrow_head_ptr)       ((_arrow_head_ptr)->perm)
#define FirstArrowOf(_arrow_head_ptr) ((_arrow_head_ptr)->first_arrow)
#define PosCntOf(_arrow_head_ptr)     ((_arrow_head_ptr)->pos_cnt)
#define NegCntOf(_arrow_head_ptr)     ((_arrow_head_ptr)->neg_cnt)

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

void InitArrows( /* void */ );
/* Initialize the arrow.c module. This routine must be called before any other
   routines in this module.
*/

void BeginMultipleArrows( /* void */ );
/* Declare that subsequent calls to AddArrow() (up until the time that the
   function EndMultipleArrows() is called) will contain identical sysnames.
   Only the *first* arrow is checked to see that an arrow with its sysname
   does not already exist. The check is disabled on the second and following
   calls to AddArrow(), up until the call to EndMultipleArrows().
*/

void EndMultipleArrows( /* void */ );
/* Turn duplicate sysname checking (disabled by BeginMultipleArrows()) back
   on.
*/

void AddArrow(
       /* SysName sys_name, Box *head, Box *tail, ArrowParity parity,
	  Permission perm, int line_no */ );
/* Add a new arrow to the arrow data structures. The new arrow is assigned
   SysName 'sys_name', head and tail boxes 'head' and 'tail' respectively, and
   ArrowParity 'parity'. The arrow will be grouped with other arrows having
   Permission 'perm'. The parameter 'line_no' should be the first line number
   (in the IFF input) of the ARROW entry (this is used for error reporting, if
   necessary).

   An error is reported if an Arrow with SysName 'sys_name' already exists.
   An error is also reported if the box '*tail' is not a user box or if
   '*head' is not a file box. In either case, no arrow is added.

   If this is the first arrow added with Permission 'perm', a new ArrowHead
   for that Permission is created.
*/

void ConvertParity( /* PropVal *p_val_ptr, Generic *val_ptr, int line_no */ );
/* Assumes 'p_val_ptr' points to a PropVal with a PropValTypeOf IdPValType
   and sets *((ArrowParity *)val_ptr) to PosParity or NegParity depending on
   whether IdValOf(p_val_ptr) is "pos" or "neg"; an error is reported, and
   *p_val_ptr is set to UnknownParity, otherwise.
*/

#ifdef MACRO_FUNCTION

String StringParity( /* ArrowParity par */ );
/* RETURNS "pos" if 'par' is PosParity, "neg" if 'par' is "NegParity", and
   "unknown" otherwise.
*/

#endif MACRO_FUNCTION

#define StringParity(_par)\
   ((_par)==PosParity ? "pos" :\
    ((_par)==NegParity ? "neg" : "unknown"))

#ifdef DEBUG
void ShowArrows( /* void */ );
/* Display the current state of the arrow data structures. For each
   Permission, the number of positive and negative arrows of that Permission
   are displayed. Then, for each arrow of that Permission, the SysName, head
   box's index, tail box's index, and parity are displayed.
*/
#endif DEBUG

#endif ARROW_H
