
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* EXTRACT.H -- Declarations for types/functions related to data-structures
                for extracting info from a parse tree Entry.

   $Header: extract.h,v 1.7 91/06/21 16:51:35 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   This module provides a general mechanism for extracting data from parse
   trees produced by the "parse" library into program variables that the
   programmer can then use. For example, from a given "INSIDE" Entry, we need
   to extract the sysname of the parent box, and the list of sysnames of child
   boxes. We need to extract these SysName's, and then use them in calls to
   assert that the parent box directly contains all listed child boxes. For
   each attribute, we may need to write a "conversion function" for converting
   the value associated with the attribute to a value we can assign to some
   designated program variables of type SysName.

   Since we don't know how many items may be in a attribute value consisting
   of a list, we need a separate mechanism for extracting data from non-list
   and list attribute values. The two mechanisms provided by this module are
   MatchPNames() and MatchNextListElement().

   Before we can extract data from an Entry, we need to specify:
     1) the names of attributes in that kind of Entry,
     2) which attributes are required to have values (those required to have
        values but not having a value should be reported as errors),
     3) the conversion function to apply to extract data from each attribute,
     4) where to put the extracted data in the event that we hope to extract a
        single value, and
     5) where to put the extracted list pointer in the event that we hope to
        extract a list.
   These 5 pieces of information associated with *each* attribute are bound up
   in a defined type called a PNameDesignator. Notice that either of (4) or
   (5) can be NULL (but not both); a non-NULL entry for (4) means we hope to
   extract a value, and a non-NULL entry for (5) means we hope to extract a
   list. If only one of them is non-NULL, but the attribute value itself is
   the other possibility, an error is reported. However, if *both* (4) and (5)
   are non-NULL, then the type of the attribute value (i.e., list or not)
   determines whether pointer (4) or (5) is used. See the documentation for
   MatchPNames() below for details.

   To specify the PNameDesignators (one for each attribute of an Entry), we
   first call StartNewPNameList(). Then, for each attribute, we call
   AddPNameDesignator(). This list of attributes remains in effect until the
   next call to StartNewPNameList().

   To extract information from an Entry, we then call MatchPNames(), passing
   it a pointer to the entry. This one call converts all the attributes
   specified in the current list of PNameDesignators, and prepares those
   having lists as values for conversion. This routine also reports errors if
   some required attribute was not found in the Entry. For optional
   attributes, we can use the routine ValFoundP() to determine if that
   attribute was matched or not.

   Lists are treated specially. Item (5) above is indicated by passing either
   NULL_LIST (to indicate that a list value is not expected) or a pointer to a
   structure of type ListDesc. When MatchPNames() matches an attribute value
   that *is* a list against a PNameDesignator having a non-NULL ListDesc
   field, it initializes that ListDesc by making the NextListEntryPtrOf()
   field of the ListDesc point to the first ListEntry of the list. Each
   successive call to MatchNextListEntry() then converts the next element in
   the list. See MatchNextListEntry() for details.

   See convert.h for a description of the conversion functions.

   WHAT THIS MODULE PROVIDES

   This module defines the following TYPES:
     ListDesc        - a list descriptor; used to extract data from attributes
                       having a list as a value
     PNameDesignator - a structure used to convert a single property name (or
                       attribute)

   It provides MACRO DEFINITIONS for accessing fields of the ListDesc and
   PNameDesignator structures, as well as the following macro definition:
     NULL_LIST       - a NULL (ListDesc *), passed to AddPNameDesignator if
                       the value associated with an attribute is not expected
		       to be a list.

   It provides the following FUNCTIONS:
     InitExtract()          - initialize this module; must be called before
                              all other routines in this module
     StartNewPNameList()    - start a new list of PNameDesignators
     AddPNameDesignator()   - add a new PNameDesignator to the current list
     ValFoundP()            - indicate if a particular attribute was matched
     MatchPNames()          - attempt to match all attributes in the list of
                              current PNameDesignators
     MatchNextListElement() - convert the next element in the specified
                              ListDesc
*/

#ifndef EXTRACT_H
#define EXTRACT_H

#include <my-types.h>
#include "parser.h"
#include "convert.h"

/* TYPES ================================================================== */

typedef struct {
    Boolean matched_p;		    /* if a match against a list succeeded */
    struct p_name_designator *desig_ptr; /* pointer to associated desig */
    int start_line_number;	    /* line on which attribute name appears */
    ListEntry *next_list_entry_ptr; /* pointer to next PropVal to convert */
} ListDesc;

typedef struct p_name_designator {
    String p_name;		/* attribute name */
    ConvertFunc convert_func;	/* pointer to conversion function to call */
    Generic *val_ptr;		/* point to value to fill in */
    ListDesc *list_desc_ptr;	/* non-NULL iff we are converting a list */
    Boolean val_found_p;	/* has this value been found? */
    Boolean val_required_p;	/* is this value is required? */
} PNameDesignator;

/* MACRO FUNCTIONS ======================================================== */

/* ListDescriptor access functions */
#define MatchedPOf(_l_desc_ptr)         ((_l_desc_ptr)->matched_p)
#define DesigPtrOf(_l_desc_ptr)         ((_l_desc_ptr)->desig_ptr)
#define StartLineNumberOf(_l_desc_ptr)  ((_l_desc_ptr)->start_line_number)
#define NextListEntryPtrOf(_l_desc_ptr) ((_l_desc_ptr)->next_list_entry_ptr)

/* PNameDesignator access functions */
#define PNameOf(_desig_ptr)             ((_desig_ptr)->p_name)
#define ConvertFuncOf(_desig_ptr)       ((_desig_ptr)->convert_func)
#define ValPtrOf(_desig_ptr)            ((_desig_ptr)->val_ptr)
#define ListDescPtrOf(_desig_ptr)       ((_desig_ptr)->list_desc_ptr)
#define ValFoundPOf(_desig_ptr)         ((_desig_ptr)->val_found_p)
#define ValRequiredPOf(_desig_ptr)      ((_desig_ptr)->val_required_p)

/* NULL ListDesc pointer */
#define NULL_LIST                       ((ListDesc *)NULL)

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

void InitExtract( /* int max_size */ );
/* Initialize this module, and allocate space for the list of
   PNameDesignators. This routine must be called before any other routine in
   this module. The parameter 'max_size' should be the size of the largest
   PNameList (i.e., the largest number of calls to AddPNameDesignator()
   between any two calls to StartNewPNameList()).
*/

void StartNewPNameList( /* void */ );
/* Indicate that subsequent calls to AddPNameDesignator() should create
   PNameDesignators of a *new* list.
*/

void AddPNameDesignator( /* String p_name, Boolean required, ConvertFunc f,
			    Generic *v_ptr, ListDesc *list_desc_ptr */ );
/* Adds a PNameDesignator for the property named 'p_name' to the current list
   of PNameDesignators. The list of PNameDesignators is used by later calls to
   MatchPNames() and MatchNextListElement().

   The 'required' value indicates whether or not this property name is
   mandatory in an entry; if the value is True, but no such property name
   appears in the entry, MatchPNames() will report an error.

   The parameter 'f' is a pointer to the function to be called by these
   routines to convert property values. It is called to convert both single
   values and top-level values within lists.

   The 'v_ptr' parameter points to a location 'f' will use to set a value.
   If 'v_ptr' is NULL, an error will be issued by MatchPNames() in the case
   where the attribute value is not a list. See MatchPNames() below for
   details.

   The parameter 'list_desc_ptr' should be non-NULL if the property value
   associated with p_name is expected to be a list. If the value
   'list_desc_ptr' is non-NULL, and if the attribute value is matched against
   a list, the DesigPtrOf(list_desc_ptr) is initialized to the added
   PNameDesignator. If 'list_desc_ptr' is NULL_LIST, an error will be issued
   by MatchPNames() in the case where the attribute value is a list. See
   MatchPNames() below for details.

   NOTE: It is a bad idea for both 'v_ptr' and 'list_desc_ptr' to be NULL. If
   they are, this attribute will never be matched. See the description of
   MatchPNames() below to understand how these parameters are used.
*/

Boolean ValFoundP( /* String p_name */ );
/* RETURNS the ValFoundPOf() value of the designator in the current list
   of PNameDesignators having the name 'p_name'. If no designator in the
   current list has name 'p_name', RETURN False.

   Note: An alternative, faster means of accomplishing the same result is
   possible if the designator in question has a non-NIL ListDescPtrOf() value.
   If the designator was created with the ListDesc *list_desc_ptr, then
   ValFoundP() can be computed faster by using the following expression:
   ValFoundPOf(DesigPtrOf(list_desc_ptr)).
*/

Boolean MatchPNames( /* Entry *entry_ptr */ );
/* Scans the attribute list associated with '*entry_ptr' and tries to match
   each attribute against the current list of PNameDesignators. There are two
   possibilities, based on the type of the value read:

   1) The value is not a list. If a NULL 'v_ptr' was specified, an error is
   reported. Otherwise, the conversion function 'f' is called to set 'v_ptr'
   to contain the value read.

   2) The value is a list. If a NULL 'list_desc_ptr' was specified, an error
   is reported. Otherwise, the ListDescPtrOf() the designator is initialized.
   Values can then be extracted from the list by subsequent calls to
   MatchNextListElement().

   Note that if *both* the 'v_ptr' and 'list_desc_ptr' of a particular
   attribute are non-NULL, then the action taken depends on whether or not the
   value read was a list. In this case, the programmer can tell which was the
   case by testing MatchedPOf(list_desc_ptr); if it is True, a list was
   matched, otherwise a non-list (single value) was matched.

   After all attributes associated with the entry have been checked, errors
   are reported for each of the *required* attributes in the current
   PNameDesignator list which were not found in '*entry_ptr'.

   If a match is not found for a particular PNameDesignator, then the values
   of *ValPtrOf() and *ListDescPtrOf() (if applicable) for that designator
   may be garbage. The best way to tell if this is the case is to call
   ValFoundP(p_name) for the designator with property name 'p_name'.

   RETURNS True iff errors were found while matching this Entry.
*/

Boolean MatchNextListElement( /* ListDesc *list_desc, Generic *v_ptr */ );
/* Calls the conversion function associated with 'list_desc' on the next
   top-level list element, and stores the result in '*v_ptr'. Also advances
   the pointer NextListEntryPtrOf(list_desc) point to the next ListEntry in
   the list.

   PRE-CONDITION: NextListEntryPtrOf(list_desc) != NULL

   RETURNS True iff errors were found while matching this list element.
*/

#endif EXTRACT_H
