
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* ID-HASH.H -- Declarations for types/functions related to data-structures
                dealing with hashing of identifiers (strings).

   $Header: id-hash.h,v 1.8 91/08/13 14:44:56 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   This module implements a single hash table for identifers. Three kinds of
   identifiers can be stored in the hash table:
     1) IDDropType = Identifier
        Identifiers expected to be encountered in the input file, such as
        names of expected entries and attributes; these are added before the
        input file is parsed.
     2) IdDropType = ArrowType
        Names of arrow permission types; these are added as they are
        encountered during the parse.
     3) IdDropType = InputId
        Identifiers which are property values; these are also added as they
        are encountered during the parse.

   The hash table follows the "bucket hashing" paradigm, where each bucket is
   implemented as a linked list of structures of type IdDrop. Each drop
   contains the name of the identifier and its type (a value of type
   IdDropType). In addition, the drops of type (1) and (2) also have an
   auxiliary value stored with them. In case (1), the value is a bit field of
   type IdType indicating properties of the identifier. In case (2), the value
   is a pointer to an ArrowHead (see arrow.h), which is a structure pointing
   to the linked list of Arrows of that type.

   Type (1) identifiers are associated with a bit-field (type IdType) which
   specifies:
     a) Whether the ID is an entry name or a property name (bits 0-1).
     b) If an entry name, a bit to indicate the EntryType. If a property name,
        bits to indicate which EntryType's this PName can be used with (bits
	4-8).
     c) If a property name, bits to indictate which type(s) of values
        (integer, identifier, quoted string, or list) to recognize in
	association with this property name (bits 16-19).

   One of the bits of part (c) can be set to indicate that a list is allowed
   to appear. Other PropValType bits should be set to indicate which types of
   values are allowed to appear in the lists. If the ListPValType bit is set,
   then a list is allowed to appear again at *any* level; each non-list value
   must be of a type such that the corresponding PropValType bit is set. For
   example, if the value for (c) were (IntPValType | ListPValType) then the
   following property values would be legal:
     2, { 2 }, { { { { 3 } } } }, { 2, 5 }, { { 2, 3 }, { 3, { 4, { 5 } } } }
   but these values would not:
     foo, { 2, foo }, { { 4, 6, abc } }.
     
   Type (2) identifiers are associated with a pointer to the head of a linked
   list of Arrow's which are all have the same type as the ID.

   The parser builds a data structure corresponding to the input file. The
   data that should be saved may vary depending on the kind of parser.
   Associated with each ID (either EntryName's or PName's) are bits specifying
   which parser(s) should recognize (i.e., save data associated with) it.
   This is not an optimal way to do things because an ID used with multiple
   EntryName's may be needed for one but not the other, and there is no way
   to specify that with this data-structure, but it is a start.

   WHAT THIS MODULE PROVIDES

   This module defines the following TYPES:
     IdType     - a bit-field value associated with type-(1) identifiers
     IdDropType - the type of each entry in the table
     IdDrop     - structure corresponding to an entry in the table

   It provides MACRO DEFINITIONS for accessing fields of the IdDrop structure,
   as well as the following macro definitions: 
     EntryType   - denotes type of entry if ID is an entry name; this is an
                   alias for IdType
     PropValType - denotes types allowed to be associated with this ID if it
                   is a property name; this is an alias for IdType
     IsEntryNameP - predicate indicating if the EntryName bit of an IdType
                    is set or not.
     IsPNameP     - predicate indicating if the PropertyName bit of an IdType
                    is set or not.
   See the source below for tag names associated with IdType's and for other
   useful definitions.

   It provides the following FUNCTIONS:
     InitIdHashTable() - Initialize this module; must be called before any
                         other functions in this module.
     AddIdDrop()       - Add an identifier to the hash table.
     FindIdDrop()      - Search for an identifier in the hash table.
     FindOrAddIdDrop() - Search for an identifier in the hash table, and add
                         it if it is not there.
     FindArrowHeadId() - Search for an identifier with IdDropType == ArrowType.
     AddArrowHeadId()  - Add an identifier with IdDropType == ArrowType.
*/

#ifndef ID_HASH_H
#define ID_HASH_H

#include <my-types.h>

/* DEFINITIONS ============================================================ */

/* Define aliases to make EntryType and PropValType = IdType */
#define EntryType	IdType
#define PropValType	IdType

/* Define aliases for PName's when used to designate an EntryType */
#define NumEntryTypes	6
#define AllEntryTypes	AllPNames
#define BoxEntry        BoxPName
#define ArrowEntry      ArrowPName
#define InsideEntry     InsidePName
#define OverlapsEntry   OverlapsPName
#define EditorEntry     EditorPName
#define TypeEntry       TypePName
#define PermEntry       PermPName

/* Definitions for common OR's of IdType's */
#define ObjectPNames    (BoxPName | ArrowPName)
#define AorEPNames      (ArrowPName | EditorPName)

/* Composite ListPVal's */
/* When the ListPVal bit is set, any elements in the this (or lists of lists
   of lists... of this list) must be of some type such that the
   corresponding IntPVal, IdPVal, or StringPVal bit is set. */
#define IntListPValType    (IntPValType | ListPValType)
#define IdListPValType     (IdPValType | ListPValType)
#define StringListPValType (StringPValType | ListPValType)
#define AnyListPValType    (IntPValType | IdPValType | \
			    StringPValType | ListPValType)

/* TYPES ================================================================== */

typedef enum {
	/* Special bits [0-3] ------------------------------------ */
		None             =        0x0,
		AllSpecial       =        0xf,
		EntryName        =        0x1,
		PropertyName     =        0x2,
	/* Property Name bits [4-15] ------------ (type EntryType) */
		AllPNames        =     0xfff0,
		BoxPName         =       0x10,
		ArrowPName       =       0x20,
		InsidePName      =       0x40,
		OverlapsPName    =       0x80,
		EditorPName      =      0x100,
		TypePName        =      0x200,
		PermPName        =      0x400,
	/* Property Value Type bits [16-23] --- (type PropValType) */
		AllPValTypes     =   0xff0000,
		IntPValType      =    0x10000,
		IdPValType       =    0x20000,
		StringPValType   =    0x40000,
		ListPValType     =    0x80000,
} IdType;

typedef enum {
    Identifier, InputId, ArrowType
} IdDropType;

#define IdDropTypeToString(_type)\
    ((_type)==Identifier?"ID":((_type)==InputId?"InputId":"Arrow"))

typedef struct id_drop_tag {
    struct id_drop_tag *next;
    String id_name;		/* name of identifier */
    IdDropType drop_type;	/* switch for how the union should be
				   interpretted */
    union {
	IdType id_types;	/* type of the identifer if
				   drop_type == Identifier */
	Generic *arrow_head_ptr; /* pointer to head of linked list if
				    drop_type == ArrowType */
    } val;
} IdDrop;

/* MACRO FUNCTIONS ======================================================== */

/* Macros to access IdDrop's */
#define IdNameOf(_id_drop_ptr)       ((_id_drop_ptr)->id_name)
#define DropTypeOf(_id_drop_ptr)     ((_id_drop_ptr)->drop_type)
#define IdTypesOf(_id_drop_ptr)      ((_id_drop_ptr)->val.id_types)
#define ArrowHeadPtrOf(_id_drop_ptr) ((_id_drop_ptr)->val.arrow_head_ptr)

/* Predicates of an IdType argument */
#define IsEntryNameP(_id_type) ((int)_id_type & (int)EntryName)
#define IsPNameP(_id_type)     ((int)_id_type & (int)PropertyName)

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

void InitIdHashTable( /* void */ );
/* Initialize the HashTable 'id_ht' so it is empty.
*/

void ShutDownIdHashTable( /* void */ );
/* Free any space allocated to entries in the HashTable 'id_ht'.
*/

void AddId( /* String name, IdType id_types */ );
/* Add the 'Identifier' with name 'name' of type 'id_types' (this parameter
   will be an OR of some of the IdTypes) to the HashTable 'id_ht'. This
   function should only be called if the caller is sure that the ID does not
   already exist in the hash table.
*/

IdDrop *AddIdDrop( /* String name, IdDropType drop_type */ );
/* Add the identifier with name 'name' to the HashTable 'id_ht', and set its
   IdDropType to 'drop_type'. This function should only be called if the
   caller is sure that the ID does not already exist in the hash table.
   RETURNS a pointer to the newly created IdDrop in the hash table.

   NOTE: the 'val' field of the IdDrop is not set by this function.
*/

IdDrop *FindIdDrop( /* String name, IdDropType drop_type */ );
/* RETURNS a pointer to the IdDrop of type 'drop_type' in the HashTable
   'id_ht' with name 'name'; NULL if none exists in the HashTable.
*/

String FindOrAddIdDrop( /* String name, IdDropType drop_type */ );
/* Search for the identifier with name 'name' & IdDropType 'drop_type' in the
   HashTable 'id_ht'; if not found, add the identifier using AddIdDrop().

   Whether the identifier is found or added, RETURNS a pointer to the IdDrop
   as it is stored in the hash table.
*/

#ifdef DEBUG
void ShowIdHashTable( /* void */ );
/* Display the current contents of the HashTable 'id_ht'. For each non-empty
   bucket in the table, the index of and number of IdDrops in the bucket is
   displayed. Then, for each IdDrop in that bucket, its name and type is
   printed. Furthermore, if the IdDropType of the IdDrop is Identifier or
   ArrowType, the value associated with the IdDrop is displayed (in hex).
   Finally, the total number of entries in the hash table is displayed.
*/
#endif

#endif
