
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* PARSER.H -- Declarations for types/functions/variables related to
               parsing the input.

   $Header: parser.h,v 1.7 91/06/21 16:15:23 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   This is the top-level module for constructing a parse tree from an IFF
   file. An IFF file is composed of a sequence of entries, each of which
   starts with ">ENTRYNAME", where ENTRYNAME is something like BOX or ARROW.
   Each entry header in the IFF file is followed by a (possibly empty) list of
   attribute value pairs taking the form "attribute = value;".

   The parse tree corresponding to an IFF file is a linked-list of items of
   type Entry, where the list appears in the same order as the entries
   appearing in the IFF file. Each Entry has the following data associated
   with it:
      1) the line number in the IFF file on which the entry's name occurs,
      2) the name of the entry,
      3) the coded type of the entry, and
      4) a pointer to a linked list of items of type AttrEntry

   Each AttrEntry encodes one attribute-value pair. The attributes are also
   known as properties, so the attribute itself is sometimes called an
   "attribute name" or "property name", and the value is sometimes called an
   "attribute value" or "property value". Each AttrEntry has the following
   data associated with it:
      1) the line number in the IFF file on which the attribute name occurs,
      2) the attribute (or property) name itself, and
      3) a pointer to the attribute (or property) value of type PropVal.

   Each PropVal encodes a single property value. Property values can be one of
   several types (as defined in id-hash.h): integers (IntPValType),
   identifiers (IdPValType), strings (StringPValType), or lists
   (ListPValType). Each PropVal has the following data associated with it:
      1) the type of the property value, and
      2) the value itself
   Item (2) will vary depending on the value of item (1). In particular, if
   the value of item (1) is ListPValType, then item (2) will be a pointer to
   a linked list of items of type ListEntry. Each ListEntry points to another
   PropVal that is the value of that list item. In this way, lists can be
   nested indefinitely.

   To use this module:
      1) call InitParser() to initialize all modules in the parse library,
      2) use calls to AddEntryName() and AddPropName() to install identifiers
         that should be recognized during the parse,
      3) call ParseFile() to parse a particular IFF file and get back its
         corresponding parse tree, and
      4) call ShutDownParser() to shut down all modules in the parse library.

   WHAT THIS MODULE PROVIDES

   This module defines the following TYPES:
     ListEntry - backbone structure for entries of a list
     PropVal   - structure for an attribute (or property) value
     AttrEntry - structure for an attribute-value pair
     Entry     - structure for an IFF file entry

   In addition to MACRO DEFINITIONS for accessing the fields of the ListEntry,
   PropVal, AttrEntry, and Entry structures, it provides the following:
     STRING_BUFF_SIZE - the maximum string size accepted as a string value
                        associated with an attribute

   The following MACRO FUNCTIONS are provided:
     AddEntryName()     - Add an Entry Name to a Parser structure
     AddPropName()      - Add a Property Name to a Parser structure

   It provides the following FUNCTIONS:
     InitParser()       - Initialize all modules in this library; must be
                          called before any other functions in this module.
     ParseFile()        - Parse a particular IFF FILE, and return the
                          corresponding parse tree.
     ShutDownParser()   - Shut down all modules in this library.
*/

#ifndef PARSER_H
#define PARSER_H

#include <stdio.h>
#include <my-types.h>
#include "id-hash.h"

/* TYPES ================================================================== */

typedef struct prop_val {
    PropValType prop_val_type;
    union prop_val_val {
	int int_val;
	String id_val;
	String string_val;
	struct list_entry *list_head;
    } val;
} PropVal;

typedef struct list_entry {
    struct list_entry *next;	/* must be first field */
    PropVal *list_prop_val_ptr; /* PropVal of this list item */
} ListEntry;

typedef struct attr_entry {
    struct attr_entry *next;	/* must be first field */
    int attr_line_number;	/* line number where the attr name appeared */
    String attr_name;		/* attribute (property) name */
    PropVal *prop_val_ptr;	/* property value associated w/ this attr */
} AttrEntry;

typedef struct entry {
    struct entry *next;		/* must be first field */
    int entry_line_number;	/* line number where the entry name appeared */
    String entry_name;		/* literal name of the entry */
    EntryType entry_type;	/* coded type of the entry */
    AttrEntry *attr_list_head;	/* pointer to head of attribute list */
} Entry;

typedef struct generic_entry {
    struct generic_entry *next; /* all entries have a next field as their 1st
				   item */
} GenericEntry;

/* lists of the above structures */

typedef struct {
    ListEntry *first,*last;
} PropValList;

typedef struct {
    AttrEntry *first,*last;
} AttrEntryList;

typedef struct {
    Entry *first,*last;
} EntryList;

typedef struct {
    GenericEntry *first,*last;
} GenericList;

/* DEFINITIONS ============================================================ */

/* maximum string size accepted as a string-value associated with an attr */
#define STRING_BUFF_SIZE 80

/* MACRO FUNCTIONS ======================================================== */

/* PropVal access functions */
#define PropValTypeOf(_prop_val_ptr)  ((_prop_val_ptr)->prop_val_type)
#define IntValOf(_prop_val_ptr)       ((_prop_val_ptr)->val.int_val)
#define IdValOf(_prop_val_ptr)        ((_prop_val_ptr)->val.id_val)
#define StringValOf(_prop_val_ptr)    ((_prop_val_ptr)->val.string_val)
#define ListHeadOf(_prop_val_ptr)     ((_prop_val_ptr)->val.list_head)

/* ListEntry access functions */
#define ListPropValPtrOf(_list_ptr)   ((_list_ptr)->list_prop_val_ptr)

/* AttrEntry access functions */
#define AttrLineNumberOf(_attr_ptr)   ((_attr_ptr)->attr_line_number)
#define AttrNameOf(_attr_ptr)         ((_attr_ptr)->attr_name)
#define PropValPtrOf(_attr_ptr)       ((_attr_ptr)->prop_val_ptr)

/* Entry access functions */
#define EntryLineNumberOf(_entry_ptr) ((_entry_ptr)->entry_line_number)
#define EntryNameOf(_entry_ptr)       ((_entry_ptr)->entry_name)
#define EntryTypeOf(_entry_ptr)       ((_entry_ptr)->entry_type)
#define AttrListHeadOf(_entry_ptr)    ((_entry_ptr)->attr_list_head)

/* List access functions */
#define FirstOf(_list_ptr)            ((_list_ptr)->first)
#define LastOf(_list_ptr)             ((_list_ptr)->last)

/* void AddEntryName(name,id_type)
     String name;
     IdType id_type;
   Adds the identifier 'name' as an EntryName to the HashTable 'id_ht',
   specifying that it is of type 'id_type'.
*/
#define AddEntryName(_name,_id_types)\
   AddId(_name,(EntryName|(_id_types)))

/* void AddPropName(name,id_types,pval_types)
     String name;
     IdType id_types;
     PropValType pval_types;
   Adds the identifier 'name' as a PName to the HashTable 'id_ht', specifying
   that it is associated with a entry types 'id_types' and that it should have
   corresponding PropValTypes 'pval_types'.
*/
#define AddPropName(_name,_id_types,_pval_types)\
   AddId(_name,(PropertyName|(_id_types)|(_pval_types)))

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

void InitParser( /* void */ );
/* Initialize the parser library and all its modules.
*/

Entry *ParseFile( /* FILE *fp */ );
/* Parses the FILE 'fp' and RETURNS a pointer to the resulting parse tree iff
   no parsing errors occurred; NULL otherwise.
*/

PropVal *CopyPropVal( /* PropVal *pv */ );
/* Return a copy of the PropVal 'pv' and all its associated structure.
*/

void FreeParseTree( /* Entry *p_tree */ );
/* Free all memory associated with the parse tree 'p_tree'. This frees all
   structures forming the parse tree with the exception of identifiers in the
   parse tree that are actually stored in the ID hash table.
*/

void ShutDownParser( /* void */ );
/* Shut down the parser module and all other modules in the parser library. */

#ifdef DEBUG
void ShowParseTree( /* Entry *p_tree */ );
/* Displays the contents of the parse tree 'p_tree'. For each Entry in the
   linked list of Entry's starting with 'p_tree', the Entry's type, line
   number, and attribute-value pairs are printed. Each attribute-value pair is
   printed as a line number, the attribute name, and the attribute value.
*/
#endif

#endif
