
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* parse_args

	This function will parse command line input into appropriate data
   structures, output error messages when appropriate and provide some
   minimal type conversion.

	Input to the function consists of the standard   argc,argv
   values, and a table which directs the parser.  Each table entry has the
   following components:

	prefix -- the (optional) switch character string, e.g. "-" "/" "="
	switch -- the command string, e.g. "o" "data" "file" "F"
	flags -- control flags, e.g.   P_CASE_INSENSITIVE, P_REQUIRED_PREFIX
	arg_count -- number of arguments this command requires, e.g. P_NO_ARGS
		     for booleans, P_ONE_ARG for filenames, P_INFINITE_ARGS
		     for input files
	result_type -- how to interpret the switch arguments, e.g. P_STRING,
		       P_CHAR, P_FILE, P_OLD_FILE, P_NEW_FILE
	result_ptr -- pointer to storage for the result, be it a table or
		      a string or whatever
	table_size -- if the arguments fill a table, the maximum number of
		      entries; if there are no arguments, the value to
		      load into the result storage

	Although the table can be used to hold a list of filenames, only
   scalar values (e.g. pointers) can be stored in the table.  No vector
   processing will be done, only pointers to string storage will be moved.

	An example entry, which could be used to parse input filenames, is:

	"-", "o", 0, oo, OLD_FILE, infilenames, INFILE_TABLE_SIZE

*/

#include <stdio.h>
#include <mwm.h>
#include <math.h>		/* For atof */
#include <ctype.h>		/* For isspace */
#include <strings.h>		/* for strcat, strcpy */

/* These three include files are system-dependent.  They are used to
   verify the status of filename arguments */

#include <sys/types.h>		/* For stat(2) definitions */
#include <sys/stat.h>
#include <sys/errno.h>		/* For error codes */

#define MAX_INPUT_SIZE 1000		/* CHANGE THIS!!!  Make it the
					   maximum reasonable value (like
					   100,000) which nothing should
					   ever exceed.  Use 1000 as the
					   default malloc'ed size, but
					   free and remalloc to allow for
					   input under 100,000 */

#define arg_prefix(x) ((x).prefix)
#define arg_string(x) ((x).string)
#define arg_flags(x) ((x).flags)
#define arg_count(x) ((x).count)
#define arg_result_type(x) ((x).result_type)
#define arg_result_ptr(x) ((x).result_ptr)
#define arg_table_size(x) ((x).table_size)


char *lower_string (/* char [], char * */);

static char *this_program = "";
static int bad_input_count = 0;


boolean parse_args (argc, argv, table, table_size, others, other_count)
int argc;
char *argv[];
arg_info table[];
int table_size;
char *others[];
int other_count;
{
    boolean arg_verify (/* table, table_size */);
    void init_store (/* table, table_size */);

    boolean result = FALSE;

    this_program = "";
    bad_input_count = 0;

    if (argv)
	this_program = argv[0];

/* Check the validity of the table and its parameters */

    result = arg_verify (table, table_size);

/* Initialize the storage values */

    init_store (table, table_size);

    if (result) {
	boolean use_prefix = TRUE;	/* use_prefix   is TRUE when a
					   prefix MUST be read; that is,
					   when the start of a command
					   line argument is being parsed */

	argc--;
	argv++;
	while (argc > 0) {
	    int index, length;

	    index = match_table (*argv, table, table_size, use_prefix,
		    &length);
	    if (index < 0) {

/* The argument doesn't match anything in the table */

		if (others) {

/* Might want to filter out those strings which appear after earlier
   switches in the current word.  That is, only treat argv as valid if
   use_prefix   is true.  Right now, any extra chars will be passed on */

		    if (other_count > 0) {
			*others++ = *argv;
			other_count--;
		    } else {
			fprintf (stderr, "%s:  too many parameters: ",
				this_program);
			fprintf (stderr, "'%s' ignored\n", *argv);
		    } /* else */
		} /* if (others) */
		argv++;
		argc--;
	    } else {

/* A match was found;   length   is the length of the switch that was
   matched */

		if (length >= strlen (*argv)) {
		    argc--;
		    argv++;
		    use_prefix = TRUE;
		} else {
		    (*argv) += length;
		    use_prefix = FALSE;
		} /* else */

/* Parse any necessary arguments */

		if (arg_count (table[index]) != P_NO_ARGS) {
		    extern int parse_arg (/* char *, arg_info * */);

/* Check for presence of actual argument */

		    if (argc <= 0) {
			char *ty2str ();

			fprintf (stderr, "%s:  Missing %s after %s%s switch\n",
				this_program, ty2str (arg_result_type
				(table[index]), (char *) NULL), arg_prefix
				(table[index]),
				arg_string (table[index]));
			continue;
		    } /* if argc <= 0 */

/* Now   length   will be used to store the number of parsed characters in
   the argument */

		    length = parse_arg (*argv, &table[index]);

/* BUG???? Why is this test here?  Shouldn't the *argv == NULL test be
   moved up and merged with "if (argc <= 0)"?    mwm 29 Oct 89 */

		    if (*argv == NULL)
			argc = 0;
		    else if (length >= strlen (*argv)) {
			argc--;
			argv++;
			use_prefix = TRUE;
		    } else {
			(*argv) += length;
			use_prefix = FALSE;
		    } /* else */
		} /* if (argv_count != P_NO_ARGS) */
		  else
		    *arg_result_ptr(table[index]) =
			    arg_table_size(table[index]);
	    } /* else */
	} /* while (argc) */
    } /* if (result) */

    if (result && bad_input_count)
	result = -bad_input_count;

    return result;
} /* parse_args */


char *ty2str (type, buf)
int type;
char *buf;
{
    static char store[100];

    if (buf == NULL)
	buf = store;

    switch (type) {
	case P_STRING:	strcpy (buf, "String");		break;
        case P_CHAR:	strcpy (buf, "Character");	break;
	case P_SHORT:
	case P_LONG:
	    if (type == P_INT)
		strcpy (buf, "Integer");
	    else if (type == P_SHORT)
		strcpy (buf, "Short Int");
	    else
		strcpy (buf, "Long Int");
	    break;
	case P_FILE:	strcpy (buf, "Filename");	break;
	case P_OLD_FILE:strcpy (buf, "Existing Filename");break;
	case P_NEW_FILE:strcpy (buf, "Unused Filename");break;
	case P_FLOAT:	strcpy (buf, "Float");		break;
	case P_DOUBLE:	strcpy (buf, "Double");		break;
	default:
	    fprintf (stderr, "%s:  bad type spec in ty2str\n", this_program);
	    strcpy (buf, "**Bad Type Spec**");
	    break;
    } /* switch */
    return buf;
} /* ty2str */

#define USAGE_MAX 1000
#define MAX_LINE_WIDTH 80

char *build_usage (table, table_size, result, result_len)
arg_info table[];
int table_size, result_len;
char *result;
{
    int i, newlines = 0;
    int max_line_width = MAX_LINE_WIDTH - 1;
    static char buff[USAGE_MAX], *ptr = buff;

    buff[USAGE_MAX - 1] = '\0';
    sprintf (ptr, "Usage:  %s", this_program);
    ptr += strlen (ptr);

    for (i = 0; i < table_size; i++) {
	arg_info *arg = &table[i];
	int count = arg_count (*arg);

	(void) strcat (ptr, " [");
	(void) strcat (ptr, arg_prefix (*arg));
	ptr += strlen (ptr);
	if (arg_flags (*arg) == P_CASE_INSENSITIVE)
	    (void) lower_string (ptr, arg_string (*arg));
	else
	    (void) strcat (ptr, arg_string (*arg));
	ptr += strlen (ptr);
	if (count != P_NO_ARGS) {
	    *ptr++ = ' ';
	    (void) ty2str (arg_result_type (*arg), ptr);
	    ptr += strlen (ptr);
	} /* if count != P_NO_ARGS */
	strcpy (ptr++, "]");
	if (count == P_INFINITE_ARGS) {
	    *ptr++ = '*';
	    (void) itoa (arg_table_size (*arg), ptr);
	    ptr += strlen (ptr);
	} /* if count == P_INFINITE_ARGS */
    } /* for i = 0 */

/* Insert newlines as appropriate, leaving room for 8-char tabs */

    ptr = buff + (int_min (strlen (buff), max_line_width));
    for (; *ptr; ptr += int_min (strlen (ptr), max_line_width - 8)) {
	char *back;

	for (back = ptr; back > buff && !isspace(*back); back--)
	    ;
	if (back > buff) {
	    *back = '\n';
	    ptr = back;
	    newlines++;
	} /* if back > buff */
    } /* for ptr = buff */

/* Copy the string into   result   storage */

    if (buff[USAGE_MAX - 1] != '\0') {
	fprintf (stderr, "%s:  Usage string too long!\n", this_program);
	buff[USAGE_MAX - 1] = '\0';
    } /* if buff[USAGE_MAX - 1] != '\0' */

    if (result_len >= 0 && strlen (buff) + newlines >= result_len) {
	fprintf (stderr, "%s:  Usage string too long, truncated\n",
		this_program);
	buff[result_len - newlines <= 0 ? 0 : result_len - 1] = '\0';
    } /* if result_len >= 0 && */

/* Add tabs after all newlines */

    if (result) {
	char *rptr = result;

	for (ptr = buff; *ptr; ptr++, rptr++) {
	    *rptr = *ptr;
	    if (*ptr == '\n')
		*++rptr = '\t';
	} /* for ptr = buff */
    } /* if result */

    return (result == NULL) ? buff : result;
} /* build_usage */    

boolean arg_verify (table, table_size)
arg_info table[];
int table_size;
{
    int i;

    for (i = 0; i < table_size; i++) {
	arg_info *arg = &table[i];

/* Check the argument flags */

	if (arg_flags (*arg) & ~(P_CASE_INSENSITIVE | P_REQUIRED_PREFIX)) {
	    fprintf (stderr, "%s [arg_verify]:  illegal ", this_program);
	    fprintf (stderr, "flags in entry %d:  '%x' (hex)\n", i,
		    arg_flags (*arg));
	} /* if */

/* Check the argument count */

	{ int count = arg_count (*arg);

	    if (count != P_NO_ARGS && count != P_ONE_ARG && count !=
		    P_INFINITE_ARGS) {
		fprintf (stderr, "%s [arg_verify]:  invalid ", this_program);
		fprintf (stderr, "argument count in entry %d:  '%d'\n", i,
			count);
	    } /* if count != P_NO_ARGS ... */

/* Check the result field; want to be able to store results */

	      else
		if (arg_result_ptr (*arg) == (int *) NULL) {
		    fprintf (stderr, "%s [arg_verify]:  ", this_program);
		    fprintf (stderr, "no argument storage given for ");
		    fprintf (stderr, "entry %d\n", i);
		} /* if arg_result_ptr */
	}

/* Check the argument type */

	{ int type = arg_result_type (*arg);

	    if (type != P_STRING &&
		type != P_CHAR &&
		type != P_SHORT &&
		type != P_LONG &&
		type != P_INT &&
		type != P_FILE &&
		type != P_OLD_FILE &&
		type != P_NEW_FILE &&
		type != P_FLOAT &&
		type != P_DOUBLE) {
		    fprintf (stderr, "%s [arg_verify]:  bad ", this_program);
		    fprintf (stderr, "arg type in entry %d:  '%d'\n", i,
			    type);
	    } /* if type != .... */
	}

/* Check table size */

	{ int size = arg_table_size (*arg);

	    if (arg_count (*arg) == P_INFINITE_ARGS && size < 1) {
		fprintf (stderr, "%s [arg_verify]:  bad ", this_program);
		fprintf (stderr, "table size in entry %d:  '%d'\n", i,
			size);
	    } /* if (arg_count == P_INFINITE_ARGS && size < 1) */
	}

    } /* for i = 0 */

    return TRUE;
} /* arg_verify */


/* match_table -- returns the index of the best entry matching the input,
   -1 if no match.  The best match is the one of longest length which
   appears lowest in the table.  The length of the match will be returned
   in   length   ONLY IF a match was found.   */

int match_table (norm_input, table, table_size, use_prefix, length)
register char *norm_input;
arg_info table[];
int table_size;
boolean use_prefix;
int *length;
{
    extern int match (/* char *, char *, arg_info *, boolean */);

    char low_input[MAX_INPUT_SIZE];
    register int i;
    int best_index = -1, best_length = 0;

/* FUNCTION BODY */

    (void) lower_string (low_input, norm_input);

    for (i = 0; i < table_size; i++) {
	int this_length = match (norm_input, low_input, &table[i], use_prefix);

	if (this_length > best_length) {
	    best_index = i;
	    best_length = this_length;
	} /* if (this_length > best_length) */
    } /* for (i = 0) */

    if (best_index > -1 && length != (int *) NULL)
	*length = best_length;

    return best_index;
} /* match_table */


/* match -- takes an input string and table entry, and returns the length
   of the longer match.

	0 ==> input doesn't match

   For example:

	INPUT	PREFIX	STRING	RESULT
----------------------------------------------------------------------
	"abcd"	"-"	"d"	0
	"-d"	"-"	"d"	2    (i.e. "-d")
	"dout"	"-"	"d"	1    (i.e. "d")
	"-d"	""	"-d"	2    (i.e. "-d")
	"dd"	"d"	"d"	2	<= here's the weird one
*/

int match (norm_input, low_input, entry, use_prefix)
char *norm_input, *low_input;
arg_info *entry;
boolean use_prefix;
{
    char *norm_prefix = arg_prefix (*entry);
    char *norm_string = arg_string (*entry);
    boolean prefix_match = FALSE, string_match = FALSE;
    int result = 0;

/* Buffers for the lowercased versions of the strings being compared.
   These are used when the switch is to be case insensitive */

    static char low_prefix[MAX_INPUT_SIZE];
    static char low_string[MAX_INPUT_SIZE];
    int prefix_length = strlen (norm_prefix);
    int string_length = strlen (norm_string);

/* Pointers for the required strings (lowered or nonlowered) */

    register char *input, *prefix, *string;

/* FUNCTION BODY */

eprintf ("match:  given '%s'", norm_input);
eprintf (" entry '%s%s'", arg_prefix (*entry), arg_string (*entry));
/* Use the appropriate strings to handle case sensitivity */

    if (arg_flags (*entry) & P_CASE_INSENSITIVE) {
	input = low_input;
	prefix = lower_string (low_prefix, norm_prefix);
	string = lower_string (low_string, norm_string);
    } else {
	input = norm_input;
	prefix = norm_prefix;
	string = norm_string;
    } /* else */

/* First, check the string formed by concatenating the prefix onto the
   switch string, but only when the prefix is not being ignored */
eprintf ("    use_prefix %d prefix %x '%s'\n\t", use_prefix, prefix,
	prefix);
    if (use_prefix && prefix != NULL && *prefix != '\0')
eprintf (" GOT PREFIX");
	 prefix_match = (strncmp (input, prefix, prefix_length) == 0) &&
		(strncmp (input + prefix_length, string, string_length) == 0);

/* Next, check just the switch string, if that's allowed */

    if (!use_prefix && (arg_flags (*entry) & P_REQUIRED_PREFIX) == 0)
	string_match = strncmp (input, string, string_length) == 0;

eprintf (" pre %d str %d", prefix_length, string_length);
    if (prefix_match)
	result = prefix_length + string_length;
    else if (string_match)
	result = string_length;

eprintf (" returning %d\n", result);
    return result;
} /* match */


char *lower_string (dest, src)
char *dest, *src;
{
    char *result = dest;

    if (dest == NULL || src == NULL)
	result = NULL;
    else
	while (*dest++ = mklow (*src++));

    return result;
} /* lower_string */


/* parse_arg -- returns the number of characters parsed for this entry */

int parse_arg (str, entry)
char *str;
arg_info *entry;
{
    int length = 0;

    if (arg_count (*entry) == P_ONE_ARG) {
	char **store = (char **) arg_result_ptr (*entry);

	length = put_one_arg (arg_result_type (*entry), str, store,
		arg_prefix (*entry), arg_string (*entry));

    } /* if (arg_count == P_ONE_ARG) */
      else { /* Must be a table of arguments */
	char **store = (char **) arg_result_ptr (*entry);

	if (store) {
	    while (*store)
		store++;

	    length = put_one_arg (arg_result_type (*entry), str, store++,
		    arg_prefix (*entry), arg_string (*entry));

	    *store = (char *) NULL;
	} /* if (store) */
    } /* else */

    return length;
} /* parse_arg */


int put_one_arg (type, str, store, prefix, string)
int type;
char *str;
char **store;
char *prefix, *string;
{
    extern double atof ();
    int length = 0;

    if (store) {
	struct stat buf;
	extern int errno;

	switch (type) {
	    case P_STRING:
	    case P_FILE:
		*store = str;
		if (str == NULL) {
		    fprintf (stderr, "%s: Missing argument after '%s%s'\n",
			    this_program, prefix, string);
		    bad_input_count++;
		} /* if str == NULL */
		length = str ? strlen (str) : 0;
		break;
	    case P_OLD_FILE:
	    case P_NEW_FILE:
		length = strlen (str);
		if (stat (str, &buf) == 0) {
		    if (type == P_NEW_FILE) {
			fprintf (stderr, "%s: File '%s' already exists, ",
				this_program);
			fprintf (stderr, "ignored.\n");
			bad_input_count++;
		    } else
			*store = str;
		} else if (type == P_NEW_FILE)
		    *store = str;
		else {
		    fprintf (stderr, "%s: Can't access '%s': ",
			    this_program, str);
		    bad_input_count++;
		    switch (errno) {
		        case ENOTDIR:
			    fprintf (stderr, "bad directory prefix");
			    break;
			case EINVAL:
			    fprintf (stderr, "non-ascii char in filename");
			    break;
			case ENAMETOOLONG:
			    fprintf (stderr, "path too long");
			    break;
			case ENOENT:
			    fprintf (stderr, "nonexistent file");
			    break;
			case EACCES:
			    fprintf (stderr, "invalid path permissions");
			    break;
			case ELOOP:
			    fprintf (stderr, "too many symbolic links");
			    break;
			case EFAULT:
			    fprintf (stderr, "*** parse_args() ERROR ***");
			    break;
			case EIO:
			    fprintf (stderr, "I/O error");
			    break;
			default:
			    fprintf (stderr, "unknown error '%d'", errno);
			    break;
		    } /* switch */
		    fprintf (stderr, "\n");
		} /* else */
		break;
	    case P_CHAR:
		*((char *) store) = *str;
		length = 1;
		break;
	    case P_SHORT: {
		long int i = atoi (str);

		*((short *) store) = (short) i;
		if (i > 32767 || i < -32768) {
		    fprintf (stderr, "%s%s parameter '%ld' is not a ",
			    prefix, string, i);
		    fprintf (stderr, "SHORT INT (truncating to %d)\n",
			    (short) *store);
		    bad_input_count++;
		} /* if i > 32767 || i < -32768 */

/* This is pessimistic, and should change */

		length = strlen (str);
		break; }
	    case P_LONG:
		*((int *) store) = atoi (str);

/* This is pessimistic too */

		length = strlen (str);
		break;
	    case P_FLOAT:
		*((float *) store) = (float) atof (str);
		length = strlen (str);
		break;
	    case P_DOUBLE:
		*((double *) store) = (double) atof (str);
		length = strlen (str);
		break;
	    default:
		fprintf (stderr, "arg_parse:  bad type '%d'\n",
			type);
		break;
	} /* switch */
    } /* if (store) */

    return length;
} /* put_one_arg */


void init_store (table, table_size)
arg_info *table;
int table_size;
{
    int index;

    for (index = 0; index < table_size; index++)
	if (arg_count (table[index]) == P_INFINITE_ARGS) {
	    char **place = (char **) arg_result_ptr (table[index]);

	    if (place)
		*place = (char *) NULL;
	} /* if arg_count == P_INFINITE_ARGS */

} /* init_store */
