/* ID-HASH.C -- functions implementing the hash table for identifiers.

   $Header: id-hash.c,v 1.10 91/08/13 14:44:21 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "id-hash.h"

/* DEFINITIONS ============================================================ */

/* Radix of the ASCII character set */
#define CharSetRadix 256

/* Hash Table Size (should be prime) */
#define IdHashTableSize 51

/* LOCAL VARIABLES ======================================================== */

/* hash table to use during the parse */
static IdDrop *id_ht[IdHashTableSize];

/* LOCAL FUNCTIONS ======================================================== */

static int HashId(name)
  String name;
{
    int hash_val = 0;
    for (; *name != '\0'; name++) {
	hash_val *= CharSetRadix;
	hash_val += (int)(*name);
	hash_val %= IdHashTableSize;
    }
    return(hash_val);
}

/* GLOBAL FUNCTIONS ======================================================= */

void InitIdHashTable()
{
    register int i;

    /* make table empty */
    StepIndex(i,0,IdHashTableSize) id_ht[i] = NULL;
}

void ShutDownIdHashTable()
{
    int i;
    IdDrop *curr,*temp;

    StepIndex(i,0,IdHashTableSize) {
	curr = id_ht[i];
	while (curr != NULL) {
	    temp = curr;
	    curr = curr->next;
	    Dealloc(temp->id_name); /* free string name */
	    Dealloc(temp);	    /* free structure */
	}
    }
}

void AddId(name,id_types)
  String name;
  IdType id_types;
{
    IdDrop *drop_ptr;

    drop_ptr = AddIdDrop(name,Identifier);
    IdTypesOf(drop_ptr) = id_types;
}

/* --------------- General Routines for All IdDropType's ----------------- */
/* ------- Used Directly for IdDrop's where IdDropType = 'InputID' ------- */

IdDrop *AddIdDrop(name,drop_type)
  String name;
  IdDropType drop_type;
{
    int hash_val;
    IdDrop *drop_ptr;

    /* Allocate/initialize the drop */
    drop_ptr = AllocOne(IdDrop);
    CopyString (IdNameOf(drop_ptr), name);
    DropTypeOf(drop_ptr) = drop_type;

    /* Add the drop to the hash table (at front of its bucket) */
    hash_val = HashId(name);
    NextOf(drop_ptr) = id_ht[hash_val];
    id_ht[hash_val] = drop_ptr;

    return(drop_ptr);
}

IdDrop *FindIdDrop(name,drop_type)
  String name;
  IdDropType drop_type;
{
    IdDrop *drop_ptr;

    StepLinkedList(drop_ptr,id_ht[HashId(name)]) {
	if ((DropTypeOf(drop_ptr) == drop_type) &&
	    (SameString(IdNameOf(drop_ptr),name)))
	    break;
    }
    return(drop_ptr);
}

String FindOrAddIdDrop(name,drop_type)
  String name;
  IdDropType drop_type;
{
    IdDrop *id_drop_ptr;

    if ((id_drop_ptr=FindIdDrop(name,drop_type)) == NULL) {
	return(IdNameOf(AddIdDrop(name,drop_type)));
    } else {
	return(IdNameOf(id_drop_ptr));
    }
}

/* -------- Routines for Showing the ID Hash Table Data Structure -------- */

#ifdef DEBUG
void ShowIdHashTable()
{
    int i,count,total;
    IdDrop *curr;

    printf("\nSHOW-ID-HASH-TABLE():\n");
    for (total=i=0; i<IdHashTableSize; i++) {
	count = 0;
	StepLinkedList(curr,id_ht[i]) {
	    count++;
	}
	total += count;
	if (count) {
	    printf("Bucket: %d\tCount: %d\n",i,count);
	    StepLinkedList(curr,id_ht[i]) {
		printf("  ** Name:  %s\n",IdNameOf(curr));
		printf("     Type:  %s\n",IdDropTypeToString(DropType(curr)));
		switch (DropTypeOf(curr)) {
		  case Identifier:
		    printf("     Value: %x\n",(int)IdTypesOf(curr));
		    break;
		  case InputId:
		    break;
		  case ArrowType:
		    printf("     Value: %x\n",(int)ArrowHeadPtrOf(curr));
		    break;
		}
	    }
	}
    }
    printf("TOTAL NUMBER OF ENTRIES: %d\n",total);
}
#endif
