%{
/* $Header: iff.y,v 1.6 91/07/30 19:20:01 heydon Exp $

   Yacc source file for recognizing Miro picture files in IFF. See Miro
   Note #4 for a description of IFF. This grammar works in conjunction
   with the lex file "iff.l".

   GRAMMAR:

   file :	entry | file entry
   entry :	'>' entry_id a_list ';'
   entry_id :	IDENTIFIER
   a_list :	a_entry | a_list ';' a_entry
   a_entry :	p_name '=' p_value
   p_name :	IDENTIFIER
   p_val_list :	p_value | p_val_list ',' p_value
   p_value :	INTEGER | IDENTIFIER | STRING | '{' p_val_list '}'
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <my-types.h>
#include <lex-globals.h>
#include "mem.h"
#include <my-defs.h>

#include "id-hash.h"
#include "error.h"
#include "parser.h"
#include "parser.g"

/* MACRO FUNCTIONS ========================================================= */

#define MakeEmptyList(_list) ((_list).first = (_list).last = NULL)

#define JoinListWithPtr(_result_ptr,_list_ptr,_entry_ptr) \
  JoinGenericListWithPtr((GenericList *)(_result_ptr), \
			 (GenericList *)(_list_ptr), \
			 (GenericEntry *)(_entry_ptr))

/* LOCAL VARIABLES ========================================================= */

/* true iff the parser needs the info assoc. w/ the current entry */
static Boolean do_read_entry;

/* true iff the parser needs the info assoc. w/ the current attribute */
static Boolean do_read_attr;

/* name of the current entry */
static String current_entry_name;

/* EntryType corresponding to the current entry */
static EntryType current_entry_type;

/* name of the current property */
static String current_prop_name;

/* allowed type(s) for the current attribute being read */
static PropValType current_prop_val_types;

/* LOCAL FUNCTIONS ========================================================= */

static void JoinGenericListWithPtr(result_ptr,list_ptr,entry_ptr)
  GenericList *result_ptr,*list_ptr;
  GenericEntry *entry_ptr;
/* Joins the entry pointed to by 'entry_ptr' to the end of the list 'list_ptr'
   (which consists of a 'first' and 'last' component), and sets 'result_ptr'
   to be the resulting list. Either 'list_ptr' or 'entry_ptr' may be empty, in
   the sense that their pointer(s) is(are) NULL.
*/
{
  FirstOf(result_ptr) =
      (FirstOf(list_ptr) != NULL) ? FirstOf(list_ptr) : entry_ptr;
  if (entry_ptr != NULL) {
      LastOf(result_ptr) = entry_ptr;
      if (FirstOf(list_ptr) != NULL) {
          NextOf(LastOf(list_ptr)) = entry_ptr;
      }
  } else {
      LastOf(result_ptr) = LastOf(list_ptr);
  }
}

static PropVal *SetPValue(p_val_type,found_type_name)
  PropValType p_val_type;
  String found_type_name;
/* This code builds a p_value if it should be created and returns a pointer to
   it. The p_value being created will be of type 'p_val_type'. If the value
   should not be read (because 'do_read_entry' or 'do_read_attr' is false) or
   if the value read is not of the proper type (according to the variable
   'current_prop_val_types'), then NULL will be returned. */
{
    PropVal *result = NULL;
    if (do_read_entry && do_read_attr) {
      if (!((int)p_val_type & (int)current_prop_val_types)) {
	  ParseErrorSS(curr_line_no,
		     "property name '%s' cannot take a value of type '%s'",
		     current_prop_name,found_type_name);
      } else {
	  result = AllocOne(PropVal);
	  PropValTypeOf(result) = (p_val_type);
      }
    }
    return(result);
}
%}

%{
/* Declare a new union type that will be the type of the yacc variable yylval.
   The Lex code returns a token by filling in the appropriate field of this
   union and then returning the appropriate %token defined below, or by
   returning an actual character instead of a %token type.
*/
%}
%union {
    int int_val;
    String id_val;
    String string_val;
    EntryList entry_list;
    AttrEntryList attr_entry_list;
    PropValList prop_val_list;
    PropVal *prop_val;
}

%{
/* Declare possible return values from the Lex code. Also declared are the
   slots to use in the %union defined above for holding the value associated
   with each %token.
*/
%}
%token <int_val> INTEGER
%token <id_val> IDENTIFIER
%token <string_val> STRING

%{
/* Declarations of the slot to use in the %union above for filling in each
   non-terminal value.
*/
%}
%type <entry_list> file entry entry_id
%type <attr_entry_list> a_list a_entry p_name
%type <prop_val_list> p_val_list
%type <prop_val> p_value

%{
/* Declare the starting non-terminal of the grammar to be 'file'. */
%}
%start file

%% /* rules section ----------------------------------------------------- */

file	:	entry
		  { $$ = $1; }	/* structure assignment (1 element list) */
	|	file entry
		  { JoinListWithPtr(&($$),&($1),$2.first); }
	;

entry	:	'>' entry_id a_list ';'
		  { /* the EntryList returned in $$ will always be such
		       that the 'first' and 'last' components are the same;
		       they both point to a 1-element AttrEntry linked list */
		    $$ = $2;	/* structure assignment */
		    if ($$.first != NULL) {
			AttrListHeadOf($$.first) = $3.first;
			NextOf($$.first) = NULL;
		    }
		  }
	;

entry_id :       IDENTIFIER
		  { /* the EntryList returned in $$ will always be such
		       that the 'first' and 'last' components are the same */
		    IdDrop *drop_ptr;
		    IdType entry_id_types;
		    do_read_entry = False;
		    if ((drop_ptr=FindIdDrop($1,Identifier)) != NULL) {
		      entry_id_types = IdTypesOf(drop_ptr);
			if (!IsEntryNameP(entry_id_types)) {
			  ParseErrorS(curr_line_no,
				      "'%s' is not an entry name",$1);
			} else {
			  do_read_entry = True;
			  current_entry_name = IdNameOf(drop_ptr);
			  current_entry_type =
			      (EntryType)((int)entry_id_types &
					  (int)AllEntryTypes);
			  $$.first = $$.last = AllocOne(Entry);
			  EntryLineNumberOf($$.first) = curr_line_no;
			  EntryNameOf($$.first) = current_entry_name;
			  EntryTypeOf($$.first) = current_entry_type;
			}
		    }
		    if (!do_read_entry) { MakeEmptyList($$); }
		  }
	;

a_list	:	a_entry
		  { $$ = $1; }	/* structure assignment (1 element list) */
	|	a_list ';' a_entry
		  { JoinListWithPtr(&($$),&($1),$3.first); }
	;

a_entry	:	p_name '=' p_value
		  { /* the AttrEntryList returned in $$ will always be such
		       that the 'first' and 'last' components are the same */
		    if ($3 == NULL) {	/* p_value not read for some reason */
			$1.first = $1.last = NULL;
		    }
    		    $$ = $1;		/* structure assignment */
		    if ($$.first != NULL) {
			PropValPtrOf($$.first) = $3;
			NextOf($$.first) = NULL;
		    }
		  }
	;

p_name :	IDENTIFIER
		  { /* the AttrEntryList returned in $$ will always be such
		       that the 'first' and 'last' components are the same */
		    IdDrop *drop_ptr;
		    IdType attr_id_types;
		    do_read_attr = False;
		    if (do_read_entry &&
			(drop_ptr=FindIdDrop($1,Identifier)) != NULL) {
		      attr_id_types = IdTypesOf(drop_ptr);
		      if (!IsPNameP(attr_id_types)) {
			  ParseErrorS(curr_line_no,
				      "'%s' is not a property name",$1);
		      } else if (!((int)attr_id_types &
				   (int)current_entry_type)) {
			  ParseErrorSS(curr_line_no,
			   "property name '%s' not allowed in entry '%s'",
			   $1,current_entry_name);
		      } else {
			  do_read_attr = True;
			  current_prop_name = IdNameOf(drop_ptr);
			  current_prop_val_types =
			      (PropValType)((Uint)attr_id_types
					    & (Uint)AllPValTypes);
			  $$.first = $$.last = AllocOne(AttrEntry);
			  AttrLineNumberOf($$.first) = curr_line_no;
			  AttrNameOf($$.first) = IdNameOf(drop_ptr);
		      }
		    }
		    if (!do_read_attr) { MakeEmptyList($$); }
		  }
	;

p_val_list :	/* empty */
		  { MakeEmptyList($$); }
	   |	p_value
		  { if ($1 == NULL) {
		      MakeEmptyList($$);
		    } else {
		      $$.first = $$.last = AllocOne(ListEntry);
		      ListPropValPtrOf($$.last) = $1;
		      NextOf($$.last) = NULL;
		    }
		  }
	   |	p_val_list ',' p_value
		  { ListEntry *list_entry_ptr;
		    if ($3 == NULL) {
			list_entry_ptr = NULL;
		    } else {
			list_entry_ptr = AllocOne(ListEntry);
			ListPropValPtrOf(list_entry_ptr) = $3;
			NextOf(list_entry_ptr) = NULL;
		    }
		    JoinListWithPtr(&($$),&($1),list_entry_ptr);
		  }
	   ;

p_value	:	INTEGER
		  { $$ = SetPValue(IntPValType,"integer");
		    if ($$ != NULL) {
			IntValOf($$) = $1;
		    }
		  }
	|	IDENTIFIER
		  { $$ = SetPValue(IdPValType,"identifier");
		    if ($$ != NULL) {
			/* If the ID is not installed in the id-hash table,
			   install a *copy* of it (this way, every ID will
			   only be stored once). */
			IdValOf($$) = FindOrAddIdDrop($1,InputId);
		    }
		  }
	|	STRING
		  { $$ = SetPValue(StringPValType,"string");
		    if ($$ != NULL) {
			StringValOf($$) = $1;
		    }
		  }
	|	'{' p_val_list '}'
		  { $$ = SetPValue(ListPValType,"list");
		    if ($$ != NULL) {
			ListHeadOf($$) = $2.first;
		    }
		  }
	;
