/* ARROW.C -- functions implementing creation/access of arrows.

   $Header: arrow.c,v 1.7 91/02/18 17:11:38 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "error.h"
#include "id-hash.h"
#include "parser.h"
#include "box.h"
#include "arrow.h"
#include "sysname-hash.h"

/* GLOBAL VARIABLE DEFINITIONS ============================================ */

ArrowHead *arrow_head_head;

/* LOCAL VARIABLE DEFINITIONS ============================================= */

/* indicates if in MultipleArrow group */
static Boolean MultipleArrows = False;

/* indicates if next arrow should be checked for duplicate sysname*/
static Boolean NewArrow = True;

/* LOCAL FUNCTIONS ======================================================== */

static Arrow *MakeArrow(sys_name,head,tail,parity)
  SysName sys_name;
  Box *head,*tail;
  ArrowParity parity;
/* Returns a pointer to a newly created Arrow with corresponding 'sys_name',
   'head', 'tail', and 'parity' values.
*/
{
    Arrow *arrow_ptr;

    arrow_ptr = AllocOne(Arrow);
    ASysNameOf(arrow_ptr) = sys_name;
    HeadOf(arrow_ptr) = head;
    TailOf(arrow_ptr) = tail;
    ParityOf(arrow_ptr) = parity;
    return(arrow_ptr);
}

static void AddArrowHead(arrow_head_ptr)
  ArrowHead *arrow_head_ptr;
/* Adds the ArrowHead '*arrow_head_ptr' to the linked list of ArrowHead's
   pointed to by 'arrow_head_head'.
*/
{
    NextOf(arrow_head_ptr) = arrow_head_head;
    arrow_head_head = arrow_head_ptr;
}

static ArrowHead *MakeArrowHead(perm)
  Permission perm;
/* Returns a pointer to a newly created ArrowHead with Permission 'perm'. This
   string should be "permanent", namely, a pointer to an identifier in the ID
   hash table.
*/
{
    ArrowHead *arrow_head_ptr;

    /* create the ArrowHead and fill in it's fields */
    arrow_head_ptr = AllocOne(ArrowHead);
    PermOf(arrow_head_ptr) = perm;
    FirstArrowOf(arrow_head_ptr) = NULL;
    PosCntOf(arrow_head_ptr) = NegCntOf(arrow_head_ptr) = 0;
    return(arrow_head_ptr);
}

static ArrowHead *FindArrowHeadName(name)
  String name;
/* RETURNS a pointer to the ArrowHead associated with the IdDrop in the ID
   hash table having name 'name' and an IdDropType of ArrowType; NULL if no
   such IdDrop is found.
*/
{
    IdDrop *drop_ptr;

    if ((drop_ptr=FindIdDrop(name,ArrowType)) == NULL)
	return(NULL);
    return((ArrowHead *)ArrowHeadPtrOf(drop_ptr));
}

static void AddArrowHeadName(name,arrow_head_ptr)
  String name;
  ArrowHead *arrow_head_ptr;
/* Add's an IdDrop with name 'name' and IdDropType of ArrowType to the ID hash
   table, and sets the value associated with this IdDrop to 'arrow_head_ptr'.
   This function should only be called if the caller is sure that the ID does
   not already exist in the hash table.
*/
{
    ArrowHeadPtrOf(AddIdDrop(name,ArrowType)) = (Generic *)arrow_head_ptr;
}

static ArrowHead *FindOrAddArrowHead(perm)
  Permission perm;
/* If the ArrowHead with Permission 'perm' does not exist, then create a
   new ArrowHead with that permission, and add the ArrowHead to the linked
   list of ArrowHead's and the ID hash table by calling AddArrowHead() and
   AddArrowHeadId() respectively.
*/
{
    ArrowHead *arrow_head_ptr;

    if ((arrow_head_ptr=FindArrowHeadName((String)perm)) != NULL) {
	return(arrow_head_ptr);
    }
    AddArrowHead(arrow_head_ptr=MakeArrowHead(perm));
    AddArrowHeadName((String)perm,arrow_head_ptr);
    return(arrow_head_ptr);
}

void ConvertParity(p_val_ptr,val_ptr,line_no)
  PropVal *p_val_ptr;
  Generic *val_ptr;
  int line_no;
{
#ifdef DEBUG
    if (PropValTypeOf(p_val_ptr) != IdPValType) {
	ProgrammerErrorI("arrow.c",
			 "prop_val_type should be IdPValType instead of 0x%x",
			 PropValTypeOf(p_val_ptr));
    }
#endif
    if (SameString("pos",IdValOf(p_val_ptr))) {
	*((ArrowParity *)val_ptr) = PosParity;
    } else if (SameString("neg",IdValOf(p_val_ptr))) {
	*((ArrowParity *)val_ptr) = NegParity;
    } else {
	ParseErrorS(line_no,
		    "arrow parity '%s' should be either 'pos' or 'neg'",
		    IdValOf(p_val_ptr));
	*((ArrowParity *)val_ptr) = UnknownParity;
    }
}

#ifdef DEBUG
#define StringParity(_par)\
   ((_par)==PosParity ? "Positive" :\
    ((_par)==NegParity ? "Negative" : "Unknown"))
#endif

#ifdef DEBUG
void ShowArrowHeadList(arrow_head_ptr)
  ArrowHead *arrow_head_ptr;
/* Display the list of arrows associated with ArrowHead '*arrow_head_ptr'.
*/
{
    int p_cnt,n_cnt;
    Arrow *arrow_ptr;

    printf("  Permission: %-10s #-of-Pos: %-8d #-of-Neg: %d\n",
	   PermOf(arrow_head_ptr),PosCntOf(arrow_head_ptr),
	   NegCntOf(arrow_head_ptr));
    p_cnt = n_cnt = 0;
    StepLinkedList(arrow_ptr,FirstArrowOf(arrow_head_ptr)) {
	printf("\tsysname=%-6d head-index=%-5d tail-index=%-5d parity=%s\n",
	       (int)ASysNameOf(arrow_ptr),IndexOf(HeadOf(arrow_ptr)),
	       IndexOf(TailOf(arrow_ptr)),StringParity(ParityOf(arrow_ptr)));
	((ParityOf(arrow_ptr) == PosParity) ? ++p_cnt : ++n_cnt);
    }
#ifdef DEBUG
    if (PosCntOf(arrow_head_ptr) != p_cnt)
	ProgrammerErrorI("arrow.c","actual # of pos arrows = %d",p_cnt);
    if (NegCntOf(arrow_head_ptr) != n_cnt)
	ProgrammerErrorI("arrow.c","actual # of neg arrows = %d",n_cnt);
#endif
}
#endif

/* GLOBAL FUNCTIONS ======================================================= */

void InitArrows()
{
    arrow_head_head = NULL;
}

void AddArrow(sys_name,head,tail,parity,perm,line_no)
  SysName sys_name;
  Box *head,*tail;
  ArrowParity parity;
  Permission perm;
  int line_no;
/* Implementation Notes:

   We use the ID hash table to find the ArrowHead corresponding to 'perm'
   quickly.
*/
{
    ArrowHead *arrow_head_ptr;
    Arrow *arrow_ptr;

/* check that an arrow of this sys_name doesn't yet exist */
    if (NewArrow) {
	if (FindArrowSysName(sys_name) != NULL) {
	    ParseErrorI(line_no,"an ARROW with sysname = %d already exists",
			(int)sys_name);
	    return;
	}
	if (MultipleArrows) { NewArrow = False; }
    } else if ((RoleOf(tail)!=UserRole) || (RoleOf(head)!=FileRole)) {
	ParseError(line_no,
		   "this ARROW does not connect a user box to a file box");
	return;
    }

/* find/create the ArrowHead */
    arrow_head_ptr = FindOrAddArrowHead(perm);

/* create the new Arrow and add it to the sysname hash table */
    arrow_ptr = MakeArrow(sys_name,head,tail,parity);
    AddSysName(sys_name,ArrowDrop,(Generic *)arrow_ptr);

/* add the arrow to the linked list for this ArrowHead */
    NextOf(arrow_ptr) = FirstArrowOf(arrow_head_ptr);
    FirstArrowOf(arrow_head_ptr) = arrow_ptr;
    ((parity == PosParity) ? ++PosCntOf(arrow_head_ptr)
	                   : ++NegCntOf(arrow_head_ptr));
}

void BeginMultipleArrows()
{
    MultipleArrows = True;
}

void EndMultipleArrows()
{
    MultipleArrows = False;
    NewArrow = True;
}

#ifdef DEBUG
void ShowArrows()
{
    ArrowHead *arrow_head_ptr;

    printf("\nSHOW-ARROWS():\n");
    StepLinkedList(arrow_head_ptr,arrow_head_head) {
	ShowArrowHeadList(arrow_head_ptr);
    }
}
#endif
