/* BOX.C -- functions implementing creation/access of boxes themselves.

   $Header: box.c,v 1.5 90/11/01 11:39:22 heydon Locked $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "error.h"
#include "parser.h"
#include "box.h"
#include "sysname-hash.h"

/* GLOBAL VARIABLE DEFINITIONS ============================================ */

Box   *all_head[NumRoles],*atom_head[NumRoles];
int    num_all_boxes[NumRoles],num_atomic_boxes[NumRoles];

/* LOCAL FUNCTIONS ======================================================== */

static Box *MakeNewBox(role,sys_name)
  BoxRole role;
  SysName sys_name;
/* Returns a pointer to a newly created ATOMIC box with role 'role' and
   SysName sys_name; it's index and role are automatically assigned.
*/
{
    Box *box_ptr;

    box_ptr = AllocOne(Box);
    BSysNameOf(box_ptr) = sys_name;
    IndexOf(box_ptr) = NumberOfAtomicBoxes(role);
    RoleOf(box_ptr) = role;
    AtomicPOf(box_ptr) = True;
    return(box_ptr);
}

static void FixBoxList(list_head_ptr_ptr,atom_list_head_ptr_ptr)
  Box **list_head_ptr_ptr;
  Box **atom_list_head_ptr_ptr;
/* Assuming '*list_head_ptr_ptr' points to a linked list of boxes, this
   routine goes down the list and changes the links so that the pointer
   '*atom_list_head_ptr_ptr' points to the middle of the list (i.e.,
   the sub-list it points to contains all those (and only those) elements
   which are atomic). The values of both the pointers '*list_head_ptr_ptr'
   and '*atom_list_head_ptr_ptr' may be changed by this routine.
*/
{
    Box *atom_list,*curr_box_ptr;

    atom_list = NULL;
    while ((curr_box_ptr = *list_head_ptr_ptr) != NULL) {
	if (AtomicPOf(curr_box_ptr)) {
	    *list_head_ptr_ptr = NextOf(curr_box_ptr);
	    NextOf(curr_box_ptr) = atom_list;
	    atom_list = curr_box_ptr;
	} else {
	    list_head_ptr_ptr = &(NextOf(curr_box_ptr));
	}
    }
    *list_head_ptr_ptr = *atom_list_head_ptr_ptr = atom_list;
}

#ifdef DEBUG
static void ShowBoxesOfRole(role)
  BoxRole role;
/* Display the index, role, and atomic status of all boxes of type 'role'.
*/
{
    int all_boxes_cnt,atomic_boxes_cnt;
    Box *box_ptr;
    Boolean atom_head_found = False;

    all_boxes_cnt = atomic_boxes_cnt = 0;
    printf("  Role: %s\n",StringBoxRole(role));
    printf("    All Boxes:\n");
    StepLinkedList(box_ptr,AllBoxesListHead(role)) {
	/* check to see if we've reached the atomic boxes */
	if (box_ptr == AtomicBoxesListHead(role)) {
	    atom_head_found = True;
	    printf("    Atomic Boxes:\n");
	}

	/* print the box info */
	printf("\tSysName=%-6d Index=%-6d Role=%-5s Atomic=%s\n",
	       BSysNameOf(box_ptr), IndexOf(box_ptr),
	       StringBoxRole(RoleOf(box_ptr)),
	       BooleanToString(AtomicPOf(box_ptr)));

	/* increment internal counts */
	all_boxes_cnt++;
	if (atom_head_found) atomic_boxes_cnt++;
    }

/* check that the counts are correct */
    printf("    Total Boxes:  %d\n",NumberOfBoxes(role));
    printf("    Atomic Boxes: %d\n",NumberOfAtomicBoxes(role));
#ifdef DEBUG
    if (all_boxes_cnt != NumberOfBoxes(role)) {
	ProgrammerErrorI("box.c","actual number of boxes is %d",
			 all_boxes_cnt);
    }
    if (atomic_boxes_cnt != NumberOfAtomicBoxes(role)) {
	ProgrammerErrorI("box.c","actual number of atomic boxes is %d",
			 atomic_boxes_cnt);
    }
#endif
}
#endif

/* GLOBAL FUNCTIONS ======================================================= */

void InitBoxes()
/* initialize the local variables
*/
{
    int role;

    StepIndex(role,FirstRole,NumRoles) {
	AllBoxesListHead(role) = AtomicBoxesListHead(role) = NULL;
	num_all_boxes[role] = num_atomic_boxes[role] = 0;
    }
}

void AddBox(sys_name,role,line_no)
  SysName sys_name;
  BoxRole role;
  int line_no;
{
    Box *box_ptr;

/* check that it's sysname does not already exist in the hash table */
    if (FindBoxSysName(sys_name) != NULL) {
	ParseErrorI(line_no,"a BOX with sysname = %d already exists",
		    (int)sys_name);
	return;
    }

/* create the box, add it to the sysname hash table, and add it to the
   front of the linked list of boxes */
    box_ptr = MakeNewBox(role,sys_name);
    AddSysName(sys_name,BoxDrop,(Generic *)box_ptr);
    NextOf(box_ptr) = AllBoxesListHead(role);
    AllBoxesListHead(role) = AtomicBoxesListHead(role) = box_ptr;

/* increment the box counts */
    NumberOfBoxes(role)++;
    NumberOfAtomicBoxes(role)++;
}

void AssertBoxNonAtomic(box_ptr)
  Box *box_ptr;
{
    if (AtomicPOf(box_ptr)) {
	AtomicPOf(box_ptr) = False;
	NumberOfAtomicBoxes(RoleOf(box_ptr))--;
    }
}

void FormAtomNonatomLists()
{
    int role;

    StepIndex(role,FirstRole,NumRoles) {
	FixBoxList(&(AllBoxesListHead(role)),&(AtomicBoxesListHead(role)));
    }
}

void ConvertBoxRole(p_val_ptr,val_ptr,line_no)
  PropVal *p_val_ptr;
  Generic *val_ptr;
  int line_no;
{
#ifdef DEBUG
    if (PropValTypeOf(p_val_ptr) != IdPValType) {
	ProgrammerErrorI("box.c",
			 "prop_val_type should be IdPValType instead of 0x%x",
			 PropValTypeOf(p_val_ptr));
    }
#endif
    if (SameString("user",IdValOf(p_val_ptr))) {
	*((BoxRole *)val_ptr) = UserRole;
    } else if (SameString("file",IdValOf(p_val_ptr))) {
	*((BoxRole *)val_ptr) = FileRole;
    } else {
	ParseErrorS(line_no,"box role '%s' should be either 'user' or 'file'",
		    IdValOf(p_val_ptr));
	*((BoxRole *)val_ptr) = UnknownRole;
    }
}

void ConvertBoxName(p_val_ptr,val_ptr,line_no)
  PropVal *p_val_ptr;
  Generic *val_ptr;
  int line_no;
{
    Box *box_ptr;

#ifdef DEBUG
    if (PropValTypeOf(p_val_ptr) != IntPValType) {
	ProgrammerErrorI("box.c",
			 "prop_val_type should be IntPValType instead of 0x%x",
			 PropValTypeOf(p_val_ptr));
    }
#endif

    /* search for the box with SysName IntValOf(p_val_ptr) */
    if ((box_ptr=FindBoxSysName((SysName)IntValOf(p_val_ptr))) == NULL) {
	/* in this case, *val_ptr will be set to NULL */
	ParseErrorI(line_no,"box with sysname %d does not exist",
		    IntValOf(p_val_ptr));
    }
    /* assign box_ptr to *val_ptr (but cast it to a (Box **) first) */
    *((Box **)val_ptr) = box_ptr;
}

#ifdef DEBUG
#define StringBoxRole(_role)\
   ((_role)==UserRole ? "User" :\
    ((_role)==FileRole ? "File" : "Unknown"))
#endif

#ifdef DEBUG
void ShowBoxes()
{
    int role;

    printf("\nSHOW-BOXES():\n");
    StepIndex(role,FirstRole,NumRoles) {
	ShowBoxesOfRole((BoxRole)role);
    }
}
#endif
