/* ID-HASH.C -- functions implementing the hash table for identifiers.

   $Header: sysname-hash.c,v 1.5 90/11/01 11:39:25 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   IMPLEMENTATION NOTES

   This module implements a hash table for Box and Arrow SysNames. The
   implmentation uses bucket hashing. Currently, the buckets are represented
   as linked lists, but these should eventually be converted to a more
   efficient structure such as AVL trees. Since an arrow and a box may have
   the same SysName, and since it is desirable not to have the same SysName
   occurring twice in the same bucket, the hash function hashes on *both* the
   SysName and the type (Box or Arrow) of object associated with that name,
   and is designed so as not to hash a Box and Arrow with the same SysName to
   the same bucket.
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "error.h"
#include "box.h"
#include "arrow.h"
#include "sysname-hash.h"

/* DEFINITIONS ============================================================ */

/* size of sysname hash table */
#define SysNameHashTableSize 1009

/* Hash function constants */
#define HashCoefficient      23	/* currently not used */
#define ArrowOffset          500

/* LOCAL VARIABLES ======================================================== */

/* hash table array */
static SysNameDrop *SysNameHashTable[SysNameHashTableSize];

/* LOCAL FUNCTIONS ======================================================== */

static int HashSysName(sys_name,drop_type)
  SysName sys_name;
  SysNameDropType drop_type;
/* this function RETURNS an integer index into the SysNameHashTable array
   corresponding to the hash value of 'sys_name' and the type of object
   associated with that SysName 'drop_type'. Boxes and Arrows with the same
   SysName are guaranteed to hash to different indices.
*/
{
    register int result;
    
/* use the SysName (which will eventually be taken % SysNameHashTableSize) as
   the index; we may want to multiply by some value if this does not give an
   adequate distribution */
    result = (int)sys_name;

/* force arrow to hash to a different index from that for a box with the
   same SysName */
    if (drop_type == ArrowDrop) {
	result += ArrowOffset;
    }

/* take index MOD SysNameHashTableSize */
    return(result % SysNameHashTableSize);
}

static SysNameDrop *FindSysName(sys_name,drop_type)
  SysName sys_name;
  SysNameDropType drop_type;
/* RETURNS a pointer to the SysNameDrop in the SysNameHashTable with SysName
   'sys_name' and SysNameDropType 'drop_type' or NULL if no such drop exists.
*/
{
    SysNameDrop *drop_ptr;

    StepLinkedList(drop_ptr,
		   SysNameHashTable[HashSysName(sys_name,drop_type)]) {
	if (SysNameOf(drop_ptr) == sys_name &&
	    SysNameDropTypeOf(drop_ptr) == drop_type)
	    break;
    }
    return(drop_ptr);
}

/* GLOBAL FUNCTIONS ======================================================= */

void InitSysNameHashTable()
{
    register int i;
    StepIndex(i,0,SysNameHashTableSize) {
	SysNameHashTable[i] = NULL;
    }
}

void AddSysName(sys_name,drop_type,element_ptr)
  SysName sys_name;
  SysNameDropType drop_type;
  Generic *element_ptr;
{
    int hash_ix;
    SysNameDrop *drop_ptr;

/* create the SysNameDrop */
    drop_ptr = AllocOne(SysNameDrop);
    SysNameOf(drop_ptr) = sys_name;
    SysNameDropTypeOf(drop_ptr) = drop_type;
    switch (drop_type) {
      case BoxDrop:
	BoxPtrOf(drop_ptr) = (Box *)element_ptr;
	break;
      case ArrowDrop:
	ArrowPtrOf(drop_ptr) = (Arrow *)element_ptr;
	break;
#ifdef DEBUG
      default:
	ProgrammerErrorI("sysname-hash.c",
			 "AddSysName(): don't know SysNameDropType %d",
			 drop_type);
	break;
#endif
    }

/* add it to the hash table */
    hash_ix = HashSysName(sys_name,drop_type);
    NextOf(drop_ptr) = SysNameHashTable[hash_ix];
    SysNameHashTable[hash_ix] = drop_ptr;
}

Box *FindBoxSysName(sys_name)
  SysName sys_name;
{
    SysNameDrop *sys_name_drop_ptr;
    if ((sys_name_drop_ptr = FindSysName(sys_name,BoxDrop)) == NULL) {
	return(NULL);
    }
    return(BoxPtrOf(sys_name_drop_ptr));
}

Arrow *FindArrowSysName(sys_name)
  SysName sys_name;
{
    SysNameDrop *sys_name_drop_ptr;
    if ((sys_name_drop_ptr = FindSysName(sys_name,ArrowDrop)) == NULL) {
	return(NULL);
    }
    return(ArrowPtrOf(sys_name_drop_ptr));
}

#ifdef DEBUG
void ShowSysNameHashTable()
{
    int i,count,total;
    SysNameDrop *curr;

    printf("\nSHOW-SYS-NAME-HASH-TABLE():\n");
    for (total=i=0; i<SysNameHashTableSize; i++) {
	count = 0;
	StepLinkedList(curr,SysNameHashTable[i]) {
	    count++;
	}
	total += count;
	if (count) {
	    printf("Bucket: %d\tCount: %d\n",i,count);
	    StepLinkedList(curr,SysNameHashTable[i]) {
		printf("  ** SysName: %d\n",SysNameOf(curr));
		printf("     Type:    %s\n",
		       SysNameDropTypeOf(curr) == BoxDrop ? "BOX" : "ARROW");
		switch (SysNameDropTypeOf(curr)) {
		  case BoxDrop:
		    printf("     Value:   %x\n",(int)BoxPtrOf(curr));
		    break;
		  case ArrowDrop:
		    printf("     Value:   %x\n",(int)ArrowPtrOf(curr));
		    break;
		}
	    }
	}
    }
    printf("TOTAL NUMBER OF ENTRIES: %d\n",total);
}
#endif
