#include <gl.h>
#include <gl/device.h>
#include <gl/sphere.h>
#include <values.h>
#include <math.h>
#include "forms.h"
#include "sphform.h"

#define	PROGNAME "sphere"
#define BOXSIZE  10.0
#define NSPHERES 6

float spheres[NSPHERES][4];

unsigned int BackgroundColor = 0xff800000;
unsigned int WhiteColor      = 0xffffffff;

float rotangle = 0.0;
int stopped = TRUE;
int Write = TRUE;
long mainwin;

static void WritePolygonsNumber()
{
  char buffer[100];
  int ibox = (Coord) BOXSIZE;

  if ( Write ) {
    sprintf(buffer, "Polygons per Sphere: %d", sphgnpolys());
    cpack(WhiteColor);
    cmov2(-ibox + ibox/10, -ibox + ibox/10);
    charstr(buffer);
  }
}

static void DrawSpheres(void)
{
  register int i;
  float rx, ry, rz;

  winset(mainwin);
  czclear(BackgroundColor, getgdesc(GD_ZMAX));
  pushmatrix();
  rx = rotangle;
  ry = rotangle * 2.0;
  rz = rotangle * 1.5;
  rot(rx, 'X'); rot(ry, 'Y'); rot(rz, 'Z');

  for(i = 0; i < NSPHERES; i++) {
    sphdraw(spheres[i]);
  }
  popmatrix();
  WritePolygonsNumber();
  swapbuffers();

  if(!stopped) rotangle += 2.0;
}

static void InitSpheres(void)
{
  register int i;
  float angle = 0.0;
  float ro = BOXSIZE/2.0;

  for(i = 0, angle = 0.0; i < NSPHERES; i++, angle += (2*M_PI/NSPHERES)) {
    spheres[i][0] = ro * cos(angle);
    spheres[i][1] = ro * sin(angle);
    spheres[i][2] = 0.0;
    spheres[i][3] = BOXSIZE/10.0 + BOXSIZE/20*i;
  }
  sphmode(SPH_DEPTH, 5);
  sphmode(SPH_TESS, SPH_OCT);
  sphmode(SPH_PRIM, SPH_MESH );
  sphmode(SPH_HEMI, TRUE);
  sphmode(SPH_ORIENT, FALSE);
  Write = TRUE;
  rotangle = 0.0;
  DrawSpheres();
}


/*****
 * Callback routines for sph_form
 ****/

void s_depth(FL_OBJECT * obj, long i)
{
  sphmode(SPH_DEPTH, (int)fl_get_slider_value(obj));
  DrawSpheres();
}

void s_tess(FL_OBJECT * obj, long i)
{
  sphmode(SPH_TESS, (int)i);
  DrawSpheres();
}

void s_draw(FL_OBJECT * obj, long i)
{
  int mode = (int) i;

  sphmode(SPH_PRIM, mode);
  if (mode == SPH_MESH || mode == SPH_POLY) Write = TRUE; 
  else Write = FALSE;
  DrawSpheres();
}

void s_hemi(FL_OBJECT * obj, long i)
{
  sphmode(SPH_HEMI, (int)i);
  DrawSpheres();
}

void s_orient(FL_OBJECT * obj, long i)
{
  sphmode(SPH_ORIENT, (int)i);
  DrawSpheres();
}

void reset_sph_form()
{
  register int i;

  fl_freeze_form(sph_form);       
  for(i = 0; i<13; i++) {
    fl_set_button(db[i], 0);
  }
  fl_set_button(db[0], 1);
  fl_set_button(db[5], 1);
  fl_set_button(db[9], 1);
  fl_set_button(db[12], 1);
  fl_set_button(s_auto, 0);
  fl_set_slider_value(ds, 5);
  fl_unfreeze_form(sph_form);
  stopped = TRUE;
  InitSpheres();
}

void OpenMainWindow(void)
{ 
  prefsize(400,400);
  mainwin = winopen("Sphere demo");
  keepaspect(1,1);
  minsize(100,100);
  maxsize(700,700);
  winconstraints();

  RGBmode();
  zbuffer(TRUE);
  doublebuffer();
  gconfig();

  mmode(MPROJECTION);
  ortho(-BOXSIZE, BOXSIZE, -BOXSIZE, BOXSIZE, -BOXSIZE, BOXSIZE);
  mmode(MVIEWING);
}
  

void main(void)
{
  FL_OBJECT * obj;
  short val;

  create_form_sph_form();

  OpenMainWindow();
  DefineLight();
  ResetLight();
  InitSpheres();
  
  fl_show_form(sph_form, FL_PLACE_SIZE, FALSE, NULL);

  do {
    if (!stopped) DrawSpheres();
    obj = fl_check_forms();
    if (obj == s_reset) reset_sph_form();
    if (obj == s_auto) stopped =  ! (int) fl_get_button(obj);
    if (obj == FL_EVENT) {
      switch (fl_qread(&val)) {
      case REDRAW:
      case REDRAWICONIC: 
	  reshapeviewport();
	  DrawSpheres();
          break;
      default:
	  break;
      }
    }
  } while (obj != s_exit);
  exit(0);
}
