/*
 * $Id: hist.c,v 1.22 1995/11/16 03:08:25 coleman Exp coleman $
 *
 * hist.c - history expansion
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1995 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */

#include "zsh.h"

extern int cs, ll;

static Histent curhistent;
/* Array of word beginnings and endings in current history line. */
short *chwords;
/* Max, actual position in chwords.
 * nwords = chwordpos/2 because we record beginning and end of words.
 */
int chwordlen, chwordpos;

/* add a character to the current history word */

/**/
void
hwaddc(int c)
{
    /* Only if history line exists and lexing has not finished. */
    if (chline && !(errflag || lexstop)) {
	/* Quote un-expanded bangs in the history line. */
	if (c == bangchar && unset(NOBANGHIST))
	    hwaddc('\\');
	*hptr++ = c;

	/* Resize history line if necessary */
	if (hptr - chline >= hlinesz) {
	    int flag = 0, oldsiz = hlinesz;

	    /* Maybe already linked to a history entry. */
	    if (curhistent->text == chline)
		flag = 1;
	    chline = realloc(chline, hlinesz = oldsiz + 16);
	    if (flag)
		curhistent->text = chline;
	    hptr = chline + oldsiz;
	}
    }
}

/**/
int
hgetc(void)
{
    int c = ingetc(), qbang = 0;

    if (!stophist && !alstackind) {
	/* If necessary, expand history characters. */
	c = histsubchar(c, &qbang);
	if (c < 0) {
	    /* bad expansion */
	    errflag = lexstop = 1;
	    return ' ';
	}
    }
    hwaddc(c);

    if (expanding && !lexstop) {
	/* When expanding zle line for history */
	if (inbufct > exlast && excs > exmax - exlast) {
	    excs += inbufct - exlast;
	    exmax += inbufct - exlast;
	}
	exlast = inbufct;
	if (qbang) {
	    spaceinline(1);
	    line[cs++] = '\\';
	}
	spaceinline(1);
	line[cs++] = itok(c) ? ztokens[c - Pound] : c;
    }

    return c;
}


/* extract :s/foo/bar/ delimiters and arguments */

/**/
int
getsubsargs(char *subline)
{
    int del;
    char *ptr1, *ptr2;

    del = ingetc();
    ptr1 = hdynread2(del);
    if (!ptr1)
	return 1;
    ptr2 = hdynread2(del);
    if (strlen(ptr1)) {
	zsfree(hsubl);
	hsubl = ptr1;
    }
    zsfree(hsubr);
    hsubr = ptr2;
    if (hsubl && !strstr(subline, hsubl)) {
	zerr("substitution failed", NULL, 0);
	inerrflush();
	return 1;
    }
    return 0;
}

/* Perform history substitution, returning the next character afterwards. */

/**/
int
histsubchar(int c, int *qbang)
{
    int ev, farg, evset = -1, larg, argc, cflag = 0, bflag = 0;
    static int mev = -1, marg = -1;
    char buf[256], *ptr;
    char *sline;
    Histent ehist;

    /* look, no goto's */
    if (isfirstch && c == hatchar) {
	/* Line begins ^foo^bar */
	isfirstch = 0;
	inungetc(hatchar);
	if (!(ehist = gethist(defev))
	    || !(sline = getargs(ehist, 0, ehist->nwords-1))
	    || getsubsargs(sline))
	    return -1;
	subst(&sline, hsubl, hsubr, 0);
    } else {
	/* Line doesn't begin ^foo^bar */
	if (c != ' ')
	    isfirstch = 0;
	if (c == '\\') {
	    int g = ingetc();

	    if (g != bangchar)
		inungetc(g);
	    else {
		*qbang = 1;
		return bangchar;
	    }
	}
	if (c != bangchar)
	    return c;
	*hptr = '\0';
	if ((c = ingetc()) == '{') {
	    bflag = cflag = 1;
	    c = ingetc();
	}
	if (c == '\"') {
	    stophist = 1;
	    return ingetc();
	}
	if ((!cflag && inblank(c)) || c == '=' || c == '(' || lexstop) {
	    if (lexstop)
		lexstop = 0;
	    else
		inungetc(c);
	    return bangchar;
	}
	cflag = 0;
	ptr = buf;

	/* get event number */

	if (c == '?') {
	    for (;;) {
		c = ingetc();
		if (c == '?' || c == '\n' || lexstop)
		    break;
		else
		    *ptr++ = c;
	    }
	    if (c != '\n' && !lexstop)
		c = ingetc();
	    *ptr = '\0';
	    mev = ev = hconsearch(hsubl = ztrdup(buf), &marg);
	    evset = 0;
	    if (ev == -1) {
		inerrflush();
		zerr("no such event: %s", buf, 0);
		return -1;
	    }
	} else {
	    int t0;

	    for (;;) {
		if (inblank(c) || c == ';' || c == ':' || c == '^' ||
		    c == '$' || c == '*' || c == '%' || c == '}' ||
		    c == '\'' || c == '"' || c == '`' || lexstop)
		    break;
		if (ptr != buf) {
		    if (c == '-')
			break;
		    if ((idigit(buf[0]) || buf[0] == '-') && !idigit(c))
			break;
		}
		*ptr++ = c;
		if (c == '#' || c == bangchar) {
		    c = ingetc();
		    break;
		}
		c = ingetc();
	    }
	    *ptr = 0;
	    if (!*buf)
		if (c != '%') {
		    if (isset(CSHJUNKIEHISTORY))
			ev = curhist - 1;
		    else
			ev = defev;
		    if (c == ':' && evset == -1)
			evset = 0;
		    else
			evset = 1;
		} else {
		    if (marg != -1)
			ev = mev;
		    else
			ev = defev;
		    evset = 0;
	    } else if ((t0 = atoi(buf))) {
		ev = (t0 < 0) ? curhist + t0 : t0;
		evset = 1;
	    } else if ((unsigned)*buf == bangchar) {
		ev = curhist - 1;
		evset = 1;
	    } else if (*buf == '#') {
		ev = curhist;
		evset = 1;
	    } else if ((ev = hcomsearch(buf)) == -1) {
		zerr("event not found: %s", buf, 0);
		while (c != '\n' && !lexstop)
		    c = ingetc();
		return -1;
	    } else
		evset = 1;
	}

	/* get the event */

	if (!(ehist = gethist(defev = ev)))
	    return -1;

	/* extract the relevant arguments */

	argc = ehist->nwords - 1;
	if (c == ':') {
	    cflag = 1;
	    c = ingetc();
	    if (c == '%' && marg != -1) {
		if (!evset) {
		    ehist = gethist(defev = mev);
		    argc = ehist->nwords-1;
		} else {
		    zerr("Ambiguous history reference", NULL, 0);
		    while (c != '\n' && !lexstop)
			c = ingetc();
		    return -1;
		}

	    }
	}
	if (c == '*') {
	    farg = 1;
	    larg = argc;
	    cflag = 0;
	} else {
	    inungetc(c);
	    larg = farg = getargspec(argc, marg, evset);
	    if (larg == -2)
		return -1;
	    if (farg != -1)
		cflag = 0;
	    c = ingetc();
	    if (c == '*') {
		cflag = 0;
		larg = argc;
	    } else if (c == '-') {
		cflag = 0;
		larg = getargspec(argc, marg, evset);
		if (larg == -2)
		    return -1;
		if (larg == -1)
		    larg = argc - 1;
	    } else
		inungetc(c);
	}
	if (farg == -1)
	    farg = 0;
	if (larg == -1)
	    larg = argc;
	if (!(sline = getargs(ehist, farg, larg)))
	    return -1;

	/* do the modifiers */

	for (;;) {
	    c = (cflag) ? ':' : ingetc();
	    cflag = 0;
	    if (c == ':') {
		int gbal = 0;

		if ((c = ingetc()) == 'g') {
		    gbal = 1;
		    c = ingetc();
		}
		switch (c) {
		case 'p':
		    histdone = HISTFLAG_DONE | HISTFLAG_NOEXEC;
		    break;
		case 'h':
		    if (!remtpath(&sline)) {
			inerrflush();
			zerr("modifier failed: h", NULL, 0);
			return -1;
		    }
		    break;
		case 'e':
		    if (!rembutext(&sline)) {
			inerrflush();
			zerr("modifier failed: e", NULL, 0);
			return -1;
		    }
		    break;
		case 'r':
		    if (!remtext(&sline)) {
			inerrflush();
			zerr("modifier failed: r", NULL, 0);
			return -1;
		    }
		    break;
		case 't':
		    if (!remlpaths(&sline)) {
			inerrflush();
			zerr("modifier failed: t", NULL, 0);
			return -1;
		    }
		    break;
		case 's':
		    if (getsubsargs(sline))
			return -1; /* fall through */
		case '&':
		    if (hsubl && hsubr)
			subst(&sline, hsubl, hsubr, gbal);
		    else {
			inerrflush();
			zerr("no previous substitution with &", NULL, 0);
			return -1;
		    }
		    break;
		case 'q':
		    quote(&sline);
		    break;
		case 'x':
		    quotebreak(&sline);
		    break;
		case 'l':
		    downcase(&sline);
		    break;
		case 'u':
		    upcase(&sline);
		    break;
		default:
		    inerrflush();
		    zerr("illegal modifier: %c", NULL, c);
		    return -1;
		}
	    } else {
		if (c != '}' || !bflag)
		    inungetc(c);
		if (c != '}' && bflag) {
		    zerr("'}' expected", NULL, 0);
		    return -1;
		}
		break;
	    }
	}

    }
    /*
     * Push the expanded value onto the input stack,
     * marking this as a history word for purposes of the alias stack.
     */

    lexstop = 0;
    if (alstackind != MAXAL) {
	c = INP_ALIAS;
	alstack[alstackind++] = NULL;
    } else
	c = INP_CONT;
    for (ptr = sline; *ptr; ptr++) {
	if (ptr[0] == '\\' && ptr[1] == (char)bangchar)
	    chuck(ptr);
    }
    inpush(sline, c); /* sline from heap, don't free */
    histdone |= HISTFLAG_DONE;
    if (isset(HISTVERIFY))
	histdone |= HISTFLAG_NOEXEC | HISTFLAG_RECALL;

    /* Don't try and re-expand line. */
    return ingetc();
}

/* reset the alias stack for lexrestore () */

/**/
void
clearalstack(void)
{
    Alias ix;

    while (alstackind) {
	ix = alstack[--alstackind];
	ix->inuse = 0;
    }
}

/* unget a char and remove it from chline */

/**/
void
hungetc(int c)
{
    if (!lexstop) {
	if (expanding) {
	    cs--;
	    ll--;
	}
#if 0
	/* debugging only */
	if (hptr == chline)
	    zerr("hungetc attempted at buffer start", NULL, 0);
	else {
#endif
	    hptr--;
	    if (*hptr == (char)bangchar && unset(NOBANGHIST))
		hptr--;
#if 0
	}
#endif
    }
    inungetc(c);
}

/* begin reading a string */

/**/
void
strinbeg(void)
{
    strin = 1;
    hbegin();
    lexinit();
}

/* done reading a string */

/**/
void
strinend(void)
{
    hend();
    strin = 0;
    isfirstch = 1;
    histdone = 0;
}

/* initialize the history mechanism */

/**/
void
hbegin(void)
{
    isfirstln = isfirstch = 1;
    histremmed = errflag = histdone = spaceflag = 0;
    stophist = !interact || isset(NOBANGHIST) || unset(SHINSTDIN);
    chline = hptr = zcalloc(hlinesz = 16);
    chwords = zalloc((chwordlen = 16)*sizeof(short));
    chwordpos = 0;

    curhistent = gethistent(curhist);
    if (!curhistent->ftim)
	curhistent->ftim = time(NULL);
    if (interact && isset(SHINSTDIN) && !strin) {
	attachtty(mypgrp);
	defev = curhist++;
	if (curhist - histsiz >= 0) {
	    struct histent *tmphist = gethistent(curhist - histsiz);
	    if (tmphist->text) {
		zsfree(tmphist->text);
		tmphist->text = NULL;
	    }
	    if (tmphist->nwords) {
		zfree(tmphist->words, tmphist->nwords*2*sizeof(short));
		tmphist->words = NULL;
		tmphist->nwords = 0;
	    }
	}
	curhistent = gethistent(curhist);
	zsfree(curhistent->text);
	if (curhistent->nwords)
	    zfree(curhistent->words, curhistent->nwords*2*sizeof(short));
	curhistent->text = chline;
	curhistent->words = NULL;
	curhistent->nwords = 0;
    } else
	histremmed = 1;
}

/* say we're done using the history mechanism */

/**/
int
hend(void)
{
    int flag, save = 1;
    Histent he;

    if (!chline)
	return 1;
    if (!interact || strin || unset(SHINSTDIN)) {
	zfree(chline, hlinesz);
	zfree(chwords, chwordlen*sizeof(short));
	chline = NULL;
	return 1;
    }
    flag = histdone;
    histdone = 0;
    if (hptr < chline + 1)
	save = 0;
    else {
	*hptr = '\0';
	if (hptr[-1] == '\n')
	    if (chline[1]) {
		*--hptr = '\0';
	    } else
		save = 0;
	he = gethistent(curhist - 1);
	if (!*chline || !strcmp(chline, "\n") ||
	    (isset(HISTIGNOREDUPS) && he->text && !strcmp(he->text, chline)) ||
	    (isset(HISTIGNORESPACE) && spaceflag))
	    save = 0;
    }
    if (flag & (HISTFLAG_DONE | HISTFLAG_RECALL)) {
	char *ptr;

	ptr = ztrdup(chline);
	if ((flag & (HISTFLAG_DONE | HISTFLAG_RECALL)) == HISTFLAG_DONE) {
	    fprintf(stderr, "%s\n", ptr);
	    fflush(stderr);
	}
	if (flag & HISTFLAG_RECALL) {
	    permalloc();
	    pushnode(bufstack, ptr);
	    lastalloc();
	    save = 0;
	} else
	    zsfree(ptr);
    }
    curhistent->stim = time(NULL);
    curhistent->ftim = 0L;
    curhistent->flags = 0;
    if (save) {
#if 0
	/* debugging only */
	if (chwordpos%2) {
	    hwend();
	    fprintf(stderr, "internal:  uncompleted line in history\n");
	}
#endif
	/* get rid of pesky \n which we've already nulled out */
	if (!chline[chwords[chwordpos-2]])
	    chwordpos -= 2;
	if ((curhistent->nwords = chwordpos/2)) {
	    curhistent->words =
		(short *)zalloc(curhistent->nwords*2*sizeof(short));
	    memcpy(curhistent->words, chwords,
		   curhistent->nwords*2*sizeof(short));
	}
    } else
	remhist();
    if (chline && !curhistent->text)
	zfree(chline, hlinesz);
    if (curhistent->text) {
	char *s = ztrdup(curhistent->text);

	zfree(curhistent->text, hlinesz);
	curhistent->text = s;
    }
    zfree(chwords, chwordlen*sizeof(short));
    chline = NULL;
    return !(flag & HISTFLAG_NOEXEC || errflag);
}

/* remove the current line from the history List */

/**/
void
remhist(void)
{
    if (!histremmed) {
	histremmed = 1;
	curhist--;
    }
}

/* Gives current expansion word if not last word before chwordpos. */
static int hwgetword = -1;

/* begin a word */

/**/
void
hwbegin(int offset)
{
    if (chwordpos%2)
	chwordpos--;	/* make sure we're on a word start, not end */
    /* If we're expanding an alias, we should overwrite the expansion
     * in the history.
     * If we're in a string, we don't need the full history line
     * and can overwrite also.
     */
    if ((alstackind || strin) && (alstackind != 1 || alstack[0]))
	hwgetword = chwordpos;
    else
	hwgetword = -1;
    chwords[chwordpos++] = hptr - chline + offset;
}

/* add a word to the history List */

/**/
void
hwend(void)
{
    if (chwordpos%2 && chline) {
	/* end of word reached and we've already begun a word */
	if (hptr > chline + chwords[chwordpos-1]) {
	    chwords[chwordpos++] = hptr - chline;
	    if (chwordpos >= chwordlen) {
		chwords = realloc(chwords,
				  (chwordlen += 16)*sizeof(short));
	    }
	    if (hwgetword > -1) {
		/* We want to reuse the current word position */
		chwordpos = hwgetword;
		/* Start from where previous word ended, if possible */
		hptr = chline + chwords[chwordpos ? chwordpos - 1 : 0];
	    }
	} else {
	    /* scrub that last word, it doesn't exist */
	    chwordpos--;
	}
    }
    if (alstat == ALSTAT_JUNK)
	alstat = 0;
}

/* Go back to immediately after the last word, skipping space. */

/**/
void
histbackword(void)
{
    if (!(chwordpos%2) && chwordpos)
	hptr = chline + chwords[chwordpos-1];
}

/* Get the start and end point of the current history word */

/**/
void
hwget(char **startptr)
{
    int pos = hwgetword > -1 ? hwgetword : chwordpos - 2;

#if 0
    /* debugging only */
    if (hwgetword == -1 && !chwordpos) {
	/* no words available */
	zerr("hwget called with no words.", NULL, 0);
	*startptr = "";
	return;
    } 
    else if (hwgetword == -1 && chwordpos%2) {
	zerr("hwget called in middle of word.", NULL, 0);
	*startptr = "";
	return;
    }
#endif

    *startptr = chline + chwords[pos];
    chline[chwords[++pos]] = '\0';
}

/* Replace the current history word with rep, if different */

/**/
void
hwrep(char *rep)
{
    char *start;
    hwget(&start);

    if (!strcmp(rep, start))
	return;
    
    hptr = start;
    chwordpos = (hwgetword > -1) ? hwgetword : chwordpos - 2;
    hwbegin(0);
    while (*rep)
	hwaddc(*rep++);
    hwend();

    /* We need the null since this is called from exalias() via spckword(),
     * which means we are expecting the history word to be NULL-terminated.
     */
    *hptr = '\0';
}

/* Get the entire current line, deleting it in the history. */

/**/
char *
hgetline(void)
{
    /* Currently only used by pushpopinput().
     * It's necessary to prevent that from getting too pally with
     * the history code.
     */
    char *ret;

    if (!chline || hptr == chline)
	return NULL;
    *hptr = '\0';
    ret = dupstring(chline);

    /* reset line */
    hptr = chline;
    chwordpos = 0;
    hwgetword = -1;

    return ret;
}

/* get an argument specification */

/**/
int
getargspec(int argc, int marg, int evset)
{
    int c, ret = -1;

    if ((c = ingetc()) == '0')
	return 0;
    if (idigit(c)) {
	ret = 0;
	while (idigit(c)) {
	    ret = ret * 10 + c - '0';
	    c = ingetc();
	}
	inungetc(c);
    } else if (c == '^')
	ret = 1;
    else if (c == '$')
	ret = argc;
    else if (c == '%') {
	if (evset) {
	    inerrflush();
	    zerr("Ambiguous history reference", NULL, 0);
	    return -2;
	}
	if (marg == -1) {
	    inerrflush();
	    zerr("%% with no previous word matched", NULL, 0);
	    return -2;
	}
	ret = marg;
    } else
	inungetc(c);
    return ret;
}

/* do ?foo? search */

/**/
int
hconsearch(char *str, int *marg)
{
    int t0, t1 = 0;
    char *s;
    Histent he;

    for (t0 = curhist - 1; (he = quietgethist(t0)); t0--)
	if ((s = strstr(he->text, str))) {
	    int pos = s - he->text;
	    while (t1 < he->nwords && he->words[2*t1] <= pos)
		t1++;
	    *marg = t1;
	    return t0;
	}
    return -1;
}

/* do !foo search */

/**/
int
hcomsearch(char *str)
{
    int t0;
    char *hs;

    for (t0 = curhist - 1; (hs = quietgetevent(t0)); t0--)
	if (!strncmp(hs, str, strlen(str)))
	    return t0;
    return -1;
}

/* various utilities for : modifiers */

/**/
int
remtpath(char **junkptr)
{
    char *str = *junkptr, *remcut;

    if ((remcut = strrchr(str, '/'))) {
	if (str != remcut)
	    *remcut = '\0';
	else
	    str[1] = '\0';
	return 1;
    }
    return 0;
}

/**/
int
remtext(char **junkptr)
{
    char *str = *junkptr, *remcut;

    if ((remcut = strrchr(str, '.')) && remcut != str) {
	*remcut = '\0';
	return 1;
    }
    return 0;
}

/**/
int
rembutext(char **junkptr)
{
    char *str = *junkptr, *remcut;

    if ((remcut = strrchr(str, '.')) && remcut != str) {
	*junkptr = dupstring(remcut + 1);	/* .xx or xx? */
	return 1;
    }
    return 0;
}

/**/
int
remlpaths(char **junkptr)
{
    char *str = *junkptr, *remcut;

    if ((remcut = strrchr(str, '/'))) {
	*remcut = '\0';
	*junkptr = dupstring(remcut + 1);
	return 1;
    }
    return 0;
}

/**/
int
makeuppercase(char **junkptr)
{
    char *str = *junkptr;

    for (; *str; str++)
	*str = tuupper(*str);
    return 1;
}

/**/
int
makelowercase(char **junkptr)
{
    char *str = *junkptr;

    for (; *str; str++)
	*str = tulower(*str);
    return 1;
}

/**/
int
makecapitals(char **junkptr)
{
    char *str = *junkptr;

    for (; *str;) {
	for (; *str && !ialnum(*str); str++);
	if (*str)
	    *str = tuupper(*str), str++;
	for (; *str && ialnum(*str); str++)
	    *str = tulower(*str);
    }
    return 1;
}

/**/
void
subst(char **strptr, char *in, char *out, int gbal)
{
    char *str = *strptr, *instr = *strptr, *substcut, *sptr, *oldstr;
    int off, inlen, outlen;

    if (!*in)
	in = str;
    if (!(substcut = (char *)strstr(str, in)))
	return;
    inlen = strlen(in);
    sptr = convamps(out, in, inlen);
    outlen = strlen(sptr);

    do {
	*substcut = '\0';
	off = substcut - *strptr + outlen;
	substcut += inlen;
	*strptr = tricat(oldstr = *strptr, sptr, substcut);
	if (oldstr != instr)
	    zsfree(oldstr);
	str = (char *)*strptr + off;
    } while (gbal && (substcut = (char *)strstr(str, in)));
}

/**/
char *
convamps(char *out, char *in, int inlen)
{
    char *ptr, *ret, *pp;
    int slen, sdup = 0;

    for (ptr = out, slen = 0; *ptr; ptr++, slen++)
	if (*ptr == '\\')
	    ptr++, sdup = 1;
	else if (*ptr == '&')
	    slen += inlen - 1, sdup = 1;
    if (!sdup)
	return out;
    ret = pp = (char *)alloc(slen + 1);
    for (ptr = out; *ptr; ptr++)
	if (*ptr == '\\')
	    *pp++ = *++ptr;
	else if (*ptr == '&') {
	    strcpy(pp, in);
	    pp += inlen;
	} else
	    *pp++ = *ptr;
    *pp = '\0';
    return ret;
}

/**/
struct histent *
quietgethist(int ev)
{
    if (ev < firsthist() || ev > curhist)
	return NULL;
    return gethistent(ev);
}

/**/
char *
quietgetevent(int ev)
{
    Histent ent = quietgethist(ev);

    return ent ? ent->text : NULL;
}

/**/
Histent
gethist(int ev)
{
    Histent ret;

    ret = quietgethist(ev);
    if (!ret) {
	inerrflush();
	zerr("no such event: %d", NULL, ev);
    }
    return ret;
}

/**/
char *
getargs(Histent elist, int arg1, int arg2)
{
    char *ret;
    int pos1;

    if (arg1 >= elist->nwords || arg2 >= elist->nwords) {
	/* remember, argN is indexed from 0, nwords is total no. of words */
	inerrflush();
	if (!elist->nwords)
	    zerr("no words available from current command", NULL, 0);
	else
	    zerr("no such word in event", NULL, 0);
	return NULL;
    }

    pos1 = elist->words[2*arg1];
    ret = dupstring(elist->text + pos1);
    ret[elist->words[2*arg2+1] - pos1] = '\0';
    
    return ret;
}

/**/
void
upcase(char **x)
{
    char *pp = *(char **)x;

    for (; *pp; pp++)
	*pp = tuupper(*pp);
}

/**/
void
downcase(char **x)
{
    char *pp = *(char **)x;

    for (; *pp; pp++)
	*pp = tulower(*pp);
}

/**/
int
quote(char **tr)
{
    char *ptr, *rptr, **str = (char **)tr;
    int len = 3;
    int inquotes = 0;

    for (ptr = *str; *ptr; ptr++, len++)
	if (*ptr == '\'') {
	    len += 3;
	    if (!inquotes)
		inquotes = 1;
	    else
		inquotes = 0;
	} else if (inblank(*ptr) && !inquotes && ptr[-1] != '\\')
	    len += 2;
    ptr = *str;
    *str = rptr = (char *)alloc(len);
    *rptr++ = '\'';
    for (; *ptr; ptr++)
	if (*ptr == '\'') {
	    if (!inquotes)
		inquotes = 1;
	    else
		inquotes = 0;
	    *rptr++ = '\'';
	    *rptr++ = '\\';
	    *rptr++ = '\'';
	    *rptr++ = '\'';
	} else if (inblank(*ptr) && !inquotes && ptr[-1] != '\\') {
	    *rptr++ = '\'';
	    *rptr++ = *ptr;
	    *rptr++ = '\'';
	} else
	    *rptr++ = *ptr;
    *rptr++ = '\'';
    *rptr++ = 0;
    str[1] = NULL;
    return 0;
}

/**/
int
quotebreak(char **tr)
{
    char *ptr, *rptr, **str = (char **)tr;
    int len = 3;

    for (ptr = *str; *ptr; ptr++, len++)
	if (*ptr == '\'')
	    len += 3;
	else if (inblank(*ptr))
	    len += 2;
    ptr = *str;
    *str = rptr = (char *)alloc(len);
    *rptr++ = '\'';
    for (; *ptr;)
	if (*ptr == '\'') {
	    *rptr++ = '\'';
	    *rptr++ = '\\';
	    *rptr++ = '\'';
	    *rptr++ = '\'';
	    ptr++;
	} else if (inblank(*ptr)) {
	    *rptr++ = '\'';
	    *rptr++ = *ptr++;
	    *rptr++ = '\'';
	} else
	    *rptr++ = *ptr++;
    *rptr++ = '\'';
    *rptr++ = '\0';
    return 0;
}

/* read an arbitrary amount of data into a buffer until stop is found */

/**/
char *
hdynread(int stop)
{
    int bsiz = 256, ct = 0, c;
    char *buf = (char *)zalloc(bsiz), *ptr;

    ptr = buf;
    while ((c = ingetc()) != stop && c != '\n' && !lexstop) {
	if (c == '\\')
	    c = ingetc();
	*ptr++ = c;
	if (++ct == bsiz) {
	    buf = realloc(buf, bsiz *= 2);
	    ptr = buf + ct;
	}
    }
    *ptr = 0;
    if (c == '\n') {
	inungetc('\n');
	zerr("delimiter expected", NULL, 0);
	zfree(buf, bsiz);
	return NULL;
    }
    return buf;
}

/**/
char *
hdynread2(int stop)
{
    int bsiz = 256, ct = 0, c;
    char *buf = (char *)zalloc(bsiz), *ptr;

    ptr = buf;
    while ((c = ingetc()) != stop && c != '\n' && !lexstop) {
	if (c == '\n') {
	    inungetc(c);
	    break;
	}
	if (c == '\\')
	    c = ingetc();
	*ptr++ = c;
	if (++ct == bsiz) {
	    buf = realloc(buf, bsiz *= 2);
	    ptr = buf + ct;
	}
    }
    *ptr = 0;
    if (c == '\n')
	inungetc('\n');
    return buf;
}

/**/
void
inithist(void)
{
    histentct = histsiz;
    histentarr = (Histent) zcalloc(histentct * sizeof *histentarr);
}

/**/
void
resizehistents(void)
{
    int newentct, t0, t1, firstlex;
    Histent newarr;

    newentct = histsiz;
    newarr = (Histent) zcalloc(newentct * sizeof *newarr);
    firstlex = curhist - histsiz + 1;
    t0 = firsthist();
    if (t0 < curhist - newentct)
	t0 = curhist - newentct;
    t1 = t0 % newentct;
    for (; t0 <= curhist; t0++) {
	newarr[t1] = *gethistent(t0);
	if (t0 < firstlex) {
	    zsfree(newarr[t1].text);
	    newarr[t1].text = NULL;
	}
	t1++;
	if (t1 == newentct)
	    t1 = 0;
    }
    free(histentarr);
    histentarr = newarr;
    histentct = newentct;
}

/**/
void
readhistfile(char *s, int err)
{
    char buf[1024];
    FILE *in;
    Histent ent;
    time_t tim = time(NULL);
    short *wordlist;
    int nwordpos, nwordlist;

    if (!s)
	return;
    if ((in = fopen(s, "r"))) {
	nwordlist = 16;
	wordlist = (short *)zalloc(nwordlist*sizeof(short));

	while (fgets(buf, sizeof(buf), in)) {
	    int l = strlen(buf);
	    char *pt, *start;

	    while (l && buf[l - 1] == '\n') {
		buf[l - 1] = '\0';
		if (l > 1 && buf[l - 2] == '\\') {
		    buf[l - 2] = '\n';
		    fgets(buf + l - 1, sizeof(buf) - (l - 1), in);
		    l = strlen(buf);
		} else
		    break;
	    }

	    ent = gethistent(++curhist);
	    pt = buf;
	    if (*pt == ':') {
		pt++;
		ent->stim = atol(pt);
		for (; *pt != ':' && *pt; pt++);
		if (*pt) {
		    pt++;
		    ent->ftim = atol(pt);
		    for (; *pt != ';' && *pt; pt++);
		    if (*pt)
			pt++;
		} else {
		    ent->ftim = tim;
		}
		if (ent->stim == 0)
		    ent->stim = tim;
		if (ent->ftim == 0)
		    ent->ftim = tim;
	    } else {
		ent->ftim = ent->stim = tim;
	    }

	    zsfree(ent->text);
	    ent->text = ztrdup(pt);
	    ent->flags = HIST_OLD;
	    if (ent->nwords)
		zfree(ent->words, ent->nwords*2*sizeof(short));

	    /* Divide up the words.  We don't know how it lexes,
	       so just look for spaces.
	       */
	    nwordpos = 0;
	    start = pt;
	    do {
		while (*pt == ' ')
		    pt++;
		if (*pt) {
		    if (nwordpos >= nwordlist)
			wordlist = realloc(wordlist,
					   (nwordlist += 16)*sizeof(short));
		    wordlist[nwordpos++] = pt - start;
		    while (*pt && *pt != ' ')
			pt++;
		    wordlist[nwordpos++] = pt - start;
		}
	    } while (*pt);

	    ent->nwords = nwordpos/2;
	    if (ent->nwords) {
		ent->words = (short *)zalloc(nwordpos*sizeof(short));
		memcpy(ent->words, wordlist, nwordpos*sizeof(short));
	    } else
		ent->words = (short *)NULL;
	}
	fclose(in);

	zfree(wordlist, nwordlist*sizeof(short));
    } else if (err)
	zerr("can't read history file", s, 0);
}

/**/
void
savehistfile(char *s, int err, int app)
{
    char *t;
    FILE *out;
    int ev;
    Histent ent;

    if (!s || !interact || savehist == 0)
	return;
    ev = curhist - savehist + 1;
    if (ev < firsthist())
	ev = firsthist();
    if (app & 1)
	out = fdopen(open(s, O_CREAT | O_WRONLY | O_APPEND, 0600), "a");
    else
	out = fdopen(open(s, O_CREAT | O_WRONLY | O_TRUNC, 0600), "w");
    if (out) {
	for (; ev <= curhist; ev++) {
	    ent = gethistent(ev);
	    if (app & 2) {
		if (ent->flags & HIST_OLD)
		    continue;
		ent->flags |= HIST_OLD;
	    }
	    t = ent->text;
	    if (isset(EXTENDEDHISTORY)) {
		fprintf(out, ": %ld:%ld;",
			(long)ent->stim,
			(long)ent->ftim);
	    } else if (*t == ':')
		fputc('\\', out);

	    for (; *t; t++) {
		if (*t == '\n')
		    fputc('\\', out);
		fputc(*t, out);
	    }
	    fputc('\n', out);
	}
	fclose(out);

	if (app & 2 && (out = fopen(s, "r"))) {
	    char **store, buf[1024], **ptr;
	    int i, l, histnum = 0;

	    store = (char **)zcalloc((savehist + 1) * sizeof *store);
	    while (fgets(buf, sizeof(buf), out)) {
		l = strlen(buf);
		if (l > 1)
		    while (l < sizeof(buf) - 1 && buf[l - 2] == '\\') {
			fgets(buf + l, sizeof buf - l, out);
			l = strlen(buf);
		    }
		if (store[i = histnum % savehist])
		    free(store[i]);
		store[i] = (char *)zalloc(l + 1);
		strcpy(store[i], buf);
		histnum++;
	    }
	    fclose(out);
	    if ((out = fdopen(open(s, O_WRONLY | O_TRUNC, 0600), "w"))) {
		if (histnum < savehist)
		    for (i = 0; i < histnum; i++)
			fprintf(out, "%s", store[i]);
		else
		    for (i = histnum; i < histnum + savehist; i++)
			fprintf(out, "%s", store[i % savehist]);
		fclose(out);
	    }
	    for (ptr = store; *ptr; ptr++)
		zsfree(*ptr);
	    free(store);
	}
    } else if (err)
	zerr("can't write history file %s", s, 0);
}

/**/
int
firsthist(void)
{
    int ev;
    Histent ent;

    ev = curhist - histentct + 1;
    if (ev < 1)
	ev = 1;
    do {
	ent = gethistent(ev);
	if (ent->text)
	    break;
	ev++;
    }
    while (ev < curhist);
    return ev;
}

