/*
 * $Id: exec.c,v 1.103 1996/04/25 03:05:53 coleman Exp coleman $
 *
 * exec.c - command execution
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1995 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */

#include "zsh.h"

extern int sigtrapped[VSIGCOUNT];
extern List sigfuncs[VSIGCOUNT];

#define execerr() { if (forked) _exit(1); \
	closemnodes(mfds); lastval = 1; return; }

static LinkList args;
static int doneps4;

/* parse string into a list */

/**/
List
parse_string(char *s)
{
    List l;

    inpush(s, 0);
    strinbeg();
    pushheap();
    l = parse_list();
    strinend();
    inpop();
    if (!l)
	popheap();
    return l;
}

/* fork and set limits */

/**/
pid_t
zfork(void)
{
    pid_t pid;

#ifdef HAVE_GETRLIMIT
    int i;
#endif

    if (thisjob >= MAXJOB - 1) {
	zerr("job table full", NULL, 0);
	return -1;
    }
    pid = fork();
    if (pid == -1) {
	zerr("fork failed: %e", NULL, errno);
	return -1;
    }
#ifdef HAVE_GETRLIMIT
    if (!pid)
	/* set resource limits for the child process */
	for (i = 0; i < RLIM_NLIMITS; i++)
	    setrlimit(i, limits + i);
#endif
    return pid;
}


int list_pipe = 0, simple_pline = 0;

static pid_t list_pipe_pid;
static int nowait, pline_level = 0;
static int list_pipe_child = 0, list_pipe_job;
static char list_pipe_text[JOBTEXTSIZE];
static LinkList last_file_list;

/**/
void
deletepipejob(void)
{
    if (!list_pipe) {
	last_file_list = jobtab[thisjob].filelist;
	jobtab[thisjob].filelist = NULL;
	deletejob(jobtab + thisjob);
    }
}

/* execute a current shell command */

/**/
int
execcursh(Cmd cmd)
{
    deletepipejob();
    execlist(cmd->u.list, 1, cmd->flags & CFLAG_EXEC);
    cmd->u.list = NULL;
    return lastval;
}

/* execve after handling $_ and #! */

#define POUNDBANGLIMIT 64

/**/
int
zexecve(char *pth, char **argv)
{
    int eno;
    static char buf[PATH_MAX * 2];
    char **eep;

    for (eep = environ; *eep; eep++)
	if (**eep == '_' && (*eep)[1] == '=')
	    break;
    buf[0] = '_';
    buf[1] = '=';
    if (*pth == '/')
	strcpy(buf + 2, pth);
    else
	sprintf(buf + 2, "%s/%s", pwd, pth);
    if (!*eep)
	eep[1] = NULL;
    *eep = buf;
    execve(pth, argv, environ);

    /* If the execve returns (which in general shouldn't happen),   *
     * then check for an errno equal to ENOEXEC.  This errno is set *
     * if the process file has the appropriate access permission,   *
     * but has an invalid magic number in its header.               */
    if ((eno = errno) == ENOEXEC) {
	char execvebuf[POUNDBANGLIMIT + 1], *ptr, *ptr2, *argv0;
	int fd, ct, t0;

	if ((fd = open(pth, O_RDONLY)) >= 0) {
	    argv0 = *argv;
	    *argv = pth;
	    ct = read(fd, execvebuf, POUNDBANGLIMIT);
	    close(fd);
	    if (ct > 0) {
		if (execvebuf[0] == '#') {
		    if (execvebuf[1] == '!') {
			for (t0 = 0; t0 != ct; t0++)
			    if (execvebuf[t0] == '\n')
				execvebuf[t0] = '\0';
			execvebuf[POUNDBANGLIMIT] = '\0';
			for (ptr = execvebuf + 2; *ptr && *ptr == ' '; ptr++);
			for (ptr2 = ptr; *ptr && *ptr != ' '; ptr++);
			if (*ptr) {
			    *ptr = '\0';
			    argv[-2] = ptr2;
			    argv[-1] = ptr + 1;
			    execve(ptr2, argv - 2, environ);
			} else {
			    argv[-1] = ptr2;
			    execve(ptr2, argv - 1, environ);
			}
		    } else {
			argv[-1] = "sh";
			execve("/bin/sh", argv - 1, environ);
		    }
		} else {
		    for (t0 = 0; t0 != ct; t0++)
			if (!execvebuf[t0])
			    break;
		    if (t0 == ct) {
			argv[-1] = "sh";
			execve("/bin/sh", argv - 1, environ);
		    }
		}
	    } else
		eno = errno;
	    *argv = argv0;
	} else
	    eno = errno;
    }
    return eno;
}

#define MAXCMDLEN (PATH_MAX*4)

/* execute an external command */

/**/
void
execute(Cmdnam not_used_yet, int dash)
{
    Cmdnam cn;
    static LinkList exargs;
    char buf[MAXCMDLEN], buf2[MAXCMDLEN];
    char *s, *z, *arg0;
    char **argv, **pp;
    int eno = 0, ee;

    /* If the parameter STTY is set in the command's environment, *
     * we first run the stty command with the value of this       *
     * parameter as it arguments.                                 */
    if (!exargs && (s = zgetenv("STTY"))) {
	char *t;

	exargs = args;	/* this prevents infinite recursion */
	args = NULL;
	t = tricat("stty", " ", s);
	execstring(t, 1, 0);
	zsfree(t);
	args = exargs;
	exargs = NULL;
    }

    arg0 = (char *) peekfirst(args);
    cn = (Cmdnam) cmdnamtab->getnode(cmdnamtab, arg0);

    /* If ARGV0 is in the commands environment, we use *
     * that as argv[0] for this external command       */
    if ((z = zgetenv("ARGV0"))) {
	setdata(firstnode(args), (void *) ztrdup(z));
	delenv(z - 6);
    } else if (dash) {
    /* Else if the pre-command `-' was given, we add `-' *
     * to the front of argv[0] for this command.         */
	sprintf(buf2, "-%s", arg0);
	setdata(firstnode(args), (void *) ztrdup(buf2));
    }

    argv = makecline(args);
    child_unblock();
    if ((int) strlen(arg0) > PATH_MAX) {
	zerr("command too long: %s", arg0, 0);
	_exit(1);
    }
    for (s = arg0; *s; s++)
	if (*s == '/') {
	    errno = zexecve(arg0, argv);
	    if (arg0 == s || unset(PATHDIRS) ||
		(arg0[0] == '.' && (arg0 + 1 == s ||
				    (arg0[1] == '.' && arg0 + 2 == s)))) {
		zerr("%e: %s", arg0, errno);
		_exit(1);
	    }
	    break;
	}

    if (cn) {
	char nn[PATH_MAX];

	if (cn->flags & HASHED)
	    strcpy(nn, cn->u.cmd);
	else {
	    for (pp = path; pp < cn->u.name; pp++)
		if (**pp == '.' && (*pp)[1] == '\0') {
		    ee = zexecve(arg0, argv);
		    if (ee != ENOENT)
			eno = ee;
		} else if (**pp != '/') {
		    z = buf;
		    strucpy(&z, *pp);
		    *z++ = '/';
		    strcpy(z, arg0);
		    ee = zexecve(buf, argv);
		    if (ee != ENOENT)
			eno = ee;
		}
	    strcpy(nn, cn->u.name ? *(cn->u.name) : "");
	    strcat(nn, "/");
	    strcat(nn, cn->nam);
	}
	ee = zexecve(nn, argv);

	if (ee != ENOENT)
	    eno = ee;
    }
    for (pp = path; *pp; pp++)
	if ((*pp)[0] == '.' && !(*pp)[1]) {
	    ee = zexecve(arg0, argv);
	    if (ee != ENOENT)
		eno = ee;
	} else {
	    z = buf;
	    strucpy(&z, *pp);
	    *z++ = '/';
	    strcpy(z, arg0);
	    ee = zexecve(buf, argv);
	    if (ee != ENOENT)
		eno = ee;
	}
    if (eno)
	zerr("%e: %s", arg0, eno);
    else
	zerr("command not found: %s", arg0, 0);
    _exit(1);
}

#define try(X) { if (iscom(X)) return ztrdup(X); }

/* get the full pathname of an external command */

/**/
char *
findcmd(char *arg0)
{
    char **pp;
    char *z, *s, buf[MAXCMDLEN];
    Cmdnam cn;

    cn = (Cmdnam) cmdnamtab->getnode(cmdnamtab, arg0);
    if (!cn && isset(HASHCMDS))
	cn = hashcmd(arg0, path);
    if ((int) strlen(arg0) > PATH_MAX)
	return NULL;
    for (s = arg0; *s; s++)
	if (*s == '/') {
	    try(arg0);
	    if (arg0 == s || unset(PATHDIRS)) {
		return NULL;
	    }
	    break;
	}
    if (cn) {
	char nn[PATH_MAX];

	if (cn->flags & HASHED)
	    strcpy(nn, cn->u.cmd);
	else {
	    for (pp = path; pp < cn->u.name; pp++)
		if (**pp != '/') {
		    z = buf;
		    strucpy(&z, *pp);
		    *z++ = '/';
		    strcpy(z, arg0);
		    try(buf);
		}
	    strcpy(nn, cn->u.name ? *(cn->u.name) : "");
	    strcat(nn, "/");
	    strcat(nn, cn->nam);
	}
	try(nn);
    }
    for (pp = path; *pp; pp++) {
	z = buf;
	strucpy(&z, *pp);
	*z++ = '/';
	strcpy(z, arg0);
	try(buf);
    }
    return NULL;
}

/**/
int
iscom(char *s)
{
    struct stat statbuf;

    return (access(s, X_OK) == 0 && stat(s, &statbuf) >= 0 &&
	    S_ISREG(statbuf.st_mode));
}

/**/
int
isrelative(char *s)
{
    if (*s != '/')
	return 1;
    for (; *s; s++)
	if (*s == '.' && s[-1] == '/' &&
	    (s[1] == '/' || s[1] == '\0' ||
	     (s[1] == '.' && (s[2] == '/' || s[2] == '\0'))))
	    return 1;
    return 0;
}

/**/
Cmdnam
hashcmd(char *arg0, char **pp)
{
    Cmdnam cn;
    char *s, buf[PATH_MAX];
    char **pq;

    for (; *pp; pp++)
	if (**pp == '/') {
	    s = buf;
	    strucpy(&s, *pp);
	    *s++ = '/';
	    strcpy(s, arg0);
	    if (iscom(buf))
		break;
	}

    if (!*pp || isrelative(*pp))
	return NULL;

    cn = (Cmdnam) zcalloc(sizeof *cn);
    cn->flags = 0;
    cn->u.name = pp;
    cmdnamtab->addnode(cmdnamtab, ztrdup(arg0), cn);

    if (isset(HASHDIRS)) {
	for (pq = pathchecked; pq <= pp; pq++)
	    hashdir(pq);
	pathchecked = pp + 1;
    }

    return cn;
}

/* execute a string */

/**/
void
execstring(char *s, int dont_change_job, int exiting)
{
    List list;

    if ((list = parse_string(s))) {
	execlist(list, dont_change_job, exiting);
	popheap();
    }
}

/* Main routine for executing a list.                                *
 * exiting means that the (sub)shell we are in is a definite goner   *
 * after the current list is finished, so we may be able to exec the *
 * last command directly instead of forking.  If dont_change_job is  *
 * nonzero, then restore the current job number after executing the  *
 * list.                                                             */

/**/
void
execlist(List list, int dont_change_job, int exiting)
{
    Sublist slist;
    static int donetrap;
    int ret, cj;
    int ornext, alt;
    int old_pline_level, old_list_pipe;

    cj = thisjob;
    old_pline_level = pline_level;
    old_list_pipe = list_pipe;

    if (sourcelevel && unset(SHINSTDIN))
	pline_level = list_pipe = 0;

    /* Loop over all sets of comands separated by newline, *
     * semi-colon or ampersand (`sublists').               */
    while (list && list != &dummy_list && !breaks && !retflag) {
	/* Reset donetrap:  this ensures that a trap is only *
	 * called once for each sublist that fails.          */
	donetrap = 0;
	simplifyright(list);

	slist = list->left;
	while (slist) {
	    /* Loop through code followed by &&, ||, or end of sublist. */
	    if (slist->type == END) {
		/* End of sublist; just execute, ignoring status. */
		execpline(slist, list->type, !list->right && exiting);
		break;
	    }
	    ret = execpline(slist, Z_SYNC, 0);
	    /* Next pipeline is either or'd or and'd with *
	     * this one; find which.                      */
	    ornext = (slist->type == ORNEXT);

	    /* Now check if either (1) it's an `or', and the return code *
	     * was zero, or (2) it's an `and', and the return code was   *
	     * non-zero.  If so then skip pipelines until we find a      *
	     * sublist followed by the opposite type.                    */
	    if (ornext ^ (ret != 0)) {
		alt = ornext ? ANDNEXT : ORNEXT;
		while ((slist = slist->right))
		    if (slist->type == alt)
			break;
		if (!slist) {
		    /* We've skipped to the end of the list, not executing *
		     * the final pipeline, so don't perform error handling *
		     * for this sublist.                                   */
		    donetrap = 1;
		    break;
		}
	    }

	    slist = slist->right;
	}

	if (sigtrapped[SIGDEBUG])
	    dotrap(SIGDEBUG);

	/* Check whether we are suppressing traps/errexit *
	 * (typically in init scripts) and if we haven't  *
	 * already performed them for this sublist.       */
	if (!noerrexit && !donetrap) {
	    if (sigtrapped[SIGZERR] && lastval) {
		dotrap(SIGZERR);
		donetrap = 1;
	    }
	    if (lastval && isset(ERREXIT)) {
		if (sigtrapped[SIGEXIT])
		    dotrap(SIGEXIT);
		exit(lastval);
	    }
	}

	/* errflag = 0; */
	list = list->right;
    }

    pline_level = old_pline_level;
    list_pipe = old_list_pipe;
    if (dont_change_job)
	thisjob = cj;
}

/* Execute a pipeline.                                                *
 * last1 is a flag that this command is the last command in a shell   *
 * that is about to exit, so we can exec instead of forking.  It gets *
 * passed all the way down to execcmd() which actually makes the      *
 * decision.  A 0 is always passed if the command is not the last in  *
 * the pipeline.                                                      */

/**/
int
execpline(Sublist l, int how, int last1)
{
    int ipipe[2], opipe[2];
    int pj, newjob;
    static int lastwj;

    pj = thisjob;
    if (!l)
	return 0;
    if (!l->left)
	return lastval = (l->flags & PFLAG_NOT) != 0;
    ipipe[0] = ipipe[1] = opipe[0] = opipe[1] = 0;
    child_block();
    if ((thisjob = newjob = getfreejob()) == -1)
	return 1;
    initjob();
    jobtab[newjob].filelist = last_file_list;
    last_file_list = NULL;
    if (how & Z_TIMED)
	jobtab[thisjob].stat |= STAT_TIMED;
    if (l->flags & PFLAG_COPROC) {
	how = Z_ASYNC;
	if (coprocin >= 0) {
	    close(coprocin);
	    close(coprocout);
	}
	mpipe(ipipe);
	mpipe(opipe);
	coprocin = ipipe[0];
	coprocout = opipe[1];
    }
    if (!pline_level++) {
	list_pipe_job = newjob;
	nowait = 0;
    }
    list_pipe_pid = lastwj = 0;
    if (pline_level == 1)
	simple_pline = (l->left->type == END);
    execpline2(l->left, how, opipe[0], ipipe[1], last1);
    pline_level--;
    if (how & Z_ASYNC) {
	lastwj = newjob;
	if (l->flags & PFLAG_COPROC)
	    close(ipipe[1]);
	spawnjob();
	child_unblock();
	return 0;
    } else {
	if (newjob != lastwj) {
	    Job jn = jobtab + newjob;

	    if (newjob == list_pipe_job && list_pipe_child)
		_exit(0);

	    lastwj = thisjob = newjob;

	    if (list_pipe)
		jn->stat |= STAT_NOPRINT;

	    if (nowait) {
		if(!pline_level) {
		    struct process *pn, *qn;

		    curjob = newjob;
		    addproc(list_pipe_pid, list_pipe_text);

		    for (pn = jobtab[jn->other].procs; pn; pn = pn->next)
			if (WIFSTOPPED(pn->status))
			    break;

		    if (pn) {
			for (qn = jn->procs; qn->next; qn = qn->next);
			qn->status = pn->status;
		    }

		    jn->stat &= ~(STAT_DONE | STAT_NOPRINT);
		    jn->stat |= STAT_STOPPED | STAT_CHANGED;
		    printjob(jn, !!isset(LONGLISTJOBS));
		}
		else
		    deletejob(jn);
	    }

	    for (; !nowait;) {
		if (list_pipe_child) {
		    jn->stat |= STAT_NOPRINT;
		    makerunning(jn);
		}
		if (!(jn->stat & STAT_LOCKED))
		    waitjobs();

		if (list_pipe_child &&
		    jn->stat & STAT_DONE &&
		    lastval2 & 0200)
		    killpg(mypgrp, lastval2 & ~0200);
		if ((list_pipe || last1) && !list_pipe_child &&
		    jn->stat & STAT_STOPPED) {
		    pid_t pid;
		    int synch[2];

		    pipe(synch);

		    if ((pid = fork()) == -1) {
			trashzle();
			close(synch[0]);
			close(synch[1]);
			putc('\n', stderr);
			fprintf(stderr, "zsh: job can't be suspended\n");
			fflush(stderr);
			makerunning(jn);
			killjb(jn, SIGCONT);
			thisjob = newjob;
		    }
		    else if (pid) {
			char dummy;

			list_pipe_pid = pid;
			nowait = errflag = 1;
			breaks = loops;
			close(synch[1]);
			read(synch[0], &dummy, 1);
			close(synch[0]);
			jobtab[list_pipe_job].other = newjob;
			jobtab[list_pipe_job].stat |= STAT_SUPERJOB;
			jn->stat |= STAT_SUBJOB | STAT_NOPRINT;
			jn->other = pid;
			killpg(jobtab[list_pipe_job].gleader, SIGSTOP);
			break;
		    }
		    else {
			close(synch[0]);
			entersubsh(Z_ASYNC, 0, 0);
			setpgrp(0L, mypgrp = jobtab[list_pipe_job].gleader);
			close(synch[1]);
			kill(getpid(), SIGSTOP);
			list_pipe = 0;
			list_pipe_child = 1;
			break;
		    }
		}
		else if (subsh && jn->stat & STAT_STOPPED)
		    thisjob = newjob;
		else
		    break;
	    }
	    child_unblock();

	    if (list_pipe && (lastval & 0200) && pj >= 0 &&
		(!(jn->stat & STAT_INUSE) || (jn->stat & STAT_DONE))) {
		jn = jobtab + pj;
		jn->stat |= STAT_NOPRINT;
		killjb(jobtab + pj, lastval & ~0200);
	    }
	    if (list_pipe_child || (list_pipe && (jn->stat & STAT_DONE)))
		deletejob(jn);
	    thisjob = pj;

	}
	if (l->flags & PFLAG_NOT)
	    lastval = !lastval;
    }
    if (!pline_level)
	simple_pline = 0;
    return lastval;
}

int subsh_close = -1;

/**/
void
execpline2(Pline pline, int how, int input, int output, int last1)
{
    pid_t pid;
    int pipes[2];
    int oldlineno = lineno;

    if (breaks || retflag)
	return;
    if (!pline)
	return;
    lineno = pline->left->lineno;
    if (pline_level == 1)
	strcpy(list_pipe_text, getjobtext((void *) pline->left));
    if (pline->type == END) {
	execcmd(pline->left, input, output, how, last1);
	pline->left = NULL;
    } else {
	int old_list_pipe = list_pipe;

	mpipe(pipes);

	/* if we are doing "foo | bar" where foo is a current *
	 * shell command, do foo in a subshell and do the     *
	 * rest of the pipeline in the current shell.         */
	if (pline->left->type >= CURSH && (how & Z_SYNC)) {
	    int synch[2];

	    pipe(synch);
	    if ((pid = fork()) == -1) {
		close(synch[0]);
		close(synch[1]);
		zerr("fork failed: %e", NULL, errno);
	    } else if (pid) {
		char dummy, *text;

		text = getjobtext((void *) pline->left);
		addproc(pid, text);
		close(synch[1]);
		read(synch[0], &dummy, 1);
		close(synch[0]);
	    } else {
		close(pipes[0]);
		close(synch[0]);
		entersubsh(how, 1, 0);
		close(synch[1]);
		execcmd(pline->left, input, pipes[1], how, 0);
		_exit(lastval);
	    }
	} else {
	/* otherwise just do the pipeline normally. */
	    subsh_close = pipes[0];
	    execcmd(pline->left, input, pipes[1], how, 0);
	}
	pline->left = NULL;
	close(pipes[1]);
	if (pline->right) {
	/* if another execpline() is invoked because the command is a list it
	   must know that we're already in a pipeline */
	    list_pipe = 1;
	    execpline2(pline->right, how, pipes[0], output, last1);
	    list_pipe = old_list_pipe;
	    close(pipes[0]);
	    subsh_close = -1;
	}
    }

    lineno = oldlineno;
}

/* make the argv array */

/**/
char **
makecline(LinkList list)
{
    int ct = 0;
    LinkNode node;
    char **argv, **ptr;

    if (isset(XTRACE)) {
	if (!doneps4)
	    fprintf(stderr, "%s", (prompt4) ? prompt4 : "");
	for (node = firstnode(list); node; incnode(node), ct++);
	ptr = argv = 2 + (char **) ncalloc((ct + 4) * sizeof(char *));

	for (node = firstnode(list); node; incnode(node))
	    if (*(char *)getdata(node)) {
		*ptr++ = (char *)getdata(node);
		untokenize(getdata(node));
		fputs(getdata(node), stderr);
		if (nextnode(node))
		    fputc(' ', stderr);
	    }
	*ptr = NULL;
	fputc('\n', stderr);
	fflush(stderr);
	return (argv);
    } else {
	for (node = firstnode(list); node; incnode(node), ct++);
	ptr = argv = 2 + (char **) ncalloc((ct + 4) * sizeof(char *));

	for (node = firstnode(list); node; incnode(node))
	    if (*(char *)getdata(node)) {
		*ptr++ = (char *)getdata(node);
		untokenize(getdata(node));
	    }
	*ptr = NULL;
	return (argv);
    }
}

/* untokenize the command line and remove null arguments */

/**/
void
fixcline(LinkList l)
{
    LinkNode node, next;

    for (node = firstnode(l); node; node = next) {
	next = nextnode(node);
	if (!*(char *)getdata(node))
	    uremnode(l, node);
	else
	    untokenize(getdata(node));
    }
}

/**/
void
untokenize(char *s)
{
    for (; *s; s++)
	if (itok(*s))
	    if (*s == Nularg)
		chuck(s--);
	    else
		*s = ztokens[*s - Pound];
}

/* nonzero if we shouldn't clobber a file */

/**/
int
dontclobber(struct redir *f)
{
    struct stat buf;

    if (unset(NOCLOBBER) || f->type & 1)
	return 0;
    if (stat(f->name, &buf) == -1)
	return 1;
    return S_ISREG(buf.st_mode);
}

/* size of buffer for tee and cat processes */
#define TCBUFSIZE 4092

/* close an multio (success) */

/**/
void
closemn(struct multio **mfds, int fd)
{
    if (mfds[fd]) {
	if (mfds[fd]->ct > 1) {
	    struct multio *mn = mfds[fd];
	    char buf[TCBUFSIZE];
	    int len, i;

	    if (zfork()) {
		for (i = 0; i < mn->ct; i++)
		    close(mn->fds[i]);
		close(mn->pipe);
		return;
	    }

	    /* pid == 0 */
	    closeallelse(mn);
	    if (mn->rflag) {
		/* tee process */
		while ((len = read(mn->pipe, buf, TCBUFSIZE)) > 0)
		    for (i = 0; i < mn->ct; i++)
			write(mn->fds[i], buf, len);
	    } else {
		/* cat process */
		for (i = 0; i < mn->ct; i++)
		    while ((len = read(mn->fds[i], buf, TCBUFSIZE)) > 0)
			write(mn->pipe, buf, len);
	    }
	    _exit(0);
	}
	mfds[fd] = NULL;
    }
}

/* close all the mnodes (failure) */

/**/
void
closemnodes(struct multio **mfds)
{
    int i, j;

    for (i = 0; i < 10; i++)
	if (mfds[i]) {
	    for (j = 0; j < mfds[i]->ct; j++)
		close(mfds[i]->fds[j]);
	    mfds[i] = NULL;
	}
}

/**/
void
closeallelse(struct multio *mn)
{
    int i, j;

    for (i = 0; i < OPEN_MAX; i++)
	if (mn->pipe != i) {
	    for (j = 0; j < mn->ct; j++)
		if (mn->fds[j] == i)
		    break;
	    if (j == mn->ct)
		close(i);
	}
}

/* A multio is a list of fds associated with a certain fd.       *
 * Thus if you do "foo >bar >ble", the multio for fd 1 will have *
 * two fds, the result of open("bar",...), and the result of     *
 * open("ble",....).                                             */

/* add a fd to an multio */

/**/
void
addfd(int forked, int *save, struct multio **mfds, int fd1, int fd2, int rflag)
{
    int pipes[2];

    /* starting a new multio */
    if (!mfds[fd1]) {
	mfds[fd1] = (struct multio *) alloc(sizeof(struct multio));

	if (!forked && fd1 != fd2 && fd1 < 10)
	    save[fd1] = movefd(fd1);
	redup(fd2, fd1);
	mfds[fd1]->ct = 1;
	mfds[fd1]->fds[0] = fd1;
	mfds[fd1]->rflag = rflag;
    } else {
	if (mfds[fd1]->rflag != rflag) {
	    zerr("file mode mismatch on fd %d", NULL, fd1);
	    return;
	}
	if (mfds[fd1]->ct == 1) {	/* split the stream */
	    mfds[fd1]->fds[0] = movefd(fd1);
	    mfds[fd1]->fds[1] = movefd(fd2);
	    mpipe(pipes);
	    mfds[fd1]->pipe = pipes[1 - rflag];
	    redup(pipes[rflag], fd1);
	    mfds[fd1]->ct = 2;
	} else			/* add another fd to an already split stream */
	    mfds[fd1]->fds[mfds[fd1]->ct++] = movefd(fd2);
    }
}

/**/
void
addvars(LinkList l, int export)
{
    Varasg v;
    LinkList vl;
    int xtr;
    char **arr, **ptr;

    xtr = isset(XTRACE);
    if (xtr && nonempty(l)) {
	fprintf(stderr, "%s", prompt4 ? prompt4 : "");
	doneps4 = 1;
    }

    while (nonempty(l)) {
	v = (Varasg) ugetnode(l);
	singsub(&v->name);
	if (errflag)
	    return;
	untokenize(v->name);
	if (xtr)
	    fprintf(stderr, "%s=", v->name);
	if (v->type == PM_SCALAR) {
	    vl = newlinklist();
	    addlinknode(vl, v->str);
	} else
	    vl = v->arr;
	prefork(vl, v->type ? 3 : 013);
	if (errflag)
	    return;
	postfork(vl, v->type ? 1 : 011);
	if (errflag)
	    return;
	if (v->type == PM_SCALAR && (empty(vl) || !nextnode(firstnode(vl)))) {
	    Param pm;
	    char *val;

	    if (empty(vl))
		val = ztrdup("");
	    else {
		untokenize(peekfirst(vl));
		val = ztrdup(ugetnode(vl));
	    }
	    if (xtr)
		fprintf(stderr, "%s ", val);
	    pm = setsparam(v->name, ztrdup(val));
	    if (errflag)
		return;
	    if (export && !(pm->flags & PM_EXPORTED))
		addenv(v->name, val);
	    zsfree(val);
	    continue;
	}
	ptr = arr = (char **) zalloc(sizeof(char **) * (countlinknodes(vl) + 1));

	while (nonempty(vl)) {
	    char *pp;

	    pp = (char *) ugetnode(vl);
	    if (*pp) {
		*ptr = ztrdup(pp);
		untokenize(*ptr++);
	    }
	}
	*ptr = NULL;
	if (xtr) {
	    fprintf(stderr, "( ");
	    for (ptr = arr; *ptr; ptr++)
		fprintf(stderr, "%s ", *ptr);
	    fprintf(stderr, ") ");
	}
	setaparam(v->name, arr);
	if (errflag)
	    return;
    }
}

/**/
void
execcmd(Cmd cmd, int input, int output, int how, int last1)
{
    HashNode hn = NULL;
    Redir fn;
    struct multio *mfds[10];
    char *text;
    int save[10];
    int fil, is_cursh, type, i;
    int nullexec = 0, assign = 0, forked = 0;
    int is_shfunc = 0, is_builtin = 0;

    doneps4 = 0;
    args = cmd->args;
    type = cmd->type;

    for (i = 0; i < 10; i++) {
	save[i] = -1;
	mfds[i] = NULL;
    }

    /* Empty command */
    if (type == SIMPLE && empty(args)) {
	if (nonempty(cmd->redir)) {
	    if (cmd->flags & CFLAG_EXEC) {
		/* Was this "exec < foobar"?  exec is a */
		/* reserved word, not a command.        */
		nullexec = 1;
	    } else if (!nullcmd || !*nullcmd) {
		zerr("redirection with no command", NULL, 0);
		errflag = lastval = 1;
		return;
	    } else if (readnullcmd && *readnullcmd &&
		       ((Redir) peekfirst(cmd->redir))->type == READ &&
		       !nextnode(firstnode(cmd->redir))) {
		addlinknode(args, dupstring(readnullcmd));
	    } else
		addlinknode(args, dupstring(nullcmd));
	} else {
	    addvars(cmd->vars, 0);
	    lastval = errflag;
	    if (isset(XTRACE)) {
		fputc('\n', stderr);
		fflush(stderr);
	    }
	    return;
	}
    }

    /* If the command begins with `%', then assume it is a *
     * reference to a job in the job table.                */
    if (nonempty(args) && *(char *)peekfirst(args) == '%') {
	pushnode(args, dupstring((how & Z_ASYNC) ? "bg" : "fg"));
	how = Z_SYNC;
    }

    /* If AUTORESUME is set, the command is SIMPLE, and doesn't have *
     * any redirections, then check if it matches as a prefix of a   *
     * job currently in the job table.  If it does, then we treat it *
     * as a command to resume this job.                              */
    if (isset(AUTORESUME) && type == SIMPLE && (how & Z_SYNC) && nonempty(args)
	&& empty(cmd->redir) && !input && !nextnode(firstnode(args))) {
	if (unset(NOTIFY))
	    scanjobs();
	if (findjobnam(peekfirst(args)) != -1)
	    pushnode(args, dupstring("fg"));
    }

    /* Get the text associated with this command. */
    if (jobbing || (how & Z_TIMED))
	text = getjobtext((void *) cmd);
    else
	text = NULL;

    /* Check if it's a builtin needing automatic MAGIC_EQUALS_SUBST      *
     * handling.  Things like typeset need this.  We can't detect the    *
     * command if it contains some tokens (e.g. x=ex; ${x}port), so this *
     * only works in simple cases.  has_token() is called to make sure   *
     * this really is a simple case.                                     */
    if(type == SIMPLE && nonempty(args)) {
	char *cmdarg = (char *) peekfirst(args);

	if(!has_token(cmdarg) && !(cmd->flags & CFLAG_COMMAND)
	    && !shfunctab->getnode(shfunctab, cmdarg)) {
	    HashNode hn2 = NULL;
	    LinkNode ln = args->first;

	    while((hn2 = builtintab->getnode(builtintab, (char *) ln->dat)) &&
	        ((Builtin) hn2)->funcid == BIN_BUILTIN &&
		(ln = ln->next) && !has_token((char *) ln->dat));
	    if(hn2)
		assign = (hn2->flags & BINF_MAGICEQUALS);
	}
    }

    /* Do prefork substitutions */
    prefork(args, (((type == CCASE) ? 010 : 0) | (assign ? 02 : isset(MAGICEQUALSUBST))));

    /* Set up special parameter $_ */
    zsfree(underscore);
    if (nonempty(args) && (underscore = ztrdup((char *) getdata(lastnode(args)))))
	untokenize(underscore); 
    else
  	underscore = ztrdup("");

    /* Warn about "rm *" */
    if (type == SIMPLE && interact && unset(RMSTARSILENT) && isset(SHINSTDIN) &&
	nonempty(args) && nextnode(firstnode(args)) && !strcmp(peekfirst(args), "rm") &&
	!(cmd->flags & CFLAG_NOGLOB)) {
	LinkNode node, next;

	for (node = nextnode(firstnode(args)); node && !errflag; node = next) {
	    char *s = (char *) getdata(node);
	    int l = strlen(s);

	    next = nextnode(node);
	    if (s[0] == Star && !s[1]) {
		if (!checkrmall(pwd))
		    uremnode(args, node);
	    } else if (l > 2 && s[l - 2] == '/' && s[l - 1] == Star) {
		char t = s[l - 2];

		s[l - 2] = 0;
		if (!checkrmall(s))
		    uremnode(args, node);
		s[l - 2] = t;
	    }
	}
	if (!nextnode(firstnode(args)))
	    errflag = 1;
    }

    if (errflag) {
	lastval = 1;
	return;
    }

    /* Resolve simple commands */
    if (type == SIMPLE && nonempty(args)) {
	char *cmdarg, *s;

	/* Get argument in command position */
	cmdarg = (char *) peekfirst(args);

	/* If the command is `[' then we must untokenize it */
	if (cmdarg[0] == Inbrack && cmdarg[1] == '\0')
	    cmdarg[0] = '[';

	/* Resolve shell functions and builtins */
	if (!(cmd->flags & CFLAG_COMMAND)) {
	    if ((hn = shfunctab->getnode(shfunctab, cmdarg))) {
		is_shfunc = 1;
	    } else if ((hn = builtintab->getnode(builtintab, cmdarg))) {
		is_builtin = 1;
	    }
	}

	/* Resolve external commands */
	if (!hn) {
	    hn = cmdnamtab->getnode(cmdnamtab, cmdarg);
	    if (!hn && isset(HASHCMDS) && strcmp(cmdarg, "..")) {
		for (s = cmdarg; *s && *s != '/'; s++);
		if (!*s)
		    hn = (HashNode) hashcmd(cmdarg, pathchecked);
	    }
	}

	/* If no command found yet, see if it is *
	 * a directory we should AUTOCD to.      */
	if (!hn && isset(AUTOCD) && isset(SHINSTDIN) && empty(cmd->redir)
	    && !nextnode(firstnode(args)) && (s = cancd(peekfirst(args)))) {
	    peekfirst(args) = (void *) s;
	    pushnode(args, dupstring("cd"));
	    if ((hn = builtintab->getnode(builtintab, "cd")))
		is_builtin = 1;
	}
    }

    /* This is nonzero if the command is a current shell procedure? */
    is_cursh = (is_builtin || is_shfunc || (type >= CURSH) || nullexec);

    /**************************************************************************
     * Do we need to fork?  We need to fork if:                               *
     * 1) The command is supposed to run in the background. (or)              *
     * 2) There is no `exec' flag, and either:                                *
     *    a) This is a builtin or shell function with output piped somewhere. *
     *    b) This is an external command and we can't do a `fake exec'.       *
     *                                                                        *
     * A `fake exec' is possible if we have all the following conditions:     *
     * 1) last1 flag is set.  This indicates that the current shell will not  *
     *    be needed after the current command.  This is typically the case    *
     *    when when the command is the last stage in a subshell, or is the    *
     *    last command after the option `-c'.                                 *
     * 2) We are not trapping EXIT or ZERR.                                   *
     * 3) We don't have any files to delete.                                  *
     *                                                                        *
     * The condition above for a `fake exec' will also work for a current     *
     * shell command such as a builtin, but doesn't really buy us anything    *
     * (doesn't save us a process), since it is already running in the        *
     * current shell.                                                         *
     **************************************************************************/

    if ((how & Z_ASYNC) || (!(cmd->flags & CFLAG_EXEC) &&
       (((is_builtin || is_shfunc) && output) ||
       (!is_cursh && (!last1 || sigtrapped[SIGZERR] ||
        sigtrapped[SIGEXIT] || havefiles()))))) {

	pid_t pid;
	int synch[2];
	char dummy;

	child_block();
	pipe(synch);

	if ((pid = zfork()) == -1) {
	    close(synch[0]);
	    close(synch[1]);
	    return;
	} if (pid) {
	    close(synch[1]);
	    read(synch[0], &dummy, 1);
	    close(synch[0]);
	    if (how & Z_ASYNC) {
		lastpid = pid;
	    } else if (!jobtab[thisjob].stty_in_env && nonempty(cmd->vars)) {
		/* search for STTY=... */
		while (nonempty(cmd->vars))
		    if (!strcmp(((Varasg) ugetnode(cmd->vars))->name, "STTY")) {
			jobtab[thisjob].stty_in_env = 1;
			break;
		    }
	    }
	    addproc(pid, text);
	    return;
	}
	/* pid == 0 */
	close(synch[0]);
	entersubsh(how, 1, 0);
	close(synch[1]);
	forked = 1;
	if (sigtrapped[SIGINT] == 2)
	    holdintr();
	/* Check if we should run background jobs at a lower priority. */
	if ((how & Z_ASYNC) && isset(BGNICE))
	    nice(5);

    } else if (is_cursh) {
	/* This is a current shell procedure that didn't need to fork.    *
	 * This includes current shell procedures that are being exec'ed, *
	 * as well as null execs.                                         */
	jobtab[thisjob].stat |= STAT_CURSH;
    } else {
	/* This is an exec (real or fake) for an external command.    *
	 * Note that any form of exec means that the subshell is fake *
	 * (but we may be in a subshell already).                     */
	entersubsh(how, 1, 1);
    }

    /* Perform postfork substitutions */
    postfork(args, !(cmd->flags & CFLAG_NOGLOB));

    if (errflag) {
	lastval = 1;
	goto err;
    } else {
	char *s;

	while (nonempty(args) && (s = (char *) peekfirst(args)) && !*s)
	    ugetnode(args);
    }

    /* Add pipeline input/output to mnodes */
    if (input)
	addfd(forked, save, mfds, 0, input, 0);
    if (output)
	addfd(forked, save, mfds, 1, output, 1);

    /* Do process substitutions */
    spawnpipes(cmd->redir);

    /* Do io redirections */
    while (nonempty(cmd->redir)) {
	fn = (Redir) ugetnode(cmd->redir);
	if (fn->type == INPIPE) {
	    if (fn->fd2 == -1) {
		fixfds(save);
		execerr();
	    }
	    addfd(forked, save, mfds, fn->fd1, fn->fd2, 0);
	} else if (fn->type == OUTPIPE) {
	    if (fn->fd2 == -1) {
		fixfds(save);
		execerr();
	    }
	    addfd(forked, save, mfds, fn->fd1, fn->fd2, 1);
	} else {
	    if (!(fn->type == HERESTR || fn->type == CLOSE || fn->type ==
		  MERGE || fn->type == MERGEOUT))
		if (xpandredir(fn, cmd->redir))
		    continue;
	    if (errflag) {
		fixfds(save);
		execerr();
	    }
	    switch(fn->type) {
	    case HERESTR:
		fil = getherestr(fn);
		if (fil == -1) {
		    fixfds(save);
		    if (errno != EINTR)
			zerr("%e", NULL, errno);
		    execerr();
		}
		addfd(forked, save, mfds, fn->fd1, fil, 0);
		break;
	    case READ:
		fil = open(fn->name, O_RDONLY);
		if (fil == -1) {
		    fixfds(save);
		    if (errno != EINTR)
			zerr("%e: %s", fn->name, errno);
		    execerr();
		}
		addfd(forked, save, mfds, fn->fd1, fil, 0);
		/* If this is 'exec < file', read from stdin, *
		 * not terminal, unless `file' is a terminal. */
		if (nullexec && fn->fd1 == 0 && isset(SHINSTDIN) && interact)
		    init_io();
		break;
	    case CLOSE:
		if (!forked && fn->fd1 < 10)
		    save[fn->fd1] = movefd(fn->fd1);
		closemn(mfds, fn->fd1);
		close(fn->fd1);
		break;
	    case MERGE:
	    case MERGEOUT:
		if (fn->fd2 == FD_COPROC)
		    fn->fd2 = (fn->type == MERGEOUT) ? coprocout : coprocin;
		closemn(mfds, fn->fd1);
		fil = dup(fn->fd2);
		if (fil == -1) {
		    char fdstr[4];

		    fixfds(save);
		    sprintf(fdstr, "%d", fn->fd2);
		    zerr("%s: %e", fdstr, errno);
		    execerr();
		}
		addfd(forked, save, mfds, fn->fd1, fil, fn->type == MERGEOUT);
		break;
	    default:
		if (fn->type >= APP)
		    fil = open(fn->name,
			       (isset(NOCLOBBER) && !(fn->type & 1)) ?
			       O_WRONLY | O_APPEND : O_WRONLY | O_APPEND | O_CREAT, 0666);
		else
		    fil = open(fn->name, dontclobber(fn) ?
			       O_WRONLY | O_CREAT | O_EXCL : O_WRONLY | O_CREAT | O_TRUNC, 0666);
		if (fil == -1) {
		    fixfds(save);
		    if (errno != EINTR)
			zerr("%e: %s", fn->name, errno);
		    execerr();
		}
		addfd(forked, save, mfds, fn->fd1, fil, 1);
		break;
	    }
	}
    }

    /* We are done with redirection.  close the mnodes, *
     * spawning tee/cat processes as necessary.         */
    for (i = 0; i < 10; i++)
	closemn(mfds, i);

    if (nullexec) {
	for (i = 0; i < 10; i++)
	    if (save[i] != -1)
		close(save[i]);
	return;
    }

    if (unset(NOEXEC)) {
	if (type >= CURSH) {
	    static int (*func[]) _((Cmd)) =
	    {
		execcursh, exectime, execfuncdef, execfor, execwhile,
		execrepeat, execif, execcase, execselect, execcond};

	    fixcline(args);
	    lastval = (func[type - CURSH]) (cmd);
	} else if (is_builtin || is_shfunc) {
	    /* builtin or shell function */

	    if (!forked && !assign) {
		permalloc();
		save_params(cmd);
		heapalloc();
	    }
	    
	    if (cmd->vars) {
		addvars(cmd->vars, 0);
		if (errflag) {
		    lastval = 1;
		    return;
		}
	    }
	    fixcline(args);

	    if (is_shfunc) {
		/* It's a shell function */
		if (subsh_close >= 0)
		    close(subsh_close);
		subsh_close = -1;
		execshfunc(cmd, (Shfunc) hn);
	    } else {
		/* It's a builtin */
		if (forked)
		    closem();
		lastval = execbuiltin(args, (Builtin) hn);
		if (isset(PRINTEXITVALUE) && isset(SHINSTDIN) && lastval && !subsh) {
		    fprintf(stderr, "zsh: exit %ld\n", (long)lastval);
		}
		fflush(stdout);
		if (save[1] == -1) {
		    if (ferror(stdout)) {
			zerr("write error: %e", NULL, errno);
			clearerr(stdout);
			errflag = 0;
		    }
		} else
		    clearerr(stdout);
	    }

	    if (cmd->flags & CFLAG_EXEC) {
		if (subsh)
		    _exit(lastval);

		/* If we are exec'ing a command, and we are not in a subshell, *
		 * then check if we should save the history file.              */
		if (unset(NORCS) && interact)
		    savehistfile(getsparam("HISTFILE"), 1, isset(APPENDHISTORY) ? 3 : 0);
		exit(lastval);
	    }

	    if (!forked && !assign)
		restore_params();

	} else {
	    if (cmd->flags & CFLAG_EXEC) {
		setiparam("SHLVL", --shlvl);
		/* If we are exec'ing a command, and we are not *
		 * in a subshell, then save the history file.   */
		if (!subsh && unset(NORCS) && interact)
		    savehistfile(getsparam("HISTFILE"), 1, isset(APPENDHISTORY) ? 3 : 0);
	    }
	    if (cmd->vars) {
		addvars(cmd->vars, 1);
		if (errflag) {
		    lastval = 1;
		    return;
		}
	    }
	    if (type == SIMPLE) {
		closem();
		execute((Cmdnam) hn, cmd->flags & CFLAG_DASH);
	    } else {		/* ( ... ) */
		list_pipe = 0;
		if (subsh_close >= 0)
		    close(subsh_close);
                subsh_close = -1;
		/* If we're forked (and we should be), no need to return */
		execlist(cmd->u.list, 0, last1 || forked);
	    }
	}
    }

  err:
    if (forked)
	_exit(lastval);
    fixfds(save);
}

/* Link list used to save parameters during *
 * execution of shfunc or builtin.          */
static LinkList restorelist;
static LinkList removelist;

/* Setup the link lists use to save parameters *
 * for shfuncs or builtins.                    */

/**/
void
init_save_params(void)
{
    restorelist = newlinklist();
    removelist  = newlinklist();
}

/* Arrange to have variables restored.                *
 * As yet special parameters won't work, nor will     *
 * parameter names that need substituting.            *
 * Note that list and nodes are in permament storage. */

/**/
void
save_params(Cmd cmd)
{
    Param pm;
    LinkNode node;
    char *s;

    for (node = firstnode(cmd->vars); node; incnode(node)) {
	s = ((Varasg) getdata(node))->name;
	if ((pm = (Param) paramtab->getnode(paramtab, s))) {
	    if (!(pm->flags & PM_SPECIAL)) {
		paramtab->removenode(paramtab, s);
		addlinknode(removelist, s);
		addlinknode(restorelist, pm);
	    }
	} else {
	    addlinknode(removelist, s);
	}
    }
}

/* Restore saved parameters after executing a shfunc or builtin */

/**/
void
restore_params(void)
{
    Param pm;
    char *s;

    /* remove temporary parameters */
    while ((s = (char *) getlinknode(removelist)))
	unsetparam(s);

    /* restore saved parameters */
    while ((pm = (Param) getlinknode(restorelist)))
	paramtab->addnode(paramtab, pm->nam, pm);
}

/* restore fds after redirecting a builtin */

/**/
void
fixfds(int *save)
{
    int old_errno = errno;
    int i;

    for (i = 0; i != 10; i++)
	if (save[i] != -1)
	    redup(save[i], i);
    errno = old_errno;
}

/**/
void
entersubsh(int how, int cl, int fake)
{
    if (sigtrapped[SIGEXIT])
	unsettrap(SIGEXIT);
    if (unset(MONITOR)) {
	if (how & Z_ASYNC) {
	    sigtrapped[SIGINT] = 2;
	    signal_ignore(SIGINT);
	    sigtrapped[SIGQUIT] = 2;
	    signal_ignore(SIGQUIT);
	    if (isatty(0)) {
		close(0);
		if (open("/dev/null", O_RDWR)) {
		    zerr("can't open /dev/null: %e", NULL, errno);
		    _exit(1);
		}
	    }
	}
    } else if (thisjob != -1 && cl) {
	if (jobtab[list_pipe_job].gleader && (list_pipe || list_pipe_child)) {
	    if (kill(jobtab[list_pipe_job].gleader, 0) == -1 ||
		setpgrp(0L, jobtab[list_pipe_job].gleader) == -1) {
		jobtab[list_pipe_job].gleader =
		    jobtab[thisjob].gleader = mypgrp;
		setpgrp(0L, mypgrp);

		if (how & Z_SYNC)
		    attachtty(jobtab[thisjob].gleader);
	    }
	}
	else if (!jobtab[thisjob].gleader) {
	    jobtab[thisjob].gleader = getpid();
	    if (list_pipe_job != thisjob &&
		!jobtab[list_pipe_job].gleader)
		jobtab[list_pipe_job].gleader = jobtab[thisjob].gleader;
	    setpgrp(0L, jobtab[thisjob].gleader);
	    if (how & Z_SYNC)
		attachtty(jobtab[thisjob].gleader);
	} else
	    setpgrp(0L, jobtab[thisjob].gleader);
    }
    if (!fake)
	subsh = 1;
    if (SHTTY != -1) {
	close(SHTTY);
	SHTTY = -1;
    }
    if (isset(MONITOR)) {
	signal_default(SIGTTOU);
	signal_default(SIGTTIN);
	signal_default(SIGTSTP);
	signal_default(SIGPIPE);
    }
    if (interact) {
	signal_default(SIGTERM);
	if (sigtrapped[SIGINT] != 2)
	    signal_default(SIGINT);
    }
    if (sigtrapped[SIGQUIT] != 2)
	signal_default(SIGQUIT);
    opts[MONITOR] = OPT_UNSET;
    opts[USEZLE] = OPT_UNSET;
    if (cl)
	clearjobtab();
    times(&shtms);
}

/* close all internal shell fds */

/**/
void
closem(void)
{
    int i;

    for (i = 10; i < OPEN_MAX; i++)
	close(i);
}

/* convert here document into a here string */

/**/
char *
gethere(char *str, int typ)
{
    char pbuf[256];
    int qt = 0, siz = 0, l, strip = 0;
    char *s, *t, *bptr;

    for (s = str; *s; s++)
	if (INULL(*s)) {
	    *s = Nularg;
	    qt = 1;
	}
    untokenize(str);
    if (typ == HEREDOCDASH) {
	strip = 1;
	while (*str == '\t')
	    str++;
    }
    t = ztrdup("");
    for (;;) {
	char *u, *v;

	if (!ingets(pbuf, sizeof(pbuf)))
	    break;
	bptr = pbuf;
	if (strip)
	    while (*bptr == '\t')
		bptr++;
	for (u = bptr, v = str; *u != '\n' && *v; u++, v++)
	    if (*u != *v)
		break;
	if (!(*u == '\n' && !*v)) {
	    l = strlen(bptr);
	    if (!qt && l > 1 && bptr[l - 1] == '\n' && bptr[l - 2] == '\\')
		bptr[l -= 2] = '\0';
	    t = realloc(t, siz + l + 1);
	    strncpy(t + siz, bptr, l);
	    siz += l;
	} else
	    break;
    }
    t[siz] = '\0';
    if (siz && t[siz - 1] == '\n')
	t[siz - 1] = '\0';
    if (!qt)
	for (s = t; *s; s++)
	    if (*s == '$') {
		*s = Qstring;
	    } else if (*s == '`') {
		*s = Qtick;
	    } else if (*s == '(') {
		*s = Inpar;
	    } else if (*s == ')') {
		*s = Outpar;
	    } else if (*s == '\\' &&
		       (s[1] == '$' || s[1] == '`'))
		chuck(s);
    s = dupstring(t);
    zsfree(t);
    return s;
}

/* open here string fd */

/**/
int
getherestr(struct redir *fn)
{
    LinkList fake;
    char *s, *t;
    int fd;

    fake = newlinklist();
    addlinknode(fake, fn->name);
    prefork(fake, 010);
    if (!errflag)
	postfork(fake, 011);
    if (errflag)
	return -1;
    s = gettempname();
    if ((fd = open(s, O_CREAT | O_WRONLY | O_EXCL, 0600)) == -1)
	return -1;
    while ((t = (char *)ugetnode(fake))) {
	untokenize(t);
	write(fd, t, strlen(t));
	if (nonempty(fake))
	    write(fd, " ", 1);
    }
    write(fd, "\n", 1);
    close(fd);
    fd = open(s, O_RDONLY);
    unlink(s);
    return fd;
}

/* $(...) */

/**/
LinkList
getoutput(char *cmd, int qt)
{
    List list;
    int pipes[2];
    pid_t pid;

    if (*cmd == '<') {
	int stream;
	char *fi, *s, x;

	for (cmd++; *cmd == ' '; cmd++);
	for (s = cmd; *s && *s != ' '; s++)
	    if (*s == '\\')
		s++;
	    else if (*s == '$')
		*s = String;
	x = *s;
	*s = '\0';
	fi = dupstring(cmd);
	*s = x;
	if (*fi == '~')
	    *fi = Tilde;
	else if (*fi == '=')
	    *fi = Equals;
	singsub(&fi);
	if (errflag)
	    return NULL;
	stream = open(fi, O_RDONLY);
	if (stream == -1) {
	    zerr("%e: %s", fi, errno);
	    return NULL;
	}
	return readoutput(stream, qt);
    }
    if (!(list = parse_string(cmd)))
	return NULL;
    mpipe(pipes);
    child_block();
    if ((cmdoutpid = pid = zfork()) == -1) {
	/* fork error */
	popheap();
	close(pipes[0]);
	close(pipes[1]);
	errflag = 1;
	cmdoutpid = 0;
	child_unblock();
	return NULL;
    } else if (pid) {
	LinkList retval;

	popheap();
	close(pipes[1]);
	retval = readoutput(pipes[0], qt);
	child_suspend(0);		/* unblocks */
	lastval = cmdoutval;
	cmdoutval = 0;
	return retval;
    }

    /* pid == 0 */
    child_unblock();
    close(pipes[0]);
    redup(pipes[1], 1);
    entersubsh(Z_SYNC, 1, 0);
    signal_ignore(SIGTSTP);
    execlist(list, 0, 1);
    close(1);
    _exit(lastval);
    zerr("exit returned in child!!", NULL, 0);
    kill(getpid(), SIGKILL);
    return NULL;
}

/* read output of command substitution */

/**/
LinkList
readoutput(int in, int qt)
{
    LinkList ret;
    char *buf, *ptr;
    int bsiz, c, cnt = 0;
    FILE *fin;

    fin = fdopen(in, "r");
    ret = newlinklist();
    ptr = buf = (char *) ncalloc(bsiz = 64);
    if (qt) {
	*ptr++ = Nularg;
	cnt++;
    }
    while ((c = fgetc(fin)) != EOF)
	if (!qt && isep(c)) {
	    if (cnt) {
		*ptr = '\0';
		if (isset(GLOBSUBST))
		    tokenize(buf);
		addlinknode(ret, buf);
		ptr = buf = (char *) ncalloc(bsiz = 64);
		cnt = 0;
	    }
	} else {
	    *ptr++ = c;
	    if (++cnt == bsiz) {
		char *pp = (char *) ncalloc(bsiz *= 2);

		memcpy(pp, buf, cnt);
		ptr = (buf = pp) + cnt;
	    }
	}
    if (ptr != buf && ptr[-1] == '\n')
	ptr[-1] = '\0';
    else
	*ptr = '\0';
    if (!qt && isset(GLOBSUBST))
	tokenize(buf);
    if (cnt)
	addlinknode(ret, buf);
    fclose(fin);
    return ret;
}

/* =(...) */

/**/
char *
getoutputfile(char *cmd)
{
    pid_t pid;
    char *nam, *str;
    List list;

    if (thisjob == -1)
	return NULL;
    for (str = cmd; *str && *str != Outpar; str++);
    if (!*str)
	zerr("oops.", NULL, 0);
    *str = '\0';
    if (!(list = parse_string(cmd)))
	return NULL;

    nam = gettempname();
    permalloc();
    if (!jobtab[thisjob].filelist)
	jobtab[thisjob].filelist = newlinklist();
    addlinknode(jobtab[thisjob].filelist, ztrdup(nam));
    heapalloc();
    child_block();

    if ((cmdoutpid = pid = zfork()) == -1) {
	/* fork error */
	popheap();
	child_unblock();
	return nam;
    } else if (pid) {
	int os;

	popheap();
	os = jobtab[thisjob].stat;
	waitforpid(pid);
	cmdoutval = 0;
	jobtab[thisjob].stat = os;
	return nam;
    }

    /* pid == 0 */
    close(1);
    entersubsh(Z_SYNC, 1, 0);
    signal_ignore(SIGTSTP);
    (void) open(nam, O_WRONLY | O_CREAT | O_EXCL, 0600); /* create the file */
    execlist(list, 0, 1);
    close(1);
    _exit(lastval);
    zerr("exit returned in child!!", NULL, 0);
    kill(getpid(), SIGKILL);
    return NULL;
}

/* get a temporary named pipe */

/**/
char *
namedpipe(void)
{
#ifdef HAVE_FIFOS
    char *tnam = gettempname();

# ifdef HAVE_MKFIFO
    if (mkfifo(tnam, 0600) < 0)
# else
    if (mknod(tnam, 0010600, 0) < 0)
# endif
	return NULL;
    return tnam;
#else
         return NULL;
#endif  /* HAVE_FIFOS */
}

/* <(...) */

/**/
char *
getoutproc(char *cmd)
{
#ifndef HAVE_FIFOS
    zerr("doesn't look like your system supports FIFOs.", NULL, 0);
    return NULL;
#else
    List list;
    int fd;
    char *pnam, *str;

    if (thisjob == -1)
	return NULL;
    for (str = cmd; *str && *str != Outpar; str++);
    if (!*str)
	zerr("oops.", NULL, 0);
    *str = '\0';
    pnam = namedpipe();
    if (!pnam)
	return NULL;
    permalloc();
    if (!jobtab[thisjob].filelist)
	jobtab[thisjob].filelist = newlinklist();
    addlinknode(jobtab[thisjob].filelist, ztrdup(pnam));
    heapalloc();
    if (!(list = parse_string(cmd)))
	return NULL;
    if (zfork()) {
	popheap();
	return pnam;
    }
    entersubsh(Z_ASYNC, 1, 0);
    closem();
    fd = open(pnam, O_WRONLY);
    if (fd == -1) {
	zerr("can't open %s: %e", pnam, errno);
	_exit(1);
    }
    redup(fd, 1);
    fd = open("/dev/null", O_RDONLY);
    redup(fd, 0);
    execlist(list, 0, 1);
    close(1);
    _exit(lastval);
    return NULL;
#endif   /* HAVE_FIFOS not defined */
}

/* >(...) */

/**/
char *
getinproc(char *cmd)
{
#ifndef HAVE_FIFOS
    zerr("doesn't look like your system supports FIFOs.", NULL, 0);
    return NULL;
#else
    List list;
    pid_t pid;
    int fd;
    char *pnam, *str;

    if (thisjob == -1)
	return NULL;
    for (str = cmd; *str && *str != Outpar; str++);
    if (!*str)
	zerr("oops.", NULL, 0);
    *str = '\0';
    pnam = namedpipe();
    if (!pnam)
	return NULL;
    permalloc();
    if (!jobtab[thisjob].filelist)
	jobtab[thisjob].filelist = newlinklist();
    addlinknode(jobtab[thisjob].filelist, ztrdup(pnam));
    heapalloc();
    if (!(list = parse_string(cmd)))
	return NULL;
    if ((pid = zfork())) {
	popheap();
	return pnam;
    }
    entersubsh(Z_ASYNC, 1, 0);
    closem();
    fd = open(pnam, O_RDONLY);
    redup(fd, 0);
    execlist(list, 0, 1);
    _exit(lastval);
    return NULL;
#endif
}

/* > >(...) (does not use named pipes) */

/**/
int
getinpipe(char *cmd)
{
    List list;
    int pipes[2];
    char *str;

    for (str = cmd; *str && *str != Outpar; str++);
    if (!*str)
	zerr("oops.", NULL, 0);
    *str = '\0';
    if (!(list = parse_string(cmd + 2)))
	return -1;
    mpipe(pipes);
    if (zfork()) {
	popheap();
	close(pipes[1]);
	return pipes[0];
    }
    close(pipes[0]);
    closem();
    entersubsh(Z_ASYNC, 1, 0);
    redup(pipes[1], 1);
    execlist(list, 0, 1);
    _exit(lastval);
    return 0;
}

/* < <(...) */

/**/
int
getoutpipe(char *cmd)
{
    List list;
    int pipes[2];
    char *str;

    for (str = cmd; *str && *str != Outpar; str++);
    if (!*str)
	zerr("oops.", NULL, 0);
    *str = '\0';
    if (!(list = parse_string(cmd + 2)))
	return -1;
    strinend();
    mpipe(pipes);
    if (zfork()) {
	popheap();
	close(pipes[0]);
	return pipes[1];
    }
    close(pipes[1]);
    entersubsh(Z_ASYNC, 1, 0);
    redup(pipes[0], 0);
    closem();
    execlist(list, 0, 1);
    _exit(lastval);
    return 0;
}

/* open pipes with fds >= 10 */

/**/
void
mpipe(int *pp)
{
    pipe(pp);
    pp[0] = movefd(pp[0]);
    pp[1] = movefd(pp[1]);
}

/* Do process substitution with redirection */

/**/
void
spawnpipes(LinkList l)
{
    LinkNode n;
    Redir f;
    char *str;

    n = firstnode(l);
    for (; n; incnode(n)) {
	f = (Redir) getdata(n);
	if (f->type == OUTPIPE) {
	    str = f->name;
	    f->fd2 = getoutpipe(str);
	}
	if (f->type == INPIPE) {
	    str = f->name;
	    f->fd2 = getinpipe(str);
	}
    }
}

/* evaluate a [[ ... ]] */

/**/
int
execcond(Cmd cmd)
{
    return !evalcond(cmd->u.cond);
}

/* perform time ... command */

/**/
int
exectime(Cmd cmd)
{
    int jb;

    jb = thisjob;
    if (!cmd->u.pline) {
	shelltime();
	return 0;
    }
    execpline(cmd->u.pline, Z_TIMED|Z_SYNC, 0);
    thisjob = jb;
    return lastval;
}

/* Define a shell function */

/**/
int
execfuncdef(Cmd cmd)
{
    Shfunc shf;
    char *s;
    int signum;

    permalloc();
    while ((s = (char *) ugetnode(cmd->args))) {
	shf = (Shfunc) zalloc(sizeof *shf);
	if (!cmd->u.list)
	    shf->funcdef = NULL;
	else
	    shf->funcdef = (List) dupstruct(cmd->u.list);
	shf->flags = 0;
	shfunctab->addnode(shfunctab, ztrdup(s), shf);

	/* is this shell function a signal trap? */
	if (!strncmp(s, "TRAP", 4)) {
	    signum = getsignum(s + 4);
	    if (signum != -1) {
		settrap(signum, cmd->u.list);
		permalloc();
	    }
	}
    }
    heapalloc();
    return 0;
}

/* Main entry point to execute a shell function.  It will retrieve *
 * an autoloaded shell function if it is currently undefined.      */

/**/
void
execshfunc(Cmd cmd, Shfunc shf)
{
    List funcdef;
    char *nam;

    if (errflag)
	return;
    deletepipejob();

    /* Are we dealing with an autoloaded shell function? */
    if (shf->flags & PM_UNDEFINED) {
	nam = (char *) peekfirst(cmd->args);
	if (!(funcdef = getfpfunc(nam))) {
	    zerr("function not found: %s", nam, 0);
	    lastval = 1;
	    return;
	}
	permalloc();
	shf->flags &= ~PM_UNDEFINED;
	funcdef = shf->funcdef = (List) dupstruct(funcdef);
	heapalloc();
	popheap();

	/* Execute the function definition, we just retrived */
	doshfunc(shf->funcdef, cmd->args, shf->flags, 0);

	/* See if this file defined the autoloaded function *
	 * by name.  If so, we execute it again.            */
	if ((shf = (Shfunc) shfunctab->getnode(shfunctab, nam))
	    && shf->funcdef && shf->funcdef != funcdef)
	    doshfunc(shf->funcdef, cmd->args, shf->flags, 0);
	return;
    }

    /* Normal shell function execution */
    doshfunc(shf->funcdef, cmd->args, shf->flags, 0);
}

/* execute a shell function */

/**/
void
doshfunc(List list, LinkList doshargs, int flags, int noreturnval)
/* If noreturnval is nonzero, then reset the current return *
 * value (lastval) to its value before the shell function   *
 * was executed.                                            */
{
    Param pm;
    char **tab, **x, *oargv0;
    int xexittr, oldzoptind, oldlastval;
    LinkList olist;
    char *s, *ou;
    List xexitfn;
    char saveopts[OPT_SIZE];

    pushheap();
    oldlastval = lastval;
    xexittr = sigtrapped[SIGEXIT];
    xexitfn = sigfuncs[SIGEXIT];
    sigtrapped[SIGEXIT] = 0;
    sigfuncs[SIGEXIT] = NULL;
    tab = pparams;
    oargv0 = argzero;
    oldzoptind = zoptind;
    zoptind = 1;

    /* We need to save the current options even if LOCALOPTIONS is *
     * not currently set.  That's because if it gets set in the    *
     * function we need to restore the original options on exit.   */
    memcpy(saveopts, opts, sizeof(opts));

    if (flags & PM_TAGGED)
	opts[XTRACE] = OPT_SET;
    opts[PRINTEXITVALUE] = OPT_UNSET;
    if (doshargs) {
	LinkNode node;

	node = doshargs->first;
	pparams = x = (char **) zcalloc(((sizeof *x) * (1 + countlinknodes(doshargs))));
	argzero = ztrdup((char *) node->dat);
	node = node->next;
	for (; node; node = node->next, x++)
	    *x = ztrdup((char *) node->dat);
    } else {
	pparams = (char **) zcalloc(sizeof *pparams);
	argzero = ztrdup(argzero);
    }
    permalloc();
    olist = locallist;		/* save the old locallist since shell functions may be nested */
    locallist = newlinklist();	/* make a new list of local variables that we have to destroy */
    heapalloc();
    locallevel++;
    ou = ztrdup(underscore);
    execlist(dupstruct(list), 1, 0);
    zsfree(underscore);
    underscore = ou;
    locallevel--;

    /* destroy the local variables we have created in this shell function */
    while ((s = (char *) getlinknode(locallist))) {
	if((pm = (Param) paramtab->getnode(paramtab, s)) && (pm->level > locallevel))
	    unsetparam(s);
	zsfree(s);
    }
    zfree(locallist, sizeof(struct linklist));

    locallist = olist;		/* restore the old list of local variables */
    breaks = retflag = 0;
    freearray(pparams);
    zsfree(argzero);
    zoptind = oldzoptind;
    argzero = oargv0;
    pparams = tab;

    if (isset(LOCALOPTIONS)) {
	/* restore all shell options except PRIVILEGED */
	saveopts[PRIVILEGED] = opts[PRIVILEGED];
	memcpy(opts, saveopts, sizeof(opts));
    } else {
	/* just restore a couple. */
	opts[XTRACE] = saveopts[XTRACE];
	opts[PRINTEXITVALUE] = saveopts[PRINTEXITVALUE];
	opts[LOCALOPTIONS] = saveopts[LOCALOPTIONS];
    }

    /* Did the shell function define a trap on SIGEXIT? */
    if (sigfuncs[SIGEXIT]) {
	dotrap(SIGEXIT);
	freestruct(sigfuncs[SIGEXIT]);
    }

    sigtrapped[SIGEXIT] = xexittr;
    sigfuncs[SIGEXIT] = xexitfn;
    if (noreturnval)
	lastval = oldlastval;
    popheap();
}

/* Search fpath for an undefined function. */

/**/
List
getfpfunc(char *s)
{
    char **pp, buf[PATH_MAX];
    off_t len;
    char *d;
    List r;
    int fd;

    pp = fpath;
    for (; *pp; pp++) {
	sprintf(buf, "%s/%s", *pp, s);
	if (!access(buf, R_OK) && (fd = open(buf, O_RDONLY)) != -1) {
	    if ((len = lseek(fd, 0, 2)) != -1) {
		lseek(fd, 0, 0);
		d = (char *) zcalloc(len + 1);
		if (read(fd, d, len) == len) {
		    close(fd);
		    r = parse_string(d);
		    zfree(d, len + 1);
		    return r;
		} else {
		    zfree(d, len + 1);
		    close(fd);
		}
	    } else {
		close(fd);
	    }
	}
    }
    return NULL;
}

/* check to see if AUTOCD applies here */

extern int doprintdir;

/**/
char *
cancd(char *s)
{
    int nocdpath = s[0] == '.' &&
    (s[1] == '/' || !s[1] || (s[1] == '.' && (s[2] == '/' || !s[1])));
    char *t;

    if (haswilds(s)) {
	LinkList l = newlinklist();
	int ne = noerrs;

	noerrs = 1;
	addlinknode(l, dupstring(s));
	postfork(l, 1);
	if (!errflag && nonempty(l) && !nextnode(firstnode(l)))
	    s = peekfirst(l);
	errflag = 0;
	noerrs = ne;
    }

    if (*s != '/') {
	char sbuf[PATH_MAX], **cp;

	if (cancd2(s))
	    return s;
	if (access(s, X_OK) == 0)
	    return NULL;
	if (!nocdpath)
	    for (cp = cdpath; *cp; cp++) {
		sprintf(sbuf, "%s/%s", *cp, s);
		if (cancd2(sbuf)) {
		    doprintdir = -1;
		    return dupstring(sbuf);
		}
	    }
	if ((t = cd_able_vars(s))) {
	    if (cancd2(t)) {
		doprintdir = -1;
		return t;
	    }
	}
	return NULL;
    }
    return cancd2(s) ? s : NULL;
}

/**/
int
cancd2(char *s)
{
    struct stat buf;

    return !(access(s, X_OK) || stat(s, &buf) || !S_ISDIR(buf.st_mode));
}

