;;; lazy-lock.el --- Lazy demand-driven fontification for fast font-lock mode.

;; Copyright (C) 1994 Simon Marshall.

;; Author: Simon Marshall <Simon.Marshall@mail.esrin.esa.it>
;; Keywords: faces files
;; Version: 1.03

;; LCD Archive Entry:
;; lazy-lock|Simon Marshall|Simon.Marshall@mail.esrin.esa.it|
;; Lazy Font Lock mode (for fast demand-driven fontification).|
;; 20-Sep-94|1.03|~/modes/lazy-lock.el.Z|

;; The archive is archive.cis.ohio-state.edu in /pub/gnu/emacs/elisp-archive.

;;; This file is not part of GNU Emacs.

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to
;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; Purpose:
;;
;; To make visiting buffers in `font-lock-mode' faster by making fontification
;; be demand-driven and stealthy.
;; Fontification only occurs when, and where, necessary.
;;
;; See caveats below.
;; See also the face-lock package.
;; See also the fast-lock package.  (But don't use the two at the same time!)

;; Installation:
;; 
;; Put this file somewhere where Emacs can find it (i.e., in one of the paths
;; in your `load-path'), `byte-compile-file' it, and put in your ~/.emacs:
;;
;; (autoload 'turn-on-lazy-lock "lazy-lock"
;;   "Unconditionally turn on Lazy Lock mode.")
;;
;; (add-hook 'font-lock-mode-hook 'turn-on-lazy-lock)
;;
;; Start up a new Emacs and use font-lock as usual (except that you can use the
;; so-called "gaudier" fontification regexps on big files without frustration).
;;
;; In a buffer (which has `font-lock-mode' enabled) which is at least
;; `lazy-lock-minimum-size' characters long, only the visible portion of the
;; buffer will be fontified.  Motion around the buffer will fontify those
;; visible portions that were not previous fontified.
;;
;; If stealth fontification is enabled (in Emacs 19.26 and up), fontification
;; will occur in invisible parts of the buffer after `lazy-lock-stealth-time'
;; seconds of Emacs idle time (i.e., there is no input during that time).

;; Advanced Use:
;;
;; You can also do fancy things with `advice'.  For example, to fontify while
;; pausing when dragging the scroll-bar, you could put in your ~/.emacs:
;;
;; (defadvice scroll-bar-drag-1 (after lazy-lock-fontify activate compile)
;;   "Fontify while scrolling with \\[scroll-bar-drag].
;; Fontifies if there is no further scrolling after `lazy-lock-delay-time'."
;;   (and (boundp 'lazy-lock-mode) lazy-lock-mode
;;        (let ((this-command (car lazy-lock-delay-commands)))
;;          (lazy-lock-post-command-hook))))
;;
;; To prevent large insertions from being fontified entirely, something like:
;;
;; (defadvice font-lock-after-change-function (around lazy-lock-fontify
;;                                             activate compile)
;;   "Fontify selectively when inserting."
;;   (if (and (pos-visible-in-window-p beg) (pos-visible-in-window-p end))
;;       ad-do-it
;;     (let ((this-command 'ignore))
;;       (put-text-property beg end 'fontified nil)
;;       (lazy-lock-fontify-window))))
;;
;; These kinds of things with `advice' aren't done automatically because they
;; cause large packages (advice.el plus bytecomp.el and friends) to be loaded.

;; Caveats:
;;
;; This is "The Other Idea" for speeding up Font Lock mode.  The idea is to
;; only fontify when, and where, absolutely necessary.  "When" means fontify
;; just as the region concerned becomes visible, and "where" means fontify just
;; the visible region where not previously fontified.
;;
;; It's an idea I didn't pursue (and so I developed fast-lock instead) as Emacs
;; 19.25 doesn't provide the functionality to properly support it, and rms
;; didn't feel that it was a "solid" solution to speeding up Font Lock mode in
;; the sense that you might expect to find text properties where none were yet
;; generated (e.g., in a region containing an as yet unfontified section).  But
;; then again I use Lazy Lock mode rather than Fast Lock mode!
;;
;; The package is really an exercise to see how effective the concept behaves;
;; it is implemented by putting the function `lazy-lock-post-command-hook' on
;; Emacs' `post-command-hook' so that fontification occurs after commands have
;; completed.  This is not the most efficient way of doing it, and not the
;; "correct" way (as of Emacs 19.25 the "correct" way is not possible).
;;
;; Redisplay occurs *before* refontification.  This is because Emacs 19.25 runs
;; `post-command-hook' before certain redisplays occur (such as those as a
;; result of commands such as `next-line' or `find-tag'), or maybe the bug is
;; that `window-start' is just as broke as `window-end'.  Or both.
;;
;; One Emacs 19.25 `window-start'/`window-end' bug means that if you open a
;; file in another frame (such as via `find-tag-other-frame'), the whole buffer
;; is fontified regardless.  Upgrade!
;;
;; Note that, for Emacs 19.25 and below, fontification by stealth is turned off
;; because of a fatal bug in `previous-single-property-change'.  Upgrade!
;;
;; Feedback is welcome.  Or just take the idea/code and run with it.

;; Feedback:
;;
;; Please send me bug reports, bug fixes, and extensions, so that I can
;; merge them into the master source.
;;     - Simon Marshall (Simon.Marshall@mail.esrin.esa.it)

;; History:
;;
;; 0.01--1.00:
;; - Changed name from fore-lock to lazy-lock.  Shame though.
;; - Dropped `advice'-wrapping completely.  Ask me if you're interested in it.
;; - Made `lazy-lock-mode' ignore `post-command-hook' and `buffer-file-name'.
;; - Made `lazy-lock-fontify-window' check `lazy-lock-mode' and `this-command'.
;; - Made `lazy-lock-fontify-window' redisplay via `sit-for'.
;; - Added `lazy-lock-minimum-size' to control `lazy-lock-mode'.
;; 1.00--1.01:
;; - Added `lazy-lock-fontify-buffer'.
;; - Made `lazy-lock-fontify-window' ignore `lazy-lock-mode'.
;; - Made `lazy-lock-fontify-window' suspicious of `window-' favourites again.
;; - Added `lazy-lock-delay-commands' (idea from William G. Dubuque).
;; - Added `lazy-lock-ignore-commands' for completeness.
;; - Added `lazy-lock-continuity-time' for normal input delay.
;; 1.01--1.02:
;; - Made `lazy-lock-fontify-window' cope with multiple unfontified regions.
;; - Made `lazy-lock-mode' remove `fontified' properties if turned off.
;; - Made `lazy-lock-fontify-window' fontify by lines.
;; - Added `lazy-lock-cache-position' buffer local to detect visibility change.
;; - Added `lazy-lock-post-command-hook' to do the waiting.
;; - Made `lazy-lock-fontify-window' just do the fontification.
;; - Made `lazy-lock-mode' append `lazy-lock-post-command-hook'.
;; - Added `lazy-lock-walk-windows' to hack multi-window motion.
;; - Made `lazy-lock-post-command-hook' `walk-windows' if variable is non-nil.
;; - Removed `lazy-lock-ignore-commands' since insertion may change window.
;; - Added `lazy-lock-fontify-stealthily' and `lazy-lock-stealth-time'.
;; - Made `lazy-lock-post-command-hook' use them.
;; 1.02--1.03:
;; - Made `lazy-lock-fontify-stealthily' do `forward-line' not `previous-line'.
;; - Made `lazy-lock-fontify-stealthily' `move-to-window-line' first.
;; - Made `lazy-lock-fontify-stealthily' use `text-property-any' for region.
;; - Made `lazy-lock-post-command-hook' loop on `lazy-lock-fontify-stealthily'.

(require 'font-lock)

;; Variables:

(defvar lazy-lock-minimum-size (* 10 1024)
  "*If non-nil, the minimum size for buffers.
Only buffers at least this size can have demand-driven fontification.
If nil, means size is irrelevant.")

(defvar lazy-lock-walk-windows t
  "If non-nil, fontify windows other than the selected window.
If `all-frames', fontify windows even on other frames.
A non-nil value is slows down redisplay.")

(defvar lazy-lock-continuity-time 0
  "*Time in seconds to delay after a command before fontification.
Window fontification occurs if there is no input within this time.")

;; `previous-single-property-change' at `point-min' up to Emacs 19.25 is fatal.
(defvar lazy-lock-stealth-time
  (if (and (boundp 'emacs-minor-version) (> emacs-minor-version 25)) 15)
  "*Time in seconds to delay before fontifying the buffer stealthily.
Stealth fontification occurs if there is no input within this time.
If nil, means no fontification by stealth.")

(defvar lazy-lock-delay-time 2
  "*Time in seconds to delay after one of `lazy-lock-delay-commands'.")

(defvar lazy-lock-delay-commands
  '(isearch-printing-char isearch-repeat-forward isearch-repeat-backward)
  "A list of commands after which fontification should delay.
Fontification occurs if there is no input after `lazy-lock-delay-time'.")

(defvar lazy-lock-mode nil)		; for modeline

;; This variable records, for each buffer, the window end position.  If this
;; has not changed, the displayed window must be the same as before.  Used to
;; make lazy-lock-fontify-window faster.
(defvar lazy-lock-cache-position -1)
(make-variable-buffer-local 'lazy-lock-cache-position)

;; Functions:

(defun lazy-lock-mode (&optional arg)
  "Toggle Lazy Lock mode.
With arg, turn Lazy Lock mode on if and only if arg is positive and the buffer
is at least `lazy-lock-minimum-size' characters long.

If Lazy Lock mode is enabled, fontification becomes demand-driven so that it
occurs only when, and where, absolutely necessary.  That is, only the visible
portion of the buffer is fontified, and only where not fontified before.
Fontification occurs following the completion of a command if the criteria
imposed by `lazy-lock-fontify-window' are met.

If Lazy Lock stealth is enabled, fontification occurs in invisible parts of the
buffer if no input arrives after `lazy-lock-stealth-time' seconds.

Use \\[lazy-lock-fontify-buffer] to fontify the whole buffer.

See variables `lazy-lock-minimum-size', `lazy-lock-walk-windows',
`lazy-lock-continuity-time', `lazy-lock-stealth-time', `lazy-lock-delay-time',
and `lazy-lock-delay-commands'."
  (interactive "P")
  (set (make-local-variable 'lazy-lock-mode)
       (and (<= (or lazy-lock-minimum-size 0) (buffer-size))
	    (if arg (> (prefix-numeric-value arg) 0) (not lazy-lock-mode))))
  (if (not lazy-lock-mode)
      (let ((modified (buffer-modified-p))
	    (inhibit-read-only t))
	(remove-hook 'post-command-hook 'lazy-lock-post-command-hook)
	(remove-text-properties (point-min) (point-max) '(fontified nil))
	(or modified (set-buffer-modified-p nil)))
    (make-local-variable 'post-command-hook)
    ;; Unfortunately, `add-hook' with APPEND up to Emacs 19.27 is broke.
;    (add-hook 'post-command-hook 'lazy-lock-post-command-hook
;	      (and (boundp 'emacs-minor-version) (> emacs-minor-version 27)))
    (if (and (boundp 'emacs-minor-version) (> emacs-minor-version 27))
	(add-hook 'post-command-hook 'lazy-lock-post-command-hook t)
      (setq post-command-hook
	    (append (delq 'lazy-lock-post-command-hook post-command-hook)
		    '(lazy-lock-post-command-hook))))
    (set (make-local-variable 'font-lock-fontified) t)))

(defun lazy-lock-post-command-hook ()
  "Do groovy things for Lazy Lock mode.

 - Fontify the visible part of the buffer where necessary.  Only the selected
   window is fontified, unless `lazy-lock-walk-windows' is non-nil.  This
   occurs if there is no input after pausing for `lazy-lock-delay-time' seconds
   (if the current command is one of `lazy-lock-delay-commands') or after
   pausing for `lazy-lock-continuity-time' seconds (otherwise).

 - Fontify invisible parts of the buffer where necessary.  This occurs if there
   is no input after pausing for `lazy-lock-stealth-time' seconds.

See `lazy-lock-mode'."
  (if (sit-for (if (memq this-command lazy-lock-delay-commands)
		   lazy-lock-delay-time
		 lazy-lock-continuity-time))
      (progn
	;;
	;; Do the visible part of the buffer(s), i.e., the window(s).
	(if (or (not lazy-lock-walk-windows) (one-window-p t))
	    ;; Fontify this window.
	    (lazy-lock-fontify-window)
	  (let ((window (selected-window)) (current (current-buffer)))
	    ;; Fontify windows by walking through them.
	    (unwind-protect
		(walk-windows
		 (function (lambda (window)
			     (select-window window)
			     (if lazy-lock-mode (lazy-lock-fontify-window))))
		 'no-minibuf (eq lazy-lock-walk-windows 'all-frames))
	      (select-window window)
	      (set-buffer current))))
	;;
	;; Do invisible parts of the buffer.
	(if (and lazy-lock-stealth-time
		 (text-property-any (point-min) (point-max) 'fontified nil)
		 (sit-for lazy-lock-stealth-time))
	    (progn
	      (if font-lock-verbose (message "Fontifying stealthily..."))
	      (while (and (lazy-lock-fontify-stealthily)
			  (sit-for lazy-lock-continuity-time)))
	      (if font-lock-verbose (message "Fontifying stealthily... done."))
	      )))))

(defun lazy-lock-fontify-window ()
  "Fontify the visible part of the buffer where necessary.
See also `lazy-lock-fontify-buffer' and `lazy-lock-fontify-stealthily'."
  ;; We rely on `window-start' and `window-end' to have reasonable values.
  ;; Only fontify if `window-end' has changed.
  (let ((we (min (max (window-end) (point-min)) (point-max))))
    (if (/= we lazy-lock-cache-position)
	;; Find where we haven't `fontified' before.
	(let* ((ws (min (max (window-start) (point-min)) (point-max)))
	       (start (or (text-property-any ws we 'fontified nil) ws))
	       (end (or (text-property-any start we 'fontified t) we))
	       (modified (buffer-modified-p)) (inhibit-read-only t))
	  ;; Fontify between `start' and `end'.
	  (while (/= start end)
	    (save-excursion
	      (if font-lock-verbose (message "Fontifying window..."))
	      ;; Unfontify and then refontify.
	      (goto-char start) (beginning-of-line 1) (setq start (point))
	      (goto-char end) (or (bolp) (forward-line 1)) (setq end (point))
	      (if font-lock-no-comments
		  (font-lock-unfontify-region start end)
		(font-lock-fontify-region start end))
	      (font-lock-hack-keywords start end)
	      ;; Use `fontified' property to flag the region and find the next.
	      (put-text-property start end 'fontified t)
	      (setq start (or (text-property-any ws we 'fontified nil) ws)
		    end (or (text-property-any start we 'fontified t) we))
	      (if font-lock-verbose (message "Fontifying window... done."))))
	  (setq lazy-lock-cache-position we)
	  (or modified (set-buffer-modified-p nil))))))

(defun lazy-lock-fontify-buffer ()
  "Fontify the current buffer the way `font-lock-mode' would.
Completely fontifies the entire buffer."
  (interactive)
  ;; Could do only where not `fontified' before, but it might not be worth it.
  (let ((modified (buffer-modified-p))
	(inhibit-read-only t))
    (font-lock-fontify-buffer)
    (put-text-property (point-min) (point-max) 'fontified t)
    (or modified (set-buffer-modified-p nil))))

(defun lazy-lock-fontify-stealthily ()
  "Fontify an invisible part of the buffer where necessary.
See also `lazy-lock-fontify-window' and `lazy-lock-fontify-buffer'."
  ;; Find where the next and previous unfontified regions begin and end.
  ;; Assumes the current window is `fontified'.
  (let ((next (next-single-property-change (point) 'fontified))
	(prev (previous-single-property-change (point) 'fontified)))
    (if (and (not prev) (not next))
	nil
      (let ((modified (buffer-modified-p)) (inhibit-read-only t)
	    start end)
	(save-excursion
	  (save-restriction
	    (widen)
	    ;; Unfontify and then refontify.
	    (move-to-window-line nil)
	    (cond ((or (null prev)
		       (and next (> (- (point) prev) (- next (point)))))
		   ;; The next region is the nearest.
		   (goto-char next) (beginning-of-line 1) (setq start (point))
		   (forward-line (window-height)) (setq end (point))
		   ;; Maybe the region is already partially fontified.
		   (goto-char
		    (or (text-property-any start end 'fontified t) end))
		   (or (bolp) (forward-line 1)) (setq end (point)))
		  (t
		   ;; The previous region is the nearest.
		   (goto-char prev) (forward-line 1) (setq end (point))
		   (forward-line (- (window-height)))
		   ;; Maybe the region is already partially fontified.
		   (goto-char (text-property-any (point) end 'fontified nil))
		   (or (bolp) (beginning-of-line 1)) (setq start (point))))
	    (if font-lock-no-comments
		(font-lock-unfontify-region start end)
	      (font-lock-fontify-region start end))
	    (font-lock-hack-keywords start end)
	    ;; Use `fontified' property to flag the region.
	    (put-text-property start end 'fontified t)
	    (or modified (set-buffer-modified-p nil))))
	t))))

(defun turn-on-lazy-lock ()
  "Unconditionally turn on Lazy Lock mode."
  (lazy-lock-mode 1))

;; Install ourselves:

(or (assq 'lazy-lock-mode minor-mode-alist)
    (setq minor-mode-alist (cons '(lazy-lock-mode " Lazy") minor-mode-alist)))

;; Provide ourselves:

(provide 'lazy-lock)

;;; lazy-lock.el ends here

