/* Perform various loop optimizations, including strength reduction.
   Copyright (C) 1987, 88, 89, 91-6, 1997 Free Software Foundation, Inc.

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


/* This is the loop optimization pass of the compiler.
   It finds invariant computations within loops and moves them
   to the beginning of the loop.  Then it identifies basic and 
   general induction variables.  Strength reduction is applied to the general
   induction variables, and induction variable elimination is applied to
   the basic induction variables.

   It also finds cases where
   a register is set within the loop by zero-extending a narrower value
   and changes these to zero the entire register once before the loop
   and merely copy the low part within the loop.

   Most of the complexity is in heuristics to decide when it is worth
   while to do these things.  */

#include <stdio.h>
#include "config.h"
#include "rtl.h"
#include "obstack.h"
#include "expr.h"
#include "basic-block.h" /* intel1 */
#include "insn-config.h"
#include "insn-flags.h"
#include "regs.h"
#include "hard-reg-set.h"
#include "recog.h"
#include "flags.h"
#include "real.h"
#include "reload.h" /* intel1 */
#include "loop.h"
#include "except.h"

/* Vector mapping INSN_UIDs to luids.
   The luids are like uids but increase monotonically always.
   We use them to see whether a jump comes from outside a given loop.  */

int *uid_luid;

/* Indexed by INSN_UID, contains the ordinal giving the (innermost) loop
   number the insn is contained in.  */

int *uid_loop_num;

/* 1 + largest uid of any insn.  */

int max_uid_for_loop;

/* 1 + luid of last insn.  */

static int max_luid;

/* Number of loops detected in current function.  Used as index to the
   next few tables.  */

static int max_loop_num;

/* Indexed by loop number, contains the first and last insn of each loop.  */

static rtx *loop_number_loop_starts, *loop_number_loop_ends;

/* For each loop, gives the containing loop number, -1 if none.  */

int *loop_outer_loop;

/* Indexed by loop number, contains a nonzero value if the "loop" isn't
   really a loop (an insn outside the loop branches into it).  */

static char *loop_invalid;

/* Indexed by loop number, links together all LABEL_REFs which refer to
   code labels outside the loop.  Used by routines that need to know all
   loop exits, such as final_biv_value and final_giv_value.

   This does not include loop exits due to return instructions.  This is
   because all bivs and givs are pseudos, and hence must be dead after a
   return, so the presense of a return does not affect any of the
   optimizations that use this info.  It is simpler to just not include return
   instructions on this list.  */

rtx *loop_number_exit_labels;

/* Indexed by loop number, counts the number of LABEL_REFs on
   loop_number_exit_labels for this loop and all loops nested inside it.  */

int *loop_number_exit_count;

/* Holds the number of loop iterations.  It is zero if the number could not be
   calculated.  Must be unsigned since the number of iterations can
   be as high as 2^wordsize-1.  For loops with a wider iterator, this number
   will be zero if the number of loop iterations is too large for an
   unsigned integer to hold.  */

unsigned HOST_WIDE_INT loop_n_iterations;

/* Nonzero if there is a subroutine call in the current loop.
   (unknown_address_altered is also nonzero in this case.)  */

static int loop_has_call;

/* Nonzero if there is a volatile memory reference in the current
   loop.  */

static int loop_has_volatile;

/* Added loop_continue which is the NOTE_INSN_LOOP_CONT of the
   current loop.  A continue statement will generate a branch to
   NEXT_INSN (loop_continue).  */

static rtx loop_continue;

/* Indexed by register number, contains the number of times the reg
   is set during the loop being scanned.
   During code motion, a negative value indicates a reg that has been
   made a candidate; in particular -2 means that it is an candidate that
   we know is equal to a constant and -1 means that it is an candidate
   not known equal to a constant.
   After code motion, regs moved have 0 (which is accurate now)
   while the failed candidates have the original number of times set.

   Therefore, at all times, == 0 indicates an invariant register;
   < 0 a conditionally invariant one.  */

static short *n_times_set;

/* Original value of n_times_set; same except that this value
   is not set negative for a reg whose sets have been made candidates
   and not set to 0 for a reg that is moved.  */

static short *n_times_used;

/* Index by register number, 1 indicates that the register
   cannot be moved or strength reduced.  */

static char *may_not_optimize;

/* Nonzero means reg N has already been moved out of one loop.
   This reduces the desire to move it out of another.  */

static char *moved_once;

/* Array of MEMs that are stored in this loop. If there are too many to fit
   here, we just turn on unknown_address_altered.  */

#define NUM_STORES 20
static rtx loop_store_mems[NUM_STORES];

/* Index of first available slot in above array.  */
static int loop_store_mems_idx;

/* Nonzero if we don't know what MEMs were changed in the current loop.
   This happens if the loop contains a call (in which case `loop_has_call'
   will also be set) or if we store into more than NUM_STORES MEMs.  */

static int unknown_address_altered;

/* Count of movable (i.e. invariant) instructions discovered in the loop.  */
static int num_movables;

/* Count of memory write instructions discovered in the loop.  */
static int num_mem_sets;

/* Number of loops contained within the current one, including itself.  */
static int loops_enclosed;

/* Bound on pseudo register number before loop optimization.
   A pseudo has valid regscan info if its number is < max_reg_before_loop.  */
int max_reg_before_loop;

/* This obstack is used in product_cheap_p to allocate its rtl.  It
   may call gen_reg_rtx which, in turn, may reallocate regno_reg_rtx.
   If we used the same obstack that it did, we would be deallocating
   that array.  */

static struct obstack temp_obstack;

/* intel1 */
extern struct obstack momentary_obstack;

/* This is where the pointer to the obstack being used for RTL is stored.  */

extern struct obstack *rtl_obstack;

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

extern char *oballoc ();

/* During the analysis of a loop, a chain of `struct movable's
   is made to record all the movable insns found.
   Then the entire chain can be scanned to decide which to move.  */

struct movable
{
  rtx insn;			/* A movable insn */
  rtx set_src;			/* The expression this reg is set from.  */
  rtx set_dest;			/* The destination of this SET.  */
  rtx dependencies;		/* When INSN is libcall, this is an EXPR_LIST
				   of any registers used within the LIBCALL.  */
  int consec;			/* Number of consecutive following insns 
				   that must be moved with this one.  */
  int regno;			/* The register it sets */
  short lifetime;		/* lifetime of that register;
				   may be adjusted when matching movables
				   that load the same value are found.  */
  short savings;		/* Number of insns we can move for this reg,
				   including other movables that force this
				   or match this one.  */
  unsigned int cond : 1;	/* 1 if only conditionally movable */
  unsigned int force : 1;	/* 1 means MUST move this insn */
  unsigned int global : 1;	/* 1 means reg is live outside this loop */
		/* If PARTIAL is 1, GLOBAL means something different:
		   that the reg is live outside the range from where it is set
		   to the following label.  */
  unsigned int done : 1;	/* 1 inhibits further processing of this */
  
  unsigned int partial : 1;	/* 1 means this reg is used for zero-extending.
				   In particular, moving it does not make it
				   invariant.  */
  unsigned int move_insn : 1;	/* 1 means that we call emit_move_insn to
				   load SRC, rather than copying INSN.  */
  unsigned int is_equiv : 1;	/* 1 means a REG_EQUIV is present on INSN.  */
  enum machine_mode savemode;   /* Nonzero means it is a mode for a low part
				   that we should avoid changing when clearing
				   the rest of the reg.  */
  struct movable *match;	/* First entry for same value */
  struct movable *forces;	/* An insn that must be moved if this is */
  struct movable *next;
};

FILE *loop_dump_stream;

/* Forward declarations.  */

static void find_and_verify_loops ();
static void mark_loop_jump ();
static void prescan_loop ();
static int reg_in_basic_block_p ();
static int consec_sets_invariant_p ();
static rtx libcall_other_reg ();
static int labels_in_range_p ();
static void count_loop_regs_set ();
static void note_addr_stored ();
static int loop_reg_used_before_p ();
static void scan_loop ();
static void replace_call_address ();
static rtx skip_consec_insns ();
static int libcall_benefit ();
static void ignore_some_movables ();
static void force_movables ();
static void combine_movables ();
static int rtx_equal_for_loop_p ();
static void move_movables ();
static void strength_reduce ();
static int valid_initial_value_p ();
static void find_mem_givs ();
static void record_biv ();
static void check_final_value ();
static void record_giv ();
static void update_giv_derive ();
static int basic_induction_var ();
static rtx simplify_giv_expr ();
static int general_induction_var ();
static int consec_sets_giv ();
static int check_dbra_loop ();
static rtx express_from ();
static int combine_givs_p ();
static void combine_givs ();
static int product_cheap_p ();
static int maybe_eliminate_biv ();
static int maybe_eliminate_biv_1 ();
static int last_use_this_basic_block ();
static void record_initial ();
static void update_reg_last_use ();
/* intel1 */
static int compare_elimination ();
static attempt_new_rtx ();
static rtx make_new_use ();
static int possible_compare_elimination ();
static void find_regs_mentioned ();
static void use_regs_source ();
static rtx replace_mems ();
static void replace_reg  ();
static int two_to_power ();
static int jumps_back_in_loop ();
static loop_copy_propogate ();
static do_replace ();
static rtx subst_cands ();
static try_unroll_propogation ();
static rtx unroll_subst_cands ();
static unroll_consistancy_maintained ();
static int do_reduce_index ();
static loop_spl ();
void loop_after_global ();
static int spill_invariant_p ();
static void move_spills_out ();
static void move_spill_movables ();
static rtx subst_stack_slot ();
static int check_if_jumps_out_of_loop ();
static void mark_regs_alive_at_exit ();
static void verify_spill_candidates ();
static void mark_spill_candidates ();
static void spill_handle_regs_mentioned ();
static int is_updated_by_const ();
static int is_updated_by_const_or_reg ();
static void peep_init_info ();
static void spill_peep_insn ();
void update_stack_offset ();
static void replace_spill_slots ();
static rtx replace_linked_regs ();
static void mark_regs_live_outside ();
static void mark_regs_mentioned ();
static int loop_contains_loop ();
static void push_load_into_loop ();
static int spill_slot_unused_between ();
static int spill_slot_mentioned ();
void init_undo_buf ();
void subst_in_insn ();
static void inner_subst_in_insn ();
void loop_undo_all ();
static void update_last_reloads ();
static void sign_extension_reduction ();
static void validate_all_references ();
static void lift_stores ();
static void mark_mem_refs ();
static void insert_mem_ref ();
static void opt_reg_use_copy ();
static int peep_spills_mark_reload_slots PROTO((rtx, rtx));

int doing_spl = 0;
int doing_loop_after_global = 0;
static char * reg_live_outside_loop;
static int cur_stack_offset = 0;


/* Relative gain of eliminating various kinds of operations.  */
int add_cost;
#if 0
int shift_cost;
int mult_cost;
#endif
/* intel1 maximum # of insns in loop for which compare elimination */
#ifndef MAX_CMP_ELIM
#define MAX_CMP_ELIM 100
#endif

/* Benefit penalty, if a giv is not replaceable, i.e. must emit an insn to
   copy the value of the strength reduced giv to its original register.  */
int copy_cost;

void
init_loop ()
{
  char *free_point = (char *) oballoc (1);
  rtx reg = gen_rtx (REG, word_mode, LAST_VIRTUAL_REGISTER + 1);

  add_cost = rtx_cost (gen_rtx (PLUS, word_mode, reg, reg), SET);

  /* We multiply by 2 to reconcile the difference in scale between
     these two ways of computing costs.  Otherwise the cost of a copy
     will be far less than the cost of an add.  */

  copy_cost = 2 * 2;

  /* Free the objects we just allocated.  */
  obfree (free_point);

  /* Initialize the obstack used for rtl in product_cheap_p.  */
  gcc_obstack_init (&temp_obstack);
}

/* Entry point of this file.  Perform loop optimization
   on the current function.  F is the first insn of the function
   and DUMPFILE is a stream for output of a trace of actions taken
   (or 0 if none should be output).  */

void
loop_optimize (f, dumpfile)
     /* f is the first instruction of a chain of insns for one function */
     rtx f;
     FILE *dumpfile;
{
  register rtx insn;
  register int i;
  rtx last_insn;

  loop_dump_stream = dumpfile;

  init_recog_no_volatile ();
  init_alias_analysis ();

  max_reg_before_loop = max_reg_num ();

  /* intel1 */
#ifdef SAVE_ON_REGS
  if (!flag_all_mem_givs && flag_opt_reg_use)
    reg_live_outside_loop = (char *) alloca(max_reg_before_loop);
#endif

  moved_once = (char *) alloca (max_reg_before_loop);
  bzero (moved_once, max_reg_before_loop);

  regs_may_share = 0;

  /* Count the number of loops.  */

  max_loop_num = 0;
  for (insn = f; insn; insn = NEXT_INSN (insn))
    {
      if (GET_CODE (insn) == NOTE
	  && NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_BEG)
	max_loop_num++;
    }

  /* Don't waste time if no loops.  */
  if (max_loop_num == 0)
    {
      /* intel1 */
#ifdef SAVE_ON_REGS
      if (!doing_loop_after_global && flag_opt_reg_use)
        {
          use_regs_source (f, NULL, 0);
          opt_reg_use_copy (f);
        }
#endif
      return;
    }

  /* Get size to use for tables indexed by uids.
     Leave some space for labels allocated by find_and_verify_loops.  */
  max_uid_for_loop = get_max_uid () + 1 + max_loop_num * 32;

  uid_luid = (int *) alloca (max_uid_for_loop * sizeof (int));
  uid_loop_num = (int *) alloca (max_uid_for_loop * sizeof (int));

  bzero ((char *) uid_luid, max_uid_for_loop * sizeof (int));
  bzero ((char *) uid_loop_num, max_uid_for_loop * sizeof (int));

  /* Allocate tables for recording each loop.  We set each entry, so they need
     not be zeroed.  */
  loop_number_loop_starts = (rtx *) alloca (max_loop_num * sizeof (rtx));
  loop_number_loop_ends = (rtx *) alloca (max_loop_num * sizeof (rtx));
  loop_outer_loop = (int *) alloca (max_loop_num * sizeof (int));
  loop_invalid = (char *) alloca (max_loop_num * sizeof (char));
  loop_number_exit_labels = (rtx *) alloca (max_loop_num * sizeof (rtx));
  loop_number_exit_count = (int *) alloca (max_loop_num * sizeof (int));

  /* Find and process each loop.
     First, find them, and record them in order of their beginnings.  */
  find_and_verify_loops (f);

  /* Now find all register lifetimes.  This must be done after
     find_and_verify_loops, because it might reorder the insns in the
     function.  */
  /* intel1  Last paramater of reg_scan used to be 1.  */
  reg_scan (f, max_reg_num (), !doing_loop_after_global);

  /* See if we went too far.  */
  if (get_max_uid () > max_uid_for_loop)
    abort ();

  /* Compute the mapping from uids to luids.
     LUIDs are numbers assigned to insns, like uids,
     except that luids increase monotonically through the code.
     Don't assign luids to line-number NOTEs, so that the distance in luids
     between two insns is not affected by -g.  */

  for (insn = f, i = 0; insn; insn = NEXT_INSN (insn))
    {
      last_insn = insn;
      if (GET_CODE (insn) != NOTE
	  || NOTE_LINE_NUMBER (insn) <= 0)
	uid_luid[INSN_UID (insn)] = ++i;
      else
	/* Give a line number note the same luid as preceding insn.  */
	uid_luid[INSN_UID (insn)] = i;
    }

  max_luid = i + 1;

  /* Don't leave gaps in uid_luid for insns that have been
     deleted.  It is possible that the first or last insn
     using some register has been deleted by cross-jumping.
     Make sure that uid_luid for that former insn's uid
     points to the general area where that insn used to be.  */
  for (i = 0; i < max_uid_for_loop; i++)
    {
      uid_luid[0] = uid_luid[i];
      if (uid_luid[0] != 0)
	break;
    }
  for (i = 0; i < max_uid_for_loop; i++)
    if (uid_luid[i] == 0)
      uid_luid[i] = uid_luid[i - 1];

  /* Create a mapping from loops to BLOCK tree nodes.  */
  if (flag_unroll_loops && write_symbols != NO_DEBUG)
    find_loop_tree_blocks ();

  /* Now scan the loops, last ones first, since this means inner ones are done
     before outer ones.  */
  for (i = max_loop_num-1; i >= 0; i--)
    if (!loop_invalid[i] && loop_number_loop_ends[i])
      scan_loop (loop_number_loop_starts[i], loop_number_loop_ends[i],
		 max_reg_num ());

  /* intel1 */
  if (flag_strength_reduce && flag_copy_prop)
    for (i = max_loop_num-1; i >= 0; i--)
      if (!loop_invalid[i] && loop_number_loop_ends[i])
        loop_copy_propogate (loop_number_loop_starts[i], loop_number_loop_ends[i]);

  /* intel1 */
  if ((flag_lift_stores || flag_runtime_lift_stores) 
      && !doing_loop_after_global)
    for (i = max_loop_num-1; i >= 0; i--)
      if (!loop_invalid[i] && loop_number_loop_ends[i])
        lift_stores (loop_number_loop_starts[i], loop_number_loop_ends[i]);

#ifdef SAVE_ON_REGS
  if (!doing_loop_after_global && flag_opt_reg_use)
    {
      use_regs_source (f, NULL, 0); 
      opt_reg_use_copy (f);
    }
#endif

  /* intel1 */
#ifdef STORE_NOT_SET_CC0
#ifdef STACK_REGS
  if (flag_sftwr_pipe && !doing_loop_after_global)
    for (i = max_loop_num-1; i >= 0; i--)
      if (!loop_invalid[i] && loop_number_loop_ends[i])
        loop_spl (loop_number_loop_starts[i], loop_number_loop_ends[i], i, 0);
#endif
#endif
  /* If debugging and unrolling loops, we must replicate the tree nodes
     corresponding to the blocks inside the loop, so that the original one
     to one mapping will remain.  */
  if (flag_unroll_loops && write_symbols != NO_DEBUG)
    unroll_block_trees ();
}

/* Optimize one loop whose start is LOOP_START and end is END.
   LOOP_START is the NOTE_INSN_LOOP_BEG and END is the matching
   NOTE_INSN_LOOP_END.  */

/* ??? Could also move memory writes out of loops if the destination address
   is invariant, the source is invariant, the memory write is not volatile,
   and if we can prove that no read inside the loop can read this address
   before the write occurs.  If there is a read of this address after the
   write, then we can also mark the memory read as invariant.  */

static void
scan_loop (loop_start, end, nregs)
     rtx loop_start, end;
     int nregs;
{
  register int i;
  register rtx p;
  /* 1 if we are scanning insns that could be executed zero times.  */
  int maybe_never = 0;
  /* 1 if we are scanning insns that might never be executed
     due to a subroutine call which might exit before they are reached.  */
  int call_passed = 0;
  /* For a rotated loop that is entered near the bottom,
     this is the label at the top.  Otherwise it is zero.  */
  rtx loop_top = 0;
  /* Jump insn that enters the loop, or 0 if control drops in.  */
  rtx loop_entry_jump = 0;
  /* Place in the loop where control enters.  */
  rtx scan_start;
  /* Number of insns in the loop.  */
  int insn_count;
  int in_libcall = 0;
  int tem;
  rtx temp;
  /* The SET from an insn, if it is the only SET in the insn.  */
  rtx set, set1;
  /* Chain describing insns movable in current loop.  */
  struct movable *movables = 0;
  /* Last element in `movables' -- so we can add elements at the end.  */
  struct movable *last_movable = 0;
  /* Ratio of extra register life span we can justify
     for saving an instruction.  More if loop doesn't call subroutines
     since in that case saving an insn makes more difference
     and more registers are available.  */
  int threshold;
  /* If we have calls, contains the insn in which a register was used
     if it was used exactly once; contains const0_rtx if it was used more
     than once.  */
  rtx *reg_single_usage = 0;
  
  /* Nonzero if we are scanning instructions in a sub-loop.  */
  int loop_depth = 0;

  /* intel1 */
  if (doing_loop_after_global
     && !(GET_CODE (PREV_INSN (end)) == JUMP_INSN
          && GET_CODE (NEXT_INSN (loop_start)) == CODE_LABEL
          && JUMP_LABEL (PREV_INSN (end)) == NEXT_INSN (loop_start)))
    return;

#ifdef SAVE_ON_REGS
  if (!flag_all_mem_givs && !doing_loop_after_global
     && !loop_contains_loop (loop_start, end) && flag_opt_reg_use)
    {
      bzero (reg_live_outside_loop, max_reg_before_loop);
      mark_regs_live_outside (loop_start, end);
    }
#endif

  n_times_set = (short *) alloca (nregs * sizeof (short));
  n_times_used = (short *) alloca (nregs * sizeof (short));
  may_not_optimize = (char *) alloca (nregs);

  /* Determine whether this loop starts with a jump down to a test at
     the end.  This will occur for a small number of loops with a test
     that is too complex to duplicate in front of the loop.

     We search for the first insn or label in the loop, skipping NOTEs.
     However, we must be careful not to skip past a NOTE_INSN_LOOP_BEG
     (because we might have a loop executed only once that contains a
     loop which starts with a jump to its exit test) or a NOTE_INSN_LOOP_END
     (in case we have a degenerate loop).

     Note that if we mistakenly think that a loop is entered at the top
     when, in fact, it is entered at the exit test, the only effect will be
     slightly poorer optimization.  Making the opposite error can generate
     incorrect code.  Since very few loops now start with a jump to the 
     exit test, the code here to detect that case is very conservative.  */

  for (p = NEXT_INSN (loop_start);
       p != end
	 && GET_CODE (p) != CODE_LABEL && GET_RTX_CLASS (GET_CODE (p)) != 'i'
	 && (GET_CODE (p) != NOTE
	     || (NOTE_LINE_NUMBER (p) != NOTE_INSN_LOOP_BEG
		 && NOTE_LINE_NUMBER (p) != NOTE_INSN_LOOP_END));
       p = NEXT_INSN (p))
    ;

  scan_start = p;

  /* Set up variables describing this loop.  */
  prescan_loop (loop_start, end);
  threshold = (loop_has_call ? 1 : 2) * (1 + n_non_fixed_regs);

  /* If loop has a jump before the first label,
     the true entry is the target of that jump.
     Start scan from there.
     But record in LOOP_TOP the place where the end-test jumps
     back to so we can scan that after the end of the loop.  */
  if (GET_CODE (p) == JUMP_INSN)
    {
      loop_entry_jump = p;

      /* Loop entry must be unconditional jump (and not a RETURN)  */
      if (simplejump_p (p)
	  && JUMP_LABEL (p) != 0
	  /* Check to see whether the jump actually
	     jumps out of the loop (meaning it's no loop).
	     This case can happen for things like
	     do {..} while (0).  If this label was generated previously
	     by loop, we can't tell anything about it and have to reject
	     the loop.  */
	  && INSN_UID (JUMP_LABEL (p)) < max_uid_for_loop
	  && INSN_LUID (JUMP_LABEL (p)) >= INSN_LUID (loop_start)
	  && INSN_LUID (JUMP_LABEL (p)) < INSN_LUID (end))
	{
	  loop_top = next_label (scan_start);
	  scan_start = JUMP_LABEL (p);
	}
    }

  /* If SCAN_START was an insn created by loop, we don't know its luid
     as required by loop_reg_used_before_p.  So skip such loops.  (This
     test may never be true, but it's best to play it safe.) 

     Also, skip loops where we do not start scanning at a label.  This
     test also rejects loops starting with a JUMP_INSN that failed the
     test above.  */

  if (INSN_UID (scan_start) >= max_uid_for_loop
      || GET_CODE (scan_start) != CODE_LABEL)
    {
      if (loop_dump_stream)
	fprintf (loop_dump_stream, "\nLoop from %d to %d is phony.\n\n",
		 INSN_UID (loop_start), INSN_UID (end));
      return;
    }

  /* Count number of times each reg is set during this loop.
     Set may_not_optimize[I] if it is not safe to move out
     the setting of register I.  If this loop has calls, set
     reg_single_usage[I].  */

  bzero ((char *) n_times_set, nregs * sizeof (short));
  bzero (may_not_optimize, nregs);

  if (loop_has_call)
    {
      reg_single_usage = (rtx *) alloca (nregs * sizeof (rtx));
      bzero ((char *) reg_single_usage, nregs * sizeof (rtx));
    }

  count_loop_regs_set (loop_top ? loop_top : loop_start, end,
		       may_not_optimize, reg_single_usage, &insn_count, nregs);
  
  /* intel1 added if but not body */
  if (!doing_loop_after_global)
    for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
      may_not_optimize[i] = 1, n_times_set[i] = 1;
  
  bcopy ((char *) n_times_set, (char *) n_times_used, nregs * sizeof (short));

  if (loop_dump_stream)
    {
      fprintf (loop_dump_stream, "\nLoop from %d to %d: %d real insns.\n",
	       INSN_UID (loop_start), INSN_UID (end), insn_count);
      if (loop_continue)
	fprintf (loop_dump_stream, "Continue at insn %d.\n",
		 INSN_UID (loop_continue));
    }

  /* Scan through the loop finding insns that are safe to move.
     Set n_times_set negative for the reg being set, so that
     this reg will be considered invariant for subsequent insns.
     We consider whether subsequent insns use the reg
     in deciding whether it is worth actually moving.

     MAYBE_NEVER is nonzero if we have passed a conditional jump insn
     and therefore it is possible that the insns we are scanning
     would never be executed.  At such times, we must make sure
     that it is safe to execute the insn once instead of zero times.
     When MAYBE_NEVER is 0, all insns will be executed at least once
     so that is not a problem.  */

  p = scan_start;
  while (1)
    {
      p = NEXT_INSN (p);
      /* At end of a straight-in loop, we are done.
	 At end of a loop entered at the bottom, scan the top.  */
      if (p == scan_start)
	break;
      if (p == end)
	{
	  if (loop_top != 0)
	    p = loop_top;
	  else
	    break;
	  if (p == scan_start)
	    break;
	}

      if (GET_RTX_CLASS (GET_CODE (p)) == 'i'
	  && find_reg_note (p, REG_LIBCALL, NULL_RTX))
	in_libcall = 1;
      else if (GET_RTX_CLASS (GET_CODE (p)) == 'i'
	       && find_reg_note (p, REG_RETVAL, NULL_RTX))
	in_libcall = 0;

      if (GET_CODE (p) == INSN
          /* intel1 */
      && (!doing_loop_after_global || GET_CODE(PATTERN(p)) == SET)
	  && (set = single_set (p))
	  && GET_CODE (SET_DEST (set)) == REG
	  && !may_not_optimize[REGNO (SET_DEST (set))])
	{
	  int tem1 = 0;
	  int tem2 = 0;
	  int move_insn = 0;
	  rtx src = SET_SRC (set);
	  rtx dependencies = 0;

	  /* Figure out what to use as a source of this insn.  If a REG_EQUIV
	     note is given or if a REG_EQUAL note with a constant operand is
	     specified, use it as the source and mark that we should move
	     this insn by calling emit_move_insn rather that duplicating the
	     insn.

	     Otherwise, only use the REG_EQUAL contents if a REG_RETVAL note
	     is present.  */
	  temp = find_reg_note (p, REG_EQUIV, NULL_RTX);
	  if (temp)
	    src = XEXP (temp, 0), move_insn = 1;
	  else 
	    {
	      temp = find_reg_note (p, REG_EQUAL, NULL_RTX);
	      if (temp && CONSTANT_P (XEXP (temp, 0)))
		src = XEXP (temp, 0), move_insn = 1;
	      if (temp && find_reg_note (p, REG_RETVAL, NULL_RTX))
		{
		  src = XEXP (temp, 0);
		  /* A libcall block can use regs that don't appear in
		     the equivalent expression.  To move the libcall,
		     we must move those regs too.  */
		  dependencies = libcall_other_reg (p, src);
		}
	    }

	  /* Don't try to optimize a register that was made
	     by loop-optimization for an inner loop.
	     We don't know its life-span, so we can't compute the benefit.  */
	  if (REGNO (SET_DEST (set)) >= max_reg_before_loop)
	    ;
	  /* In order to move a register, we need to have one of three cases:
	     (1) it is used only in the same basic block as the set
	     (2) it is not a user variable and it is not used in the
	         exit test (this can cause the variable to be used
		 before it is set just like a user-variable).
	     (3) the set is guaranteed to be executed once the loop starts,
	         and the reg is not used until after that.  */
	  else if (! ((! maybe_never
		       && ! loop_reg_used_before_p (set, p, loop_start,
						    scan_start, end))
		      ||  (!doing_loop_after_global 
		           && /* intel1 */!REG_USERVAR_P (SET_DEST (set))
		           && ! REG_LOOP_TEST_P (SET_DEST (set)))
		      || reg_in_basic_block_p (p, SET_DEST (set))))
	    ;
	  else if (((tem = invariant_p (src))
	             /* intel1 */ || (doing_loop_after_global 
	             && RTX_IS_SPILL_P (p) == 1 && spill_invariant_p (src))
	           )
	           /* intel1 */
                    && !(doing_loop_after_global 
	            && call_used_regs[REGNO (SET_DEST (set))]
	            && loop_has_call)
		   && (dependencies == 0
		       || (tem2 = invariant_p (dependencies)) != 0)
		   && (n_times_set[REGNO (SET_DEST (set))] == 1
		       || (tem1
			   = consec_sets_invariant_p (SET_DEST (set),
						      n_times_set[REGNO (SET_DEST (set))],
						      p)))
		   /* If the insn can cause a trap (such as divide by zero),
		      can't move it unless it's guaranteed to be executed
		      once loop is entered.  Even a function call might
		      prevent the trap insn from being reached
		      (since it might exit!)  */
		   && ! ((maybe_never || call_passed)
			 && may_trap_p (src)))
	    {
	      register struct movable *m;
	      register int regno = REGNO (SET_DEST (set));

	      /* A potential lossage is where we have a case where two insns
		 can be combined as long as they are both in the loop, but
		 we move one of them outside the loop.  For large loops,
		 this can lose.  The most common case of this is the address
		 of a function being called.  

		 Therefore, if this register is marked as being used exactly
		 once if we are in a loop with calls (a "large loop"), see if
		 we can replace the usage of this register with the source
		 of this SET.  If we can, delete this insn. 

		 Don't do this if P has a REG_RETVAL note or if we have
		 SMALL_REGISTER_CLASSES and SET_SRC is a hard register.  */

	      if (reg_single_usage && reg_single_usage[regno] != 0
		  && reg_single_usage[regno] != const0_rtx
		  && regno_first_uid[regno] == INSN_UID (p)
		  && (regno_last_uid[regno]
		      == INSN_UID (reg_single_usage[regno]))
		  && n_times_set[REGNO (SET_DEST (set))] == 1
		  && ! side_effects_p (SET_SRC (set))
		  && ! find_reg_note (p, REG_RETVAL, NULL_RTX)
#ifdef SMALL_REGISTER_CLASSES
		  && ! (SMALL_REGISTER_CLASSES
			&& GET_CODE (SET_SRC (set)) == REG
			&& REGNO (SET_SRC (set)) < FIRST_PSEUDO_REGISTER)
#endif
		  /* This test is not redundant; SET_SRC (set) might be
		     a call-clobbered register and the life of REGNO
		     might span a call.  */
		  && ! modified_between_p (SET_SRC (set), p,
					   reg_single_usage[regno])
		  && !doing_loop_after_global /* intel1 */
		  && no_labels_between_p (p, reg_single_usage[regno])
		  && validate_replace_rtx (SET_DEST (set), SET_SRC (set),
					   reg_single_usage[regno]))
		{
		  /* Replace any usage in a REG_EQUAL note.  Must copy the
		     new source, so that we don't get rtx sharing between the
		     SET_SOURCE and REG_NOTES of insn p.  */
		  REG_NOTES (reg_single_usage[regno])
		    = replace_rtx (REG_NOTES (reg_single_usage[regno]),
				   SET_DEST (set), copy_rtx (SET_SRC (set)));
				   
		  PUT_CODE (p, NOTE);
		  NOTE_LINE_NUMBER (p) = NOTE_INSN_DELETED;
		  NOTE_SOURCE_FILE (p) = 0;
		  n_times_set[regno] = 0;
		  continue;
		}

	      m = (struct movable *) alloca (sizeof (struct movable));
	      m->next = 0;
	      m->insn = p;
	      m->set_src = src;
	      m->dependencies = dependencies;
	      m->set_dest = SET_DEST (set);
	      m->force = 0;
	      m->consec = n_times_set[REGNO (SET_DEST (set))] - 1;
	      m->done = 0;
	      m->forces = 0;
	      m->partial = 0;
	      m->move_insn = move_insn;
	      m->is_equiv = (find_reg_note (p, REG_EQUIV, NULL_RTX) != 0);
	      m->savemode = VOIDmode;
	      m->regno = regno;
	      /* Set M->cond if either invariant_p or consec_sets_invariant_p
		 returned 2 (only conditionally invariant).  */
	      m->cond = ((tem | tem1 | tem2) > 1);
	      m->global = (uid_luid[regno_last_uid[regno]] > INSN_LUID (end)
			   || uid_luid[regno_first_uid[regno]] < INSN_LUID (loop_start));
	      m->match = 0;
	      m->lifetime = (uid_luid[regno_last_uid[regno]]
			     - uid_luid[regno_first_uid[regno]]);
	      m->savings = n_times_used[regno];
	      if (find_reg_note (p, REG_RETVAL, NULL_RTX))
		m->savings += libcall_benefit (p);
	      n_times_set[regno] = move_insn ? -2 : -1;
	      /* Add M to the end of the chain MOVABLES.  */
	      if (movables == 0)
		movables = m;
	      else
		last_movable->next = m;
	      last_movable = m;

	      if (m->consec > 0)
		{
		  /* Skip this insn, not checking REG_LIBCALL notes.  */
		  p = next_nonnote_insn (p);
		  /* Skip the consecutive insns, if there are any.  */
		  p = skip_consec_insns (p, m->consec);
		  /* Back up to the last insn of the consecutive group.  */
		  p = prev_nonnote_insn (p);

		  /* We must now reset m->move_insn, m->is_equiv, and possibly
		     m->set_src to correspond to the effects of all the
		     insns.  */
		  temp = find_reg_note (p, REG_EQUIV, NULL_RTX);
		  if (temp)
		    m->set_src = XEXP (temp, 0), m->move_insn = 1;
		  else
		    {
		      temp = find_reg_note (p, REG_EQUAL, NULL_RTX);
		      if (temp && CONSTANT_P (XEXP (temp, 0)))
			m->set_src = XEXP (temp, 0), m->move_insn = 1;
		      else
			m->move_insn = 0;

		    }
		  m->is_equiv = (find_reg_note (p, REG_EQUIV, NULL_RTX) != 0);
		}
	    }
	  /* If this register is always set within a STRICT_LOW_PART
	     or set to zero, then its high bytes are constant.
	     So clear them outside the loop and within the loop
	     just load the low bytes.
	     We must check that the machine has an instruction to do so.
	     Also, if the value loaded into the register
	     depends on the same register, this cannot be done.  */
	  else if (SET_SRC (set) == const0_rtx
		   && GET_CODE (NEXT_INSN (p)) == INSN
		   && (set1 = single_set (NEXT_INSN (p)))
		   && GET_CODE (set1) == SET
		   && (GET_CODE (SET_DEST (set1)) == STRICT_LOW_PART)
		   && (GET_CODE (XEXP (SET_DEST (set1), 0)) == SUBREG)
		   && (SUBREG_REG (XEXP (SET_DEST (set1), 0))
		       == SET_DEST (set))
		   && !reg_mentioned_p (SET_DEST (set), SET_SRC (set1)))
	    {
	      register int regno = REGNO (SET_DEST (set));
	      if (n_times_set[regno] == 2)
		{
		  register struct movable *m;
		  m = (struct movable *) alloca (sizeof (struct movable));
		  m->next = 0;
		  m->insn = p;
		  m->set_dest = SET_DEST (set);
		  m->dependencies = 0;
		  m->force = 0;
		  m->consec = 0;
		  m->done = 0;
		  m->forces = 0;
		  m->move_insn = 0;
		  m->partial = 1;
		  /* If the insn may not be executed on some cycles,
		     we can't clear the whole reg; clear just high part.
		     Not even if the reg is used only within this loop.
		     Consider this:
		     while (1)
		       while (s != t) {
		         if (foo ()) x = *s;
			 use (x);
		       }
		     Clearing x before the inner loop could clobber a value
		     being saved from the last time around the outer loop.
		     However, if the reg is not used outside this loop
		     and all uses of the register are in the same
		     basic block as the store, there is no problem.

		     If this insn was made by loop, we don't know its
		     INSN_LUID and hence must make a conservative
		     assumption.  */
		  m->global = (INSN_UID (p) >= max_uid_for_loop
			       || (uid_luid[regno_last_uid[regno]]
				   > INSN_LUID (end))
			       || (uid_luid[regno_first_uid[regno]]
				   < INSN_LUID (p))
			       || (labels_in_range_p
				   (p, uid_luid[regno_first_uid[regno]])));
		  if (maybe_never && m->global)
		    m->savemode = GET_MODE (SET_SRC (set1));
		  else
		    m->savemode = VOIDmode;
		  m->regno = regno;
		  m->cond = 0;
		  m->match = 0;
		  m->lifetime = (uid_luid[regno_last_uid[regno]]
				 - uid_luid[regno_first_uid[regno]]);
		  m->savings = 1;
		  n_times_set[regno] = -1;
		  /* Add M to the end of the chain MOVABLES.  */
		  if (movables == 0)
		    movables = m;
		  else
		    last_movable->next = m;
		  last_movable = m;
		}
	    }
	}
      /* Past a call insn, we get to insns which might not be executed
	 because the call might exit.  This matters for insns that trap.
	 Call insns inside a REG_LIBCALL/REG_RETVAL block always return,
	 so they don't count.  */
      else if (GET_CODE (p) == CALL_INSN && ! in_libcall)
	call_passed = 1;
      /* Past a label or a jump, we get to insns for which we
	 can't count on whether or how many times they will be
	 executed during each iteration.  Therefore, we can
	 only move out sets of trivial variables
	 (those not used after the loop).  */
      /* Similar code appears twice in strength_reduce.  */
      else if ((GET_CODE (p) == CODE_LABEL || GET_CODE (p) == JUMP_INSN)
	       /* If we enter the loop in the middle, and scan around to the
		  beginning, don't set maybe_never for that.  This must be an
		  unconditional jump, otherwise the code at the top of the
		  loop might never be executed.  Unconditional jumps are
		  followed a by barrier then loop end.  */
               && ! (GET_CODE (p) == JUMP_INSN && JUMP_LABEL (p) == loop_top
		     && NEXT_INSN (NEXT_INSN (p)) == end
		     && simplejump_p (p)))
	maybe_never = 1;
      else if (GET_CODE (p) == NOTE)
	{
	  /* At the virtual top of a converted loop, insns are again known to
	     be executed: logically, the loop begins here even though the exit
	     code has been duplicated.  */
	  if (NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_VTOP && loop_depth == 0)
	    maybe_never = call_passed = 0;
	  else if (NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_BEG)
	    loop_depth++;
	  else if (NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_END)
	    loop_depth--;
	}
    }

  /* If one movable subsumes another, ignore that other.  */

  ignore_some_movables (movables);

  /* For each movable insn, see if the reg that it loads
     leads when it dies right into another conditionally movable insn.
     If so, record that the second insn "forces" the first one,
     since the second can be moved only if the first is.  */

  force_movables (movables);

  /* See if there are multiple movable insns that load the same value.
     If there are, make all but the first point at the first one
     through the `match' field, and add the priorities of them
     all together as the priority of the first.  */

  combine_movables (movables, nregs);
	
  /* Now consider each movable insn to decide whether it is worth moving.
     Store 0 in n_times_set for each reg that is moved.  */

  move_movables (movables, threshold,
		 insn_count, loop_start, end, nregs);

  /* intel1 */
  if (doing_loop_after_global && !loop_has_call)
    move_spills_out(loop_start, end, nregs);

  /* Now candidates that still are negative are those not moved.
     Change n_times_set to indicate that those are not actually invariant.  */
  for (i = 0; i < nregs; i++)
    if (n_times_set[i] < 0)
      n_times_set[i] = n_times_used[i];

  /* intel1 */
#ifdef SAVE_ON_REGS
   if (!doing_loop_after_global && flag_opt_reg_use)
     use_regs_source(loop_start, end, 1); 
#endif

  if (flag_strength_reduce)
    strength_reduce (scan_start, end, loop_top,
		     insn_count, loop_start, end);
}

/* Add elements to *OUTPUT to record all the pseudo-regs
   mentioned in IN_THIS but not mentioned in NOT_IN_THIS.  */

void
record_excess_regs (in_this, not_in_this, output)
     rtx in_this, not_in_this;
     rtx *output;
{
  enum rtx_code code;
  char *fmt;
  int i;

  code = GET_CODE (in_this);

  switch (code)
    {
    case PC:
    case CC0:
    case CONST_INT:
    case CONST_DOUBLE:
    case CONST:
    case SYMBOL_REF:
    case LABEL_REF:
      return;

    case REG:
      if (REGNO (in_this) >= FIRST_PSEUDO_REGISTER
	  && ! reg_mentioned_p (in_this, not_in_this))
	*output = gen_rtx (EXPR_LIST, VOIDmode, in_this, *output);
      return;
    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      int j;

      switch (fmt[i])
	{
	case 'E':
	  for (j = 0; j < XVECLEN (in_this, i); j++)
	    record_excess_regs (XVECEXP (in_this, i, j), not_in_this, output);
	  break;

	case 'e':
	  record_excess_regs (XEXP (in_this, i), not_in_this, output);
	  break;
	}
    }
}

/* Check what regs are referred to in the libcall block ending with INSN,
   aside from those mentioned in the equivalent value.
   If there are none, return 0.
   If there are one or more, return an EXPR_LIST containing all of them.  */

static rtx
libcall_other_reg (insn, equiv)
     rtx insn, equiv;
{
  rtx note = find_reg_note (insn, REG_RETVAL, NULL_RTX);
  rtx p = XEXP (note, 0);
  rtx output = 0;

  /* First, find all the regs used in the libcall block
     that are not mentioned as inputs to the result.  */

  while (p != insn)
    {
      if (GET_CODE (p) == INSN || GET_CODE (p) == JUMP_INSN
	  || GET_CODE (p) == CALL_INSN)
	record_excess_regs (PATTERN (p), equiv, &output);
      p = NEXT_INSN (p);
    }

  return output;
}

/* Return 1 if all uses of REG
   are between INSN and the end of the basic block.  */

static int 
reg_in_basic_block_p (insn, reg)
     rtx insn, reg;
{
  int regno = REGNO (reg);
  rtx p;

  if (regno_first_uid[regno] != INSN_UID (insn))
    return 0;

  /* Search this basic block for the already recorded last use of the reg.  */
  for (p = insn; p; p = NEXT_INSN (p))
    {
      switch (GET_CODE (p))
	{
	case NOTE:
	  break;

	case INSN:
	case CALL_INSN:
	  /* Ordinary insn: if this is the last use, we win.  */
	  if (regno_last_uid[regno] == INSN_UID (p))
	    return 1;
	  break;

	case JUMP_INSN:
	  /* Jump insn: if this is the last use, we win.  */
	  if (regno_last_uid[regno] == INSN_UID (p))
	    return 1;
	  /* Otherwise, it's the end of the basic block, so we lose.  */
	  return 0;

	case CODE_LABEL:
	case BARRIER:
	  /* It's the end of the basic block, so we lose.  */
	  return 0;
	}
    }

  /* The "last use" doesn't follow the "first use"??  */
  abort ();
}

/* Compute the benefit of eliminating the insns in the block whose
   last insn is LAST.  This may be a group of insns used to compute a
   value directly or can contain a library call.  */

static int
libcall_benefit (last)
     rtx last;
{
  rtx insn;
  int benefit = 0;

  for (insn = XEXP (find_reg_note (last, REG_RETVAL, NULL_RTX), 0);
       insn != last; insn = NEXT_INSN (insn))
    {
      if (GET_CODE (insn) == CALL_INSN)
	benefit += 10;		/* Assume at least this many insns in a library
				   routine.  */
      else if (GET_CODE (insn) == INSN
	       && GET_CODE (PATTERN (insn)) != USE
	       && GET_CODE (PATTERN (insn)) != CLOBBER)
	benefit++;
    }

  return benefit;
}

/* Skip COUNT insns from INSN, counting library calls as 1 insn.  */

static rtx
skip_consec_insns (insn, count)
     rtx insn;
     int count;
{
  for (; count > 0; count--)
    {
      rtx temp;

      /* If first insn of libcall sequence, skip to end.  */
      /* Do this at start of loop, since INSN is guaranteed to 
	 be an insn here.  */
      if (GET_CODE (insn) != NOTE
	  && (temp = find_reg_note (insn, REG_LIBCALL, NULL_RTX)))
	insn = XEXP (temp, 0);

      do insn = NEXT_INSN (insn);
      while (GET_CODE (insn) == NOTE);
    }

  return insn;
}

/* Ignore any movable whose insn falls within a libcall
   which is part of another movable.
   We make use of the fact that the movable for the libcall value
   was made later and so appears later on the chain.  */

static void
ignore_some_movables (movables)
     struct movable *movables;
{
  register struct movable *m, *m1;

  for (m = movables; m; m = m->next)
    {
      /* Is this a movable for the value of a libcall?  */
      rtx note = find_reg_note (m->insn, REG_RETVAL, NULL_RTX);
      if (note)
	{
	  rtx insn;
	  /* Check for earlier movables inside that range,
	     and mark them invalid.  We cannot use LUIDs here because
	     insns created by loop.c for prior loops don't have LUIDs.
	     Rather than reject all such insns from movables, we just
	     explicitly check each insn in the libcall (since invariant
	     libcalls aren't that common).  */
	  for (insn = XEXP (note, 0); insn != m->insn; insn = NEXT_INSN (insn))
	    for (m1 = movables; m1 != m; m1 = m1->next)
	      if (m1->insn == insn)
		m1->done = 1;
	}
    }
}	  

/* For each movable insn, see if the reg that it loads
   leads when it dies right into another conditionally movable insn.
   If so, record that the second insn "forces" the first one,
   since the second can be moved only if the first is.  */

static void
force_movables (movables)
     struct movable *movables;
{
  register struct movable *m, *m1;
  for (m1 = movables; m1; m1 = m1->next)
    /* Omit this if moving just the (SET (REG) 0) of a zero-extend.  */
    if (!m1->partial && !m1->done)
      {
	int regno = m1->regno;
	for (m = m1->next; m; m = m->next)
	  /* ??? Could this be a bug?  What if CSE caused the
	     register of M1 to be used after this insn?
	     Since CSE does not update regno_last_uid,
	     this insn M->insn might not be where it dies.
	     But very likely this doesn't matter; what matters is
	     that M's reg is computed from M1's reg.  */
	  if (INSN_UID (m->insn) == regno_last_uid[regno]
	      && !m->done)
	    break;
	if (m != 0 && m->set_src == m1->set_dest
	    /* If m->consec, m->set_src isn't valid.  */
	    && m->consec == 0)
	  m = 0;

	/* Increase the priority of the moving the first insn
	   since it permits the second to be moved as well.  */
	if (m != 0)
	  {
	    m->forces = m1;
	    m1->lifetime += m->lifetime;
	    m1->savings += m1->savings;
	  }
      }
}

/* Find invariant expressions that are equal and can be combined into
   one register.  */

static void
combine_movables (movables, nregs)
     struct movable *movables;
     int nregs;
{
  register struct movable *m;
  char *matched_regs = (char *) alloca (nregs);
  enum machine_mode mode;

  /* Regs that are set more than once are not allowed to match
     or be matched.  I'm no longer sure why not.  */
  /* Perhaps testing m->consec_sets would be more appropriate here?  */

  for (m = movables; m; m = m->next)
    if (m->match == 0 && n_times_used[m->regno] == 1 && !m->partial)
      {
	register struct movable *m1;
	int regno = m->regno;

	bzero (matched_regs, nregs);
	matched_regs[regno] = 1;

	for (m1 = movables; m1; m1 = m1->next)
	  if (m != m1 && m1->match == 0 && n_times_used[m1->regno] == 1
	      /* A reg used outside the loop mustn't be eliminated.  */
	      && !m1->global
	      /* A reg used for zero-extending mustn't be eliminated.  */
	      && !m1->partial
	      && (matched_regs[m1->regno]
		  || 
		  (
		   /* Can combine regs with different modes loaded from the
		      same constant only if the modes are the same or
		      if both are integer modes with M wider or the same
		      width as M1.  The check for integer is redundant, but
		      safe, since the only case of differing destination
		      modes with equal sources is when both sources are
		      VOIDmode, i.e., CONST_INT.  */
		   (GET_MODE (m->set_dest) == GET_MODE (m1->set_dest)
		    || (GET_MODE_CLASS (GET_MODE (m->set_dest)) == MODE_INT
			&& GET_MODE_CLASS (GET_MODE (m1->set_dest)) == MODE_INT
			&& (GET_MODE_BITSIZE (GET_MODE (m->set_dest))
			    >= GET_MODE_BITSIZE (GET_MODE (m1->set_dest)))))
		   /* See if the source of M1 says it matches M.  */
		   && ((GET_CODE (m1->set_src) == REG
			&& matched_regs[REGNO (m1->set_src)])
		       || rtx_equal_for_loop_p (m->set_src, m1->set_src,
						movables))))
	      && ((m->dependencies == m1->dependencies)
		  || rtx_equal_p (m->dependencies, m1->dependencies)))
	    {
	      m->lifetime += m1->lifetime;
	      m->savings += m1->savings;
	      m1->done = 1;
	      m1->match = m;
	      matched_regs[m1->regno] = 1;
	    }
      }

  /* Now combine the regs used for zero-extension.
     This can be done for those not marked `global'
     provided their lives don't overlap.  */

  for (mode = GET_CLASS_NARROWEST_MODE (MODE_INT); mode != VOIDmode;
       mode = GET_MODE_WIDER_MODE (mode))
    {
      register struct movable *m0 = 0;

      /* Combine all the registers for extension from mode MODE.
	 Don't combine any that are used outside this loop.  */
      for (m = movables; m; m = m->next)
	if (m->partial && ! m->global
	    && mode == GET_MODE (SET_SRC (PATTERN (NEXT_INSN (m->insn)))))
	  {
	    register struct movable *m1;
	    int first = uid_luid[regno_first_uid[m->regno]];
	    int last = uid_luid[regno_last_uid[m->regno]];

	    if (m0 == 0)
	      {
		/* First one: don't check for overlap, just record it.  */
		m0 = m;
		  continue;
	      }

	    /* Make sure they extend to the same mode.
	       (Almost always true.)  */
	    if (GET_MODE (m->set_dest) != GET_MODE (m0->set_dest))
		continue;

	    /* We already have one: check for overlap with those
	       already combined together.  */
	    for (m1 = movables; m1 != m; m1 = m1->next)
	      if (m1 == m0 || (m1->partial && m1->match == m0))
		if (! (uid_luid[regno_first_uid[m1->regno]] > last
		       || uid_luid[regno_last_uid[m1->regno]] < first))
		  goto overlap;

	    /* No overlap: we can combine this with the others.  */
	    m0->lifetime += m->lifetime;
	    m0->savings += m->savings;
	    m->done = 1;
	    m->match = m0;

	  overlap: ;
	  }
    }
}

/* Return 1 if regs X and Y will become the same if moved.  */

static int
regs_match_p (x, y, movables)
     rtx x, y;
     struct movable *movables;
{
  int xn = REGNO (x);
  int yn = REGNO (y);
  struct movable *mx, *my;

  for (mx = movables; mx; mx = mx->next)
    if (mx->regno == xn)
      break;

  for (my = movables; my; my = my->next)
    if (my->regno == yn)
      break;

  return (mx && my
	  && ((mx->match == my->match && mx->match != 0)
	      || mx->match == my
	      || mx == my->match));
}

/* Return 1 if X and Y are identical-looking rtx's.
   This is the Lisp function EQUAL for rtx arguments.

   If two registers are matching movables or a movable register and an
   equivalent constant, consider them equal.  */

static int
rtx_equal_for_loop_p (x, y, movables)
     rtx x, y;
     struct movable *movables;
{
  register int i;
  register int j;
  register struct movable *m;
  register enum rtx_code code;
  register char *fmt;

  if (x == y)
    return 1;
  if (x == 0 || y == 0)
    return 0;

  code = GET_CODE (x);

  /* If we have a register and a constant, they may sometimes be
     equal.  */
  if (GET_CODE (x) == REG && n_times_set[REGNO (x)] == -2
      && CONSTANT_P (y))
    for (m = movables; m; m = m->next)
      if (m->move_insn && m->regno == REGNO (x)
	  && rtx_equal_p (m->set_src, y))
	return 1;

  else if (GET_CODE (y) == REG && n_times_set[REGNO (y)] == -2
	   && CONSTANT_P (x))
    for (m = movables; m; m = m->next)
      if (m->move_insn && m->regno == REGNO (y)
	  && rtx_equal_p (m->set_src, x))
	return 1;

  /* Otherwise, rtx's of different codes cannot be equal.  */
  if (code != GET_CODE (y))
    return 0;

  /* (MULT:SI x y) and (MULT:HI x y) are NOT equivalent.
     (REG:SI x) and (REG:HI x) are NOT equivalent.  */

  if (GET_MODE (x) != GET_MODE (y))
    return 0;

  /* These three types of rtx's can be compared nonrecursively.  */
  if (code == REG)
    return (REGNO (x) == REGNO (y) || regs_match_p (x, y, movables));

  if (code == LABEL_REF)
    return XEXP (x, 0) == XEXP (y, 0);
  if (code == SYMBOL_REF)
    return XSTR (x, 0) == XSTR (y, 0);

  /* Compare the elements.  If any pair of corresponding elements
     fail to match, return 0 for the whole things.  */

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      switch (fmt[i])
	{
	case 'w':
	  if (XWINT (x, i) != XWINT (y, i))
	    return 0;
	  break;

	case 'i':
	  if (XINT (x, i) != XINT (y, i))
	    return 0;
	  break;

	case 'E':
	  /* Two vectors must have the same length.  */
	  if (XVECLEN (x, i) != XVECLEN (y, i))
	    return 0;

	  /* And the corresponding elements must match.  */
	  for (j = 0; j < XVECLEN (x, i); j++)
	    if (rtx_equal_for_loop_p (XVECEXP (x, i, j), XVECEXP (y, i, j), movables) == 0)
	      return 0;
	  break;

	case 'e':
	  if (rtx_equal_for_loop_p (XEXP (x, i), XEXP (y, i), movables) == 0)
	    return 0;
	  break;

	case 's':
	  if (strcmp (XSTR (x, i), XSTR (y, i)))
	    return 0;
	  break;

	case 'u':
	  /* These are just backpointers, so they don't matter.  */
	  break;

	case '0':
	  break;

	  /* It is believed that rtx's at this level will never
	     contain anything but integers and other rtx's,
	     except for within LABEL_REFs and SYMBOL_REFs.  */
	default:
	  abort ();
	}
    }
  return 1;
}

/* If X contains any LABEL_REF's, add REG_LABEL notes for them to all
  insns in INSNS which use thet reference.  */

static void
add_label_notes (x, insns)
     rtx x;
     rtx insns;
{
  enum rtx_code code = GET_CODE (x);
  int i, j;
  char *fmt;
  rtx insn;

  if (code == LABEL_REF && !LABEL_REF_NONLOCAL_P (x))
    {
      rtx next = next_real_insn (XEXP (x, 0));

      /* Don't record labels that refer to dispatch tables.
	 This is not necessary, since the tablejump references the same label.
	 And if we did record them, flow.c would make worse code.  */
      if (next == 0
	  || ! (GET_CODE (next) == JUMP_INSN
		&& (GET_CODE (PATTERN (next)) == ADDR_VEC
		    || GET_CODE (PATTERN (next)) == ADDR_DIFF_VEC)))
	{
	  for (insn = insns; insn; insn = NEXT_INSN (insn))
	    if (reg_mentioned_p (XEXP (x, 0), insn))
	      REG_NOTES (insn) = gen_rtx (EXPR_LIST, REG_LABEL, XEXP (x, 0),
					  REG_NOTES (insn));
	}
      return;
    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'e')
	add_label_notes (XEXP (x, i), insns);
      else if (fmt[i] == 'E')
	for (j = XVECLEN (x, i) - 1; j >= 0; j--)
	  add_label_notes (XVECEXP (x, i, j), insns);
    }
}

/* Scan MOVABLES, and move the insns that deserve to be moved.
   If two matching movables are combined, replace one reg with the
   other throughout.  */

static void
move_movables (movables, threshold, insn_count, loop_start, end, nregs)
     struct movable *movables;
     int threshold;
     int insn_count;
     rtx loop_start;
     rtx end;
     int nregs;
{
/* intel1 */
#ifdef SAVE_ON_REGS
  rtx sinsn,last_sym;
  int num_sym_adds;
#endif
  /* intel1 */
  int *insns_deleted;
  rtx new_start = 0;
  register struct movable *m;
  register rtx p;
  /* Map of pseudo-register replacements to handle combining
     when we move several insns that load the same value
     into different pseudo-registers.  */
  rtx *reg_map = (rtx *) alloca (nregs * sizeof (rtx));
  char *already_moved = (char *) alloca (nregs);

  bzero (already_moved, nregs);
  bzero ((char *) reg_map, nregs * sizeof (rtx));

  num_movables = 0;

  /* intel1 */
  num_movables = 0;

/* intel1 Any insn moved out of loop must be deleted from the log_links
          of all other insns in the loop.
*/
  if (doing_loop_after_global)
    {
        insns_deleted = (int *) alloca ((max_uid_for_loop+1) * sizeof (int));
        bzero(insns_deleted,(max_uid_for_loop+1) * sizeof (int));
    }
/* intel1 */
#ifdef SAVE_ON_REGS
  num_sym_adds=0;
  last_sym = NULL;
  for (sinsn=loop_start; flag_opt_reg_use && (sinsn!=end && num_sym_adds < 2); sinsn=NEXT_INSN (sinsn))
    {
      if (GET_CODE (sinsn) == INSN
         && GET_CODE (PATTERN (sinsn)) == SET
         && GET_CODE (SET_DEST (PATTERN (sinsn))) == REG
         && GET_CODE (SET_SRC (PATTERN (sinsn))) == PLUS
         && GET_CODE (XEXP (SET_SRC (PATTERN (sinsn)), 0)) == REG
         && GET_CODE (XEXP (SET_SRC (PATTERN (sinsn)), 1)) == SYMBOL_REF)
        {
          if (last_sym)
            {
              if (!rtx_equal_p (XEXP (SET_SRC (PATTERN (sinsn)), 1),
                              last_sym))
                {
                  num_sym_adds++;
                }
            }
          else
            {
              last_sym = XEXP (SET_SRC (PATTERN (sinsn)), 1);
            }
        }
    }
#endif /* SAVE_ON_REGS */
  for (m = movables; m; m = m->next)
    {
      /* Describe this movable insn.  */

      if (loop_dump_stream)
	{
	  fprintf (loop_dump_stream, "Insn %d: regno %d (life %d), ",
		   INSN_UID (m->insn), m->regno, m->lifetime);
	  if (m->consec > 0)
	    fprintf (loop_dump_stream, "consec %d, ", m->consec);
	  if (m->cond)
	    fprintf (loop_dump_stream, "cond ");
	  if (m->force)
	    fprintf (loop_dump_stream, "force ");
	  if (m->global)
	    fprintf (loop_dump_stream, "global ");
	  if (m->done)
	    fprintf (loop_dump_stream, "done ");
	  if (m->move_insn)
	    fprintf (loop_dump_stream, "move-insn ");
	  if (m->match)
	    fprintf (loop_dump_stream, "matches %d ",
		     INSN_UID (m->match->insn));
	  if (m->forces)
	    fprintf (loop_dump_stream, "forces %d ",
		     INSN_UID (m->forces->insn));
	}

      /* Count movables.  Value used in heuristics in strength_reduce.  */
      num_movables++;

      /* Ignore the insn if it's already done (it matched something else).
	 Otherwise, see if it is now safe to move.  */

      if (!m->done
	  && (! m->cond
	      || (1 == invariant_p (m->set_src)
		  && (m->dependencies == 0
		      || 1 == invariant_p (m->dependencies))
		  && (m->consec == 0
		      || 1 == consec_sets_invariant_p (m->set_dest,
						       m->consec + 1,
						       m->insn))))
	  && (! m->forces || m->forces->done))
	{
	  register int regno;
	  register rtx p;
	  int savings = m->savings;

	  /* We have an insn that is safe to move.
	     Compute its desirability.  */

	  p = m->insn;
	  regno = m->regno;

	  if (loop_dump_stream)
	    fprintf (loop_dump_stream, "savings %d ", savings);

	  if (moved_once[regno])
	    {
	      insn_count *= 2;

	      if (loop_dump_stream)
		fprintf (loop_dump_stream, "halved since already moved ");
	    }

	  /* An insn MUST be moved if we already moved something else
	     which is safe only if this one is moved too: that is,
	     if already_moved[REGNO] is nonzero.  */

	  /* An insn is desirable to move if the new lifetime of the
	     register is no more than THRESHOLD times the old lifetime.
	     If it's not desirable, it means the loop is so big
	     that moving won't speed things up much,
	     and it is liable to make register usage worse.  */

	  /* It is also desirable to move if it can be moved at no
	     extra cost because something else was already moved.  */
#ifdef SAVE_ON_REGS
  	  /* intel1 if the insn is regx <- regy + symbol_ref then don't move
  	     if there is more than one.  
  	  */
  	  if (flag_opt_reg_use
  	     && num_sym_adds > 1 
  	     && GET_CODE (PATTERN (m->insn)) == SET 
  	     && GET_CODE (SET_DEST (PATTERN (m->insn))) == REG 
  	     && GET_CODE (SET_SRC (PATTERN (m->insn))) == PLUS 
  	     && GET_CODE (XEXP (SET_SRC (PATTERN (m->insn)), 0)) == REG 
  	     && GET_CODE (XEXP (SET_SRC (PATTERN (m->insn)), 1)) == SYMBOL_REF)
	    {
	      savings = 0;
	    } 
#endif  	     
/* intel1 */
#ifdef STACK_REGS
#ifdef IS_STACK_MODE
          if (IS_STACK_MODE (DFmode)
             && GET_CODE (PATTERN (m->insn)) == SET 
             && GET_CODE (SET_SRC (PATTERN (m->insn))) == CONST_DOUBLE)
            savings = 0; /* not worthwhile */
#endif
#endif
	  if (already_moved[regno]
	      || flag_move_all_movables
	      || (threshold * savings * m->lifetime) >= insn_count
	      || (m->forces && m->forces->done
		  && n_times_used[m->forces->regno] == 1))
	    {
	      int count;
	      register struct movable *m1;
	      rtx first;

	      /* Now move the insns that set the reg.  */

	      if (m->partial && m->match)
		{
		  rtx newpat, i1;
		  rtx r1, r2;
		  /* Find the end of this chain of matching regs.
		     Thus, we load each reg in the chain from that one reg.
		     And that reg is loaded with 0 directly,
		     since it has ->match == 0.  */
		  for (m1 = m; m1->match; m1 = m1->match);
		  newpat = gen_move_insn (SET_DEST (PATTERN (m->insn)),
					  SET_DEST (PATTERN (m1->insn)));
		  i1 = emit_insn_before (newpat, loop_start);

		  /* Mark the moved, invariant reg as being allowed to
		     share a hard reg with the other matching invariant.  */
		  REG_NOTES (i1) = REG_NOTES (m->insn);
		  r1 = SET_DEST (PATTERN (m->insn));
		  r2 = SET_DEST (PATTERN (m1->insn));
		  regs_may_share = gen_rtx (EXPR_LIST, VOIDmode, r1,
					    gen_rtx (EXPR_LIST, VOIDmode, r2,
						     regs_may_share));
                  /* intel1 */
                  if (doing_loop_after_global 
                     && INSN_UID (m->insn) <= max_uid_for_loop)
                    insns_deleted[INSN_UID (m->insn)] = 1;

		  delete_insn (m->insn);

		  if (new_start == 0)
		    new_start = i1;

                  /* intel1 */
                  if (doing_loop_after_global)
                    {
                      RTX_IS_SPILL_P (i1) = RTX_IS_SPILL_P (m->insn);
                      SPILL_PSEUDO (i1) = SPILL_PSEUDO (m->insn);
                    }
                    
		  if (loop_dump_stream)
		    fprintf (loop_dump_stream, " moved to %d (case 1)", INSN_UID (i1));
		}
	      /* If we are to re-generate the item being moved with a
		 new move insn, first delete what we have and then emit
		 the move insn before the loop.  */
	      else if (m->move_insn)
		{
		  rtx i1, temp;

		  for (count = m->consec; count >= 0; count--)
		    {
		      /* If this is the first insn of a library call sequence,
			 skip to the end.  */
		      if (GET_CODE (p) != NOTE
			  && (temp = find_reg_note (p, REG_LIBCALL, NULL_RTX)))
			p = XEXP (temp, 0);

		      /* If this is the last insn of a libcall sequence, then
			 delete every insn in the sequence except the last.
			 The last insn is handled in the normal manner.  */
		      if (GET_CODE (p) != NOTE
			  && (temp = find_reg_note (p, REG_RETVAL, NULL_RTX)))
			{
			  temp = XEXP (temp, 0);
			  while (temp != p)
                            {
                              /* intel1 */
                              if (doing_loop_after_global
                                  && INSN_UID( temp) <= max_uid_for_loop)
                                insns_deleted[INSN_UID( temp)] = 1;

                              temp = delete_insn (temp);
                            }
                	}
                      
		      /* intel1 */
                      if (doing_loop_after_global
                          && INSN_UID( p) <= max_uid_for_loop)
                        insns_deleted[INSN_UID( p)] = 1;

		      p = delete_insn (p);
		      while (p && GET_CODE (p) == NOTE)
			p = NEXT_INSN (p);
		    }

		  start_sequence ();
		  emit_move_insn (m->set_dest, m->set_src);
		  temp = get_insns ();
		  end_sequence ();

		  add_label_notes (m->set_src, temp);

		  i1 = emit_insns_before (temp, loop_start);
		  if (! find_reg_note (i1, REG_EQUAL, NULL_RTX))
		    REG_NOTES (i1)
		      = gen_rtx (EXPR_LIST,
				 m->is_equiv ? REG_EQUIV : REG_EQUAL,
				 m->set_src, REG_NOTES (i1));

                  /* intel1 */
                  if (doing_loop_after_global)
                    {
                      RTX_IS_SPILL_P (i1) = RTX_IS_SPILL_P (m->insn);
                      SPILL_PSEUDO (i1) = SPILL_PSEUDO (m->insn);
                    }

		  if (loop_dump_stream)
		    fprintf (loop_dump_stream, " moved to %d (case 2)", INSN_UID (i1));

		  /* The more regs we move, the less we like moving them.  */
		  threshold -= 3;
		}
	      else
		{
		  for (count = m->consec; count >= 0; count--)
		    {
		      rtx i1, temp;

		      /* If first insn of libcall sequence, skip to end.  */
		      /* Do this at start of loop, since p is guaranteed to 
			 be an insn here.  */
		      if (GET_CODE (p) != NOTE
			  && (temp = find_reg_note (p, REG_LIBCALL, NULL_RTX)))
			p = XEXP (temp, 0);

		      /* If last insn of libcall sequence, move all
			 insns except the last before the loop.  The last
			 insn is handled in the normal manner.  */
		      if (GET_CODE (p) != NOTE
			  && (temp = find_reg_note (p, REG_RETVAL, NULL_RTX)))
			{
			  rtx fn_address = 0;
			  rtx fn_reg = 0;
			  rtx fn_address_insn = 0;

			  first = 0;
			  for (temp = XEXP (temp, 0); temp != p;
			       temp = NEXT_INSN (temp))
			    {
			      rtx body;
			      rtx n;
			      rtx next;

			      if (GET_CODE (temp) == NOTE)
				continue;

			      body = PATTERN (temp);

			      /* Find the next insn after TEMP,
				 not counting USE or NOTE insns.  */
			      for (next = NEXT_INSN (temp); next != p;
				   next = NEXT_INSN (next))
				if (! (GET_CODE (next) == INSN
				       && GET_CODE (PATTERN (next)) == USE)
				    && GET_CODE (next) != NOTE)
				  break;
			      
			      /* If that is the call, this may be the insn
				 that loads the function address.

				 Extract the function address from the insn
				 that loads it into a register.
				 If this insn was cse'd, we get incorrect code.

				 So emit a new move insn that copies the
				 function address into the register that the
				 call insn will use.  flow.c will delete any
				 redundant stores that we have created.  */
			      if (GET_CODE (next) == CALL_INSN
				  && GET_CODE (body) == SET
				  && GET_CODE (SET_DEST (body)) == REG
				  && (n = find_reg_note (temp, REG_EQUAL,
							 NULL_RTX)))
				{
				  fn_reg = SET_SRC (body);
				  if (GET_CODE (fn_reg) != REG)
				    fn_reg = SET_DEST (body);
				  fn_address = XEXP (n, 0);
				  fn_address_insn = temp;
				}
			      /* We have the call insn.
				 If it uses the register we suspect it might,
				 load it with the correct address directly.  */
			      if (GET_CODE (temp) == CALL_INSN
				  && fn_address != 0
				  && reg_referenced_p (fn_reg, body))
				emit_insn_after (gen_move_insn (fn_reg,
								fn_address),
						 fn_address_insn);

			      if (GET_CODE (temp) == CALL_INSN)
				{
				  i1 = emit_call_insn_before (body, loop_start);
				  /* Because the USAGE information potentially
				     contains objects other than hard registers
				     we need to copy it.  */
				  if (CALL_INSN_FUNCTION_USAGE (temp))
				    CALL_INSN_FUNCTION_USAGE (i1) =
				      copy_rtx (CALL_INSN_FUNCTION_USAGE (temp));
				}
			      else
				i1 = emit_insn_before (body, loop_start);
			      if (first == 0)
				first = i1;
			      if (temp == fn_address_insn)
				fn_address_insn = i1;
			      REG_NOTES (i1) = REG_NOTES (temp);

                              /* intel1 */
                              if (doing_loop_after_global 
                                 && INSN_UID (temp) <= max_uid_for_loop)
                                insns_deleted[INSN_UID (temp)] = 1;

			      delete_insn (temp);
			    }
			}
		      if (m->savemode != VOIDmode)
			{
			  /* P sets REG to zero; but we should clear only
			     the bits that are not covered by the mode
			     m->savemode.  */
			  rtx reg = m->set_dest;
			  rtx sequence;
			  rtx tem;
		      
			  start_sequence ();
			  tem = expand_binop
			    (GET_MODE (reg), and_optab, reg,
			     GEN_INT ((((HOST_WIDE_INT) 1
					<< GET_MODE_BITSIZE (m->savemode)))
				      - 1),
			     reg, 1, OPTAB_LIB_WIDEN);
			  if (tem == 0)
			    abort ();
			  if (tem != reg)
			    emit_move_insn (reg, tem);
			  sequence = gen_sequence ();
			  end_sequence ();
			  i1 = emit_insn_before (sequence, loop_start);
			}
		      else if (GET_CODE (p) == CALL_INSN)
			{
			  i1 = emit_call_insn_before (PATTERN (p), loop_start);
			  /* Because the USAGE information potentially
			     contains objects other than hard registers
			     we need to copy it.  */
			  if (CALL_INSN_FUNCTION_USAGE (p))
			    CALL_INSN_FUNCTION_USAGE (i1) =
			      copy_rtx (CALL_INSN_FUNCTION_USAGE (p));
			}
		      else
			i1 = emit_insn_before (PATTERN (p), loop_start);

		      REG_NOTES (i1) = REG_NOTES (p);

		      /* If there is a REG_EQUAL note present whose value is
			 not loop invariant, then delete it, since it may
			 cause problems with later optimization passes.
			 It is possible for cse to create such notes
			 like this as a result of record_jump_cond.  */
		      
		      if ((temp = find_reg_note (i1, REG_EQUAL, NULL_RTX))
			  && ! invariant_p (XEXP (temp, 0)))
			remove_note (i1, temp);

		      if (new_start == 0)
			new_start = i1;

                      /* intel1 */
                      if (doing_loop_after_global)
                        {
                          RTX_IS_SPILL_P (i1) = RTX_IS_SPILL_P (m->insn);
                          SPILL_PSEUDO (i1) = SPILL_PSEUDO (m->insn);
                        }
                        
		      if (loop_dump_stream)
			fprintf (loop_dump_stream, " moved to %d (case 3)",
				 INSN_UID (i1));

#if 0
		      /* This isn't needed because REG_NOTES is copied
			 below and is wrong since P might be a PARALLEL.  */
		      if (REG_NOTES (i1) == 0
			  && ! m->partial /* But not if it's a zero-extend clr.  */
			  && ! m->global /* and not if used outside the loop
					    (since it might get set outside).  */
			  && CONSTANT_P (SET_SRC (PATTERN (p))))
			REG_NOTES (i1)
			  = gen_rtx (EXPR_LIST, REG_EQUAL,
				     SET_SRC (PATTERN (p)), REG_NOTES (i1));
#endif

		      /* If library call, now fix the REG_NOTES that contain
			 insn pointers, namely REG_LIBCALL on FIRST
			 and REG_RETVAL on I1.  */
		      if (temp = find_reg_note (i1, REG_RETVAL, NULL_RTX))
			{
			  XEXP (temp, 0) = first;
			  temp = find_reg_note (first, REG_LIBCALL, NULL_RTX);
			  XEXP (temp, 0) = i1;
			}

                      /* intel1 */
                      if (doing_loop_after_global 
                         && INSN_UID (p) <= max_uid_for_loop)
                        insns_deleted[INSN_UID (p)] = 1;

		      delete_insn (p);
		      do p = NEXT_INSN (p);
		      while (p && GET_CODE (p) == NOTE);
		    }

		  /* The more regs we move, the less we like moving them.  */
		  threshold -= 3;
		}

	      /* Any other movable that loads the same register
		 MUST be moved.  */
	      already_moved[regno] = 1;

	      /* This reg has been moved out of one loop.  */
	      moved_once[regno] = 1;

	      /* The reg set here is now invariant.  */
	      if (! m->partial)
		n_times_set[regno] = 0;

	      m->done = 1;

	      /* Change the length-of-life info for the register
		 to say it lives at least the full length of this loop.
		 This will help guide optimizations in outer loops.  */

	      if (uid_luid[regno_first_uid[regno]] > INSN_LUID (loop_start))
		/* This is the old insn before all the moved insns.
		   We can't use the moved insn because it is out of range
		   in uid_luid.  Only the old insns have luids.  */
		regno_first_uid[regno] = INSN_UID (loop_start);
	      if (uid_luid[regno_last_uid[regno]] < INSN_LUID (end))
		regno_last_uid[regno] = INSN_UID (end);

	      /* Combine with this moved insn any other matching movables.  */

	      if (! m->partial)
		for (m1 = movables; m1; m1 = m1->next)
		  if (m1->match == m)
		    {
		      rtx temp;

		      /* Schedule the reg loaded by M1
			 for replacement so that shares the reg of M.
			 If the modes differ (only possible in restricted
			 circumstances, make a SUBREG.  */
		      if (GET_MODE (m->set_dest) == GET_MODE (m1->set_dest))
			reg_map[m1->regno] = m->set_dest;
		      else
			reg_map[m1->regno]
			  = gen_lowpart_common (GET_MODE (m1->set_dest),
						m->set_dest);
		    
		      /* Get rid of the matching insn
			 and prevent further processing of it.  */
		      m1->done = 1;

		      /* if library call, delete all insn except last, which
			 is deleted below */
		      if (temp = find_reg_note (m1->insn, REG_RETVAL,
						NULL_RTX))
			{
			  for (temp = XEXP (temp, 0); temp != m1->insn;
			       temp = NEXT_INSN (temp))
			    {
                              /* intel1 */
                              if (doing_loop_after_global 
                                 && INSN_UID (temp) <= max_uid_for_loop)
                                insns_deleted[INSN_UID (temp)] = 1;

			      delete_insn (temp);
			    }
			}

                      /* intel1 */
                      if (doing_loop_after_global 
                         && INSN_UID (m1->insn) <= max_uid_for_loop)
                        insns_deleted[INSN_UID (m1->insn)] = 1;

		      delete_insn (m1->insn);

		      /* Any other movable that loads the same register
			 MUST be moved.  */
		      already_moved[m1->regno] = 1;

		      /* The reg merged here is now invariant,
			 if the reg it matches is invariant.  */
		      if (! m->partial)
			n_times_set[m1->regno] = 0;
		    }
	    }
	  else if (loop_dump_stream)
	    fprintf (loop_dump_stream, "not desirable");
	}
      else if (loop_dump_stream && !m->match)
	fprintf (loop_dump_stream, "not safe");

      if (loop_dump_stream)
	fprintf (loop_dump_stream, "\n");
    }

  if (new_start == 0)
    new_start = loop_start;

  /* Go through all the instructions in the loop, making
     all the register substitutions scheduled in REG_MAP.  */
  for (p = new_start; p != end; p = NEXT_INSN (p))
    if (GET_CODE (p) == INSN || GET_CODE (p) == JUMP_INSN
	|| GET_CODE (p) == CALL_INSN)
      {
	replace_regs (PATTERN (p), reg_map, nregs, 0);
	replace_regs (REG_NOTES (p), reg_map, nregs, 0);
	INSN_CODE (p) = -1;
      }
  /* intel1 */
  /* remove all deleted insns from the log_links of any insn in which
     they appear */
  if (doing_loop_after_global)
    {
        int i;
        rtx rn,prev,link;

        for (rn=loop_start; rn!=end; rn=NEXT_INSN (rn))
          {
              if ((GET_CODE (rn) == INSN || GET_CODE (rn) == JUMP_INSN
	               || GET_CODE (rn) == CALL_INSN) && LOG_LINKS(rn))
                {
                    for (prev = 0, link = LOG_LINKS (rn); link;
                         prev = link, link = XEXP (link, 1))
                      {
                          if (insns_deleted[INSN_UID(XEXP (link, 0))])
	                    {
	                      if (prev)
	                        XEXP (prev, 1) = XEXP (link, 1);
	                      else
	                        LOG_LINKS (rn) = XEXP (link, 1);
	                    }
                      }

                }
          }
    }
}

#if 0
/* Scan X and replace the address of any MEM in it with ADDR.
   REG is the address that MEM should have before the replacement.  */

static void
replace_call_address (x, reg, addr)
     rtx x, reg, addr;
{
  register enum rtx_code code;
  register int i;
  register char *fmt;

  if (x == 0)
    return;
  code = GET_CODE (x);
  switch (code)
    {
    case PC:
    case CC0:
    case CONST_INT:
    case CONST_DOUBLE:
    case CONST:
    case SYMBOL_REF:
    case LABEL_REF:
    case REG:
      return;

    case SET:
      /* Short cut for very common case.  */
      replace_call_address (XEXP (x, 1), reg, addr);
      return;

    case CALL:
      /* Short cut for very common case.  */
      replace_call_address (XEXP (x, 0), reg, addr);
      return;

    case MEM:
      /* If this MEM uses a reg other than the one we expected,
	 something is wrong.  */
      if (XEXP (x, 0) != reg)
	abort ();
      XEXP (x, 0) = addr;
      return;
    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'e')
	replace_call_address (XEXP (x, i), reg, addr);
      if (fmt[i] == 'E')
	{
	  register int j;
	  for (j = 0; j < XVECLEN (x, i); j++)
	    replace_call_address (XVECEXP (x, i, j), reg, addr);
	}
    }
}
#endif

/* Return the number of memory refs to addresses that vary
   in the rtx X.  */

static int
count_nonfixed_reads (x)
     rtx x;
{
  register enum rtx_code code;
  register int i;
  register char *fmt;
  int value;

  if (x == 0)
    return 0;

  code = GET_CODE (x);
  switch (code)
    {
    case PC:
    case CC0:
    case CONST_INT:
    case CONST_DOUBLE:
    case CONST:
    case SYMBOL_REF:
    case LABEL_REF:
    case REG:
      return 0;

    case MEM:
      return ((invariant_p (XEXP (x, 0)) != 1)
	      + count_nonfixed_reads (XEXP (x, 0)));
    }

  value = 0;
  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'e')
	value += count_nonfixed_reads (XEXP (x, i));
      if (fmt[i] == 'E')
	{
	  register int j;
	  for (j = 0; j < XVECLEN (x, i); j++)
	    value += count_nonfixed_reads (XVECEXP (x, i, j));
	}
    }
  return value;
}


#if 0
/* P is an instruction that sets a register to the result of a ZERO_EXTEND.
   Replace it with an instruction to load just the low bytes
   if the machine supports such an instruction,
   and insert above LOOP_START an instruction to clear the register.  */

static void
constant_high_bytes (p, loop_start)
     rtx p, loop_start;
{
  register rtx new;
  register int insn_code_number;

  /* Try to change (SET (REG ...) (ZERO_EXTEND (..:B ...)))
     to (SET (STRICT_LOW_PART (SUBREG:B (REG...))) ...).  */

  new = gen_rtx (SET, VOIDmode,
		 gen_rtx (STRICT_LOW_PART, VOIDmode,
			  gen_rtx (SUBREG, GET_MODE (XEXP (SET_SRC (PATTERN (p)), 0)),
				   SET_DEST (PATTERN (p)),
				   0)),
		 XEXP (SET_SRC (PATTERN (p)), 0));
  insn_code_number = recog (new, p);

  if (insn_code_number)
    {
      register int i;

      /* Clear destination register before the loop.  */
      emit_insn_before (gen_rtx (SET, VOIDmode,
				 SET_DEST (PATTERN (p)),
				 const0_rtx),
			loop_start);

      /* Inside the loop, just load the low part.  */
      PATTERN (p) = new;
    }
}
#endif

/* Scan a loop setting the variables `unknown_address_altered',
   `num_mem_sets', `loop_continue', loops_enclosed', `loop_has_call',
   and `loop_has_volatile'.
   Also, fill in the array `loop_store_mems'.  */

static void
prescan_loop (start, end)
     rtx start, end;
{
  register int level = 1;
  register rtx insn;

  unknown_address_altered = 0;
  loop_has_call = 0;
  loop_has_volatile = 0;
  loop_store_mems_idx = 0;

  num_mem_sets = 0;
  loops_enclosed = 1;
  loop_continue = 0;

  for (insn = NEXT_INSN (start); insn != NEXT_INSN (end);
       insn = NEXT_INSN (insn))
    {
      if (GET_CODE (insn) == NOTE)
	{
	  if (NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_BEG)
	    {
	      ++level;
	      /* Count number of loops contained in this one.  */
	      loops_enclosed++;
	    }
	  else if (NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_END)
	    {
	      --level;
	      if (level == 0)
		{
		  end = insn;
		  break;
		}
	    }
	  else if (NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_CONT)
	    {
	      if (level == 1)
		loop_continue = insn;
	    }
	}
      else if (GET_CODE (insn) == CALL_INSN)
	{
	  unknown_address_altered = 1;
	  loop_has_call = 1;
	}
      else
	{
	  if (GET_CODE (insn) == INSN || GET_CODE (insn) == JUMP_INSN)
	    {
	      if (volatile_refs_p (PATTERN (insn)))
		loop_has_volatile = 1;

	      note_stores (PATTERN (insn), note_addr_stored);
	    }
	}
    }
}

/* Scan the function looking for loops.  Record the start and end of each loop.
   Also mark as invalid loops any loops that contain a setjmp or are branched
   to from outside the loop.  */

static void
find_and_verify_loops (f)
     rtx f;
{
  rtx insn, label;
  int current_loop = -1;
  int next_loop = -1;
  int loop;

  /* If there are jumps to undefined labels,
     treat them as jumps out of any/all loops.
     This also avoids writing past end of tables when there are no loops.  */
  uid_loop_num[0] = -1;

  /* Find boundaries of loops, mark which loops are contained within
     loops, and invalidate loops that have setjmp.  */

  for (insn = f; insn; insn = NEXT_INSN (insn))
    {
      if (GET_CODE (insn) == NOTE)
	switch (NOTE_LINE_NUMBER (insn))
	  {
	  case NOTE_INSN_LOOP_BEG:
	    loop_number_loop_starts[++next_loop] =  insn;
	    loop_number_loop_ends[next_loop] = 0;
	    loop_outer_loop[next_loop] = current_loop;
	    loop_invalid[next_loop] = 0;
	    loop_number_exit_labels[next_loop] = 0;
	    loop_number_exit_count[next_loop] = 0;
	    current_loop = next_loop;
	    break;

	  case NOTE_INSN_SETJMP:
	    /* In this case, we must invalidate our current loop and any
	       enclosing loop.  */
	    for (loop = current_loop; loop != -1; loop = loop_outer_loop[loop])
	      {
		loop_invalid[loop] = 1;
		if (loop_dump_stream)
		  fprintf (loop_dump_stream,
			   "\nLoop at %d ignored due to setjmp.\n",
			   INSN_UID (loop_number_loop_starts[loop]));
	      }
	    break;

	  case NOTE_INSN_LOOP_END:
	    if (current_loop == -1)
	      abort ();

	    loop_number_loop_ends[current_loop] = insn;
	    current_loop = loop_outer_loop[current_loop];
	    break;

	  }

      /* Note that this will mark the NOTE_INSN_LOOP_END note as being in the
	 enclosing loop, but this doesn't matter.  */
      uid_loop_num[INSN_UID (insn)] = current_loop;
    }

  /* Any loop containing a label used in an initializer must be invalidated,
     because it can be jumped into from anywhere.  */

  for (label = forced_labels; label; label = XEXP (label, 1))
    {
      int loop_num;

      for (loop_num = uid_loop_num[INSN_UID (XEXP (label, 0))];
	   loop_num != -1;
	   loop_num = loop_outer_loop[loop_num])
	loop_invalid[loop_num] = 1;
    }

  /* Any loop containing a label used for an exception handler must be
     invalidated, because it can be jumped into from anywhere.  */

  for (label = exception_handler_labels; label; label = XEXP (label, 1))
    {
      int loop_num;

      for (loop_num = uid_loop_num[INSN_UID (XEXP (label, 0))];
	   loop_num != -1;
	   loop_num = loop_outer_loop[loop_num])
	loop_invalid[loop_num] = 1;
    }


#if HAVE_movstricthi
  if (flag_sign_extension_elim)
    {
      int i;
      for (i = max_loop_num-1; i >= 0; i--)
        if (! loop_invalid[i] && loop_number_loop_ends[i])
          sign_extension_reduction (loop_number_loop_starts[i],
                                    loop_number_loop_ends [i]);
    }
#endif
  /* Now scan all insn's in the function.  If any JUMP_INSN branches into a
     loop that it is not contained within, that loop is marked invalid.
     If any INSN or CALL_INSN uses a label's address, then the loop containing
     that label is marked invalid, because it could be jumped into from
     anywhere.

     Also look for blocks of code ending in an unconditional branch that
     exits the loop.  If such a block is surrounded by a conditional 
     branch around the block, move the block elsewhere (see below) and
     invert the jump to point to the code block.  This may eliminate a
     label in our loop and will simplify processing by both us and a
     possible second cse pass.  */


  for (insn = f; insn; insn = NEXT_INSN (insn))
    if (GET_RTX_CLASS (GET_CODE (insn)) == 'i')
      {
	int this_loop_num = uid_loop_num[INSN_UID (insn)];
        
	if (GET_CODE (insn) == INSN || GET_CODE (insn) == CALL_INSN)
	  {
	    rtx note = find_reg_note (insn, REG_LABEL, NULL_RTX);
	    if (note)
	      {
		int loop_num;

		for (loop_num = uid_loop_num[INSN_UID (XEXP (note, 0))];
		     loop_num != -1;
		     loop_num = loop_outer_loop[loop_num])
		  loop_invalid[loop_num] = 1;
	      }
	  }

	if (GET_CODE (insn) != JUMP_INSN)
	  continue;

	mark_loop_jump (PATTERN (insn), this_loop_num);

	/* See if this is an unconditional branch outside the loop.  */
	if (this_loop_num != -1
	    && (GET_CODE (PATTERN (insn)) == RETURN
		|| (simplejump_p (insn)
		    && (uid_loop_num[INSN_UID (JUMP_LABEL (insn))]
			!= this_loop_num)))
	    && get_max_uid () < max_uid_for_loop)
	  {
	    rtx p;
	    rtx our_next = next_real_insn (insn);
	    int dest_loop;
	    int outer_loop = -1;

	    /* Go backwards until we reach the start of the loop, a label,
	       or a JUMP_INSN.  */
	    for (p = PREV_INSN (insn);
		 GET_CODE (p) != CODE_LABEL
		 && ! (GET_CODE (p) == NOTE
		       && NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_BEG)
		 && GET_CODE (p) != JUMP_INSN;
		 p = PREV_INSN (p))
	      ;

	    /* Check for the case where we have a jump to an inner nested
	       loop, and do not perform the optimization in that case.  */

	    if (JUMP_LABEL (insn))
	      {
		dest_loop = uid_loop_num[INSN_UID (JUMP_LABEL (insn))];
		if (dest_loop != -1)
		  {
		    for (outer_loop = dest_loop; outer_loop != -1;
			 outer_loop = loop_outer_loop[outer_loop])
		      if (outer_loop == this_loop_num)
			break;
		  }
	      }

	    /* Make sure that the target of P is within the current loop.  */

	    if (GET_CODE (p) == JUMP_INSN && JUMP_LABEL (p)
		&& uid_loop_num[INSN_UID (JUMP_LABEL (p))] != this_loop_num)
	      outer_loop = this_loop_num;

	      /* intel1 */  
            if (flag_opt_jumps_out)
              {
                /* Retarget a jump around a sequence that contains jumps out 
                   of a loop to jump out of the loop rather than jump around.  */
                while (!(GET_CODE (p) == JUMP_INSN
		        && JUMP_LABEL (p) != 0
		        && condjump_p (p)
		        && ! simplejump_p (p)
		        && next_real_insn (JUMP_LABEL (p)) == our_next) &&
		      !(GET_CODE (p) == JUMP_INSN 
		        && (JUMP_LABEL (p) == 0 
		         || (uid_loop_num[INSN_UID (JUMP_LABEL (p))]
			    == this_loop_num)))
		      && GET_CODE (p) != CODE_LABEL 
		      && !(GET_CODE (p) == NOTE
		           && NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_BEG) 
		      && !(GET_CODE (p) == NOTE
		           && NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_END))
                  p = PREV_INSN (p);
              }

	    /* If we stopped on a JUMP_INSN to the next insn after INSN,
	       we have a block of code to try to move.

	       We look backward and then forward from the target of INSN
	       to find a BARRIER at the same loop depth as the target.
	       If we find such a BARRIER, we make a new label for the start
	       of the block, invert the jump in P and point it to that label,
	       and move the block of code to the spot we found.  */

	    if (outer_loop == -1
		&& GET_CODE (p) == JUMP_INSN
		&& JUMP_LABEL (p) != 0
		/* Just ignore jumps to labels that were never emitted.
		   These always indicate compilation errors.  */
		&& INSN_UID (JUMP_LABEL (p)) != 0
		&& condjump_p (p)
		&& ! simplejump_p (p)
		&& next_real_insn (JUMP_LABEL (p)) == our_next)
	      {
		rtx target
		  = JUMP_LABEL (insn) ? JUMP_LABEL (insn) : get_last_insn ();
		int target_loop_num = uid_loop_num[INSN_UID (target)];
		rtx loc;

		for (loc = target; loc; loc = PREV_INSN (loc))
		  if (GET_CODE (loc) == BARRIER
		      && uid_loop_num[INSN_UID (loc)] == target_loop_num)
		    break;

		if (loc == 0)
		  for (loc = target; loc; loc = NEXT_INSN (loc))
		    if (GET_CODE (loc) == BARRIER
			&& uid_loop_num[INSN_UID (loc)] == target_loop_num)
		      break;

		if (loc 
		    /* intel1 Enable the optimization even if no barrier
		              is found.  This requires adding a jump
		              around the section of code moved out - but
		              this can win if jumps NOT taken are preferred.  
		    */
		    || (BRANCH_COSTS(1,1) < BRANCH_COSTS(1,0)
		        && flag_opt_jumps_out 
		        && LABEL_NUSES (JUMP_LABEL (p)) == 1))
		  {
		    rtx cond_label = JUMP_LABEL (p);
		    rtx new_label = get_label_after (p);

		    /* Ensure our label doesn't go away.  */
		    LABEL_NUSES (cond_label)++;

		    /* Verify that uid_loop_num is large enough and that
		       we can invert P.  */
		   if (invert_jump (p, new_label))
		     {
		       rtx q, r;

		       /* Include the BARRIER after INSN and copy the
			  block after LOC.  */
		       new_label = squeeze_notes (new_label, NEXT_INSN (insn));
		       /* intel1 */
		       reorder_insns (new_label, NEXT_INSN (insn), loc ? loc : PREV_INSN (target));

		       /* All those insns are now in TARGET_LOOP_NUM.  */
		       for (q = new_label; q != NEXT_INSN (NEXT_INSN (insn));
			    q = NEXT_INSN (q))
			 uid_loop_num[INSN_UID (q)] = target_loop_num;

		       /* The label jumped to by INSN is no longer a loop exit.
			  Unless INSN does not have a label (e.g., it is a
			  RETURN insn), search loop_number_exit_labels to find
			  its label_ref, and remove it.  Also turn off
			  LABEL_OUTSIDE_LOOP_P bit.  */
		       if (JUMP_LABEL (insn))
			 {
			   int loop_num;

			   for (q = 0,
				r = loop_number_exit_labels[this_loop_num];
				r; q = r, r = LABEL_NEXTREF (r))
			     if (XEXP (r, 0) == JUMP_LABEL (insn))
			       {
				 LABEL_OUTSIDE_LOOP_P (r) = 0;
				 if (q)
				   LABEL_NEXTREF (q) = LABEL_NEXTREF (r);
				 else
				   loop_number_exit_labels[this_loop_num]
				     = LABEL_NEXTREF (r);
				 break;
			       }

			   for (loop_num = this_loop_num;
				loop_num != -1 && loop_num != target_loop_num;
				loop_num = loop_outer_loop[loop_num])
			     loop_number_exit_count[loop_num]--;

			   /* If we didn't find it, then something is wrong.  */
			   if (! r)
			     abort ();
			 }

		       /* P is now a jump outside the loop, so it must be put
			  in loop_number_exit_labels, and marked as such.
			  The easiest way to do this is to just call
			  mark_loop_jump again for P.  */
        
		       mark_loop_jump (PATTERN (p), this_loop_num);
                       /* intel1 */
                       if (loc)
                         {
		           /* If INSN now jumps to the insn after it,
			      delete INSN.  */
		           if (JUMP_LABEL (insn) != 0
			       && (next_real_insn (JUMP_LABEL (insn))
			           == next_real_insn (insn)))
			     delete_insn (insn);
			 }
		       else
		         {
		           /* move the jump insn to just before the new_label */
		           reorder_insns (insn, NEXT_INSN (insn),
		                          PREV_INSN (new_label));
		         }
		     }

		    /* Continue the loop after where the conditional
		       branch used to jump, since the only branch insn
		       in the block (if it still remains) is an inter-loop
		       branch and hence needs no processing.  */
		    insn = NEXT_INSN (cond_label);

                       /* intel1 */
                    if (!loc || --LABEL_NUSES (cond_label) == 0)
		       delete_insn (cond_label);

		    /* This loop will be continued with NEXT_INSN (insn).  */
		    insn = PREV_INSN (insn);
		  }
	      }
	  }
      }
}

/* If any label in X jumps to a loop different from LOOP_NUM and any of the
   loops it is contained in, mark the target loop invalid.

   For speed, we assume that X is part of a pattern of a JUMP_INSN.  */

static void
mark_loop_jump (x, loop_num)
     rtx x;
     int loop_num;
{
  int dest_loop;
  int outer_loop;
  int i;

  switch (GET_CODE (x))
    {
    case PC:
    case USE:
    case CLOBBER:
    case REG:
    case MEM:
    case CONST_INT:
    case CONST_DOUBLE:
    case RETURN:
      return;

    case CONST:
      /* There could be a label reference in here.  */
      mark_loop_jump (XEXP (x, 0), loop_num);
      return;

    case PLUS:
    case MINUS:
    case MULT:
      mark_loop_jump (XEXP (x, 0), loop_num);
      mark_loop_jump (XEXP (x, 1), loop_num);
      return;

    case SIGN_EXTEND:
    case ZERO_EXTEND:
      mark_loop_jump (XEXP (x, 0), loop_num);
      return;

    case LABEL_REF:
      dest_loop = uid_loop_num[INSN_UID (XEXP (x, 0))];

      /* Link together all labels that branch outside the loop.  This
	 is used by final_[bg]iv_value and the loop unrolling code.  Also
	 mark this LABEL_REF so we know that this branch should predict
	 false.  */

      /* A check to make sure the label is not in an inner nested loop,
	 since this does not count as a loop exit.  */
      if (dest_loop != -1)
	{
	  for (outer_loop = dest_loop; outer_loop != -1;
	       outer_loop = loop_outer_loop[outer_loop])
	    if (outer_loop == loop_num)
	      break;
	}
      else
	outer_loop = -1;

      if (loop_num != -1 && outer_loop == -1)
	{
	  LABEL_OUTSIDE_LOOP_P (x) = 1;
	  LABEL_NEXTREF (x) = loop_number_exit_labels[loop_num];
	  loop_number_exit_labels[loop_num] = x;

	  for (outer_loop = loop_num;
	       outer_loop != -1 && outer_loop != dest_loop;
	       outer_loop = loop_outer_loop[outer_loop])
	    loop_number_exit_count[outer_loop]++;
	}

      /* If this is inside a loop, but not in the current loop or one enclosed
	 by it, it invalidates at least one loop.  */

      if (dest_loop == -1)
	return;

      /* We must invalidate every nested loop containing the target of this
	 label, except those that also contain the jump insn.  */

      for (; dest_loop != -1; dest_loop = loop_outer_loop[dest_loop])
	{
	  /* Stop when we reach a loop that also contains the jump insn.  */
	  for (outer_loop = loop_num; outer_loop != -1;
	       outer_loop = loop_outer_loop[outer_loop])
	    if (dest_loop == outer_loop)
	      return;

	  /* If we get here, we know we need to invalidate a loop.  */
	  if (loop_dump_stream && ! loop_invalid[dest_loop])
	    fprintf (loop_dump_stream,
		     "\nLoop at %d ignored due to multiple entry points.\n",
		     INSN_UID (loop_number_loop_starts[dest_loop]));
	  
	  /* intel1
	     If doing loop_spl we don't care if there are multiple jumps
	     to the loop entry label.  */
	  if (!doing_spl 
	      || NEXT_INSN (loop_number_loop_starts[dest_loop]) 
	           != XEXP (x, 0))
	    loop_invalid[dest_loop] = 1;
	}
      return;

    case SET:
      /* If this is not setting pc, ignore.  */
      if (SET_DEST (x) == pc_rtx)
	mark_loop_jump (SET_SRC (x), loop_num);
      return;

    case IF_THEN_ELSE:
      mark_loop_jump (XEXP (x, 1), loop_num);
      mark_loop_jump (XEXP (x, 2), loop_num);
      return;

    case PARALLEL:
    case ADDR_VEC:
      for (i = 0; i < XVECLEN (x, 0); i++)
	mark_loop_jump (XVECEXP (x, 0, i), loop_num);
      return;

    case ADDR_DIFF_VEC:
      for (i = 0; i < XVECLEN (x, 1); i++)
	mark_loop_jump (XVECEXP (x, 1, i), loop_num);
      return;

    default:
      /* Treat anything else (such as a symbol_ref)
	 as a branch out of this loop, but not into any loop.  */

      if (loop_num != -1)
	{
	  loop_number_exit_labels[loop_num] = x;

	  for (outer_loop = loop_num; outer_loop != -1;
	       outer_loop = loop_outer_loop[outer_loop])
	    loop_number_exit_count[outer_loop]++;
	}
      return;
    }
}

/* Return nonzero if there is a label in the range from
   insn INSN to and including the insn whose luid is END
   INSN must have an assigned luid (i.e., it must not have
   been previously created by loop.c).  */

static int
labels_in_range_p (insn, end)
     rtx insn;
     int end;
{
  while (insn && INSN_LUID (insn) <= end)
    {
      if (GET_CODE (insn) == CODE_LABEL)
	return 1;
      insn = NEXT_INSN (insn);
    }

  return 0;
}

/* Record that a memory reference X is being set.  */

static void
note_addr_stored (x)
     rtx x;
{
  register int i;

  if (x == 0 || GET_CODE (x) != MEM)
    return;

  /* Count number of memory writes.
     This affects heuristics in strength_reduce.  */
  num_mem_sets++;

  /* BLKmode MEM means all memory is clobbered.  */
  if (GET_MODE (x) == BLKmode)
    unknown_address_altered = 1;

  if (unknown_address_altered)
    return;

  for (i = 0; i < loop_store_mems_idx; i++)
    if (rtx_equal_p (XEXP (loop_store_mems[i], 0), XEXP (x, 0))
	&& MEM_IN_STRUCT_P (x) == MEM_IN_STRUCT_P (loop_store_mems[i]))
      {
	/* We are storing at the same address as previously noted.  Save the
	   wider reference.  */
	if (GET_MODE_SIZE (GET_MODE (x))
	    > GET_MODE_SIZE (GET_MODE (loop_store_mems[i])))
	  loop_store_mems[i] = x;
	break;
      }

  if (i == NUM_STORES)
    unknown_address_altered = 1;

  else if (i == loop_store_mems_idx)
    loop_store_mems[loop_store_mems_idx++] = x;
}

/* Return nonzero if the rtx X is invariant over the current loop.

   The value is 2 if we refer to something only conditionally invariant.

   If `unknown_address_altered' is nonzero, no memory ref is invariant.
   Otherwise, a memory ref is invariant if it does not conflict with
   anything stored in `loop_store_mems'.  */

int
invariant_p (x)
     register rtx x;
{
  register int i;
  register enum rtx_code code;
  register char *fmt;
  int conditional = 0;

  if (x == 0)
    return 1;
  code = GET_CODE (x);
  switch (code)
    {
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CONST:
      return 1;

    case LABEL_REF:
      /* A LABEL_REF is normally invariant, however, if we are unrolling
	 loops, and this label is inside the loop, then it isn't invariant.
	 This is because each unrolled copy of the loop body will have
	 a copy of this label.  If this was invariant, then an insn loading
	 the address of this label into a register might get moved outside
	 the loop, and then each loop body would end up using the same label.

	 We don't know the loop bounds here though, so just fail for all
	 labels.  */
      if (flag_unroll_loops)
	return 0;
      else
	return 1;

    case PC:
    case CC0:
    case UNSPEC_VOLATILE:
      return 0;

    case REG:
      /* We used to check RTX_UNCHANGING_P (x) here, but that is invalid
	 since the reg might be set by initialization within the loop.  */
      if (x == frame_pointer_rtx || x == hard_frame_pointer_rtx
	  || x == arg_pointer_rtx)
	return 1;
      if (loop_has_call
	  && REGNO (x) < FIRST_PSEUDO_REGISTER && call_used_regs[REGNO (x)])
	return 0;
      if (n_times_set[REGNO (x)] < 0)
	return 2;
      return n_times_set[REGNO (x)] == 0;

    case MEM:
      /* Volatile memory references must be rejected.  Do this before
	 checking for read-only items, so that volatile read-only items
	 will be rejected also.  */
      if (MEM_VOLATILE_P (x))
	return 0;

      /* Read-only items (such as constants in a constant pool) are
	 invariant if their address is.  */
      if (RTX_UNCHANGING_P (x))
	break;

      /* If we filled the table (or had a subroutine call), any location
	 in memory could have been clobbered.  */
      if (unknown_address_altered)
	return 0;

      /* See if there is any dependence between a store and this load.  */
      for (i = loop_store_mems_idx - 1; i >= 0; i--)
	if (true_dependence (loop_store_mems[i], x))
	  return 0;

      /* It's not invalidated by a store in memory
	 but we must still verify the address is invariant.  */
      break;

    case ASM_OPERANDS:
      /* Don't mess with insns declared volatile.  */
      if (MEM_VOLATILE_P (x))
	return 0;
    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'e')
	{
	  int tem = invariant_p (XEXP (x, i));
	  if (tem == 0)
	    return 0;
	  if (tem == 2)
	    conditional = 1;
	}
      else if (fmt[i] == 'E')
	{
	  register int j;
	  for (j = 0; j < XVECLEN (x, i); j++)
	    {
	      int tem = invariant_p (XVECEXP (x, i, j));
	      if (tem == 0)
		return 0;
	      if (tem == 2)
		conditional = 1;
	    }

	}
    }

  return 1 + conditional;
}

/* Return nonzero if all the insns in the loop that set REG
   are INSN and the immediately following insns,
   and if each of those insns sets REG in an invariant way
   (not counting uses of REG in them).

   The value is 2 if some of these insns are only conditionally invariant.

   We assume that INSN itself is the first set of REG
   and that its source is invariant.  */

static int
consec_sets_invariant_p (reg, n_sets, insn)
     int n_sets;
     rtx reg, insn;
{
  register rtx p = insn;
  register int regno = REGNO (reg);
  rtx temp;
  /* Number of sets we have to insist on finding after INSN.  */
  int count = n_sets - 1;
  int old = n_times_set[regno];
  int value = 0;
  int this;

  /* If N_SETS hit the limit, we can't rely on its value.  */
  if (n_sets == 127)
    return 0;

  n_times_set[regno] = 0;

  while (count > 0)
    {
      register enum rtx_code code;
      rtx set;

      p = NEXT_INSN (p);
      code = GET_CODE (p);

      /* If library call, skip to end of of it.  */
      if (code == INSN && (temp = find_reg_note (p, REG_LIBCALL, NULL_RTX)))
	p = XEXP (temp, 0);

      this = 0;
      if (code == INSN
	  && (set = single_set (p))
	  && GET_CODE (SET_DEST (set)) == REG
	  && REGNO (SET_DEST (set)) == regno)
	{
	  this = invariant_p (SET_SRC (set));
	  if (this != 0)
	    value |= this;
	  else if (temp = find_reg_note (p, REG_EQUAL, NULL_RTX))
	    {
	      /* If this is a libcall, then any invariant REG_EQUAL note is OK.
		 If this is an ordinary insn, then only CONSTANT_P REG_EQUAL
		 notes are OK.  */
	      this = (CONSTANT_P (XEXP (temp, 0))
		      || (find_reg_note (p, REG_RETVAL, NULL_RTX)
			  && invariant_p (XEXP (temp, 0))));
	      if (this != 0)
		value |= this;
	    }
	}
      if (this != 0)
	count--;
      else if (code != NOTE)
	{
	  n_times_set[regno] = old;
	  return 0;
	}
    }

  n_times_set[regno] = old;
  /* If invariant_p ever returned 2, we return 2.  */
  return 1 + (value & 2);
}

#if 0
/* I don't think this condition is sufficient to allow INSN
   to be moved, so we no longer test it.  */

/* Return 1 if all insns in the basic block of INSN and following INSN
   that set REG are invariant according to TABLE.  */

static int
all_sets_invariant_p (reg, insn, table)
     rtx reg, insn;
     short *table;
{
  register rtx p = insn;
  register int regno = REGNO (reg);

  while (1)
    {
      register enum rtx_code code;
      p = NEXT_INSN (p);
      code = GET_CODE (p);
      if (code == CODE_LABEL || code == JUMP_INSN)
	return 1;
      if (code == INSN && GET_CODE (PATTERN (p)) == SET
	  && GET_CODE (SET_DEST (PATTERN (p))) == REG
	  && REGNO (SET_DEST (PATTERN (p))) == regno)
	{
	  if (!invariant_p (SET_SRC (PATTERN (p)), table))
	    return 0;
	}
    }
}
#endif /* 0 */

/* Look at all uses (not sets) of registers in X.  For each, if it is
   the single use, set USAGE[REGNO] to INSN; if there was a previous use in
   a different insn, set USAGE[REGNO] to const0_rtx.  */

static void
find_single_use_in_loop (insn, x, usage)
     rtx insn;
     rtx x;
     rtx *usage;
{
  enum rtx_code code = GET_CODE (x);
  char *fmt = GET_RTX_FORMAT (code);
  int i, j;

  if (code == REG)
    usage[REGNO (x)]
      = (usage[REGNO (x)] != 0 && usage[REGNO (x)] != insn)
	? const0_rtx : insn;

  else if (code == SET)
    {
      /* Don't count SET_DEST if it is a REG; otherwise count things
	 in SET_DEST because if a register is partially modified, it won't
	 show up as a potential movable so we don't care how USAGE is set 
	 for it.  */
      if (GET_CODE (SET_DEST (x)) != REG)
	find_single_use_in_loop (insn, SET_DEST (x), usage);
      find_single_use_in_loop (insn, SET_SRC (x), usage);
    }
  else
    for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
      {
	if (fmt[i] == 'e' && XEXP (x, i) != 0)
	  find_single_use_in_loop (insn, XEXP (x, i), usage);
	else if (fmt[i] == 'E')
	  for (j = XVECLEN (x, i) - 1; j >= 0; j--)
	    find_single_use_in_loop (insn, XVECEXP (x, i, j), usage);
      }
}

/* Increment N_TIMES_SET at the index of each register
   that is modified by an insn between FROM and TO.
   If the value of an element of N_TIMES_SET becomes 127 or more,
   stop incrementing it, to avoid overflow.

   Store in SINGLE_USAGE[I] the single insn in which register I is
   used, if it is only used once.  Otherwise, it is set to 0 (for no
   uses) or const0_rtx for more than one use.  This parameter may be zero,
   in which case this processing is not done.

   Store in *COUNT_PTR the number of actual instruction
   in the loop.  We use this to decide what is worth moving out.  */

/* last_set[n] is nonzero iff reg n has been set in the current basic block.
   In that case, it is the insn that last set reg n.  */

static void
count_loop_regs_set (from, to, may_not_move, single_usage, count_ptr, nregs)
     register rtx from, to;
     char *may_not_move;
     rtx *single_usage;
     int *count_ptr;
     int nregs;
{
  register rtx *last_set = (rtx *) alloca (nregs * sizeof (rtx));
  register rtx insn;
  register int count = 0;
  register rtx dest;

  bzero ((char *) last_set, nregs * sizeof (rtx));
  for (insn = from; insn != to; insn = NEXT_INSN (insn))
    {
      if (GET_RTX_CLASS (GET_CODE (insn)) == 'i')
	{
	  ++count;

	  /* If requested, record registers that have exactly one use.  */
	  if (single_usage)
	    {
	      find_single_use_in_loop (insn, PATTERN (insn), single_usage);

	      /* Include uses in REG_EQUAL notes.  */
	      if (REG_NOTES (insn))
		find_single_use_in_loop (insn, REG_NOTES (insn), single_usage);
	    }

	  if (GET_CODE (PATTERN (insn)) == CLOBBER
	      && GET_CODE (XEXP (PATTERN (insn), 0)) == REG)
	    /* Don't move a reg that has an explicit clobber.
	       We might do so sometimes, but it's not worth the pain.  */
	    may_not_move[REGNO (XEXP (PATTERN (insn), 0))] = 1;

	  if (GET_CODE (PATTERN (insn)) == SET
	      || GET_CODE (PATTERN (insn)) == CLOBBER)
	    {
	      dest = SET_DEST (PATTERN (insn));
	      while (GET_CODE (dest) == SUBREG
		     || GET_CODE (dest) == ZERO_EXTRACT
		     || GET_CODE (dest) == SIGN_EXTRACT
		     || GET_CODE (dest) == STRICT_LOW_PART)
		dest = XEXP (dest, 0);
	      if (GET_CODE (dest) == REG)
		{
		  register int regno = REGNO (dest);
		  /* If this is the first setting of this reg
		     in current basic block, and it was set before,
		     it must be set in two basic blocks, so it cannot
		     be moved out of the loop.  */
		  if (n_times_set[regno] > 0 && last_set[regno] == 0)
		    may_not_move[regno] = 1;
		  /* If this is not first setting in current basic block,
		     see if reg was used in between previous one and this.
		     If so, neither one can be moved.  */
		  if (last_set[regno] != 0
		      && reg_used_between_p (dest, last_set[regno], insn))
		    may_not_move[regno] = 1;
		  if (n_times_set[regno] < 127)
		    ++n_times_set[regno];
		  /* intel1 */
		  if (doing_loop_after_global 
		     && HARD_REGNO_NREGS (regno, GET_MODE (dest)) > 1)
		    { /* Count each hard reg in the group.  */
		        register int i,next_regno;

		        i = HARD_REGNO_NREGS (regno, GET_MODE (dest));
		        next_regno = regno + 1;
		        while (i!=1)
		          {
		              if (n_times_set[next_regno] < 127)
		                ++n_times_set[next_regno];
		              next_regno++;
		              i--;
		          }
		    }
		  last_set[regno] = insn;
		}
	    }
	  else if (GET_CODE (PATTERN (insn)) == PARALLEL)
	    {
	      register int i;
	      for (i = XVECLEN (PATTERN (insn), 0) - 1; i >= 0; i--)
		{
		  register rtx x = XVECEXP (PATTERN (insn), 0, i);
		  if (GET_CODE (x) == CLOBBER && GET_CODE (XEXP (x, 0)) == REG)
		    /* Don't move a reg that has an explicit clobber.
		       It's not worth the pain to try to do it correctly.  */
		    may_not_move[REGNO (XEXP (x, 0))] = 1;

		  if (GET_CODE (x) == SET || GET_CODE (x) == CLOBBER)
		    {
		      dest = SET_DEST (x);
		      while (GET_CODE (dest) == SUBREG
			     || GET_CODE (dest) == ZERO_EXTRACT
			     || GET_CODE (dest) == SIGN_EXTRACT
			     || GET_CODE (dest) == STRICT_LOW_PART)
			dest = XEXP (dest, 0);
		      if (GET_CODE (dest) == REG)
			{
			  register int regno = REGNO (dest);
			  if (n_times_set[regno] > 0 && last_set[regno] == 0)
			    may_not_move[regno] = 1;
			  if (last_set[regno] != 0
			      && reg_used_between_p (dest, last_set[regno], insn))
			    may_not_move[regno] = 1;
			  if (n_times_set[regno] < 127)
			    ++n_times_set[regno];
			  last_set[regno] = insn;
			}
		    }
		}
	    }
	}

      if (GET_CODE (insn) == CODE_LABEL || GET_CODE (insn) == JUMP_INSN)
	bzero ((char *) last_set, nregs * sizeof (rtx));
    }
  *count_ptr = count;
}

/* Given a loop that is bounded by LOOP_START and LOOP_END
   and that is entered at SCAN_START,
   return 1 if the register set in SET contained in insn INSN is used by
   any insn that precedes INSN in cyclic order starting
   from the loop entry point.

   We don't want to use INSN_LUID here because if we restrict INSN to those
   that have a valid INSN_LUID, it means we cannot move an invariant out
   from an inner loop past two loops.  */

static int
loop_reg_used_before_p (set, insn, loop_start, scan_start, loop_end)
     rtx set, insn, loop_start, scan_start, loop_end;
{
  rtx reg = SET_DEST (set);
  rtx p;

  /* Scan forward checking for register usage.  If we hit INSN, we
     are done.  Otherwise, if we hit LOOP_END, wrap around to LOOP_START.  */
  for (p = scan_start; p != insn; p = NEXT_INSN (p))
    {
      if (GET_RTX_CLASS (GET_CODE (p)) == 'i'
	  && reg_overlap_mentioned_p (reg, PATTERN (p)))
	return 1;

      if (p == loop_end)
	p = loop_start;
    }

  return 0;
}

/* A "basic induction variable" or biv is a pseudo reg that is set
   (within this loop) only by incrementing or decrementing it.  */
/* A "general induction variable" or giv is a pseudo reg whose
   value is a linear function of a biv.  */

/* Bivs are recognized by `basic_induction_var';
   Givs by `general_induct_var'.  */

/* Indexed by register number, indicates whether or not register is an
   induction variable, and if so what type.  */

enum iv_mode *reg_iv_type;

/* Indexed by register number, contains pointer to `struct induction'
   if register is an induction variable.  This holds general info for
   all induction variables.  */

struct induction **reg_iv_info;

/* Indexed by register number, contains pointer to `struct iv_class'
   if register is a basic induction variable.  This holds info describing
   the class (a related group) of induction variables that the biv belongs
   to.  */

struct iv_class **reg_biv_class;

/* The head of a list which links together (via the next field)
   every iv class for the current loop.  */

struct iv_class *loop_iv_list;

/* Communication with routines called via `note_stores'.  */

static rtx note_insn;

/* Dummy register to have non-zero DEST_REG for DEST_ADDR type givs.  */

static rtx addr_placeholder;

/* ??? Unfinished optimizations, and possible future optimizations,
   for the strength reduction code.  */

/* ??? There is one more optimization you might be interested in doing: to
   allocate pseudo registers for frequently-accessed memory locations.
   If the same memory location is referenced each time around, it might
   be possible to copy it into a register before and out after.
   This is especially useful when the memory location is a variable which
   is in a stack slot because somewhere its address is taken.  If the
   loop doesn't contain a function call and the variable isn't volatile,
   it is safe to keep the value in a register for the duration of the
   loop. One tricky thing is that the copying of the value back from the
   register has to be done on all exits from the loop.  You need to check that
   all the exits from the loop go to the same place.  */

/* ??? The interaction of biv elimination, and recognition of 'constant'
   bivs, may cause problems.  */

/* ??? Add heuristics so that DEST_ADDR strength reduction does not cause
   performance problems.

   Perhaps don't eliminate things that can be combined with an addressing
   mode.  Find all givs that have the same biv, mult_val, and add_val;
   then for each giv, check to see if its only use dies in a following
   memory address.  If so, generate a new memory address and check to see
   if it is valid.   If it is valid, then store the modified memory address,
   otherwise, mark the giv as not done so that it will get its own iv.  */

/* ??? Could try to optimize branches when it is known that a biv is always
   positive.  */

/* ??? When replace a biv in a compare insn, we should replace with closest
   giv so that an optimized branch can still be recognized by the combiner,
   e.g. the VAX acb insn.  */

/* ??? Many of the checks involving uid_luid could be simplified if regscan
   was rerun in loop_optimize whenever a register was added or moved.
   Also, some of the optimizations could be a little less conservative.  */

/* Perform strength reduction and induction variable elimination.  */

/* Pseudo registers created during this function will be beyond the last
   valid index in several tables including n_times_set and regno_last_uid.
   This does not cause a problem here, because the added registers cannot be
   givs outside of their loop, and hence will never be reconsidered.
   But scan_loop must check regnos to make sure they are in bounds.  */

static void
strength_reduce (scan_start, end, loop_top, insn_count,
		 loop_start, loop_end)
     rtx scan_start;
     rtx end;
     rtx loop_top;
     int insn_count;
     rtx loop_start;
     rtx loop_end;
{
  rtx p;
  rtx set;
  rtx inc_val;
  rtx mult_val;
  rtx dest_reg;
  /* This is 1 if current insn is not executed at least once for every loop
     iteration.  */
  int not_every_iteration = 0;
  /* This is 1 if current insn may be executed more than once for every
     loop iteration.  */
  int maybe_multiple = 0;
  /* Temporary list pointers for traversing loop_iv_list.  */
  struct iv_class *bl, **backbl;
  /* Ratio of extra register life span we can justify
     for saving an instruction.  More if loop doesn't call subroutines
     since in that case saving an insn makes more difference
     and more registers are available.  */
  /* ??? could set this to last value of threshold in move_movables */
  int threshold = (loop_has_call ? 1 : 2) * (3 + n_non_fixed_regs);
  /* Map of pseudo-register replacements.  */
  rtx *reg_map;
  int call_seen;
  rtx test;
  rtx end_insert_before;
  int try_compare_elimination, done_compare_elimination; /* intel1 */
  rtx compare_insn, jump_insn, tinsn; /* intel1 */
  int *times_reg_used; /* intel1 */
  int loop_depth = 0;

  reg_iv_type = (enum iv_mode *) alloca (max_reg_before_loop
					 * sizeof (enum iv_mode *));
  bzero ((char *) reg_iv_type, max_reg_before_loop * sizeof (enum iv_mode *));
  reg_iv_info = (struct induction **)
    alloca (max_reg_before_loop * sizeof (struct induction *));
  bzero ((char *) reg_iv_info, (max_reg_before_loop
				* sizeof (struct induction *)));
  reg_biv_class = (struct iv_class **)
    alloca (max_reg_before_loop * sizeof (struct iv_class *));
  bzero ((char *) reg_biv_class, (max_reg_before_loop
				  * sizeof (struct iv_class *)));

  loop_iv_list = 0;
  addr_placeholder = gen_reg_rtx (Pmode);

  /* Save insn immediately after the loop_end.  Insns inserted after loop_end
     must be put before this insn, so that they will appear in the right
     order (i.e. loop order). 

     If loop_end is the end of the current function, then emit a 
     NOTE_INSN_DELETED after loop_end and set end_insert_before to the
     dummy note insn.  */
  if (NEXT_INSN (loop_end) != 0)
    end_insert_before = NEXT_INSN (loop_end);
  else
    end_insert_before = emit_note_after (NOTE_INSN_DELETED, loop_end);

  /* Scan through loop to find all possible bivs.  */

  p = scan_start;
  while (1)
    {
      p = NEXT_INSN (p);
      /* At end of a straight-in loop, we are done.
	 At end of a loop entered at the bottom, scan the top.  */
      if (p == scan_start)
	break;
      if (p == end)
	{
	  if (loop_top != 0)
	    p = loop_top;
	  else
	    break;
	  if (p == scan_start)
	    break;
	}

      if (GET_CODE (p) == INSN
	  && (set = single_set (p))
	  && GET_CODE (SET_DEST (set)) == REG)
	{
	  dest_reg = SET_DEST (set);
	  if (REGNO (dest_reg) < max_reg_before_loop
	      && REGNO (dest_reg) >= FIRST_PSEUDO_REGISTER
	      && reg_iv_type[REGNO (dest_reg)] != NOT_BASIC_INDUCT)
	    {
	      if (basic_induction_var (SET_SRC (set), GET_MODE (SET_SRC (set)),
				       dest_reg, p, &inc_val, &mult_val))
		{
		  /* It is a possible basic induction variable.
		     Create and initialize an induction structure for it.  */

		  struct induction *v
		    = (struct induction *) alloca (sizeof (struct induction));

		  record_biv (v, p, dest_reg, inc_val, mult_val,
			      not_every_iteration, maybe_multiple);
		  reg_iv_type[REGNO (dest_reg)] = BASIC_INDUCT;
		}
	      else if (REGNO (dest_reg) < max_reg_before_loop)
		reg_iv_type[REGNO (dest_reg)] = NOT_BASIC_INDUCT;
	    }
	}

      /* Past CODE_LABEL, we get to insns that may be executed multiple
	 times.  The only way we can be sure that they can't is if every
	 every jump insn between here and the end of the loop either
	 returns, exits the loop, is a forward jump, or is a jump
	 to the loop start.  */

      if (GET_CODE (p) == CODE_LABEL)
	{
	  rtx insn = p;

	  maybe_multiple = 0;

	  while (1)
	    {
	      insn = NEXT_INSN (insn);
	      if (insn == scan_start)
		break;
	      if (insn == end)
		{
		  if (loop_top != 0)
		    insn = loop_top;
		  else
		    break;
		  if (insn == scan_start)
		    break;
		}

	      if (GET_CODE (insn) == JUMP_INSN
		  && GET_CODE (PATTERN (insn)) != RETURN
		  && (! condjump_p (insn)
		      || (JUMP_LABEL (insn) != 0
			  && JUMP_LABEL (insn) != scan_start
			  && (INSN_UID (JUMP_LABEL (insn)) >= max_uid_for_loop
			      || INSN_UID (insn) >= max_uid_for_loop
			      || (INSN_LUID (JUMP_LABEL (insn))
				  < INSN_LUID (insn))))))
		{
		  maybe_multiple = 1;
		  break;
		}
	    }
	}

      /* Past a jump, we get to insns for which we can't count
	 on whether they will be executed during each iteration.  */
      /* This code appears twice in strength_reduce.  There is also similar
	 code in scan_loop.  */
      if (GET_CODE (p) == JUMP_INSN
	  /* If we enter the loop in the middle, and scan around to the
	     beginning, don't set not_every_iteration for that.
	     This can be any kind of jump, since we want to know if insns
	     will be executed if the loop is executed.  */
	  && ! (JUMP_LABEL (p) == loop_top
		&& ((NEXT_INSN (NEXT_INSN (p)) == loop_end && simplejump_p (p))
		    || (NEXT_INSN (p) == loop_end && condjump_p (p))))
     /*intel1*/ && !(GET_CODE (p) == JUMP_INSN
                     && jumps_back_in_loop(p,loop_start,0,0)))
	{
	  rtx label = 0;

	  /* If this is a jump outside the loop, then it also doesn't
	     matter.  Check to see if the target of this branch is on the
	     loop_number_exits_labels list.  */
	     
	  for (label = loop_number_exit_labels[uid_loop_num[INSN_UID (loop_start)]];
	       label;
	       label = LABEL_NEXTREF (label))
	    if (XEXP (label, 0) == JUMP_LABEL (p))
	      break;

	  if (! label)
	    not_every_iteration = 1;
	}

      else if (GET_CODE (p) == NOTE)
	{
	  /* At the virtual top of a converted loop, insns are again known to
	     be executed each iteration: logically, the loop begins here
	     even though the exit code has been duplicated.  */
	  if (NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_VTOP && loop_depth == 0)
	    not_every_iteration = 0;
	  else if (NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_BEG)
	    loop_depth++;
	  else if (NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_END)
	    loop_depth--;
	}

      /* Unlike in the code motion pass where MAYBE_NEVER indicates that
	 an insn may never be executed, NOT_EVERY_ITERATION indicates whether
	 or not an insn is known to be executed each iteration of the
	 loop, whether or not any iterations are known to occur.

	 Therefore, if we have just passed a label and have no more labels
	 between here and the test insn of the loop, we know these insns
	 will be executed each iteration.  */

      if (not_every_iteration && GET_CODE (p) == CODE_LABEL
	  && no_labels_between_p (p, loop_end))
	not_every_iteration = 0;
    }

  /* Scan loop_iv_list to remove all regs that proved not to be bivs.
     Make a sanity check against n_times_set.  */
  for (backbl = &loop_iv_list, bl = *backbl; bl; bl = bl->next)
    {
      if (reg_iv_type[bl->regno] != BASIC_INDUCT
	  /* Above happens if register modified by subreg, etc.  */
	  /* Make sure it is not recognized as a basic induction var: */
	  || n_times_set[bl->regno] != bl->biv_count
	  /* If never incremented, it is invariant that we decided not to
	     move.  So leave it alone.  */
	  || ! bl->incremented)
	{
	  if (loop_dump_stream)
	    fprintf (loop_dump_stream, "Reg %d: biv discarded, %s\n",
		     bl->regno,
		     (reg_iv_type[bl->regno] != BASIC_INDUCT
		      ? "not induction variable"
		      : (! bl->incremented ? "never incremented"
			 : "count error")));
	  
	  reg_iv_type[bl->regno] = NOT_BASIC_INDUCT;
	  *backbl = bl->next;
	}
      else
	{
	  backbl = &bl->next;

	  if (loop_dump_stream)
	    fprintf (loop_dump_stream, "Reg %d: biv verified\n", bl->regno);
	}
    }

  /* Exit if there are no bivs.  */
  if (! loop_iv_list)
    {
      /* Can still unroll the loop anyways, but indicate that there is no
	 strength reduction info available.  */
      if (flag_unroll_loops)
	unroll_loop (loop_end, insn_count, loop_start, end_insert_before, 0);

      return;
    }

  /* Find initial value for each biv by searching backwards from loop_start,
     halting at first label.  Also record any test condition.  */

  call_seen = 0;
  for (p = loop_start; p && GET_CODE (p) != CODE_LABEL; p = PREV_INSN (p))
    {
      note_insn = p;

      if (GET_CODE (p) == CALL_INSN)
	call_seen = 1;

      if (GET_CODE (p) == INSN || GET_CODE (p) == JUMP_INSN
	  || GET_CODE (p) == CALL_INSN)
	note_stores (PATTERN (p), record_initial);

      /* Record any test of a biv that branches around the loop if no store
	 between it and the start of loop.  We only care about tests with
	 constants and registers and only certain of those.  */
      if (GET_CODE (p) == JUMP_INSN
	  && JUMP_LABEL (p) != 0
	  && next_real_insn (JUMP_LABEL (p)) == next_real_insn (loop_end)
	  && (test = get_condition_for_loop (p)) != 0
	  && GET_CODE (XEXP (test, 0)) == REG
	  && REGNO (XEXP (test, 0)) < max_reg_before_loop
	  && (bl = reg_biv_class[REGNO (XEXP (test, 0))]) != 0
	  && valid_initial_value_p (XEXP (test, 1), p, call_seen, loop_start)
	  && bl->init_insn == 0)
	{
	  /* If an NE test, we have an initial value!  */
	  if (GET_CODE (test) == NE)
	    {
	      bl->init_insn = p;
	      bl->init_set = gen_rtx (SET, VOIDmode,
				      XEXP (test, 0), XEXP (test, 1));
	    }
	  else
	    bl->initial_test = test;
	}
    }

  /* Now see if it is worthwhile to do compare elimination, remember it only
     saves one instruction.  intel1
  */
  try_compare_elimination = 0;
  if (flag_compare_elim 
     && loop_number_exit_labels[uid_loop_num[INSN_UID (loop_start)]] == NULL)
    {
      possible_compare_elimination(loop_start, loop_end, &compare_insn, 
                                   &jump_insn, &try_compare_elimination);
      done_compare_elimination = 0;
    }

  /* Look at the each biv and see if we can say anything better about its
     initial value from any initializing insns set up above.  (This is done
     in two passes to avoid missing SETs in a PARALLEL.)  */
  for (bl = loop_iv_list; bl; bl = bl->next)
    {
      rtx src;

      if (! bl->init_insn)
	continue;

      src = SET_SRC (bl->init_set);

      if (loop_dump_stream)
	fprintf (loop_dump_stream,
		 "Biv %d initialized at insn %d: initial value ",
		 bl->regno, INSN_UID (bl->init_insn));

      if ((GET_MODE (src) == GET_MODE (regno_reg_rtx[bl->regno])
	   || GET_MODE (src) == VOIDmode)
	  && valid_initial_value_p (src, bl->init_insn, call_seen, loop_start))
	{
	  /* intel1 */
	  if (flag_compare_elim
	     && try_compare_elimination && !done_compare_elimination 
	     && bl->biv_count == 1 
	     && bl->biv->mult_val == const1_rtx 
	     && REGNO (XEXP (SET_SRC (PATTERN (compare_insn)), 0)) == bl->regno 
	     && (done_compare_elimination = compare_elimination (bl, loop_start, loop_end, 
								 compare_insn, jump_insn, 0)))
	    {
    	      rtx tem = find_reg_note (bl->init_insn, REG_EQUAL, 0);
	      if (tem)
		XEXP (tem, 0) = SET_SRC (bl->init_set);
	      src = SET_SRC (bl->init_set);
	    }
	  bl->initial_value = src;

	  if (loop_dump_stream)
	    {
	      if (GET_CODE (src) == CONST_INT)
		fprintf (loop_dump_stream, "%d\n", INTVAL (src));
	      else
		{
		  print_rtl (loop_dump_stream, src);
		  fprintf (loop_dump_stream, "\n");
		}
	    }
	}
      else
	{
	  /* Biv initial value is not simple move,
	     so let it keep initial value of "itself".  */

	  if (loop_dump_stream)
	    fprintf (loop_dump_stream, "is complex\n");
	}
    }

  /* Search the loop for general induction variables.  */

  /* A register is a giv if: it is only set once, it is a function of a
     biv and a constant (or invariant), and it is not a biv.  */

  not_every_iteration = 0;
  loop_depth = 0;
  p = scan_start;
  while (1)
    {
      p = NEXT_INSN (p);
      /* At end of a straight-in loop, we are done.
	 At end of a loop entered at the bottom, scan the top.  */
      if (p == scan_start)
	break;
      if (p == end)
	{
	  if (loop_top != 0)
	    p = loop_top;
	  else
	    break;
	  if (p == scan_start)
	    break;
	}

      /* Look for a general induction variable in a register.  */
      if (GET_CODE (p) == INSN
	  && (set = single_set (p))
	  && GET_CODE (SET_DEST (set)) == REG
	  && ! may_not_optimize[REGNO (SET_DEST (set))])
	{
	  rtx src_reg;
	  rtx add_val;
	  rtx mult_val;
	  int benefit;
	  rtx regnote = 0;

	  dest_reg = SET_DEST (set);
	  if (REGNO (dest_reg) < FIRST_PSEUDO_REGISTER)
	    continue;

	  if (/* SET_SRC is a giv.  */
	      ((benefit = general_induction_var (SET_SRC (set),
						 &src_reg, &add_val,
						 &mult_val))
	       /* Equivalent expression is a giv.  */
	       || ((regnote = find_reg_note (p, REG_EQUAL, NULL_RTX))
		   && (benefit = general_induction_var (XEXP (regnote, 0),
							&src_reg,
							&add_val, &mult_val))))
	      /* Don't try to handle any regs made by loop optimization.
		 We have nothing on them in regno_first_uid, etc.  */
	      && REGNO (dest_reg) < max_reg_before_loop
	      /* Don't recognize a BASIC_INDUCT_VAR here.  */
	      && dest_reg != src_reg
	      /* This must be the only place where the register is set.  */
	      && (n_times_set[REGNO (dest_reg)] == 1
		  /* or all sets must be consecutive and make a giv.  */
		  || (benefit = consec_sets_giv (benefit, p,
						 src_reg, dest_reg,
						 &add_val, &mult_val))))
	    {
	      int count;
	      struct induction *v
		= (struct induction *) alloca (sizeof (struct induction));
	      rtx temp;

	      /* If this is a library call, increase benefit.  */
	      if (find_reg_note (p, REG_RETVAL, NULL_RTX))
		benefit += libcall_benefit (p);

	      /* Skip the consecutive insns, if there are any.  */
	      for (count = n_times_set[REGNO (dest_reg)] - 1;
		   count > 0; count--)
		{
		  /* If first insn of libcall sequence, skip to end.
		     Do this at start of loop, since INSN is guaranteed to
		     be an insn here.  */
		  if (GET_CODE (p) != NOTE
		      && (temp = find_reg_note (p, REG_LIBCALL, NULL_RTX)))
		    p = XEXP (temp, 0);

		  do p = NEXT_INSN (p);
		  while (GET_CODE (p) == NOTE);
		}

	      record_giv (v, p, src_reg, dest_reg, mult_val, add_val, benefit,
			  DEST_REG, not_every_iteration, NULL_PTR, loop_start,
			  loop_end);

	    }
	}

#ifndef DONT_REDUCE_ADDR
      /* Look for givs which are memory addresses.  */
      /* This resulted in worse code on a VAX 8600.  I wonder if it
	 still does.  */
      if (GET_CODE (p) == INSN)
	find_mem_givs (PATTERN (p), p, not_every_iteration, loop_start,
		       loop_end);
#endif

      /* Update the status of whether giv can derive other givs.  This can
	 change when we pass a label or an insn that updates a biv.  */
      if (GET_CODE (p) == INSN || GET_CODE (p) == JUMP_INSN
	|| GET_CODE (p) == CODE_LABEL)
	update_giv_derive (p);

      /* Past a jump, we get to insns for which we can't count
	 on whether they will be executed during each iteration.  */
      /* This code appears twice in strength_reduce.  There is also similar
	 code in scan_loop.  */
      if (GET_CODE (p) == JUMP_INSN
	  /* If we enter the loop in the middle, and scan around to the
	     beginning, don't set not_every_iteration for that.
	     This can be any kind of jump, since we want to know if insns
	     will be executed if the loop is executed.  */
	  && ! (JUMP_LABEL (p) == loop_top
		&& ((NEXT_INSN (NEXT_INSN (p)) == loop_end && simplejump_p (p))
		    || (NEXT_INSN (p) == loop_end && condjump_p (p))))
     /*intel1*/ && !(GET_CODE (p) == JUMP_INSN
                     && jumps_back_in_loop(p,loop_start,0,0)))
	{
	  rtx label = 0;

	  /* If this is a jump outside the loop, then it also doesn't
	     matter.  Check to see if the target of this branch is on the
	     loop_number_exits_labels list.  */
	     
	  for (label = loop_number_exit_labels[uid_loop_num[INSN_UID (loop_start)]];
	       label;
	       label = LABEL_NEXTREF (label))
	    if (XEXP (label, 0) == JUMP_LABEL (p))
	      break;

	  if (! label)
	    not_every_iteration = 1;
	}

      else if (GET_CODE (p) == NOTE)
	{
	  /* At the virtual top of a converted loop, insns are again known to
	     be executed each iteration: logically, the loop begins here
	     even though the exit code has been duplicated.  */
	  if (NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_VTOP && loop_depth == 0)
	    not_every_iteration = 0;
	  else if (NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_BEG)
	    loop_depth++;
	  else if (NOTE_LINE_NUMBER (p) == NOTE_INSN_LOOP_END)
	    loop_depth--;
	}

      /* Unlike in the code motion pass where MAYBE_NEVER indicates that
	 an insn may never be executed, NOT_EVERY_ITERATION indicates whether
	 or not an insn is known to be executed each iteration of the
	 loop, whether or not any iterations are known to occur.

	 Therefore, if we have just passed a label and have no more labels
	 between here and the test insn of the loop, we know these insns
	 will be executed each iteration.  */

      if (not_every_iteration && GET_CODE (p) == CODE_LABEL
	  && no_labels_between_p (p, loop_end))
	not_every_iteration = 0;
    }

  /* Try to calculate and save the number of loop iterations.  This is
     set to zero if the actual number can not be calculated.  This must
     be called after all giv's have been identified, since otherwise it may
     fail if the iteration variable is a giv.  */

  loop_n_iterations = loop_iterations (loop_start, loop_end);

  /* Now for each giv for which we still don't know whether or not it is
     replaceable, check to see if it is replaceable because its final value
     can be calculated.  This must be done after loop_iterations is called,
     so that final_giv_value will work correctly.  */

  for (bl = loop_iv_list; bl; bl = bl->next)
    {
      struct induction *v;

      for (v = bl->giv; v; v = v->next_iv)
	if (! v->replaceable && ! v->not_replaceable)
	  check_final_value (v, loop_start, loop_end);
    }

  /* Try to prove that the loop counter variable (if any) is always
     nonnegative; if so, record that fact with a REG_NONNEG note
     so that "decrement and branch until zero" insn can be used.  */
  check_dbra_loop (loop_end, insn_count, loop_start);

  /* Create reg_map to hold substitutions for replaceable giv regs.  */
  reg_map = (rtx *) alloca (max_reg_before_loop * sizeof (rtx));
  bzero ((char *) reg_map, max_reg_before_loop * sizeof (rtx));

  /* Examine each iv class for feasibility of strength reduction/induction
     variable elimination.  */

  for (bl = loop_iv_list; bl; bl = bl->next)
    {
      struct induction *v;
      /* intel1 */
#ifdef SAVE_ON_REGS
      int num_add_vals_live_outside =0;
      int no_mem_givs = 0;
      int contains_loop =0;
#endif
      int benefit;
      int all_reduced;
      rtx final_value = 0;

      /* Test whether it will be possible to eliminate this biv
	 provided all givs are reduced.  This is possible if either
	 the reg is not used outside the loop, or we can compute
	 what its final value will be.

	 For architectures with a decrement_and_branch_until_zero insn,
	 don't do this if we put a REG_NONNEG note on the endtest for
	 this biv.  */

      /* Compare against bl->init_insn rather than loop_start.
	 We aren't concerned with any uses of the biv between
	 init_insn and loop_start since these won't be affected
	 by the value of the biv elsewhere in the function, so
	 long as init_insn doesn't use the biv itself.
	 March 14, 1989 -- self@bayes.arc.nasa.gov */

      if ((uid_luid[regno_last_uid[bl->regno]] < INSN_LUID (loop_end)
	   && bl->init_insn
	   && INSN_UID (bl->init_insn) < max_uid_for_loop
	   && uid_luid[regno_first_uid[bl->regno]] >= INSN_LUID (bl->init_insn)
#ifdef HAVE_decrement_and_branch_until_zero
	   && ! bl->nonneg
#endif
	   && ! reg_mentioned_p (bl->biv->dest_reg, SET_SRC (bl->init_set)))
	  || ((final_value = final_biv_value (bl, loop_start, loop_end))
#ifdef HAVE_decrement_and_branch_until_zero
	      && ! bl->nonneg
#endif
	      ))
	bl->eliminable = maybe_eliminate_biv (bl, loop_start, end, 0,
					      threshold, insn_count);
      else
	{
	  if (loop_dump_stream)
	    {
	      fprintf (loop_dump_stream,
		       "Cannot eliminate biv %d.\n",
		       bl->regno);
	      fprintf (loop_dump_stream,
		       "First use: insn %d, last use: insn %d.\n",
		       regno_first_uid[bl->regno],
		       regno_last_uid[bl->regno]);
	    }
	}

      /* Combine all giv's for this iv_class.  */
      combine_givs (bl);

      /* This will be true at the end, if all givs which depend on this
	 biv have been strength reduced.
	 We can't (currently) eliminate the biv unless this is so.  */
      all_reduced = 1;

      /* intel1 */
#ifdef SAVE_ON_REGS
      /* Only reduce DEST_ADDR givs of this biv if they will not increase
         register pressure.  
      */
      if (!flag_all_mem_givs && flag_opt_reg_use)
        {
          int num_mult_val_saved = 0;
          int num_givs_to_update = 0;
          num_add_vals_live_outside = 0;
          no_mem_givs = 0;
          contains_loop = loop_contains_loop (loop_start, end);
        
          for (v = bl->giv; v; v = v->next_iv)
            {
              if (v->giv_type == DEST_ADDR)
                num_mult_val_saved++;
              if (v->giv_type == DEST_ADDR && !v->same)
                {
                  num_givs_to_update++;
                  if (v->add_val && GET_CODE (v->add_val) == REG 
                     && REGNO (v->add_val) >= FIRST_PSEUDO_REGISTER 
                     && (contains_loop 
                         || REGNO (v->add_val)<max_reg_before_loop))
                    {
                      if (loop_dump_stream 
                         && reg_live_outside_loop[REGNO (v->add_val)])
                        {
                          fprintf (loop_dump_stream, 
                          "dest addr giv at %d has add_val live outside reg%d\n", 
                          INSN_UID (v->insn), REGNO (v->add_val));
                        }
                      if (contains_loop)
                        num_add_vals_live_outside += 2;
                      else
                        {
                          num_add_vals_live_outside += 
                            reg_live_outside_loop[REGNO (v->add_val)];
                        }
                    }
                  else if (v->add_val && GET_CODE (v->add_val) == CONST_INT 
                           && v->mult_val == const1_rtx)
                    { /* Disable this reduction. */
                      num_add_vals_live_outside = (1 << ((sizeof (int)*8)-4));
                    }
                }
            }
#ifdef MAX_DEST_ADDR_GIVS
        if (num_givs_to_update > MAX_DEST_ADDR_GIVS)
          {
            no_mem_givs = 1;
          }
#endif
        /* Heuristic here:
           a) If the biv is eliminated that saves one register in the loop.  
           b) Each giv_to_update must be updated at the end of the loop.  
           c) Each add_val_live_outside may cause a spill somewhere.  
        */
        if (loop_dump_stream)
          {
            fprintf (loop_dump_stream,
            "eliminable %d num_updates %d mults_saved %d adds_outside %d heu %d\n",
            bl->eliminable, num_givs_to_update, num_mult_val_saved,
            num_add_vals_live_outside,
            (((int)(bl->eliminable) * -3) 
             + (num_givs_to_update+((int)(bl->eliminable) * -1)) 
             + (num_mult_val_saved*-1)
             + (num_add_vals_live_outside*3) 
            ));
          }
        if (no_mem_givs 
            ||  
            (((int)(bl->eliminable) * -3) 
             + (num_givs_to_update+((int)(bl->eliminable) * -1)) 
             + (num_mult_val_saved*-1)
             + (num_add_vals_live_outside*3) > 0
            )
           )
          {
            all_reduced = 0;
            for (v = bl->giv; v; v = v->next_iv)
              {
                if (v->giv_type == DEST_ADDR)
                  {
                    v->ignore = 1;
                  }
              }
          }
      }
#endif /* SAVE_ON_REGS */

      /* Check each giv in this class to see if we will benefit by reducing
	 it.  Skip giv's combined with others.  */
      for (v = bl->giv; v; v = v->next_iv)
	{
	  struct induction *tv;

	  if (v->ignore || v->same)
	    continue;

	  benefit = v->benefit;

	  /* Reduce benefit if not replaceable, since we will insert
	     a move-insn to replace the insn that calculates this giv.
	     Don't do this unless the giv is a user variable, since it
	     will often be marked non-replaceable because of the duplication
	     of the exit code outside the loop.  In such a case, the copies
	     we insert are dead and will be deleted.  So they don't have
	     a cost.  Similar situations exist.  */
	  /* ??? The new final_[bg]iv_value code does a much better job
	     of finding replaceable giv's, and hence this code may no longer
	     be necessary.  */
	  if (! v->replaceable && ! bl->eliminable
	      && REG_USERVAR_P (v->dest_reg))
	    benefit -= copy_cost;

	  /* Decrease the benefit to count the add-insns that we will
	     insert to increment the reduced reg for the giv.  */
	  benefit -= add_cost * bl->biv_count;

	  /* Decide whether to strength-reduce this giv or to leave the code
	     unchanged (recompute it from the biv each time it is used).
	     This decision can be made independently for each giv.  */

	  /* ??? Perhaps attempt to guess whether autoincrement will handle
	     some of the new add insns; if so, can increase BENEFIT
	     (undo the subtraction of add_cost that was done above).  */

	  /* If an insn is not to be strength reduced, then set its ignore
	     flag, and clear all_reduced.  */

	  /* A giv that depends on a reversed biv must be reduced if it is
	     used after the loop exit, otherwise, it would have the wrong
	     value after the loop exit.  To make it simple, just reduce all
	     of such giv's whether or not we know they are used after the loop
	     exit.  */

	  if ( ! flag_reduce_all_givs && v->lifetime * threshold * benefit < insn_count
	      && ! bl->reversed )
	    {
	      if (loop_dump_stream)
		fprintf (loop_dump_stream,
			 "giv of insn %d not worth while, %d vs %d.\n",
			 INSN_UID (v->insn),
			 v->lifetime * threshold * benefit, insn_count);
	      v->ignore = 1;
	      all_reduced = 0;
	    }
	  else
	    {
	      /* Check that we can increment the reduced giv without a
		 multiply insn.  If not, reject it.  */

	      for (tv = bl->biv; tv; tv = tv->next_iv)
		if (tv->mult_val == const1_rtx
		    && ! product_cheap_p (tv->add_val, v->mult_val))
		  {
		    if (loop_dump_stream)
		      fprintf (loop_dump_stream,
			       "giv of insn %d: would need a multiply.\n",
			       INSN_UID (v->insn));
		    v->ignore = 1;
		    all_reduced = 0;
		    break;
		  }
	    }
	}

      /* Reduce each giv that we decided to reduce.  */

      for (v = bl->giv; v; v = v->next_iv)
	{
	  struct induction *tv;
	  if (! v->ignore && v->same == 0)
	    {
	      int auto_inc_opt = 0;

	      v->new_reg = gen_reg_rtx (v->mode);

#ifdef AUTO_INC_DEC
	      /* If the target has auto-increment addressing modes, and
		 this is an address giv, then try to put the increment
		 immediately after its use, so that flow can create an
		 auto-increment addressing mode.  */
	      if (v->giv_type == DEST_ADDR && bl->biv_count == 1
		  && bl->biv->always_executed && ! bl->biv->maybe_multiple
		  /* We don't handle reversed biv's because bl->biv->insn
		     does not have a valid INSN_LUID.  */
		  && ! bl->reversed
		  && v->always_executed && ! v->maybe_multiple)
		{
		  /* If other giv's have been combined with this one, then
		     this will work only if all uses of the other giv's occur
		     before this giv's insn.  This is difficult to check.

		     We simplify this by looking for the common case where
		     there is one DEST_REG giv, and this giv's insn is the
		     last use of the dest_reg of that DEST_REG giv.  If the
		     the increment occurs after the address giv, then we can
		     perform the optimization.  (Otherwise, the increment
		     would have to go before other_giv, and we would not be
		     able to combine it with the address giv to get an
		     auto-inc address.)  */
		  if (v->combined_with)
		    {
		      struct induction *other_giv = 0;

		      for (tv = bl->giv; tv; tv = tv->next_iv)
			if (tv->same == v)
			  {
			    if (other_giv)
			      break;
			    else
			      other_giv = tv;
			  }
		      if (! tv && other_giv
			  && (regno_last_uid[REGNO (other_giv->dest_reg)]
			      == INSN_UID (v->insn))
			  && INSN_LUID (v->insn) < INSN_LUID (bl->biv->insn))
			auto_inc_opt = 1;
		    }
		  /* Check for case where increment is before the the address
		     giv.  */
		  else if (INSN_LUID (v->insn) > INSN_LUID (bl->biv->insn))
		    auto_inc_opt = -1;
		  else
		    auto_inc_opt = 1;

#ifdef HAVE_cc0
		  {
		    rtx prev;

		    /* We can't put an insn immediately after one setting
		       cc0, or immediately before one using cc0.  */
		    if ((auto_inc_opt == 1 && sets_cc0_p (PATTERN (v->insn)))
			|| (auto_inc_opt == -1
			    && (prev = prev_nonnote_insn (v->insn)) != 0
			    && GET_RTX_CLASS (GET_CODE (prev)) == 'i'
			    && sets_cc0_p (PATTERN (prev))))
		      auto_inc_opt = 0;
		  }
#endif

		  if (auto_inc_opt)
		    v->auto_inc_opt = 1;
		}
#endif

	      /* For each place where the biv is incremented, add an insn
		 to increment the new, reduced reg for the giv.  */
	      for (tv = bl->biv; tv; tv = tv->next_iv)
		{
		  rtx insert_before;

		  if (! auto_inc_opt)
		    insert_before = tv->insn;
		  else if (auto_inc_opt == 1)
		    insert_before = NEXT_INSN (v->insn);
		  else
		    insert_before = v->insn;

		  if (tv->mult_val == const1_rtx)
		    emit_iv_add_mult (tv->add_val, v->mult_val,
				      v->new_reg, v->new_reg, insert_before);
		  else /* tv->mult_val == const0_rtx */
		    /* A multiply is acceptable here
		       since this is presumed to be seldom executed.  */
		    emit_iv_add_mult (tv->add_val, v->mult_val,
				      v->add_val, v->new_reg, insert_before);
		}

	      /* Add code at loop start to initialize giv's reduced reg.  */

	      emit_iv_add_mult (bl->initial_value, v->mult_val,
				v->add_val, v->new_reg, loop_start);
	    }
	}

      /* Rescan all givs.  If a giv is the same as a giv not reduced, mark it
	 as not reduced.
	 
	 For each giv register that can be reduced now: if replaceable,
	 substitute reduced reg wherever the old giv occurs;
	 else add new move insn "giv_reg = reduced_reg".

	 Also check for givs whose first use is their definition and whose
	 last use is the definition of another giv.  If so, it is likely
	 dead and should not be used to eliminate a biv.  */
      for (v = bl->giv; v; v = v->next_iv)
	{
	  if (v->same && v->same->ignore)
	    v->ignore = 1;

	  if (v->ignore)
	    continue;

	  if (v->giv_type == DEST_REG
	      && regno_first_uid[REGNO (v->dest_reg)] == INSN_UID (v->insn))
	    {
	      struct induction *v1;

	      for (v1 = bl->giv; v1; v1 = v1->next_iv)
		if (regno_last_uid[REGNO (v->dest_reg)] == INSN_UID (v1->insn))
		  v->maybe_dead = 1;
	    }

	  /* Update expression if this was combined, in case other giv was
	     replaced.  */
	  if (v->same)
	    v->new_reg = replace_rtx (v->new_reg,
				      v->same->dest_reg, v->same->new_reg);

	  if (v->giv_type == DEST_ADDR)
	    /* Store reduced reg as the address in the memref where we found
	       this giv.  */
	    validate_change (v->insn, v->location, v->new_reg, 0);
	  else if (v->replaceable)
	    {
	      reg_map[REGNO (v->dest_reg)] = v->new_reg;

#if 0
	      /* I can no longer duplicate the original problem.  Perhaps
		 this is unnecessary now?  */

	      /* Replaceable; it isn't strictly necessary to delete the old
		 insn and emit a new one, because v->dest_reg is now dead.

		 However, especially when unrolling loops, the special
		 handling for (set REG0 REG1) in the second cse pass may
		 make v->dest_reg live again.  To avoid this problem, emit
		 an insn to set the original giv reg from the reduced giv.
		 We can not delete the original insn, since it may be part
		 of a LIBCALL, and the code in flow that eliminates dead
		 libcalls will fail if it is deleted.  */
	      emit_insn_after (gen_move_insn (v->dest_reg, v->new_reg),
			       v->insn);
#endif
	    }
	  else
	    {
	      /* Not replaceable; emit an insn to set the original giv reg from
		 the reduced giv, same as above.  */
	      emit_insn_after (gen_move_insn (v->dest_reg, v->new_reg),
			       v->insn);
	    }

	  /* When a loop is reversed, givs which depend on the reversed
	     biv, and which are live outside the loop, must be set to their
	     correct final value.  This insn is only needed if the giv is
	     not replaceable.  The correct final value is the same as the
	     value that the giv starts the reversed loop with.  */
	  if (bl->reversed && ! v->replaceable)
	    emit_iv_add_mult (bl->initial_value, v->mult_val,
			      v->add_val, v->dest_reg, end_insert_before);
	  else if (v->final_value)
	    {
	      rtx insert_before;

	      /* If the loop has multiple exits, emit the insn before the
		 loop to ensure that it will always be executed no matter
		 how the loop exits.  Otherwise, emit the insn after the loop,
		 since this is slightly more efficient.  */
	      if (loop_number_exit_count[uid_loop_num[INSN_UID (loop_start)]])
		insert_before = loop_start;
	      else
		insert_before = end_insert_before;
	      emit_insn_before (gen_move_insn (v->dest_reg, v->final_value),
				insert_before);

#if 0
	      /* If the insn to set the final value of the giv was emitted
		 before the loop, then we must delete the insn inside the loop
		 that sets it.  If this is a LIBCALL, then we must delete
		 every insn in the libcall.  Note, however, that
		 final_giv_value will only succeed when there are multiple
		 exits if the giv is dead at each exit, hence it does not
		 matter that the original insn remains because it is dead
		 anyways.  */
	      /* Delete the insn inside the loop that sets the giv since
		 the giv is now set before (or after) the loop.  */
	      delete_insn (v->insn);
#endif
	    }

	  if (loop_dump_stream)
	    {
	      fprintf (loop_dump_stream, "giv at %d reduced to ",
		       INSN_UID (v->insn));
	      print_rtl (loop_dump_stream, v->new_reg);
	      fprintf (loop_dump_stream, "\n");
	    }
	}

      /* All the givs based on the biv bl have been reduced if they
	 merit it.  */

      /* For each giv not marked as maybe dead that has been combined with a
	 second giv, clear any "maybe dead" mark on that second giv.
	 v->new_reg will either be or refer to the register of the giv it
	 combined with.

	 Doing this clearing avoids problems in biv elimination where a
	 giv's new_reg is a complex value that can't be put in the insn but
	 the giv combined with (with a reg as new_reg) is marked maybe_dead.
	 Since the register will be used in either case, we'd prefer it be
	 used from the simpler giv.  */

      for (v = bl->giv; v; v = v->next_iv)
	if (! v->maybe_dead && v->same)
	  v->same->maybe_dead = 0;

      /* Try to eliminate the biv, if it is a candidate.
	 This won't work if ! all_reduced,
	 since the givs we planned to use might not have been reduced.

	 We have to be careful that we didn't initially think we could eliminate
	 this biv because of a giv that we now think may be dead and shouldn't
	 be used as a biv replacement.  

	 Also, there is the possibility that we may have a giv that looks
	 like it can be used to eliminate a biv, but the resulting insn
	 isn't valid.  This can happen, for example, on the 88k, where a 
	 JUMP_INSN can compare a register only with zero.  Attempts to
	 replace it with a compare with a constant will fail.

	 Note that in cases where this call fails, we may have replaced some
	 of the occurrences of the biv with a giv, but no harm was done in
	 doing so in the rare cases where it can occur.  */

      /* intel1 Retry compare elimination this time for loop trip test.  */
      if (flag_compare_elim
         && !done_compare_elimination && bl->retry_celim > 1 
         && all_reduced == 1 && bl->eliminable)
	compare_elimination (bl, loop_start, loop_end, 
			     compare_insn, jump_insn, 1);

      if (all_reduced == 1 && bl->eliminable
	  && maybe_eliminate_biv (bl, loop_start, end, 1,
				  threshold, insn_count))

	{
	  /* ?? If we created a new test to bypass the loop entirely,
	     or otherwise drop straight in, based on this test, then
	     we might want to rewrite it also.  This way some later
	     pass has more hope of removing the initialization of this
	     biv entirely.  */

	  /* If final_value != 0, then the biv may be used after loop end
	     and we must emit an insn to set it just in case.

	     Reversed bivs already have an insn after the loop setting their
	     value, so we don't need another one.  We can't calculate the
	     proper final value for such a biv here anyways.  */
	  if (final_value != 0 && ! bl->reversed)
	    {
	      rtx insert_before;

	      /* If the loop has multiple exits, emit the insn before the
		 loop to ensure that it will always be executed no matter
		 how the loop exits.  Otherwise, emit the insn after the
		 loop, since this is slightly more efficient.  */
	      if (loop_number_exit_count[uid_loop_num[INSN_UID (loop_start)]])
		insert_before = loop_start;
	      else
		insert_before = end_insert_before;

	      emit_insn_before (gen_move_insn (bl->biv->dest_reg, final_value),
				end_insert_before);
	    }

#if 0
	  /* Delete all of the instructions inside the loop which set
	     the biv, as they are all dead.  If is safe to delete them,
	     because an insn setting a biv will never be part of a libcall.  */
	  /* However, deleting them will invalidate the regno_last_uid info,
	     so keeping them around is more convenient.  Final_biv_value
	     will only succeed when there are multiple exits if the biv
	     is dead at each exit, hence it does not matter that the original
	     insn remains, because it is dead anyways.  */
	  for (v = bl->biv; v; v = v->next_iv)
	    delete_insn (v->insn);
#endif

	  if (loop_dump_stream)
	    fprintf (loop_dump_stream, "Reg %d: biv eliminated\n",
		     bl->regno);
	}
      /* intel1 Retry compare elimination this time handling ashifhts. */
      else if (flag_compare_elim
               && !done_compare_elimination && bl->retry_celim == 1)
        {
	  if (compare_elimination (bl, loop_start, loop_end, 
	                         compare_insn, jump_insn, 1))
	    {
    	      rtx tem = find_reg_note (bl->init_insn, REG_EQUAL, 0);
	      if (tem)
		XEXP (tem, 0) = SET_SRC (bl->init_set);
 	      bl->initial_value = SET_SRC (bl->init_set);;
	    }
        } 
    }

  /* Go through all the instructions in the loop, making all the
     register substitutions scheduled in REG_MAP.  */

  for (p = loop_start; p != end; p = NEXT_INSN (p))
    if (GET_CODE (p) == INSN || GET_CODE (p) == JUMP_INSN
 	|| GET_CODE (p) == CALL_INSN)
      {
	replace_regs (PATTERN (p), reg_map, max_reg_before_loop, 0);
	replace_regs (REG_NOTES (p), reg_map, max_reg_before_loop, 0);
	INSN_CODE (p) = -1;
      }

  /* Unroll loops from within strength reduction so that we can use the
     induction variable information that strength_reduce has already
     collected.  */
  
  if (flag_unroll_loops)
    unroll_loop (loop_end, insn_count, loop_start, end_insert_before, 1);

  if (loop_dump_stream)
    fprintf (loop_dump_stream, "\n");
}

/* Return 1 if X is a valid source for an initial value (or as value being
   compared against in an initial test).

   X must be either a register or constant and must not be clobbered between
   the current insn and the start of the loop.

   INSN is the insn containing X.  */

static int
valid_initial_value_p (x, insn, call_seen, loop_start)
     rtx x;
     rtx insn;
     int call_seen;
     rtx loop_start;
{
  if (CONSTANT_P (x))
    return 1;

  /* Only consider pseudos we know about initialized in insns whose luids
     we know.  */
  if (GET_CODE (x) != REG
      || REGNO (x) >= max_reg_before_loop)
    return 0;

  /* Don't use call-clobbered registers across a call which clobbers it.  On
     some machines, don't use any hard registers at all.  */
  if (REGNO (x) < FIRST_PSEUDO_REGISTER
      && (
#ifdef SMALL_REGISTER_CLASSES
          SMALL_REGISTER_CLASSES
#else
	  0
#endif
	    || (call_used_regs[REGNO (x)] && call_seen))
      )
    return 0;

  /* Don't use registers that have been clobbered before the start of the
     loop.  */
  if (reg_set_between_p (x, insn, loop_start))
    return 0;

  return 1;
}

/* Scan X for memory refs and check each memory address
   as a possible giv.  INSN is the insn whose pattern X comes from.
   NOT_EVERY_ITERATION is 1 if the insn might not be executed during
   every loop iteration.  */

static void
find_mem_givs (x, insn, not_every_iteration, loop_start, loop_end)
     rtx x;
     rtx insn;
     int not_every_iteration;
     rtx loop_start, loop_end;
{
  register int i, j;
  register enum rtx_code code;
  register char *fmt;
  /* intel1 */
  rtx tx;

  if (x == 0)
    return;

  code = GET_CODE (x);
  switch (code)
    {
    case REG:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
    case ADDR_VEC:
    case ADDR_DIFF_VEC:
    case USE:
    case CLOBBER:
      return;

    case MEM:
      {
	rtx src_reg;
	rtx add_val;
	rtx mult_val;
	int benefit;

	benefit = general_induction_var (XEXP (x, 0),
					 &src_reg, &add_val, &mult_val);

	/* intel1
	   For now lets strip off consty and try to recognize the result
	   as a possible mem_giv.  */
#ifdef REWRITE_ADDRESS
	if (flag_opt_reg_use
	   && benefit == 0 && GET_CODE (XEXP (x, 0)) == PLUS 
	   && GET_CODE (XEXP (XEXP (x, 0), 0)) == PLUS 
	   &&  (GET_CODE (XEXP (XEXP (x, 0), 1)) == CONST_INT 
	    || GET_CODE (XEXP (XEXP (x, 0), 1)) == CONST 
	    || GET_CODE (XEXP (XEXP (x, 0), 1)) == SYMBOL_REF))
	  {
	    tx = x;
	    x = XEXP (x, 0);
	    benefit = general_induction_var (XEXP (x, 0),
                                             &src_reg, &add_val, &mult_val);
	  }
#endif

	/* Don't make a DEST_ADDR giv with mult_val == 1 && add_val == 0.
	   Such a giv isn't useful.  */
	if (benefit > 0 && (mult_val != const1_rtx || add_val != const0_rtx))
	  {
	    /* Found one; record it.  */
	    struct induction *v
	      = (struct induction *) oballoc (sizeof (struct induction));

/* intel1 */
#ifdef REDUCE_INDEX
#ifdef REWRITE_ADDRESS
#ifdef LEGITIMATE_INDEX_P
	rtx * index_part;
	/* try to eliminate index terms */		
	if (REDUCE_INDEX && flag_reduce_index_givs
	   && do_reduce_index (x, &add_val))
	  { 
	    index_part = &XEXP (XEXP (x, 0), 0);
	    record_giv (v, insn, src_reg, addr_placeholder, mult_val,
	               add_val, benefit, DEST_ADDR, not_every_iteration,
	               index_part, loop_start, loop_end);
	    v->is_index_induc = 1;
	  }
	else
#endif
#endif
#endif
	    record_giv (v, insn, src_reg, addr_placeholder, mult_val,
			add_val, benefit, DEST_ADDR, not_every_iteration,
			&XEXP (x, 0), loop_start, loop_end);

	    v->mem_mode = GET_MODE (x);
	  }
	return;
      }
    }

  /* Recursively scan the subexpressions for other mem refs.  */

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      find_mem_givs (XEXP (x, i), insn, not_every_iteration, loop_start,
		     loop_end);
    else if (fmt[i] == 'E')
      for (j = 0; j < XVECLEN (x, i); j++)
	find_mem_givs (XVECEXP (x, i, j), insn, not_every_iteration,
		       loop_start, loop_end);
}

/* Fill in the data about one biv update.
   V is the `struct induction' in which we record the biv.  (It is
   allocated by the caller, with alloca.)
   INSN is the insn that sets it.
   DEST_REG is the biv's reg.

   MULT_VAL is const1_rtx if the biv is being incremented here, in which case
   INC_VAL is the increment.  Otherwise, MULT_VAL is const0_rtx and the biv is
   being set to INC_VAL.

   NOT_EVERY_ITERATION is nonzero if this biv update is not know to be
   executed every iteration; MAYBE_MULTIPLE is nonzero if this biv update
   can be executed more than once per iteration.  If MAYBE_MULTIPLE
   and NOT_EVERY_ITERATION are both zero, we know that the biv update is
   executed exactly once per iteration.  */

static void
record_biv (v, insn, dest_reg, inc_val, mult_val,
	    not_every_iteration, maybe_multiple)
     struct induction *v;
     rtx insn;
     rtx dest_reg;
     rtx inc_val;
     rtx mult_val;
     int not_every_iteration;
     int maybe_multiple;
{
  struct iv_class *bl;

  v->insn = insn;
  v->src_reg = dest_reg;
  v->dest_reg = dest_reg;
  v->mult_val = mult_val;
  v->add_val = inc_val;
  v->mode = GET_MODE (dest_reg);
  v->always_computable = ! not_every_iteration;
  v->always_executed = ! not_every_iteration;
  v->maybe_multiple = maybe_multiple;

  /* Add this to the reg's iv_class, creating a class
     if this is the first incrementation of the reg.  */

  bl = reg_biv_class[REGNO (dest_reg)];
  if (bl == 0)
    {
      /* Create and initialize new iv_class.  */

      bl = (struct iv_class *) oballoc (sizeof (struct iv_class));

      bl->regno = REGNO (dest_reg);
      bl->biv = 0;
      bl->giv = 0;
      bl->biv_count = 0;
      bl->giv_count = 0;

      /* Set initial value to the reg itself.  */
      bl->initial_value = dest_reg;
      /* We haven't seen the initializing insn yet */
      bl->init_insn = 0;
      bl->init_set = 0;
      bl->initial_test = 0;
      bl->incremented = 0;
      bl->eliminable = 0;
      bl->nonneg = 0;
      bl->reversed = 0;
      /* intel1 */
      bl->retry_celim = 0;
      bl->total_benefit = 0;

      /* Add this class to loop_iv_list.  */
      bl->next = loop_iv_list;
      loop_iv_list = bl;

      /* Put it in the array of biv register classes.  */
      reg_biv_class[REGNO (dest_reg)] = bl;
    }

  /* Update IV_CLASS entry for this biv.  */
  v->next_iv = bl->biv;
  bl->biv = v;
  bl->biv_count++;
  if (mult_val == const1_rtx)
    bl->incremented = 1;

  if (loop_dump_stream)
    {
      fprintf (loop_dump_stream,
	       "Insn %d: possible biv, reg %d,",
	       INSN_UID (insn), REGNO (dest_reg));
      if (GET_CODE (inc_val) == CONST_INT)
	fprintf (loop_dump_stream, " const = %d\n",
		 INTVAL (inc_val));
      else
	{
	  fprintf (loop_dump_stream, " const = ");
	  print_rtl (loop_dump_stream, inc_val);
	  fprintf (loop_dump_stream, "\n");
	}
    }
}

/* Fill in the data about one giv.
   V is the `struct induction' in which we record the giv.  (It is
   allocated by the caller, with alloca.)
   INSN is the insn that sets it.
   BENEFIT estimates the savings from deleting this insn.
   TYPE is DEST_REG or DEST_ADDR; it says whether the giv is computed
   into a register or is used as a memory address.

   SRC_REG is the biv reg which the giv is computed from.
   DEST_REG is the giv's reg (if the giv is stored in a reg).
   MULT_VAL and ADD_VAL are the coefficients used to compute the giv.
   LOCATION points to the place where this giv's value appears in INSN.  */

static void
record_giv (v, insn, src_reg, dest_reg, mult_val, add_val, benefit,
	    type, not_every_iteration, location, loop_start, loop_end)
     struct induction *v;
     rtx insn;
     rtx src_reg;
     rtx dest_reg;
     rtx mult_val, add_val;
     int benefit;
     enum g_types type;
     int not_every_iteration;
     rtx *location;
     rtx loop_start, loop_end;
{
  struct induction *b;
  struct iv_class *bl;
  rtx set = single_set (insn);
  rtx p;

  v->insn = insn;
  v->src_reg = src_reg;
  v->giv_type = type;
  v->dest_reg = dest_reg;
  v->mult_val = mult_val;
  v->add_val = add_val;
  v->benefit = benefit;
  v->location = location;
  v->cant_derive = 0;
  v->combined_with = 0;
  v->maybe_multiple = 0;
  v->maybe_dead = 0;
  v->derive_adjustment = 0;
  v->same = 0;
  v->ignore = 0;
  v->new_reg = 0;
  v->final_value = 0;
  v->same_insn = 0;
  /* intel1 */
#ifdef REDUCE_INDEX
  v->is_index_induc = 0;
#endif
  v->auto_inc_opt = 0;

  /* The v->always_computable field is used in update_giv_derive, to
     determine whether a giv can be used to derive another giv.  For a
     DEST_REG giv, INSN computes a new value for the giv, so its value
     isn't computable if INSN insn't executed every iteration.
     However, for a DEST_ADDR giv, INSN merely uses the value of the giv;
     it does not compute a new value.  Hence the value is always computable
     regardless of whether INSN is executed each iteration.  */

  if (type == DEST_ADDR)
    v->always_computable = 1;
  else
    v->always_computable = ! not_every_iteration;

  v->always_executed = ! not_every_iteration;

  if (type == DEST_ADDR)
    {
      v->mode = GET_MODE (*location);
      v->lifetime = 1;
      v->times_used = 1;
    }
  else /* type == DEST_REG */
    {
      v->mode = GET_MODE (SET_DEST (set));

      v->lifetime = (uid_luid[regno_last_uid[REGNO (dest_reg)]]
		     - uid_luid[regno_first_uid[REGNO (dest_reg)]]);

      v->times_used = n_times_used[REGNO (dest_reg)];

      /* If the lifetime is zero, it means that this register is
	 really a dead store.  So mark this as a giv that can be
	 ignored.  This will not prevent the biv from being eliminated.  */
      if (v->lifetime == 0)
	v->ignore = 1;

      reg_iv_type[REGNO (dest_reg)] = GENERAL_INDUCT;
      reg_iv_info[REGNO (dest_reg)] = v;
    }

  /* Add the giv to the class of givs computed from one biv.  */

  bl = reg_biv_class[REGNO (src_reg)];
  if (bl)
    {
      v->next_iv = bl->giv;
      bl->giv = v;
      /* Don't count DEST_ADDR.  This is supposed to count the number of
	 insns that calculate givs.  */
      if (type == DEST_REG)
	bl->giv_count++;
      bl->total_benefit += benefit;
    }
  else
    /* Fatal error, biv missing for this giv?  */
    abort ();

  if (type == DEST_ADDR)
    v->replaceable = 1;
  else
    {
      /* The giv can be replaced outright by the reduced register only if all
	 of the following conditions are true:
 	 - the insn that sets the giv is always executed on any iteration
	   on which the giv is used at all
	   (there are two ways to deduce this:
	    either the insn is executed on every iteration,
	    or all uses follow that insn in the same basic block),
 	 - the giv is not used outside the loop
	 - no assignments to the biv occur during the giv's lifetime.  */

      if (regno_first_uid[REGNO (dest_reg)] == INSN_UID (insn)
	  /* Previous line always fails if INSN was moved by loop opt.  */
	  && uid_luid[regno_last_uid[REGNO (dest_reg)]] < INSN_LUID (loop_end)
	  && (! not_every_iteration
	      || last_use_this_basic_block (dest_reg, insn)))
 	{
	  /* Now check that there are no assignments to the biv within the
	     giv's lifetime.  This requires two separate checks.  */

	  /* Check each biv update, and fail if any are between the first
	     and last use of the giv.
	     
	     If this loop contains an inner loop that was unrolled, then
	     the insn modifying the biv may have been emitted by the loop
	     unrolling code, and hence does not have a valid luid.  Just
	     mark the biv as not replaceable in this case.  It is not very
	     useful as a biv, because it is used in two different loops.
	     It is very unlikely that we would be able to optimize the giv
	     using this biv anyways.  */

	  v->replaceable = 1;
	  for (b = bl->biv; b; b = b->next_iv)
	    {
	      if (INSN_UID (b->insn) >= max_uid_for_loop
		  || ((uid_luid[INSN_UID (b->insn)]
		       >= uid_luid[regno_first_uid[REGNO (dest_reg)]])
		      && (uid_luid[INSN_UID (b->insn)]
			  <= uid_luid[regno_last_uid[REGNO (dest_reg)]])))
		{
		  v->replaceable = 0;
		  v->not_replaceable = 1;
		  break;
 		}
	    }

	  /* If there are any backwards branches that go from after the
	     biv update to before it, then this giv is not replaceable.  */
	  if (v->replaceable)
	    for (b = bl->biv; b; b = b->next_iv)
	      if (back_branch_in_range_p (b->insn, loop_start, loop_end))
		{
		  v->replaceable = 0;
		  v->not_replaceable = 1;
		  break;
		}
	}
      else
	{
	  /* May still be replaceable, we don't have enough info here to
	     decide.  */
	  v->replaceable = 0;
	  v->not_replaceable = 0;
	}
    }

  if (loop_dump_stream)
    {
      if (type == DEST_REG)
 	fprintf (loop_dump_stream, "Insn %d: giv reg %d",
		 INSN_UID (insn), REGNO (dest_reg));
      else
 	fprintf (loop_dump_stream, "Insn %d: dest address",
 		 INSN_UID (insn));

      fprintf (loop_dump_stream, " src reg %d benefit %d",
	       REGNO (src_reg), v->benefit);
      fprintf (loop_dump_stream, " used %d lifetime %d",
	       v->times_used, v->lifetime);

      if (v->replaceable)
 	fprintf (loop_dump_stream, " replaceable");

      if (GET_CODE (mult_val) == CONST_INT)
	fprintf (loop_dump_stream, " mult %d",
 		 INTVAL (mult_val));
      else
	{
	  fprintf (loop_dump_stream, " mult ");
	  print_rtl (loop_dump_stream, mult_val);
	}

      if (GET_CODE (add_val) == CONST_INT)
	fprintf (loop_dump_stream, " add %d",
		 INTVAL (add_val));
      else
	{
	  fprintf (loop_dump_stream, " add ");
	  print_rtl (loop_dump_stream, add_val);
	}
    }

  if (loop_dump_stream)
    fprintf (loop_dump_stream, "\n");

}


/* All this does is determine whether a giv can be made replaceable because
   its final value can be calculated.  This code can not be part of record_giv
   above, because final_giv_value requires that the number of loop iterations
   be known, and that can not be accurately calculated until after all givs
   have been identified.  */

static void
check_final_value (v, loop_start, loop_end)
     struct induction *v;
     rtx loop_start, loop_end;
{
  struct iv_class *bl;
  rtx final_value = 0;

  bl = reg_biv_class[REGNO (v->src_reg)];

  /* DEST_ADDR givs will never reach here, because they are always marked
     replaceable above in record_giv.  */

  /* The giv can be replaced outright by the reduced register only if all
     of the following conditions are true:
     - the insn that sets the giv is always executed on any iteration
       on which the giv is used at all
       (there are two ways to deduce this:
        either the insn is executed on every iteration,
        or all uses follow that insn in the same basic block),
     - its final value can be calculated (this condition is different
       than the one above in record_giv)
     - no assignments to the biv occur during the giv's lifetime.  */

#if 0
  /* This is only called now when replaceable is known to be false.  */
  /* Clear replaceable, so that it won't confuse final_giv_value.  */
  v->replaceable = 0;
#endif

  if ((final_value = final_giv_value (v, loop_start, loop_end))
      && (v->always_computable || last_use_this_basic_block (v->dest_reg, v->insn)))
    {
      int biv_increment_seen = 0;
      rtx p = v->insn;
      rtx last_giv_use;

      v->replaceable = 1;

      /* When trying to determine whether or not a biv increment occurs
	 during the lifetime of the giv, we can ignore uses of the variable
	 outside the loop because final_value is true.  Hence we can not
	 use regno_last_uid and regno_first_uid as above in record_giv.  */

      /* Search the loop to determine whether any assignments to the
	 biv occur during the giv's lifetime.  Start with the insn
	 that sets the giv, and search around the loop until we come
	 back to that insn again.

	 Also fail if there is a jump within the giv's lifetime that jumps
	 to somewhere outside the lifetime but still within the loop.  This
	 catches spaghetti code where the execution order is not linear, and
	 hence the above test fails.  Here we assume that the giv lifetime
	 does not extend from one iteration of the loop to the next, so as
	 to make the test easier.  Since the lifetime isn't known yet,
	 this requires two loops.  See also record_giv above.  */

      last_giv_use = v->insn;

      while (1)
	{
	  p = NEXT_INSN (p);
	  if (p == loop_end)
	    p = NEXT_INSN (loop_start);
	  if (p == v->insn)
	    break;

	  if (GET_CODE (p) == INSN || GET_CODE (p) == JUMP_INSN
	      || GET_CODE (p) == CALL_INSN)
	    {
	      if (biv_increment_seen)
		{
		  if (reg_mentioned_p (v->dest_reg, PATTERN (p)))
		    {
		      v->replaceable = 0;
		      v->not_replaceable = 1;
		      break;
		    }
		}
	      else if (GET_CODE (PATTERN (p)) == SET
		       && SET_DEST (PATTERN (p)) == v->src_reg)
		biv_increment_seen = 1;
	      else if (reg_mentioned_p (v->dest_reg, PATTERN (p)))
		last_giv_use = p;
	    }
	}
      
      /* Now that the lifetime of the giv is known, check for branches
	 from within the lifetime to outside the lifetime if it is still
	 replaceable.  */

      if (v->replaceable)
	{
	  p = v->insn;
	  while (1)
	    {
	      p = NEXT_INSN (p);
	      if (p == loop_end)
		p = NEXT_INSN (loop_start);
	      if (p == last_giv_use)
		break;

	      if (GET_CODE (p) == JUMP_INSN && JUMP_LABEL (p)
		  && LABEL_NAME (JUMP_LABEL (p))
		  && ((INSN_UID (JUMP_LABEL (p)) >= max_uid_for_loop)
		      || (INSN_UID (v->insn) >= max_uid_for_loop)
		      || (INSN_UID (last_giv_use) >= max_uid_for_loop)
		      || (INSN_LUID (JUMP_LABEL (p)) < INSN_LUID (v->insn)
			  && INSN_LUID (JUMP_LABEL (p)) > INSN_LUID (loop_start))
		      || (INSN_LUID (JUMP_LABEL (p)) > INSN_LUID (last_giv_use)
			  && INSN_LUID (JUMP_LABEL (p)) < INSN_LUID (loop_end))))
		{
		  v->replaceable = 0;
		  v->not_replaceable = 1;

		  if (loop_dump_stream)
		    fprintf (loop_dump_stream,
			     "Found branch outside giv lifetime.\n");

		  break;
		}
	    }
	}

      /* If it is replaceable, then save the final value.  */
      if (v->replaceable)
	v->final_value = final_value;
    }

  if (loop_dump_stream && v->replaceable)
    fprintf (loop_dump_stream, "Insn %d: giv reg %d final_value replaceable\n",
	     INSN_UID (v->insn), REGNO (v->dest_reg));
}

/* Update the status of whether a giv can derive other givs.

   We need to do something special if there is or may be an update to the biv
   between the time the giv is defined and the time it is used to derive
   another giv.

   In addition, a giv that is only conditionally set is not allowed to
   derive another giv once a label has been passed.

   The cases we look at are when a label or an update to a biv is passed.  */

static void
update_giv_derive (p)
     rtx p;
{
  struct iv_class *bl;
  struct induction *biv, *giv;
  rtx tem;
  int dummy;

  /* Search all IV classes, then all bivs, and finally all givs.

     There are three cases we are concerned with.  First we have the situation
     of a giv that is only updated conditionally.  In that case, it may not
     derive any givs after a label is passed.

     The second case is when a biv update occurs, or may occur, after the
     definition of a giv.  For certain biv updates (see below) that are
     known to occur between the giv definition and use, we can adjust the
     giv definition.  For others, or when the biv update is conditional,
     we must prevent the giv from deriving any other givs.  There are two
     sub-cases within this case.

     If this is a label, we are concerned with any biv update that is done
     conditionally, since it may be done after the giv is defined followed by
     a branch here (actually, we need to pass both a jump and a label, but
     this extra tracking doesn't seem worth it).

     If this is a jump, we are concerned about any biv update that may be
     executed multiple times.  We are actually only concerned about
     backward jumps, but it is probably not worth performing the test
     on the jump again here.

     If this is a biv update, we must adjust the giv status to show that a
     subsequent biv update was performed.  If this adjustment cannot be done,
     the giv cannot derive further givs.  */

  for (bl = loop_iv_list; bl; bl = bl->next)
    for (biv = bl->biv; biv; biv = biv->next_iv)
      if (GET_CODE (p) == CODE_LABEL || GET_CODE (p) == JUMP_INSN
	  || biv->insn == p)
	{
	  for (giv = bl->giv; giv; giv = giv->next_iv)
	    {
	      /* If cant_derive is already true, there is no point in
		 checking all of these conditions again.  */
	      if (giv->cant_derive)
		continue;

	      /* If this giv is conditionally set and we have passed a label,
		 it cannot derive anything.  */
	      if (GET_CODE (p) == CODE_LABEL && ! giv->always_computable)
		giv->cant_derive = 1;

	      /* Skip givs that have mult_val == 0, since
		 they are really invariants.  Also skip those that are
		 replaceable, since we know their lifetime doesn't contain
		 any biv update.  */
	      else if (giv->mult_val == const0_rtx || giv->replaceable)
		continue;

	      /* The only way we can allow this giv to derive another
		 is if this is a biv increment and we can form the product
		 of biv->add_val and giv->mult_val.  In this case, we will
		 be able to compute a compensation.  */
	      else if (biv->insn == p)
		{
		  tem = 0;

		  if (biv->mult_val == const1_rtx)
		    tem = simplify_giv_expr (gen_rtx (MULT, giv->mode,
						      biv->add_val,
						      giv->mult_val),
					     &dummy);

		  if (tem && giv->derive_adjustment)
		    tem = simplify_giv_expr (gen_rtx (PLUS, giv->mode, tem,
						      giv->derive_adjustment),
					     &dummy);
		  if (tem)
		    giv->derive_adjustment = tem;
		  else
		    giv->cant_derive = 1;
		}
	      else if ((GET_CODE (p) == CODE_LABEL && ! biv->always_computable)
		       || (GET_CODE (p) == JUMP_INSN && biv->maybe_multiple))
		giv->cant_derive = 1;
	    }
	}
}

/* Check whether an insn is an increment legitimate for a basic induction var.
   X is the source of insn P, or a part of it.
   MODE is the mode in which X should be interpreted.

   DEST_REG is the putative biv, also the destination of the insn.
   We accept patterns of these forms:
     REG = REG + INVARIANT (includes REG = REG - CONSTANT)
     REG = INVARIANT + REG

   If X is suitable, we return 1, set *MULT_VAL to CONST1_RTX,
   and store the additive term into *INC_VAL.

   If X is an assignment of an invariant into DEST_REG, we set
   *MULT_VAL to CONST0_RTX, and store the invariant into *INC_VAL.

   We also want to detect a BIV when it corresponds to a variable
   whose mode was promoted via PROMOTED_MODE.  In that case, an increment
   of the variable may be a PLUS that adds a SUBREG of that variable to
   an invariant and then sign- or zero-extends the result of the PLUS
   into the variable.

   Most GIVs in such cases will be in the promoted mode, since that is the
   probably the natural computation mode (and almost certainly the mode
   used for addresses) on the machine.  So we view the pseudo-reg containing
   the variable as the BIV, as if it were simply incremented.

   Note that treating the entire pseudo as a BIV will result in making
   simple increments to any GIVs based on it.  However, if the variable
   overflows in its declared mode but not its promoted mode, the result will
   be incorrect.  This is acceptable if the variable is signed, since 
   overflows in such cases are undefined, but not if it is unsigned, since
   those overflows are defined.  So we only check for SIGN_EXTEND and
   not ZERO_EXTEND.

   If we cannot find a biv, we return 0.  */

static int
basic_induction_var (x, mode, dest_reg, p, inc_val, mult_val)
     register rtx x;
     enum machine_mode mode;
     rtx p;
     rtx dest_reg;
     rtx *inc_val;
     rtx *mult_val;
{
  register enum rtx_code code;
  rtx arg;
  rtx insn, set = 0;

  code = GET_CODE (x);
  switch (code)
    {
    case PLUS:
      if (XEXP (x, 0) == dest_reg
	  || (GET_CODE (XEXP (x, 0)) == SUBREG
	      && SUBREG_PROMOTED_VAR_P (XEXP (x, 0))
	      && SUBREG_REG (XEXP (x, 0)) == dest_reg))
 	arg = XEXP (x, 1);
      else if (XEXP (x, 1) == dest_reg
	       || (GET_CODE (XEXP (x, 1)) == SUBREG
		   && SUBREG_PROMOTED_VAR_P (XEXP (x, 1))
		   && SUBREG_REG (XEXP (x, 1)) == dest_reg))
	arg = XEXP (x, 0);
      else
 	return 0;

      if (invariant_p (arg) != 1)
	return 0;

      *inc_val = convert_modes (GET_MODE (dest_reg), GET_MODE (x), arg, 0);
      *mult_val = const1_rtx;
      return 1;

    case SUBREG:
      /* If this is a SUBREG for a promoted variable, check the inner
	 value.  */
      if (SUBREG_PROMOTED_VAR_P (x))
	return basic_induction_var (SUBREG_REG (x), GET_MODE (SUBREG_REG (x)),
				    dest_reg, p, inc_val, mult_val);
      return 0;

    case REG:
      /* If this register is assigned in the previous insn, look at its
	 source, but don't go outside the loop or past a label.  */

      for (insn = PREV_INSN (p);
	   (insn && GET_CODE (insn) == NOTE
	    && NOTE_LINE_NUMBER (insn) != NOTE_INSN_LOOP_BEG);
	   insn = PREV_INSN (insn))
	;

      if (insn)
	set = single_set (insn);

      if (set != 0
	  && (SET_DEST (set) == x
	      || (GET_CODE (SET_DEST (set)) == SUBREG
		  && (GET_MODE_SIZE (GET_MODE (SET_DEST (set)))
		      <= UNITS_PER_WORD)
		  && SUBREG_REG (SET_DEST (set)) == x)))
	return basic_induction_var (SET_SRC (set),
				    (GET_MODE (SET_SRC (set)) == VOIDmode
				     ? GET_MODE (x)
				     : GET_MODE (SET_SRC (set))),
				    dest_reg, insn,
				    inc_val, mult_val);
      /* ... fall through ...  */

      /* Can accept constant setting of biv only when inside inner most loop.
  	 Otherwise, a biv of an inner loop may be incorrectly recognized
	 as a biv of the outer loop,
	 causing code to be moved INTO the inner loop.  */
    case MEM:
      if (invariant_p (x) != 1)
	return 0;
    case CONST_INT:
    case SYMBOL_REF:
    case CONST:
      if (loops_enclosed == 1)
 	{
	  /* Possible bug here?  Perhaps we don't know the mode of X.  */
	  *inc_val = convert_modes (GET_MODE (dest_reg), mode, x, 0);
 	  *mult_val = const0_rtx;
 	  return 1;
 	}
      else
 	return 0;

    case SIGN_EXTEND:
      return basic_induction_var (XEXP (x, 0), GET_MODE (XEXP (x, 0)),
				  dest_reg, p, inc_val, mult_val);
    case ASHIFTRT:
      /* Similar, since this can be a sign extension.  */
      for (insn = PREV_INSN (p);
	   (insn && GET_CODE (insn) == NOTE
	    && NOTE_LINE_NUMBER (insn) != NOTE_INSN_LOOP_BEG);
	   insn = PREV_INSN (insn))
	;

      if (insn)
	set = single_set (insn);

      if (set && SET_DEST (set) == XEXP (x, 0)
	  && GET_CODE (XEXP (x, 1)) == CONST_INT
	  && INTVAL (XEXP (x, 1)) >= 0
	  && GET_CODE (SET_SRC (set)) == ASHIFT
	  && XEXP (x, 1) == XEXP (SET_SRC (set), 1))
	return basic_induction_var (XEXP (SET_SRC (set), 0),
				    GET_MODE (XEXP (x, 0)),
				    dest_reg, insn, inc_val, mult_val);
      return 0;

    default:
      return 0;
    }
}

/* A general induction variable (giv) is any quantity that is a linear
   function   of a basic induction variable,
   i.e. giv = biv * mult_val + add_val.
   The coefficients can be any loop invariant quantity.
   A giv need not be computed directly from the biv;
   it can be computed by way of other givs.  */

/* Determine whether X computes a giv.
   If it does, return a nonzero value
     which is the benefit from eliminating the computation of X;
   set *SRC_REG to the register of the biv that it is computed from;
   set *ADD_VAL and *MULT_VAL to the coefficients,
     such that the value of X is biv * mult + add;  */

static int
general_induction_var (x, src_reg, add_val, mult_val)
     rtx x;
     rtx *src_reg;
     rtx *add_val;
     rtx *mult_val;
{
  rtx orig_x = x;
  int benefit = 0;
  char *storage;

  /* If this is an invariant, forget it, it isn't a giv.  */
  if (invariant_p (x) == 1)
    return 0;

  /* See if the expression could be a giv and get its form.
     Mark our place on the obstack in case we don't find a giv.  */
  storage = (char *) oballoc (0);
  x = simplify_giv_expr (x, &benefit);
  if (x == 0)
    {
      obfree (storage);
      return 0;
    }

  switch (GET_CODE (x))
    {
    case USE:
    case CONST_INT:
      /* Since this is now an invariant and wasn't before, it must be a giv
	 with MULT_VAL == 0.  It doesn't matter which BIV we associate this
	 with.  */
      *src_reg = loop_iv_list->biv->dest_reg;
      *mult_val = const0_rtx;
      *add_val = x;
      break;

    case REG:
      /* This is equivalent to a BIV.  */
      *src_reg = x;
      *mult_val = const1_rtx;
      *add_val = const0_rtx;
      break;

    case PLUS:
      /* Either (plus (biv) (invar)) or
	 (plus (mult (biv) (invar_1)) (invar_2)).  */
      if (GET_CODE (XEXP (x, 0)) == MULT)
	{
	  *src_reg = XEXP (XEXP (x, 0), 0);
	  *mult_val = XEXP (XEXP (x, 0), 1);
	}
      else
	{
	  *src_reg = XEXP (x, 0);
	  *mult_val = const1_rtx;
	}
      *add_val = XEXP (x, 1);
      break;

    case MULT:
      /* ADD_VAL is zero.  */
      *src_reg = XEXP (x, 0);
      *mult_val = XEXP (x, 1);
      *add_val = const0_rtx;
      break;

    default:
      abort ();
    }

  /* Remove any enclosing USE from ADD_VAL and MULT_VAL (there will be
     unless they are CONST_INT).  */
  if (GET_CODE (*add_val) == USE)
    *add_val = XEXP (*add_val, 0);
  if (GET_CODE (*mult_val) == USE)
    *mult_val = XEXP (*mult_val, 0);

  benefit += rtx_cost (orig_x, SET);

  /* Always return some benefit if this is a giv so it will be detected
     as such.  This allows elimination of bivs that might otherwise
     not be eliminated.  */
  return benefit == 0 ? 1 : benefit;
}

/* Given an expression, X, try to form it as a linear function of a biv.
   We will canonicalize it to be of the form
   	(plus (mult (BIV) (invar_1))
	      (invar_2))
   with possible degeneracies.

   The invariant expressions must each be of a form that can be used as a
   machine operand.  We surround then with a USE rtx (a hack, but localized
   and certainly unambiguous!) if not a CONST_INT for simplicity in this
   routine; it is the caller's responsibility to strip them.

   If no such canonicalization is possible (i.e., two biv's are used or an
   expression that is neither invariant nor a biv or giv), this routine
   returns 0.

   For a non-zero return, the result will have a code of CONST_INT, USE,
   REG (for a BIV), PLUS, or MULT.  No other codes will occur.  

   *BENEFIT will be incremented by the benefit of any sub-giv encountered.  */

static rtx
simplify_giv_expr (x, benefit)
     rtx x;
     int *benefit;
{
  enum machine_mode mode = GET_MODE (x);
  rtx arg0, arg1;
  rtx tem;

  /* If this is not an integer mode, or if we cannot do arithmetic in this
     mode, this can't be a giv.  */
  if (mode != VOIDmode
      && (GET_MODE_CLASS (mode) != MODE_INT
	  || GET_MODE_BITSIZE (mode) > HOST_BITS_PER_WIDE_INT))
    return 0;

  switch (GET_CODE (x))
    {
    case PLUS:
      arg0 = simplify_giv_expr (XEXP (x, 0), benefit);
      arg1 = simplify_giv_expr (XEXP (x, 1), benefit);
      if (arg0 == 0 || arg1 == 0)
	return 0;

      /* Put constant last, CONST_INT last if both constant.  */
      if ((GET_CODE (arg0) == USE
	   || GET_CODE (arg0) == CONST_INT)
	  && GET_CODE (arg1) != CONST_INT)
	tem = arg0, arg0 = arg1, arg1 = tem;

      /* Handle addition of zero, then addition of an invariant.  */
      if (arg1 == const0_rtx)
	return arg0;
      else if (GET_CODE (arg1) == CONST_INT || GET_CODE (arg1) == USE)
	switch (GET_CODE (arg0))
	  {
	  case CONST_INT:
	  case USE:
	    /* Both invariant.  Only valid if sum is machine operand.
	       First strip off possible USE on first operand.  */
	    if (GET_CODE (arg0) == USE)
	      arg0 = XEXP (arg0, 0);

	    tem = 0;
	    if (CONSTANT_P (arg0) && GET_CODE (arg1) == CONST_INT)
	      {
		tem = plus_constant (arg0, INTVAL (arg1));
		if (GET_CODE (tem) != CONST_INT)
		  tem = gen_rtx (USE, mode, tem);
	      }

	    return tem;

	  case REG:
	  case MULT:
	    /* biv + invar or mult + invar.  Return sum.  */
	    return gen_rtx (PLUS, mode, arg0, arg1);

	  case PLUS:
	    /* (a + invar_1) + invar_2.  Associate.  */
	    return simplify_giv_expr (gen_rtx (PLUS, mode,
					       XEXP (arg0, 0),
					       gen_rtx (PLUS, mode,
							XEXP (arg0, 1), arg1)),
				      benefit);

	  default:
	    abort ();
	  }

      /* Each argument must be either REG, PLUS, or MULT.  Convert REG to
	 MULT to reduce cases.  */
      if (GET_CODE (arg0) == REG)
	arg0 = gen_rtx (MULT, mode, arg0, const1_rtx);
      if (GET_CODE (arg1) == REG)
	arg1 = gen_rtx (MULT, mode, arg1, const1_rtx);

      /* Now have PLUS + PLUS, PLUS + MULT, MULT + PLUS, or MULT + MULT.
	 Put a MULT first, leaving PLUS + PLUS, MULT + PLUS, or MULT + MULT.
	 Recurse to associate the second PLUS.  */
      if (GET_CODE (arg1) == MULT)
	tem = arg0, arg0 = arg1, arg1 = tem;

      if (GET_CODE (arg1) == PLUS)
	  return simplify_giv_expr (gen_rtx (PLUS, mode,
					     gen_rtx (PLUS, mode,
						      arg0, XEXP (arg1, 0)),
					     XEXP (arg1, 1)),
				    benefit);

      /* Now must have MULT + MULT.  Distribute if same biv, else not giv.  */
      if (GET_CODE (arg0) != MULT || GET_CODE (arg1) != MULT)
	abort ();

      if (XEXP (arg0, 0) != XEXP (arg1, 0))
	return 0;

      return simplify_giv_expr (gen_rtx (MULT, mode,
					 XEXP (arg0, 0),
					 gen_rtx (PLUS, mode,
						  XEXP (arg0, 1),
						  XEXP (arg1, 1))),
				benefit);

    case MINUS:
      /* Handle "a - b" as "a + b * (-1)".  */
      return simplify_giv_expr (gen_rtx (PLUS, mode,
					 XEXP (x, 0),
					 gen_rtx (MULT, mode,
						  XEXP (x, 1), constm1_rtx)),
				benefit);

    case MULT:
      arg0 = simplify_giv_expr (XEXP (x, 0), benefit);
      arg1 = simplify_giv_expr (XEXP (x, 1), benefit);
      if (arg0 == 0 || arg1 == 0)
	return 0;

      /* Put constant last, CONST_INT last if both constant.  */
      if ((GET_CODE (arg0) == USE || GET_CODE (arg0) == CONST_INT)
	  && GET_CODE (arg1) != CONST_INT)
	tem = arg0, arg0 = arg1, arg1 = tem;

      /* If second argument is not now constant, not giv.  */
      if (GET_CODE (arg1) != USE && GET_CODE (arg1) != CONST_INT)
	return 0;

      /* Handle multiply by 0 or 1.  */
      if (arg1 == const0_rtx)
	return const0_rtx;

      else if (arg1 == const1_rtx)
	return arg0;

      switch (GET_CODE (arg0))
	{
	case REG:
	  /* biv * invar.  Done.  */
	  return gen_rtx (MULT, mode, arg0, arg1);

	case CONST_INT:
	  /* Product of two constants.  */
	  return GEN_INT (INTVAL (arg0) * INTVAL (arg1));

	case USE:
	  /* invar * invar.  Not giv.  */
	  return 0;

	case MULT:
	  /* (a * invar_1) * invar_2.  Associate.  */
	  return simplify_giv_expr (gen_rtx (MULT, mode,
					     XEXP (arg0, 0),
					     gen_rtx (MULT, mode,
						      XEXP (arg0, 1), arg1)),
				    benefit);

	case PLUS:
	  /* (a + invar_1) * invar_2.  Distribute.  */
	  return simplify_giv_expr (gen_rtx (PLUS, mode,
					     gen_rtx (MULT, mode,
						      XEXP (arg0, 0), arg1),
					     gen_rtx (MULT, mode,
						      XEXP (arg0, 1), arg1)),
				    benefit);

	default:
	  abort ();
	}

    case ASHIFT:
      /* Shift by constant is multiply by power of two.  */
      if (GET_CODE (XEXP (x, 1)) != CONST_INT)
	return 0;

      return simplify_giv_expr (gen_rtx (MULT, mode,
					 XEXP (x, 0),
					 GEN_INT ((HOST_WIDE_INT) 1
						  << INTVAL (XEXP (x, 1)))),
				benefit);

    case NEG:
      /* "-a" is "a * (-1)" */
      return simplify_giv_expr (gen_rtx (MULT, mode, XEXP (x, 0), constm1_rtx),
				benefit);

    case NOT:
      /* "~a" is "-a - 1". Silly, but easy.  */
      return simplify_giv_expr (gen_rtx (MINUS, mode,
					 gen_rtx (NEG, mode, XEXP (x, 0)),
					 const1_rtx),
				benefit);

    case USE:
      /* Already in proper form for invariant.  */
      return x;

    case REG:
      /* If this is a new register, we can't deal with it.  */
      if (REGNO (x) >= max_reg_before_loop)
	return 0;

      /* Check for biv or giv.  */
      switch (reg_iv_type[REGNO (x)])
	{
	case BASIC_INDUCT:
	  return x;
	case GENERAL_INDUCT:
	  {
	    struct induction *v = reg_iv_info[REGNO (x)];

	    /* Form expression from giv and add benefit.  Ensure this giv
	       can derive another and subtract any needed adjustment if so.  */
	    *benefit += v->benefit;
	    if (v->cant_derive)
	      return 0;

	    tem = gen_rtx (PLUS, mode, gen_rtx (MULT, mode,
						v->src_reg, v->mult_val),
			   v->add_val);
	    if (v->derive_adjustment)
	      tem = gen_rtx (MINUS, mode, tem, v->derive_adjustment);
	    return simplify_giv_expr (tem, benefit);
	  }
	}

      /* Fall through to general case.  */
    default:
      /* If invariant, return as USE (unless CONST_INT).
	 Otherwise, not giv.  */
      if (GET_CODE (x) == USE)
	x = XEXP (x, 0);

      if (invariant_p (x) == 1)
	{
	  if (GET_CODE (x) == CONST_INT)
	    return x;
	  else
	    return gen_rtx (USE, mode, x);
	}
      else
	return 0;
    }
}

/* Help detect a giv that is calculated by several consecutive insns;
   for example,
      giv = biv * M
      giv = giv + A
   The caller has already identified the first insn P as having a giv as dest;
   we check that all other insns that set the same register follow
   immediately after P, that they alter nothing else,
   and that the result of the last is still a giv.

   The value is 0 if the reg set in P is not really a giv.
   Otherwise, the value is the amount gained by eliminating
   all the consecutive insns that compute the value.

   FIRST_BENEFIT is the amount gained by eliminating the first insn, P.
   SRC_REG is the reg of the biv; DEST_REG is the reg of the giv.

   The coefficients of the ultimate giv value are stored in
   *MULT_VAL and *ADD_VAL.  */

static int
consec_sets_giv (first_benefit, p, src_reg, dest_reg,
		 add_val, mult_val)
     int first_benefit;
     rtx p;
     rtx src_reg;
     rtx dest_reg;
     rtx *add_val;
     rtx *mult_val;
{
  int count;
  enum rtx_code code;
  int benefit;
  rtx temp;
  rtx set;

  /* Indicate that this is a giv so that we can update the value produced in
     each insn of the multi-insn sequence. 

     This induction structure will be used only by the call to
     general_induction_var below, so we can allocate it on our stack.
     If this is a giv, our caller will replace the induct var entry with
     a new induction structure.  */
  struct induction *v
    = (struct induction *) alloca (sizeof (struct induction));
  v->src_reg = src_reg;
  v->mult_val = *mult_val;
  v->add_val = *add_val;
  v->benefit = first_benefit;
  v->cant_derive = 0;
  v->derive_adjustment = 0;

  reg_iv_type[REGNO (dest_reg)] = GENERAL_INDUCT;
  reg_iv_info[REGNO (dest_reg)] = v;

  count = n_times_set[REGNO (dest_reg)] - 1;

  while (count > 0)
    {
      p = NEXT_INSN (p);
      code = GET_CODE (p);

      /* If libcall, skip to end of call sequence.  */
      if (code == INSN && (temp = find_reg_note (p, REG_LIBCALL, NULL_RTX)))
	p = XEXP (temp, 0);

      if (code == INSN
	  && (set = single_set (p))
	  && GET_CODE (SET_DEST (set)) == REG
	  && SET_DEST (set) == dest_reg
	  && ((benefit = general_induction_var (SET_SRC (set), &src_reg,
						add_val, mult_val))
	      /* Giv created by equivalent expression.  */
	      || ((temp = find_reg_note (p, REG_EQUAL, NULL_RTX))
		  && (benefit = general_induction_var (XEXP (temp, 0), &src_reg,
						       add_val, mult_val))))
	  && src_reg == v->src_reg)
	{
	  if (find_reg_note (p, REG_RETVAL, NULL_RTX))
	    benefit += libcall_benefit (p);

	  count--;
	  v->mult_val = *mult_val;
	  v->add_val = *add_val;
	  v->benefit = benefit;
	}
      else if (code != NOTE)
	{
	  /* Allow insns that set something other than this giv to a
	     constant.  Such insns are needed on machines which cannot
	     include long constants and should not disqualify a giv.  */
	  if (code == INSN
	      && (set = single_set (p))
	      && SET_DEST (set) != dest_reg
	      && CONSTANT_P (SET_SRC (set)))
	    continue;

	  reg_iv_type[REGNO (dest_reg)] = UNKNOWN_INDUCT;
	  return 0;
	}
    }

  return v->benefit;
}

/* Return an rtx, if any, that expresses giv G2 as a function of the register
   represented by G1.  If no such expression can be found, or it is clear that
   it cannot possibly be a valid address, 0 is returned. 

   To perform the computation, we note that
   	G1 = a * v + b		and
	G2 = c * v + d
   where `v' is the biv.

   So G2 = (c/a) * G1 + (d - b*c/a)  */

#ifdef ADDRESS_COST
static rtx
express_from (g1, g2)
     struct induction *g1, *g2;
{
  rtx mult, add;

  /* The value that G1 will be multiplied by must be a constant integer.  Also,
     the only chance we have of getting a valid address is if b*c/a (see above
     for notation) is also an integer.  */
  if (GET_CODE (g1->mult_val) != CONST_INT
      || GET_CODE (g2->mult_val) != CONST_INT
      || GET_CODE (g1->add_val) != CONST_INT
      || g1->mult_val == const0_rtx
      || INTVAL (g2->mult_val) % INTVAL (g1->mult_val) != 0)
    return 0;

  mult = GEN_INT (INTVAL (g2->mult_val) / INTVAL (g1->mult_val));
  add = plus_constant (g2->add_val, - INTVAL (g1->add_val) * INTVAL (mult));

  /* Form simplified final result.  */
  if (mult == const0_rtx)
    return add;
  else if (mult == const1_rtx)
    mult = g1->dest_reg;
  else
    mult = gen_rtx (MULT, g2->mode, g1->dest_reg, mult);

  if (add == const0_rtx)
    return mult;
  else
    return gen_rtx (PLUS, g2->mode, mult, add);
}
#endif

/* Return 1 if giv G2 can be combined with G1.  This means that G2 can use
   (either directly or via an address expression) a register used to represent
   G1.  Set g2->new_reg to a represtation of G1 (normally just
   g1->dest_reg).  */

static int
combine_givs_p (g1, g2)
     struct induction *g1, *g2;
{
  rtx tem;

  /* If these givs are identical, they can be combined.  */
  if (rtx_equal_p (g1->mult_val, g2->mult_val)
      && rtx_equal_p (g1->add_val, g2->add_val))
    {
      g2->new_reg = g1->dest_reg;
      return 1;
    }

#ifdef ADDRESS_COST
  /* If G2 can be expressed as a function of G1 and that function is valid
     as an address and no more expensive than using a register for G2,
     the expression of G2 in terms of G1 can be used.  */
  if (g2->giv_type == DEST_ADDR
      && (tem = express_from (g1, g2)) != 0
/* intel1 */
#ifdef REDUCE_INDEX
      && ((g2->is_index_induc && LEGITIMATE_INDEX_P(tem)) 
           || (g2->is_index_induc == 0 && memory_address_p (g2->mem_mode, tem)
           && ADDRESS_COST (tem) <= ADDRESS_COST (*g2->location))
         )
#else
      && memory_address_p (g2->mem_mode, tem)
      && ADDRESS_COST (tem) <= ADDRESS_COST (*g2->location)
#endif
     )
    {
/* intel1 */
#ifdef SAVE_ON_REGS
      if (flag_opt_reg_use && ADDRESS_COST (tem) != 1)
        return 0;
#endif
      g2->new_reg = tem;
      return 1;
    }
#endif

  return 0;
}

#ifdef GIV_SORT_CRITERION
/* Compare two givs and sort the most desirable one for combinations first.
   This is used only in one qsort call below.  */

static int
giv_sort (x, y)
     struct induction **x, **y;
{
  GIV_SORT_CRITERION (*x, *y);

  return 0;
}
#endif

/* Check all pairs of givs for iv_class BL and see if any can be combined with
   any other.  If so, point SAME to the giv combined with and set NEW_REG to
   be an expression (in terms of the other giv's DEST_REG) equivalent to the
   giv.  Also, update BENEFIT and related fields for cost/benefit analysis.  */

static void
combine_givs (bl)
     struct iv_class *bl;
{
  struct induction *g1, *g2, **giv_array, *temp_iv;
  int i, j, giv_count, pass;

  /* Count givs, because bl->giv_count is incorrect here.  */
  giv_count = 0;
  for (g1 = bl->giv; g1; g1 = g1->next_iv)
    giv_count++;

  giv_array
    = (struct induction **) alloca (giv_count * sizeof (struct induction *));
  i = 0;
  for (g1 = bl->giv; g1; g1 = g1->next_iv)
    giv_array[i++] = g1;

#ifdef GIV_SORT_CRITERION
  /* Sort the givs if GIV_SORT_CRITERION is defined.
     This is usually defined for processors which lack
     negative register offsets so more givs may be combined.  */

  if (loop_dump_stream)
    fprintf (loop_dump_stream, "%d givs counted, sorting...\n", giv_count);

  qsort (giv_array, giv_count, sizeof (struct induction *), giv_sort);
#endif

  for (i = 0; i < giv_count; i++)
    {
      g1 = giv_array[i];
      for (pass = 0; pass <= 1; pass++)
	for (j = 0; j < giv_count; j++)
	  {
	    g2 = giv_array[j];
	    if (g1 != g2
		/* First try to combine with replaceable givs, then all givs.  */
		&& (g1->replaceable || pass == 1)
		/* If either has already been combined or is to be ignored, can't
		   combine.  */
		&& ! g1->ignore && ! g2->ignore && ! g1->same && ! g2->same
		/* If something has been based on G2, G2 cannot itself be based
		   on something else.  */
		&& ! g2->combined_with
		&& combine_givs_p (g1, g2))
	      {
		/* g2->new_reg set by `combine_givs_p'  */
		g2->same = g1;
		g1->combined_with = 1;

		/* If one of these givs is a DEST_REG that was only used
		   once, by the other giv, this is actually a single use.
		   The DEST_REG has the correct cost, while the other giv
		   counts the REG use too often.  */
		if (g2->giv_type == DEST_REG
		    && n_times_used[REGNO (g2->dest_reg)] == 1
		    && reg_mentioned_p (g2->dest_reg, PATTERN (g1->insn)))
		  g1->benefit = g2->benefit;
		else if (g1->giv_type != DEST_REG
			 || n_times_used[REGNO (g1->dest_reg)] != 1
			 || ! reg_mentioned_p (g1->dest_reg,
					       PATTERN (g2->insn)))
		  {
		    g1->benefit += g2->benefit;
		    g1->times_used += g2->times_used;
		  }
		/* ??? The new final_[bg]iv_value code does a much better job
		   of finding replaceable giv's, and hence this code may no
		   longer be necessary.  */
		if (! g2->replaceable && REG_USERVAR_P (g2->dest_reg))
		  g1->benefit -= copy_cost;
		g1->lifetime += g2->lifetime;
		
		if (loop_dump_stream)
		  fprintf (loop_dump_stream, "giv at %d combined with giv at %d\n",
			   INSN_UID (g2->insn), INSN_UID (g1->insn));
	      }
	  }
    }
}

/* EMIT code before INSERT_BEFORE to set REG = B * M + A.  */

void
emit_iv_add_mult (b, m, a, reg, insert_before)
     rtx b;          /* initial value of basic induction variable */
     rtx m;          /* multiplicative constant */
     rtx a;          /* additive constant */
     rtx reg;        /* destination register */
     rtx insert_before;
{
  rtx seq;
  rtx result;

  /* Prevent unexpected sharing of these rtx.  */
  a = copy_rtx (a);
  b = copy_rtx (b);

  /* Increase the lifetime of any invariants moved further in code.  */
  update_reg_last_use (a, insert_before);
  update_reg_last_use (b, insert_before);
  update_reg_last_use (m, insert_before);

  start_sequence ();
  result = expand_mult_add (b, reg, m, a, GET_MODE (reg), 0);
  if (reg != result)
    emit_move_insn (reg, result);
  seq = gen_sequence ();
  end_sequence ();

  emit_insn_before (seq, insert_before);
}

/* Test whether A * B can be computed without
   an actual multiply insn.  Value is 1 if so.  */

static int
product_cheap_p (a, b)
     rtx a;
     rtx b;
{
  int i;
  rtx tmp;
  struct obstack *old_rtl_obstack = rtl_obstack;
  char *storage = (char *) obstack_alloc (&temp_obstack, 0);
  int win = 1;

  /* If only one is constant, make it B.  */
  if (GET_CODE (a) == CONST_INT)
    tmp = a, a = b, b = tmp;

  /* If first constant, both constant, so don't need multiply.  */
  if (GET_CODE (a) == CONST_INT)
    return 1;

  /* If second not constant, neither is constant, so would need multiply.  */
  if (GET_CODE (b) != CONST_INT)
    return 0;

  /* One operand is constant, so might not need multiply insn.  Generate the
     code for the multiply and see if a call or multiply, or long sequence
     of insns is generated.  */

  rtl_obstack = &temp_obstack;
  start_sequence ();
  expand_mult (GET_MODE (a), a, b, NULL_RTX, 0);
  tmp = gen_sequence ();
  end_sequence ();

  if (GET_CODE (tmp) == SEQUENCE)
    {
      if (XVEC (tmp, 0) == 0)
	win = 1;
      else if (XVECLEN (tmp, 0) > 3)
	win = 0;
      else
	for (i = 0; i < XVECLEN (tmp, 0); i++)
	  {
	    rtx insn = XVECEXP (tmp, 0, i);

	    if (GET_CODE (insn) != INSN
		|| (GET_CODE (PATTERN (insn)) == SET
		    && GET_CODE (SET_SRC (PATTERN (insn))) == MULT)
		|| (GET_CODE (PATTERN (insn)) == PARALLEL
		    && GET_CODE (XVECEXP (PATTERN (insn), 0, 0)) == SET
		    && GET_CODE (SET_SRC (XVECEXP (PATTERN (insn), 0, 0))) == MULT))
	      {
		win = 0;
		break;
	      }
	  }
    }
  else if (GET_CODE (tmp) == SET
	   && GET_CODE (SET_SRC (tmp)) == MULT)
    win = 0;
  else if (GET_CODE (tmp) == PARALLEL
	   && GET_CODE (XVECEXP (tmp, 0, 0)) == SET
	   && GET_CODE (SET_SRC (XVECEXP (tmp, 0, 0))) == MULT)
    win = 0;

  /* Free any storage we obtained in generating this multiply and restore rtl
     allocation to its normal obstack.  */
  obstack_free (&temp_obstack, storage);
  rtl_obstack = old_rtl_obstack;

  return win;
}

/* Check to see if loop can be terminated by a "decrement and branch until
   zero" instruction.  If so, add a REG_NONNEG note to the branch insn if so.
   Also try reversing an increment loop to a decrement loop
   to see if the optimization can be performed.
   Value is nonzero if optimization was performed.  */

/* This is useful even if the architecture doesn't have such an insn,
   because it might change a loops which increments from 0 to n to a loop
   which decrements from n to 0.  A loop that decrements to zero is usually
   faster than one that increments from zero.  */

/* ??? This could be rewritten to use some of the loop unrolling procedures,
   such as approx_final_value, biv_total_increment, loop_iterations, and
   final_[bg]iv_value.  */

static int
check_dbra_loop (loop_end, insn_count, loop_start)
     rtx loop_end;
     int insn_count;
     rtx loop_start;
{
  struct iv_class *bl;
  rtx reg;
  rtx jump_label;
  rtx final_value;
  rtx start_value;
  rtx new_add_val;
  rtx comparison;
  rtx before_comparison;
  rtx p;

  /* If last insn is a conditional branch, and the insn before tests a
     register value, try to optimize it.  Otherwise, we can't do anything.  */

  comparison = get_condition_for_loop (PREV_INSN (loop_end));
  if (comparison == 0)
    return 0;

  /* Check all of the bivs to see if the compare uses one of them.
     Skip biv's set more than once because we can't guarantee that
     it will be zero on the last iteration.  Also skip if the biv is
     used between its update and the test insn.  */

  for (bl = loop_iv_list; bl; bl = bl->next)
    {
      if (bl->biv_count == 1
	  && bl->biv->dest_reg == XEXP (comparison, 0)
	  && ! reg_used_between_p (regno_reg_rtx[bl->regno], bl->biv->insn,
				   PREV_INSN (PREV_INSN (loop_end))))
	break;
    }

  if (! bl)
    return 0;

  /* Look for the case where the basic induction variable is always
     nonnegative, and equals zero on the last iteration.
     In this case, add a reg_note REG_NONNEG, which allows the
     m68k DBRA instruction to be used.  */

  if (((GET_CODE (comparison) == GT
	&& GET_CODE (XEXP (comparison, 1)) == CONST_INT
	&& INTVAL (XEXP (comparison, 1)) == -1)
       || (GET_CODE (comparison) == NE && XEXP (comparison, 1) == const0_rtx))
      && GET_CODE (bl->biv->add_val) == CONST_INT
      && INTVAL (bl->biv->add_val) < 0)
    {
      /* Initial value must be greater than 0,
	 init_val % -dec_value == 0 to ensure that it equals zero on
	 the last iteration */

      if (GET_CODE (bl->initial_value) == CONST_INT
	  && INTVAL (bl->initial_value) > 0
	  && (INTVAL (bl->initial_value) %
	      (-INTVAL (bl->biv->add_val))) == 0)
	{
	  /* register always nonnegative, add REG_NOTE to branch */
	  REG_NOTES (PREV_INSN (loop_end))
	    = gen_rtx (EXPR_LIST, REG_NONNEG, NULL_RTX,
		       REG_NOTES (PREV_INSN (loop_end)));
	  bl->nonneg = 1;

	  return 1;
	}

      /* If the decrement is 1 and the value was tested as >= 0 before
	 the loop, then we can safely optimize.  */
      for (p = loop_start; p; p = PREV_INSN (p))
	{
	  if (GET_CODE (p) == CODE_LABEL)
	    break;
	  if (GET_CODE (p) != JUMP_INSN)
	    continue;

	  before_comparison = get_condition_for_loop (p);
	  if (before_comparison
	      && XEXP (before_comparison, 0) == bl->biv->dest_reg
	      && GET_CODE (before_comparison) == LT
	      && XEXP (before_comparison, 1) == const0_rtx
	      && ! reg_set_between_p (bl->biv->dest_reg, p, loop_start)
	      && INTVAL (bl->biv->add_val) == -1)
	    {
	      REG_NOTES (PREV_INSN (loop_end))
		= gen_rtx (EXPR_LIST, REG_NONNEG, NULL_RTX,
			   REG_NOTES (PREV_INSN (loop_end)));
	      bl->nonneg = 1;

	      return 1;
	    }
	}
    }
  else if (num_mem_sets <= 1)
    {
      /* Try to change inc to dec, so can apply above optimization.  */
      /* Can do this if:
	 all registers modified are induction variables or invariant,
	 all memory references have non-overlapping addresses
	 (obviously true if only one write)
	 allow 2 insns for the compare/jump at the end of the loop.  */
      /* Also, we must avoid any instructions which use both the reversed
	 biv and another biv.  Such instructions will fail if the loop is
	 reversed.  We meet this condition by requiring that either
	 no_use_except_counting is true, or else that there is only
	 one biv.  */
      int num_nonfixed_reads = 0;
      /* 1 if the iteration var is used only to count iterations.  */
      int no_use_except_counting = 0;
      /* 1 if the loop has no memory store, or it has a single memory store
	 which is reversible.  */
      int reversible_mem_store = 1;

      for (p = loop_start; p != loop_end; p = NEXT_INSN (p))
	if (GET_RTX_CLASS (GET_CODE (p)) == 'i')
	  num_nonfixed_reads += count_nonfixed_reads (PATTERN (p));

      if (bl->giv_count == 0
	  && ! loop_number_exit_count[uid_loop_num[INSN_UID (loop_start)]])
	{
	  rtx bivreg = regno_reg_rtx[bl->regno];

	  /* If there are no givs for this biv, and the only exit is the
	     fall through at the end of the the loop, then
	     see if perhaps there are no uses except to count.  */
	  no_use_except_counting = 1;
	  for (p = loop_start; p != loop_end; p = NEXT_INSN (p))
	    if (GET_RTX_CLASS (GET_CODE (p)) == 'i')
	      {
		rtx set = single_set (p);

		if (set && GET_CODE (SET_DEST (set)) == REG
		    && REGNO (SET_DEST (set)) == bl->regno)
		  /* An insn that sets the biv is okay.  */
		  ;
		else if (p == prev_nonnote_insn (prev_nonnote_insn (loop_end))
			 || p == prev_nonnote_insn (loop_end))
		  /* Don't bother about the end test.  */
		  ;
		else if (reg_mentioned_p (bivreg, PATTERN (p)))
		  /* Any other use of the biv is no good.  */
		  {
		    no_use_except_counting = 0;
		    break;
		  }
	      }
	}

      /* If the loop has a single store, and the destination address is
	 invariant, then we can't reverse the loop, because this address
	 might then have the wrong value at loop exit.
	 This would work if the source was invariant also, however, in that
	 case, the insn should have been moved out of the loop.  */

      if (num_mem_sets == 1)
	reversible_mem_store
	  = (! unknown_address_altered
	     && ! invariant_p (XEXP (loop_store_mems[0], 0)));

      /* This code only acts for innermost loops.  Also it simplifies
	 the memory address check by only reversing loops with
	 zero or one memory access.
	 Two memory accesses could involve parts of the same array,
	 and that can't be reversed.  */

      if (num_nonfixed_reads <= 1
	  && !loop_has_call
	  && !loop_has_volatile
	  && reversible_mem_store
	  && (no_use_except_counting
	      || ((bl->giv_count + bl->biv_count + num_mem_sets
		   + num_movables + 2 == insn_count)
		  && (bl == loop_iv_list && bl->next == 0))))
	{
	  rtx tem;

	  /* Loop can be reversed.  */
	  if (loop_dump_stream)
	    fprintf (loop_dump_stream, "Can reverse loop\n");

	  /* Now check other conditions:
	     initial_value must be zero,
	     final_value % add_val == 0, so that when reversed, the
	     biv will be zero on the last iteration.

	     This test can probably be improved since +/- 1 in the constant
	     can be obtained by changing LT to LE and vice versa; this is
	     confusing.  */

	  if (comparison && bl->initial_value == const0_rtx
	      && GET_CODE (XEXP (comparison, 1)) == CONST_INT
	      /* LE gets turned into LT */
	      && GET_CODE (comparison) == LT
	      && (INTVAL (XEXP (comparison, 1))
		  % INTVAL (bl->biv->add_val)) == 0)
	    {
	      /* Register will always be nonnegative, with value
		 0 on last iteration if loop reversed */

	      /* Save some info needed to produce the new insns.  */
	      reg = bl->biv->dest_reg;
	      jump_label = XEXP (SET_SRC (PATTERN (PREV_INSN (loop_end))), 1);
	      if (jump_label == pc_rtx)
		jump_label = XEXP (SET_SRC (PATTERN (PREV_INSN (loop_end))), 2);
	      new_add_val = GEN_INT (- INTVAL (bl->biv->add_val));

	      final_value = XEXP (comparison, 1);
	      start_value = GEN_INT (INTVAL (XEXP (comparison, 1))
				     - INTVAL (bl->biv->add_val));

	      /* Initialize biv to start_value before loop start.
		 The old initializing insn will be deleted as a
		 dead store by flow.c.  */
	      emit_insn_before (gen_move_insn (reg, start_value), loop_start);

	      /* Add insn to decrement register, and delete insn
		 that incremented the register.  */
	      p = emit_insn_before (gen_add2_insn (reg, new_add_val),
				    bl->biv->insn);
	      delete_insn (bl->biv->insn);
		      
	      /* Update biv info to reflect its new status.  */
	      bl->biv->insn = p;
	      bl->initial_value = start_value;
	      bl->biv->add_val = new_add_val;

	      /* Inc LABEL_NUSES so that delete_insn will
		 not delete the label.  */
	      LABEL_NUSES (XEXP (jump_label, 0)) ++;

	      /* Emit an insn after the end of the loop to set the biv's
		 proper exit value if it is used anywhere outside the loop.  */
	      if ((regno_last_uid[bl->regno]
		   != INSN_UID (PREV_INSN (PREV_INSN (loop_end))))
		  || ! bl->init_insn
		  || regno_first_uid[bl->regno] != INSN_UID (bl->init_insn))
		emit_insn_after (gen_move_insn (reg, final_value),
				 loop_end);

	      /* Delete compare/branch at end of loop.  */
	      delete_insn (PREV_INSN (loop_end));
	      delete_insn (PREV_INSN (loop_end));

	      /* Add new compare/branch insn at end of loop.  */
	      start_sequence ();
	      emit_cmp_insn (reg, const0_rtx, GE, NULL_RTX,
			     GET_MODE (reg), 0, 0);
	      emit_jump_insn (gen_bge (XEXP (jump_label, 0)));
	      tem = gen_sequence ();
	      end_sequence ();
	      emit_jump_insn_before (tem, loop_end);

	      for (tem = PREV_INSN (loop_end);
		   tem && GET_CODE (tem) != JUMP_INSN; tem = PREV_INSN (tem))
		;
	      if (tem)
		{
		  JUMP_LABEL (tem) = XEXP (jump_label, 0);

		  /* Increment of LABEL_NUSES done above.  */
		  /* Register is now always nonnegative,
		     so add REG_NONNEG note to the branch.  */
		  REG_NOTES (tem) = gen_rtx (EXPR_LIST, REG_NONNEG, NULL_RTX,
					     REG_NOTES (tem));
		}

	      bl->nonneg = 1;

	      /* Mark that this biv has been reversed.  Each giv which depends
		 on this biv, and which is also live past the end of the loop
		 will have to be fixed up.  */

	      bl->reversed = 1;

	      if (loop_dump_stream)
		fprintf (loop_dump_stream,
			 "Reversed loop and added reg_nonneg\n");

	      return 1;
	    }
	}
    }

  return 0;
}

/* Verify whether the biv BL appears to be eliminable,
   based on the insns in the loop that refer to it.
   LOOP_START is the first insn of the loop, and END is the end insn.

   If ELIMINATE_P is non-zero, actually do the elimination.

   THRESHOLD and INSN_COUNT are from loop_optimize and are used to
   determine whether invariant insns should be placed inside or at the
   start of the loop.  */

static int
maybe_eliminate_biv (bl, loop_start, end, eliminate_p, threshold, insn_count)
     struct iv_class *bl;
     rtx loop_start;
     rtx end;
     int eliminate_p;
     int threshold, insn_count;
{
  rtx reg = bl->biv->dest_reg;
  rtx p;

  /* Scan all insns in the loop, stopping if we find one that uses the
     biv in a way that we cannot eliminate.  */

  for (p = loop_start; p != end; p = NEXT_INSN (p))
    {
      enum rtx_code code = GET_CODE (p);
      rtx where = threshold >= insn_count ? loop_start : p;

      if ((code == INSN || code == JUMP_INSN || code == CALL_INSN)
	  && reg_mentioned_p (reg, PATTERN (p))
	  && ! maybe_eliminate_biv_1 (PATTERN (p), p, bl, eliminate_p, where))
	{
	  if (loop_dump_stream)
	    fprintf (loop_dump_stream,
		     "Cannot eliminate biv %d: biv used in insn %d.\n",
		     bl->regno, INSN_UID (p));
	  break;
	}
    }

  if (p == end)
    {
      if (loop_dump_stream)
	fprintf (loop_dump_stream, "biv %d %s eliminated.\n",
		 bl->regno, eliminate_p ? "was" : "can be");
      return 1;
    }

  return 0;
}

/* If BL appears in X (part of the pattern of INSN), see if we can
   eliminate its use.  If so, return 1.  If not, return 0.

   If BIV does not appear in X, return 1.

   If ELIMINATE_P is non-zero, actually do the elimination.  WHERE indicates
   where extra insns should be added.  Depending on how many items have been
   moved out of the loop, it will either be before INSN or at the start of
   the loop.  */

static int
maybe_eliminate_biv_1 (x, insn, bl, eliminate_p, where)
     rtx x, insn;
     struct iv_class *bl;
     int eliminate_p;
     rtx where;
{
  enum rtx_code code = GET_CODE (x);
  rtx reg = bl->biv->dest_reg;
  enum machine_mode mode = GET_MODE (reg);
  struct induction *v;
  rtx arg, new, tem;
  int arg_operand;
  char *fmt;
  int i, j;

  switch (code)
    {
    case REG:
      /* If we haven't already been able to do something with this BIV,
	 we can't eliminate it.  */
      if (x == reg)
	return 0;
      return 1;

    case SET:
      /* If this sets the BIV, it is not a problem.  */
      if (SET_DEST (x) == reg)
	return 1;

      /* If this is an insn that defines a giv, it is also ok because
	 it will go away when the giv is reduced.  */
      for (v = bl->giv; v; v = v->next_iv)
	if (v->giv_type == DEST_REG && SET_DEST (x) == v->dest_reg)
	  return 1;

#ifdef HAVE_cc0
      if (SET_DEST (x) == cc0_rtx && SET_SRC (x) == reg)
	{
	  /* Can replace with any giv that was reduced and
	     that has (MULT_VAL != 0) and (ADD_VAL == 0).
	     Require a constant for MULT_VAL, so we know it's nonzero.
	     ??? We disable this optimization to avoid potential
	     overflows.  */

	  for (v = bl->giv; v; v = v->next_iv)
	    if (CONSTANT_P (v->mult_val) && v->mult_val != const0_rtx
		&& v->add_val == const0_rtx
		&& ! v->ignore && ! v->maybe_dead && v->always_computable
		&& v->mode == mode
		&& 0)
	      {
		/* If the giv V had the auto-inc address optimization applied
		   to it, and INSN occurs between the giv insn and the biv
		   insn, then we must adjust the value used here.
		   This is rare, so we don't bother to do so.  */
		if (v->auto_inc_opt
		    && ((INSN_LUID (v->insn) < INSN_LUID (insn)
			 && INSN_LUID (insn) < INSN_LUID (bl->biv->insn))
			|| (INSN_LUID (v->insn) > INSN_LUID (insn)
			    && INSN_LUID (insn) > INSN_LUID (bl->biv->insn))))
		  continue;

		if (! eliminate_p)
		  return 1;

		/* If the giv has the opposite direction of change,
		   then reverse the comparison.  */
		if (INTVAL (v->mult_val) < 0)
		  new = gen_rtx (COMPARE, GET_MODE (v->new_reg),
				 const0_rtx, v->new_reg);
		else
		  new = v->new_reg;

		/* We can probably test that giv's reduced reg.  */
		if (validate_change (insn, &SET_SRC (x), new, 0))
		  return 1;
	      }

	  /* Look for a giv with (MULT_VAL != 0) and (ADD_VAL != 0);
	     replace test insn with a compare insn (cmp REDUCED_GIV ADD_VAL).
	     Require a constant for MULT_VAL, so we know it's nonzero.
	     ??? Do this only if ADD_VAL is a pointer to avoid a potential
	     overflow problem.  */

	  for (v = bl->giv; v; v = v->next_iv)
	    if (CONSTANT_P (v->mult_val) && v->mult_val != const0_rtx
		&& ! v->ignore && ! v->maybe_dead && v->always_computable
		&& v->mode == mode
		&& (GET_CODE (v->add_val) == SYMBOL_REF
		    || GET_CODE (v->add_val) == LABEL_REF
		    || GET_CODE (v->add_val) == CONST
		    || (GET_CODE (v->add_val) == REG
			&& REGNO_POINTER_FLAG (REGNO (v->add_val)))))
	      {
		/* If the giv V had the auto-inc address optimization applied
		   to it, and INSN occurs between the giv insn and the biv
		   insn, then we must adjust the value used here.
		   This is rare, so we don't bother to do so.  */
		if (v->auto_inc_opt
		    && ((INSN_LUID (v->insn) < INSN_LUID (insn)
			 && INSN_LUID (insn) < INSN_LUID (bl->biv->insn))
			|| (INSN_LUID (v->insn) > INSN_LUID (insn)
			    && INSN_LUID (insn) > INSN_LUID (bl->biv->insn))))
		  continue;

		if (! eliminate_p)
		  return 1;

		/* If the giv has the opposite direction of change,
		   then reverse the comparison.  */
		if (INTVAL (v->mult_val) < 0)
		  new = gen_rtx (COMPARE, VOIDmode, copy_rtx (v->add_val),
				 v->new_reg);
		else
		  new = gen_rtx (COMPARE, VOIDmode, v->new_reg,
				 copy_rtx (v->add_val));

		/* Replace biv with the giv's reduced register.  */
		update_reg_last_use (v->add_val, insn);
		if (validate_change (insn, &SET_SRC (PATTERN (insn)), new, 0))
		  return 1;

		/* Insn doesn't support that constant or invariant.  Copy it
		   into a register (it will be a loop invariant.)  */
		tem = gen_reg_rtx (GET_MODE (v->new_reg));

		emit_insn_before (gen_move_insn (tem, copy_rtx (v->add_val)),
				  where);

		if (validate_change (insn, &SET_SRC (PATTERN (insn)),
				     gen_rtx (COMPARE, VOIDmode,
					      v->new_reg, tem), 0))
		  return 1;
	      }
	}
#endif
      break;

    case COMPARE:
    case EQ:  case NE:
    case GT:  case GE:  case GTU:  case GEU:
    case LT:  case LE:  case LTU:  case LEU:
      /* See if either argument is the biv.  */
      if (XEXP (x, 0) == reg)
	arg = XEXP (x, 1), arg_operand = 1;
      else if (XEXP (x, 1) == reg)
	arg = XEXP (x, 0), arg_operand = 0;
      else
	break;

      if (CONSTANT_P (arg))
	{
	  /* First try to replace with any giv that has constant positive
	     mult_val and constant add_val.  We might be able to support
	     negative mult_val, but it seems complex to do it in general.  */

	  for (v = bl->giv; v; v = v->next_iv)
	    if (CONSTANT_P (v->mult_val) && INTVAL (v->mult_val) > 0
		&& (GET_CODE (v->add_val) == SYMBOL_REF
		    || GET_CODE (v->add_val) == LABEL_REF
		    || GET_CODE (v->add_val) == CONST
		    || (GET_CODE (v->add_val) == REG
			&& REGNO_POINTER_FLAG (REGNO (v->add_val))))
		&& ! v->ignore && ! v->maybe_dead && v->always_computable
		&& v->mode == mode)
	      {
		/* If the giv V had the auto-inc address optimization applied
		   to it, and INSN occurs between the giv insn and the biv
		   insn, then we must adjust the value used here.
		   This is rare, so we don't bother to do so.  */
		if (v->auto_inc_opt
		    && ((INSN_LUID (v->insn) < INSN_LUID (insn)
			 && INSN_LUID (insn) < INSN_LUID (bl->biv->insn))
			|| (INSN_LUID (v->insn) > INSN_LUID (insn)
			    && INSN_LUID (insn) > INSN_LUID (bl->biv->insn))))
		  continue;

		if (! eliminate_p)
		  return 1;

		/* Replace biv with the giv's reduced reg.  */
		XEXP (x, 1-arg_operand) = v->new_reg;

		/* If all constants are actually constant integers and
		   the derived constant can be directly placed in the COMPARE,
		   do so.  */
		if (GET_CODE (arg) == CONST_INT
		    && GET_CODE (v->mult_val) == CONST_INT
		    && GET_CODE (v->add_val) == CONST_INT
		    && validate_change (insn, &XEXP (x, arg_operand),
					GEN_INT (INTVAL (arg)
						 * INTVAL (v->mult_val)
						 + INTVAL (v->add_val)), 0))
		  return 1;

		/* Otherwise, load it into a register.  */
		tem = gen_reg_rtx (mode);
		emit_iv_add_mult (arg, v->mult_val, v->add_val, tem, where);
		if (validate_change (insn, &XEXP (x, arg_operand), tem, 0))
		  return 1;

		/* If that failed, put back the change we made above.  */
		XEXP (x, 1-arg_operand) = reg;
	      }
	  
	  /* Look for giv with positive constant mult_val and nonconst add_val.
	     Insert insns to calculate new compare value.  
	     ??? Turn this off due to possible overflow.  */

	  for (v = bl->giv; v; v = v->next_iv)
	    if (CONSTANT_P (v->mult_val) && INTVAL (v->mult_val) > 0
		&& ! v->ignore && ! v->maybe_dead && v->always_computable
		&& v->mode == mode
		&& 0)
	      {
		rtx tem;

		/* If the giv V had the auto-inc address optimization applied
		   to it, and INSN occurs between the giv insn and the biv
		   insn, then we must adjust the value used here.
		   This is rare, so we don't bother to do so.  */
		if (v->auto_inc_opt
		    && ((INSN_LUID (v->insn) < INSN_LUID (insn)
			 && INSN_LUID (insn) < INSN_LUID (bl->biv->insn))
			|| (INSN_LUID (v->insn) > INSN_LUID (insn)
			    && INSN_LUID (insn) > INSN_LUID (bl->biv->insn))))
		  continue;

		if (! eliminate_p)
		  return 1;

		tem = gen_reg_rtx (mode);

		/* Replace biv with giv's reduced register.  */
		validate_change (insn, &XEXP (x, 1 - arg_operand),
				 v->new_reg, 1);

		/* Compute value to compare against.  */
		emit_iv_add_mult (arg, v->mult_val, v->add_val, tem, where);
		/* Use it in this insn.  */
		validate_change (insn, &XEXP (x, arg_operand), tem, 1);
		if (apply_change_group ())
		  return 1;
	      }
	}
      else if (GET_CODE (arg) == REG || GET_CODE (arg) == MEM)
	{
	  if (invariant_p (arg) == 1)
	    {
	      /* Look for giv with constant positive mult_val and nonconst
		 add_val. Insert insns to compute new compare value. 
		 ??? Turn this off due to possible overflow.  */

	      for (v = bl->giv; v; v = v->next_iv)
		if (CONSTANT_P (v->mult_val) && INTVAL (v->mult_val) > 0
		    && ! v->ignore && ! v->maybe_dead && v->always_computable
		    && v->mode == mode
		    && 0)
		  {
		    rtx tem;

		    /* If the giv V had the auto-inc address optimization applied
		       to it, and INSN occurs between the giv insn and the biv
		       insn, then we must adjust the value used here.
		       This is rare, so we don't bother to do so.  */
		    if (v->auto_inc_opt
			&& ((INSN_LUID (v->insn) < INSN_LUID (insn)
			     && INSN_LUID (insn) < INSN_LUID (bl->biv->insn))
			    || (INSN_LUID (v->insn) > INSN_LUID (insn)
				&& INSN_LUID (insn) > INSN_LUID (bl->biv->insn))))
		      continue;

		    if (! eliminate_p)
		      return 1;

		    tem = gen_reg_rtx (mode);

		    /* Replace biv with giv's reduced register.  */
		    validate_change (insn, &XEXP (x, 1 - arg_operand),
				     v->new_reg, 1);

		    /* Compute value to compare against.  */
		    emit_iv_add_mult (arg, v->mult_val, v->add_val,
				      tem, where);
		    validate_change (insn, &XEXP (x, arg_operand), tem, 1);
		    if (apply_change_group ())
		      return 1;
		  }
	    }

	  /* This code has problems.  Basically, you can't know when
	     seeing if we will eliminate BL, whether a particular giv
	     of ARG will be reduced.  If it isn't going to be reduced,
	     we can't eliminate BL.  We can try forcing it to be reduced,
	     but that can generate poor code.

	     The problem is that the benefit of reducing TV, below should
	     be increased if BL can actually be eliminated, but this means
	     we might have to do a topological sort of the order in which
	     we try to process biv.  It doesn't seem worthwhile to do
	     this sort of thing now.  */

#if 0
	  /* Otherwise the reg compared with had better be a biv.  */
	  if (GET_CODE (arg) != REG
	      || reg_iv_type[REGNO (arg)] != BASIC_INDUCT)
	    return 0;

	  /* Look for a pair of givs, one for each biv,
	     with identical coefficients.  */
	  for (v = bl->giv; v; v = v->next_iv)
	    {
	      struct induction *tv;

	      if (v->ignore || v->maybe_dead || v->mode != mode)
		continue;

	      for (tv = reg_biv_class[REGNO (arg)]->giv; tv; tv = tv->next_iv)
		if (! tv->ignore && ! tv->maybe_dead
		    && rtx_equal_p (tv->mult_val, v->mult_val)
		    && rtx_equal_p (tv->add_val, v->add_val)
		    && tv->mode == mode)
		  {
		    /* If the giv V had the auto-inc address optimization applied
		       to it, and INSN occurs between the giv insn and the biv
		       insn, then we must adjust the value used here.
		       This is rare, so we don't bother to do so.  */
		    if (v->auto_inc_opt
			&& ((INSN_LUID (v->insn) < INSN_LUID (insn)
			     && INSN_LUID (insn) < INSN_LUID (bl->biv->insn))
			    || (INSN_LUID (v->insn) > INSN_LUID (insn)
				&& INSN_LUID (insn) > INSN_LUID (bl->biv->insn))))
		      continue;

		    if (! eliminate_p)
		      return 1;

		    /* Replace biv with its giv's reduced reg.  */
		    XEXP (x, 1-arg_operand) = v->new_reg;
		    /* Replace other operand with the other giv's
		       reduced reg.  */
		    XEXP (x, arg_operand) = tv->new_reg;
		    return 1;
		  }
	    }
#endif
	}

      /* If we get here, the biv can't be eliminated.  */
      return 0;

    case MEM:
      /* If this address is a DEST_ADDR giv, it doesn't matter if the
	 biv is used in it, since it will be replaced.  */
      for (v = bl->giv; v; v = v->next_iv)
	if (v->giv_type == DEST_ADDR && v->location == &XEXP (x, 0))
	  return 1;
/* intel1 */
#ifdef REDUCE_INDEX
#ifdef LEGITIMATE_INDEX_P
      for (v = bl->giv; v; v = v->next_iv)
	if (REDUCE_INDEX && v->giv_type == DEST_ADDR && v->is_index_induc 
	    && v->location == &XEXP (XEXP(x,0), 0))
	  return 1;
#endif
#endif
      break;
    }

  /* See if any subexpression fails elimination.  */
  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      switch (fmt[i])
	{
	case 'e':
	  if (! maybe_eliminate_biv_1 (XEXP (x, i), insn, bl, 
				       eliminate_p, where))
	    return 0;
	  break;

	case 'E':
	  for (j = XVECLEN (x, i) - 1; j >= 0; j--)
	    if (! maybe_eliminate_biv_1 (XVECEXP (x, i, j), insn, bl,
					 eliminate_p, where))
	      return 0;
	  break;
	}
    }

  return 1;
}  

/* Return nonzero if the last use of REG
   is in an insn following INSN in the same basic block.  */

static int
last_use_this_basic_block (reg, insn)
     rtx reg;
     rtx insn;
{
  rtx n;
  for (n = insn;
       n && GET_CODE (n) != CODE_LABEL && GET_CODE (n) != JUMP_INSN;
       n = NEXT_INSN (n))
    {
      if (regno_last_uid[REGNO (reg)] == INSN_UID (n))
	return 1;
    }
  return 0;
}

/* Called via `note_stores' to record the initial value of a biv.  Here we
   just record the location of the set and process it later.  */

static void
record_initial (dest, set)
     rtx dest;
     rtx set;
{
  struct iv_class *bl;

  if (GET_CODE (dest) != REG
      || REGNO (dest) >= max_reg_before_loop
      || reg_iv_type[REGNO (dest)] != BASIC_INDUCT)
    return;

  bl = reg_biv_class[REGNO (dest)];

  /* If this is the first set found, record it.  */
  if (bl->init_insn == 0)
    {
      bl->init_insn = note_insn;
      bl->init_set = set;
    }
}

/* If any of the registers in X are "old" and currently have a last use earlier
   than INSN, update them to have a last use of INSN.  Their actual last use
   will be the previous insn but it will not have a valid uid_luid so we can't
   use it.  */

static void
update_reg_last_use (x, insn)
     rtx x;
     rtx insn;
{
  /* Check for the case where INSN does not have a valid luid.  In this case,
     there is no need to modify the regno_last_uid, as this can only happen
     when code is inserted after the loop_end to set a pseudo's final value,
     and hence this insn will never be the last use of x.  */
  if (GET_CODE (x) == REG && REGNO (x) < max_reg_before_loop
      && INSN_UID (insn) < max_uid_for_loop
      && uid_luid[regno_last_uid[REGNO (x)]] < uid_luid[INSN_UID (insn)])
    regno_last_uid[REGNO (x)] = INSN_UID (insn);
  else
    {
      register int i, j;
      register char *fmt = GET_RTX_FORMAT (GET_CODE (x));
      for (i = GET_RTX_LENGTH (GET_CODE (x)) - 1; i >= 0; i--)
	{
	  if (fmt[i] == 'e')
	    update_reg_last_use (XEXP (x, i), insn);
	  else if (fmt[i] == 'E')
	    for (j = XVECLEN (x, i) - 1; j >= 0; j--)
	      update_reg_last_use (XVECEXP (x, i, j), insn);
	}
    }
}

/* Given a jump insn JUMP, return the condition that will cause it to branch
   to its JUMP_LABEL.  If the condition cannot be understood, or is an
   inequality floating-point comparison which needs to be reversed, 0 will
   be returned.

   If EARLIEST is non-zero, it is a pointer to a place where the earliest
   insn used in locating the condition was found.  If a replacement test
   of the condition is desired, it should be placed in front of that
   insn and we will be sure that the inputs are still valid.

   The condition will be returned in a canonical form to simplify testing by
   callers.  Specifically:

   (1) The code will always be a comparison operation (EQ, NE, GT, etc.).
   (2) Both operands will be machine operands; (cc0) will have been replaced.
   (3) If an operand is a constant, it will be the second operand.
   (4) (LE x const) will be replaced with (LT x <const+1>) and similarly
       for GE, GEU, and LEU.  */

rtx
get_condition (jump, earliest)
     rtx jump;
     rtx *earliest;
{
  enum rtx_code code;
  rtx prev = jump;
  rtx set;
  rtx tem;
  rtx op0, op1;
  int reverse_code = 0;
  int did_reverse_condition = 0;

  /* If this is not a standard conditional jump, we can't parse it.  */
  if (GET_CODE (jump) != JUMP_INSN
      || ! condjump_p (jump) || simplejump_p (jump))
    return 0;

  code = GET_CODE (XEXP (SET_SRC (PATTERN (jump)), 0));
  op0 = XEXP (XEXP (SET_SRC (PATTERN (jump)), 0), 0);
  op1 = XEXP (XEXP (SET_SRC (PATTERN (jump)), 0), 1);

  if (earliest)
    *earliest = jump;

  /* If this branches to JUMP_LABEL when the condition is false, reverse
     the condition.  */
  if (GET_CODE (XEXP (SET_SRC (PATTERN (jump)), 2)) == LABEL_REF
      && XEXP (XEXP (SET_SRC (PATTERN (jump)), 2), 0) == JUMP_LABEL (jump))
    code = reverse_condition (code), did_reverse_condition ^= 1;

  /* If we are comparing a register with zero, see if the register is set
     in the previous insn to a COMPARE or a comparison operation.  Perform
     the same tests as a function of STORE_FLAG_VALUE as find_comparison_args
     in cse.c  */

  while (GET_RTX_CLASS (code) == '<' && op1 == CONST0_RTX (GET_MODE (op0)))
    {
      /* Set non-zero when we find something of interest.  */
      rtx x = 0;

#ifdef HAVE_cc0
      /* If comparison with cc0, import actual comparison from compare
	 insn.  */
      if (op0 == cc0_rtx)
	{
	  if ((prev = prev_nonnote_insn (prev)) == 0
	      || GET_CODE (prev) != INSN
	      || (set = single_set (prev)) == 0
	      || SET_DEST (set) != cc0_rtx)
	    return 0;

	  op0 = SET_SRC (set);
	  op1 = CONST0_RTX (GET_MODE (op0));
	  if (earliest)
	    *earliest = prev;
	}
#endif

      /* If this is a COMPARE, pick up the two things being compared.  */
      if (GET_CODE (op0) == COMPARE)
	{
	  op1 = XEXP (op0, 1);
	  op0 = XEXP (op0, 0);
	  continue;
	}
      else if (GET_CODE (op0) != REG)
	break;

      /* Go back to the previous insn.  Stop if it is not an INSN.  We also
	 stop if it isn't a single set or if it has a REG_INC note because
	 we don't want to bother dealing with it.  */

      if ((prev = prev_nonnote_insn (prev)) == 0
	  || GET_CODE (prev) != INSN
	  || FIND_REG_INC_NOTE (prev, 0)
	  || (set = single_set (prev)) == 0)
	break;

      /* If this is setting OP0, get what it sets it to if it looks
	 relevant.  */
      if (rtx_equal_p (SET_DEST (set), op0))
	{
	  enum machine_mode inner_mode = GET_MODE (SET_SRC (set));

	  if ((GET_CODE (SET_SRC (set)) == COMPARE
	       || (((code == NE
		     || (code == LT
			 && GET_MODE_CLASS (inner_mode) == MODE_INT
			 && (GET_MODE_BITSIZE (inner_mode)
			     <= HOST_BITS_PER_WIDE_INT)
			 && (STORE_FLAG_VALUE
			     & ((HOST_WIDE_INT) 1
				<< (GET_MODE_BITSIZE (inner_mode) - 1))))
#ifdef FLOAT_STORE_FLAG_VALUE
		     || (code == LT
			 && GET_MODE_CLASS (inner_mode) == MODE_FLOAT
			 && FLOAT_STORE_FLAG_VALUE < 0)
#endif
		     ))
		   && GET_RTX_CLASS (GET_CODE (SET_SRC (set))) == '<')))
	    x = SET_SRC (set);
	  else if (((code == EQ
		     || (code == GE
			 && (GET_MODE_BITSIZE (inner_mode)
			     <= HOST_BITS_PER_WIDE_INT)
			 && GET_MODE_CLASS (inner_mode) == MODE_INT
			 && (STORE_FLAG_VALUE
			     & ((HOST_WIDE_INT) 1
				<< (GET_MODE_BITSIZE (inner_mode) - 1))))
#ifdef FLOAT_STORE_FLAG_VALUE
		     || (code == GE
			 && GET_MODE_CLASS (inner_mode) == MODE_FLOAT
			 && FLOAT_STORE_FLAG_VALUE < 0)
#endif
		     ))
		   && GET_RTX_CLASS (GET_CODE (SET_SRC (set))) == '<')
	    {
	      /* We might have reversed a LT to get a GE here.  But this wasn't
		 actually the comparison of data, so we don't flag that we
		 have had to reverse the condition.  */
	      did_reverse_condition ^= 1;
	      reverse_code = 1;
	      x = SET_SRC (set);
	    }
	  else
	    break;
	}

      else if (reg_set_p (op0, prev))
	/* If this sets OP0, but not directly, we have to give up.  */
	break;

      if (x)
	{
	  if (GET_RTX_CLASS (GET_CODE (x)) == '<')
	    code = GET_CODE (x);
	  if (reverse_code)
	    {
	      code = reverse_condition (code);
	      did_reverse_condition ^= 1;
	      reverse_code = 0;
	    }

	  op0 = XEXP (x, 0), op1 = XEXP (x, 1);
	  if (earliest)
	    *earliest = prev;
	}
    }

  /* If constant is first, put it last.  */
  if (CONSTANT_P (op0))
    code = swap_condition (code), tem = op0, op0 = op1, op1 = tem;

  /* If OP0 is the result of a comparison, we weren't able to find what
     was really being compared, so fail.  */
  if (GET_MODE_CLASS (GET_MODE (op0)) == MODE_CC)
    return 0;

  /* Canonicalize any ordered comparison with integers involving equality
     if we can do computations in the relevant mode and we do not
     overflow.  */

  if (GET_CODE (op1) == CONST_INT
      && GET_MODE (op0) != VOIDmode
      && GET_MODE_BITSIZE (GET_MODE (op0)) <= HOST_BITS_PER_WIDE_INT)
    {
      HOST_WIDE_INT const_val = INTVAL (op1);
      unsigned HOST_WIDE_INT uconst_val = const_val;
      unsigned HOST_WIDE_INT max_val
	= (unsigned HOST_WIDE_INT) GET_MODE_MASK (GET_MODE (op0));

      switch (code)
	{
	case LE:
	  if (const_val != max_val >> 1)
	    code = LT,	op1 = GEN_INT (const_val + 1);
	  break;

	case GE:
	  if (const_val
	      != (((HOST_WIDE_INT) 1
		   << (GET_MODE_BITSIZE (GET_MODE (op0)) - 1))))
	    code = GT, op1 = GEN_INT (const_val - 1);
	  break;

	case LEU:
	  if (uconst_val != max_val)
	    code = LTU, op1 = GEN_INT (uconst_val + 1);
	  break;

	case GEU:
	  if (uconst_val != 0)
	    code = GTU, op1 = GEN_INT (uconst_val - 1);
	  break;
	}
    }

  /* If this was floating-point and we reversed anything other than an
     EQ or NE, return zero.  */
  if (TARGET_FLOAT_FORMAT == IEEE_FLOAT_FORMAT
      && did_reverse_condition && code != NE && code != EQ
      && ! flag_fast_math
      && GET_MODE_CLASS (GET_MODE (op0)) == MODE_FLOAT)
    return 0;

#ifdef HAVE_cc0
  /* Never return CC0; return zero instead.  */
  if (op0 == cc0_rtx)
    return 0;
#endif

  return gen_rtx (code, VOIDmode, op0, op1);
}

/* Similar to above routine, except that we also put an invariant last
   unless both operands are invariants.  */

rtx
get_condition_for_loop (x)
     rtx x;
{
  rtx comparison = get_condition (x, NULL_PTR);

  if (comparison == 0
      || ! invariant_p (XEXP (comparison, 0))
      || invariant_p (XEXP (comparison, 1)))
    return comparison;

  return gen_rtx (swap_condition (GET_CODE (comparison)), VOIDmode,
		  XEXP (comparison, 1), XEXP (comparison, 0));
}

/* new code added by intel1 */
struct new_rtx
  {
    rtx pat;
    rtx insn;
    rtx supplemental_pat;	/* Pattern that should be
				   emited after this insn.  */
    struct new_rtx *next;
  };

/* If bl is also the loop counter try to optimize out the
   compare instruction by counting up to 0.  Return 1 if
   successful.
*/
static
int 
compare_elimination (bl, loop_start, loop_end, compare_insn, jump_insn, second_time)
     struct iv_class *bl;
     rtx loop_start, loop_end, compare_insn, jump_insn;
     int second_time;
{
  rtx insn, new_init_src, copy_of_jump, new_biv_set, loop_trip_reg;
  int init_val, compare_val, i, adjust_val, desired_init_val;
  int num_times_loop_execs, add_one, biv_init_to_const, biv_end_value;
  int num_sups, compare_to_const;
  struct new_rtx *new_rtx_element, *new_rtx_head, *new_rtx_tail;
  char *storage;

  if (GET_CODE (SET_DEST (bl->init_set)) != REG)
    return 0;

  if (!rtx_equal_p (SET_DEST (bl->init_set),
		    XEXP (SET_SRC (PATTERN (compare_insn)), 0)))
    return 0;

  copy_of_jump = copy_rtx (PATTERN (jump_insn));
  PUT_CODE (XEXP (SET_SRC (copy_of_jump), 0), NE);
  if (recog (copy_of_jump, jump_insn, 0) == -1)
    return 0;

  if (GET_CODE (XEXP (SET_SRC (PATTERN (compare_insn)), 1)) == CONST_INT)
    {
      if (bl->initial_test != NULL)
	return 0;

      compare_val = INTVAL (XEXP (SET_SRC (PATTERN (compare_insn)), 1));
      compare_to_const = 1;
    }
  else
    {
      compare_to_const = 0;
      if (INTVAL (bl->biv->add_val) != 1)
	{
	  return 0;
	}
      else if (second_time == 0)
	{
	  bl->retry_celim = 2;	/* Signals that this is loop trip test
	                           candidate.  */
	  return 0;
	}
    }
  if (GET_CODE (SET_SRC (bl->init_set)) != CONST_INT)
    {
      if (!compare_to_const)
	{
	  return 0;
	}
      else
	{
	  if (abs (INTVAL (bl->biv->add_val)) != 1)
	    return 0;
	  
	  biv_init_to_const = 0;
	  /* Just so checks on LE GE ... pass.  */
	  init_val = compare_val - INTVAL (bl->biv->add_val);
	}
    }
  else
    {
      init_val = INTVAL (SET_SRC (bl->init_set));
      biv_init_to_const = 1;
    }
  insn = PREV_INSN (compare_insn);
  while (GET_CODE (insn) == NOTE)
    insn = PREV_INSN (insn);
  
  if (!(GET_CODE (insn) == INSN 
	&& rtx_equal_p (SET_DEST (PATTERN (insn)), SET_DEST (bl->init_set)) 
	&& bl->biv->insn == insn))
    return 0;
  
  switch (GET_CODE (XEXP (SET_SRC (PATTERN (jump_insn)), 0)))
    {
    case LE:
      if (compare_to_const 
	  && (INTVAL (bl->biv->add_val) <= 0 || compare_val < init_val))
	return 0;

      add_one = 1;
      break;

    case GE:
      if (compare_to_const 
	  && (INTVAL (bl->biv->add_val) >= 0 || compare_val > init_val))
	return 0;
      
      add_one = 1;
      break;

    case LT:
      if (compare_to_const 
	  && (INTVAL (bl->biv->add_val) <= 0 || compare_val <= init_val))
	return 0;
      
      add_one = 0;
      break;

    case GT:
      if (compare_to_const 
	  && (INTVAL (bl->biv->add_val) >= 0 || compare_val >= init_val))
	return 0;
	
      add_one = 0;
      break;

    default:
      return 0;
    }
  
  if (!compare_to_const)
    { /* We are doing loop trip test.  */
      if (n_times_set[REGNO (XEXP (SET_SRC (PATTERN (compare_insn)), 1))])
	return 0;
      
      loop_trip_reg = gen_reg_rtx (GET_MODE (SET_DEST (bl->init_set)));
      desired_init_val = (-1 * init_val) + add_one;
      /* Emit insn to initialize loop trip reg.  */
      if (desired_init_val)
	{
	  emit_insn_before (gen_rtx (SET, 0, loop_trip_reg,
				     gen_rtx (PLUS, GET_MODE (loop_trip_reg),
				 XEXP (SET_SRC (PATTERN (compare_insn)), 1),
				 gen_rtx (CONST_INT, 0, desired_init_val))),
			    loop_start);
	}
      else
	{
	  emit_insn_before (gen_rtx (SET, 0, loop_trip_reg,
				XEXP (SET_SRC (PATTERN (compare_insn)), 1)),
			    loop_start);
	}
      /* Emit insn to decrement loop trip reg.  */
      emit_insn_before (gen_rtx (SET, 0, loop_trip_reg,
				 gen_rtx (PLUS, GET_MODE (loop_trip_reg),
					  loop_trip_reg,
					  gen_rtx (CONST_INT, 0, -1))),
			compare_insn);
      /* Change comparison */
      XEXP (SET_SRC (PATTERN (compare_insn)), 1) = const0_rtx;
      XEXP (SET_SRC (PATTERN (compare_insn)), 0) = loop_trip_reg;
      /* Change jump insn */
      PUT_CODE (XEXP (SET_SRC (PATTERN (jump_insn)), 0), NE);
      INSN_CODE (jump_insn) = recog (PATTERN (jump_insn), jump_insn, 0);
      bl->retry_celim = 0;
      return 1;
    }

  if (biv_init_to_const)
    {
      num_times_loop_execs
	  = ((abs (compare_val - init_val) + add_one - 1) / abs (INTVAL (bl->biv->add_val))) + 1;
      desired_init_val = -1 * num_times_loop_execs * INTVAL (bl->biv->add_val);
      adjust_val = init_val - desired_init_val;
      new_init_src = gen_rtx (CONST_INT, 0, desired_init_val);

      biv_end_value = num_times_loop_execs * INTVAL (bl->biv->add_val) + init_val;
    }
  else
    {/* biv is incremented by +-1 so the adjust_val is */
      adjust_val 
	  = (compare_val + (add_one * INTVAL (bl->biv->add_val)));
      desired_init_val = -1 * adjust_val;
      biv_end_value = compare_val + (add_one * INTVAL (bl->biv->add_val));
      /* Make a check that we will be able to adjust the bivs init_val
         by desired_init_val.  */
      new_init_src = gen_rtx (PLUS, GET_MODE (SET_DEST (bl->init_set)),
			      SET_SRC (bl->init_set),
			      gen_rtx (CONST_INT, 0, desired_init_val));
      new_biv_set = gen_rtx (SET, GET_MODE (SET_SRC (bl->init_set)),
			     SET_DEST (bl->init_set), new_init_src);
      if (recog (new_biv_set, bl->init_set, 0) == -1)
	return 0;
    }
  
  /* Now see if we can replace register bl->regno with
     bl->regno+(adjust_val) in all insns from bl->init_insn to
     loop_end.  
     Signal that if we fail we should retry again at the end of
     strength reduction.  */
  if (second_time == 0)
    bl->retry_celim = 1;
  
  /* Mark our place on the obstack in case of failure.  */
  storage = (char *) oballoc (0);
  num_sups = 0;
  new_rtx_head = new_rtx_tail = NULL;
  
  insn = NEXT_INSN (bl->init_insn);
  while (insn != compare_insn)
    {
      if (GET_CODE (insn) == INSN)
        {
          if (GET_CODE (PATTERN (insn)) != SET)
            {
              obfree(storage);
              return 0;
            }
          else if (!rtx_equal_p (SET_DEST (PATTERN (insn)), SET_DEST (bl->init_set)))
	    {
	      new_rtx_element = (struct new_rtx *) alloca (sizeof (struct new_rtx));
	      new_rtx_element->insn = insn;
	      new_rtx_element->supplemental_pat = NULL;
	      if (!attempt_new_rtx (new_rtx_element, insn, bl, adjust_val, second_time))
	        {
	          obfree (storage);
	          return 0;
	        }
	      else if (!rtx_equal_p (SET_SRC (insn), SET_SRC (new_rtx_element->pat)) 
		       || !rtx_equal_p (SET_DEST (insn), SET_DEST (new_rtx_element->pat)))
	        {
	          if (new_rtx_head == NULL)
		    new_rtx_head = new_rtx_element;
	          else
		    new_rtx_tail->next = new_rtx_element;

	          new_rtx_tail = new_rtx_element;
	          new_rtx_element->next = 0;
	        }

	      if (new_rtx_element->supplemental_pat)
	        {
	          num_sups++;
	          if (num_sups > 4)
		    { /* heuristic */
		      obfree (storage);
		      return 0;
		    }
	        }
	    }
	}
      insn = NEXT_INSN (insn);
    }
  
  /* Replacement can be done now do it.  */
  new_rtx_element = new_rtx_head;
  while (new_rtx_element)
    {
      PATTERN (new_rtx_element->insn) = new_rtx_element->pat;
      if (new_rtx_element->supplemental_pat)
	{
	  emit_insn_after (
		gen_move_insn (SET_DEST (new_rtx_element->supplemental_pat),
			       SET_SRC (new_rtx_element->supplemental_pat)),
			    new_rtx_element->insn);
	}
      new_rtx_element = new_rtx_element->next;
    }
  /* Finally update the initialization and comparison of the biv.  */
  PUT_CODE (XEXP (SET_SRC (PATTERN (jump_insn)), 0), NE);
  INSN_CODE (jump_insn) = recog (PATTERN (jump_insn), jump_insn, 0);
  XEXP (SET_SRC (PATTERN (compare_insn)), 1) = const0_rtx;
  SET_SRC (bl->init_set) = new_init_src;
  INSN_CODE (bl->init_insn) = recog ( PATTERN (bl->init_insn), bl->init_insn, 0);
  emit_insn_after (gen_rtx (SET, GET_MODE (SET_DEST (bl->init_set)),
			    SET_DEST (bl->init_set),
			    gen_rtx (CONST_INT, 0, biv_end_value)),
		   loop_end);
  bl->retry_celim = 0;
  return 1;
}

/* Try to rewrite insn replacing the biv with biv+biv_adjust_const.  if
   the substitution is not immediately recognizable then try to
   rewrite the insn to make it recognizable.  If try_supplemental is 1
   then an additional insn may be generated to make the substitution
   recognizable.  The generated pattern is pointed to by
   new_rtx_element->pat.  
*/
static 
attempt_new_rtx (new_rtx_element, insn, bl, biv_adjust_const, try_supplemental)
     rtx insn;
     struct new_rtx *new_rtx_element;
     struct iv_class *bl;
     int biv_adjust_const;
     int try_supplemental;
{
  rtx new_src, new_dest, make_new_use (), try_to_make_good ();

  new_src = make_new_use (SET_SRC (PATTERN (insn)), SET_DEST (bl->init_set),
		          biv_adjust_const, insn);
  new_dest = make_new_use (SET_DEST (PATTERN (insn)), SET_DEST (bl->init_set),
		          biv_adjust_const, insn);
  new_rtx_element->pat = gen_rtx (SET, 0, new_dest, new_src);

  if (recog (new_rtx_element->pat, insn, 0) < 0)
    {
      try_to_make_good (SET_SRC (new_rtx_element->pat));
      try_to_make_good (SET_DEST (new_rtx_element->pat));
    }

  if (recog (new_rtx_element->pat, insn, 0) < 0)
    {
      if (!try_supplemental)
	return 0;
      
      if (GET_CODE (SET_DEST (new_rtx_element->pat)) == REG 
	  && GET_CODE (SET_SRC (new_rtx_element->pat)) == ASHIFT 
	  && GET_CODE (XEXP (SET_SRC (new_rtx_element->pat), 0)) == PLUS 
	  && GET_CODE (XEXP (SET_SRC (new_rtx_element->pat), 1)) == CONST_INT 
          && GET_CODE (XEXP (XEXP (SET_SRC (new_rtx_element->pat), 0), 0)) == REG 
	  && GET_CODE (XEXP (XEXP (SET_SRC (new_rtx_element->pat), 0), 1)) == CONST_INT)
	{
	  /* try to rewrite:
           (set (reg:SI x)
              (ashift:SI (plus:SI (reg/v:SI y)
                                  (const_int a))
                         (const_int b)))

           into two patterns:
           (set (reg:SI x)
              (ashift:SI (reg/v:SI y)
                         (const_int b)))

           (set (reg:SI x)
              (plus:SI (reg/v:SI x)
                       (const_int a*(2**b))))
        */
	  int add_val;
	  add_val 
	    = two_to_power (INTVAL (XEXP (SET_SRC (new_rtx_element->pat), 1))) 
	      * INTVAL (XEXP (XEXP (SET_SRC (new_rtx_element->pat), 0), 1));
	  new_rtx_element->pat = PATTERN (new_rtx_element->insn);
	  new_rtx_element->supplemental_pat 
	    = gen_rtx (SET, GET_MODE (SET_DEST (new_rtx_element->pat)),
		     SET_DEST (new_rtx_element->pat),
	  gen_rtx (PLUS, GET_MODE (XEXP (SET_SRC (new_rtx_element->pat), 0)),
		   SET_DEST (new_rtx_element->pat),
		   gen_rtx (CONST_INT, 0, add_val)));
	}
      else if (GET_CODE (SET_DEST (new_rtx_element->pat)) == REG 
	       && GET_CODE (SET_SRC (new_rtx_element->pat)) == PLUS 
	       && GET_CODE (XEXP (SET_SRC (new_rtx_element->pat), 0)) == REG 
	       && GET_CODE (XEXP (SET_SRC (new_rtx_element->pat), 1)) == PLUS 
	       && GET_CODE (XEXP (XEXP (SET_SRC (new_rtx_element->pat), 1), 0)) == REG 
	       && GET_CODE (XEXP (XEXP (SET_SRC (new_rtx_element->pat), 1), 1)) == CONST_INT
	)
	{
	  /* try to rewrite:
           (set (reg:SI x)
                (plus:SI (reg:SI y)
                          (plus:SI   (reg:SI z)
                                     (const_int a))))

           into two patterns:
           (set (reg:SI x)
              (plus:SI (reg/v:SI y)
                       (reg/v:SI z)))

           (set (reg:SI x)
              (plus:SI (reg/v:SI x)
                       (const_int a)))
        */
	  new_rtx_element->supplemental_pat 
	    = gen_rtx (SET, GET_MODE (SET_DEST (new_rtx_element->pat)),
		     SET_DEST (new_rtx_element->pat),
	  gen_rtx (PLUS, GET_MODE (XEXP (SET_SRC (new_rtx_element->pat), 0)),
		   SET_DEST (new_rtx_element->pat),
		   XEXP (XEXP (SET_SRC (new_rtx_element->pat), 1), 1)));
	  new_rtx_element->pat = PATTERN (new_rtx_element->insn);
	}
      else
	return 0;
      
      if (recog (new_rtx_element->supplemental_pat, insn, 0) < 0)
	return 0;
    }
  return 1;
}


/* Generate a new rtx which is a copy of orig with each occurence of
   biv_reg replaced by biv_reg+biv_adjust_const.  
*/
static rtx
make_new_use (orig, biv_reg, biv_adjust_const, insn)
     register rtx orig, biv_reg;
     rtx insn;
     int biv_adjust_const;
{
  register rtx copy;
  rtx non_biv_operand, ret_rtx;
  register int i, j;
  register RTX_CODE code;
  register char *format_ptr;

  code = GET_CODE (orig);

  switch (code)
    {
    case REG:
      if (rtx_equal_p (orig, biv_reg))
	{
	  ret_rtx 
	    = gen_rtx (PLUS, GET_MODE (orig), orig,
		     gen_rtx (CONST_INT, 0, biv_adjust_const));
	  return ret_rtx;
	}
      else
	return orig;

      break;

    case PLUS:
    case MINUS:
      if (GET_CODE (XEXP (orig, 0)) == REG && GET_CODE (XEXP (orig, 1)) == CONST_INT 
	  && rtx_equal_p (XEXP (orig, 0), biv_reg))
	{ /* Ajdust the const_int part by biv_adjust_const.  */
	  if (GET_CODE (orig) == PLUS)
	    {
	      non_biv_operand 
		   = gen_rtx (CONST_INT, 0,
			 INTVAL (XEXP (orig, 1)) + biv_adjust_const);
	    }
	  else
	    {
	      non_biv_operand 
		= gen_rtx (CONST_INT, 0,
			 INTVAL (XEXP (orig, 1)) - biv_adjust_const);
	    }
	  ret_rtx = gen_rtx (code, GET_MODE (orig), XEXP (orig, 0), non_biv_operand);
	  return ret_rtx;
	}
      else
	/* commutativity */ 
	if (GET_CODE (XEXP (orig, 1)) == REG && GET_CODE (XEXP (orig, 0)) == CONST_INT 
	    && rtx_equal_p (XEXP (orig, 1), biv_reg))
	{/* Ajdust the const_int part by biv_adjust_const.  */
	  if (GET_CODE (orig) == PLUS)
	    {
	      non_biv_operand 
		= gen_rtx (CONST_INT, 0,
			 INTVAL (XEXP (orig, 0)) + biv_adjust_const);
	    }
	  else
	    {
	      non_biv_operand 
		= gen_rtx (CONST_INT, 0,
			 INTVAL (XEXP (orig, 0)) - biv_adjust_const);
	    }
	  ret_rtx = gen_rtx (code, GET_MODE (orig), non_biv_operand, XEXP (orig, 1));
	  return ret_rtx;
	}
      break;

    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
    case PC:
    case CC0:
      return orig;
    }

  copy = rtx_alloc (code);
  PUT_MODE (copy, GET_MODE (orig));
  copy->in_struct = orig->in_struct;
  copy->volatil = orig->volatil;
  copy->unchanging = orig->unchanging;
  copy->integrated = orig->integrated;

  format_ptr = GET_RTX_FORMAT (GET_CODE (copy));

  for (i = 0; i < GET_RTX_LENGTH (GET_CODE (copy)); i++)
    {
      switch (*format_ptr++)
	{
	case 'e':
	  XEXP (copy, i) = XEXP (orig, i);
	  if (XEXP (orig, i) != NULL)
	    XEXP (copy, i) 
	      = make_new_use (XEXP (orig, i), biv_reg,
			    biv_adjust_const, insn);
	  break;

	case 'E':
	case 'V':
	  XVEC (copy, i) = XVEC (orig, i);
	  if (XVEC (orig, i) != NULL)
	    {
	      XVEC (copy, i) = rtvec_alloc (XVECLEN (orig, i));
	      for (j = 0; j < XVECLEN (copy, i); j++)
		XVECEXP (copy, i, j) 
		  = make_new_use (XVECEXP (orig, i, j), biv_reg,
				biv_adjust_const, insn);
	    }
	  break;

	default:
	  XINT (copy, i) = XINT (orig, i);
	  break;
	}
    }
  return copy;
}

/* Try to put orig into a form that is recognizable using rewrite_address
   on the relevant portion of orig.  
*/
rtx
try_to_make_good (orig)
     register rtx orig;
{
  void rewrite_address ();
  register int i, j;
  register RTX_CODE code;
  register char *fmt;

  code = GET_CODE (orig);

  switch (code)
    {
    case REG:
    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
    case PC:
    case CC0:
      return orig;

    case MEM:
      GO_IF_LEGITIMATE_ADDRESS (GET_MODE (orig), orig, memok);
#ifdef REWRITE_ADDRESS
      REWRITE_ADDRESS (orig);
#endif
    memok:
      return orig;
    }


  fmt = GET_RTX_FORMAT (code);

  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      XEXP (orig, i) = try_to_make_good (XEXP (orig, i));
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (orig, i); j++)
	  XVECEXP (orig, i, j) 
	    = try_to_make_good (XVECEXP (orig, i, j));
      }
  return orig;
}

/* Check wether insn1 is strictly after insn2 */
static int
insn_after (insn1, insn2)
     rtx insn1, insn2;
{
  if (INSN_UID (insn1) >= max_uid_for_loop
      || INSN_UID (insn2) >= max_uid_for_loop)
    {
      /* lets do it the hard way */
      while (insn2 = NEXT_INSN (insn2))
        if (insn1 == insn2)
          return 1;
      return 0;
    }
  else
    return INSN_LUID (insn1) > INSN_LUID (insn2);
}

/* Check to see if compare elimination is possible for this loop.  
   Return 1 if yes and set up *compare_insn to point to the compare
   at the end of the loop and *jump_insn to point to the jump back
   to the beginning of the loop.  
*/
static int
possible_compare_elimination (loop_start, loop_end, compare_insn, jump_insn, try_compare_elimination)
     rtx loop_end, loop_start, *compare_insn, *jump_insn;
     int *try_compare_elimination;
{
  rtx insn;
  int i;

  insn = loop_start;
  i = 0;
  *try_compare_elimination = 0;

  while (insn && insn != loop_end)
    {
      if (GET_CODE (insn) == CALL_INSN)
	{
	  insn = NULL;
	}
      else if (GET_CODE (insn) == INSN)
	{
	  i++;
	  if (i > MAX_CMP_ELIM
	      || (GET_CODE (PATTERN (insn)) != SET
	          && GET_CODE (PATTERN (insn)) != USE))
	    insn = NULL;
	}
      else if (GET_CODE (insn) == JUMP_INSN
               && condjump_p (insn)
               && insn_after (JUMP_LABEL (insn), loop_end))
        {
          /* intel3 */
          /* returns are not included in loop_exit_labels, but */
          /* since we only need to know wether there are */
          /* ANY exists, we just look at the jumps */
          /* this is very conservative, i.e. there could */
          /* be ways around this problem */
          insn = NULL;
        }
      if (insn)
	insn = NEXT_INSN (insn);
    }

  if (insn != NULL)
    *try_compare_elimination = 1;

  /* Look for compare jcc on this biv at the end of the loop.  */
  insn = PREV_INSN (loop_end);
  if (!(GET_CODE (insn) == JUMP_INSN 
	&& GET_CODE (PATTERN (insn)) == SET 
	&& SET_DEST (PATTERN (insn)) == pc_rtx 
	&& GET_CODE (SET_SRC (PATTERN (insn))) == IF_THEN_ELSE 
	&& XEXP (XEXP (SET_SRC (PATTERN (insn)), 0), 0) == cc0_rtx 
	&& XEXP (XEXP (SET_SRC (PATTERN (insn)), 0), 1) == const0_rtx))
    *try_compare_elimination = 0;
  
  *jump_insn = insn;
  
  insn = PREV_INSN (insn);
  if (!(GET_CODE (insn) == INSN 
        && GET_CODE (PATTERN (insn)) == SET 
        && SET_DEST (PATTERN (insn)) == cc0_rtx 
        && GET_CODE (SET_SRC (PATTERN (insn))) == COMPARE 
        && (GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) == CONST_INT 
	    || GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) == REG) 
	&& GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 0)) == REG))
    *try_compare_elimination = 0;
  
  *compare_insn = insn;
}

struct regs_mentioned
  {
    rtx reg;				/* The register.  */
    int in_mem;				/* Is 1 if register is part of
    					   a MEM_REF.  */
    struct regs_mentioned *next;
  };
   
static struct regs_mentioned *giv_reg_header = NULL, *use_reg_header = NULL;

/* Build a linked list of (struct regs_mentioned) describing the
   registers used in the pattern pat.  The head of the linked list
   to place them on is a_header.  If in_mem is 1 this means that
   pat is subexpression of a MEM_REF.  
*/
static void
find_regs_mentioned (pat, a_header, in_mem)
     rtx pat;
     struct regs_mentioned **a_header;
     int in_mem;
{
  register char *fmt;
  register int i;
  register enum rtx_code code;
  struct regs_mentioned *one_reg;

  code = GET_CODE (pat);

  switch (code)
    {
    case REG:
      if (REGNO (pat) < FIRST_PSEUDO_REGISTER)
	{
	  return;
	}
      one_reg = (struct regs_mentioned *)
	obstack_alloc (&momentary_obstack, sizeof (struct regs_mentioned));
      one_reg->reg = pat;
      one_reg->in_mem = in_mem;
      one_reg->next = *a_header;
      *a_header = one_reg;
      return;

      /* These codes have no constituent expressions
	 and are unique.  */
    case SCRATCH:
    case CC0:
    case PC:
    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
      return;

    case MEM:
      in_mem = 1;
      break;

    }


  fmt = GET_RTX_FORMAT (code);

  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'E')
	{
	  register int j;
	  for (j = XVECLEN (pat, i) - 1; j >= 0; j--)
	    find_regs_mentioned (XVECEXP (pat, i, j), a_header, in_mem);
	}
      else if (fmt[i] == 'e')
	find_regs_mentioned (XEXP (pat, i), a_header, in_mem);
    }
}

#ifdef REDUCE_INDEX
/* Create givs which are only index terms of memory addresses.  */
static int
do_reduce_index (x, add_val)
     rtx x, *add_val;
{
  rtx mem_address, sym_part, constint_part, save_add_val;
  save_add_val = *add_val;
  mem_address = XEXP (x, 0);
  if (GET_CODE (mem_address) != PLUS)
    return 0;

  if (GET_CODE (XEXP (mem_address, 0)) == MULT)
    {
      if (!LEGITIMATE_INDEX_P (XEXP (mem_address, 0)) 
	  || GET_CODE (XEXP (mem_address, 1)) == REG)
	{ /* Will have index term anyway.  */
	  return 0;
	}
    }
  else
    return 0;

  /* Strip from *(add_val) the non index part.  */
  if (rtx_equal_p (*(add_val), XEXP (mem_address, 1)))
    {
      *(add_val) = const0_rtx;
    }
  else if (GET_CODE (*(add_val)) == CONST_INT)
    {
      if (GET_CODE (XEXP (mem_address, 1)) != CONST_INT)
	{
	  return 0;
	}
      else
	{
	  *(add_val) 
	    = gen_rtx (CONST_INT, 0,
		      INTVAL (*(add_val)) - INTVAL (XEXP (mem_address, 1)));
	}
    }
  else if (GET_CODE (*(add_val)) == CONST)
    {
      if (!(GET_CODE (XEXP (*(add_val), 0)) == PLUS 
	    && GET_CODE (XEXP (XEXP (*(add_val), 0), 0)) == SYMBOL_REF 
	    && GET_CODE (XEXP (XEXP (*(add_val), 0), 1)) == CONST_INT))
	return 0;
      
      *(add_val) = XEXP (*(add_val), 0);
      mem_address = XEXP (mem_address, 1);
      if (GET_CODE (mem_address) == CONST)
	mem_address = XEXP (mem_address, 0);

      if (GET_CODE (mem_address) == PLUS)
	{
	  sym_part = XEXP (mem_address, 0);
	  constint_part = XEXP (mem_address, 1);
	}
      else
	{
	  sym_part = mem_address;
	  constint_part = const0_rtx;
	}
      if (!rtx_equal_p (XEXP (*(add_val), 0), sym_part))
	{
	  *(add_val) = save_add_val;
	  return 0;
	}
      else
	{
	  *(add_val) = XEXP (*(add_val), 1);
	  *(add_val) 
	    = gen_rtx (CONST_INT, 0, INTVAL (*(add_val)) - INTVAL (constint_part));
	}
    }
  else
    return 0;

  return 1;
}

#endif

struct regopt_info
{
  rtx insn_set;		/* The insn that loads the current value in the register */
  rtx corresp_mem;	/* The memory that contains the same value as the register */
  rtx reg; 		/* The register */
  char replace_reg;	/* If 1 replace reg with corresp_mem (for stack regs)
			   otherwise replace corresp_mem with reg */
  char do_replace;	/* If 1 then this is a stack reg that has already
			   been used in this extended basic block -
			   try to replace susequent uses with this regs
			   corresponding mem */
};
static struct regopt_info *rinfo;
struct undo
{
  rtx *where;
  rtx old_contents;
  int is_int;
};

#define MAX_UNDO 50

struct undobuf
{
  int num_undo;
  char *storage;
  struct undo undo[MAX_UNDO];
  rtx other_insn;
};

static struct undobuf undobuf;
#define LOOP_SUBST(WHERE, NEWVAL)  \
 do { if (undobuf.num_undo < MAX_UNDO)					\
	{								\
	  undobuf.undo[undobuf.num_undo].where = WHERE;			\
	  undobuf.undo[undobuf.num_undo].old_contents = *WHERE;		\
	  undobuf.undo[undobuf.num_undo].is_int = 0;			\
	  *WHERE = NEWVAL;						\
	  undobuf.num_undo++;						\
	}								\
    } while (0)


static int mem_max_reg_num;
#define SYMREF_PLUS_REG(insn)						\
     (GET_CODE (SET_SRC (PATTERN (insn))) == PLUS &&			\
     GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 0)) == REG && 		\
     GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) == SYMBOL_REF)

/*  Try to replace a register use with the value that sets that the
    register.  This is done for the following cases:
    1) regx <- regy ashift const_int
    2) regx <- symbol_ref + regy
    3) regx <- (regy * const_int) + const_int

    If only_symref_plus_reg==1 then regx is replaced only if it
    used as a base or index register and is of type 2)
    otherwise it is always replaced.  

    Calling use_regs_source with only_symref_plus_reg==0 on a loop
    before strength reduction causes strength reduction to better handle
    mem givs.  

    Calling use_regs_source with only_symref_plus_reg==1 at the end of
    loop optimization will do the replacement even if combine would not.  

    also if only_symref_plus_reg==1 and if we have
    1) mem1 <- regx
    then attempt to replace susequent uses of mem1 with regx


*/

static
void 
use_regs_source (start, end, only_symref_plus_reg)
     rtx start, end;
     int only_symref_plus_reg;
{
  rtx insn, src, new_src, dest, tinsn, tmp_rtx;
  struct regs_mentioned *header, *a_reg_used;
  int recog_ok, i, insn_code_num, nested;
  char *storage1, *storage2;
  int cur_max_reg = max_reg_num () + 1;
  mem_max_reg_num = cur_max_reg;

  undobuf.num_undo = 0;
  rinfo 
    = (struct regopt_info *) alloca (cur_max_reg * sizeof (struct regopt_info));
  bzero (rinfo, cur_max_reg * sizeof (struct regopt_info));

  nested = 0;
  insn = NEXT_INSN (start);
  while (insn != end)
    {
      switch (GET_CODE (insn))
	{

	case NOTE:
	  if (NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_BEG 
	      && only_symref_plus_reg)
	    {
	      nested++;
	      bzero (rinfo, cur_max_reg * sizeof (struct regopt_info));
	    }
	  if (NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_END 
	      && only_symref_plus_reg)
	    {
	      nested--;
	      bzero (rinfo, cur_max_reg * sizeof (struct regopt_info));
	    }
	  break;

	case JUMP_INSN:
	  if (nested == 0 || !only_symref_plus_reg)
	    {
	      if (SET_DEST (PATTERN (insn)) == pc_rtx
		  && GET_CODE (SET_SRC (PATTERN (insn))) == LABEL_REF)
		{
		  bzero (rinfo, cur_max_reg * sizeof (struct regopt_info));
		}
	      else if (!only_symref_plus_reg)
		{
		  for (i = 0; i < cur_max_reg; i++)
		    {
		      rinfo[i].corresp_mem = NULL;
		    }
		}
	    }
	  break;

	case CALL_INSN:
	  if (nested == 0 || !only_symref_plus_reg)
	    {
	      bzero (rinfo, cur_max_reg * sizeof (struct regopt_info));
	    }
	  break;

	case CODE_LABEL:
	case BARRIER:
	  if (nested == 0 || !only_symref_plus_reg)
	    {
	      /* Forget everything */
	      bzero (rinfo, cur_max_reg * sizeof (struct regopt_info));
	    }
	  break;

	case INSN:
	  if (!(nested == 0 || !only_symref_plus_reg))
	    break;
          if (GET_CODE (PATTERN (insn)) == CLOBBER)
            {
              dest = SET_DEST (PATTERN (insn));
	      if (GET_CODE (dest) == REG)
		{
		  rinfo[ REGNO (dest)].insn_set = NULL;
		  for (i = 0; i < cur_max_reg; i++)
		    {
		      if (rinfo[i].insn_set
			  && reg_overlap_mentioned_p (dest, 
			        SET_SRC (PATTERN (rinfo[i].insn_set)))
			)
			{ /* current value of reg dest overwritten */
			  rinfo[i].insn_set = NULL;
		          if (rinfo[i].corresp_mem 
		              && reg_overlap_mentioned_p (dest, 
		                   rinfo[i].corresp_mem))
			    { /* Not the same memory address anymore.  */
			      rinfo[i].corresp_mem = NULL;
			    }
			}
		    }
		}
	      else if (GET_CODE (dest) == MEM && !only_symref_plus_reg 
		       && flag_replace_mem)
		{
		  for (i = 0; i < cur_max_reg; i++)
		    {
		      if (rinfo[i].corresp_mem 
			  && true_dependence (dest, rinfo[i].corresp_mem))
			{ /* Memory address maybe overwritten.  */
			  rinfo[i].corresp_mem = NULL;
			}
		    }
		}
			} else if (GET_CODE (PATTERN (insn)) == PARALLEL)
	    {
				dest = SET_DEST (PATTERN (insn));
	      for (i = 0; i < XVECLEN (PATTERN (insn), 0); i++)
		{
		  rtx cur_rtx;
		  cur_rtx = XVECEXP (PATTERN (insn), 0, i);
		  if ((GET_CODE (cur_rtx) == SET || GET_CODE (cur_rtx) == CLOBBER)
		      && GET_CODE (SET_DEST (cur_rtx)) == REG)
		    {
		      rinfo[ REGNO (SET_DEST (cur_rtx))].insn_set = NULL;
		      for (i = 0; i < cur_max_reg; i++)
		        {
		          if (rinfo[i].insn_set
			      && reg_overlap_mentioned_p (SET_DEST (cur_rtx), 
			             SET_SRC (PATTERN (rinfo[i].insn_set)))
			    )
			    { /* Current value of reg dest overwritten */
			      rinfo[i].insn_set = NULL;
			    }
		          if (rinfo[i].corresp_mem 
		              && reg_overlap_mentioned_p (SET_DEST (cur_rtx), 
		                   rinfo[i].corresp_mem))
			    { /* Not the same memory address anymore.  */
			      rinfo[i].corresp_mem = NULL;
			    }
		        }
		    }
	          else if ((GET_CODE (cur_rtx) == SET || GET_CODE (cur_rtx) == CLOBBER)
		          && GET_CODE (SET_DEST (cur_rtx)) == MEM
	                  && GET_CODE (dest) == MEM && !only_symref_plus_reg 
		          && flag_replace_mem)
		    {
		      dest = SET_DEST (cur_rtx);
		      for (i = 0; i < cur_max_reg; i++)
		        {
		          if (rinfo[i].corresp_mem 
			      && true_dependence (dest, rinfo[i].corresp_mem))
			    { /* Memory address maybe overwritten.  */
			      rinfo[i].corresp_mem = NULL;
			    }
		        }
		    }
		  else if (!(GET_CODE (cur_rtx) == SET  
		       || GET_CODE (cur_rtx) == CLOBBER
		       || GET_CODE (cur_rtx) == USE))
	            bzero (rinfo, cur_max_reg * sizeof (struct regopt_info));
		}
	    }
	  else if (GET_CODE (PATTERN (insn)) == SET)
	    {
	      dest = SET_DEST (PATTERN (insn));
	      if (GET_CODE (SET_SRC (PATTERN (insn))) != REG)
		{
		  header = NULL;
		  storage2 = (char *) obstack_alloc (&momentary_obstack, 0);
		  find_regs_mentioned (SET_SRC (PATTERN (insn)), &header, 0);
		  a_reg_used = header;
		  while (a_reg_used)
		    {
		      undobuf.num_undo = 0;
		      undobuf.storage = (char *) oballoc (0);
		      if (((a_reg_used->in_mem || !only_symref_plus_reg) 
			   && rinfo[REGNO (a_reg_used->reg)].insn_set 
			   && rinfo[REGNO (a_reg_used->reg)].replace_reg == 0 
			   && (!only_symref_plus_reg 
			       || SYMREF_PLUS_REG (rinfo[REGNO (a_reg_used->reg)].insn_set))) 
			  || (rinfo[REGNO (a_reg_used->reg)].corresp_mem 
			   && rinfo[REGNO (a_reg_used->reg)].insn_set 
			   && rinfo[REGNO (a_reg_used->reg)].replace_reg == 1 
			   && rinfo[REGNO (a_reg_used->reg)].do_replace == 1 
#ifdef STACK_REGS
#ifdef IS_STACK_MODE
			  && !((GET_CODE (SET_SRC (PATTERN (insn))) == FLOAT_EXTEND 
		      || GET_CODE (SET_SRC (PATTERN (insn))) == FLOAT_TRUNCATE)
		      && (IS_STACK_MODE (SFmode) || IS_STACK_MODE (DFmode)))
			   
#endif
#endif
			   && !only_symref_plus_reg && flag_replace_stack_mem)
			)
			{ /* Try to replace the register with it's current
		              value.  */
			  replace_reg (SET_SRC (PATTERN (insn)),
				       &SET_SRC (PATTERN (insn)),
				       a_reg_used->reg,
				       &(rinfo[REGNO (a_reg_used->reg)]));
			  if ((insn_code_num = recog (PATTERN (insn), insn, 0)) < 0)
			    {
			      recog_ok = 0;
			      try_to_make_good (SET_SRC (PATTERN (insn)));
			    }
			  else
			    {
			      recog_ok = 1;
			    }
			  if (recog_ok 
			      || (insn_code_num = recog (PATTERN (insn), insn, 0)) >= 0)
			    {
			      INSN_CODE (insn) = insn_code_num;
			      /* canonicalize */
			      if (GET_CODE (SET_SRC (PATTERN (insn))) == PLUS 
				  && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) == MULT 
				  && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 0)) == REG)
				{
				  tmp_rtx = XEXP (SET_SRC (PATTERN (insn)), 0);
				  XEXP (SET_SRC (PATTERN (insn)), 0) 
				    = XEXP (SET_SRC (PATTERN (insn)), 1);
				  XEXP (SET_SRC (PATTERN (insn)), 1) = tmp_rtx;
				}
			    }
			  else
			    {
			      loop_undo_all ();
			    }
			}
#ifdef STACK_REGS
#ifdef IS_STACK_MODE
		      if (IS_STACK_MODE (GET_MODE (a_reg_used->reg)))
			{
			  rinfo[REGNO (a_reg_used->reg)].do_replace = 1;
			}
#endif
#endif
		      a_reg_used = a_reg_used->next;
		    }
		  if (header != NULL)
		    {
		      obstack_free (&momentary_obstack, storage2);
		    }
		  if (!only_symref_plus_reg && flag_replace_mem)
		    replace_mems (insn, SET_SRC (PATTERN (insn)));
		}
	      src = SET_SRC (PATTERN (insn));
	      if (GET_CODE (dest) != REG)
		{
		  header = NULL;
		  storage2 = (char *) obstack_alloc (&momentary_obstack, 0);
		  find_regs_mentioned (SET_DEST (PATTERN (insn)), &header, 0);
		  a_reg_used = header;
		  while (a_reg_used)
		    {
		      undobuf.num_undo = 0;
		      undobuf.storage = (char *) oballoc (0);
		      if (a_reg_used->in_mem 
			  && rinfo[REGNO (a_reg_used->reg)].insn_set 
			  && (!only_symref_plus_reg 
			   || SYMREF_PLUS_REG (rinfo[REGNO (a_reg_used->reg)].insn_set))
			)
			{ /* Try to replace the register with it's current
		             value.  */
			  replace_reg (SET_DEST (PATTERN (insn)),
				       &SET_DEST (PATTERN (insn)),
				       a_reg_used->reg,
				       &(rinfo[REGNO (a_reg_used->reg)]));

			  if (recog (PATTERN (insn), insn, 0) < 0)
			    {
			      recog_ok = 0;
			      try_to_make_good (SET_DEST (PATTERN (insn)));
			    }
			  else
			    {
			      recog_ok = 1;
			    }
			  if (recog_ok 
			      || recog (PATTERN (insn), insn, 0) >= 0)
			    {
			      ;
			    }
			  else
			    {
			      loop_undo_all ();
			    }
			}
		      a_reg_used = a_reg_used->next;
		    }
		  if (header != NULL)
		    {
		      obstack_free (&momentary_obstack, storage2);
		    }
		  if (!only_symref_plus_reg 
		      && GET_CODE (src) == REG && GET_CODE (dest) == MEM 
		      && !MEM_VOLATILE_P (dest) 
		      && GET_CODE (XEXP (dest, 0)) != PRE_DEC 
		      && GET_CODE (XEXP (dest, 0)) != PRE_INC
#ifdef STACK_REGS
#ifdef IS_STACK_MODE
		      && !(IS_STACK_MODE (GET_MODE (src)))
#endif
#endif
		    )
		    {
		      rinfo[REGNO (src)].corresp_mem = dest;
		      rinfo[REGNO (src)].reg = src;
		      rinfo[REGNO (src)].replace_reg = 0;
		    }
		}
	      if (GET_CODE (dest) == MEM && !only_symref_plus_reg 
		  && flag_replace_mem)
		{
		  for (i = 0; i < cur_max_reg; i++)
		    {
		      if (rinfo[i].corresp_mem && !(GET_CODE (src) == REG && i == REGNO (src)) 
			  && true_dependence (dest, rinfo[i].corresp_mem))
			{ /* Memory address maybe overwritten.  */
			  rinfo[i].corresp_mem = NULL;
			}
		    }
		}
	      if (GET_CODE (dest) == REG)
		{
		  for (i = 0; i < cur_max_reg; i++)
		    {
		      if (rinfo[i].insn_set 
			  && reg_overlap_mentioned_p (dest, SET_SRC (PATTERN (rinfo[i].insn_set)))
			)
			{ /* Current value of reg dest overwritten.  */
			  rinfo[i].insn_set = NULL;
			}
		      if (rinfo[i].corresp_mem 
		          && reg_overlap_mentioned_p (dest, rinfo[i].corresp_mem))
			{ /* Not the same memory address anymore.  */
			  rinfo[i].corresp_mem = NULL;
			}
		    }
		  rinfo[REGNO (dest)].corresp_mem = NULL;
		  if (reg_overlap_mentioned_p (dest, SET_SRC (PATTERN (insn))))
		    {
		      rinfo[REGNO (dest)].insn_set = NULL;
		    }
		  else
		    {
		      if ((GET_CODE (SET_SRC (PATTERN (insn))) == ASHIFT 
		           && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 0)) == REG 
			   && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) == CONST_INT) 
			   || (GET_CODE (SET_SRC (PATTERN (insn))) == PLUS 
		               && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 0)) == REG 
			       && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) == SYMBOL_REF) 
			   || (GET_CODE (SET_SRC (PATTERN (insn))) == PLUS 
			       && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 0)) == MULT 
			       && GET_CODE (XEXP (XEXP (SET_SRC (PATTERN (insn)), 0), 0)) == REG 
			       && GET_CODE (XEXP (XEXP (SET_SRC (PATTERN (insn)), 0), 1)) == CONST_INT 
			       && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) == CONST_INT)
			)
			{ /* These are the patterns of interest.  */
			  rinfo[REGNO (dest)].insn_set = insn;
			}
		      else
			{
			  rinfo[REGNO (dest)].insn_set = NULL;
			}
#ifdef STACK_REGS
#ifdef IS_STACK_MODE
		      if (IS_STACK_MODE (GET_MODE (dest)) 
			      && ((GET_CODE (src) == MEM 
			           && !MEM_VOLATILE_P (src)) 
		               || ((IS_STACK_MODE (SFmode) || IS_STACK_MODE (DFmode)) 
		                    && (GET_CODE (src) == FLOAT_EXTEND 
			                    || GET_CODE (src) == FLOAT_TRUNCATE) 
		          && GET_CODE (XEXP (src, 0)) == MEM 
		          && !MEM_VOLATILE_P (XEXP (src, 0))))
		          && flag_replace_stack_mem)
			{
			  rinfo[REGNO (dest)].corresp_mem = src;
			  rinfo[REGNO (dest)].reg = dest;
			  rinfo[REGNO (dest)].insn_set = insn;
			  rinfo[REGNO (dest)].replace_reg = 1;
			  rinfo[REGNO (dest)].do_replace = 0;
			}
#endif
#endif
		    }
		}
	    }
	  else if (GET_CODE (PATTERN (insn)) != USE)
	    {
	      bzero (rinfo, cur_max_reg * sizeof (struct regopt_info));
	    }
	  break;

	default:
	  bzero (rinfo, cur_max_reg * sizeof (struct regopt_info));
	  break;
	}
      insn = NEXT_INSN (insn);
    }
}

void
loop_undo_all ()
{
  register int i;
  if (undobuf.num_undo > MAX_UNDO)
    undobuf.num_undo = MAX_UNDO;
  for (i = undobuf.num_undo - 1; i >= 0; i--)
    *undobuf.undo[i].where = undobuf.undo[i].old_contents;

  obfree (undobuf.storage);
  undobuf.num_undo = 0;
}

/* Replace in pat of insn each MEM with a corresponding reg if one exists.  */
static
rtx 
replace_mems (insn, pat)
     rtx insn, pat;
{
  register RTX_CODE code = GET_CODE (pat);
  register int i, j;
  register char *fmt;

  switch (code)
    {
    case HIGH:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
    case REG:
      return pat;


    }

  if (GET_CODE (pat) == MEM)
    {
      for (i = 0; i < mem_max_reg_num; i++)
	{
	  if (rinfo[i].corresp_mem && rinfo[i].replace_reg == 0 
	      && rtx_equal_p (rinfo[i].corresp_mem, pat))
	    return rinfo[i].reg;
	}
    }
  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      XEXP (pat, i) = replace_mems (insn, XEXP (pat, i));
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (pat, i); j++)
	  XVECEXP (pat, i, j) 
	    = replace_mems (insn, XVECEXP (pat, i, j));
      }
  return pat;
}


/* Replace in orig each occurrence of reg_rplc with the value
   described by *rptr.  The location in orig of reg_rplc is where.  
*/
static void
replace_reg (orig, where, reg_rplc, rptr)
     register rtx orig, reg_rplc;
     rtx *where;
     struct regopt_info *rptr;
{
  rtx non_biv_operand, ret_rtx;
  register int i, j;
  register RTX_CODE code;
  register char *fmt;

  code = GET_CODE (orig);

  switch (code)
    {
    case REG:
      if (rtx_equal_p (orig, reg_rplc))
	{
	  if (GET_CODE (SET_SRC (PATTERN (rptr->insn_set))) == ASHIFT 
	      && GET_CODE (XEXP (SET_SRC (PATTERN (rptr->insn_set)), 1)) == CONST_INT
	    )
	    {
	      LOOP_SUBST (where,
	      (gen_rtx (MULT, GET_MODE (SET_SRC (PATTERN (rptr->insn_set))),
			XEXP (SET_SRC (PATTERN (rptr->insn_set)), 0),
			gen_rtx (CONST_INT, 0,
				 two_to_power (INTVAL (XEXP (SET_SRC (PATTERN (rptr->insn_set)), 1)))))));
	    }
	  else
	    {
	      LOOP_SUBST (where, (SET_SRC (PATTERN (rptr->insn_set))));
	    }
	}
      break;

    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
    case PC:
    case CC0:
      return;
    }


  fmt = GET_RTX_FORMAT (code);

  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      replace_reg (XEXP (orig, i), &XEXP (orig, i), reg_rplc, rptr);
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (orig, i); j++)
	  replace_reg (XVECEXP (orig, i, j), &XVECEXP (orig, i, j), reg_rplc, rptr);
      }
}

static int
two_to_power (x)
     int x;
{
  int retval, i;

  retval = 1;
  for (i = 0; i < x; i++)
    retval *= 2;
  return retval;
}

/* Return 1 if insn p jumps backward to a label that is located
   between loop_start and p,
   if strict==1 we also require that any jumps between the target label and
   p jump to labels that are between the target label and p.  */
static int
jumps_back_in_loop (p, loop_start, use1, strict)
     rtx p, loop_start;
     int use1;		/* If 1 then we require each label within to be accessed only once */
     int strict;
{
  rtx insn, target, last_jump_found;
  int jumps_found, labels_found;

  if (!flag_jump_back)
    return 0;

  target = JUMP_LABEL (p);
  insn = PREV_INSN (p);
  jumps_found = 0;
  labels_found = 0;
  while (insn != target && insn != loop_start)
    {
      if (GET_CODE (insn) == JUMP_INSN)
	{
	  jumps_found++;
	}
      else if (GET_CODE (insn) == CODE_LABEL)
	{
	  if (use1 && LABEL_NUSES (insn) != 1)
	    return 0;
	  
	  labels_found++;
	}

      insn = PREV_INSN (insn);
    }
  if (insn != target)
    {
      return 0;
    }
  else
    {
      if (use1 && jumps_found != labels_found)
	return 0;

      if (jumps_found == 0 || strict == 0)
	{
	  return 1;
	}
      else
	{ /* Must check each jump to see that it jumps to label
	     between target and p.  */
	  last_jump_found = target;
	  while (jumps_found)
	    {
	      /* Find the next jump.  */
	      insn = NEXT_INSN (last_jump_found);
	      while (GET_CODE (insn) != JUMP_INSN)
		insn = NEXT_INSN (insn);

	      last_jump_found = insn;
	      /* Now find the target. */
	      while (insn != p && insn != JUMP_LABEL (last_jump_found))
		{ /* Search forward.  */
		  insn = NEXT_INSN (insn);
		}
	      if (insn == JUMP_LABEL (last_jump_found) 
		  && (!use1 || LABEL_NUSES (JUMP_LABEL (last_jump_found)) == 1))
		{
		  jumps_found--;
		}
	      else
		{
		  while (insn != loop_start && insn != JUMP_LABEL (last_jump_found))
		    { /* Search backwards. */
		      insn = PREV_INSN (insn);
		    }
		  if (insn == JUMP_LABEL (last_jump_found) 
		      && (!use1 || LABEL_NUSES (JUMP_LABEL (last_jump_found)) == 1))
		    jumps_found--;
		  else
		    break;
		}
	    }
	  return jumps_found == 0;
	}
    }
}


struct propogate_cand
{
  rtx srcpat;
  rtx add_reg0;
  rtx add_reg1;
  rtx mul_reg;
  rtx reg;
  int mul_val;
  char cand_type;
};

struct unroll_combine_cand
{
  rtx reg;
  struct unroll_combine_cand *next;
};

#define COPY 1
#define GIV 2

/* Try to propogate copies into loop as follows:
   If before the loop we have
   regx <- regy
   and in the loop regx is only used, then replace each use of regx
   in the loop with regy
*/
static
loop_copy_propogate (loop_start, loop_end)
     rtx loop_start, loop_end;
{
  struct propogate_cand *copy_candidates, *possible_giv_candidate;
  char *regs_set, dummyc;
  rtx insn, loop_label, loop_jump, dest, src, *split_giv_src;
  int num_copy_candidates, num_giv_candidates,
    stop_looking, cur_max_regnum, i, dummyi, num_unroll_candidates;
  struct unroll_combine_cand **unroll_cands, *an_unroll_cand;

  cur_max_regnum = max_reg_num () + 1;
  if (loop_dump_stream)
    {
      fprintf (loop_dump_stream, "copy prop from %d to %d\n",
	       INSN_UID (loop_start), INSN_UID (loop_end));
    }
  copy_candidates = (struct propogate_cand *)
    alloca (cur_max_regnum * sizeof (struct propogate_cand));
  bzero (copy_candidates, (cur_max_regnum * sizeof (struct propogate_cand)));
  regs_set = (char *) alloca (cur_max_regnum);
  bzero (regs_set, cur_max_regnum);
  n_times_set = (short *) alloca (cur_max_regnum * sizeof (short));
  bzero (n_times_set, cur_max_regnum * sizeof (short));
  may_not_optimize = (char *) alloca (cur_max_regnum);	/* not used */
  if (flag_unroll_loops)
    {
      unroll_cands 
	= (struct unroll_combine_cand **)
	alloca (cur_max_regnum * sizeof (struct unroll_combine_cand *));
      bzero (unroll_cands, cur_max_regnum * sizeof (struct unroll_combine_cand *));
      split_giv_src = (rtx *) alloca (cur_max_regnum * sizeof (rtx));
      bzero (split_giv_src, cur_max_regnum * sizeof (rtx));
    }
  /* First check that the label that is the start of the loop
     is only jumped to from within the loop - if not return
  */
  insn = loop_start;
  while (insn && GET_CODE (insn) != CODE_LABEL)
    {
      insn = NEXT_INSN (insn);
    }
  loop_label = insn;
  insn = loop_end;
  while (insn && GET_CODE (insn) != JUMP_INSN)
    {
      insn = PREV_INSN (insn);
    }
  loop_jump = insn;
  if (!loop_label || !loop_jump 
      || (JUMP_LABEL (loop_jump) != loop_label) 
      || (LABEL_NUSES (JUMP_LABEL (loop_jump)) != 1)
    )
    {
      return;
    }
  /* Compute n_times_set */
  count_loop_regs_set (loop_start, loop_end, may_not_optimize, NULL,
		       &dummyi, cur_max_regnum);

  /* Look backwards from loop_start until a label barier or
     uncoditional jump is reached,
     searching for insns of the form regx <- regy and regx
     NOT set in loop  - these are candidates for copy propogation.  
     Also search for registers that are set rega <- regb*C + regd
     - rega is a candidate for inter loop giv combination.  
  */
  insn = loop_start;
  num_copy_candidates = 0;
  num_giv_candidates = 0;
  num_unroll_candidates = 0;
  stop_looking = 0;
  possible_giv_candidate = NULL;
  while (insn && !stop_looking)
    {
      switch (GET_CODE (insn))
	{
	case NOTE:
	  break;

	case JUMP_INSN:
	  if (SET_DEST (PATTERN (insn)) == pc_rtx
	      && GET_CODE (SET_SRC (PATTERN (insn))) == LABEL_REF)
	    {
	      stop_looking = 1;
	    }
	  break;

	case CODE_LABEL:
	case BARRIER:
	  stop_looking = 1;
	  break;

	case INSN:
	  if (GET_CODE (PATTERN (insn)) != SET)
	    {
	      stop_looking = 1;
	    }
	  else
	    {
	      dest = SET_DEST (PATTERN (insn));
	      src = SET_SRC (PATTERN (insn));
	      if (GET_CODE (dest) == REG)
		{
		  if (flag_unroll_loops && dest != cc0_rtx 
		      && REGNO (dest) >= FIRST_PSEUDO_REGISTER 
		      && dest != pc_rtx && regs_set[REGNO (dest)] == 0 
		      && GET_CODE (src) == REG && regs_set[REGNO (src)] == 0
		      && REGNO (src) >= FIRST_PSEUDO_REGISTER )
		    { /* The dest may be a split giv which we want
                         to combine */
		      if (split_giv_src[REGNO (dest)] == NULL)
			{
			  split_giv_src[REGNO (dest)] = src;
			  an_unroll_cand 
			    = (struct unroll_combine_cand *)
			    alloca (sizeof (struct unroll_combine_cand));
			  an_unroll_cand->reg = dest;
			  an_unroll_cand->next = unroll_cands[REGNO (src)];
			  unroll_cands[REGNO (src)] = an_unroll_cand;
			  num_unroll_candidates++;
			}
		      else
			{
			  split_giv_src[REGNO (dest)] = (rtx) - 1;
			}
		    }

		  if (dest != cc0_rtx && dest != pc_rtx 
		      && REGNO (dest) >= FIRST_PSEUDO_REGISTER 
		      && regs_set[REGNO (dest)] == 0)
		    {
		      if (GET_CODE (src) == REG 
			  && REGNO (src) >= FIRST_PSEUDO_REGISTER 
			  && n_times_set[REGNO (src)] == 0 
			  && regs_set[REGNO (src)] == 0)
			{
			  if (n_times_set[REGNO (dest)] == 0)
			    {
			      num_copy_candidates++;
			      copy_candidates[REGNO (dest)].srcpat = src;
			      copy_candidates[REGNO (dest)].cand_type = COPY;
			    }
			}
		      else
			{
			  if (GET_CODE (src) == PLUS 
			      && GET_CODE (XEXP (src, 0)) == REG 
			      && REGNO (XEXP (src, 0)) >= FIRST_PSEUDO_REGISTER 
			      && GET_CODE (XEXP (src, 1)) == REG 
			      && REGNO (XEXP (src, 1)) >= FIRST_PSEUDO_REGISTER 
			      && regs_set[REGNO (XEXP (src, 0))] == 0 
			      && regs_set[REGNO (XEXP (src, 1))] == 0 
			      && n_times_set[REGNO (XEXP (src, 1))] == 0 
			      && copy_candidates[REGNO (dest)].srcpat == NULL 
			      && copy_candidates[REGNO (dest)].add_reg0 == NULL)
			    {
			      copy_candidates[REGNO (dest)].add_reg0 
				= XEXP (src, 0);
			      copy_candidates[REGNO (dest)].add_reg1 
				= XEXP (src, 1);
			      possible_giv_candidate 
				= &copy_candidates[REGNO (dest)];
			      possible_giv_candidate->reg = dest;
			    }
			  else if (GET_CODE (src) == ASHIFT 
				   && GET_CODE (XEXP (src, 0)) == REG 
				   && GET_CODE (XEXP (src, 1)) == CONST_INT 
				   && regs_set[REGNO (XEXP (src, 0))] == 0 
				   && possible_giv_candidate 
				   && possible_giv_candidate->mul_reg == NULL 
				   && rtx_equal_p (dest, possible_giv_candidate->add_reg0)
			    )
			    {
			      possible_giv_candidate->mul_reg = XEXP (src, 0);
			      possible_giv_candidate->mul_val 
				= two_to_power (INTVAL (XEXP (src, 1)));
			      possible_giv_candidate->cand_type = GIV;
			      num_giv_candidates++;
			    }
			}
		    }
		  regs_set[REGNO (dest)] = 1;
		}
	    }
	  break;

	default:
	  stop_looking = 1;
	  break;
	}
      insn = PREV_INSN (insn);
    }
  if (num_copy_candidates != 0)
    {
      /* Now for each candidate do the replacement */
      do_replace (loop_start, loop_end, copy_candidates);
    }
  if (num_unroll_candidates != 0)
    {
      try_unroll_propogation (loop_start, loop_end, unroll_cands, split_giv_src);
    }
}

/*
  Each occurence of a copy_candidate in an insn between loop_start
  and loop_end is replaced by the corresponding register.  
*/
static
do_replace (loop_start, loop_end, copy_candidates)
     rtx loop_start, loop_end;
     struct propogate_cand *copy_candidates;
{
  rtx insn, subst_cands ();

  for (insn = loop_start; insn != loop_end; insn = NEXT_INSN (insn))
    {
      if (GET_CODE (insn) == INSN)
	{
	  subst_cands (insn, PATTERN (insn), copy_candidates);
	}
    }
}


/* Each register in pat which is a copy_candidate of type COPY is replaced with
   it's copy from outside the loop.  
*/
static
rtx 
subst_cands (insn, pat, copy_candidates)
     rtx insn, pat;
     struct propogate_cand *copy_candidates;
{
  register RTX_CODE code = GET_CODE (pat);
  register int i, j;
  register char *fmt;

  switch (code)
    {
    case HIGH:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
      return pat;

    case REG:
      {
	register int regno = REGNO (pat);

	if (copy_candidates[regno].srcpat 
	    && copy_candidates[regno].cand_type == COPY)
	  {
	    if (loop_dump_stream)
	      {
		fprintf (loop_dump_stream, "replacing reg%d with reg%d in insn %d\n",
			 regno, REGNO (copy_candidates[regno].srcpat), INSN_UID (insn));
	      }
	    return copy_candidates[regno].srcpat;
	  }
      }
      return pat;

    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      XEXP (pat, i) = subst_cands (insn, XEXP (pat, i), copy_candidates);
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (pat, i); j++)
	  XVECEXP (pat, i, j) 
	    = subst_cands (insn, XVECEXP (pat, i, j), copy_candidates);
      }
  return pat;
}

struct reg_incs
{
  int inc_val;
  char invalid;
};

/* Try to combine split givs from unrolling into one giv to decrease
   register pressure and increase scheduling possiblities
*/
static
try_unroll_propogation (loop_start, loop_end, unroll_cands, split_giv_src)
     rtx loop_start, loop_end;
     struct unroll_combine_cand **unroll_cands;
     rtx *split_giv_src;
{
  rtx insn, src, dest, ninsn, nsrc, ndest, unroll_subst_cands ();
  struct reg_incs *reg_inc;
  int i, cur_max_regnum, found;

  if (loop_dump_stream)
    {
      fprintf (loop_dump_stream, " unroll propogate from %d to %d\n",
	       INSN_UID (loop_start), INSN_UID (loop_end));
    }
  cur_max_regnum = max_reg_num () + 1;
  reg_inc 
    = (struct reg_incs *) alloca (cur_max_regnum * (sizeof (struct reg_incs)));
  bzero (reg_inc, cur_max_regnum * (sizeof (struct reg_incs)));
  insn = NEXT_INSN (loop_start);
  while (insn != loop_end)
    {
      switch (GET_CODE (insn))
	{

	case NOTE:
	  break;

	case JUMP_INSN:
	case CODE_LABEL:
	case BARRIER:
	  unroll_consistancy_maintained (cur_max_regnum, unroll_cands,
					 reg_inc, insn, split_giv_src);
	  /* forget everything */
	  bzero (reg_inc, cur_max_regnum * (sizeof (struct reg_incs)));
	  break;

	case INSN:
	  if (GET_CODE (PATTERN (insn)) == PARALLEL)
	    {
	      return;
	    }
	  if (GET_CODE (PATTERN (insn)) != SET)
	    {
	      return;
	    }
	  dest = SET_DEST (PATTERN (insn));
	  if (GET_CODE (dest) == REG)
	    {
	      if (GET_CODE (SET_SRC (PATTERN (insn))) == PLUS 
		  && rtx_equal_p (XEXP (SET_SRC (PATTERN (insn)), 0), dest) 
		  && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) == CONST_INT)
		{ /* reg incremented by constatnt */
		  reg_inc[REGNO (dest)].inc_val 
		    += INTVAL (XEXP (SET_SRC (PATTERN (insn)), 1));
		}
	      else
		{
		  reg_inc[REGNO (dest)].invalid = 1;
		}
	    }
	  break;

	default:
	  return;
	  break;
	}
      insn = NEXT_INSN (insn);
    }
  insn = NEXT_INSN (loop_start);
  /* Combine each valid unroll candidate */
  for (insn = loop_start; insn != loop_end; insn = NEXT_INSN (insn))
    {
      if (GET_CODE (insn) == INSN)
	{
	  unroll_subst_cands (insn, PATTERN (insn), unroll_cands, split_giv_src);
	}
    }
}


/* Try to combine split givs from unrolling into one giv to decrease
   register pressure and increase scheduling possiblities
*/
static
rtx 
unroll_subst_cands (insn, pat, unroll_cands, split_giv_src)
     rtx insn, pat;
     struct unroll_combine_cand **unroll_cands;
     rtx *split_giv_src;
{
  register RTX_CODE code = GET_CODE (pat);
  register int i, j;
  register char *fmt;

  if (GET_CODE (pat) == REG && pat == SET_DEST (PATTERN (insn)))
    return pat;

  switch (code)
    {
    case HIGH:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
      return pat;

    case REG:
      {
	register int regno = REGNO (pat);

	if (split_giv_src[regno] && split_giv_src[regno] != (rtx) - 1 
	    && unroll_cands[REGNO (split_giv_src[regno])])
	  {
	    if (loop_dump_stream)
	      {
		fprintf (loop_dump_stream, "   replacing reg%d with reg%d in insn %d\n",
		      regno, REGNO (split_giv_src[regno]), INSN_UID (insn));
	      }
	    return split_giv_src[regno];
	  }
      }
      return pat;

    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      XEXP (pat, i) 
	= unroll_subst_cands (insn, XEXP (pat, i), unroll_cands, split_giv_src);
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (pat, i); j++)
	  XVECEXP (pat, i, j) 
	    = unroll_subst_cands (insn, XVECEXP (pat, i, j), unroll_cands,
			  split_giv_src);
      }
  return pat;
}

/* Check that spilt givs remain consistant */
static
unroll_consistancy_maintained (cur_max_regnum, unroll_cands, reg_inc, insn,
			       split_giv_src)
     int cur_max_regnum;
     struct unroll_combine_cand **unroll_cands;
     struct reg_incs *reg_inc;
     rtx *split_giv_src;
     rtx insn;
{
  int i;
  struct unroll_combine_cand *an_unroll_cand;

  for (i = 0; i < cur_max_regnum; i++)
    {
      if (unroll_cands[i])
	{
	  if (reg_inc[i].invalid)
	    {
	      unroll_cands[i] = NULL;
	    }
	  else
	    { /* Check each candidate to see that it was incremented
                 by same amount.  */
	      an_unroll_cand = unroll_cands[i];
	      while (an_unroll_cand)
		{
		  if (reg_inc[REGNO (an_unroll_cand->reg)].invalid 
		      || reg_inc[REGNO (an_unroll_cand->reg)].inc_val !=
		      reg_inc[i].inc_val)
		    {
		      split_giv_src[REGNO (an_unroll_cand->reg)] = (rtx) - 1;
		    }
		  an_unroll_cand = an_unroll_cand->next;
		}
	    }
	}
    }
}


#ifdef STORE_NOT_SET_CC0
#ifdef STACK_REGS
/* do_spl is called after reload because the reloader decides to spill
   fp regs, for no appareant reason, if it is done before.
*/
void
do_spl (f)
     /* f is the first instruction of a chain of insns for one function */
     rtx f;
{
  register rtx insn;
  register int i;
  rtx end;
  rtx last_insn;

#ifndef HAVE_cc0
  return;
#endif

  /* Count the number of loops. */

  max_loop_num = 0;
  for (insn = f; insn; insn = NEXT_INSN (insn))
    if (GET_CODE (insn) == NOTE
	&& NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_BEG)
      max_loop_num++;

  /* Don't waste time if no loops.  */
  if (max_loop_num == 0)
    return;

  doing_spl = 1;
  /* Get size to use for tables indexed by uids.
     Leave some space for labels allocated by find_and_verify_loops.  */
  max_uid_for_loop = get_max_uid () + 1 + max_loop_num * 4;

  uid_luid = (int *) alloca (max_uid_for_loop * sizeof (int));
  uid_loop_num = (int *) alloca (max_uid_for_loop * sizeof (int));

  bzero (uid_luid, max_uid_for_loop * sizeof (int));
  bzero (uid_loop_num, max_uid_for_loop * sizeof (int));

  /* Allocate tables for recording each loop.  We set each entry, so they need
     not be zeroed.  */
  loop_number_loop_starts = (rtx *) alloca (max_loop_num * sizeof (rtx));
  loop_number_loop_ends = (rtx *) alloca (max_loop_num * sizeof (rtx));
  loop_outer_loop = (int *) alloca (max_loop_num * sizeof (int));
  loop_invalid = (char *) alloca (max_loop_num * sizeof (char));
  loop_number_exit_labels = (rtx *) alloca (max_loop_num * sizeof (rtx));
  loop_number_exit_count = (int *) alloca (max_loop_num * sizeof (int));

  /* Find and spl each loop.
     First, find them, and record them in order of their beginnings.  */
  find_and_verify_loops (f);

  /* See if we went too far.  */
  if (get_max_uid () > max_uid_for_loop)
    abort ();

  for (i = max_loop_num - 1; i >= 0; i--)
    {
      if (loop_number_loop_ends[i])
	loop_spl (loop_number_loop_starts[i], loop_number_loop_ends[i], i, 1);
    }
  doing_spl = 0;
}

/*
   A mini software pipelining - consists of two steps
   1) try to get the floating point store to be the last instruction
      executed before the jump at the end of loop.
   2) try to get the jump to be executed before the floating point
      store by retargeting it to the begining of loop and moving
      the floating point store to there.  Note that the fall into the
      loop will be replaced by a jump around this floating point store
   If do_all is 0 then just do step 1
*/
static
loop_spl (loop_start, loop_end, loop_num, do_all)
     rtx loop_start, loop_end;
     int loop_num, do_all;
{
  rtx insn, jump_insn, mem_addr, base_reg, index_reg, copy_pat, label_insn,
    label, p, *index_where, *base_where, pinsn, cc0_insn, where_to_move;
  struct reg_incs *reg_inc;
  int cur_max_regnum, scale;
  char *storage;

  if (loop_dump_stream)
    {
      fprintf (loop_dump_stream, "loop_spl from %d to %d\n",
	       INSN_UID (loop_start), INSN_UID (loop_end));
    }
  cur_max_regnum = max_reg_num () + 1;
  reg_inc 
    = (struct reg_incs *) alloca (cur_max_regnum * (sizeof (struct reg_incs)));
  bzero (reg_inc, cur_max_regnum * (sizeof (struct reg_incs)));
  insn = PREV_INSN (loop_end);
  while (GET_CODE (insn) == INSN && GET_CODE (PATTERN (insn)) == SET 
	 && rtx_equal_p (SET_DEST (PATTERN (insn)), SET_SRC (PATTERN (insn))))
    insn = PREV_INSN (insn);

  if (GET_CODE (insn) != JUMP_INSN)
    return;

  jump_insn = insn;
  insn = PREV_INSN (jump_insn);
  if (!(GET_CODE (insn) == INSN && GET_CODE (PATTERN (insn)) == SET 
	&& SET_DEST (PATTERN (insn)) == cc0_rtx))
    return;

  cc0_insn = insn;
  base_reg = NULL;
  index_reg = NULL;
  scale = 1;
  insn = prev_nonnote_insn (insn);
  while (1)
    {
      if (GET_CODE (insn) != INSN)
	return;

      if (GET_CODE (PATTERN (insn)) != SET)
	return;

      if (GET_CODE (SET_DEST (PATTERN (insn))) == REG 
	  && (GET_CODE (SET_SRC (PATTERN (insn))) == PLUS 
	   || GET_CODE (SET_SRC (PATTERN (insn))) == MINUS) 
	  && rtx_equal_p (SET_DEST (PATTERN (insn)),
		       XEXP (SET_SRC (PATTERN (insn)), 0)) 
	  && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) == CONST_INT)
	{
	  reg_inc[REGNO (SET_DEST (PATTERN (insn)))].inc_val 
	    += INTVAL (XEXP (SET_SRC (PATTERN (insn)), 1));
	  if (GET_CODE (SET_SRC (PATTERN (insn))) == MINUS)
	    reg_inc[REGNO (SET_DEST (PATTERN (insn)))].inc_val *= -1;
	  insn = prev_nonnote_insn (insn);
	}
      else
	break;
    }
  if (!(GET_CODE (insn) == INSN && GET_CODE (PATTERN (insn)) == SET 
	&& (GET_CODE (SET_SRC (PATTERN (insn))) == REG 
	|| (GET_CODE (SET_SRC (PATTERN (insn))) == FLOAT_TRUNCATE 
	&& GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 0)) == REG) 
	|| (GET_CODE (SET_SRC (PATTERN (insn))) == FLOAT_EXTEND 
	&& GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 0)) == REG))
	&& GET_CODE (SET_DEST (PATTERN (insn))) == MEM 
	&& (GET_MODE (SET_DEST (PATTERN (insn))) == SFmode 
	|| GET_MODE (SET_DEST (PATTERN (insn))) == DFmode) 
        && GET_MODE (SET_DEST (PATTERN (insn))) == 
            GET_MODE (SET_SRC (PATTERN (insn)))))
    return;

  /* if previous fp instruction is fld - no gain */
  pinsn = prev_nonnote_insn (insn);
  if (INSN_UID (pinsn) < INSN_UID (loop_start))
    return;

  while (pinsn != loop_start)
    {
      if (GET_CODE (pinsn) == JUMP_INSN || GET_CODE (pinsn) == CODE_LABEL 
	  || (GET_CODE (pinsn) == INSN && GET_CODE (PATTERN (pinsn)) == SET 
	   && (GET_MODE (SET_DEST (PATTERN (pinsn))) == SFmode 
	    || GET_MODE (SET_DEST (PATTERN (pinsn))) == DFmode)))
	break;

      pinsn = PREV_INSN (pinsn);
    }

  if (pinsn == loop_start || GET_CODE (pinsn) == JUMP_INSN 
      || GET_CODE (pinsn) == CODE_LABEL)
    return;

  if ((GET_CODE (SET_SRC (PATTERN (pinsn))) == MEM 
	   || GET_CODE (SET_SRC (PATTERN (pinsn))) == CONST_DOUBLE 
	   || GET_CODE (SET_SRC (PATTERN (pinsn))) == REG))
    return;

  /* We have found the floating point store insn - now try to move it down */
  /* Find the registers that are used as bases for the memory access in the
     insn */
  storage = (char *) oballoc (0);
  copy_pat = copy_rtx (PATTERN (insn));
  mem_addr = XEXP (SET_DEST (copy_pat), 0);
  if (GET_CODE (mem_addr) == PLUS)
    {
      if (GET_CODE (XEXP (mem_addr, 0)) == MULT)
	{
	  if (GET_CODE (XEXP (XEXP (mem_addr, 0), 0)) == REG 
	      && GET_CODE (XEXP (XEXP (mem_addr, 0), 1)) == CONST_INT)
	    {
	      index_where = &XEXP (XEXP (mem_addr, 0), 0);
	      index_reg = XEXP (XEXP (mem_addr, 0), 0);
	      scale = INTVAL (XEXP (XEXP (mem_addr, 0), 1));
	    }
	  else
	    return;

	  if (GET_CODE (XEXP (mem_addr, 1)) == REG)
	    {
	      base_where = &XEXP (mem_addr, 1);
	      base_reg = XEXP (mem_addr, 1);
	    }
	  else if (GET_CODE (XEXP (mem_addr, 1)) == SYMBOL_REF 
		   || GET_CODE (XEXP (mem_addr, 1)) == CONST)
	    {
	      ;
	    }
	  else
	    return;
	}
      else if (GET_CODE (XEXP (mem_addr, 0)) == PLUS)
	{
	  /* Take care of non canonicalized expressions */
	  rtx tmp;
	  if (GET_CODE (XEXP (XEXP (mem_addr, 0), 0)) == REG 
	      && GET_CODE (XEXP (XEXP (mem_addr, 0), 1)) == MULT)
	    {
	      tmp = XEXP (XEXP (mem_addr, 0), 0);
	      XEXP (XEXP (mem_addr, 0), 0) = XEXP (XEXP (mem_addr, 0), 1);
	      XEXP (XEXP (mem_addr, 0), 1) = tmp;
	    }

	  if (GET_CODE (XEXP (XEXP (mem_addr, 0), 0)) != MULT)
	    return;

	  if (GET_CODE (XEXP (XEXP (XEXP (mem_addr, 0), 0), 0)) == REG 
	      && GET_CODE (XEXP (XEXP (XEXP (mem_addr, 0), 0), 1)) == CONST_INT)
	    {
	      index_where = &XEXP (XEXP (XEXP (mem_addr, 0), 0), 0);
	      index_reg = XEXP (XEXP (XEXP (mem_addr, 0), 0), 0);
	      scale = INTVAL (XEXP (XEXP (XEXP (mem_addr, 0), 0), 1));
	    }
	  else
	    return;

	  if (GET_CODE (XEXP (XEXP (mem_addr, 0), 1)) != REG)
	    return;

	  base_where = &XEXP (XEXP (mem_addr, 0), 1);
	  base_reg = XEXP (XEXP (mem_addr, 0), 1);
	  if (GET_CODE (XEXP (mem_addr, 1)) == SYMBOL_REF 
	      || GET_CODE (XEXP (mem_addr, 1)) == CONST 
	      || GET_CODE (XEXP (mem_addr, 1)) == CONST_INT)
	    {
	      ;
	    }
	  else
	    return;
	}
      else if (GET_CODE (XEXP (mem_addr, 0)) == REG)
	{
	  base_where = &XEXP (mem_addr, 0);
	  base_reg = XEXP (mem_addr, 0);
	  if (GET_CODE (XEXP (mem_addr, 1)) == REG)
	    {
	      index_where = &XEXP (mem_addr, 1);
	      index_reg = XEXP (mem_addr, 1);
	    }
	}
      else
	return;
    }
  else if (GET_CODE (mem_addr) == SYMBOL_REF || GET_CODE (mem_addr) == CONST)
    {
      ;
    }
  else if (GET_CODE (mem_addr) == REG)
    {
      base_reg = mem_addr;
      base_where = &XEXP (SET_DEST (copy_pat), 0);
    }
  else
    return;

  /* Now we know the base index and scale */
  if (index_reg && reg_inc[REGNO (index_reg)].inc_val)
    {
      *index_where = gen_rtx (PLUS, GET_MODE (index_reg), index_reg,
	   gen_rtx (CONST_INT, 0, reg_inc[REGNO (index_reg)].inc_val * -1));
    }
  if (base_reg && reg_inc[REGNO (base_reg)].inc_val)
    {
      *base_where = gen_rtx (PLUS, GET_MODE (base_reg), base_reg,
	    gen_rtx (CONST_INT, 0, reg_inc[REGNO (base_reg)].inc_val * -1));
    }
#ifdef REWRITE_ADDRESS
  if (recog (copy_pat, insn, 0) < 0)
    try_to_make_good (SET_DEST (copy_pat));
#endif
  if (recog (copy_pat, insn, 0) < 0)
    {
      obfree (storage);
      return;
    }

  if (do_all == 0)
    where_to_move = cc0_insn;
  else
    where_to_move = jump_insn;

  /* Now move the store with it's new pattern */
  PREV_INSN (NEXT_INSN (insn)) = PREV_INSN (insn);
  NEXT_INSN (PREV_INSN (insn)) = NEXT_INSN (insn);
  NEXT_INSN (insn) = where_to_move;
  PREV_INSN (insn) = PREV_INSN (where_to_move);
  NEXT_INSN (PREV_INSN (where_to_move)) = insn;
  PREV_INSN (where_to_move) = insn;
  PATTERN (insn) = copy_pat;
  if (loop_dump_stream)
    fprintf (loop_dump_stream, " moved fp store down\n");

  if (do_all == 0)
    return;

  /* Now move the store again and add a new jump and label */
  if (loop_invalid[loop_num])
    {
      emit_insn_before (gen_rtx (USE, 0, cc0_rtx), insn);
      return;
    }
  if (GET_CODE (NEXT_INSN (loop_start)) != CODE_LABEL)
    {
      emit_insn_before (gen_rtx (USE, 0, cc0_rtx), insn);
      return;
    }
  label_insn = NEXT_INSN (loop_start);
  if (GET_CODE (SET_SRC (PATTERN (jump_insn))) != IF_THEN_ELSE)
    {
      emit_insn_before (gen_rtx (USE, 0, cc0_rtx), insn);
      return;
    }
#if 0
  if (LABEL_NUSES (label_insn) != 1)
    {
      emit_insn_before (gen_rtx (USE, 0, cc0_rtx), insn);
      return;
    }
#endif
  /* Make a jump to the old label just before the label*/
  p = emit_jump_insn_after (gen_jump (label_insn), 
                            PREV_INSN (label_insn));
  JUMP_LABEL (p) = label_insn;
  emit_barrier_after (p);
  /* Add a new label just before the old one */
  label = gen_label_rtx ();
  label_insn = emit_label_after (label, PREV_INSN (label_insn));
  /* Make the jump at the end of the loop jump to the new lable */
  if (XEXP (SET_SRC (PATTERN (jump_insn)), 1) == pc_rtx)
    {
      XEXP (SET_SRC (PATTERN (jump_insn)), 2)
        = gen_rtx (LABEL_REF, VOIDmode, label);
    }
  else
    {
      XEXP (SET_SRC (PATTERN (jump_insn)), 1)
        = gen_rtx (LABEL_REF, VOIDmode, label);
    }
  ++LABEL_NUSES (label_insn);
  JUMP_LABEL (jump_insn) = label_insn;
  /* Move the store to just after the new label */
  PREV_INSN (NEXT_INSN (insn)) = PREV_INSN (insn);
  NEXT_INSN (PREV_INSN (insn)) = NEXT_INSN (insn);
  NEXT_INSN (insn) = NEXT_INSN (label_insn);
  PREV_INSN (insn) = label_insn;
  PREV_INSN (NEXT_INSN (label_insn)) = insn;
  NEXT_INSN (label_insn) = insn;
  /* Add the store after the jump at the end of the loop */
  p = emit_insn_after (PATTERN (insn), jump_insn);
  REG_NOTES (p) = REG_NOTES (insn);
  if (loop_dump_stream)
    fprintf (loop_dump_stream, " spl successful\n");
}

#endif
#endif

/* Element N is first insn in basic block N.
   This info lasts until we finish compiling the function.  */

extern rtx *basic_block_head;

/* Element N is last insn in basic block N.
   This info lasts until we finish compiling the function.  */

extern rtx *basic_block_end;

/* Number of basic blocks in the current function.  */

extern int n_basic_blocks;

extern regset *basic_block_live_at_start;

/*        Move spills out of loops if possible i.e.
          if can find a register in the loop that is
          alway used to hold a certian spill slot then
          1) load the spill slot into that register before the
             loop.
          2) replace all references to the spill slot in the loop with
             references to that register
          3) if the spill slot was updated in the loop and the pseudo
             that corresponds to that spill slot is alive at the loop
             fallout then write that register back to the spill slot
             at the end of the loop.  NOTE that if the pseudo is
             alive at any other of the loop exits (if there are any)
             and the spill slot is updated in the loop then steps
             1) and 2) are NOT done.  
          NOTE
          a) if the spill slot is only loaded into the register once
             throughout the loop and the spill slot is not updated in
             the loop then loop_optimize() called from here will
             probably handle this.  I say probably because maybe_never
             of scan_loop is rather over cautious since it is set
             when a jump_insn is encountered but not reset when
             the corresponding label is encountered.  

          Other optimizations which are done here are trying
          to push invariants that were spilled back into the loop
          (see push_load_into_loop()).  

          Also a straight line pass over the function is made
          and whenever a spill slot is loaded into a register
          if that spill slot is also valid in some register
          then that spill slot is replaced with that register.  
          (see peep_spills()).  

*/
void
loop_after_global (f, dumpfile)
     /* f is the first instruction of a chain of insns for one function */
     rtx f;
     FILE *dumpfile;
{
  rtx insn;
  int i;
  rtx nonlocal_label_list = nonlocal_label_rtx_list ();
  register regset tem;

  {
    int sav_flag_copy_prop = flag_copy_prop;
    int sav_flag_jump_back = flag_jump_back;
    int sav_flag_compare_elim = flag_compare_elim;
    int sav_flag_unroll_loops = flag_unroll_loops;
    int sav_flag_strength_reduce = flag_strength_reduce;

    flag_copy_prop =
    flag_jump_back =
    flag_compare_elim =
    flag_unroll_loops =
    flag_strength_reduce = 0;

    cur_stack_offset = 0;
    doing_loop_after_global = 1;
    
    loop_optimize (f, dumpfile);
    
    doing_loop_after_global = 0;

    flag_copy_prop = sav_flag_copy_prop;
    flag_jump_back = sav_flag_jump_back;
    flag_compare_elim = sav_flag_compare_elim;
    flag_unroll_loops = sav_flag_unroll_loops;
    flag_strength_reduce = sav_flag_strength_reduce;
  }
    
  { /* Update basic block start and ends */
    register RTX_CODE prev_code = JUMP_INSN;
    register RTX_CODE code;
    for (insn = f, i = 0; insn; insn = NEXT_INSN (insn))
      {
	code = GET_CODE (insn);
	if (code == CODE_LABEL
	    || (GET_RTX_CLASS (code) == 'i'
		&& (prev_code == JUMP_INSN
		    || (prev_code == CALL_INSN
			&& nonlocal_label_list != 0)
		    || prev_code == BARRIER)))
	  i++;
	if (code != NOTE)
	  prev_code = code;
      }
    if (i > n_basic_blocks)
      {
        n_basic_blocks = i;
        basic_block_head = (rtx *) oballoc (n_basic_blocks * sizeof (rtx));
        basic_block_end = (rtx *) oballoc (n_basic_blocks * sizeof (rtx));
      }
    
    n_basic_blocks = i;
    prev_code = JUMP_INSN;
    
    for (insn = f, i = -1; insn; insn = NEXT_INSN (insn))
      {
	code = GET_CODE (insn);
	/* A basic block starts at label, or after something that can jump.  */
	if (code == CODE_LABEL
	    || (GET_RTX_CLASS (code) == 'i'
		&& (prev_code == JUMP_INSN
		    || (prev_code == CALL_INSN
			&& nonlocal_label_list != 0)
		    || prev_code == BARRIER)))
	  {
	    basic_block_head[++i] = insn;
	    if (i >= n_basic_blocks)
	      {
		abort ();
	      }
	    basic_block_end[i] = insn;
	  }
	else if (GET_RTX_CLASS (code) == 'i')
	  {
	    basic_block_end[i] = insn;
	  }

	if (code != NOTE
	    && !(prev_code == CALL_INSN && code == INSN
		 && GET_CODE (PATTERN (insn)) == CLOBBER))
	  prev_code = code;
      }
  }
}

/* x is a spill slot and hence is unaliased */

static int
spill_invariant_p (x)
     register rtx x;
{
  register int i;
  register enum rtx_code code;
  register char *fmt;
  int conditional = 0;

  if (x == 0)
    return 1;
  code = GET_CODE (x);
  switch (code)
    {
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CONST:
      return 1;

    case LABEL_REF:
      /* A LABEL_REF is normally invariant, however, if we are unrolling
	 loops, and this label is inside the loop, then it isn't invariant.
	 This is because each unrolled copy of the loop body will have
	 a copy of this label.  If this was invariant, then an insn loading
	 the address of this label into a register might get moved outside
	 the loop, and then each loop body would end up using the same label.

	 We don't know the loop bounds here though, so just fail for all
	 labels.  */
      if (flag_unroll_loops)
	return 0;
      else
	return 1;

    case PC:
    case CC0:
    case UNSPEC_VOLATILE:
      return 0;

    case REG:
      /* We used to check RTX_UNCHANGING_P (x) here, but that is invalid
	 since the reg might be set by initialization within the loop.  */
      if (x == frame_pointer_rtx || x == arg_pointer_rtx)
	return 1;
      if (loop_has_call
	  && REGNO (x) < FIRST_PSEUDO_REGISTER && call_used_regs[REGNO (x)])
	return 0;
      if (n_times_set[REGNO (x)] < 0)
	return 2;
      return n_times_set[REGNO (x)] == 0;

    case MEM:
      /* Read-only items (such as constants in a constant pool) are
	 invariant if their address is.  */
      if (RTX_UNCHANGING_P (x))
	break;

      /* If we filled the table (or had a subroutine call), any location
	 in memory could have been clobbered.  */
      if (unknown_address_altered
      /* Don't mess with volatile memory references.  */
	  || MEM_VOLATILE_P (x))
	return 0;

      /* See if there is any dependence between a store and this load.  */
      for (i = loop_store_mems_idx - 1; i >= 0; i--)
	if (rtx_equal_p (XEXP (loop_store_mems[i], 0), XEXP (x, 0)))
	  {
	    return 0;
	  }

      /* It's not invalidated by a store in memory
	 but we must still verify the address is invariant.  */
      break;

    case ASM_OPERANDS:
      /* Don't mess with insns declared volatile.  */
      if (MEM_VOLATILE_P (x))
	return 0;
    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'e')
	{
	  int tem = invariant_p (XEXP (x, i));
	  if (tem == 0)
	    return 0;
	  if (tem == 2)
	    conditional = 1;
	}
      else if (fmt[i] == 'E')
	{
	  register int j;
	  for (j = 0; j < XVECLEN (x, i); j++)
	    {
	      int tem = invariant_p (XVECEXP (x, i, j));
	      if (tem == 0)
		return 0;
	      if (tem == 2)
		conditional = 1;
	    }

	}
    }

  return 1 + conditional;
}

extern struct reload_slot_info reload_slots[];
extern num_reload_stack_slots;
extern char call_used_regs[];
extern char **insn_name_ptr;

static struct spill_candidates
  {
    /* The stack slot reloaded into this candidate */
    rtx reload_slot;
    /* The insn that does the reloading */
    rtx reload_insn;
    /* The rtx of the hard register that is reloaded into */
    rtx hard_reg;
    /* The mode of the reload slot */
    enum machine_mode mode;
    /* Number of times this candidate is set */
    int num_times_set;
    /* The pseduo register corresponding to the stack slot */
    int pseudo_reg;
    int cur_stack_offset;
    int reload_indx;
    /* A linked list of candidates on the same reload stack slot */
    struct spill_candidates *link_cand;
    struct spill_candidates *head_cand;
    unsigned is_head:1;
    unsigned has_link:1;
    /* if 1 this candidate is invalid */
    unsigned disregard:1;
    /* if 1 the reload stack slot is stored into */
    unsigned slot_updated_in_loop:1;
    /* if 1 the pseduo register is live at the loop fall out */
    unsigned live_at_fall_out:1;
    /* if 1 the pseduo register is live at a loop jump out */
    unsigned live_at_jump_out:1;
    /* if 1 the value in the candidate hard register must be
       updated into the reload stack slot */
    unsigned must_update_reload:1;
    /* if 1 then the candidate hard register was loaded with the
       reload stack slot before it was used */
    unsigned first_set_to_reload:1;
    /* if 1 then the candidate's spill slot has been updated and
       the candidate is out of synch with it's spill slot intel1a */
    unsigned slot_updated:1;
  }

spill_candidate[FIRST_PSEUDO_REGISTER];

static int all_candidates_invalid = 0;
static int num_spill_candidates_found = 0;
static int *insns_moved_out;
static int have_flow_analysis = 0;

/* Find all candidates for spill slot optimization over the loop and
   move them out. Also try to push spilled invariants back into the
   loop
*/
static
void 
move_spills_out (loop_start, loop_end)
     rtx loop_start;
     rtx loop_end;
{
  rtx insn;
  int i, is_loop_exit, regno, indx;

  insns_moved_out = (int *) alloca ((max_uid_for_loop + 1) * sizeof (int));
  bzero (insns_moved_out, (max_uid_for_loop + 1) * sizeof (int));
  num_reload_stack_slots = 0;
  num_spill_candidates_found = 0;
  bzero (spill_candidate, FIRST_PSEUDO_REGISTER * sizeof (spill_candidate[0]));
  for (i=0; i<FIRST_PSEUDO_REGISTER; i++)
    spill_candidate[i].reload_indx = -1;

  insn = loop_start;
  while (insn != PREV_INSN (loop_end))
    {
      if (GET_CODE (insn) == INSN && RTX_IS_SPILL_P (insn) 
	  && GET_CODE (PATTERN (insn)) == SET 
	  && GET_CODE (SET_DEST (PATTERN (insn))) == REG 
	  && GET_CODE (SET_SRC (PATTERN (insn))) == MEM 
	  && is_a_stack_slot (SET_SRC (PATTERN (insn)))
	  && (regno = REGNO (SET_DEST (PATTERN (insn)))) < FIRST_PSEUDO_REGISTER)
	{
	  indx = -1;
	  if (!is_a_reload_stack_slot (SET_SRC (PATTERN (insn)), &indx, 0, 0))
	    {
	      if (num_reload_stack_slots >= MAX_RELOAD_STACK_SLOTS)
	        return;

	      reload_slots[num_reload_stack_slots].cur_stack_offset = 0;
	      reload_slots[num_reload_stack_slots].reload_stack_mode = -1;
	      reload_slots[num_reload_stack_slots].is_loaded_to_reg = -1;
	      reload_slots[num_reload_stack_slots].reload_slot_updated = 0;
	      reload_slots[num_reload_stack_slots].stack_slot_pseudo_reg = SPILL_PSEUDO (insn);
	      reload_slots[num_reload_stack_slots].reload_insn = insn; 
	      reload_slots[num_reload_stack_slots].reload_stack_slot = SET_SRC (PATTERN (insn));
	      num_reload_stack_slots++;
	    }
	  else
	    { /* We have already encountered this reload slot - it
                 must always be a reload slot of the same pseudo */
	      if (reload_slots[indx].stack_slot_pseudo_reg !=
		  SPILL_PSEUDO (insn))
		reload_slots[indx].reload_stack_mode == -2;
	    }

	  if (indx == -1) 
	    indx = num_reload_stack_slots - 1;

	  if (!(spill_candidate[regno].pseudo_reg == 0 
		|| spill_candidate[regno].pseudo_reg == SPILL_PSEUDO (insn)))
	    { /* Candidate can be for only one spilled pseudo reg */
              reload_slots[indx].reload_stack_mode = -2;
              if (spill_candidate[regno].reload_indx != -1)
                reload_slots[spill_candidate[regno].reload_indx].reload_stack_mode = -2;
	    }
          else if (spill_candidate[regno].reload_indx != -1
                   && spill_candidate[regno].reload_indx != indx)
            { /* Candidates can only be for one reload slot */
              reload_slots[indx].reload_stack_mode = -2;
              reload_slots[spill_candidate[regno].reload_indx].reload_stack_mode = -2;
            }
          else 
	    {
              spill_candidate[regno].pseudo_reg = SPILL_PSEUDO (insn);
              spill_candidate[regno].reload_indx = indx;
	    }
	}
      insn = NEXT_INSN (insn);
    }

  all_candidates_invalid = 0;
  insn = loop_start;
  while (insn != (loop_end))
    {
      if (GET_CODE (insn) == INSN)
	{
	  mark_spill_candidates (insn, PATTERN (insn));
	  if (all_candidates_invalid)
	    return;
	}
      else if (GET_CODE (insn) == JUMP_INSN)
	{ /* Check if this is a loop exit */
	  is_loop_exit = check_if_jumps_out_of_loop (PATTERN (insn), 1);
	  /* Verify which spill candidates remian valid */
	  if (num_spill_candidates_found)
	    verify_spill_candidates (insn, is_loop_exit);
	  if (all_candidates_invalid)
	    return;
	}
      else if (GET_CODE (insn) == CODE_LABEL && num_spill_candidates_found)
	{
	  /* Verify which spill candidates remian valid */
	  verify_spill_candidates (insn, 0);
	}
      insn = NEXT_INSN (insn);
    }
  if (num_spill_candidates_found && !all_candidates_invalid)
    {
      int i;
      rtx rn, prev, link;
      if (!simplejump_p (PREV_INSN (loop_end)))
	mark_regs_alive_at_exit (NEXT_INSN (loop_end), 1);
      if (!all_candidates_invalid)
	{
	  move_spill_movables (loop_start, loop_end);

	  for (rn = loop_start; rn != loop_end; rn = NEXT_INSN (rn))
	    {
	      if ((GET_CODE (rn) == INSN || GET_CODE (rn) == JUMP_INSN
		   || GET_CODE (rn) == CALL_INSN) && LOG_LINKS (rn))
		{
		  for (prev = 0, link = LOG_LINKS (rn); link;
		       prev = link, link = XEXP (link, 1))
		    {
		      if (insns_moved_out[INSN_UID (XEXP (link, 0))])
			{
			  if (prev)
			    XEXP (prev, 1) = XEXP (link, 1);
			  else
			    LOG_LINKS (rn) = XEXP (link, 1);
			}
		    }

		}
	    }
	}
    }
  if (unknown_address_altered)
    return;

  for (i = 0; i < num_reload_stack_slots; i++)
    {
      if (flag_push_load_into_loop
	  && reload_slots[i].reload_slot_updated == 0 
	  /*&& reload_slots[i].reload_stack_mode != -2 
	  && reload_slots[i].is_loaded_to_reg != -1 
	  && !spill_candidate[reload_slots[i].is_loaded_to_reg].has_link 
	  && spill_candidate[reload_slots[i].is_loaded_to_reg].disregard 
	  && spill_candidate[reload_slots[i].is_loaded_to_reg].live_at_jump_out == 0 
          && spill_candidate[reload_slots[i].is_loaded_to_reg].live_at_fall_out == 0
          */
	)
	{
	  /* This is a reload slot that was never updated in loop
             but could not be lifted only because the register it is
             loaded into was also set to something else in loop.
             Lets see if the reload slot was set in the basic block
             just prior to the loop by some inexpensive operation.
             (particularly a move from memory into a register and
             a move of that register onto stack). If so lets
             replace each load of the reload slot in the loop with
             a load of the memory and delete the code that loaded
             and spilled it outside the loop.
             This corrects the preloading of loop invariants that
             could not be held in registers over the loop
           */
	  push_load_into_loop (loop_start, loop_end, i);
	}
    }
}

/* Move the valid spill slot candidates out of the loop and
   replace the spill slot with the register throughout the loop.
   Also update the spill slot after the loop if this is necessary */

static
void 
move_spill_movables (loop_start, loop_end)
     rtx loop_start, loop_end;
{
  rtx insn;
  int i, indx, all_are_ok;
  struct spill_candidates *link_cand;
  for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
    {
      if (spill_candidate[i].disregard == 0 && spill_candidate[i].reload_insn 
	  && spill_candidate[i].num_times_set == n_times_set[i])
	{
	  is_a_reload_stack_slot (spill_candidate[i].reload_slot, &indx, 0, 0);
          /* Sanity check */
          if (spill_candidate[i].reload_indx != indx)
            abort ();

	  if (reload_slots[indx].reload_stack_mode == -2)
	    {/* The stack slot was used in some other mode than the
                 load to the hard reg */
              spill_candidate[i].disregard = 1;
	      continue;
	    }
	  if (!spill_candidate[i].has_link)
	    {
	      all_are_ok = 1;
	    }
	  else if (spill_candidate[i].is_head)
	    {
	      /* This is a candidate that is the head of a chain of
                 candidates that all contain the same reload slot.
                 Go over the linked list to see if all are ok;
              */
	      all_are_ok = 1;
	      link_cand = spill_candidate[i].link_cand;
	      while (link_cand)
		{
		  all_are_ok = all_are_ok 
		    && link_cand->disregard == 0 
		    && link_cand->num_times_set == 1 
		    && n_times_set[REGNO (link_cand->hard_reg)] == 1 
		    && link_cand->reload_insn;
		  /* So it won't be processed again */
		  link_cand->disregard = 1;
		  link_cand = link_cand->link_cand;
		}

	      if (spill_candidate[i].num_times_set != 1)
		all_are_ok = 0;

	      if (all_are_ok == 0)
		spill_candidate[i].disregard = 1;
	    }
	  else
	    {
	      all_are_ok = 0;
	    }

	  if (all_are_ok)
	    {
	      /* Move the reload insn to just before loop_start */
	      if (loop_dump_stream)
	        {
	          fprintf (loop_dump_stream,
	          "insn %d moved out of loop, stack slot held in reg %d over loop\n",
	          INSN_UID (spill_candidate[i].reload_insn), i);
	          if (spill_candidate[i].is_head)
	            {
		      fprintf (loop_dump_stream, " these regs also shadow the stack slot:");
	              link_cand = spill_candidate[i].link_cand;
	              while (link_cand)
		        {
		          fprintf (loop_dump_stream, " %d", REGNO (link_cand->hard_reg));
		          link_cand = link_cand->link_cand;
		        }
		      fprintf (loop_dump_stream, "\n");
	            }
	        }
	      insn = spill_candidate[i].reload_insn;
	      NEXT_INSN (PREV_INSN (insn)) = NEXT_INSN (insn);
	      PREV_INSN (NEXT_INSN (insn)) = PREV_INSN (insn);
	      NEXT_INSN (insn) = loop_start;
	      PREV_INSN (insn) = PREV_INSN (loop_start);
	      NEXT_INSN (PREV_INSN (loop_start)) = insn;
	      PREV_INSN (loop_start) = insn;
	      if (INSN_UID (insn) <= max_uid_for_loop)
		insns_moved_out[INSN_UID (insn)] = 1;

	      if (spill_candidate[i].live_at_fall_out 
		  && spill_candidate[i].slot_updated_in_loop)
		{
		  emit_insn_after (gen_rtx (SET, 0, spill_candidate[i].reload_slot,
					    spill_candidate[i].hard_reg),
				   loop_end);
		}
	    }
	}
      else
	{
	  spill_candidate[i].disregard = 1;
	  if (spill_candidate[i].has_link && spill_candidate[i].is_head)
	    {  /* invalidate the whole chain */
	      link_cand = spill_candidate[i].link_cand;
	      while (link_cand)
		{
		  link_cand->disregard = 1;
		  link_cand = link_cand->link_cand;
		}
	    }
	}
    }
  /* Where ever a reload_slot that is contained is a valid candidate
     appears in the loop - replace it with that hard_reg */
  insn = loop_start;
  while (insn != loop_end)
    {
      if (GET_CODE (insn) == INSN)
	{
          undobuf.num_undo = 0;
          undobuf.storage = (char *) oballoc (0);
	  replace_spill_slots (insn, PATTERN (insn), &PATTERN (insn));
	  replace_linked_regs (insn, PATTERN (insn));
	}
      insn = NEXT_INSN (insn);
    }

}


/* If pat jumps out of the loop then mark all candidates that
   correspond to pseudos that are alive at the target of the jump,
   and return 1
*/
static
int 
check_if_jumps_out_of_loop (pat, do_mark)
     rtx pat;
     int do_mark;
{
  rtx target;
  int ret_val = 0;
  int i;

  switch (GET_CODE (pat))
    {
    case PC:
    case USE:
    case CLOBBER:
    case REG:
    case MEM:
    case CONST_INT:
    case CONST_DOUBLE:
    case RETURN:
      return ret_val;

    case CONST:
      /* There could be a label reference in here.  */
      return (check_if_jumps_out_of_loop (XEXP (pat, 0), do_mark));

    case PLUS:
    case MINUS:
    case MULT:
    case LSHIFT:
      ret_val = check_if_jumps_out_of_loop (XEXP (pat, 0), do_mark);
      ret_val = ret_val | check_if_jumps_out_of_loop (XEXP (pat, 1), do_mark);
      return ret_val;

    case SIGN_EXTEND:
    case ZERO_EXTEND:
      return (check_if_jumps_out_of_loop (XEXP (pat, 0), do_mark));

    case LABEL_REF:
      if (LABEL_OUTSIDE_LOOP_P (pat))
	{ /* This is a jump to outside of loop */
	  if (do_mark)
	    {
	      target = XEXP (pat, 0);
	      /* Now mark all those pseudo regs not dead at entry
	         to the target block */
	      mark_regs_alive_at_exit (target, 0);
	    }
	  ret_val = 1;
	}

      return ret_val;

    case SET:
      /* If this is not setting pc, ignore.  */
      if (SET_DEST (pat) == pc_rtx)
	ret_val = check_if_jumps_out_of_loop (SET_SRC (pat), do_mark);
      return ret_val;

    case IF_THEN_ELSE:
      ret_val = check_if_jumps_out_of_loop (XEXP (pat, 1), do_mark);
      ret_val = ret_val | check_if_jumps_out_of_loop (XEXP (pat, 2), do_mark);
      return ret_val;

    case PARALLEL:
    case ADDR_VEC:
    case ADDR_DIFF_VEC:
      all_candidates_invalid = 1;
      return 1;

    default:
      /* Nothing else should occur in a JUMP_INSN.  */
      abort ();
    }
}

/*
  block_head, or the first nonnote insn after block_head (if block_head
  is a note insn), is the head of a block that is the target of a loop
  exit. if is_loop_fallout is 1 then the loop exit is the normal end of
  the loop where the loop falls out, otherwise the loop exit was a jump
  out of the loop. Any candidates whose pseudos are alive at the
  start of the block are marked (i.e. the stack slot which
  corresponds to the pseudo must be updated at this loop exit)
  if the loop exit is NOT the loop fallout then we mark this
  candidate as not valid
*/

static
void 
mark_regs_alive_at_exit (block_head, is_loop_fallout)
     rtx block_head;
     int is_loop_fallout;
{
  int i, found, pseudo_regno, block_num, offset, bit;
  rtx org_block_head = block_head;

  i = 0;
  found = 0;

  /* Find which block block_head is the head of */
  if (is_loop_fallout)
    while (i < n_basic_blocks && !found)
      if (basic_block_head[i] == block_head)
	found = 1;
      else
	i++;

  if (!found)
    {
      while (block_head && GET_CODE (block_head) != CODE_LABEL 
	     && GET_RTX_CLASS (GET_CODE (block_head)) != 'i')
        block_head = NEXT_INSN (block_head);

      if (block_head == NULL)
        {
          if (!is_loop_fallout)
	    { /* block_head must have been a label upon entry and hence
                 should not be null now */
	      abort ();
	    }
          return;
        }
      i = 0;
      while (i < n_basic_blocks && !found)
        if (basic_block_head[i] == block_head)
	  found = 1;
        else
	  i++;
    }
  if (!found)
    {
      all_candidates_invalid = 1;
      return;
    }
  block_num = i;
  for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
    {
      if (spill_candidate[i].reload_slot == NULL)
        { /* A jump out of the loop before it is known
             if this slot will become a candidate - so
             just mark it to never become a candidate.
             Actually we could remember this jump and
             mark the candidate accordingly in the future
             but this seems like too much work */
          spill_candidate[i].disregard = 1;
        }
      else
      if (spill_candidate[i].disregard == 0 
	  && spill_candidate[i].reload_slot)
	{ /* This candidate is still valid */
	  /* Find the pseudo reg that corresponds to this spill slot */
	  pseudo_regno = spill_candidate[i].pseudo_reg;
	  if (pseudo_regno < FIRST_PSEUDO_REGISTER)
	    abort ();

	  /* See if this pseudo reg is alive at the beginning of
             the block that begins at block_head*/
	  offset = pseudo_regno / REGSET_ELT_BITS;
	  bit = 1 << (pseudo_regno % REGSET_ELT_BITS);
	  if (basic_block_live_at_start[block_num][offset] & bit)
	    {
	      if (!is_loop_fallout)
		{ /* Don't want to handle pseudo's that are alive
                     at targets of jumps out of the loop */
		  spill_candidate[i].disregard = 1;
		  spill_candidate[i].live_at_jump_out = 1;
		}
	      else
		{
		  spill_candidate[i].live_at_fall_out = 1;
		}
	    }
	}
    }

}


/* 
  Verfiy that all spill candidates are still valid
*/
static
void 
verify_spill_candidates (insn, is_loop_exit)
     rtx insn;
     int is_loop_exit;
{
  int i;
  for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
    {
      /* intel1a */
      spill_candidate[i].slot_updated = 0;
      if (spill_candidate[i].must_update_reload)
	spill_candidate[i].disregard = 1;

      if (GET_CODE (insn) == CODE_LABEL)
	spill_candidate[i].first_set_to_reload = 0;
    }
}

#ifndef max
#define max(a,b) ((a) < (b) ? (b) : (a))
#endif

/* Update the spill slot candidate info according to what body does */
static
void 
mark_spill_candidates (insn, body)
     rtx insn;
     rtx body;
{
  int regno, i, dummy, indx, cand_num, j, n_regs;

  if (GET_CODE (body) == SET)
    {
      if (GET_CODE (SET_DEST (body)) != REG
          && GET_CODE (SET_DEST (body)) != SUBREG)
	spill_handle_regs_mentioned (SET_DEST (body));

      spill_handle_regs_mentioned (SET_SRC (body));
      if (GET_CODE (SET_DEST (body)) == SUBREG
           && GET_CODE (SUBREG_REG (SET_DEST (body))) == REG 
	   && REGNO (SUBREG_REG (SET_DEST (body))) < FIRST_PSEUDO_REGISTER)

	{
	  regno = REGNO (SUBREG_REG (SET_DEST (body)));
	  n_regs 
	    = max (HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body))),
	           HARD_REGNO_NREGS (regno, 
	                             GET_MODE (SUBREG_REG (SET_DEST (body)))));
	  for (j = 0; j < n_regs; j++)
	    {
	      spill_candidate[regno + j].disregard = 1;
	      if (spill_candidate[regno + j].reload_indx != -1)
	        {
	          reload_slots[spill_candidate[regno + j].
	              reload_indx].reload_stack_mode = -2;
	        }
	    }
	}
      else if (GET_CODE (SET_DEST (body)) == REG 
	       && REGNO (SET_DEST (body)) < FIRST_PSEUDO_REGISTER)
	{
	  /* if body sets a hard register it is of interest */
	  regno = REGNO (SET_DEST (body));
          n_regs = HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body)));
          if (n_regs > 1)
            for (j = 0; j < n_regs; j++)
	      spill_candidate[regno + j].disregard = 1;

	  spill_candidate[regno].num_times_set++;
	  if (GET_CODE (SET_SRC (body)) == MEM 
	      && is_a_reload_stack_slot (SET_SRC (body), &indx, 0, 0))
	    { /* It sets it to the value in a reload stack slot 
	         The stack slot must always be used in the same mode
                 as it is used to reload the hard register
              */
              /* intel1a */
	      spill_candidate[regno].slot_updated = 0;
	      if (reload_slots[indx].reload_stack_mode == -1)
		{
		  reload_slots[indx].reload_stack_mode 
		     = GET_MODE (SET_SRC (body));
		}
	      else if (reload_slots[indx].reload_stack_mode == -2 
		       || reload_slots[indx].reload_stack_mode 
		             != GET_MODE (SET_SRC (body)))
		{
		  /* reload slot not always used in same mode */
		  reload_slots[indx].reload_stack_mode = -2;
		  spill_candidate[regno].disregard = 1;
		}
	      if (!spill_candidate[regno].disregard)
		{
		  if (spill_candidate[regno].reload_indx != -1
		      && spill_candidate[regno].reload_indx != indx)
		    { /* Same reg loads more than one reload slot -
                         disregard it */
		      reload_slots[indx].reload_stack_mode = -2;
		      reload_slots[spill_candidate[regno].reload_indx].reload_stack_mode = -2;
		      spill_candidate[regno].disregard = 1;
		    }
		  else
		    {
                      num_spill_candidates_found++;
                      spill_candidate[regno].first_set_to_reload = 1;
                      spill_candidate[regno].reload_slot = SET_SRC (body);
                      spill_candidate[regno].hard_reg = SET_DEST (body);
                      spill_candidate[regno].reload_insn = insn;
                      spill_candidate[regno].mode = GET_MODE (SET_DEST (body));

                      /* Sanity check */
                      if (spill_candidate[regno].reload_indx != indx
                          && spill_candidate[regno].reload_indx != -1)
                        abort ();

                      spill_candidate[regno].reload_indx = indx;
                      if (spill_candidate[regno].slot_updated_in_loop == 0)
                        {
                          spill_candidate[regno].slot_updated_in_loop 
                    	    = reload_slots[indx].reload_slot_updated;
                        }

                      if (spill_candidate[regno].pseudo_reg == 0)
                        spill_candidate[regno].disregard = 1;
		    }
		}
	      if (reload_slots[indx].is_loaded_to_reg == -1)
	        {
	          if (reload_slots[indx].reload_stack_mode != -2
	              && spill_candidate[regno].disregard == 0)
		    reload_slots[indx].is_loaded_to_reg = regno;
		}
	      else if (regno != reload_slots[indx].is_loaded_to_reg)
		{
		  int loaded_reg;
		  loaded_reg = reload_slots[indx].is_loaded_to_reg;
		  /* Same spill slot already loaded to different hard reg */
		  /* Create a linked list of candidates that load the
                     same reload slot */
		  if (spill_candidate[reload_slots[indx].is_loaded_to_reg].disregard == 0
		      && GET_MODE (reload_slots[indx].reload_stack_slot) 
		        == GET_MODE (SET_SRC (body)))
		    {
		      if (spill_candidate[regno].has_link == 0)
			{
			  spill_candidate[regno].has_link = 1;
			  spill_candidate[loaded_reg].has_link = 1;
			  spill_candidate[loaded_reg].is_head = 1;
			  spill_candidate[regno].link_cand = spill_candidate[loaded_reg].link_cand;
			  spill_candidate[loaded_reg].link_cand = &spill_candidate[regno];
			  spill_candidate[regno].head_cand = &spill_candidate[loaded_reg];
			}
		    }
		  else
		    {
		      spill_candidate[loaded_reg].disregard = 1;
		      spill_candidate[regno].disregard = 1;
		      reload_slots[indx].reload_stack_mode = -2;
		    }
		}
	    }
	  else
	    {
	      /* reg regno is set to some other value than
                 a spill stack slot */
	      if (!is_updated_by_const (insn))
		{
		  spill_candidate[regno].disregard = 1;
		  if (spill_candidate[regno].reload_indx != -1)
		    reload_slots[spill_candidate[regno].reload_indx].reload_stack_mode = -2;
		}
	      /* It sets it to some other value - signal that by
                 the end of this block that value must have been
                 stored into the hard registers reload stack slot
              */
	      spill_candidate[regno].must_update_reload = 1;
	    }
	}
      else if (is_a_reload_stack_slot (SET_DEST (body), &indx, 0, 0))
	{ /* This is an update of a reload_stack_slot */
	  if (GET_CODE (SET_DEST (body)) == SUBREG)
	    reload_slots[indx].reload_stack_mode = -2;

	  reload_slots[indx].reload_slot_updated = 1;
	  cand_num = reload_slots[indx].is_loaded_to_reg;
	  /* intel1a */
	  if (cand_num != -1)
	    spill_candidate[cand_num].slot_updated = 1;

	  if (cand_num == -1)
	    {
	      if (!is_updated_by_const_or_reg (insn))
		{
		  /* An update before a load - must not
                     become a candidate */
		  reload_slots[indx].reload_stack_mode = -2;
		}
	    }
	  else if (GET_CODE (SET_SRC (body)) == REG 
		   && REGNO (SET_SRC (body)) < FIRST_PSEUDO_REGISTER 
		   && GET_MODE (SET_SRC (body)) == spill_candidate[REGNO (SET_SRC (body))].mode 
		   && rtx_equal_p (XEXP (spill_candidate[REGNO (SET_SRC (body))].reload_slot, 0),
				   XEXP (SET_DEST (body), 0)))
	    {
	      /* It is an update from the appropriate hard reg
                 and was expected */
	      spill_candidate[cand_num].must_update_reload = 0;
	      spill_candidate[cand_num].slot_updated_in_loop = 1;
	    }
	  else if (spill_candidate[cand_num].must_update_reload != 0)
	    { /* It was expected to be updated by the hard
                 register but it wasn't - the hard register
                 that has this as a reload stack slot is no
                 longer to be considered a candidate */
	      spill_candidate[cand_num].disregard = 1;
	      if (spill_candidate[cand_num].reload_indx != -1)
		reload_slots[spill_candidate[cand_num].reload_indx].reload_stack_mode = -2;
	    }
	  else if (!is_updated_by_const_or_reg (insn))
	    {
	      spill_candidate[cand_num].disregard = 1;
	      if (spill_candidate[cand_num].reload_indx != -1)
		reload_slots[spill_candidate[cand_num].reload_indx].reload_stack_mode = -2;
	    }
	  else
	    {
	      spill_candidate[cand_num].slot_updated_in_loop = 1;
	    }
	}
    }
  else if (GET_CODE (body) == CLOBBER)
    {
      if (GET_CODE (SET_DEST (body)) == REG 
	  && REGNO (SET_DEST (body)) < FIRST_PSEUDO_REGISTER)
	{
	  regno = REGNO (SET_DEST (body));
	  n_regs = HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body)));
	  for (j = 0; j < n_regs; j++)
	    {
	      spill_candidate[regno + j].disregard = 1;
	      if (spill_candidate[regno + j].reload_indx != -1)
	        reload_slots[spill_candidate[regno + j].reload_indx].reload_stack_mode = -2;
	    }
	}
      else if (GET_CODE (SET_DEST (body)) == SUBREG
               && GET_CODE (SUBREG_REG (SET_DEST (body))) == REG 
	       && REGNO (SUBREG_REG (SET_DEST (body))) < FIRST_PSEUDO_REGISTER)

	{
	  regno = REGNO (SUBREG_REG (SET_DEST (body)));
	  n_regs = max (HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body))),
	                HARD_REGNO_NREGS (regno, 
	                                  GET_MODE (SUBREG_REG (SET_DEST (body)))));
	  for (j = 0; j < n_regs; j++)
	    {
	      spill_candidate[regno + j].disregard = 1;
	      if (spill_candidate[regno + j].reload_indx != -1)
	        reload_slots[spill_candidate[regno + j].reload_indx].reload_stack_mode = -2;
	    }
	}
    }
  else if (GET_CODE (body) == PARALLEL)
    {
      for (i = 0; i < XVECLEN (body, 0); i++)
	mark_spill_candidates (insn, XVECEXP (body, 0, i));
    }
  else
    {
      all_candidates_invalid = 1;
    }
}


/*
  Called to mark candidates that are invalid at this point */
static
void 
spill_handle_regs_mentioned (pat)
     rtx pat;
{
  register char *fmt;
  register int i;
  register enum rtx_code code;
  int indx;

  code = GET_CODE (pat);

  switch (code)
    {
    case REG:
      if (REGNO (pat) < FIRST_PSEUDO_REGISTER)
	{
	  /* intel1a */
	  if (spill_candidate[REGNO (pat)].slot_updated)
	    {
	      spill_candidate[REGNO (pat)].disregard = 1;
	      for (i=0; i < num_reload_stack_slots; i++)
	        if (reload_slots[i].is_loaded_to_reg == REGNO (pat))
	          reload_slots[i].reload_stack_mode = -2;
	    }
	  if (!spill_candidate[REGNO (pat)].first_set_to_reload)
	    {
	      spill_candidate[REGNO (pat)].disregard = 1;
	      for (i=0; i < num_reload_stack_slots; i++)
	        if (reload_slots[i].is_loaded_to_reg == REGNO (pat))
	          reload_slots[i].reload_stack_mode = -2;
	    }
	}
      return;

      /* These codes have no constituent expressions
	 and are unique.  */
    case SCRATCH:
    case CC0:
    case PC:
    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
      return;

    }

  /* The reload stack slot must always be accessed in the same mode */
  if ((GET_CODE (pat) == MEM || GET_CODE (pat) == SUBREG)
      && is_a_reload_stack_slot (pat, &indx, 0, 0))
    {
      if (GET_CODE (pat) == SUBREG)
        {
	  /* Signal invalid */
	  reload_slots[indx].reload_stack_mode = -2;
        }
      else if (reload_slots[indx].reload_stack_mode == -1)
	{
	  reload_slots[indx].reload_stack_mode = GET_MODE (pat);
	}
      else if (reload_slots[indx].reload_stack_mode != GET_MODE (pat))
	{
	  /* Signal invalid */
	  reload_slots[indx].reload_stack_mode = -2;
	}
    }
  fmt = GET_RTX_FORMAT (code);

  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'E')
	{
	  register int j;
	  for (j = XVECLEN (pat, i) - 1; j >= 0; j--)
	    spill_handle_regs_mentioned (XVECEXP (pat, i, j));
	}
      else if (fmt[i] == 'e')
	spill_handle_regs_mentioned (XEXP (pat, i));
    }
}


/*
  Return 1 if insn updates a value by a constant value */
static
int 
is_updated_by_const (insn)
     rtx insn;
{
  if (GET_CODE (PATTERN (insn)) != SET)
    return 0;

  if (GET_RTX_LENGTH (GET_CODE (SET_SRC (PATTERN (insn)))) != 2)
    return 0;

  if (!rtx_equal_p (XEXP (SET_SRC (PATTERN (insn)), 0),
		    SET_DEST (PATTERN (insn))))
    return 0;

  if (GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) != CONST_INT)
    return 0;

  return 1;
}


/*
  Return 1 if insn updates a value by a constant value or register */
static
int 
is_updated_by_const_or_reg (insn)
     rtx insn;
{
  if (GET_CODE (PATTERN (insn)) != SET)
    return 0;

  if (GET_RTX_LENGTH (GET_CODE (SET_SRC (PATTERN (insn)))) != 2)
    return 0;

  if (!rtx_equal_p (XEXP (SET_SRC (PATTERN (insn)), 0),
		    SET_DEST (PATTERN (insn))))
    return 0;

  if (GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) != CONST_INT
      && GET_CODE (XEXP (SET_SRC (PATTERN (insn)), 1)) != REG)
    return 0;

  return 1;
}

static int
peep_spills_mark_reload_slots(insn, x)
     rtx insn;
     rtx x;
{
  int indx;
  
  if (GET_CODE (x) == SET)
    {
      return peep_spills_mark_reload_slots(insn, SET_SRC (x))
          && peep_spills_mark_reload_slots(insn, SET_DEST(x));
    }
  else if (GET_CODE (x) == PARALLEL)
    {
      int i;
      
      for (i = 0; i < XVECLEN (x, 0); i++)
        if(!peep_spills_mark_reload_slots(insn, XVECEXP (x, 0, i)))
          return 0;
    }
  else if (GET_CODE (x) == MEM 
           && is_a_stack_slot (x)
           && !is_a_reload_stack_slot (x, &indx, cur_stack_offset, 0) )
    {
      if (num_reload_stack_slots >= MAX_RELOAD_STACK_SLOTS)
          return 0;

      if (loop_dump_stream)
        {
          fprintf (loop_dump_stream, "reload slot at insn %d offset %d %d %d\n",
          INSN_UID (insn), cur_stack_offset,
          RTX_IS_SPILL_P (x),
          RTX_IS_SPILL_P (XEXP (x, 0)));
        }

      reload_slots[num_reload_stack_slots].is_loaded_to_reg = -1;
      reload_slots[num_reload_stack_slots].last_store = NULL;
      reload_slots[num_reload_stack_slots].cur_stack_offset = cur_stack_offset;
      reload_slots[num_reload_stack_slots].reload_stack_slot = x;
      reload_slots[num_reload_stack_slots].reload_insn = insn; 
      num_reload_stack_slots++;
    }

  return 1;
}

/* Go over the whole function and do

   spill_slotA <- regY                     spill_slotA <- regY
        .  sequence of code not                 .  sequence of code not
        .  affecting regY or       =>           .  affecting regY or
        .  spill_slotA                          .  spill_slotA
   regX <- spill_slotA                     regX <- regY
*/
void 
peep_spills (f, dumpfile)
     rtx f;
     FILE * dumpfile;
{
  rtx insn, next, note;
  int i, insn_code_number, success;

  undobuf.num_undo = 0;
  num_reload_stack_slots = 0;
  cur_stack_offset = 0;
  bzero (spill_candidate, FIRST_PSEUDO_REGISTER * sizeof (spill_candidate[0]));
  loop_dump_stream = dumpfile;
  if (loop_dump_stream)
    fprintf (loop_dump_stream, "\n\nDoing peep spills\n\n");
  
  insn = f;
  while (insn)
    {
      if (GET_RTX_CLASS (GET_CODE (insn)) == 'i')
        {
          if (!peep_spills_mark_reload_slots(insn, PATTERN (insn)))
            return;
          
          if (GET_CODE (PATTERN (insn)) == SET)
	    update_stack_offset (PATTERN (insn), &cur_stack_offset);
          else if (GET_CODE (PATTERN (insn)) == PARALLEL)
            for (i = 0; i < XVECLEN (PATTERN (insn), 0); i++)
	      update_stack_offset (XVECEXP (PATTERN (insn), 0, i), &cur_stack_offset);
        }

      insn = NEXT_INSN (insn);
    }

  cur_stack_offset = 0;
  insn = f;
  while (insn)
    {
      if (GET_RTX_CLASS (GET_CODE (insn)) == 'i')
	{
	  spill_peep_insn (insn, PATTERN (insn));
	  if (have_flow_analysis 
	      && GET_CODE (PATTERN (insn)) == SET
	      && GET_CODE (SET_DEST (PATTERN (insn))) == REG
	      && GET_CODE (SET_SRC (PATTERN (insn))) == REG
	      && (next = next_nonnote_insn (insn))
	      && GET_CODE (next) == INSN
	      && GET_CODE (next) == INSN
	      && (note = find_regno_note (next, 
		                    REG_DEAD, 
		                    REGNO (SET_DEST (PATTERN (insn)))))
	      && rtx_equal_p (XEXP (note, 0), SET_DEST (PATTERN (insn))))
            { /* we have reg1 <- reg2 and reg1 dead in next insn
              replace reg1 with reg2 in next insn and delete this
              insn */
              init_undo_buf ();
              subst_in_insn (next,
                             PATTERN (next),
                             SET_DEST (PATTERN (insn)),
                             SET_SRC (PATTERN (insn)),
                             0,
                             0);
              if (reg_overlap_mentioned_p (XEXP (note, 0), PATTERN (next))
                  || (insn_code_number = recog (PATTERN (next), next, 0)) < 0)
                {
                  loop_undo_all ();
                  success = 0;
                }
              else
                {
                  success = 1;
#ifdef REGISTER_CONSTRAINTS
                  INSN_CODE (next) = insn_code_number;
                  insn_extract (next);
                  if (!constrain_operands (insn_code_number, 1))
                    {
                      loop_undo_all ();
                      INSN_CODE (next) 
                        = recog (PATTERN (next), 
                                 next, 0);
                      success = 0;
                    }
#endif
                  if (success)
                    {
                      if (loop_dump_stream)
                        {
                          fprintf (loop_dump_stream,
                          "replace reg %d with reg %d in insn %d, delete insn %d\n",
                          REGNO (SET_DEST (PATTERN (insn))),
                          REGNO (SET_SRC (PATTERN (insn))),
                          INSN_UID (next), INSN_UID (insn));
                        }
		      PUT_CODE (insn, NOTE);
		      NOTE_LINE_NUMBER (insn) = NOTE_INSN_DELETED;
		      NOTE_SOURCE_FILE (insn) = 0;
		      remove_note (next, note);
                    }
                }
            }
        }
      
      if (GET_CODE (insn) == CODE_LABEL
          || GET_CODE (insn) == CALL_INSN) /* call_insn's can clobber anything */
	peep_init_info ();

      insn = NEXT_INSN (insn);
    }
  have_flow_analysis = 0;
}

static
void 
peep_init_info ()
{
  int i;
  for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
    {
      spill_candidate[i].reload_slot = 0;
      spill_candidate[i].disregard = 0;
    }
  for (i=0; i < num_reload_stack_slots; i++)
    {
      reload_slots[i].is_loaded_to_reg = -1;
      reload_slots[i].last_store = NULL;
    }
}

/*
  In body of insn replace any reload stack slots with registers that
  have the same value
*/
static
void 
spill_peep_insn (insn, body)
     rtx insn;
     rtx body;
{
  enum rtx_code code;
  int i, j, indx, regno, cand_num, insn_code_number, cur_code_number, success,
      n_regs;

  undobuf.num_undo = 0;
  code = GET_CODE (body);

  if (code == SET)
    {
      if (/*cur_stack_offset == 0*/ 1)
	{
	  if (GET_CODE (SET_SRC (body)) == MEM 
	      && is_a_reload_stack_slot (SET_SRC (body), &indx, cur_stack_offset, 0))
	    {
	      cand_num = reload_slots[indx].is_loaded_to_reg;
	      if (cand_num != -1 && spill_candidate[cand_num].disregard == 0 
		  && spill_candidate[cand_num].reload_slot 
		  && GET_MODE (spill_candidate[cand_num].reload_slot)
		       == GET_MODE (SET_SRC (body))
		  && are_same_reload_slots (spill_candidate[cand_num].reload_slot,
		               spill_candidate[cand_num].cur_stack_offset,
			       SET_SRC (body), cur_stack_offset, 1))
		{
		  SET_SRC (body) = spill_candidate[cand_num].hard_reg;
		  if ((insn_code_number = recog (PATTERN (insn), insn, 0)) < 0)
		    {
		      SET_SRC (body) = spill_candidate[cand_num].reload_slot;
		    }
		  else
		    {
		      success = 1;
#ifdef REGISTER_CONSTRAINTS
		      INSN_CODE (insn) = insn_code_number;
		      insn_extract (insn);
		      if (!constrain_operands (insn_code_number, 1))
			{
			  SET_SRC (body) 
			    = spill_candidate[cand_num].reload_slot;
			  INSN_CODE (insn) = recog (PATTERN (insn), insn, 0);
			  success = 0;
			}
#endif
		      if (success && loop_dump_stream)
			{
			  fprintf (loop_dump_stream,
			     " replacing stack slot (src) with reg%d in insn %d\n",
				   REGNO (spill_candidate[cand_num].hard_reg), INSN_UID (insn));
			}
		    }
		}
	    }
	  else
	    {
              undobuf.storage = (char *) oballoc (0);
	      replace_spill_slots (insn, SET_SRC (body), &SET_SRC (body));
	      if (undobuf.num_undo)
	        {
		  if ((insn_code_number = recog (PATTERN (insn), insn, 0)) < 0)
		    {
		      loop_undo_all ();
		      if (loop_dump_stream)
			 fprintf (loop_dump_stream, "  replacment did not succeed %d\n", INSN_UID (insn));
		    }
#ifdef REGISTER_CONSTRAINTS
		  else
		    {
		      INSN_CODE (insn) = insn_code_number;
		      insn_extract (insn);
		      if (!constrain_operands (insn_code_number, 1))
			{
			  loop_undo_all ();
		          INSN_CODE (insn) = recog (PATTERN (insn), insn, 0);
			  if (loop_dump_stream)
			    fprintf (loop_dump_stream, "  replacment did not succeed %d\n", INSN_UID (insn));
			}
		    }
#endif
                  undobuf.num_undo = 0;
                }
	    }
	}
      if (GET_CODE (SET_DEST (body)) == REG
	  && REGNO (SET_DEST (body)) < FIRST_PSEUDO_REGISTER)
	{
	  regno = REGNO (SET_DEST (body));
	  if (HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body))) > 1)
	    {
	      regno = REGNO (SET_DEST (body));
	      n_regs = HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body)));
	      for (j = 0; j < n_regs; j++)
	        {
		      spill_candidate[regno + j].disregard = 1;
	              spill_candidate[regno + j].reload_slot = NULL;
	              for (i=0; i < num_reload_stack_slots; i++)
	                if (reload_slots[i].is_loaded_to_reg == regno + i)
	                  reload_slots[i].is_loaded_to_reg = -1;
	        }
	    }
	  if (GET_CODE (SET_SRC (body)) == MEM 
	      && is_a_reload_stack_slot (SET_SRC (body), &indx, cur_stack_offset, 0) 
	      && HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body))) == 1)
	    { /* It sets it to the value in a reload stack slot */
	      spill_candidate[regno].reload_slot = SET_SRC (body);
	      spill_candidate[regno].hard_reg = SET_DEST (body);
	      spill_candidate[regno].disregard = 0;
	      spill_candidate[regno].reload_insn = insn;
	      spill_candidate[regno].cur_stack_offset = cur_stack_offset;
              reload_slots[indx].last_store = NULL;
	      reload_slots[indx].is_loaded_to_reg = regno;
	    }
	  else
	    {
	      spill_candidate[regno].disregard = 1;
	      spill_candidate[regno].reload_slot = NULL;
	      for (i=0; i < num_reload_stack_slots; i++)
	        if (reload_slots[i].is_loaded_to_reg == regno)
	          reload_slots[i].is_loaded_to_reg = -1;
	    }
	}
      else if (GET_CODE (SET_DEST (body)) == SUBREG
               && GET_CODE (SUBREG_REG (SET_DEST (body))) == REG
	       && REGNO (SUBREG_REG (SET_DEST (body))) < FIRST_PSEUDO_REGISTER)
	{ /* Simple handling of subregs - invalidate the subreg being set */
	  regno = REGNO (SUBREG_REG (SET_DEST (body)));
	  n_regs = HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body)));
	  for (j = 0; j < n_regs; j++)
	    {
		  spill_candidate[regno + j].disregard = 1;
	          spill_candidate[regno + j].reload_slot = NULL;
	          for (i=0; i < num_reload_stack_slots; i++)
	            if (reload_slots[i].is_loaded_to_reg == regno + j)
	              reload_slots[i].is_loaded_to_reg = -1;
	    }
	}
      else if (is_a_reload_stack_slot (SET_DEST (body), &indx, cur_stack_offset, 0))
	{ /* This is an update of a reload_stack_slot */
              /* Check if last write into this reload slot can be deleted */
              if (reload_slots[indx].last_store
                  && GET_CODE (SET_DEST (body)) != SUBREG
                  /* && GET_CODE (SET_SRC (body)) == REG */
		  && REGNO (SET_SRC (body)) < FIRST_PSEUDO_REGISTER
 	          && GET_MODE (SET_DEST ( PATTERN (reload_slots[indx].last_store)))
	              == GET_MODE (SET_DEST (body))
                  && spill_slot_unused_between (
                      SET_DEST ( PATTERN (reload_slots[indx].last_store)),
                      reload_slots[indx].last_store, insn))
                {
                  if (loop_dump_stream)
                    {
                      fprintf (loop_dump_stream, "delete redundant spill store at %d becuase of re-store at %d\n",
                               INSN_UID (reload_slots[indx].last_store),
                               INSN_UID (insn));
                    }
		  PUT_CODE (reload_slots[indx].last_store, NOTE);
		  NOTE_LINE_NUMBER (reload_slots[indx].last_store) = NOTE_INSN_DELETED;
		  NOTE_SOURCE_FILE (reload_slots[indx].last_store) = 0;
		  reload_slots[indx].is_loaded_to_reg = -1;
		  reload_slots[indx].last_store = NULL;
                 }
	      if (GET_CODE (SET_SRC (body)) == REG 
                  && GET_CODE (SET_DEST (body)) != SUBREG
	          && HARD_REGNO_NREGS(REGNO(SET_SRC(body)),
			  		GET_MODE(SET_DEST(body))) == 1
		  && REGNO (SET_SRC (body)) < FIRST_PSEUDO_REGISTER)
		{
		  regno = REGNO (SET_SRC (body));
		  spill_candidate[regno].reload_slot = SET_DEST (body);
		  spill_candidate[regno].hard_reg = SET_SRC (body);
		  spill_candidate[regno].disregard = 0;
		  spill_candidate[regno].reload_insn = insn;
        	  spill_candidate[regno].cur_stack_offset = cur_stack_offset;
        	  reload_slots[indx].last_store = insn;
		  reload_slots[indx].is_loaded_to_reg = regno;
		}
	      else
	        {
                  regno = reload_slots[indx].is_loaded_to_reg;
                  if (regno != -1)
                    { /* This candidate is now invalid */
		       spill_candidate[regno].disregard = 1;
		       spill_candidate[regno].reload_slot = NULL;
                    }
		  reload_slots[indx].is_loaded_to_reg = -1;
		  reload_slots[indx].last_store = NULL;
	        }
	}
      update_stack_offset (body, &cur_stack_offset);
    }
  else if (code == CLOBBER)
    {
      if (GET_CODE (SET_DEST (body)) == REG 
	  && REGNO (SET_DEST (body)) < FIRST_PSEUDO_REGISTER)
	{
	  regno = REGNO (SET_DEST (body));
	  n_regs = HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body)));
	  for (j = 0; j < n_regs; j++)
	    {
	      spill_candidate[regno + j].disregard = 1;
	      for (i=0; i < num_reload_stack_slots; i++)
	        if (reload_slots[i].is_loaded_to_reg == (regno + j))
	          reload_slots[i].is_loaded_to_reg = -1;
	    }
	}
      else if (GET_CODE (SET_DEST (body)) == SUBREG
               && GET_CODE (SUBREG_REG (SET_DEST (body))) == REG 
	       && REGNO (SUBREG_REG (SET_DEST (body))) < FIRST_PSEUDO_REGISTER)
	{
	  regno = REGNO (SUBREG_REG (SET_DEST (body)));
	  n_regs 
	    = max (HARD_REGNO_NREGS (regno, GET_MODE (SET_DEST (body))),
	           HARD_REGNO_NREGS (regno, 
	                             GET_MODE (SUBREG_REG (SET_DEST (body)))));
	  for (j = 0; j < n_regs; j++)
	    {
	      spill_candidate[regno + j].disregard = 1;
	      for (i=0; i < num_reload_stack_slots; i++)
	        if (reload_slots[i].is_loaded_to_reg == (regno + j))
	          reload_slots[i].is_loaded_to_reg = -1;
	    }
	}
      else if (is_a_reload_stack_slot (SET_DEST (body), &indx, cur_stack_offset, 0))
        {
          regno = reload_slots[indx].is_loaded_to_reg;
          if (regno != -1)
            { /* This candidate is now invalid */
		spill_candidate[regno].disregard = 1;
		spill_candidate[regno].reload_slot = NULL;
	    }
          reload_slots[indx].is_loaded_to_reg = -1;
	  reload_slots[indx].last_store = NULL;
        }
    }
  if (code == PARALLEL)
    {
      for (i = 0; i < XVECLEN (body, 0); i++)
	spill_peep_insn (insn, XVECEXP (body, 0, i));
    }
  if (GET_CODE (insn) == CALL_INSN)
    { /* All call clobbered registers are no longer valid */
      for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
	{
	  if (call_used_regs[i])
	    {
	      spill_candidate[i].disregard = 1;
	      for (j=0; j < num_reload_stack_slots; j++)
	        if (reload_slots[j].is_loaded_to_reg == i)
	          reload_slots[j].is_loaded_to_reg = -1;
	    }
	}
    }
}

/*
  Return 1 if we are sure that the spill_slot  (or any subpart of it)
  is not referenced between start and end.  Also there must be no 
  CALL JUMP or LABELs in between them
*/
static int
spill_slot_unused_between (spill_slot, start, end)
  rtx spill_slot, start, end;
{
  rtx tinsn;
  int mcur_stack_offset = 0;

  for (tinsn = next_nonnote_insn (start);
       tinsn && tinsn != end;
       tinsn = next_nonnote_insn (tinsn))
    {
      if (GET_CODE (tinsn) == CALL_INSN || GET_CODE (tinsn) == JUMP_INSN 
          || GET_CODE (tinsn) == CODE_LABEL)
        return 0;

      if (spill_slot_mentioned(spill_slot, PATTERN (tinsn), mcur_stack_offset))
        return 0;

      update_stack_offset (PATTERN (tinsn), &mcur_stack_offset);
    }

  return 1;
}

/*
  Return 1 if spill_slot or any part of it is mentioned in pat.
  relative_stack_offset is the offset of the stack pointer relative
  to the point at which spill_slot was stored into.  
*/

static int
spill_slot_mentioned (spill_slot, pat, relative_stack_offset)
     rtx pat, spill_slot;
     int relative_stack_offset;
{
  register char *fmt;
  register int i;
  register enum rtx_code code;

  code = GET_CODE (pat);

  switch (code)
    {
      /* These codes have no constituent expressions
	 and are unique.  */
    case REG:
    case SCRATCH:
    case CC0:
    case PC:
    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
      return 0;

    case MEM:
      if (is_a_stack_slot (pat)
          && are_same_reload_slots (spill_slot, 0, pat, relative_stack_offset, 0))
        return 1;

      return 0;
    }


  fmt = GET_RTX_FORMAT (code);

  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'E')
	{
	  register int j;
	  for (j = XVECLEN (pat, i) - 1; j >= 0; j--)
	    if (spill_slot_mentioned (spill_slot, XVECEXP (pat, i, j),
	                              relative_stack_offset))
	      return 1;
	}
      else if (fmt[i] == 'e')
	if (spill_slot_mentioned (spill_slot, XEXP (pat, i),
	                              relative_stack_offset))
	  return 1;
    }
  
  return 0;
}


/*
  Keep track of the depth of the stack
*/
/* static */
void 
update_stack_offset (pat, addr_stack_offset)
     rtx pat;
     int *addr_stack_offset;
{
  int i;
  if (GET_CODE (pat) != SET)
    return;

  if (GET_CODE (SET_DEST (pat)) == REG
      && REGNO (SET_DEST (pat)) == STACK_POINTER_REGNUM
      && GET_CODE (SET_SRC (pat)) == PLUS
      && GET_CODE (XEXP (SET_SRC (pat), 0)) == REG
      && REGNO (XEXP (SET_SRC (pat), 0)) == STACK_POINTER_REGNUM)
    {
      *addr_stack_offset -= XINT (XEXP (SET_SRC (pat), 1), 0);
    }
  else if (GET_CODE (SET_DEST (pat)) == MEM 
	   && GET_CODE (XEXP (SET_DEST (pat), 0)) == PRE_DEC
	   && XEXP (XEXP (SET_DEST (pat), 0), 0) == stack_pointer_rtx)
    {
      spill_candidate[STACK_POINTER_REGNUM].disregard = 1;
      for (i=0; i < num_reload_stack_slots; i++)
	if (reload_slots[i].is_loaded_to_reg == STACK_POINTER_REGNUM)
	  reload_slots[i].is_loaded_to_reg = -1;

      *addr_stack_offset += GET_MODE_SIZE (GET_MODE (SET_DEST (pat)));
    }
  else if (GET_CODE (SET_DEST (pat)) == MEM 
	   && GET_CODE (XEXP (SET_DEST (pat), 0)) == PRE_INC
	   && XEXP (XEXP (SET_DEST (pat), 0), 0) == stack_pointer_rtx)
    {
      spill_candidate[STACK_POINTER_REGNUM].disregard = 1;
      for (i=0; i < num_reload_stack_slots; i++)
	if (reload_slots[i].is_loaded_to_reg == STACK_POINTER_REGNUM)
	  reload_slots[i].is_loaded_to_reg = -1;

      *addr_stack_offset -= GET_MODE_SIZE (GET_MODE (SET_DEST (pat)));
    }
}


/*
  Replace stack spill slots in pat of insn with the appropriate spill
  candidate
*/
static
void
replace_spill_slots (insn, pat, where)
     rtx insn;
     rtx pat;
     rtx *where;
{
  register RTX_CODE code = GET_CODE (pat);
  register int i, j;
  register char *fmt;
  int cand_num, indx;
  rtx note;

  switch (code)
    {
    case HIGH:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
    case REG:
      return;

    default:
      break;
    }
  if (GET_CODE (pat) == MEM && is_a_reload_stack_slot (pat, &indx, cur_stack_offset, 0))
    {
      cand_num = reload_slots[indx].is_loaded_to_reg;
      if (cand_num != -1 && spill_candidate[cand_num].disregard == 0 
	  && spill_candidate[cand_num].reload_slot 
	  && GET_MODE (spill_candidate[cand_num].reload_slot)
	       == GET_MODE (pat)
	  && are_same_reload_slots (spill_candidate[cand_num].reload_slot,
	                            spill_candidate[cand_num].cur_stack_offset,
	                            pat, cur_stack_offset, 1))
	{
	  note = find_reg_note (spill_candidate[cand_num].reload_insn,
				REG_DEAD,
				spill_candidate[cand_num].hard_reg);
	  if (note)
	    {
	      remove_note (spill_candidate[cand_num].reload_insn, note);
	    }
	  if (loop_dump_stream)
	    {
	      fprintf (loop_dump_stream, " replacing stack slot with reg%d in insn %d\n",
	       REGNO (spill_candidate[cand_num].hard_reg), INSN_UID (insn));
	    }
	  LOOP_SUBST (where, spill_candidate[cand_num].hard_reg);
	}
    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      replace_spill_slots (insn, XEXP (pat, i), &XEXP (pat, i));
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (pat, i); j++)
	  replace_spill_slots (insn, XVECEXP (pat, i, j), &XVECEXP (pat, i, j));
      }
}

/*
  Replace any hard registers in pat of insn, that are on a linked
  list of spill candidates, with the head candidate of the list
*/
static
rtx 
replace_linked_regs (insn, pat)
     rtx insn;
     rtx pat;
{
  register RTX_CODE code = GET_CODE (pat);
  register int i, j;
  register char *fmt;
  int cand_num, indx;
  rtx note;

  switch (code)
    {
    case HIGH:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
      return pat;

    case REG:
      if (spill_candidate[REGNO (pat)].head_cand 
	  && (spill_candidate[REGNO (pat)].head_cand)->disregard == 0)
	{
	  if (loop_dump_stream)
	    {
	      fprintf (loop_dump_stream, " replacing reg%d with reg%d in insn %d\n",
		       REGNO (pat),
		 REGNO ((spill_candidate[REGNO (pat)].head_cand)->hard_reg),
		       INSN_UID (insn));
	    }
	  return ((spill_candidate[REGNO (pat)].head_cand)->hard_reg);
	}
      else
	return pat;

    default:
      break;
    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      XEXP (pat, i) = replace_linked_regs (insn, XEXP (pat, i));
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (pat, i); j++)
	  XVECEXP (pat, i, j) 
	    = replace_linked_regs (insn, XVECEXP (pat, i, j));
      }
  return pat;
}


/*
  Mark all registers mentioned in insns between loop_start and loop_end
  and that are also mentioned outside of the loop
*/
static void
mark_regs_live_outside (loop_start, loop_end)
     rtx loop_start;
     rtx loop_end;
{
  rtx insn;
  int in_loop = 0;

  insn = get_insns ();
  while (insn)
    {
      if (insn == loop_start)
	{
	  in_loop = 1;
	}
      else if (insn == loop_end)
	{
	  in_loop = 0;
	}
      else if (in_loop == 0 
	       && (GET_CODE (insn) == INSN || GET_CODE (insn) == CALL_INSN 
		|| GET_CODE (insn) == JUMP_INSN))
	{
	  mark_regs_mentioned (PATTERN (insn), insn);
	}
      insn = NEXT_INSN (insn);
    }
}


/*
  Each register mentioned in pat, except for the destination register of
  the insn, is marked as reg_live_outside_loop
*/
static void
mark_regs_mentioned (pat, insn)
     rtx pat, insn;
{
  register char *fmt;
  register int i;
  register enum rtx_code code;

  if (GET_CODE (pat) == SET
      && GET_CODE (SET_DEST (pat)) == REG && is_a_param_slot (SET_SRC (pat)))
    {
      return;
    }
    
  code = GET_CODE (pat);

  switch (code)
    {
    case REG:
      if (REGNO (pat) < max_reg_before_loop)
	{
	  reg_live_outside_loop[REGNO (pat)] = 1;
	}
      return;

      /* These codes have no constituent expressions
	 and are unique.  */
    case SCRATCH:
    case CC0:
    case PC:
    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
      return;


    }


  fmt = GET_RTX_FORMAT (code);

  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'E')
	{
	  register int j;
	  for (j = XVECLEN (pat, i) - 1; j >= 0; j--)
	    mark_regs_mentioned (XVECEXP (pat, i, j), insn);
	}
      else if (fmt[i] == 'e')
	mark_regs_mentioned (XEXP (pat, i), insn);
    }
}

/* Return 1 iff there is a loop_begin note between loop_start and loop_end
   otherwise return 0 */

static int
loop_contains_loop (loop_start, loop_end)
     rtx loop_start;
     rtx loop_end;
{
  rtx insn;

  insn = NEXT_INSN (loop_start);
  while (insn && insn != loop_end)
    {
      if (GET_CODE (insn) == NOTE
	  && NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_BEG)
	return 1;

      insn = NEXT_INSN (insn);
    }
  
  return 0;
}


/* reload_slots[indx] is a reload slot that was never updated in loop
   but could not be lifted only because the register it is
   loaded into was also set to something else in loop.
   Lets see if the reload slot was set in the basic block
   just prior to the loop by some inexpensive operation.
   (particularly a move from memory into a register and
   a move of that register onto stack). If so lets
   replace each load of the reload slot in the loop with
   a load of the memory and delete the code that loaded
   and spilled it outside the loop.
   This corrects the preloading of loop invariants that
   could not be held in registers over the loop
 */
static
void 
push_load_into_loop (loop_start, loop_end, indx)
     rtx loop_start;
     rtx loop_end;
     int indx;
{
  rtx insn, insn_sets_reload_slot, val_in_reload_slot, insns_deleted[2],
    update_log_links_start, rn, prev, link;
  int rindx, i, found_set, num_deleted;

  if (loop_dump_stream)
    {
      fprintf (loop_dump_stream, "\nTry to push load of pseudo %d into loop\n",
	       reload_slots[indx].stack_slot_pseudo_reg);
    }
  /* Look for insn before the loop that sets this reload slot */
  insn = PREV_INSN (loop_start);
  insn_sets_reload_slot = 0;
  while (insn && !insn_sets_reload_slot)
    {
      if (GET_CODE (insn) == CALL_INSN || GET_CODE (insn) == JUMP_INSN 
	  || GET_CODE (insn) == CODE_LABEL 
	  || (GET_CODE (insn) == NOTE
	       && (NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_BEG 
	           || NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_END
	          )
	     ) 
	  || (GET_CODE (insn) == INSN && GET_CODE (PATTERN (insn)) != SET))
	{
	  break;
	}
      else if (GET_CODE (insn) == INSN 
	       && rtx_equal_p (SET_DEST (PATTERN (insn)), reload_slots[indx].reload_stack_slot))
	{
	  insn_sets_reload_slot = insn;
	}
      insn = PREV_INSN (insn);
    }
  if (!insn_sets_reload_slot)
    {
      return;
    }
  /* Now see if the source is a register and that register is
     set by the immediately preceeding insn - then source of
     the immediately preceeding insn is the value in the reload
     slot throughout the loop */
  if (GET_CODE (SET_SRC (PATTERN (insn_sets_reload_slot))) == REG 
      && GET_CODE (PREV_INSN (insn_sets_reload_slot)) == INSN 
      && GET_CODE (PATTERN (PREV_INSN (insn_sets_reload_slot))) == SET 
      && rtx_equal_p (SET_DEST (PATTERN (PREV_INSN (insn_sets_reload_slot))),
		   SET_SRC (PATTERN (insn_sets_reload_slot))))
    {
      val_in_reload_slot = SET_SRC (PATTERN (PREV_INSN (insn_sets_reload_slot)));
    }
  else
    {
      return;
    }
  if (is_a_reload_stack_slot (val_in_reload_slot, &rindx, cur_stack_offset, 0))
    { /* Don't want to handle these */
      return;
    }
  /* Now make sure that this value is not changed in loop */
  if (GET_CODE (val_in_reload_slot) != MEM || !invariant_p (val_in_reload_slot))
    {
      return;
    }
  for (i = loop_store_mems_idx - 1; i >= 0; i--)
    {
      if (true_dependence (loop_store_mems[i], val_in_reload_slot) 
	  && !is_a_reload_stack_slot (loop_store_mems[i], &rindx, cur_stack_offset, 0)
	)
	{
	  return;
	}
    }
  /* Now we know that the value can be substituted and that the store into
     the reload slot outside the loop (i.e. insn_sets_reload_slot)
     can be deleted.
     we also want to delete the load of the reload slot
     (i.e PREV_INSN(insn_sets_reload_slot)). we can do this if the register
     loaded (i.e. SET_DEST(PATTERN(PREV_INSN(insn_sets_reload_slot))) is
     set before used after PREV_INSN(insn_sets_reload_slot))
  */
  insn = NEXT_INSN (insn_sets_reload_slot);
  found_set = 0;
  while (!found_set && insn && insn != loop_end)
    {
      if (GET_CODE (insn) == CALL_INSN || GET_CODE (insn) == JUMP_INSN 
	  || (GET_CODE (insn) == CODE_LABEL && PREV_INSN (insn) != loop_start))
	{
	  break;
	}
      if (GET_CODE (insn) == INSN)
	{
	  if (reg_overlap_mentioned_p (
		     SET_DEST (PATTERN (PREV_INSN (insn_sets_reload_slot))),
					PATTERN (insn)))
	    {
	      break;
	    }
	  if (GET_CODE (PATTERN (insn)) == SET 
	      && rtx_equal_p (SET_DEST (PATTERN (insn)),
		    SET_DEST (PATTERN (PREV_INSN (insn_sets_reload_slot)))))
	    {
	      found_set = 1;
	    }
	}
      insn = NEXT_INSN (insn);
    }
  /* Do the replacement throughout the loop */
  if (loop_dump_stream)
    {
      fprintf (loop_dump_stream, "replacing reload slot %d with: throughout loop\n",
	       indx);
	  tdebug_rtx (val_in_reload_slot, loop_dump_stream);
    }
  insn = NEXT_INSN (insn_sets_reload_slot);
  while (insn != loop_end)
    {
      if (GET_CODE (insn) == INSN || GET_CODE (insn) == CALL_INSN 
	  || GET_CODE (insn) == JUMP_INSN)
	{
          init_undo_buf ();
	  subst_in_insn (insn, PATTERN (insn),
			 reload_slots[indx].reload_stack_slot,
			 val_in_reload_slot, 1, 0);

	}
      insn = NEXT_INSN (insn);
    }
  insns_deleted[0] = NULL;
  insns_deleted[1] = NULL;
  num_deleted = 0;
  update_log_links_start = PREV_INSN (insn_sets_reload_slot);
  if (found_set)
    {
      if (loop_dump_stream)
	fprintf (loop_dump_stream, " delete insn %d\n",
		 INSN_UID (PREV_INSN (insn_sets_reload_slot)));

      insns_deleted[num_deleted++] = PREV_INSN (insn_sets_reload_slot);
      delete_insn (PREV_INSN (insn_sets_reload_slot));
    }

  if (loop_dump_stream)
    fprintf (loop_dump_stream, " delete insn %d\n",
	     INSN_UID (insn_sets_reload_slot));

  insns_deleted[num_deleted++] = insn_sets_reload_slot;
  delete_insn (insn_sets_reload_slot);
  /* Remove these deleted insns from any log links they are on */
  for (rn = update_log_links_start; rn != loop_start; rn = NEXT_INSN (rn))
    {
      if ((GET_CODE (rn) == INSN || GET_CODE (rn) == JUMP_INSN
	   || GET_CODE (rn) == CALL_INSN) && LOG_LINKS (rn))
	{
	  for (prev = 0, link = LOG_LINKS (rn); link;
	       prev = link, link = XEXP (link, 1))
	    {
	      if (XEXP (link, 0) == insns_deleted[0] 
		  || (num_deleted == 2 && XEXP (link, 0) == insns_deleted[1]))
		{
		  if (prev)
		    XEXP (prev, 1) = XEXP (link, 1);
		  else
		    LOG_LINKS (rn) = XEXP (link, 1);
		}
	    }
	}
    }
}

void
init_undo_buf ()
{
  undobuf.storage = (char *) oballoc (0);
  undobuf.num_undo = 0;
}

/* In pat of insn replace org with new. If not_dest is 1 then
   don't do the rplacement if org is ae destination of a SET
*/
void 
subst_in_insn (insn, pat, org, new, not_dest, is_dest)
     rtx insn, pat, org, new;
     int not_dest;
     int is_dest;
{
  inner_subst_in_insn (insn, pat, &pat, org, new, not_dest, is_dest);
}

/* In pat of insn replace org with new. If not_dest is 1 then
   don't do the rplacement if org is ae destination of a SET
*/
static void 
inner_subst_in_insn (insn, pat, where, org, new, not_dest, is_dest)
     rtx insn, pat, org, new, *where;
     int not_dest;
     int is_dest;
{
  register RTX_CODE code = GET_CODE (pat);
  register int i, j;
  register char *fmt;

  if (not_dest && is_dest)
    {
      return;
    }
  switch (code)
    {
    case HIGH:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
      return ;


    }

  if (rtx_equal_p (pat, org))
    {
      LOOP_SUBST (where, new);
      return;
    }
  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      {
        if (not_dest && code == SET && i == 0)
          is_dest = 1;
        else
          is_dest = 0;
        if( XEXP (pat, i))
          inner_subst_in_insn (insn, XEXP (pat, i), &XEXP (pat, i), org, new, not_dest, is_dest);
      }
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (pat, i); j++)
	  inner_subst_in_insn (insn, XVECEXP (pat, i, j), &XVECEXP (pat, i, j), org, new, not_dest, 0);
      }
}


/* tsl - begin */

/* Indexed by i, each entry indicates the last memory, register i was loaded
   with, in the current loop (NULL if no reload until now) */
   
static rtx last_reloads[FIRST_PSEUDO_REGISTER];

#define LAST_RELOAD(REGNO) last_reloads[(REGNO)]

/* If entry > 0 then the higher it is the less recently used it is.  
   If entry == 0 then that register number has not been used as reload
   since last lable.  
 */

static int reload_priority[FIRST_PSEUDO_REGISTER];

/* Set of registers that are free and ready to be used at this moment.  */

extern HARD_REG_SET free_regs;

extern void update_free_regs PROTO((rtx));
extern int find_free_reg PROTO((enum machine_mode));
static int find_last_reload PROTO((rtx));

/* Replaces the reload registers of reload insns as follows:

   for each insn which loads memory into a register,
   if the reload register is loaded with a memory, which was previously
   loaded, and the reload register dies in the next insn, we
   replace the reload register, with that already loaded with the stack slot
   (assuming that register is free - we use the same free register scanning
   method of the memory_simplify() function). 
   If the reload register was already used since the last label or call then
   try to find a free register which was used less recently and replace the
   reload register with the register found.  
   */

void
replace_reload_registers (f, file)
     rtx f;
     FILE *file;
{
  HARD_REG_SET saved_free_regs; /* for testing find_free_regs_between () */

  /* indicate wheter we are in a loop now */

  int in_loop_f = 0;
  int save_code;
  int i;

  /* make flow_analysis without changing the insns and saving the pseudo
     registers information in basic_block_live_at_start */

  hard_reg_flow_analysis (f, file, 1 /* save pseudo register information */);
  have_flow_analysis = 1;
  bzero (last_reloads, sizeof last_reloads);

  if (n_basic_blocks > 0)
    {
      register int regno, block;

      /* To determine which registers are live after any insn, start from
	 the beginning of the basic block and scan insns, noting
         which registers are set by each insn and which die there. */

      for (block = 0; block < n_basic_blocks; block++)
	{
	  register rtx insn;
	  int end_of_block_f;

	  if (file)
	    fprintf (file, "\nblock number: %d:\n", block);

	  for (regno = 0; regno < FIRST_PSEUDO_REGISTER; regno++)
	    /* at the beginning of a block, each non-live hard register is
	       considered free. */
	    {
	      register int offset = regno / REGSET_ELT_BITS;
	      register int bit = 1 << (regno % REGSET_ELT_BITS);
	      if (basic_block_live_at_start[block][offset] & bit)
		CLEAR_HARD_REG_BIT (free_regs, regno);
	      else
		SET_HARD_REG_BIT (free_regs, regno);
	    }

	  end_of_block_f = 0;	/* signals the end of block reached */

	  /* Traverse the block, an insn by insn.  Maintain a history
	     of free registers and registers used to load memory
	     for use only in next insn (these are called reload_registers).  
	     At CALLs and CODE_LABELs restart the history of reload_registers.
	  */

	  for (insn = basic_block_head[block];
	       ! end_of_block_f;
	       insn = NEXT_INSN (insn))
	    {
	      rtx note, next, pat;
	      rtx loop_start; /* use to test find_free_regs_between () */
	      int regno;

	      if (GET_CODE (insn) == CALL_INSN)
		{
	          bzero (last_reloads, sizeof last_reloads);
	          bzero (reload_priority, sizeof reload_priority);
	        }
	      if (GET_CODE (insn) == CODE_LABEL)
		{
	          bzero (last_reloads, sizeof last_reloads);
	          bzero (reload_priority, sizeof reload_priority);
	          loop_start = insn;
	          if (GET_CODE (PREV_INSN (insn)) == NOTE
	              && NOTE_LINE_NUMBER (PREV_INSN (insn)) 
	                == NOTE_INSN_LOOP_BEG)
	            {
	              in_loop_f = 1;
	              COPY_HARD_REG_SET (saved_free_regs, free_regs);
	            }
	        }
	      if (GET_CODE (insn) == JUMP_INSN
		  && GET_CODE (NEXT_INSN (insn)) == NOTE
		  && NOTE_LINE_NUMBER (NEXT_INSN (insn)) == NOTE_INSN_LOOP_END)
		{
		  register int regno;
#if 0
	          if (in_loop_f)
		    {
		      find_free_regs_between (& saved_free_regs,
		  			      loop_start,
		  			      insn);
		      if (file)
		        {
		          fprintf (file,
		      	           "free regs found in loop (insns %d - %d): ",
		      	           INSN_UID (loop_start), INSN_UID (insn));
		          for (regno = 0; regno < FIRST_PSEUDO_REGISTER; regno++)
		            if (TEST_HARD_REG_BIT (saved_free_regs, regno))
		              fprintf (file, "%d ", regno);
		          fprintf (file, "\n");
		        }
		    }
#endif
		  in_loop_f = 0;
		}
	    
	      if (/*in_loop_f*/ 1)
	        {
		  if (GET_RTX_CLASS (GET_CODE (insn)) == 'i'
		      /*&& RTX_IS_SPILL_P (insn)*/
		      && (pat = PATTERN (insn))
	              && GET_CODE (pat) == SET
	              && GET_CODE (SET_DEST (pat)) == REG
	              && GET_CODE (SET_SRC (pat)) == MEM)
	            {
	              regno = REGNO (SET_DEST (pat));

	              if ((next = next_nonnote_insn (insn))
	                  && GET_CODE (next) == INSN
	                  && (note = find_regno_note (next,
						      REG_DEAD,
						      regno)))
			{
			  /* look for new_regno to replace with */
		          int new_regno = find_last_reload (SET_SRC (pat));
		          enum machine_mode mode = GET_MODE (SET_DEST (pat));

                          /* tevid */
                          if (new_regno != FIRST_PSEUDO_REGISTER)
                            { /* The mode must be ok */
                              if (!HARD_REGNO_MODE_OK (new_regno, mode))
                                new_regno = FIRST_PSEUDO_REGISTER;
                            }
		          if (new_regno == FIRST_PSEUDO_REGISTER
			      || ! TEST_HARD_REG_BIT (free_regs, new_regno))
			    if (/*LAST_RELOAD (regno)*/
			        reload_priority[regno])
			      /* if we already used regno for reload in this
			         loop it is better to reload the new memory
			         to another register (if we find free one) */
			      {
			        new_regno = find_free_reg (mode);
			        if (reload_priority [new_regno] &&
			            reload_priority [regno] >=
			            reload_priority [new_regno])
			          {
			            new_regno = FIRST_PSEUDO_REGISTER;
			          }
			      }
			    else
			      /* this is the first time we're using this
			         register for reload in the current loop -
			         so we need no replacement */
			      new_regno = FIRST_PSEUDO_REGISTER;





			  if (new_regno != FIRST_PSEUDO_REGISTER
			      /* if regno == new_regno no replcement needed */
			      && new_regno != regno)
	                    {
	                      register int nregs, i;
	                      int stop_replacement = 0;

		              rtx new_reg = gen_rtx (REG, mode, new_regno);
	                      rtx old_reg = XEXP (note, 0);

		              /* replace this register in the next insn */
                              init_undo_buf ();
		              subst_in_insn (next,
		    		             PATTERN (next),
		    		             old_reg,
		    		             new_reg,
		    		             0,
		    		             0);

#ifdef REGISTER_CONSTRAINTS
                              save_code = INSN_CODE (next);

	                      if ((INSN_CODE (next) = recog (PATTERN (next),
							     next,
							     0)) != -1)
		                insn_extract (next);

		              if (INSN_CODE (next) == -1
		                  || !constrain_operands (INSN_CODE (next), 1))
		                {
			          /* restore the previous situation */
			          INSN_CODE (next) = save_code;
			          loop_undo_all ();
		                  /*subst_in_insn (next,
		    		                 PATTERN (next),
		    		                 new_reg,
		    		                 old_reg,
		    		                 0,
		    		                 0);*/
		                  stop_replacement = 1;
 		                }
#endif
 		              if (! stop_replacement)
 		                {
 		                  rtx next_dest = SET_DEST (PATTERN (next));

	                          /* replace this insn's register with regno */
	                          SET_DEST (pat) = new_reg;
#if 0
                                  init_undo_buf ();
		    	          subst_in_insn (next,
						 note,
						 old_reg,
						 new_reg,
						 0,
						 0);
#endif

				  /* The regno in the REG_DEAD note should be
				     changed */

				  XEXP (note, 0) = new_reg;
		          
				  /* Now, if we replaced a REG_DEAD note, and
				     the new_reg is the destination of the next
				     insn, we should remove this note */
				  /* tevid replaced regno with new_regno,
				     twice */

				  if (GET_CODE (next_dest) == REG
				      && REGNO (next_dest) == new_regno)
				    remove_death (new_regno, next);
			  
				  /* replace the registers in the free_register
				     list */
				  nregs = HARD_REGNO_NREGS (new_regno, mode);
				  for (i = 0; i < nregs; i++)
				    CLEAR_HARD_REG_BIT (free_regs, new_regno + i);
				  nregs = HARD_REGNO_NREGS (regno, mode);
				  for (i = 0; i < nregs; i++)
				    SET_HARD_REG_BIT (free_regs, regno + i);

				  if (file)
				    {
				      fprintf (file, "insn %d: reload register",
							       INSN_UID (insn));
				      fprintf (file, " replaced (old %d new %d)\n",
				      regno, new_regno);
				    }
#ifdef REGISTER_CONSTRAINTS
				}  /* if (! stop_replacement) */
#endif

			    }  /* if (new_regno != FIRST_PSEUDO_REGISTER
			    	      && new_regno != regno) */

			    else if (file)
			       {
			         fprintf (file, "insn %d: no free register for",
							       INSN_UID (insn));
				 fprintf (file, " reload replacement\n");
			       }
			}  /* if (next = next_nonnote (insn))
				  && (note = find_regno_note (next,
				  			      REG_DEAD,
				  			      regno))) */
		      for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
		        if (reload_priority[i] > 0)
		          reload_priority[i]++;
		      reload_priority[REGNO (SET_DEST (PATTERN (insn)))] = 1;
	            }  /* if (GET_RTX_CLASS (GET_CODE (insn)) == 'i'
		              && RTX_IS_SPILL_P (insn)
		              && (pat = PATTERN (insn))
	                      && GET_CODE (pat) == SET
	                      && GET_CODE (SET_DEST (pat)) == REG */

		  /* update the last_reloads array */

		  if (GET_RTX_CLASS (GET_CODE (insn)) == 'i'
		      && (pat = PATTERN (insn))
	              && GET_CODE (pat) == SET
	              && GET_CODE (SET_DEST (pat)) == REG
	              && GET_CODE (SET_SRC (pat)) == MEM)
	            LAST_RELOAD (REGNO (SET_DEST (pat ))) = SET_SRC (pat);

	        }  /* if (in_loop_f) */

	      end_of_block_f = (insn == basic_block_end[block]);

	      if (GET_RTX_CLASS (GET_CODE (insn)) == 'i')
	        {
	          register int j;
	          rtx set;

	          if (GET_CODE (insn) == CALL_INSN)
		    /* all call clobbered registers are now free */
		    for (j = 0; j < FIRST_PSEUDO_REGISTER; j++)
		      if (call_used_regs[j])
		        SET_HARD_REG_BIT (free_regs, j);

		  update_free_regs (PATTERN (insn));

		  /* Update LAST_RELOADs that are no longer valid */
		  update_last_reloads ( PATTERN (insn));
		     

		  /* add each dead register to the free_regs set */
		  for (note = REG_NOTES (insn); note; note = XEXP (note, 1))
		    if ((REG_NOTE_KIND (note) == REG_DEAD
		         || REG_NOTE_KIND (note) == REG_UNUSED)
		        && GET_CODE (XEXP (note, 0)) == REG)
		      {
			register int i;
		        register int regno = REGNO (XEXP (note, 0));
		        register int nregs = 
			    HARD_REGNO_NREGS (regno, GET_MODE (XEXP (note, 0)));

		        for (i = 0; i < nregs; i++)
			  SET_HARD_REG_BIT (free_regs, regno + i);
		      }
		  if ((note = find_reg_note (insn, REG_WAS_0, 0)))
		    {
		      set = single_set (insn);
		      if (set == NULL 
		          || GET_CODE (SET_DEST (set)) == REG)
		        { /* Since reg may have been reused.  */
		          remove_note (insn, note);
		        }
		    }
		}

	    }  /* for (insn = basic_block_head[block]; */
	}  /* for (block = 0; block < n_basic_blocks; block++) */
    }  /* if (n_basic_blocks > 0) */
}  /* replace_reload_registers (file, f) */



/* Returns a hard register regno, if the contents of entry regno in
   last_reloads array equals to MEM, FIRST_PSEUDO_REGISTER otherwise. */

static int
find_last_reload (mem)
  rtx mem;
{
  register int regno;
  
  if (GET_CODE (mem) != MEM)
    abort ();

  for (regno = 0; regno < FIRST_PSEUDO_REGISTER; regno++)
    /* tevid added test of LAST_RELOAD (regno) != NULL and XEXPs */
    if (LAST_RELOAD (regno) != NULL
        && rtx_equal_p (XEXP (LAST_RELOAD (regno), 0), XEXP (mem, 0)))
      return regno;

  return FIRST_PSEUDO_REGISTER;
}

/* Keep track of effect of pat on last_reloads[]. Note that making
   an error will not effect the correctness of the code.  */
   
static void
update_last_reloads (pat)
  rtx pat;
{
  register int i, j, regno, nregs;
  rtx set_src, set_dest;
  register char *fmt;
  register RTX_CODE code = GET_CODE (pat);

  switch (GET_CODE (pat))
    {
      case SET:
      case CLOBBER:
        set_dest = SET_DEST (pat);
        if (GET_CODE (set_dest) == REG)
          {
            regno = REGNO (set_dest);
            nregs = HARD_REGNO_NREGS (regno, GET_MODE (set_dest));

            if (nregs != 1)
              {
	        bzero (last_reloads, sizeof last_reloads);
              }
            else
              {
                for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
                  {
                    if (LAST_RELOAD (i))
                      {
                        if (reg_overlap_mentioned_p (set_dest, LAST_RELOAD (i)))
                          {
                            LAST_RELOAD (i) = NULL;
                          }
                      }
                  }
              }
          }
        else if (GET_CODE (set_dest) == SUBREG)
          {
            if (GET_CODE (SUBREG_REG (set_dest)) == REG)
              {
                regno = REGNO (SUBREG_REG (set_dest)) + SUBREG_WORD (set_dest);
                nregs 
                  = HARD_REGNO_NREGS (regno, GET_MODE (SUBREG_REG (set_dest)));

                if (nregs != 1)
                  {
	            bzero (last_reloads, sizeof last_reloads);
                  }
                else
                  {
                    for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
                      {
                        if (LAST_RELOAD (i))
                          {
                            if (reg_overlap_mentioned_p (SUBREG_REG (set_dest), 
                            LAST_RELOAD (i)))
                              {
                                LAST_RELOAD (i) = NULL;
                              }
                          }
                      }
                  }
              }
            return;
          }
        else if (GET_CODE (set_dest) == MEM)
          {
	    for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
	      if (LAST_RELOAD (i)
	          && true_dependence (set_dest, LAST_RELOAD (i)))
	        LAST_RELOAD (i) = 0;
	    update_last_reloads (XEXP (set_dest, 0));
            return;
          }
      case CALL:
	for (j = 0; j < FIRST_PSEUDO_REGISTER; j++)
	  if (call_used_regs[j])
	    LAST_RELOAD (j) = NULL;
        return;
      case PRE_DEC:
      case PRE_INC:
      case POST_DEC:
      case POST_INC:
        if (GET_CODE (XEXP (pat ,0)) == REG)
          {
            regno = REGNO (XEXP (pat ,0));
            nregs = HARD_REGNO_NREGS (regno, GET_MODE (XEXP (pat ,0)));

            if (nregs != 1)
              {
	        bzero (last_reloads, sizeof last_reloads);
              }
            else
              {
                for (i = 0; i < FIRST_PSEUDO_REGISTER; i++)
                  {
                    if (LAST_RELOAD (i))
                      {
                        if (reg_overlap_mentioned_p (XEXP (pat ,0), LAST_RELOAD (i)))
                          {
                            LAST_RELOAD (i) = NULL;
                          }
                      }
                  }
              }
          }
        return;
    case HIGH:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
    case REG:
    /*case SUBREG:*/
        /* do nothing */
        return;
  }  /* switch (GET_CODE (pat)) */

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      {
        if( XEXP (pat, i))
          update_last_reloads (XEXP (pat, i));
      }
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (pat, i); j++)
	  update_last_reloads (XVECEXP (pat, i, j));
      }
}  

/* tsl - end */

static unsigned int largest_himode_const;

#define MAX_COMP_REGS 4

struct sign_extension_info {
  rtx insn_set;
  int num_comp_regs;
  struct comp_to_reg {
    int regno;
    rtx comp_insn;
    rtx *where;
    unsigned char for_equality;
  } compare_to_reg [MAX_COMP_REGS];
};
static struct sign_extension_info * sign_extension_infos;

static void
sign_extension_reduction (loop_start, loop_end)
  rtx loop_start, loop_end;
{
  int cur_max_regnum = max_reg_num ();
  int found_sign_extension = 0, regno, i, j;
  rtx insn, sign_ext_reg;

  if (sizeof (unsigned int) <= GET_MODE_SIZE (HImode))
    return;
  
  largest_himode_const = (1<<(GET_MODE_SIZE(HImode)*8))-1;
  sign_extension_infos 
    = (struct sign_extension_info *) 
      alloca ( sizeof (struct sign_extension_info) * cur_max_regnum);
  bzero (sign_extension_infos, 
         sizeof (struct sign_extension_info) * cur_max_regnum);
  /* Go over loop - mark registers set by sign extension */
  for (insn = loop_start; insn && insn != loop_end; insn = NEXT_INSN (insn))
    {
      if (GET_CODE (insn) == INSN && GET_CODE (PATTERN (insn)) == SET
          && GET_CODE (SET_DEST (PATTERN (insn))) == REG 
          && GET_CODE (SET_SRC (PATTERN (insn))) == SIGN_EXTEND
          && GET_MODE (SET_DEST (PATTERN (insn))) == SImode
          && GET_MODE (XEXP (SET_SRC (PATTERN (insn)), 0)) == HImode)
         { /* Found one.  */
           regno = REGNO (SET_DEST (PATTERN (insn)));
           if ( regno >= cur_max_regnum)
             return;

           found_sign_extension++;
           if (sign_extension_infos[regno].insn_set)
             return;

           sign_extension_infos[regno].insn_set = insn;
         }
    }
  if (!found_sign_extension || insn != loop_end)
    return;

  /* Find all uses of each register marked above */
  for (i = FIRST_PSEUDO_REGISTER; i < cur_max_regnum; i++)
    {
      if (sign_extension_infos[i].insn_set)
        {
          for (insn = loop_start; insn != loop_end; insn = NEXT_INSN (insn))
            {
              if (GET_CODE (insn) == INSN || GET_CODE (insn) == CALL_INSN
                  || GET_CODE (insn) == JUMP_INSN)
                validate_all_references (i, insn, PATTERN (insn));
            }
        }
    }
  for (i = FIRST_PSEUDO_REGISTER; i < cur_max_regnum; i++)
    {
      if (sign_extension_infos[i].insn_set)
        {
          sign_ext_reg = SET_DEST (PATTERN (sign_extension_infos[i].insn_set));
          if (reg_used_between_p (sign_ext_reg,
                                  get_insns (), loop_start) ||
              reg_used_between_p (sign_ext_reg,
                                  loop_end, NULL))
            sign_extension_infos[i].insn_set = NULL;
        }
    }
  /* Check which registers can be sign extension reduced */
  for (i = FIRST_PSEUDO_REGISTER; i < cur_max_regnum; i++)
    {
      if (sign_extension_infos[i].insn_set)
        {
          j = 0;
          while (j < sign_extension_infos[i].num_comp_regs
                 && sign_extension_infos[i].insn_set)
            {
              regno = sign_extension_infos[i].compare_to_reg[j].regno;
              if (regno < FIRST_PSEUDO_REGISTER)
                sign_extension_infos[i].insn_set = NULL;

              if (sign_extension_infos[regno].insn_set == NULL)
                sign_extension_infos[i].insn_set = NULL;

              j++;
            }
        }
    }
  /* Do the optimization.  */
  for (i = FIRST_PSEUDO_REGISTER; i < cur_max_regnum; i++)
    {
      if (sign_extension_infos[i].insn_set)
        {
          rtx subreg, strict_subreg;

          subreg 
            = gen_rtx (SUBREG, HImode, 
              SET_DEST (PATTERN (sign_extension_infos[i].insn_set)),
              0);
          strict_subreg = gen_rtx (STRICT_LOW_PART, 0, subreg);
          emit_insn_before (gen_rtx (SET, 0, 
             SET_DEST (PATTERN (sign_extension_infos[i].insn_set)),
             const0_rtx), loop_start);
          emit_insn_before (gen_rtx (USE, 0, 
             SET_DEST (PATTERN (sign_extension_infos[i].insn_set))),
             loop_start);
          SET_DEST (PATTERN (sign_extension_infos[i].insn_set)) 
            = strict_subreg;
          SET_SRC (PATTERN (sign_extension_infos[i].insn_set)) 
            = XEXP (SET_SRC (PATTERN (sign_extension_infos[i].insn_set)), 0);
          INSN_CODE (sign_extension_infos[i].insn_set) = -1;
          j = 0;
          while (j < sign_extension_infos[i].num_comp_regs)
            {
              if (sign_extension_infos[i].compare_to_reg[j].for_equality == 0)
                {
                  *sign_extension_infos[i].compare_to_reg[j].where
                    = subreg;
                  INSN_CODE (sign_extension_infos[i].compare_to_reg[j].comp_insn)
                    = -1;
                }
              j++;
            }
        }
    }
}


static void
validate_all_references (regno, insn, pat)
  int regno;
  rtx insn, pat;
{
  register RTX_CODE code = GET_CODE (pat);
  register int i, j, val;
  register char *fmt;
  int num_comp_regs, comp_regno, for_equality = 0;
  rtx next;

  switch (code)
    {
    case HIGH:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
      return ;

    case REG:
      if (REGNO (pat) == regno)
        sign_extension_infos[regno].insn_set = NULL;

      break;
      
    case SET:
      if (GET_CODE (SET_DEST (pat)) == REG 
          && REGNO (SET_DEST (pat)) == regno
          && sign_extension_infos[REGNO (SET_DEST (pat))].insn_set != insn)
        {
          if (GET_CODE (SET_SRC (pat)) != CONST_INT)
            sign_extension_infos[regno].insn_set = NULL;

          if (GET_MODE (SET_DEST (pat)) != SImode)
            sign_extension_infos[regno].insn_set = NULL;

          val = XINT (SET_SRC (pat), 0);
          if (val < 0 || val > largest_himode_const)
            sign_extension_infos[regno].insn_set = NULL;

          validate_all_references (regno, insn, SET_SRC (pat));
          return;
        }
      else if (GET_CODE (SET_DEST (pat)) == REG 
          && REGNO (SET_DEST (pat)) == regno
          && sign_extension_infos[REGNO (SET_DEST (pat))].insn_set == insn)
        return;
      break;

    case AND:
      if (GET_CODE (XEXP (pat, 0)) == REG 
          && REGNO (XEXP (pat, 0)) == regno)
        {
          if (GET_CODE (XEXP (pat, 1)) != CONST_INT)
            sign_extension_infos[regno].insn_set = NULL;

          if (GET_MODE (XEXP (pat, 0)) != SImode)
            sign_extension_infos[regno].insn_set = NULL;

          val = XINT (XEXP (pat, 1), 0);
          if (val < 0 || val > largest_himode_const)
            sign_extension_infos[regno].insn_set = NULL;

          validate_all_references (regno, insn, XEXP (pat, 1));
          return;
        }
      else if (GET_CODE (XEXP (pat, 1)) == REG 
          && REGNO (XEXP (pat, 1)) == regno)
        {
          if (GET_CODE (XEXP (pat, 0)) != CONST_INT)
            sign_extension_infos[regno].insn_set = NULL;

          if (GET_MODE (XEXP (pat, 1)) != SImode)
            sign_extension_infos[regno].insn_set = NULL;

          val = XINT (XEXP (pat, 0), 0);
          if (val < 0 || val > largest_himode_const)
            sign_extension_infos[regno].insn_set = NULL;

          validate_all_references (regno, insn, XEXP (pat, 0));
          return;
        }
      break;

    case COMPARE:
      next = NEXT_INSN (insn);
      if (GET_CODE (next) == JUMP_INSN 
	  && GET_CODE (PATTERN (next)) == SET 
	  && SET_DEST (PATTERN (next)) == pc_rtx 
	  && GET_CODE (SET_SRC (PATTERN (next))) == IF_THEN_ELSE
	  && (GET_CODE (XEXP (SET_SRC (PATTERN (next)), 0)) == NE
	      ||
	      GET_CODE (XEXP (SET_SRC (PATTERN (next)), 0)) == EQ))
	{
	  for_equality = 1;
	}
      if (GET_CODE (XEXP (pat, 0)) == REG 
          && REGNO (XEXP (pat, 0)) == regno)
        {
          if (GET_MODE (XEXP (pat, 0)) != SImode)
            sign_extension_infos[regno].insn_set = NULL;

          if (GET_CODE (XEXP (pat, 1)) == CONST_INT)
            {
              val = XINT (XEXP (pat, 1), 0);
              if (for_equality == 0 || val < 0 || val > largest_himode_const)
                sign_extension_infos[regno].insn_set = NULL;
            }
          else if (GET_CODE (XEXP (pat, 1)) == REG)
            {
              if (GET_MODE (XEXP (pat, 1)) != SImode)
                sign_extension_infos[regno].insn_set = NULL;

#ifndef HAVE_cmphi
              sign_extension_infos[regno].insn_set = NULL;
#else
              if (!HAVE_cmphi)
                sign_extension_infos[regno].insn_set = NULL;
#endif
              comp_regno = REGNO (XEXP (pat, 1));
              if (!sign_extension_infos[comp_regno].insn_set)
                sign_extension_infos[regno].insn_set = NULL;
              else
                {
                  num_comp_regs 
                    = sign_extension_infos[regno].num_comp_regs;
                  if (num_comp_regs >= MAX_COMP_REGS)
                    sign_extension_infos[regno].insn_set = NULL;
                  else
                    {
                      sign_extension_infos[regno].
                          compare_to_reg[num_comp_regs].where = 
                        &XEXP (pat, 0);
                      sign_extension_infos[regno].
                          compare_to_reg[num_comp_regs].comp_insn = insn;
                      sign_extension_infos[regno].
                          compare_to_reg[num_comp_regs].regno = 
                        comp_regno;
                      sign_extension_infos[regno].
    	                compare_to_reg[num_comp_regs].for_equality = for_equality;
                      sign_extension_infos[regno].num_comp_regs++;
                    }
                }
            }
          else
            {
              sign_extension_infos[regno].insn_set = NULL;
              validate_all_references (regno, insn, XEXP (pat, 1));
            }
          return;
        }
      else if (GET_CODE (XEXP (pat, 1)) == REG 
          && REGNO (XEXP (pat, 1)) == regno)
        {
          if (GET_MODE (XEXP (pat, 1)) != SImode)
            sign_extension_infos[regno].insn_set = NULL;
          if (GET_CODE (XEXP (pat, 0)) == CONST_INT)
            {
              val = XINT (XEXP (pat, 0), 0);
              if (for_equality == 0 || val < 0 || val > largest_himode_const)
                sign_extension_infos[regno].insn_set = NULL;
            }
          else if (GET_CODE (XEXP (pat, 0)) == REG)
            {
              if (GET_MODE (XEXP (pat, 0)) != SImode)
                sign_extension_infos[regno].insn_set = NULL;
#ifndef HAVE_cmphi
              sign_extension_infos[regno].insn_set = NULL;
#else
              if (!HAVE_cmphi)
                sign_extension_infos[regno].insn_set = NULL;
#endif
              comp_regno = REGNO (XEXP (pat, 0));
              if (!sign_extension_infos[comp_regno].insn_set)
                sign_extension_infos[regno].insn_set = NULL;
              else
                {
                  num_comp_regs 
                    = sign_extension_infos[regno].num_comp_regs;
                  if (num_comp_regs >= MAX_COMP_REGS)
                    sign_extension_infos[regno].insn_set = NULL;
                  else
                    {
                      sign_extension_infos[regno].
                          compare_to_reg[num_comp_regs].where = 
                        &XEXP (pat, 1);
                      sign_extension_infos[regno].
                          compare_to_reg[num_comp_regs].comp_insn = insn;
                      sign_extension_infos[regno].
                          compare_to_reg[num_comp_regs].regno = 
                        comp_regno;
    	              sign_extension_infos[regno].
    	                 compare_to_reg[num_comp_regs].for_equality = for_equality;
                      sign_extension_infos[regno].num_comp_regs++;
                    }
                }
            }
          else
            {
              sign_extension_infos[regno].insn_set = NULL;
              validate_all_references (regno, insn, XEXP (pat, 0));
            }
          return;
        }
      break;

    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      {
        if( XEXP (pat, i))
          validate_all_references (regno, insn, XEXP (pat, i));
      }
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      {
	for (j = 0; j < XVECLEN (pat, i); j++)
          validate_all_references (regno, insn, XVECEXP (pat, i, j));
      }
}

#define MAX_WHERE 6
#define MAX_DISAMBIGS 5
#define MAX_MEM_REFS 6
#define NO_REG -1
#define UNKNOWN -2

static struct mem_ref {
  /*  Pointers to where this mem_ref is used.  */
  rtx * where[MAX_WHERE];
  /* Number of references to this mem_ref in the loop.  */
  int num_ref;
  /* The base register number if any, else -1.  */
  int base_regno;
  /* The base register number if.  */
  rtx base_reg;
  /* The index register number if any, else -1.  */
  int index_regno;
  /* The index register number if.  */
  rtx index_reg;
  /* Bit-map if bit# i is set then this element is true_dependence on
     element# i. Assumes MAX_MEM_REFS < (sizeof (int) * 8) */
  int conflict;
  /* If 1 then this element is a varying address over the loop - or
     should not be lifted into register for some other reason.  */
  unsigned int varying:1;
  /* If 1 then this mem_ref is stored into in the loop.  */
  unsigned int stored:1;
  /* If 1 then the first reference to the mem_ref in the loop is a store.  */
  unsigned int first_stored:1;
  } *mem_refs;

static struct disambig{
  rtx reg;
  int inc_val;
}	runtime_disambigs [MAX_DISAMBIGS];

static int num_mem_refs = 0;
static int max_regs_before_lift;

extern rtx gen_jump ();
/*
  Shadow memory locations that are stored into in the loop and can
  be memory disambiguated over the loop. These are loaded into a register
  before the loop, accessed from and into the register during the loop
  and stored from the register at the loop end.  
*/
static void
lift_stores (loop_start, loop_end)
  rtx loop_start, loop_end;
{
#ifdef LIFT_STORE_OK
  rtx insn, lift_reg, set, new_label, new_label_insn, new_jump_insn, new_jump,
      new_loop_label, new_loop_jump, new_compare, new_compare_insn, copy_insn;
  int i, j, num_candidates, jmask, imask, num_lifted, num_possible,
      indx_possible, num_runtime_disambigs, disambig_mask, found, dummy;
  char *may_not_optimize;

  if (loop_dump_stream)
    {
      fprintf (loop_dump_stream, "lift stores from %d to %d\n",
      INSN_UID (loop_start), INSN_UID (loop_end));
    }

  /* Handle only loops with no inner jumps labels or calls */
  if (GET_CODE (PREV_INSN (loop_end)) != JUMP_INSN)
    return;

  if (GET_CODE (SET_SRC (PATTERN (PREV_INSN (loop_end)))) != IF_THEN_ELSE)
    return;

  if (GET_CODE (NEXT_INSN (loop_start)) != CODE_LABEL)
    return;

  for (insn = NEXT_INSN (NEXT_INSN (loop_start)); 
       insn != PREV_INSN (loop_end); insn = NEXT_INSN (insn))
    if (GET_CODE (insn) == JUMP_INSN || GET_CODE (insn) == CALL_INSN
        || GET_CODE (insn) == CODE_LABEL)
       return;

  /* Compute n_times_set */
  max_regs_before_lift = max_reg_num () + 1;
  n_times_set = (short *) alloca (max_regs_before_lift * sizeof (short));
  bzero (n_times_set, max_regs_before_lift * sizeof (short));
  may_not_optimize = (char *) alloca (max_regs_before_lift);	/* not used */
  count_loop_regs_set (loop_start, loop_end, may_not_optimize, NULL,
		       &dummy, max_regs_before_lift);

  mem_refs = (struct mem_ref *) alloca (sizeof (struct mem_ref) * MAX_MEM_REFS);
  bzero (mem_refs, sizeof (struct mem_ref) * MAX_MEM_REFS);
  /* Locate all memory refs into the mem_refs array.  */
  num_mem_refs = 0;
  for (insn = loop_start; insn != loop_end; insn = NEXT_INSN (insn))
    if (GET_CODE (insn) == INSN)
      {
        mark_mem_refs (PATTERN (insn), &PATTERN (insn));
        if (num_mem_refs >= MAX_MEM_REFS)
          return;
      }

  if (num_mem_refs == 0)
    return;

  /* Check each non varying mem_ref against all others to see if it can
     be memory disambiguated */
  num_candidates = 0;
  for (i = 0; i < num_mem_refs; i++)
    {
      if (mem_refs[i].varying == 0 && mem_refs[i].stored == 1)
        {
          num_candidates++;
          imask = 1 << i;
          for (j = 0; j < num_mem_refs; j++)
            {
              jmask = (1 << j);
              if (j != i && !(mem_refs[i].conflict & jmask))
                {
                  if (true_dependence (*(mem_refs[i].where[0]),
                                       *(mem_refs[j].where[0])))
                    {
                      mem_refs[i].conflict |= jmask;
                      mem_refs[j].conflict |= imask;
                    }
                }
            }
        }
    }
  /* Lift all the mem_refs that can be lifted.  */
  num_lifted = 0;
  num_possible = 0;
  for (i = 0; i < num_mem_refs; i++)
    {
      if (mem_refs[i].varying == 0 && mem_refs[i].stored == 1)
        {
          num_possible++;
          indx_possible = i;
        }
      if (mem_refs[i].varying == 0 && mem_refs[i].stored == 1
          && mem_refs[i].conflict == 0
          && LIFT_STORE_OK (*(mem_refs[i].where[0]))) /* intel3 dont lift exotic modes */
        {
          num_lifted++;
          lift_reg = gen_reg_rtx (GET_MODE (*(mem_refs[i].where[0])));
          if (mem_refs[i].first_stored)
            emit_insn_before ( gen_rtx (SET, 0, lift_reg, 
                                        CONST0_RTX ( GET_MODE (lift_reg))),
                               loop_start);
          else
            emit_insn_before ( gen_rtx (SET, 0, lift_reg, 
                                        copy_rtx (*(mem_refs[i].where[0]))),
                               loop_start);

          emit_insn_after ( gen_rtx (SET, 0, copy_rtx (*(mem_refs[i].where[0])),
                                              lift_reg),
                                 loop_end);
          /*emit_insn_after (gen_rtx (USE, 0, lift_reg),
                          NEXT_INSN (loop_end));*/
          for (j = 0; j < mem_refs[i].num_ref; j++)
            *(mem_refs[i].where[j]) = lift_reg;
          
          if (loop_dump_stream)
            {
              fprintf (loop_dump_stream, 
              "The following memory operand lifted into reg %d during loop:\n",
              REGNO (lift_reg));
              tdebug_rtx (SET_DEST (PATTERN (NEXT_INSN (loop_end))),
                          loop_dump_stream);
              fprintf (loop_dump_stream, "\n");
            }
        }
    }
  if (num_lifted == 0 && num_possible == 1 && flag_runtime_lift_stores)
    {  /* Do runtime disambiguation if possible */
      if (mem_refs[indx_possible].index_regno != NO_REG
          || mem_refs[indx_possible].base_regno == NO_REG
          || mem_refs[indx_possible].base_regno == UNKNOWN
          || GET_MODE (mem_refs[indx_possible].base_reg) != SImode)
        return;
      
      num_runtime_disambigs = 0;
      for (i = 0; i < num_mem_refs; i++)
        {
          disambig_mask = 1 << i;
          if (mem_refs[indx_possible].conflict & disambig_mask)
            {
              if (num_runtime_disambigs >= MAX_DISAMBIGS)
                return;

              if (mem_refs[i].base_regno == NO_REG 
                  || mem_refs[i].base_regno == UNKNOWN
                  || (mem_refs[i].index_regno != UNKNOWN
                      && mem_refs[i].index_regno != NO_REG))
                return;

              if (n_times_set[mem_refs[i].base_regno] > 1)
                return;

              if (GET_MODE (mem_refs[i].base_reg) 
                   != GET_MODE (mem_refs[indx_possible].base_reg))
                return;
              
              runtime_disambigs[num_runtime_disambigs].reg = mem_refs[i].base_reg;
              runtime_disambigs[num_runtime_disambigs].inc_val = 0;
              num_runtime_disambigs++;
            }
        }
      /* Now find the setting of each of the registers of the disambigs */
      for (i = 0; i < num_runtime_disambigs; i++)
        {
          if (n_times_set[REGNO (runtime_disambigs[i].reg)])
            {
              insn = loop_start;
              found = 0;
              while (insn != loop_end && !found)
                {
                  if (GET_CODE (insn) == INSN)
                    {
                      set = single_set (insn);
                      if (set && GET_CODE (SET_DEST (set)) == REG
                          && REGNO (SET_DEST (set)) 
                              == REGNO (runtime_disambigs[i].reg) )
                        {
                          found = 1;
                          if (!rtx_equal_p (SET_DEST (set),
                                            runtime_disambigs[i].reg))
                            return;
                          if ((GET_CODE (SET_SRC (set)) == PLUS
                                || GET_CODE (SET_SRC (set)) == MINUS)
                              && rtx_equal_p (SET_DEST (set), 
                                              XEXP (SET_SRC (set), 0))
                              && GET_CODE (XEXP (SET_SRC (set), 1)) == CONST_INT)
                            {
                              runtime_disambigs[i].inc_val 
                                = XINT (XEXP (SET_SRC (set), 1), 0) 
                                   * (GET_CODE (SET_SRC (set)) == PLUS
                                      ? 1 : -1);
                            }
                          else
                            return;
                        }
                    }
                  if (!found)
                    insn = NEXT_INSN (insn);
                }
              if (!found)
                return;
            }
        }
      
      /* intel3 dont lift BLKmode and other exotic modes */
      if (!LIFT_STORE_OK(*(mem_refs[indx_possible].where[0])))
        return;

      /* The runtime disambiguation is possible - now do it.  */
      /* Make a copy of the loop.  */
      new_label = gen_label_rtx ();
      new_label_insn 
        = emit_label_after (new_label, PREV_INSN (loop_start));
      new_loop_label = new_label_insn;
      ++LABEL_NUSES (new_label_insn);
      new_jump = copy_rtx (PATTERN (PREV_INSN (loop_end)));
      if (XEXP (SET_SRC (new_jump), 1) == pc_rtx)
        XEXP (SET_SRC (new_jump), 2)
          = gen_rtx (LABEL_REF, VOIDmode, new_label);
      else
        XEXP (SET_SRC (new_jump), 1)
          = gen_rtx (LABEL_REF, VOIDmode, new_label);
      
      new_jump_insn = emit_jump_insn_after (new_jump, new_label_insn);
      INSN_CODE (new_jump_insn) = recog (new_jump, new_jump_insn);
      new_loop_jump = new_jump_insn;
      JUMP_LABEL (new_jump_insn) = new_label_insn;
      /* The body of the original loop is moved to the new loop */
      NEXT_INSN (new_label_insn) = NEXT_INSN (NEXT_INSN (loop_start));
      PREV_INSN (NEXT_INSN (new_label_insn)) = new_label_insn;
      PREV_INSN (new_jump_insn) = PREV_INSN (PREV_INSN (loop_end));
      NEXT_INSN ( PREV_INSN (new_jump_insn)) = new_jump_insn;
      NEXT_INSN (NEXT_INSN (loop_start)) = PREV_INSN (loop_end);
      PREV_INSN (PREV_INSN (loop_end)) = NEXT_INSN (loop_start);
      /* Now make a copy of the loop in between the original label and jump.  */
      insn = NEXT_INSN (new_label_insn);
      copy_insn = NEXT_INSN (loop_start);
      while (insn != new_jump_insn)
        {
          if (GET_CODE (insn) == INSN)
            copy_insn = emit_insn_after (copy_rtx (PATTERN (insn)), copy_insn);
          insn = NEXT_INSN (insn);
        }
      /* Now make a jump around the second copy of the loop.  */
      new_label = gen_label_rtx ();
      new_label_insn = emit_label_after (new_label, loop_end);
      new_jump_insn = emit_jump_insn_after (gen_jump (new_label_insn),
                                            PREV_INSN (loop_start));
      JUMP_LABEL (new_jump_insn) = new_label_insn;
      ++LABEL_NUSES (new_label_insn);
      emit_barrier_after (new_jump_insn);
      /* Now make a label just before loop start */
      new_label = gen_label_rtx ();
      new_label_insn = emit_label_after (new_label, PREV_INSN (loop_start));
      ++LABEL_NUSES (new_label_insn);
      /* Now compare the base_reg of the mem_ref to be lifted with
         each of the base_regs that it conflicts with.  Jump around the
         disambiguated loop if they conflict at runtime.  */
      for (i = 0; i < num_runtime_disambigs; i++)
        {
          if (runtime_disambigs[i].inc_val != 0)
            {
              new_compare 
                = gen_cmpsi_1 (runtime_disambigs[i].reg,
                                mem_refs[indx_possible].base_reg);
              new_compare_insn 
                = emit_insn_before (new_compare, new_loop_label);
              new_jump = copy_rtx (new_jump);
              if (runtime_disambigs[i].inc_val > 0)
                GET_CODE (XEXP (SET_SRC (new_jump), 0)) = LE;
              else
                GET_CODE (XEXP (SET_SRC (new_jump), 0)) = GE;

	      XEXP (SET_SRC (new_jump), 1) 
	        = gen_rtx (LABEL_REF, VOIDmode, new_label);
	      XEXP (SET_SRC (new_jump), 2) = pc_rtx;
              ++LABEL_NUSES (new_label_insn);
              new_jump_insn = emit_jump_insn_after (new_jump, new_compare_insn);
              JUMP_LABEL (new_jump_insn) = new_label_insn;
              INSN_CODE (new_jump_insn) = recog (new_jump, new_jump_insn);
            }
        }
      /* Now do the lifting */
      lift_reg = gen_reg_rtx (GET_MODE (*(mem_refs[indx_possible].where[0])));
      if (mem_refs[indx_possible].first_stored)
        emit_insn_before ( gen_rtx (SET, 0, lift_reg, 
                                    CONST0_RTX ( GET_MODE (lift_reg))),
                           new_loop_label);
      else
        emit_insn_before ( gen_rtx (SET, 0, lift_reg, 
                                    copy_rtx (*(mem_refs[indx_possible].where[0]))),
                           new_loop_label);

      emit_insn_after ( gen_rtx (SET, 0, 
                                 copy_rtx (*(mem_refs[indx_possible].where[0])),
                                 lift_reg),
                        new_loop_jump);
      /*emit_insn_after (gen_rtx (USE, 0, lift_reg),
                      NEXT_INSN (new_loop_jump));*/
      emit_insn_after (gen_rtx (CLOBBER, 0, lift_reg), loop_end);
      for (j = 0; j < mem_refs[indx_possible].num_ref; j++)
        *(mem_refs[indx_possible].where[j]) = lift_reg;

      emit_note_before (NOTE_INSN_LOOP_BEG, new_loop_label);
      emit_note_before (NOTE_INSN_LOOP_END, NEXT_INSN (new_loop_jump));
      if (loop_dump_stream)
        {
          fprintf (loop_dump_stream, 
          "The following memory operand RUNTIME lifted into reg %d during loop:\n",
          REGNO (lift_reg));
          tdebug_rtx (SET_DEST (PATTERN (NEXT_INSN (NEXT_INSN (new_loop_jump)))),
                      loop_dump_stream);
          fprintf (loop_dump_stream, "\n");
        }
    }
#endif
}


/* 
   Go over PAT locate all mem_refs and insert them into the mem_refs
   structure.  WHERE is the location of PAT.  */
static void 
mark_mem_refs (pat, where)
     rtx pat;
     rtx *where;
{
  register RTX_CODE code = GET_CODE (pat);
  register int i, j;
  register char *fmt;

  switch (code)
    {
    case HIGH:
    case CONST_INT:
    case CONST:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case LABEL_REF:
    case PC:
    case CC0:
      return;

    case SET:
      mark_mem_refs (SET_SRC (pat), &SET_SRC (pat));
      if (GET_CODE (SET_DEST (pat)) == MEM)
        insert_mem_ref (SET_DEST (pat), &SET_DEST (pat), 1);

      return;

     case MEM:
       insert_mem_ref (pat, where, 0);
       return;

    }

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    if (fmt[i] == 'e')
      {
        if( XEXP (pat, i))
          mark_mem_refs (XEXP (pat, i), &XEXP (pat, i));
      }
    else if (fmt[i] == 'E' || fmt[i] == 'V')
      for (j = 0; j < XVECLEN (pat, i); j++)
	mark_mem_refs (XVECEXP (pat, i, j), &XVECEXP (pat, i, j));
}


/*
   Insert the memory reference MEM into the mem_refs structure.  
   WHERE is location of MEM.  
   if IS_STORE is 1 then MEM is being stored into.  */
   
static void
insert_mem_ref (mem, where, is_store)
  rtx mem, *where;
  int is_store;
{
  int i, j, indx, found, base_regno, index_regno;
  rtx mem_addr, base_reg, index_reg;

  if (num_mem_refs >= MAX_MEM_REFS)
    return;
  
  /* Find where in table to put mem.  */
  i = 0;
  found = 0;
  while (i < num_mem_refs && !found)
    {
      if (rtx_equal_p (mem, *(mem_refs[i].where[0])))
        {
          found = 1;
        }
      else
        i++;
    }

  if (!found)
    num_mem_refs++;
  
  if (i >= MAX_MEM_REFS)
    return;

  indx = i;
  j = mem_refs[indx].num_ref;
  if (j >= MAX_WHERE)
    {
      mem_refs[indx].varying = 1;
      return;
    }
  mem_refs[indx].where[j] = where;
  mem_refs[indx].num_ref++;
  if (is_store)
    {
      mem_refs[indx].stored = 1;
      if (j == 0)
        mem_refs[indx].first_stored = 1;
    }

  if (found == 1)
    return;

  mem_refs[indx].varying = rtx_addr_varies_p (mem);
  /* Now analyze the mem to find it's base and index registers if any */
  mem_refs[indx].base_regno = NO_REG;
  mem_refs[indx].index_regno = NO_REG;
  index_reg = NULL;
  base_regno = NO_REG;
  index_regno = NO_REG;
  base_reg = NULL;
  mem_addr = XEXP (mem, 0);
  if (GET_CODE (mem_addr) == SYMBOL_REF)
    return;

  if (GET_CODE (mem_addr) == REG)
    {
      base_regno = REGNO (mem_addr);
      base_reg = (mem_addr);
    }

  if (GET_CODE (mem_addr) == PLUS)
    {
      if (GET_CODE (XEXP (mem_addr, 0)) == REG)
	{
	  base_regno = REGNO (XEXP (mem_addr, 0));
	  base_reg = (XEXP (mem_addr, 0));
	  if (GET_CODE (XEXP (mem_addr, 1)) == REG)
	    {
	      index_regno = REGNO (XEXP (mem_addr, 1));
	      index_reg = (XEXP (mem_addr, 1));
	    }
	  else if (GET_CODE (XEXP (mem_addr, 1)) != CONST_INT 
	           && GET_CODE (XEXP (mem_addr, 1)) != CONST
	           && GET_CODE (XEXP (mem_addr, 1)) != SYMBOL_REF)
	    index_regno = UNKNOWN;
	}
      else if (GET_CODE (XEXP (mem_addr, 0)) == MULT)
	{
	  if (GET_CODE (XEXP (mem_addr, 1)) == REG)
	    {
	      base_regno = REGNO (XEXP (mem_addr, 1));
	      base_reg =  (XEXP (mem_addr, 1));
	    }
	  else if (GET_CODE (XEXP (mem_addr, 1)) != CONST_INT 
	           && GET_CODE (XEXP (mem_addr, 1)) != CONST
	           && GET_CODE (XEXP (mem_addr, 1)) != SYMBOL_REF)
	    base_regno = UNKNOWN;
	  if (GET_CODE (XEXP (XEXP (mem_addr, 0), 0)) == REG
	      && GET_CODE (XEXP (XEXP (mem_addr, 0), 1)) == CONST_INT)
	   {
	     index_regno = REGNO (XEXP (XEXP (mem_addr, 0), 0));
	     index_reg = (XEXP (XEXP (mem_addr, 0), 0));
	   }
	 else
	   index_regno = UNKNOWN;
	}
      else if (GET_CODE (XEXP (mem_addr, 0)) == PLUS)
	{
          if (GET_CODE (XEXP (XEXP (mem_addr, 0), 0)) == REG)
	    {
	      base_regno = REGNO (XEXP (XEXP (mem_addr, 0), 0));
	      base_reg = (XEXP (XEXP (mem_addr, 0), 0));
	      if (GET_CODE (XEXP (XEXP (mem_addr, 0), 1)) == REG)
	        {
	          index_regno = REGNO (XEXP (XEXP (mem_addr, 0), 1));
	          index_reg = (XEXP (XEXP (mem_addr, 0), 1));
	        }
      	      else if (GET_CODE (XEXP (XEXP (mem_addr, 0), 1)) != CONST_INT 
	               && GET_CODE (XEXP (XEXP (mem_addr, 0), 1)) != CONST
	               && GET_CODE (XEXP (XEXP (mem_addr, 0), 1)) != SYMBOL_REF)
	        index_regno = UNKNOWN;
	    }
          else if (GET_CODE (XEXP (XEXP (mem_addr, 0), 0)) == MULT)
	    {
	      if (GET_CODE (XEXP (XEXP (mem_addr, 0), 1)) == REG)
	        {
	          base_regno = REGNO (XEXP (XEXP (mem_addr, 0), 1));
	          base_reg = (XEXP (XEXP (mem_addr, 0), 1));
	        }
	      if (GET_CODE (XEXP (XEXP (XEXP (mem_addr, 0), 0), 0)) == REG
	          && GET_CODE (XEXP (XEXP (XEXP (mem_addr, 0), 0), 1)) == CONST_INT)
	       {
	         index_regno = REGNO (XEXP (XEXP (XEXP (mem_addr, 0), 0), 0));
	         index_reg = (XEXP (XEXP (XEXP (mem_addr, 0), 0), 0));
	       }
	     else
	       index_regno = UNKNOWN;
	    }
	  if (GET_CODE (XEXP (mem_addr, 1)) != SYMBOL_REF
	      && GET_CODE (XEXP (mem_addr, 1)) != CONST
	      && GET_CODE (XEXP (mem_addr, 1)) != CONST_INT)
	    {
	      index_regno = UNKNOWN;
	      base_regno = UNKNOWN;
	    }
	}
    }
  else if (GET_CODE (mem_addr) == MULT)
    {
      if (GET_CODE (XEXP (mem_addr, 0)) == REG 
          && GET_CODE (XEXP (mem_addr, 1)) == CONST_INT)
        {
          index_regno = REGNO (XEXP (mem_addr, 0));
          index_reg = (XEXP (mem_addr, 0));
        }
      else
        index_regno = UNKNOWN;
    }
  mem_refs[indx].base_reg = base_reg;
  mem_refs[indx].base_regno = base_regno;
  mem_refs[indx].index_reg = index_reg;
  mem_refs[indx].index_regno = index_regno;
  if (base_regno == UNKNOWN || index_regno == UNKNOWN)
    {
      mem_refs[indx].varying = 1;
      return;
    }
  if (base_regno >= max_regs_before_lift || index_regno >= max_regs_before_lift)
    {
      mem_refs[indx].varying = 1;
      return;
    }
  if ( (base_regno == NO_REG || n_times_set[base_regno] == 0)
      && (index_regno == NO_REG || n_times_set[index_regno] == 0))
    mem_refs[indx].varying = 0;

  if (MEM_VOLATILE_P (mem))
    mem_refs[indx].varying = 1;
}
 

static void opt_reg_use_copy (f)
  rtx f;
{
  rtx insn, src, prev;
  int i, length;
  char * fmt;
  for (insn = f; insn; insn = NEXT_INSN (insn))
    {
      if (GET_CODE (insn) == INSN)
        {
  	  if (GET_CODE (PATTERN (insn)) == SET
	      && GET_CODE (SET_DEST (PATTERN (insn))) == REG
	      && GET_CODE (SET_SRC (PATTERN (insn))) == REG
	      && (prev = PREV_INSN (insn))
	      && GET_CODE (prev) == INSN
	      && GET_CODE (PATTERN (prev)) == SET
	      && rtx_equal_p (SET_SRC (PATTERN (insn)), 
	                      SET_DEST (PATTERN (prev))))
	    {
              src = SET_SRC (PATTERN (prev));
              fmt = GET_RTX_FORMAT (GET_CODE (src));
              length = GET_RTX_LENGTH (GET_CODE (src));
              for (i=0; i<length; i++)
                {
                  if (fmt[i] == 'e' 
                      && GET_CODE (XEXP (src, i)) == REG
                      && rtx_equal_p (XEXP (src, i), SET_DEST (PATTERN (insn))))
                    {  
                      SET_DEST (PATTERN (insn)) = SET_SRC (PATTERN (insn));
                      SET_SRC (PATTERN (insn)) = XEXP (src, i);
                      SET_DEST (PATTERN (prev)) = XEXP (src, i);
                      emit_insn_after (gen_rtx (USE, 0, 
                                                SET_DEST (PATTERN (prev))),
                                       prev);
                      break;
                    }
                }
	    }
	}
    }
}
