/*
** main.c - Main entrypoint for Pidentd 3.0
**
** Copyright (c) 1997 Peter Eriksson <pen@lysator.liu.se>
**
** This program is free software; you can redistribute it and/or
** modify it as you wish - as long as you don't claim that you wrote
** it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <signal.h>

#include "pidentd.h"

#if !defined(HAVE_LIBTHREAD) && defined(HAVE_THR_SETCONCURRENCY)
#  include <thread.h>
#endif

extern char *optarg;
extern int optind;


int debug = 0;
int server_uid = -1;
int server_gid = 0;
char *pidfile_path = PATH_PIDFILE;

#ifdef HAVE_LIBDES
int encrypt_flag = 0;
char *encrypt_keyfile = PATH_KEYFILE;
#endif

pthread_attr_t cattr_detached;


static void
usage(FILE *fp)
{
    fputs("Usage: pidentd [options]\n", fp);
    fputs("Options:\n", fp);
    fputs("  -d           Enable debug mode\n", fp);
    fputs("  -h           Print this information\n", fp);
    fputs("  -V           Print version and OS information\n", fp);
    
    fputs("  -w           Start in Inetd 'wait' mode\n", fp);
    fputs("  -i           Start in Inetd 'nowait' mode\n", fp);
    fputs("  -I           Start in Init mode\n", fp);
    fputs("  -b           Start in Standalone mode\n", fp);
    
    fputs("  -e           Enable protocol extensions\n", fp);
    fputs("  -m           Enable multiquery mode\n", fp);
    
#ifdef HAVE_LIBDES
    fputs("  -E           Enable DES encrypted replies\n", fp);
#endif
    
    fputs("  -n           Send uid numbers instead of usernames\n", fp);
    fputs("  -o           Return OTHER instead of UNIX\n", fp);
    fputs("  -N           Check for .noident files\n", fp);
    
    fputs("  -t<time>     Request timeout limit in seconds\n", fp);
    fputs("  -p<port>     Port to listen for connections on\n", fp);
    fputs("  -g<group>    Group name/number to run as\n", fp);
    fputs("  -u<user>     User name/number to run as\n", fp);
    fputs("  -C<file>     Config file to include\n", fp);
    fputs("  -P<file>     Where to write the process id number\n", fp);
    fputs("  -K<threads>  Number of kernel lookup threads\n", fp);
}



int
main(int argc, char *argv[])
{
    int c, socket_type = -1;
    int init_mode = 0;

    
#ifdef SIGTTOU    
    signal(SIGTTOU, SIG_IGN);
#endif
    
    openlog("pidentd", LOG_PID|LOG_ODELAY, LOG_DAEMON);
    
    conf_parse(PATH_CFGFILE, 1);
    
    while ((c = getopt(argc, argv, "NVEdhbwiIemnp:u:g:t:C:P:K:")) != -1)
	switch (c)
	{
	  case 'E':
#ifdef HAVE_LIBDES
	    encrypt_flag = 1;
	    break;
#else
	    usage(stderr);
	    exit(1);
#endif
	    
	  case 'n':
	    uidonly_flag = 1;
	    break;
	    
	  case 'd':
	    debug = 1;
	    break;
	    
	  case 'h':
	    usage(stdout);
	    exit(0);
	    
	  case 'e':
	    extensions_enabled = 1;
	    break;
	    
	  case 'm':
	    multiquery_enabled = 1;
	    break;
	    
	  case 'w':
	    listen_sock = 0;
	    socket_type = SOCKTYPE_LISTEN;
	    break;
	    
	  case 'i':
	    listen_sock = 0;
	    socket_type = SOCKTYPE_CONNECTED;
	    break;
	    
	  case 'I':
	    listen_sock = -1;
	    socket_type = SOCKTYPE_NOTSOCKET;
	    init_mode = 1;
	    break;
	    
	  case 'b':
	    listen_sock = -1;
	    socket_type = SOCKTYPE_NOTSOCKET;
	    break;
	    
	  case 'p':
	    if (str2port(optarg, &listen_port) < 0)
		syslog(LOG_ERR, "invalid argument to '-p': %s", optarg);
	    break;

	  case 't':
	    if (str2int(optarg, &request_timeout) < 0)
		syslog(LOG_ERR, "invalid argument to '-t': %s", optarg);
	    break;
	    
	  case 'g':
	    if (str2gid(optarg, &server_gid) < 0)
		syslog(LOG_ERR, "invalid argument to '-g': %s", optarg);
	    break;
	    
	  case 'u':
	    if (str2uid(optarg, &server_uid, &server_gid) < 0)
		syslog(LOG_ERR, "invalid argument to '-u': %s", optarg);
	    break;

	  case 'C':
	    if (conf_parse(optarg, 0) < 0)
	    {
		fprintf(stderr, "%s: error parsing config file: %s\n",
			argv[0], optarg);
		exit(1);
	    }
	    break;
	    
	  case 'P':
	    pidfile_path = s_strdup(optarg);
	    break;

	  case 'K':
	    if (str2int(optarg, &kernel_threads) < 0)
		syslog(LOG_ERR, "invalid argument to '-K': %s", optarg);
	    break;

	  case 'o':
	    opsys = "OTHER";
	    break;

	  case 'N':
	    noident_flag = 1;
	    break;

	  case 'V':
	    printf("Pidentd version %s (compiled for %s)\n",
	    	   server_version, osinfo_build);
	    exit(0);
	    
	  default:
	    usage(stderr);
	    exit(1);
	}

    if (debug)
	fprintf(stderr, "pidentd %s starting\n", server_version);

    if (ka_init())
    {
	syslog(LOG_ERR, "OS version mismatch - compiled for %s", osinfo_build);
	fprintf(stderr,
		"%s: OS version mismatch - compiled for: %s\n",
		argv[0], osinfo_build);
	exit(1);
    }

    
    if (socket_type < 0)
    {
	/*
	** Try to autodetect how we was started.
	*/
	socket_type = socktype(0);
	
	if (debug)
	    fprintf(stderr, "socktype = %d\n", socket_type);

	if (socket_type == SOCKTYPE_LISTEN ||
	    socket_type == SOCKTYPE_CONNECTED)
	    listen_sock = 0;
    }
    
    if (!debug &&
	getppid() != 1 && !init_mode &&
	socket_type != SOCKTYPE_CONNECTED &&
	listen_sock < 0)
    {
	become_daemon();
    }

#ifdef HAVE_THR_SETCONCURRENCY
#if 1
    thr_setconcurrency(kernel_threads+8);
#else
    thr_setconcurrency(sysconf(_SC_NPROCESSORS_ONLN));
#endif
#endif
    
    syslog(LOG_DEBUG, "started");


    pthread_attr_init(&cattr_detached);
    pthread_attr_setdetachstate(&cattr_detached, PTHREAD_CREATE_DETACHED);
    
    
    if (socket_type != SOCKTYPE_CONNECTED)
    {
	if (!debug && pidfile_path)
	    pidfile_create(pidfile_path);
	
	if (server_init() < 0)
	{
	    if (debug)
		fprintf(stderr, "%s: failed binding to the TCP/IP socket\n",
			argv[0]);
	    goto Exit;
	}
    }

    if (kernel_init() < 0)
    {
	if (debug)
	    fprintf(stderr, "%s: failed opening kernel devices\n",
		    argv[0]);
	goto Exit;
    }

#ifdef HAVE_LIBDES
    if (encrypt_flag)
    {
	if (pdes_init(NULL) < 0)
	{
	    syslog(LOG_ERR, "encryption could not be initalized: %m");
	    if (debug)
	    {
		fprintf(stderr, "%s: encryption could not be initialized: ");
		perror("");
	    }
	    goto Exit;
	}
    }
#endif

    if (server_uid == -1)
    {
	if (str2uid("nobody", &server_uid, &server_gid) < 0)
	    server_uid = 0;
    }
    
    if (server_gid)
	setgid(server_gid);
    
    if (server_uid)
	setuid(server_uid);

    timeout_init();
    request_init();
    
    if (socket_type != SOCKTYPE_CONNECTED)
    {
	if (debug)
	    fprintf(stderr, "entering server main loop\n");
    
	server_run();
    }
    else
	exit(request_run(listen_sock, 1));

  Exit:
    syslog(LOG_DEBUG, "terminating");
    exit(1);
}
