/******************************** -*- C -*- ****************************
 *
 * System specific implementation module.
 *
 * This module contains implementations of various operating system
 * specific routines.  This module should encapsulate most (or all) of
 * these calls so that the rest of the code is portable.
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/


#include "gst.h"
#include "gstpriv.h"

#include <errno.h>
#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/times.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#ifdef HAVE_SYS_FILE_H
# include <sys/file.h>
#endif

#ifdef HAVE_TERMIOS_H
# include <termios.h>
#endif

#ifdef HAVE_STROPTS_H
#include <stropts.h>
#endif

#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif

#ifdef HAVE_SYS_TIMEB_H
#include <sys/timeb.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef HAVE_GETRUSAGE
# include <sys/resource.h>
#endif

#if defined(HAVE_SYS_TIME_H) || defined(TIME_WITH_SYS_TIME)
# include <sys/time.h>
#endif
#if !defined(HAVE_SYS_TIME_H) || defined(TIME_WITH_SYS_TIME)
# include <time.h>
#endif

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif /* STDC_HEADERS */

#include <stdarg.h>

#ifdef WIN32
#  include <windows.h>
#endif

#ifdef __OS2__
#  include <direct.h>
#endif

#ifdef HAVE_IO_H
# include <io.h>
#endif

#ifndef F_OK
#define F_OK 0
#define X_OK 1
#define W_OK 2
#define R_OK 4
#endif

#ifndef PATH_MAX
#define PATH_MAX  1024		/* max length of a file and path */
#endif


/* Yield A - B, measured in seconds.
   This function is copied from the GNU C Library.  */
static int tm_diff (struct tm *a,
		    struct tm *b);

#undef INTERRUPTS_DONE
#ifdef SIG_BLOCK
#define INTERRUPTS_DONE
static int disabled = false;

int_state
_gst_disable_interrupts (void)
{
  sigset_t newSet;
  static sigset_t oldSet;

  if (disabled)
    {
      /* let only the outermost calls actually block and unblock
         signals */
      return ((int_state) NULL);
    }
  disabled = true;
  sigfillset (&newSet);
  sigprocmask (SIG_BLOCK, &newSet, &oldSet);
  return ((int_state) & oldSet);
}

void
_gst_enable_interrupts (int_state newSet)
{
  if (newSet)
    {
      disabled = false;
      sigprocmask (SIG_SETMASK, (sigset_t *) newSet, NULL);
    }
}
#endif

#if defined(HAVE_SIGSETMASK) && !defined(INTERRUPTS_DONE)	/* BSD
								   style 
								 */
#define INTERRUPTS_DONE
int_state
_gst_disable_interrupts ()
{
  return (sigsetmask (-1));
}

void
_gst_enable_interrupts (mask)
     int_state mask;
{
  sigsetmask (mask);
}
#endif

#if defined(HAVE_SIGHOLD) && !defined(INTERRUPTS_DONE)	/* SVID style */
#define INTERRUPTS_DONE
static mst_Boolean disabled;

int_state
_gst_disable_interrupts ()
{
  int i;

  if (disabled)
    /* let only the outermost calls actually block and unblock signals */
    return (0);

  disabled = true;
  for (i = 1; i <= 8 * SIZEOF_LONG; i++)
    sighold (i);		/* want it blocked - ok if it already
				   is */

  return (1);
}

void
_gst_enable_interrupts (mask)
     int_state mask;
{
  int i;

  if (!mask)
    return;

  disabled = false;
  for (i = 1; i <= 8 * SIZEOF_LONG; i++)
    sigrelse (i);		/* want it unblocked if it used to be */
}
#endif

#ifndef INTERRUPTS_DONE
int_state
_gst_disable_interrupts ()
{
  return ((int_state) 0);	/* no recognized interrupt mechanism */
}

void
_gst_enable_interrupts (mask)
     int_state mask;
{
  /* no way dude */
}
#endif

SigHandler
_gst_set_signal_handler (int signum,
			 SigHandler handlerFunc)
{
#ifdef _POSIX_VERSION
  /* If we are running on a posix-compliant system, then do things the
     Posix way. */
  struct sigaction act, o_act;

  act.sa_handler = handlerFunc;
  act.sa_flags = 0;

  sigemptyset (&act.sa_mask);
  sigaction (signum, &act, &o_act);
  return o_act.sa_handler;

#else
  return signal (signum, handlerFunc);
#endif
}



unsigned long
_gst_get_milli_time (void)
{
#undef MILLI_DONE
#ifdef WIN32
#define MILLI_DONE
  /* time() seems not to work... so we hack. This method to obtain the
     time is complex, but it is the most precise. */
  static long frequency = 0, frequencyH, adjust = 0;
  long milli;
  LARGE_INTEGER counter;
  SYSTEMTIME st;

  if (!frequency)
    {
      if (QueryPerformanceFrequency (&counter))
	{
	  /* frequencyH = 1000 * 2^32 / frequency */
	  frequency = counter.HighPart ? -1 : counter.LowPart;
	  frequencyH = MulDiv (1000 * (1 << 16), (1 << 16), frequency);
	}
      else
	frequency = -1;
    }

  if (frequency == -1)
    {
      /* QueryPerformanceCounter not supported, always use GetLocalTime 
       */
      adjust = milli = 0;
    }
  else
    {
      QueryPerformanceCounter (&counter);
      /* milli = (high * 2^32 + low) * 1000 / freq = high * (1000 *
         2^32 / freq) + (low * 1000 / freq) = (high * frequencyH) +
         (low / 4) * 4000 / freq) Dividing and multiplying
         counter.LowPart by 4 is needed because MulDiv accepts signed
         integers but counter.LowPart is unsigned. */
      milli = counter.HighPart * frequencyH;
      milli += MulDiv (counter.LowPart >> 2, 4000, frequency);
    }

  if (!adjust)
    {
      GetLocalTime (&st);
      adjust = st.wMilliseconds;
      adjust += st.wSecond * 1000;
      adjust += st.wMinute * 60000;
      adjust += st.wHour * 3600000;
      adjust -= milli;
      milli += adjust;
    }
  else
    {
      milli += adjust;
      while (milli > 86400000)
	{
	  milli -= 86400000;
	  adjust -= 86400000;
	}
    }
  return (milli);
#endif

#if defined(HAVE_GETTIMEOFDAY) && !defined(MILLI_DONE)	/* BSD style */
#define MILLI_DONE
  struct timeval t;

  gettimeofday (&t, NULL);
  t.tv_sec %= 86400;
  return (t.tv_sec * 1000 + t.tv_usec / 1000);
#endif

#ifndef MILLI_DONE
  /* Assume that ftime (System V) is available */
  struct timeb t;
  ftime (&t);
  return t.time * 1000 + t.millitm;
#endif
}

#define TM_YEAR_BASE 1900

static int
tm_diff (struct tm *a,
	 struct tm *b)
{
  /* Compute intervening leap days correctly even if year is negative.
     Take care to avoid int overflow in leap day calculations, but it's 
     OK to assume that A and B are close to each other.  */
  int a4 = (a->tm_year >> 2) + (TM_YEAR_BASE >> 2) - !(a->tm_year & 3);
  int b4 = (b->tm_year >> 2) + (TM_YEAR_BASE >> 2) - !(b->tm_year & 3);
  int a100 = a4 / 25 - (a4 % 25 < 0);
  int b100 = b4 / 25 - (b4 % 25 < 0);
  int a400 = a100 >> 2;
  int b400 = b100 >> 2;
  int intervening_leap_days = (a4 - b4) - (a100 - b100) + (a400 - b400);
  int years = a->tm_year - b->tm_year;
  int days = (365 * years + intervening_leap_days
	      + (a->tm_yday - b->tm_yday));
  return (60 * (60 * (24 * days + (a->tm_hour - b->tm_hour))
		+ (a->tm_min - b->tm_min)) + (a->tm_sec - b->tm_sec));
}

long
_gst_adjust_time_zone (time_t t)
{
  struct tm save_tm, *decoded_time;
  long bias;

#ifdef LOCALTIME_CACHE
  tzset ();
#endif
  decoded_time = localtime (&t);
  save_tm = *decoded_time;
  decoded_time = gmtime (&t);
  bias = tm_diff (&save_tm, decoded_time);

  return (((long) t) + bias);
}

long
_gst_current_time_zone_bias (void)
{
  time_t now;
  long bias;
  struct tm save_tm, *decoded_time;

  time (&now);

#ifdef LOCALTIME_CACHE
  tzset ();
#endif

  decoded_time = localtime (&now);
  save_tm = *decoded_time;
  decoded_time = gmtime (&now);
  bias = tm_diff (&save_tm, decoded_time);
  return (bias);
}

char *
_gst_current_time_zone_name (void)
{
  char *zone;
  zone = getenv ("TZ");
  if (!zone)
    {
#ifdef WIN32
      long bias = _gst_current_time_zone_bias () / 60;
      TIME_ZONE_INFORMATION tzi;
      LPCWSTR name;
      static char buffer[32];
      GetTimeZoneInformation (&tzi);
      zone = buffer;
      name = (bias == (tzi.Bias + tzi.StandardBias))
	? tzi.StandardName : tzi.DaylightName;

      WideCharToMultiByte (CP_ACP, 0,
			   name, lstrlenW (name), zone, 32, NULL, NULL);

#else
      /* Maybe we could guess it */
      zone = "XXX";
#endif
    }

  return strdup (zone);
}

long
_gst_get_time (void)
{
  time_t now;
  time (&now);

  return (_gst_adjust_time_zone (now));
}



#ifdef WIN32
struct
{
  HANDLE hNewWaitEvent;
  HANDLE hCritEvent;
  long sleepTime;
}
alarms;

/* thread for precise alarm callbacks */
void CALLBACK
alarm_thread (unused)
     LPVOID unused;
{
  WaitForSingleObject (alarms.hNewWaitEvent, INFINITE);
  for (;;)
    {
      int sleepTime;

      WaitForSingleObject (alarms.hCritEvent, INFINITE);
      sleepTime = alarms.sleepTime;
      SetEvent (alarms.hCritEvent);

      if (sleepTime > 0)
	{
	  if (WaitForSingleObject (alarms.hNewWaitEvent, sleepTime) !=
	      WAIT_TIMEOUT)
	    {
	      /* The old wait was canceled by a new one */
	      continue;
	    }
	}
      kill (getpid (), SIGALRM);
      WaitForSingleObject (alarms.hNewWaitEvent, INFINITE);
    }
}

static void
initialize_alarms ()
{
  HANDLE hthread;
  DWORD tid;

  /* Starts as non-signaled, so alarm_thread will wait */
  alarms.hNewWaitEvent = CreateEvent (NULL, FALSE, FALSE, NULL);

  /* Starts as signaled (i.e. the first wait won't block) */
  alarms.hCritEvent = CreateEvent (NULL, FALSE, TRUE, NULL);

  /* Start alarm_thread with a 1024 bytes stack */
  hthread = CreateThread (NULL,
			  1024,
			  (LPTHREAD_START_ROUTINE) alarm_thread,
			  NULL, 0, &tid);

  /* This does not terminate the thread - it only releases our handle */
  CloseHandle (hthread);
}
#endif

/* Please feel free to make this more accurate for your operating system
 * and send me the changes.
 */
void
_gst_signal_after (int deltaMilli,
		   SigHandler func,
		   int kind)
{
  _gst_set_signal_handler (kind, func);

  if (deltaMilli <= 0)
    {
      kill (getpid (), kind);
      return;
    }

  if (kind == TIMER_PROCESS)
    {
#if defined(ITIMER_VIRTUAL)
      struct itimerval value;
      value.it_interval.tv_sec = value.it_interval.tv_usec = 0;
      value.it_value.tv_sec = deltaMilli / 1000;
      value.it_value.tv_usec = (deltaMilli % 1000) * 1000;
      setitimer (ITIMER_VIRTUAL, &value, (struct itimerval *) 0);
#endif

    }
  else if (kind == TIMER_REAL)
    {

#ifdef WIN32
#define ALARM_DONE
      WaitForSingleObject (alarms.hCritEvent, INFINITE);
      alarms.sleepTime = deltaMilli;
      SetEvent (alarms.hCritEvent);
      SetEvent (alarms.hNewWaitEvent);
#endif

#if defined(ITIMER_REAL) && !defined(ALARM_DONE)
#define ALARM_DONE
      struct itimerval value;
      value.it_interval.tv_sec = value.it_interval.tv_usec = 0;
      value.it_value.tv_sec = deltaMilli / 1000;
      value.it_value.tv_usec = (deltaMilli % 1000) * 1000;
      setitimer (ITIMER_REAL, &value, (struct itimerval *) 0);
#endif

#if defined(HAVE_FORK) && !defined(ALARM_DONE)
#define ALARM_DONE
      static pid_t pid = -1;
      long end, ticks;
      if (pid != -1)
	kill (pid, SIGTERM);

      switch (pid = fork ())
	{
	case -1:
	  /* Error, try to recover */
	  kill (getpid (), SIGALRM);
	  break;

	case 0:
	  /* Child process */
	  end = _gst_get_milli_time () + deltaMilli;
	  do
	    {
	      ticks = end - _gst_get_milli_time ();
	      if (ticks > 1100)	/* +100 is arbitrary */
		sleep ((int) (ticks / 1000));
	    }
	  while (ticks > 0);
	  kill (getppid (), SIGALRM) _exit (0);
	}
#endif

#if defined(HAVE_ALARM) && !defined(ALARM_DONE)
#define ALARM_DONE
      alarm (deltaMilli / 1000);
#endif

#if defined(ALARM_DONE)
#undef ALARM_DONE
#else
      /* Cannot do anything better than this */
      kill (getpid (), SIGALRM);
#endif
    }
}

void
_gst_set_file_interrupt (int fd,
			 SigHandler func)
{
  static SigHandler old_func;
  if (func != old_func) 
    {
      old_func = func;
      _gst_set_signal_handler (SIGCHLD, func);
#ifdef SIGPOLL
      _gst_set_signal_handler (SIGPOLL, func);
#else
      _gst_set_signal_handler (SIGIO, func);
#endif
#ifdef SIGURG
      _gst_set_signal_handler (SIGURG, func);
#endif
    }

#if defined(F_SETOWN) && (defined(FASYNC)||defined(O_ASYNC))
#ifndef O_ASYNC
#define O_ASYNC FASYNC
#endif

  {
    int oldflags;

    oldflags = fcntl (fd, F_GETFL, 0);
    if (((oldflags & O_ASYNC)
	 || (fcntl (fd, F_SETFL, oldflags | O_ASYNC) != -1))
	&& fcntl (fd, F_SETOWN, getpid ()) != -1)
      return;
  }
#endif

#ifdef I_SETSIG
  if (ioctl (fd, I_SETSIG, S_INPUT | S_OUTPUT | S_HIPRI) > -1)
    return;
#endif

#ifdef FIOSSAIOSTAT
#ifdef FIOSSAIOOWN
  {
    int stat_flags = 1;
    int own_flags = getpid();

    if (ioctl (fd, FIOSSAIOSTAT, &stat_flags) != -1
	&& ioctl (fd, FIOSSAIOOWN, &own_flags) != -1)
      return;
  }
#endif
#endif

#ifdef FIOASYNC
  {
    int argFIOASYNC = 1;
#if defined (SIOCSPGRP)
    int argSIOCSPGRP = getpid ();

    if (ioctl (fd, SIOCSPGRP, &argSIOCSPGRP) > -1 ||
	ioctl (fd, FIOASYNC, &argFIOASYNC) > -1)
      return;
#elif defined (O_ASYNC)
    int oldflags;

    oldflags = fcntl (fd, F_GETFL, 0);
    if (((oldflags & O_ASYNC)
         || (fcntl (fd, F_SETFL, oldflags | O_ASYNC) != -1))
        && ioctl (fd, FIOASYNC, &argFIOASYNC) > -1)
      return;
#else
    if (ioctl (fd, FIOASYNC, &argFIOASYNC) > -1)
      return;
#endif
  }
#endif /* FIOASYNC */
}



char *
_gst_get_cur_dir_name (void)
{
#ifdef HAVE_GETCWD
  char *cwd;
  char *ret;
  unsigned path_max;
  int save_errno;

  path_max = (unsigned) PATH_MAX;
  path_max += 2;		/* The getcwd docs say to do this. */

  cwd = xmalloc (path_max);

  errno = 0;
  do
    {
      ret = getcwd (cwd, path_max);
      if (ret)
	return (cwd);

      if (errno != ERANGE)
	break;

      errno = 0;
      path_max += 128;
      cwd = xrealloc (cwd, path_max);
    }
  while (!errno);

  save_errno = errno;
  xfree (cwd);
  errno = save_errno;
  return (NULL);
#else
  char name[PATH_MAX + 2];
  getwd (name);
  return (strdup (name));
#endif
}


char *
_gst_get_full_file_name (const char *fileName)
{
  char *fullFileName;
  static char *fullPath = NULL;

  if (fileName[0] == '/')	/* absolute, so don't need to change */
    return (strdup (fileName));

  if (fullPath == NULL)
    {
      /* Only need to do this once, then cache the result */
      fullPath = _gst_get_cur_dir_name ();
    }

  /* 
   * ### canonicalize filename and full path here in the future (remove any
   * extraneous .. or . directories, etc.)
   */

  fullFileName = (char *) xmalloc (strlen (fullPath) + strlen (fileName) + 1	/* slash 
										 */
				   + 1 /* trailing nul */ );
  sprintf (fullFileName, "%s/%s", fullPath, fileName);
  return (fullFileName);
}



long
_gst_get_file_modify_time (const char *fileName)
{
  struct stat st;

  if (stat (fileName, &st) < 0)
    return ((unsigned long) 0);

  else
    return (_gst_adjust_time_zone (st.st_mtime));
}


mst_Boolean
_gst_file_is_readable (const char *fileName)
{
  return (access (fileName, R_OK) == 0);
}

mst_Boolean
_gst_file_is_writeable (const char *fileName)
{
  return (access (fileName, W_OK) == 0);
}

mst_Boolean
_gst_file_is_executable (const char *fileName)
{
  return (access (fileName, X_OK) == 0);
}

static mst_Boolean signal_ok = false;

int
_gst_open_pipe (const char *command,
		const char *mode)
{
  /* Most of the code for ptys was adapted from librep */

  static mst_Boolean initialized;
#ifdef HAVE_GRANTPT
  static mst_Boolean have_devptmx;
#endif
  static int firstpty;

  int access, i;
  char slavenam[32];
  int master;
  int slave;
  struct termios st;

  if (!signal_ok)
    {
      _gst_set_signal_handler (SIGPIPE, SIG_IGN);
      signal_ok = true;
    }

  switch (*mode)
    {
    case 'a':
    case 'w':
      access = O_WRONLY;
      break;
    case 'r':
      access = O_RDONLY;
      break;
    default:
      return -1;
    }

  for (i = 1; i < 3; ++i)
    {
      ++mode;
      if (*mode == '\0')
	break;
      else if (*mode == '+')
	access = O_RDWR;
    }

  if (!initialized)
    {
      char s[] = "/dev/pty?0";
      int n;
#ifdef HAVE_GRANTPT
      have_devptmx = _gst_file_is_readable ("/dev/ptmx");
#endif

      /* Search for the first letter to be tried when using pty devices 
       */
      firstpty = -1;
      for (n = 26, s[8] = 'p'; n--;
	   s[8] = (s[8] == 'z') ? 'a' : s[8] + 1)
	{
	  struct stat statb;
	  if (stat (s, &statb) >= 0)
	    {
	      firstpty = s[strlen ("/dev/pty")];
	      break;
	    }
	}
    }

#ifdef HAVE_GRANTPT
  if (have_devptmx)
    {
      master = open ("/dev/ptmx", O_RDWR);
      if (master >= 0)
	{
	  char *buf;
	  grantpt (master);
	  unlockpt (master);
	  buf = (char *) ptsname (master);
	  if (!buf)
	    close (master);

	  else
	    {
	      strcpy (slavenam, buf);
	      goto got_the_pty;
	    }
	}
    }
#endif

  if (firstpty != -1)
    {
      /* Assume /dev/ptyXN and /dev/ttyXN naming system. The
         FIRST_PTY_LETTER gives the first X to try. We try in the
         sequence FIRST_PTY_LETTER, .., 'z', 'a', .., FIRST_PTY_LETTER. 
         Is this worthwhile, or just over-zealous? */

      char *ptyId = slavenam + strlen ("/dev/pty");
      int n;
      strcpy (slavenam, "/dev/pty?0");

      /* Search for the first letter to be tried when using pty devices 
       */
      for (n = 26, ptyId[0] = firstpty;
	   n--; ptyId[0] = (ptyId[0] == 'z') ? 'a' : ptyId[0] + 1)
	{

	  for (ptyId[1] = '0'; ptyId[1] <= 'f';
	       ptyId[1] = (ptyId[1] == '9') ? 'a' : ptyId[1] + 1)
	    {

	      struct stat statb;
	      if (stat (slavenam, &statb) < 0)
		return -1;
	      if ((master = open (slavenam, O_RDWR)) >= 0)
		{
		  slavenam[strlen ("/dev/")] = 't';
		  if (_gst_file_is_readable (slavenam)
		      && _gst_file_is_writeable (slavenam))
		    goto got_the_pty;

		  slavenam[strlen ("/dev/")] = 'p';
		  close (master);
		}		/* open master */
	    }			/* hex digit */
	}			/* alphabetic char */
    }
  return -1;

got_the_pty:

  /* We have a pty in master and slavenam */

  switch (fork ())
    {
    case -1:
      {
	int save_errno;
	save_errno = errno;
	close (master);
	errno = save_errno;
	return (-1);
      }

    case 0:
      /* Child process */
      close (master);

      /* Order can be either open/setsid/TIOCSCTTY or setsid/open.
         Anyway we need to make this process a session group leader,
         because it is on a new PTY, and things like job control
         simply will not work correctly unless there is a session
         group leader and a process group leader (which a session
         group leader automatically is).  This also disassociates
         us from our old controlling tty.  */
#ifdef TIOCSCTTY
      if ((slave = open (slavenam, O_RDWR)) < 0)
        {
          perror ("open(slave)");
          exit (255);
        }

      if (setsid () < 0)
        {
          perror ("setsid");
          exit (255);
        }

      ioctl (slave, TIOCSCTTY, 0);
#else
      if (setsid () < 0)
        {
          perror ("setsid");
          exit (255);
        }

      if ((slave = open (slavenam, O_RDWR)) < 0)
        {
          perror ("open(slave)");
          exit (255);
        }
#endif

#ifdef I_PUSH
      /* Push the necessary modules onto the slave to get terminal
         semantics. */
      ioctl (slave, I_PUSH, "ptem");
      ioctl (slave, I_PUSH, "ldterm");
#endif

      tcgetattr (slave, &st);
      st.c_iflag &= ~(ISTRIP | IGNCR | INLCR | IXOFF);
      st.c_iflag |= (ICRNL | IGNPAR | BRKINT | IXON);
      st.c_cflag &= ~CSIZE;
      st.c_cflag |= CREAD | CS8 | CLOCAL;
      st.c_oflag &= ~OPOST;
      st.c_lflag &= ~(ECHO | ECHOE | ECHOK | NOFLSH | TOSTOP);
      st.c_lflag |= ISIG;
#if 0
      st.c_cc[VMIN] = 1;
      st.c_cc[VTIME] = 0;
#endif

      /* Set some control codes to default values */
#ifdef VINTR
      st.c_cc[VINTR] = '\003';  /* ^c */
#endif
#ifdef VQUIT
      st.c_cc[VQUIT] = '\034';  /* ^| */
#endif
#ifdef VERASE
      st.c_cc[VERASE] = '\177'; /* ^? */
#endif
#ifdef VKILL
      st.c_cc[VKILL] = '\025';  /* ^u */
#endif
#ifdef VEOF
      st.c_cc[VEOF] = '\004';   /* ^d */
#endif

      tcsetattr (slave, TCSANOW, &st);

      /* Duplicate the handles.  Which handles are to be hooked is
         anti-intuitive: remember that access gives the parent's point 
         of view, not the child's, so `read only' means the child
         should write to the pipe and `write only' means the child
         should read from the pipe. */
      if (access != O_RDONLY)
	dup2 (slave, 0);

      if (access != O_WRONLY)
	{
	  dup2 (slave, 1);
	  dup2 (slave, 2);
	}

      signal (SIGPIPE, SIG_DFL);
      if (slave > 2)
	close (slave);

      _exit (system (command) >= 0);
      /*NOTREACHED*/

    default:
      return (master);
    }
}

int
_gst_open_file (const char *filename,
		const char *mode)
{
  mst_Boolean create = false;
  int oflags = 0, access = 0;
  int fd, i;

  if (!signal_ok)
    {
      _gst_set_signal_handler (SIGPIPE, SIG_IGN);
      signal_ok = true;
    }

  switch (*mode)
    {
    case 'a':
      access = O_WRONLY;
      oflags |= O_APPEND;
      create = 1;
      break;
    case 'w':
      access = O_WRONLY;
      oflags |= O_TRUNC;
      create = 1;
      break;
    case 'r':
      access = O_RDONLY;
      break;
    default:
      return -1;
    }

  for (i = 1; i < 3; ++i)
    {
      ++mode;
      if (*mode == '\0')
	break;
      else if (*mode == '+')
	access = O_RDWR;
      else if (*mode == 'x')
	oflags |= O_EXCL;
    }

  if (create)
    fd = open (filename, oflags | access | O_CREAT, 0666);
  else
    fd = open (filename, oflags | access);

  if (fd < 0)
    return -1;

#ifdef FD_CLOEXEC
  fcntl (fd, F_SETFD, fcntl (fd, F_GETFD, 0) | FD_CLOEXEC);
#endif

  return fd;
}

mst_Boolean
_gst_is_pipe (int fd)
{
  struct stat st;

  fstat (fd, &st);
#ifdef S_IFREG
  return !(st.st_mode & S_IFREG);
#else
  return !S_ISREG (st.st_mode);
#endif
}

int
_gst_full_write (int fd,
		 PTR buffer,
		 int size)
{
  char *buf = (char *) buffer;
  static mst_Boolean signal_ok = false;
  int num = 0, sofar = 0;

  if (!signal_ok)
    {
      _gst_set_signal_handler (SIGPIPE, SIG_IGN);
      signal_ok = true;
    }
  for (; size; buf += num, size -= num, sofar += num)
    {
      num = write (fd, buf, size);
      if (num <= 0)
	{
	  if (errno == EINTR)
	    {
	      num = 0;
	      errno = 0;
	    }
	  else
	    break;
	}
    }

  return (num < 0 ? num : sofar);
}



void
_gst_init_sysdep (void)
{
#ifdef WIN32
  initialize_alarms ();
#endif /* WIN32 */
  tzset ();
}

void
_gst_debugf (const char *fmt,
	     ...)
{
  char buf[1024];
  va_list args;
  va_start (args, fmt);

  vsprintf (buf, fmt, args);

#if defined(WIN32) && !defined(__CYGWIN__)&& !defined(__CYGWIN32__)
  /* VC++ has a Win32-friendly debugger, otherwise winning Cygwin has
     not */
  OutputDebugString (buf);
#else /* !WIN32 */
  {
    static FILE *debFile = NULL;
    if (debFile == NULL)
      debFile = fopen ("mst.deb", "w");

    fputs (buf, debFile);
    fflush (debFile);
  }
#endif /* !WIN32 */

  va_end (args);
}

void
_gst_debug (void)
{
  /* kill(getpid(), SIGTRAP); */
  /* getchar(); */
}
