/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { AmountJson } from "@gnu-taler/taler-util";

export interface TextFieldHandler {
  onInput?: SafeHandler<string>;
  value: string;
  error?: string | Error;
}

export interface AmountFieldHandler {
  onInput?: SafeHandler<AmountJson>;
  value: AmountJson;
  error?: string | Error;
}

declare const __safe_handler: unique symbol;
export type SafeHandler<T> = {
  <Req extends T>(req: Req): Promise<void>;
  (): Promise<void>;
  [__safe_handler]: true;
};

type UnsafeHandler<T> = ((p: T) => Promise<void>) | ((p: T) => void);

export function withSafe<T>(
  handler: UnsafeHandler<T>,
  onError: (e: Error) => void,
): SafeHandler<T> {
  const sh = async function (p: T): Promise<void> {
    try {
      await handler(p);
    } catch (e) {
      if (e instanceof Error) {
        onError(e);
      } else {
        onError(new Error(String(e)));
      }
    }
  };
  return sh as SafeHandler<T>;
}

export const nullFunction = async function (): Promise<void> {
  //do nothing
} as SafeHandler<void>;

//FIXME: UI button should required SafeHandler but
//useStateComponent should not be required to create SafeHandlers
//so this need to be split in two:
// * ButtonHandlerUI =>  with i18n
// * ButtonHandlerLogic => without i18n
export interface ButtonHandler {
  onClick?: SafeHandler<void>;
  // error?: TalerError;
}

export interface ToggleHandler {
  value?: boolean;
  button: ButtonHandler;
}

export interface SelectFieldHandler {
  onChange?: SafeHandler<string>;
  error?: string;
  value: string;
  isDirty?: boolean;
  list: Record<string, string>;
}
