dnl -*- Autoconf -*-
dnl Copyright (C) 1993-2024 Free Software Foundation, Inc.
dnl This file is free software, distributed under the terms of the GNU
dnl General Public License as published by the Free Software Foundation;
dnl either version 2 of the License, or (at your option) any later version.
dnl As a special exception to the GNU General Public License, this file
dnl may be distributed as part of a program that contains a configuration
dnl script generated by Autoconf, under the same distribution terms as
dnl the rest of that program.

dnl From Bruno Haible, Marcus Daniels, Sam Steingold.

AC_PREREQ([2.63])

AC_DEFUN([FFCALL_CODEEXEC],
[
  AC_REQUIRE([AC_CANONICAL_HOST])
  AC_REQUIRE([gl_HOST_CPU_C_ABI])
  AC_REQUIRE([gl_FUNC_MMAP_ANON])

  AC_CACHE_CHECK([whether code in malloc()ed memory is executable],
    [ffcall_cv_codeexec],
    [dnl The test below does not work on platforms with the following ABIs:
     dnl - hppa, because function pointers are actually pointers into(!)
     dnl   a two-pointer struct.
     dnl - hppa64, because function pointers are actually pointers to a
     dnl   four-pointer struct.
     dnl - powerpc on AIX, powerpc64, because function pointers are actually
     dnl   pointers to a three-pointer struct.
     dnl - ia64, because function pointers are actually pointers to a
     dnl   two-pointer struct.
     case "$HOST_CPU_C_ABI--$host_os" in
       hppa--* | hppa64--* | powerpc--aix* | powerpc64--* | ia64--*)
         dnl On these platforms, it's irrelevant whether malloc'ed memory is
         dnl executable, because the trampolines are built without executable
         dnl code.
         ffcall_cv_codeexec="irrelevant"
         ;;
       arm64--freebsd*)
         dnl On this platform, malloc()ed memory is not executable, and the
         dnl test program loops endlessly.
         ffcall_cv_codeexec=no
         ;;
       *)
         AC_RUN_IFELSE(
           [AC_LANG_SOURCE([
              GL_NOCRASH
              [#include <sys/types.h>
               /* declare malloc() */
               #include <stdlib.h>
               int fun () { return 31415926; }
               int main ()
               { nocrash_init();
                {long size = (char*)&main - (char*)&fun;
                 char* funcopy = (char*) malloc(size);
                 int i;
                 for (i = 0; i < size; i++) { funcopy[i] = ((char*)&fun)[i]; }
                 return !((*(int(*)())funcopy)() == 31415926);
               }}
              ]])
           ],
           [ffcall_cv_codeexec=yes],
           [ffcall_cv_codeexec=no],
           [dnl When cross-compiling, assume the known behaviour.
            dnl If we don't know, assume the worst.
            case "$host_os" in
              cygwin*)
                case "$HOST_CPU_C_ABI" in
                  i386)
                    ffcall_cv_codeexec="guessing yes" ;;
                  x86_64)
                    ffcall_cv_codeexec="guessing no" ;;
                  *)
                    ffcall_cv_codeexec="guessing no" ;;
                esac
                ;;
              darwin*)
                case "$HOST_CPU_C_ABI" in
                  i386 | powerpc)
                    ffcall_cv_codeexec="guessing yes" ;;
                  x86_64)
                    ffcall_cv_codeexec="guessing no" ;;
                  *)
                    ffcall_cv_codeexec="guessing no" ;;
                esac
                ;;
              irix*)
                ffcall_cv_codeexec="guessing no" ;;
              linux*)
                case "$HOST_CPU_C_ABI" in
                  alpha | ia64)
                    ffcall_cv_codeexec="guessing yes" ;;
                  arm | armhf | arm64 | i386 | mips* | s390 | s390x | sparc | sparc64 | x86_64*)
                    ffcall_cv_codeexec="guessing no" ;;
                  *)
                    ffcall_cv_codeexec="guessing no" ;;
                esac
                ;;
              solaris*)
                case "$HOST_CPU_C_ABI" in
                  i386 | sparc | sparc64)
                    ffcall_cv_codeexec="guessing yes" ;;
                  x86_64)
                    ffcall_cv_codeexec="guessing no" ;;
                  *)
                    ffcall_cv_codeexec="guessing no" ;;
                esac
                ;;
              *)
                ffcall_cv_codeexec="guessing no" ;;
            esac
           ])
         ;;
     esac
    ])
  case "$ffcall_cv_codeexec" in
    *yes | irrelevant)
     AC_DEFINE([CODE_EXECUTABLE], [], [whether code in malloc()ed memory is executable])
     ;;
    *no) ;;
  esac

  AC_CHECK_HEADER([sys/mman.h], [], [no_mmap=1])
  if test -z "$no_mmap"; then
    AC_CHECK_FUNC([mmap], [], [no_mmap=1])
    if test -z "$no_mmap"; then
      mmap_prog_1='
        #include <stdlib.h>
        #ifdef HAVE_UNISTD_H
         #include <unistd.h>
        #endif
        #include <fcntl.h>
        #include <sys/types.h>
        #include <sys/mman.h>
        int main ()
        {
      '
      mmap_prog_2='
          if (mmap(NULL,0x100000,PROT_READ|PROT_WRITE,flags,fd,0) == (void*)-1)
            exit(1);
          exit(0);
        }
      '
      AC_CACHE_CHECK([for working mmap with MAP_ANONYMOUS],
        [ffcall_cv_func_mmap_anonymous],
        [AC_RUN_IFELSE(
           [AC_LANG_SOURCE([
              GL_NOCRASH
              [$mmap_prog_1
               int flags = MAP_ANONYMOUS | MAP_PRIVATE;
               int fd = -1;
               nocrash_init();
               $mmap_prog_2
              ]])
           ],
           [have_mmap_anon=1
            ffcall_cv_func_mmap_anonymous=yes],
           [ffcall_cv_func_mmap_anonymous=no],
           [dnl When cross-compiling, assume the known behaviour.
            dnl If we don't know, don't assume anything.
            case "$host_os" in
              aix* | cygwin* | darwin* | hpux* | linux* | solaris*)
                ffcall_cv_func_mmap_anonymous="guessing yes" ;;
              *)
                ffcall_cv_func_mmap_anonymous="guessing no" ;;
            esac
           ])
        ])
      case "$ffcall_cv_func_mmap_anonymous" in
        *yes)
          AC_DEFINE([HAVE_MMAP_ANONYMOUS], [1],
            [<sys/mman.h> defines MAP_ANONYMOUS and mmaping with MAP_ANONYMOUS works])
          ;;
        *)
          dnl This is needed for IRIX.
          AC_CACHE_CHECK([for working mmap of /dev/zero],
            [ffcall_cv_func_mmap_devzero],
            [AC_RUN_IFELSE(
               [AC_LANG_SOURCE([
                  GL_NOCRASH
                  [$mmap_prog_1
                   #ifndef MAP_FILE
                    #define MAP_FILE 0
                   #endif
                   int flags = MAP_FILE | MAP_PRIVATE;
                   int fd = open("/dev/zero",O_RDONLY,0666);
                   if (fd<0)
                     exit(1);
                   nocrash_init();
                   $mmap_prog_2
                  ]])
               ],
               [have_mmap_devzero=1
                ffcall_cv_func_mmap_devzero=yes],
               [ffcall_cv_func_mmap_devzero=no],
               [dnl When cross-compiling, assume the known behaviour.
                dnl If we don't know, don't assume anything.
                case "$host_os" in
                  aix* | cygwin* | hpux* | irix* | linux* | solaris*)
                    ffcall_cv_func_mmap_devzero="guessing yes" ;;
                  *)
                    ffcall_cv_func_mmap_devzero="guessing no" ;;
                esac
               ])
            ])
          case "$ffcall_cv_func_mmap_devzero" in
            *yes)
              AC_DEFINE([HAVE_MMAP_DEVZERO], [1],
                [mmaping of the special device /dev/zero works])
              ;;
          esac
          ;;
      esac
    fi
  fi

  AC_CHECK_FUNCS([mprotect])
  if test $ac_cv_func_mprotect = yes; then
    AC_CACHE_CHECK([for working mprotect], [cl_cv_func_mprotect_works],
      [if test $cross_compiling = no; then
         mprotect_prog='
           #include <sys/types.h>
           /* Declare malloc().  */
           #include <stdlib.h>
           /* Declare getpagesize().  */
           #ifdef HAVE_UNISTD_H
            #include <unistd.h>
           #endif
           #ifdef __hpux
            extern
            #ifdef __cplusplus
            "C"
            #endif
            int getpagesize (void);
           #endif
           /* Declare mprotect().  */
           #include <sys/mman.h>
           char foo;
           int main ()
           {
             unsigned long pagesize = getpagesize();
           #define page_align(address)  (char*)((unsigned long)(address) & -pagesize)
         '
         AC_RUN_IFELSE(
           [AC_LANG_SOURCE([
              [$mprotect_prog
                 if ((pagesize-1) & pagesize) exit(1);
                 exit(0);
               }
              ]])
           ],
           [],
           [no_mprotect=1],
           [dnl When cross-compiling, don't assume anything.
            no_mprotect=1
           ])
         mprotect_prog="$mprotect_prog"'
           char* area = (char*) malloc(6*pagesize);
           char* fault_address = area + pagesize*7/2;
         '
         if test -z "$no_mprotect"; then
           AC_RUN_IFELSE(
             [AC_LANG_SOURCE([
                GL_NOCRASH
                [$mprotect_prog
                   nocrash_init();
                   if (mprotect(page_align(fault_address),pagesize,PROT_NONE) < 0)
                     exit(0);
                   foo = *fault_address; /* this should cause an exception or signal */
                   exit(0);
                 }
                ]])
             ],
             [no_mprotect=1],
             [],
             [dnl When cross-compiling, don't assume anything.
              :
             ])
         fi
         if test -z "$no_mprotect"; then
           AC_RUN_IFELSE(
             [AC_LANG_SOURCE([
                GL_NOCRASH
                [$mprotect_prog
                   nocrash_init();
                   if (mprotect(page_align(fault_address),pagesize,PROT_NONE) < 0)
                     exit(0);
                   *fault_address = 'z'; /* this should cause an exception or signal */
                   exit(0);
                 }
                ]])
             ],
             [no_mprotect=1],
             [],
             [dnl When cross-compiling, don't assume anything.
              :
             ])
         fi
         if test -z "$no_mprotect"; then
           AC_RUN_IFELSE(
             [AC_LANG_SOURCE([
                GL_NOCRASH
                [$mprotect_prog
                   nocrash_init();
                   if (mprotect(page_align(fault_address),pagesize,PROT_READ) < 0)
                     exit(0);
                   *fault_address = 'z'; /* this should cause an exception or signal */
                   exit(0);
                 }
                ]])
             ],
             [no_mprotect=1],
             [],
             [dnl When cross-compiling, don't assume anything.
              :
             ])
         fi
         if test -z "$no_mprotect"; then
           AC_RUN_IFELSE(
             [AC_LANG_SOURCE([
                GL_NOCRASH
                [$mprotect_prog
                   nocrash_init();
                   if (mprotect(page_align(fault_address),pagesize,PROT_READ) < 0)
                     exit(1);
                   if (mprotect(page_align(fault_address),pagesize,PROT_READ|PROT_WRITE) < 0)
                     exit(1);
                   *fault_address = 'z'; /* this should not cause an exception or signal */
                   exit(0);
                 }
                ]])
             ],
             [],
             [no_mprotect=1],
             [dnl When cross-compiling, don't assume anything.
              :
             ])
         fi
         if test -z "$no_mprotect"; then
           cl_cv_func_mprotect_works=yes
         else
           cl_cv_func_mprotect_works=no
         fi
       else
         dnl When cross-compiling, assume the known behaviour.
         dnl If we don't know, don't assume anything.
         case "$host_os" in
           aix* | cygwin* | darwin* | hpux* | irix* | linux* | solaris*)
             cl_cv_func_mprotect_works="guessing yes" ;;
           mingw* | windows*)
             cl_cv_func_mprotect_works="guessing no" ;;
           *)
             cl_cv_func_mprotect_works="guessing no" ;;
         esac
       fi
      ])
    case "$cl_cv_func_mprotect_works" in
      *yes)
        AC_DEFINE([HAVE_WORKING_MPROTECT], [1],
          [have a working mprotect() function])
        ;;
    esac
  fi

  dnl Test how to use the mprotect function to make memory executable.
  dnl Test against the mprotect limitations found in PaX enabled Linux kernels
  dnl and HardenedBSD.
  case "$ffcall_cv_codeexec" in
    *yes | irrelevant) ;;
    *)
      case "$ac_cv_func_mprotect--$cl_cv_func_mprotect_works" in
        yes--*yes)
          AC_CACHE_CHECK([whether mprotect can make malloc()ed memory executable],
            [ffcall_cv_malloc_mprotect_can_exec],
            [dnl On RHEL 6 / CentOS 6 with SELinux enabled, the result of
             dnl this test depends on SELinux flags that can be changed at
             dnl runtime: By default, the result is 'no'. However, when the flag
             dnl allow_execheap is turned on, the result is 'yes'. But the flag
             dnl can be turned off again at any moment.
             if test "$cross_compiling" != yes -a -d /etc/selinux; then
               ffcall_cv_malloc_mprotect_can_exec='determined by SELinux at runtime'
             else
               AC_RUN_IFELSE(
                 [AC_LANG_SOURCE([[
                    #include <errno.h>
                    #include <stdlib.h>
                    /* Declare getpagesize().  */
                    #ifdef HAVE_UNISTD_H
                     #include <unistd.h>
                    #endif
                    #ifdef __hpux
                     extern
                     #ifdef __cplusplus
                     "C"
                     #endif
                     int getpagesize (void);
                    #endif
                    #include <fcntl.h>
                    /* Declare mprotect().  */
                    #include <sys/mman.h>
                    int
                    main ()
                    {
                      unsigned int pagesize = getpagesize ();
                      char *p = (char *) malloc (50);
                      int ret;
                      if (p == (char*) -1)
                        /* malloc is not working as expected. */
                        return 1;
                      p[5] = 0x77;
                      ret = mprotect (p - ((unsigned int) p & (pagesize - 1)), pagesize, PROT_READ | PROT_WRITE | PROT_EXEC);
                      if (ret < 0
                          && (errno == EACCES || errno == ENOMEM || errno == ENOTSUP))
                        /* mprotect is forbidden to make malloc()ed pages executable that were writable earlier. */
                        return 2;
                      return 0;
                    }
                    ]])
                 ],
                 [ffcall_cv_malloc_mprotect_can_exec=yes],
                 [ffcall_cv_malloc_mprotect_can_exec=no],
                 [dnl When cross-compiling, assume SELinux on Linux.
                  dnl If we don't know, assume the worst.
                  case "$host_os" in
                    linux*)
                      ffcall_cv_malloc_mprotect_can_exec='determined by SELinux at runtime' ;;
                    aix* | cygwin* | darwin* | irix* | solaris*)
                      ffcall_cv_malloc_mprotect_can_exec="guessing yes" ;;
                    *)
                      ffcall_cv_malloc_mprotect_can_exec="guessing no" ;;
                  esac
                 ])
             fi
            ])
          case "$ffcall_cv_malloc_mprotect_can_exec" in
            *yes)      MPROTECT_AFTER_MALLOC_CAN_EXEC=1 ;;
            *no)       MPROTECT_AFTER_MALLOC_CAN_EXEC=0 ;;
            *runtime*) MPROTECT_AFTER_MALLOC_CAN_EXEC='-1' ;;
          esac
          AC_DEFINE_UNQUOTED([HAVE_MPROTECT_AFTER_MALLOC_CAN_EXEC], [$MPROTECT_AFTER_MALLOC_CAN_EXEC],
            [have an mprotect() function that can make malloc()ed memory pages executable])
          case "$ffcall_cv_malloc_mprotect_can_exec" in
            *yes) ;;
            *)
              AC_CACHE_CHECK([whether mprotect can make mmap()ed memory executable],
                [ffcall_cv_mmap_mprotect_can_exec],
                [dnl On RHEL 6 / CentOS 6 with SELinux enabled, the result of
                 dnl this test depends on SELinux flags that can be changed at
                 dnl runtime: By default, the result is 'yes'. However, when the flags
                 dnl allow_execmem and allow_execstack are turned off, the result is
                 dnl 'no'.
                 if test "$cross_compiling" != yes -a -d /etc/selinux; then
                   ffcall_cv_mmap_mprotect_can_exec='determined by SELinux at runtime'
                 else
                   AC_RUN_IFELSE(
                     [AC_LANG_SOURCE([[
                        #include <errno.h>
                        #include <stdlib.h>
                        /* Declare getpagesize().  */
                        #ifdef HAVE_UNISTD_H
                         #include <unistd.h>
                        #endif
                        #ifdef __hpux
                         extern
                         #ifdef __cplusplus
                         "C"
                         #endif
                         int getpagesize (void);
                        #endif
                        #include <fcntl.h>
                        /* Declare mmap(), mprotect().  */
                        #include <sys/mman.h>
                        #ifndef MAP_FILE
                         #define MAP_FILE 0
                        #endif
                        #ifndef MAP_VARIABLE
                         #define MAP_VARIABLE 0
                        #endif
                        int
                        main ()
                        {
                          unsigned int pagesize = getpagesize ();
                          char *p;
                          int ret;
                        #if HAVE_MMAP_ANONYMOUS
                          p = (char *) mmap (NULL, pagesize, PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANONYMOUS | MAP_VARIABLE, -1, 0);
                        #elif HAVE_MMAP_DEVZERO
                          int zero_fd = open("/dev/zero", O_RDONLY, 0666);
                          if (zero_fd < 0)
                            return 1;
                          p = (char *) mmap (NULL, pagesize, PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_FILE | MAP_VARIABLE, zero_fd, 0);
                        #else
                          ??
                        #endif
                          if (p == (char*) -1)
                            /* mmap is not working as expected. */
                            return 1;
                          p[5] = 0x77;
                          ret = mprotect (p, pagesize, PROT_READ | PROT_WRITE | PROT_EXEC);
                          if (ret < 0
                              && (errno == EACCES || errno == ENOMEM || errno == ENOTSUP))
                            /* mprotect is forbidden to make mmap()ed pages executable that were writable earlier. */
                            return 2;
                          return 0;
                        }
                        ]])
                     ],
                     [ffcall_cv_mmap_mprotect_can_exec=yes],
                     [ffcall_cv_mmap_mprotect_can_exec=no],
                     [dnl When cross-compiling, assume SELinux on Linux.
                      dnl If we don't know, assume the worst.
                      case "$host_os" in
                        linux*) ffcall_cv_mmap_mprotect_can_exec='determined by SELinux at runtime' ;;
                        *)      ffcall_cv_mmap_mprotect_can_exec="guessing no" ;;
                      esac
                     ])
                 fi
                ])
              case "$ffcall_cv_mmap_mprotect_can_exec" in
                *yes)      MPROTECT_AFTER_MMAP_CAN_EXEC=1 ;;
                *no)       MPROTECT_AFTER_MMAP_CAN_EXEC=0 ;;
                *runtime*) MPROTECT_AFTER_MMAP_CAN_EXEC='-1' ;;
              esac
              AC_DEFINE_UNQUOTED([HAVE_MPROTECT_AFTER_MMAP_CAN_EXEC], [$MPROTECT_AFTER_MMAP_CAN_EXEC],
                [have an mprotect() function that can make mmap()ed memory pages executable])
              case "$ffcall_cv_mmap_mprotect_can_exec" in
                *yes) ;;
                *)
                  case "$host_os" in
                    darwin*)
                      # This is expected to work on macOS only.
                      # Here we use the approach with Mach primitives that exist
                      # since macOS 10.4. It was invented by Joelle van Dyne for QEMU.
                      # It is a lot simpler than the MAP_JIT approach that is
                      # propagated by Apple but incompletely documented and
                      # changes behaviour whenever Apple wants to improve "security".
                      AC_CACHE_CHECK([whether a shared mmap with macOS primitives can make memory pages executable],
                        [ffcall_cv_mmap_shared_macos_can_exec],
                        [AC_RUN_IFELSE(
                           [AC_LANG_SOURCE([[
                              #include <fcntl.h>
                              #include <stdlib.h>
                              /* Declare getpagesize().  */
                              #include <unistd.h>
                              /* Declare mmap().  */
                              #include <sys/mman.h>
                              /* Declare mach_vm_remap.  */
                              #include <mach/mach.h>
                              /* Declaring it ourselves is easier than
                                 including <mach/mach_vm.h>.  */
                              extern
                              #ifdef __cplusplus
                              "C"
                              #endif
                              kern_return_t mach_vm_remap (vm_map_t target_task,
                                                           mach_vm_address_t *target_address, /* in/out */
                                                           mach_vm_size_t size,
                                                           mach_vm_offset_t mask,
                                                           int flags,
                                                           vm_map_t src_task,
                                                           mach_vm_address_t src_address,
                                                           boolean_t copy,
                                                           vm_prot_t *cur_protection, /* out */
                                                           vm_prot_t *max_protection, /* out */
                                                           vm_inherit_t inheritance);
                              int
                              main ()
                              {
                                unsigned int pagesize = getpagesize ();
                                char *pw;
                                char *px;
                                pw = (char *) mmap (NULL, pagesize, PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANON, -1, 0);
                                if (pw == (char*) -1)
                                  return 2;
                                pw[5] = 0xc3;
                                {
                                  vm_map_t self = mach_task_self ();
                                  mach_vm_address_t target_address = 0;
                                  vm_prot_t cur_prot;
                                  vm_prot_t max_prot;
                                  kern_return_t ret = mach_vm_remap (self, &target_address, pagesize, 0, VM_FLAGS_ANYWHERE, self, (mach_vm_address_t) (unsigned long) pw, FALSE, &cur_prot, &max_prot, VM_INHERIT_NONE);
                                  if (ret != KERN_SUCCESS)
                                    return 3;
                                  px = (char *) (unsigned long) target_address;
                                }
                                if (mprotect (px, pagesize, PROT_READ | PROT_EXEC) < 0)
                                  return 4;
                                if ((char)px[5] != (char)0xc3)
                                  return 5;
                                /* On i386 and x86_64 this is a 'ret' instruction that we can invoke. */
                              #if (defined __i386 || defined __i386__ || defined _I386 || defined _M_IX86 || defined _X86_) || (defined __x86_64__ || defined __amd64__)
                                ((void (*) (void)) (px + 5)) ();
                              #endif
                                return 0;
                              }
                              ]])
                           ],
                           [ffcall_cv_mmap_shared_macos_can_exec=yes],
                           [ffcall_cv_mmap_shared_macos_can_exec=no],
                           [dnl When cross-compiling, assume yes, since this is the result
                            dnl on all the platforms where we have tested it.
                            ffcall_cv_mmap_shared_macos_can_exec="guessing yes"
                           ])
                        ])
                      case "$ffcall_cv_mmap_shared_macos_can_exec" in
                        *yes)
                          AC_DEFINE([HAVE_MMAP_SHARED_MACOS_CAN_EXEC], [1],
                            [have an mmap() function that, together with mach_vm_remap(), can make memory pages executable])
                          ;;
                      esac
                      ;;
                  esac
                  AC_CACHE_CHECK([whether a shared mmap of a RAM-only region can make memory pages executable],
                    [ffcall_cv_mmap_shared_memfd_can_exec],
                    [filename="trampdata$$"
                     AC_RUN_IFELSE(
                       [AC_LANG_SOURCE([[
                          /* Enable the memfd_create declaration on Linux.  */
                          #ifndef _GNU_SOURCE
                           #define _GNU_SOURCE 1
                          #endif
                          #include <fcntl.h>
                          #include <stdlib.h>
                          /* Declare getpagesize().  */
                          #ifdef HAVE_UNISTD_H
                           #include <unistd.h>
                          #endif
                          /* Declare mmap() and memfd_create().  */
                          #include <sys/mman.h>
                          #ifndef MAP_FILE
                           #define MAP_FILE 0
                          #endif
                          #ifndef MAP_VARIABLE
                           #define MAP_VARIABLE 0
                          #endif
                          int
                          main ()
                          {
                            unsigned int pagesize = getpagesize ();
                            int fd;
                            char *pw;
                            char *px;
                            fd = memfd_create ("$filename", 0);
                            if (fd < 0)
                              return 1;
                            if (ftruncate (fd, pagesize) < 0)
                              return 2;
                            pw = (char *) mmap (NULL, pagesize, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
                            if (pw == (char*) -1)
                              return 3;
                            pw[5] = 0xc3;
                            px = (char *) mmap (NULL, pagesize, PROT_READ | PROT_EXEC, MAP_SHARED, fd, 0);
                            if (px == (char*) -1)
                              return 4;
                            if ((char)px[5] != (char)0xc3)
                              return 5;
                            /* On i386 and x86_64 this is a 'ret' instruction that we can invoke. */
                          #if (defined __i386 || defined __i386__ || defined _I386 || defined _M_IX86 || defined _X86_) || (defined __x86_64__ || defined __amd64__)
                            ((void (*) (void)) (px + 5)) ();
                          #endif
                            return 0;
                          }
                          ]])
                       ],
                       [ffcall_cv_mmap_shared_memfd_can_exec=yes],
                       [ffcall_cv_mmap_shared_memfd_can_exec=no],
                       [dnl When cross-compiling, assume yes, since this is the result
                        dnl on all the platforms where we have tested it.
                        ffcall_cv_mmap_shared_memfd_can_exec="guessing yes"
                       ])
                    ])
                  case "$ffcall_cv_mmap_shared_memfd_can_exec" in
                    *yes)
                      AC_DEFINE([HAVE_MMAP_SHARED_MEMFD_CAN_EXEC], [1],
                        [have an mmap() function that, with MAP_SHARED on a memfd descriptor, can make memory pages executable])
                      ;;
                  esac
                  AC_CACHE_CHECK([whether a shared mmap of a file can make memory pages executable],
                    [ffcall_cv_mmap_shared_posix_can_exec],
                    [filename="/tmp/trampdata$$.data"
                     AC_RUN_IFELSE(
                       [AC_LANG_SOURCE([[
                          #include <fcntl.h>
                          #include <stdlib.h>
                          /* Declare getpagesize().  */
                          #ifdef HAVE_UNISTD_H
                           #include <unistd.h>
                          #endif
                          #ifdef __hpux
                           extern
                           #ifdef __cplusplus
                           "C"
                           #endif
                           int getpagesize (void);
                          #endif
                          /* Declare mmap().  */
                          #include <sys/mman.h>
                          #ifndef MAP_FILE
                           #define MAP_FILE 0
                          #endif
                          #ifndef MAP_VARIABLE
                           #define MAP_VARIABLE 0
                          #endif
                          int
                          main ()
                          {
                            unsigned int pagesize = getpagesize ();
                            int fd;
                            char *pw;
                            char *px;
                            fd = open ("$filename", O_CREAT | O_RDWR | O_TRUNC, 0700);
                            if (fd < 0)
                              return 1;
                            if (ftruncate (fd, pagesize) < 0)
                              return 2;
                            pw = (char *) mmap (NULL, pagesize, PROT_READ | PROT_WRITE, MAP_SHARED | MAP_FILE | MAP_VARIABLE, fd, 0);
                            if (pw == (char*) -1)
                              return 3;
                            pw[5] = 0xc3;
                            px = (char *) mmap (NULL, pagesize, PROT_READ | PROT_EXEC, MAP_SHARED | MAP_FILE | MAP_VARIABLE, fd, 0);
                            if (px == (char*) -1)
                              return 4;
                            if ((char)px[5] != (char)0xc3)
                              return 5;
                            /* On i386 and x86_64 this is a 'ret' instruction that we can invoke. */
                          #if (defined __i386 || defined __i386__ || defined _I386 || defined _M_IX86 || defined _X86_) || (defined __x86_64__ || defined __amd64__)
                            ((void (*) (void)) (px + 5)) ();
                          #endif
                            return 0;
                          }
                          ]])
                       ],
                       [ffcall_cv_mmap_shared_posix_can_exec=yes],
                       [ffcall_cv_mmap_shared_posix_can_exec=no],
                       [dnl When cross-compiling, assume yes, since this is the result
                        dnl on all the platforms where we have tested it.
                        ffcall_cv_mmap_shared_posix_can_exec="guessing yes"
                       ])
                     rm -f "$filename"
                    ])
                  case "$ffcall_cv_mmap_shared_posix_can_exec" in
                    *yes)
                      AC_DEFINE([HAVE_MMAP_SHARED_POSIX_CAN_EXEC], [1],
                        [have an mmap() function that, with MAP_SHARED on a file, can make memory pages executable])
                      ;;
                  esac
                  ;;
              esac
              ;;
          esac
          ;;
      esac
      ;;
  esac
])
