/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * gtkemuvt102: vt102 emulation for GtkTty
 * Copyright (C) 1997 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "gtkemuvt102.h"
#include <stdio.h>


/* --- limits --- */
#define	MAX_PARAMS	(16)

/* --- strings --- */
#define VT100ID		("\033[?1;2c")
#define	VT102ID		("\033[?6c")
#define	VT_OK		("\033[0n")
#define	VT_TERM		("vt102")

/* --- macros --- */
#define	gtk_emu_vt102_gotoxry(t,e,x,y)	gtk_emu_vt102_gotoxy((t), (e), (x), (e)->relative_origin ? (((GtkTerm*) (t))->top + (y)) : (y))

/* --- typedefs --- */
typedef enum
{
  ESC_Normal,
  ESC_Escape,
  ESC_NonStd,
  ESC_Percent,
  ESC_Square,
  ESC_GetParams,
  ESC_GotParams,
  ESC_Set_G0,
  ESC_Set_G1,
  ESC_Hash,
  ESC_Palette,
  ESC_FuncKey
} TtyEscStates;

typedef	struct
{
  TtyEscStates	state;
  gboolean	disp_ctrl;
  guint32	display_bitmap;
  guint32	display_bitmap_restricted;
  gboolean	insert_mode;
  gboolean	cr_and_lf;
  gboolean	need_wrap;
  gboolean	dec_priv_mode;
  gboolean	relative_origin;
  guint		cur_x;
  guint		cur_y;
  GtkCursorMode	cursor_mode;
  gboolean	cursor_blinking;
  gulong	tab_stop[5];
  guint		n_params;
  gulong	param[MAX_PARAMS];
} GtkTtyEmuData;

/* --- structures --- */
static	GtkTtyEmuData emu_data_default =
{
  ESC_Normal		/* state */,
  FALSE			/* disp_ctrl */,
  0xffffffff &		/* these are taken from the linux console */
  ~(0x00000000 |
    ( 1 << '\0') |
    ( 1 << '\b') |
    ( 1 << '\n') |
    ( 1 << '\f') |
    ( 1 << '\r') |
    ( 1 <<  14 ) | /* shift out */
    ( 1 <<  15 ) | /* shift in */
    ( 1 <<  27 ) | /* escape */
    0)			/* display_bitmap */,
  0xffffffff &		/* these are taken from the linux console */
  ~(0x00000000 |
    ( 1 << '\0') |
    ( 1 << '\a') |
    ( 1 << '\b') |
    ( 1 << '\t') |
    ( 1 << '\n') |
    ( 1 << '\v') |
    ( 1 << '\f') |
    ( 1 << '\r') |
    ( 1 <<  14 ) | /* shift out */
    ( 1 <<  15 ) | /* shift in */
    ( 1 <<  24 ) | /* CAN */
    ( 1 <<  26 ) | /* SUB */
    ( 1 <<  27 ) | /* escape */
    0)			/* display_bitmap_restricted */,
  FALSE			/* insert_mode */,
  FALSE			/* cr_and_lf */,
  FALSE			/* need_wrap */,
  FALSE			/* dec_priv_mode */,
  FALSE			/* relative_origin */,
  0			/* cur_x */,
  0			/* cur_y */,
  GTK_CURSOR_UNDERLINE	/* cursor_mode */,
  TRUE			/* cursor_blinking */,
  {
    0x01010100		/* tab_stop[0] */,
    0x01010101		/* tab_stop[1] */,
    0x01010101		/* tab_stop[2] */,
    0x01010101		/* tab_stop[3] */,
    0x01010101		/* tab_stop[4] */,
  },
  0			/* n_params */,
  {
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
  }			/* param[0...15] */,
};


/* --- prototypes --- */
static	void		gtk_emu_vt102_lf	    (GtkTty		*tty,
						     GtkTtyEmuData	*emu);
static	void		gtk_emu_vt102_reverse_lf    (GtkTty		*tty,
						     GtkTtyEmuData	*emu);
static	void		gtk_emu_vt102_cr	    (GtkTty		*tty,
						     GtkTtyEmuData	*emu);
static	void		gtk_emu_vt102_bs	    (GtkTty		*tty,
						     GtkTtyEmuData	*emu);
static	void		gtk_emu_vt102_gotoxy	    (GtkTty		*tty,
						     GtkTtyEmuData	*emu,
						     guint		new_x,
						     guint		new_y);
static	void		gtk_emu_vt102_set_mode	    (GtkTty		*tty,
						     GtkTtyEmuData	*emu,
						     gboolean		on_off);
static	void		gtk_emu_vt102_set_term_mode (GtkTty		*tty,
						     GtkTtyEmuData	*emu);
static	void		gtk_emu_vt102_linux_setterm (GtkTty		*tty,
						     GtkTtyEmuData	*emu);



/* --- functions --- */

gpointer
gtk_emu_vt102_new (void)
{
  GtkTtyEmuData *emu_data;

  emu_data = g_new (GtkTtyEmuData, 1);
  *emu_data = emu_data_default;

  return emu_data;
}

void
gtk_emu_vt102_free (gpointer write_func_data)
{
  g_free (write_func_data);
}

gchar*
gtk_emu_vt102_reset (GtkTty	*tty,
		     gpointer	write_func_data,
		     gboolean	blank_screen)
{
  GtkTtyEmuData *emu;
  
  emu = write_func_data;
  
  *emu = emu_data_default;
  tty->freeze_leds = FALSE;
  gtk_tty_leds_changed (tty);
  gtk_term_set_cursor_mode (GTK_TERM (tty), emu->cursor_mode, emu->cursor_blinking);
  gtk_term_reset (GTK_TERM (tty));
  if (blank_screen)
    gtk_term_clear (GTK_TERM (tty), TRUE, TRUE);

  return VT_TERM;
}

guint
gtk_emu_vt102 (GtkTty	      *tty,
	     gpointer	    write_func_data,
	     guchar	    *buffer,
	     guint	    count)
{
  /* this function is modeled much after
   * linux-2.0.29/drivers/char/console.c:con_write().
   * for the moment we skip unicode and charset conversions
   */
  
  GtkTerm *term;
  guint n_processed;
  GtkTtyEmuData *emu;
  guint old_x, old_y;

  term = GTK_TERM (tty);
  emu = write_func_data;

  gtk_term_set_cursor_mode (term, GTK_CURSOR_INVISIBLE, emu->cursor_blinking);

  gtk_term_get_cursor (term, &emu->cur_x, &emu->cur_y);
  old_x = emu->cur_x;
  old_y = emu->cur_y;

  n_processed = 0;
  while (n_processed < count)
  {
    guchar ch;
    gboolean display_ok;

    if (emu->cur_x != old_x ||
	emu->cur_y != old_y)
    {
      gtk_term_set_cursor (term, emu->cur_x, emu->cur_y);
      old_x = emu->cur_x;
      old_y = emu->cur_y;
    }

    ch = buffer[n_processed++];
    display_ok = (ch
		  && (ch >= 32 ||
		      (((emu->disp_ctrl ?
			 emu->display_bitmap :
			 emu->display_bitmap_restricted) >> ch) & 1))
		  && (ch != 127 || emu->disp_ctrl)
		  && (ch != 128+27));
    
    /* write normal char
     */
    if (emu->state == ESC_Normal && display_ok)
    {
      if (emu->need_wrap)
      {
	gtk_emu_vt102_cr (tty, emu);
	gtk_emu_vt102_lf (tty, emu);
      }

      /* FIXME: autowrap hint */
      emu->need_wrap = gtk_term_putc (term, ch, emu->insert_mode);
      gtk_term_get_cursor (term, &emu->cur_x, &emu->cur_y);

      continue;
    }

    /* control characters can be used in the middle of an escape sequence
     */
    switch (ch)
    {
    case  0:
      continue;

    case  '\a': /* audible bell */
      gtk_term_bell (term);
      continue;

    case  '\b': /* backspace */
      gtk_emu_vt102_bs (tty, emu);
      continue;

    case  '\t': /* tabulator */
      while (emu->cur_x < MIN (160, term->term_width) - 1)
      {
	emu->cur_x++;
	if (emu->tab_stop[emu->cur_x >> 5] & (1 << (emu->cur_x & 31)))
	{
	  emu->need_wrap = FALSE;
	  break;
	}
      }
      gtk_term_set_cursor (term, emu->cur_x, emu->cur_y);
      continue;

    case  '\n': /* linefeed */
    case  '\v': /* vertical tabulator */
    case  '\f': /* formfeed */
      gtk_emu_vt102_lf (tty, emu);
      if (emu->cr_and_lf)
	gtk_emu_vt102_cr (tty, emu);
      continue;

    case  '\r': /* carriage return */
      gtk_emu_vt102_cr (tty, emu);
      continue;

    case  14: /* shift out */
      /* FIXME: charset */
      continue;

    case  15: /* shift in */
      /* FIXME: charset */
      continue;

    case  24: /* cancel */
    case  26: /* SUB */
      emu->state = ESC_Normal;
      continue;

    case  27: /* escape */
      emu->state = ESC_Escape;
      continue;

    case  128 + 27: /* ??? */
      emu->state = ESC_Square;
      continue;
    }

    /* handle various escape states
     */
    switch (emu->state)
    {
    case  ESC_Escape:
      emu->state = ESC_Normal;
      switch (ch)
      {
      case  '[':
	emu->state = ESC_Square;
	continue;
	
      case  ']':
	emu->state = ESC_NonStd;
	continue;
	
      case  '%':
	emu->state = ESC_Percent;
	continue;
	
      case  'E':
	gtk_emu_vt102_cr (tty, emu);
	gtk_emu_vt102_lf (tty, emu);
	continue;
	
      case  'M':
	gtk_emu_vt102_reverse_lf (tty, emu);
	continue;
	
      case  'D':
	gtk_emu_vt102_lf (tty, emu);
	continue;
	
      case 'H':
	emu->tab_stop[MIN (160, emu->cur_x) >> 5] |= (1 << (emu->cur_x & 31));
	continue;
	
      case 'Z':
	gtk_tty_put_in (tty, VT102ID, sizeof (VT102ID) - 1);
	printf ("DEBUG: `\033%s' (%d)\n", VT102ID, sizeof (VT102ID) - 1);
	continue;
	
      case '7':
	gtk_term_save_cursor (term);
	continue;
	
      case '8':
	gtk_term_restore_cursor (term);
	gtk_term_get_cursor (term, &emu->cur_x, &emu->cur_y);
	emu->need_wrap = FALSE;
	continue;
	
      case '(':
	emu->state = ESC_Set_G0;
	continue;
	
      case ')':
	emu->state = ESC_Set_G1;
	continue;
	
      case '#':
	emu->state = ESC_Hash;
	continue;
	
      case 'c':
	gtk_emu_vt102_reset (tty, emu, TRUE);
	continue;
	
      case '>':
	/* FIXME: numeric keypad */
	continue;
	
      case '=':
	/* FIXME: application keypad */
	continue;
      }
      continue;

    case  ESC_NonStd:
      if (ch == 'P') /* palette escape sequence */
      {
	for (emu->n_params = 0; emu->n_params < MAX_PARAMS; emu->n_params++)
	  emu->param[emu->n_params] = 0;
	emu->n_params = 0;
	emu->state = ESC_Palette;
	continue;
      }
      else if (ch == 'R') /* reset palette */
	emu->state = ESC_Normal;
      else
	emu->state = ESC_Normal;
      continue;

    case ESC_Palette:
      if ((ch >= '0' && ch <= '9') || (ch >= 'A' && ch <= 'F') || (ch >= 'a' &&ch <= 'f'))
      {
	emu->param[emu->n_params++] = (ch > '9' ? (ch & 0xDF) - 'A' + 10 : ch - '0');
	if (emu->n_params == 7)
	  emu->state = ESC_Normal;
      }
      else
	emu->state = ESC_Normal;
      continue;

    case ESC_Square:
      for (emu->n_params = 0; emu->n_params < MAX_PARAMS; emu->n_params++)
	emu->param[emu->n_params] = 0;
      emu->n_params = 0;
      emu->state = ESC_GetParams;
      if (ch == '[') /* function key */
      {
	emu->state = ESC_FuncKey;
	continue;
      }
      emu->dec_priv_mode = ch == '?';
      if (emu->dec_priv_mode)
	continue;
    case  ESC_GetParams:
      if (ch == ';' && emu->n_params < MAX_PARAMS - 1)
      {
	emu->n_params++;
	continue;
      }
      else if (ch >= '0' && ch <= '9')
      {
	emu->param[emu->n_params] *= 10;
	emu->param[emu->n_params] += ch - '0';
	continue;
      }
      else
	emu->state = ESC_GotParams;
    case  ESC_GotParams:
      emu->state = ESC_Normal;
      switch (ch)
      {

      case  'h':
	gtk_emu_vt102_set_mode (tty, emu, TRUE);
	continue;

      case 'l':
	gtk_emu_vt102_set_mode (tty, emu, FALSE);
	continue;

      case 'n':
	if (!emu->dec_priv_mode)
	  if (emu->param[0] == 5)
	    gtk_tty_put_in (tty, VT_OK, sizeof (VT_OK) - 1);
	  else if (emu->param[0] == 6)
	  {
	    guchar string[40];

	    sprintf (string,
		     "\033[%d;%dR",
		     emu->cur_y + (emu->relative_origin ? term->top + 1 : 1),
		     emu->cur_x + 1);
	    gtk_tty_put_in (tty, string, strlen (string));
	  }
	continue;
      }
      if (emu->dec_priv_mode)
      {
	emu->dec_priv_mode = FALSE;
	continue;
      }
      switch(ch)
      {

      case  'G':
      case  '`':
	if (emu->param[0])
	  emu->param[0]--;
	gtk_emu_vt102_gotoxy (tty, emu, emu->param[0], emu->cur_y);
	continue;

      case  'A':
	if (!emu->param[0])
	  emu->param[0] = 1;
	if (emu->param[0] > emu->cur_y)
	  emu->param[0] = emu->cur_y;
	gtk_emu_vt102_gotoxy (tty, emu, emu->cur_x, emu->cur_y - emu->param[0]);
	continue;

      case  'B':
      case  'e':
	if (!emu->param[0])
	  emu->param[0] = 1;
	gtk_emu_vt102_gotoxy (tty, emu, emu->cur_x, emu->cur_y + emu->param[0]);
	continue;

      case  'C':
      case  'a':
	if (!emu->param[0])
	  emu->param[0] = 1;
	gtk_emu_vt102_gotoxy (tty, emu, emu->cur_x + emu->param[0], emu->cur_y);
	continue;

      case  'D':
	if (!emu->param[0])
	  emu->param[0] = 1;
	if (emu->param[0] > emu->cur_x)
	  emu->param[0] = emu->cur_x;
	gtk_emu_vt102_gotoxy (tty, emu, emu->cur_x - emu->param[0], emu->cur_y);
	continue;

      case  'E':
	if (!emu->param[0])
	  emu->param[0] = 1;
	gtk_emu_vt102_gotoxy (tty, emu, 0, emu->cur_y + emu->param[0]);
	continue;
	
      case  'F':
	if (!emu->param[0])
	  emu->param[0] = 1;
	gtk_emu_vt102_gotoxy (tty, emu, 0, emu->cur_y - emu->param[0]);
	continue;
	
      case  'd':
	if (emu->param[0])
	  emu->param[0]--;
	gtk_emu_vt102_gotoxry (tty, emu, emu->cur_x, emu->param[0]);
	continue;
	
      case  'H':
      case  'f':
	if (emu->param[0])
	  emu->param[0]--;
	if (emu->param[1])
	  emu->param[1]--;
	gtk_emu_vt102_gotoxry (tty, emu, emu->param[1], emu->param[0]);
	continue;
	
      case  'J':
	gtk_term_clear (term,
			emu->param[0] == 1 ||
			emu->param[0] == 2,
			emu->param[0] == 0 ||
			emu->param[0] == 2);
	emu->need_wrap = FALSE;
	continue;
	
      case  'K':
	gtk_term_clear_line (term,
			     emu->param[0] == 1 ||
			     emu->param[0] == 2,
			     emu->param[0] == 0 ||
			     emu->param[0] == 2);
	emu->need_wrap = FALSE;
	continue;
	
      case  'L':
	gtk_term_insert_lines (term, emu->param[0]);
	emu->need_wrap = FALSE;
	continue;
	
      case  'M':
	gtk_term_delete_lines (term, emu->param[0]);
	emu->need_wrap = FALSE;
	continue;
	
      case  'P':
	gtk_term_delete_chars (term, emu->param[0]);
	emu->need_wrap = FALSE;
	continue;
	
      case  'c':
	if (!emu->param[0])
	  gtk_tty_put_in (tty, VT102ID, sizeof (VT102ID) - 1);
	continue;

      case  'g':
	if (!emu->param[0])
	  emu->tab_stop[MIN (emu->cur_x, 160) >> 5] &= ~(1 << (MIN (emu->cur_x, 160) & 31));
	else if (emu->param[0] == 3)
	{
	  emu->tab_stop[0] = 0;
	  emu->tab_stop[1] = 0;
	  emu->tab_stop[2] = 0;
	  emu->tab_stop[3] = 0;
	  emu->tab_stop[4] = 0;
	}
	continue;

      case  'm':
	gtk_emu_vt102_set_term_mode (tty, emu);
	continue;

      case  'q':
	if (emu->param[0] < 8)
	{
	  tty->freeze_leds = TRUE;
	  if (!emu->param[0])
	    tty->leds = GTK_TTY_STATE_NONE;
	  else
	    tty->leds = 1 << (emu->param[0] - 1);
	  gtk_tty_leds_changed (tty);
	}
	continue;

      case  'r':
	if (!emu->param[0])
	  emu->param[0] = 1;
	if (!emu->param[1])
	  emu->param[1] = term->term_height;
	if (emu->param[0] < emu->param[1] &&
	    emu->param[1] <= term->term_height)
	{
	  gtk_term_set_scroll_reg (term, emu->param[0] - 1, emu->param[1] - 1);
	  gtk_emu_vt102_gotoxry (tty, emu, 0, 0);
	}
	continue;

      case  's':
	gtk_term_save_cursor (term);
	continue;

      case  'u':
	gtk_term_restore_cursor (term);
	gtk_term_get_cursor (term, &emu->cur_x, &emu->cur_y);
	emu->need_wrap = FALSE;
	continue;

      case  'X':
	gtk_term_erase_chars (term, emu->param[0]);
	emu->need_wrap = FALSE;
	continue;

      case  '@':
	gtk_term_insert_chars (term, emu->param[0]);
	continue;

      case  ']':
	gtk_emu_vt102_linux_setterm (tty, emu);
	continue;
      }
      continue;

    case  ESC_Percent:
      emu->state = ESC_Normal;
      continue;

    case  ESC_FuncKey:
      emu->state = ESC_Normal;
      continue;

    case  ESC_Hash:
      emu->state = ESC_Normal;
      continue;

    case  ESC_Set_G0:
      emu->state = ESC_Normal;
      if (ch == '0')
      {
	/* FIXME: charset_G0 = GRAF_MAP */
      }
      else if (ch == 'B')
      {
	/* FIXME: charset_G0 = LAT1_MAP */
      }
      else if (ch == 'U')
      {
	/* FIXME: charset_G0 = IBMPC_MAP */
      }
      else if (ch == 'K')
      {
	/* FIXME: charset_G0 = USER_MAP */
      }
      continue;

    case  ESC_Set_G1:
      emu->state = ESC_Normal;
      if (ch == '0')
      {
	/* FIXME: charset_G1 = GRAF_MAP */
      }
      else if (ch == 'B')
      {
	/* FIXME: charset_G1 = LAT1_MAP */
      }
      else if (ch == 'U')
      {
	/* FIXME: charset_G1 = IBMPC_MAP */
      }
      else if (ch == 'K')
      {
	/* FIXME: charset_G1 = USER_MAP */
      }
      continue;

    case  ESC_Normal:
      continue;

    default:
      emu->state = ESC_Normal;
    }
  }

  gtk_term_set_cursor_mode (term, emu->cursor_mode, emu->cursor_blinking);

  return n_processed;
}

static void
gtk_emu_vt102_lf (GtkTty	 *tty,
		  GtkTtyEmuData	 *emu)
{
  if (emu->cur_y == ((GtkTerm*) tty)->bottom)
    gtk_term_scroll ((GtkTerm*) tty, 1, FALSE);
  else if (emu->cur_y < ((GtkTerm*) tty)->term_height - 1)
  {
    emu->cur_y++;
    gtk_term_set_cursor ((GtkTerm*) tty, emu->cur_x, emu->cur_y);
  }
  emu->need_wrap = FALSE;
}

static void
gtk_emu_vt102_reverse_lf (GtkTty	 *tty,
			  GtkTtyEmuData	 *emu)
{
  if (emu->cur_y == ((GtkTerm*) tty)->top)
    gtk_term_scroll (((GtkTerm*) tty), 1, TRUE);
  else if (emu->cur_y > 0)
  {
    emu->cur_y--;
    gtk_term_set_cursor ((GtkTerm*) tty, emu->cur_x, emu->cur_y);
  }
  emu->need_wrap = FALSE;
}

static void
gtk_emu_vt102_cr (GtkTty	 *tty,
		  GtkTtyEmuData	 *emu)
{
  if (emu->cur_x != 0)
  {
    emu->cur_x = 0;
    gtk_term_set_cursor ((GtkTerm*) tty, emu->cur_x, emu->cur_y);
  }
  emu->need_wrap = FALSE;
}

static void
gtk_emu_vt102_bs (GtkTty	 *tty,
		  GtkTtyEmuData	 *emu)
{
  if (emu->cur_x)
  {
    emu->cur_x--;
    gtk_term_set_cursor ((GtkTerm*) tty, emu->cur_x, emu->cur_y);
    emu->need_wrap = FALSE;
  }
}

static void
gtk_emu_vt102_gotoxy (GtkTty	       *tty,
		      GtkTtyEmuData    *emu,
		      guint	       new_x,
		      guint	       new_y)
{
  guint min_y, max_y;
  guint height, width;
  GtkTerm *term;
  
  term = GTK_TERM(tty);
  height = term->term_height;
  width = term->term_width;
  
  if (new_x >= width)
    emu->cur_x = width - 1;
  else
    emu->cur_x = new_x;
  
  if (emu->relative_origin)
  {
    min_y = term->top;
    max_y = term->bottom;
  }
  else
  {
    min_y = 0;
    max_y = height - 1;
  }
  
  if (new_y < min_y)
    emu->cur_y = min_y;
  else if (new_y > max_y)
    emu->cur_y = max_y;
  else
    emu->cur_y = new_y;
  emu->need_wrap = FALSE;
  
  gtk_term_set_cursor (term, emu->cur_x, emu->cur_y);
}

static void
gtk_emu_vt102_linux_setterm (GtkTty		*tty,
			     GtkTtyEmuData	*emu)
{
  switch(emu->param[0])
  {
    
  case 1:
    /* set color for underline mode */
    break;
    
  case 2:
    /* set color for half intensity mode */
    break;
    
  case 8:
    /* store colors as defaults */
    break;
    
  case 9:
    /* set blanking interval */
    break;
    
  case 10:
    /* set bell frequency in Hz from param[1] */
    break;
    
  case 11:
    /* set bell duration in msec from param[1] */
    break;
    
  case 12:
    /* bring specified console to the front,
     * we could "misuse" this for notebooks... ;)
     */
    break;
    
  case 13:
    /* unblank the screen */
    break;
    
  case 14:
    /* set vesa powerdown interval */
    break;
  }
}

static void
gtk_emu_vt102_set_mode (GtkTty		*tty,
			GtkTtyEmuData	*emu,
			gboolean	on_off)
{
  guint i;
  
  on_off = on_off != FALSE;
  
  for (i = 0; i <= emu->n_params; i++)
    if (emu->dec_priv_mode)
    {
      switch (emu->param[i]) /* DEC private modes set/reset */
      {
      case  1:
	/* FIXME: Cursor keys send ^[Ox/^[[x */
	break;
	
      case  3:
	/* 80/132 mode switch unimplemented */
	break;
	
      case  5:
	/* Inverted screen on/off */
	if (GTK_TERM (tty)->term_inversed != on_off)
	{
	  gtk_term_set_inversed (GTK_TERM (tty), on_off);
	}
	break;
	
      case  6:
	/* Origin relative/absolute */
	emu->relative_origin = on_off;
	gtk_emu_vt102_gotoxry (tty, emu, emu->cur_x, emu->cur_y);
	break;
	
      case  7:
	/* FIXME: autowrap hint */
	/* emu->autowrap_hint = on_off; */
	break;
	
      case  8:
	/* Autorepeat on/off unimplemented */
	break;
	
      case  9:
	/* FIXME: report_mouse on/off -> 1/0 */
	break;
	
      case  25:
	if (on_off)
	  emu->cursor_mode = emu_data_default.cursor_mode;
	else
	  emu->cursor_mode = GTK_CURSOR_INVISIBLE;
	gtk_term_set_cursor_mode (GTK_TERM (tty), emu->cursor_mode, emu->cursor_blinking);
	break;
	
      case  1000:
	/* FIXME: report_mouse on/off -> 2/0 */
	break;
      }
    }
    else
    {
      switch (emu->param[i]) /* ANSI modes set/reset */
      {
      case  3:
	emu->disp_ctrl = on_off;
	break;
	
      case  4:
	/* Insert Mode on/off */
	emu->insert_mode = on_off;
	break;
	
      case  20:
	/* Lf, Enter == CrLf/Lf */
	if (on_off)
	  emu->cr_and_lf = TRUE;
	else
	  emu->cr_and_lf = FALSE;
	break;
      }
    }
}

static void
gtk_emu_vt102_set_term_mode (GtkTty		*tty,
			     GtkTtyEmuData	*emu)
{
  guint i;
  
  for (i = 0; i <= emu->n_params; i++)
    switch (emu->param[i])
    {
    case  0:
      /* all attributes off */
      gtk_term_select_color (GTK_TERM (tty), GTK_TERM_MAX_COLORS - 1, 0);
      gtk_term_set_bold (GTK_TERM (tty), FALSE);
      gtk_term_set_dim (GTK_TERM (tty), FALSE);
      gtk_term_set_underline (GTK_TERM (tty), FALSE);
      /* gtk_term_set_blink (GTK_TERM (tty), FALSE); */
      gtk_term_set_reverse (GTK_TERM (tty), FALSE);
      break;
      
    case  1:
      gtk_term_set_bold (GTK_TERM (tty), TRUE);
      break;
      
    case  2:
      gtk_term_set_dim (GTK_TERM (tty), TRUE);
      break;
      
    case  4:
      gtk_term_set_underline (GTK_TERM (tty), TRUE);
      break;
      
    case  5:
      /* we can't blink... ;( */
      gtk_term_set_bold (GTK_TERM (tty), TRUE);
      break;
      
    case  7:
      gtk_term_set_reverse (GTK_TERM (tty), TRUE);
      break;
      
    case  10:
      /* FIXME: charset_G0/charset_G1 */
      emu->disp_ctrl = FALSE;
      break;
      
    case  11:
      /* FIXME: charset_G0/charset_G1 */
      emu->disp_ctrl = TRUE;
      break;
      
    case  12:
      /* FIXME: charset_G0/charset_G1 */
      emu->disp_ctrl = TRUE;
      break;
      
    case  21:
      gtk_term_set_bold (GTK_TERM (tty), FALSE);
      break;
      
    case  22:
      gtk_term_set_dim (GTK_TERM (tty), FALSE);
      break;
      
    case  24:
      gtk_term_set_underline (GTK_TERM (tty), FALSE);
      break;
      
    case  25:
      /* we can't blink... ;( */
      gtk_term_set_bold (GTK_TERM (tty), FALSE);
      break;
      
    case  27:
      gtk_term_set_reverse (GTK_TERM (tty), FALSE);
      break;
      
    case  30 ... 37:
      /* set foreground color */
      gtk_term_select_color (GTK_TERM (tty), emu->param[i] - 30, GTK_TERM (tty)->i_back);
      break;
      
    case  38:
      /* enable underscore mode: white foreground + underline */
      gtk_term_set_underline (GTK_TERM (tty), TRUE);
      break;
      
    case  39:
      /* disable underscore mode */
      gtk_term_set_underline (GTK_TERM (tty), FALSE);
      break;
      
    case  40 ... 47:
      /* set background color */
      gtk_term_select_color (GTK_TERM (tty), GTK_TERM (tty)->i_fore, emu->param[i] - 40);
      break;
      
    case  49:
      /* reset background color */
      gtk_term_select_color (GTK_TERM (tty), GTK_TERM (tty)->i_fore, 0);
      break;
    }
}
