/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"gleeditor.h"
#include	"gleprivate.h"
#include	"glegarg.h"
#include	<string.h>


/* --- signals --- */
enum
{
  SIGNAL_REFRESH_VALUES,
  SIGNAL_APPLY_VALUES,
  SIGNAL_RESET_VALUES,
  SIGNAL_RESTORE_VALUES,
  SIGNAL_LAST
};


/* --- prototypes --- */
static void	  gle_editor_destroy		(GtkObject		*object);
static void	  gle_editor_update_names	(GleEditor		*editor);
static void	  gle_editor_class_init		(GleEditorClass		*class);
static void	  gle_editor_init		(GleEditor		*editor);
static GtkWidget* gle_editor_field_create	(GleGArg		*garg,
						 const gchar		*name,
						 GtkBox			*parent_box);
static void	  gle_editor_field_refresh	(GtkWidget		*widget,
						 GtkArg			*arg);
static gint	  gle_editor_field_to_arg	(GtkWidget		*widget,
						 GtkArg			*arg);
static void	gle_editor_real_refresh_values	(GleEditor		*editor);
static void	gle_editor_real_apply_values	(GleEditor		*editor);
static void	gle_editor_real_reset_values	(GleEditor		*editor);
static void	gle_editor_real_restore_values	(GleEditor		*editor);



/* -- variables --- */
static GtkWindowClass	*parent_class = NULL;
static gint		 editor_signals[SIGNAL_LAST] = { 0 };
static const gchar	*gle_editor_garg_key = "gle-editor-garg-key";


/* --- functions --- */
GtkType
gle_editor_get_type (void)
{
  static GtkType editor_type = 0;

  if (!editor_type)
    {
      GtkTypeInfo editor_info =
      {
	"GleEditor",
	sizeof (GleEditor),
	sizeof (GleEditorClass),
	(GtkClassInitFunc) gle_editor_class_init,
	(GtkObjectInitFunc) gle_editor_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      editor_type = gtk_type_unique (gtk_window_get_type (), &editor_info);
    }

  return editor_type;
}

static void
gle_editor_class_init (GleEditorClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkWindowClass *window_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  window_class = (GtkWindowClass*) class;

  parent_class = gtk_type_class (gtk_window_get_type ());

  editor_signals[SIGNAL_REFRESH_VALUES] =
    gtk_signal_new ("refresh_values",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, refresh_values),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  editor_signals[SIGNAL_APPLY_VALUES] =
    gtk_signal_new ("apply_values",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, apply_values),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  editor_signals[SIGNAL_RESET_VALUES] =
    gtk_signal_new ("reset_values",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, reset_values),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  editor_signals[SIGNAL_RESTORE_VALUES] =
    gtk_signal_new ("restore_values",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, restore_values),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  gtk_object_class_add_signals (object_class, editor_signals, SIGNAL_LAST);

  object_class->destroy = gle_editor_destroy;

  class->refresh_values = gle_editor_real_refresh_values;
  class->apply_values = gle_editor_real_apply_values;
  class->reset_values = gle_editor_real_reset_values;
  class->restore_values = gle_editor_real_restore_values;
}

static void
gle_editor_init (GleEditor *editor)
{
  GtkWidget *main_vbox;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *any;
  GtkWidget *button;

  editor->gobject = NULL;
  editor->args_dirty = FALSE;
  editor->is_widget = FALSE;
  editor->garg_list = NULL;

  gtk_widget_set (GTK_WIDGET (editor),
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::title", "GLE-Editor",
		  "GtkWindow::window_position", GTK_WIN_POS_CENTER,
		  "GtkWindow::allow_shrink", FALSE,
		  "GtkWindow::allow_grow", TRUE,
		  "GtkWindow::auto_shrink", FALSE,
		  NULL);
  main_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", editor,
		    "GtkWidget::visible", TRUE,
		    NULL);

  /* gle name
   */
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  editor->gle_name_label =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "<gle_name_label>",
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), editor->gle_name_label, FALSE, TRUE, 0);
  editor->gle_name_entry =
    gtk_widget_new (gtk_entry_get_type (),
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::activate", gtk_window_activate_default, editor,
		    "GtkObject::signal::destroy", gtk_widget_destroyed, &editor->gle_name_entry,
		    NULL);
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), any, FALSE, TRUE, 0);

  /* object name label
   */
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  editor->object_name_label =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "<object_name_label>",
		    "GtkWidget::visible", TRUE,
		    "GtkObject::signal::destroy", gtk_widget_destroyed, &editor->object_name_label,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), editor->object_name_label, FALSE, TRUE, 0);
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), any, FALSE, TRUE, 0);

  /* widget argument notebook
   */
  vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), vbox, TRUE, TRUE, 0);
  editor->notebook =
    gtk_widget_new (gtk_notebook_get_type (),
		    "GtkWidget::parent", vbox,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::signal::destroy", gtk_widget_destroyed, &editor->notebook,
		    NULL);
  gtk_notebook_set_scrollable (GTK_NOTEBOOK (editor->notebook), TRUE);
  gtk_notebook_set_tab_border (GTK_NOTEBOOK (editor->notebook), 0);
  gtk_notebook_set_show_border (GTK_NOTEBOOK (editor->notebook), FALSE);
  gtk_notebook_popup_enable (GTK_NOTEBOOK (editor->notebook));
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), any, FALSE, TRUE, 0);

  /* Reset/Restore buttons
   */
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Reset",
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::sensitive", FALSE,
		    "GtkObject::object_signal::clicked", gle_editor_reset_values, editor,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Restore",
		    "GtkWidget::sensitive", FALSE,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::clicked", gle_editor_restore_values, editor,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), any, FALSE, TRUE, 0);

  /* Apply/Close buttons
   */
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_end (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Apply",
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::can_default", TRUE,
		    "GtkWidget::has_default", TRUE,
		    "GtkObject::object_signal::clicked", gle_editor_apply_values, editor,
		    NULL);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Close",
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::can_default", TRUE,
		    "GtkObject::object_signal::clicked", gtk_widget_destroy, editor,
		    NULL);
}

static void
gle_editor_update_names (GleEditor *editor)
{
  static const gchar *generic_prompt = "Generic GLE Name:";
  static const guint prompt_offs = 8;
  gchar *string;

  if (editor->is_widget)
    {
      GtkWidget *widget;

      widget = GTK_WIDGET (editor->gobject->object);
      string = g_strconcat (gtk_type_name (editor->gobject->type),
			    "::",
			    "\"",
			    widget->name ? widget->name : "",
			    "\"",
			    NULL);
    }
  else
    string = g_strconcat (gtk_type_name (editor->gobject->type));

  gtk_label_set (GTK_LABEL (editor->object_name_label), string);
  g_free (string);

  if (GLE_GOBJECT_GENERIC_GLE_NAME (editor->gobject))
    gtk_label_set (GTK_LABEL (editor->gle_name_label), generic_prompt);
  else
    gtk_label_set (GTK_LABEL (editor->gle_name_label), generic_prompt + prompt_offs);

  gtk_entry_set_text (GTK_ENTRY (editor->gle_name_entry), editor->gobject->gle_name);
}

static void
gle_editor_construct (GleEditor	 *editor,
		      GleGObject *gobject)
{
  GtkType last_type;
  GList *list;
  GtkWidget *vbox;
  
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));
  g_return_if_fail (!GLE_GOBJECT_HAS_EDITOR (gobject));
  
  editor->gobject = gobject;
  editor->is_widget = gtk_type_is_a (gobject->type, GTK_TYPE_WIDGET);
  
  gle_gobject_register_editor (gobject, GTK_WIDGET (editor));
  
  gtk_signal_connect_object_while_alive (gobject->object,
					 "destroy",
					 GTK_SIGNAL_FUNC (gtk_widget_destroy),
					 GTK_OBJECT (editor));
  
  
  /* create argument fields in notebook
   */
  last_type = GTK_TYPE_INVALID;
  vbox = NULL;
  for (list = editor->gobject->gargs; list; list = list->next)
    {
      GleGArg *garg;
      gchar *name;
      
      garg = list->data;
      if (!GLE_GARG_IS_READWRITE (garg))
	continue;

      gle_garg_get (garg, editor->gobject->object);
      gle_garg_save (garg);

      if (last_type != garg->info->class_type)
	{
	  last_type = garg->info->class_type;
	  vbox =
	    gtk_widget_new (gtk_vbox_get_type (),
			    "GtkBox::homogeneous", FALSE,
			    "GtkBox::spacing", 0,
			    "GtkContainer::border_width", 5,
			    "GtkWidget::visible", TRUE,
			    NULL);
	  gtk_notebook_prepend_page (GTK_NOTEBOOK (editor->notebook),
				     vbox, gtk_label_new (gtk_type_name (last_type)));
	}
      
      name = strchr (garg->arg_name, ':') + 2;
      garg->widget = gle_editor_field_create (garg, name, GTK_BOX (vbox));
      if (GLE_GARG_IS_CONSTRUCT (garg))
	gtk_widget_set_sensitive (garg->widget, FALSE);
    }
  gtk_notebook_set_page (GTK_NOTEBOOK (editor->notebook), 0);

  gle_editor_refresh_values (editor);
}

GtkWidget*
gle_editor_new (GleGObject *gobject)
{
  GtkWidget *editor;
  
  g_return_val_if_fail (gobject != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GOBJECT (gobject), NULL);

  editor = gtk_type_new (gle_editor_get_type ());

  gle_editor_construct (GLE_EDITOR (editor), gobject);

  return editor;
}

static void
gle_editor_destroy (GtkObject *object)
{
  GleEditor *editor;
  GList *list;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GLE_IS_EDITOR (object));

  editor = GLE_EDITOR (object);

  for (list = editor->gobject->gargs; list; list = list->next)
    {
      GleGArg *garg;

      garg = list->data;

      garg->widget = NULL;
    }

  editor->gobject->editor = NULL;
  editor->gobject = NULL;

  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

void
gle_editor_refresh_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_REFRESH_VALUES]);
}

void
gle_editor_apply_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));

  gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_APPLY_VALUES]);
}

void
gle_editor_reset_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));

  gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_RESET_VALUES]);
}

void
gle_editor_restore_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));

  gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_RESTORE_VALUES]);
}

static GtkWidget*
gle_editor_field_create (GleGArg	    *garg,
			 const gchar	    *name,
			 GtkBox		    *parent_box)
{
  register GtkWidget *widget;
  GtkType arg_type;
  
  g_return_val_if_fail (garg != NULL, NULL);
  g_return_val_if_fail (parent_box != NULL, NULL);
  
  if (!name)
    name = garg->arg_name;
  
  widget = NULL;
  arg_type = garg->info->arg->type;
  switch (arg_type)
    {
      GtkWidget *box;
      GtkWidget *label;
      GtkWidget *entry;
      
    case GTK_TYPE_BOOL:
      widget = gtk_widget_new (gtk_check_button_get_type (),
			       "GtkButton::label", name,
			       "GtkWidget::visible", TRUE,
			       NULL);
      gtk_misc_set_alignment (GTK_MISC (GTK_BUTTON (widget)->child), 0, 0.5);
      gtk_box_pack_start (parent_box, widget, FALSE, FALSE, 0);
      break;
      
    case GTK_TYPE_INT:
    case GTK_TYPE_UINT:
    case GTK_TYPE_LONG:
    case GTK_TYPE_ULONG:
      box = gtk_widget_new (gtk_hbox_get_type (),
			    "GtkBox::homogeneous", FALSE,
			    "GtkBox::spacing", 10,
			    "GtkContainer::border_width", 0,
			    "GtkWidget::visible", TRUE,
			    NULL);
      gtk_box_pack_start (parent_box, box, FALSE, TRUE, 0);
      label = gtk_widget_new (gtk_label_get_type (),
			      "GtkLabel::label", name,
			      "GtkLabel::justify", GTK_JUSTIFY_LEFT,
			      "GtkWidget::visible", TRUE,
			      NULL);
      gtk_box_pack_start (GTK_BOX (box), label, FALSE, TRUE, 0);
      entry = gtk_widget_new (gtk_entry_get_type (),
			      "GtkWidget::width", 50,
			      "GtkWidget::visible", TRUE,
			      NULL);
      gtk_box_pack_end (GTK_BOX (box), entry, FALSE, TRUE, 0);
      widget = entry;
      break;

    case GTK_TYPE_FLOAT:
    case GTK_TYPE_DOUBLE:
    case GTK_TYPE_STRING:
      box = gtk_widget_new (gtk_hbox_get_type (),
			    "GtkBox::homogeneous", FALSE,
			    "GtkBox::spacing", 10,
			    "GtkContainer::border_width", 0,
			    "GtkWidget::visible", TRUE,
			    NULL);
      gtk_box_pack_start (parent_box, box, FALSE, TRUE, 0);
      label = gtk_widget_new (gtk_label_get_type (),
			      "GtkLabel::label", name,
			      "GtkLabel::justify", GTK_JUSTIFY_LEFT,
			      "GtkWidget::visible", TRUE,
			      NULL);
      gtk_box_pack_start (GTK_BOX (box), label, FALSE, TRUE, 0);
      entry = gtk_widget_new (gtk_entry_get_type (),
			      "GtkWidget::width", arg_type == GTK_TYPE_STRING ? 140 : 50,
			      "GtkWidget::visible", TRUE,
			      NULL);
      gtk_box_pack_end (GTK_BOX (box), entry, FALSE, TRUE, 0);
      widget = entry;
      break;
      
    default:
      widget = gtk_widget_new (gtk_label_get_type (),
			       "GtkLabel::label", garg->arg_name,
			       "GtkWidget::sensitive", FALSE,
			       "GtkWidget::visible", TRUE,
			       NULL);
      gtk_misc_set_alignment (GTK_MISC (widget), 0, 0.5);
      gtk_box_pack_start (parent_box, widget, FALSE, FALSE, 0);
      break;
    }
  
  if (widget)
    gtk_object_set_data (GTK_OBJECT (widget), gle_editor_garg_key, garg);
  
  return widget;
}

static void
gle_editor_field_refresh (GtkWidget *widget,
			  GtkArg    *arg)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (arg != NULL);
  
  switch (arg->type)
    {
      gchar *string;
      gchar buffer[64];
      
    case GTK_TYPE_BOOL:
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (widget),
				   arg->d.bool_data);
      break;
      
    case GTK_TYPE_INT:
      sprintf (buffer, "%d", arg->d.int_data);
      gtk_entry_set_text (GTK_ENTRY (widget), buffer);
      break;

    case GTK_TYPE_LONG:
      sprintf (buffer, "%ld", arg->d.long_data);
      gtk_entry_set_text (GTK_ENTRY (widget), buffer);
      break;

    case GTK_TYPE_UINT:
      sprintf (buffer, "%u", arg->d.uint_data);
      gtk_entry_set_text (GTK_ENTRY (widget), buffer);
      break;

    case GTK_TYPE_ULONG:
      sprintf (buffer, "%lu", arg->d.ulong_data);
      gtk_entry_set_text (GTK_ENTRY (widget), buffer);
      break;

    case GTK_TYPE_FLOAT:
      sprintf (buffer, "%f", arg->d.float_data);
      gtk_entry_set_text (GTK_ENTRY (widget), buffer);
      break;

    case GTK_TYPE_DOUBLE:
      sprintf (buffer, "%f", arg->d.double_data);
      gtk_entry_set_text (GTK_ENTRY (widget), buffer);
      break;

    case GTK_TYPE_STRING:
      string = arg->d.string_data;
      gtk_entry_set_text (GTK_ENTRY (widget), string ? string : "");
      break;
      
    default:
      break;
    }
}

static gint
gle_editor_field_to_arg (GtkWidget *widget,
			 GtkArg    *arg)
{
  gboolean dirty;
  gchar *dummy = NULL;
    
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (arg != NULL, FALSE);

  dirty = FALSE;
  switch (arg->type)
    {
      gboolean bool_data;
      gchar *string;
      gchar *string2;
      gulong ulong_data;
      glong long_data;
      gdouble double_data;
      
    case GTK_TYPE_BOOL:
      bool_data = GTK_TOGGLE_BUTTON (widget)->active;
      if (bool_data != arg->d.bool_data)
	{
	  arg->d.bool_data = bool_data;
	  dirty = TRUE;
	}
      break;
      
    case GTK_TYPE_INT:
      long_data = strtol (gtk_entry_get_text (GTK_ENTRY (widget)), &dummy, 10);
      if (long_data != arg->d.int_data)
	{
	  arg->d.int_data = long_data;
	  dirty = TRUE;
	}
      break;

    case GTK_TYPE_LONG:
      long_data = strtol (gtk_entry_get_text (GTK_ENTRY (widget)), &dummy, 10);
      if (long_data != arg->d.long_data)
	{
	  arg->d.long_data = long_data;
	  dirty = TRUE;
	}
      break;

    case GTK_TYPE_UINT:
      ulong_data = strtol (gtk_entry_get_text (GTK_ENTRY (widget)), &dummy, 10);
      if (ulong_data != arg->d.uint_data)
	{
	  arg->d.uint_data = ulong_data;
	  dirty = TRUE;
	}
      break;

    case GTK_TYPE_ULONG:
      ulong_data = strtol (gtk_entry_get_text (GTK_ENTRY (widget)), &dummy, 10);
      if (ulong_data != arg->d.ulong_data)
	{
	  arg->d.ulong_data = ulong_data;
	  dirty = TRUE;
	}
      break;

    case GTK_TYPE_FLOAT:
      double_data = g_strtod (gtk_entry_get_text (GTK_ENTRY (widget)), &dummy);
      if (double_data != arg->d.float_data)
	{
	  arg->d.float_data = double_data;
	  dirty = TRUE;
	}
      break;

    case GTK_TYPE_DOUBLE:
      double_data = g_strtod (gtk_entry_get_text (GTK_ENTRY (widget)), &dummy);
      if (double_data != arg->d.double_data)
	{
	  arg->d.double_data = double_data;
	  dirty = TRUE;
	}
      break;

    case GTK_TYPE_STRING:
      string = gtk_entry_get_text (GTK_ENTRY (widget));
      string2 = arg->d.string_data;
      if (!string2)
	string2 = "";
      if (strcmp (string, string2) != 0)
	{
	  g_free (arg->d.string_data);
	  arg->d.string_data = g_strdup (string);
	  dirty = TRUE;
	}
      break;
      
    default:
      break;
    }

  return dirty;
}

static void
gle_editor_real_refresh_values (GleEditor *editor)
{
  GList *list;

  for (list = editor->gobject->gargs; list; list = list->next)
    {
      GleGArg *garg;

      garg = list->data;

      if (garg->widget)
	gle_editor_field_refresh (garg->widget, &garg->object_arg);
    }
  
  gle_editor_update_names (editor);
}

static void
gle_editor_real_apply_values (GleEditor *editor)
{
  GList *list;
  gchar *gle_name;
  
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  gle_name = gtk_entry_get_text (GTK_ENTRY (editor->gle_name_entry));
  if (editor->is_widget)
    gle_name = gle_canonicalize_gle_name (g_strdup (gle_name));
  else
    gle_name = gle_canonicalize_object_name (g_strdup (gle_name));
  if (gle_name[0] &&
      !g_str_equal ((gchar*) editor->gobject->gle_name, gle_name) &&
      !gle_gobject_lookup (gle_name))
    gle_gobject_set_gle_name (editor->gobject, gle_name);
  g_free (gle_name);

  for (list = editor->gobject->gargs; list; list = list->next)
    {
      GleGArg *garg;

      garg = list->data;

      if (GLE_GARG_IS_READWRITE (garg) && garg->widget)
	garg->needs_set = gle_editor_field_to_arg (garg->widget, &garg->object_arg);
    }
  gle_gobject_apply_args (editor->gobject);

  gle_editor_refresh_values (editor);
}

static void
gle_editor_real_reset_values (GleEditor *editor)
{
  gle_editor_refresh_values (editor);
}

static void
gle_editor_real_restore_values (GleEditor *editor)
{
  gle_editor_refresh_values (editor);
}

/*

static void
gle_toggle_blink_widget (GtkWidget *widget)
{
  if (gle_get_flash_widget () != widget)
    gle_set_flash_widget (widget);
  else
    gle_set_flash_widget (NULL);
}

static void
gle_toggle_button_set_arg (GtkToggleButton	*toggle,
			   GtkArg		*arg)
{
  printf ("gle_toggle_button_set_arg: %d\n", toggle->active);
}

static void
gle_entry_set_arg (GtkEntry	 *entry,
		   GtkArg	 *arg)
{
  printf ("gle_entry_set_arg: \"%s\"\n", entry->text);
}

void
gle_editor_reset (GtkWindow	*editor)
{
  GleEditorData *data;

  g_return_if_fail (editor != NULL);
  g_return_if_fail (GTK_IS_WINDOW (editor));
  data = GLE_EDITOR_DATA (editor);
}
*/
