/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"glegarg.h"
#include	"glemem.h"
#include	<string.h>
#include	<stdio.h>


/* --- variables --- */
static GHashTable	*gle_arg_info_ht = NULL;

/* --- functions --- */
GleGArg*
gle_garg_new (const GleArgInfo *info)
{
  GtkType	object_type;
  GleGArg	*garg;

  g_return_val_if_fail (info != NULL, NULL);
  g_return_val_if_fail (info->arg != NULL, NULL);
  g_return_val_if_fail (info->arg->type != GTK_TYPE_INVALID, NULL);
  g_return_val_if_fail (info->arg->type != GTK_TYPE_NONE, NULL);

  object_type = gtk_type_from_arg_name (info->arg->name, NULL);
  g_return_val_if_fail (gtk_type_is_a (object_type, gtk_object_get_type ()), NULL);

  garg = gle_garg_chunk_new0 ();
  garg->arg_name = info->arg->name;
  garg->info = info;
  garg->setting_failed = FALSE;
  garg->needs_set = FALSE;
  garg->widget = NULL;
  garg->saved_arg.type = GTK_TYPE_INVALID;
  garg->saved_arg.name = info->arg->name;
  garg->object_arg.type = GTK_TYPE_INVALID;
  garg->object_arg.name = info->arg->name;

  gle_garg_reset (garg);

  return garg;
}

void
gle_garg_reset (GleGArg	*garg)
{
  g_return_if_fail (garg != NULL);

  gle_arg_reset (&garg->object_arg);
  gle_arg_copy_value (garg->info->arg, &garg->object_arg);
  gle_garg_save (garg);
  garg->setting_failed = FALSE;
  garg->needs_set = FALSE;
}

void
gle_garg_save (GleGArg *garg)
{
  g_return_if_fail (garg != NULL);

  if (garg->object_arg.type > GTK_TYPE_NONE)
    gle_arg_copy_value (&garg->object_arg, &garg->saved_arg);
  else
    {
      gle_arg_reset (&garg->object_arg);
      gle_arg_reset (&garg->saved_arg);
    }
  garg->setting_failed = FALSE;
}

void
gle_garg_restore (GleGArg *garg)
{
  g_return_if_fail (garg != NULL);

  if (garg->saved_arg.type > GTK_TYPE_NONE)
    gle_arg_copy_value (&garg->saved_arg, &garg->object_arg);
  else
    {
      gle_arg_reset (&garg->saved_arg);
      gle_arg_reset (&garg->object_arg);
    }
  garg->setting_failed = FALSE;
}

void
gle_garg_free (GleGArg*        garg)
{
  g_return_if_fail (garg != NULL);

  if (garg->widget)
    g_warning ("GLE: freeing GleGArg with widget != NULL");
  
  gle_arg_reset (&garg->saved_arg);
  gle_arg_reset (&garg->object_arg);
  
  gle_garg_chunk_free (garg);
}

void
gle_garg_init (GleGArg        *garg,
	       GtkObject      *object)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (object));
  g_return_if_fail (gtk_type_is_a (GTK_OBJECT_TYPE (object), GLE_GARG_CLASS_TYPE (garg)));

  if (GLE_GARG_IS_READABLE (garg))
    {
      gle_garg_get (garg, object);
      if (garg->setting_failed)
	gle_garg_reset (garg);
      else
	gle_garg_save (garg);
    }
  else
    gle_garg_reset (garg);
  garg->setting_failed = FALSE;
  garg->needs_set = FALSE;
}

void
gle_garg_get (GleGArg        *garg,
	      GtkObject      *object)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (object));
  g_return_if_fail (GLE_GARG_IS_READABLE (garg));
  g_return_if_fail (gtk_type_is_a (GTK_OBJECT_TYPE (object), GLE_GARG_CLASS_TYPE (garg)));

  gle_arg_reset (&garg->object_arg);
  garg->object_arg.type = garg->info->arg->type;

  gtk_object_getv (object, 1, &garg->object_arg);
  garg->setting_failed = FALSE;
}

void
gle_garg_set (GleGArg        *garg,
	      GtkObject      *object)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (object));
  g_return_if_fail (GLE_GARG_IS_WRITABLE (garg));
  g_return_if_fail (gtk_type_is_a (GTK_OBJECT_TYPE (object), GLE_GARG_CLASS_TYPE (garg)));


  if (garg->object_arg.type > GTK_TYPE_NONE)
    {
      GtkArg arg;

      g_memmove (&arg, &garg->object_arg, sizeof (arg));
      gtk_object_setv (object, 1, &arg);
      garg->setting_failed = arg.type > GTK_TYPE_NONE;
    }
  else
    garg->setting_failed = TRUE;
  garg->needs_set = FALSE;
}

void
gle_arg_copy_value (const GtkArg   *src_arg,
		    GtkArg         *dest_arg)
{
  g_return_if_fail (src_arg != NULL);
  g_return_if_fail (dest_arg != NULL);
  g_return_if_fail (src_arg->type > GTK_TYPE_NONE);
  
  gle_arg_reset (dest_arg);
  g_memmove (&dest_arg->d, &src_arg->d, sizeof (dest_arg->d));
  if (src_arg->type == GTK_TYPE_STRING)
    dest_arg->d.string_data = g_strdup (src_arg->d.string_data);
  dest_arg->type = src_arg->type;
}

void
gle_arg_reset (GtkArg *arg)
{
  g_return_if_fail (arg != NULL);

  if (arg->type != GTK_TYPE_INVALID)
    {
      if (arg->type == GTK_TYPE_STRING)
	g_free (arg->d.string_data);
      arg->type = GTK_TYPE_INVALID;
    }
}

const GleArgInfo*
gle_arg_get_info (const gchar *arg_name)
{
  GleArgInfo *info;
  
  g_return_val_if_fail (arg_name != NULL, NULL);
  
  if (!gle_arg_info_ht)
    {
      gle_arg_info_ht = g_hash_table_new (g_str_hash,
					  g_str_equal);
      info = NULL;
    }
  else
    info = g_hash_table_lookup (gle_arg_info_ht, (gpointer) arg_name);
  
  if (!info)
    {
      GtkType		object_type;
      GtkObject		*object;
      GtkArg		*args;
      guint		n_args;
      guint32		*flags;

      object_type = gtk_type_from_arg_name (arg_name, NULL);
      g_return_val_if_fail (gtk_type_is_a (object_type, gtk_object_get_type ()), NULL);

      object = gtk_type_new (object_type);
      n_args = 0;
      flags = NULL;
      args = gtk_object_query_args (GTK_OBJECT_TYPE (object), &flags, &n_args);

      for (; n_args > 0; n_args--)
	if (args[n_args - 1].type)
	  {
	    memset (&args[n_args - 1].d, 0, sizeof (args[n_args - 1].d));
	    if (flags[n_args - 1] & GTK_ARG_READABLE &&
		args[n_args - 1].type < GTK_TYPE_OBJECT)
	      gtk_object_getv (object, 1, &args[n_args - 1]);
	    else if (gtk_type_is_a (args[n_args - 1].type, GTK_TYPE_OBJECT))
	      args[n_args - 1].d.object_data = NULL;
	    info = gle_arg_info_chunk_new ();
	    info->class_type = object_type;
	    info->arg_flags = flags[n_args - 1];
	    info->seq_num = n_args - 1;
	    info->arg = &args[n_args - 1];
	    g_hash_table_insert (gle_arg_info_ht, info->arg->name, info);
	  }

      g_free (flags);
      if (GTK_OBJECT_FLOATING (object))
	gtk_object_sink (object);
      else
	gtk_object_destroy (object);

      info = g_hash_table_lookup (gle_arg_info_ht, (gpointer) arg_name);
    }

  return info;
}

void
gle_arg_set_from_string (GtkArg         *arg,
			 const gchar    *value_string)
{
  gchar *string;

  g_return_if_fail (arg != NULL);
  g_return_if_fail (value_string != NULL);

  string = g_strdup (value_string);

  switch (arg->type)
    {
    case  GTK_TYPE_INVALID:
      g_warning ("gle_arg_set_from_string() used with invalid type");
      break;
    case  GTK_TYPE_CHAR:
      if (strlen (string) > 1)
	{
	  arg->d.int_data = 0;
	  sscanf (string, "%d", &arg->d.int_data);
	  arg->d.char_data = arg->d.int_data;
	}
      else
	arg->d.char_data = string[0];
      break;
    case  GTK_TYPE_BOOL:
      g_strdown (string);
      if ((string[0] > '0' && string[0] <= '9') ||
	  g_str_equal (string, "true") ||
	  g_str_equal (string, "yes"))
	arg->d.bool_data = TRUE;
      else
	arg->d.bool_data = FALSE;
      break;
    case  GTK_TYPE_INT:
      arg->d.int_data = 0;
      sscanf (string, "%d", &arg->d.int_data);
      break;
    case  GTK_TYPE_UINT:
      arg->d.uint_data = 0;
      sscanf (string, "%u", &arg->d.int_data);
      break;
    case  GTK_TYPE_LONG:
      arg->d.long_data = 0;
      sscanf (string, "%ld", &arg->d.long_data);
      break;
    case  GTK_TYPE_ULONG:
      arg->d.ulong_data = 0;
      sscanf (string, "%lu", &arg->d.ulong_data);
      break;
    case  GTK_TYPE_FLOAT:
      arg->d.float_data = 0;
      sscanf (string, "%f", &arg->d.float_data);
      break;
    case  GTK_TYPE_DOUBLE:
      arg->d.double_data = 0;
      sscanf (string, "%lf", &arg->d.double_data);
      break;
    case  GTK_TYPE_STRING:
      arg->d.string_data = g_strdup (value_string);
      break;
    default:
      g_warning ("gle_arg_set_from_string() used with type <%s>", gtk_type_name (arg->type));
      break;
    }
  g_free (string);
}
