/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"glegobject.h"
#include	"glegwidget.h"
#include	"gleprivate.h"
#include	"glemem.h"


/* --- prototypes --- */
extern gpointer		_gle_gobject_new	(GtkType        type,
						 gchar		*gle_name,
						 gboolean       generic_name,
						 guint          alloc_type);
static void		gle_gobject_setup_args	(GleGObject	*gobject);


/* --- variables --- */
static GHashTable	*gobject_hash_table = NULL;
static const gchar	*key_gle_gobject = GLE_PRIVATE_KEY (gle-gobject);


/* --- functions --- */
gpointer
_gle_gobject_new (GtkType        type,
		  gchar         *gle_name,
		  gboolean	 generic_name,
		  guint		 alloc_type)
{
  GleGObject *gobject;

  switch (alloc_type)
    {
    case 1:
      gobject = gle_gobject_chunk_new0 ();
      break;
    case 2:
      gobject = (GleGObject*) gle_gwidget_chunk_new0 ();
      break;
    default:
      gobject = NULL;
      break;
    }

  gobject->type = type;
  gobject->gle_name = gle_name;
  gobject->flags = (generic_name ? GLE_GENERIC_GLE_NAME : 0);
  gobject->object = NULL;
  gobject->gargs = NULL;
  gobject->object_gargs = NULL;
  gobject->connections = NULL;
  gobject->editor = NULL;

  g_hash_table_insert (gobject_hash_table, (gpointer) gobject->gle_name, gobject);

  gle_gobject_setup_args (gobject);

  return gobject;
}

GleGObject*
gle_gobject_new (GtkType         type,
		 const gchar    *gle_name,
		 gboolean	 generic_name)
{
  GleGObject *gobject;
  gchar *name;

  g_return_val_if_fail (gtk_type_is_a (type, GTK_TYPE_OBJECT), NULL);
  g_return_val_if_fail (!gtk_type_is_a (type, GTK_TYPE_WIDGET), NULL);
  g_return_val_if_fail (gle_name != NULL, NULL);
  g_return_val_if_fail (gle_name[0] != 0, NULL);

  name = gle_canonicalize_object_name (g_strdup (gle_name));

  if (gle_gobject_lookup (name))
    {
      g_warning ("gle_gobject_new(): GleGObject \"%s\" already exists", name);
      g_free (name);
      return NULL;
    }

  gobject = _gle_gobject_new (type, name, generic_name, 1);

  return gobject;
}

GleGObject*
gle_gobject_lookup (const gchar    *gle_name)
{
  g_return_val_if_fail (gle_name != NULL, NULL);

  if (!gobject_hash_table)
    gobject_hash_table = g_hash_table_new (g_str_hash, g_str_equal);

  return g_hash_table_lookup (gobject_hash_table, (gpointer) gle_name);
}

static void
gle_gobject_remove_object (GleGObject     *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));
  g_return_if_fail (!GLE_GOBJECT_HAS_EDITOR (gobject));

  gobject->object = NULL;
}

void
gle_gobject_associate (GleGObject     *gobject,
		       GtkObject      *object)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (object));
  g_return_if_fail (!GLE_OBJECT_HAS_GOBJECT (object));
  g_return_if_fail (!GLE_HAS_TAG (object));
  g_return_if_fail (!GLE_GOBJECT_IS_INSTANTIATED (gobject));

  gobject->object = object;
  gtk_object_set_data_full (object,
			    key_gle_gobject,
			    gobject,
			    (GtkDestroyNotify) gle_gobject_remove_object);
}

GtkObject*
gle_gobject_instantiate (GleGObject     *gobject)
{
  g_return_val_if_fail (gobject != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GOBJECT (gobject), NULL);
  g_return_val_if_fail (!GLE_GOBJECT_IS_INSTANTIATED (gobject), NULL);

  return NULL;
}

void
gle_gobject_disassociate (GleGObject     *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));

  if (gobject->editor)
    gtk_widget_destroy (gobject->editor);

  gtk_object_remove_data (gobject->object, key_gle_gobject);
}

void
gle_gobject_connect (GleGObject     *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));

  if (!GTK_OBJECT_DESTROYED (gobject->object))
    {
    }
}

void
gle_gobject_disconnect (GleGObject     *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));

  if (!GTK_OBJECT_DESTROYED (gobject->object))
    {
    }
}

static void
gle_gobject_setup_args (GleGObject     *gobject)
{
  GtkType type;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (gobject->gargs == NULL);
  g_return_if_fail (gobject->object_gargs == NULL);

  type = gobject->type;

  while (type)
    {
      GtkArg  *args;
      GleGArg *garg;
      guint    nargs;
      guint    i;
      
      args = gtk_object_query_args (type, NULL, &nargs);
      
      for (i = 0; i < nargs; i++)
	{
	  const GleArgInfo *garg_info;
	  
	  garg_info = gle_arg_get_info (args[i].name);
	  if (!garg_info)
	    {
	      g_warning ("unresolved object arg: \"%s\"", args[i].name);
	      continue;
	    }

	  if (!(garg_info->arg_flags & GTK_ARG_CONSTRUCT) ||
	      garg_info->class_type == gobject->type)
	    {
	      garg = gle_garg_new (garg_info);
	      
	      if (GLE_GARG_ARG_TYPE (garg) < GTK_TYPE_OBJECT)
		gobject->gargs = g_list_prepend (gobject->gargs, garg);
	      else
		gobject->object_gargs = g_list_prepend (gobject->object_gargs, garg);
	    }
	}
      
      g_free (args);

      type = gtk_type_parent (type);
    }

  gobject->gargs = g_list_reverse (gobject->gargs);
  gobject->object_gargs = g_list_reverse (gobject->object_gargs);
}

void
gle_gobject_apply_args (GleGObject     *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));

  if (!GTK_OBJECT_DESTROYED (gobject->object))
    {
      GList *list;

      for (list = gobject->gargs; list; list = list->next)
	{
	  GleGArg *garg;

	  garg = list->data;
	  if (GLE_GARG_IS_WRITABLE (garg) &&
	      GLE_GARG_NEEDS_SET (garg))
	    gle_garg_set (garg, gobject->object);
	}
    }
}

void
gle_gobject_update_args (GleGObject     *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));

  if (!GTK_OBJECT_DESTROYED (gobject->object))
    {
      GList *list;

      for (list = gobject->gargs; list; list = list->next)
	{
	  GleGArg *garg;

	  garg = list->data;
	  if (GLE_GARG_IS_READABLE (garg))
	    {
	      GLE_GARG_NEEDS_SET (garg) = FALSE;
	      
	      gle_garg_get (garg, gobject->object);
	    }
	}
    }
}

void
gle_gobject_set_gle_name (GleGObject     *gobject,
			  const gchar    *gle_name)
{
  gchar *name;

  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (gle_name != NULL);
  g_return_if_fail (gle_name[0] != 0);

  if (GLE_IS_GWIDGET (gobject))
    name = gle_canonicalize_gle_name (g_strdup (gle_name));
  else
    name = gle_canonicalize_object_name (g_strdup (gle_name));

  if (gle_gobject_lookup (name))
    {
      g_warning ("gle_gobject_set_gle_name(): GleGObject \"%s\" already exists", name);
      g_free (name);
      return;
    }

  GLE_GOBJECT_UNSET_FLAG (gobject, GLE_GENERIC_GLE_NAME);

  g_hash_table_remove (gobject_hash_table, (gpointer) gobject->gle_name);

  g_free ((gchar*) gobject->gle_name);
  gobject->gle_name = name;

  g_hash_table_insert (gobject_hash_table, (gpointer) gobject->gle_name, gobject);
}

void
gle_gobject_register_editor (GleGObject     *gobject,
			     GtkWidget      *editor)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GTK_IS_WIDGET (editor));
  g_return_if_fail (!GLE_GOBJECT_HAS_EDITOR (gobject));

  gobject->editor = editor;
  gtk_signal_connect (GTK_OBJECT (editor),
		      "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroyed),
		      &gobject->editor);
}

GleGObject*
gle_object_get_gobject (GtkObject      *object)
{
  GleGObject *gobject;

  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (GTK_IS_OBJECT (object), NULL);
  
  if (!GTK_OBJECT_DESTROYED (object))
    {
      gobject = gtk_object_get_data (object, key_gle_gobject);
    }
  else
    gobject = NULL;

  return gobject;
}

GleGObject*
gle_object_force_gobject (GtkObject      *object)
{
  GleGObject *gobject;

  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (GTK_IS_OBJECT (object), NULL);
  g_return_val_if_fail (!GLE_HAS_TAG (object), NULL);

  if (GTK_IS_WIDGET (object))
    return (GleGObject*) gle_widget_force_gwidget (GTK_WIDGET (object));

  gobject = gtk_object_get_data (object, key_gle_gobject);
  if (!gobject)
    {
      gobject = gle_gobject_new (GTK_OBJECT_TYPE (object),
				 gle_object_create_gle_name (object),
				 TRUE);
      gle_gobject_associate (gobject, object);
    }
  
  return gobject;
}

const gchar*
gle_object_get_gle_name (GtkObject      *object)
{
  GleGObject *gobject;

  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (GTK_IS_OBJECT (object), NULL);
  
  if (!GTK_OBJECT_DESTROYED (object))
    {
      gobject = gtk_object_get_data (object, key_gle_gobject);
    }
  else
    gobject = NULL;

  if (gobject)
    return gobject->gle_name;

  return NULL;
}

gchar*
gle_object_create_gle_name (GtkObject      *object)
{
  gchar *gle_name;
  guint counter;

  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (GTK_IS_OBJECT (object), NULL);

  if (GTK_IS_WIDGET (object))
    return gle_widget_create_gle_name (GTK_WIDGET (object));

  gle_name = NULL;
  counter = 1;
  do
    {
      gchar buffer[64];

      sprintf (buffer, "-%u", counter++);
      g_free (gle_name);
      gle_name = (gchar*) gle_canonicalize_object_name_static (gtk_type_name (GTK_OBJECT_TYPE (object)));
      gle_name = g_strconcat (gle_name,
			      buffer,
			      NULL);
      gle_name = gle_canonicalize_object_name (gle_name);
    }
  while (gle_gobject_lookup (gle_name));

  return gle_name;
}
