/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"glepopups.h"
#include	<stdio.h>
#include	<stdarg.h>



/* --- external prototypes --- */
extern	char*	g_vsprintf	(gchar *fmt, va_list *args1, va_list *args2);


/* --- functions --- */
static GtkWidget*
gle_popup_generic (gpointer	   nullify_pointer,
		   const gchar	  *window_title,
		   const gchar	  *action_name,
		   GtkSignalFunc   signal_func,
		   gpointer	   func_data,
		   GtkWidget	  **vbox_p,
		   const gchar	  *format,
		   va_list	  *args1_p,
		   va_list	  *args2_p)
{
  GtkWidget *window;
  GtkWidget *window_vbox;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *button;
  GtkWidget *any;
  
  /* main window
   */
  window	= gtk_widget_new (gtk_window_get_type (),
				  "GtkWindow::title", window_title,
				  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
				  "GtkWindow::allow_shrink", vbox_p ? TRUE : FALSE,
				  "GtkWindow::allow_grow", vbox_p ? TRUE : FALSE,
				  "GtkWindow::auto_shrink", vbox_p ? FALSE : TRUE,
				  NULL);
  window_vbox	= gtk_widget_new (gtk_vbox_get_type (),
				  "GtkBox::homogeneous", FALSE,
				  "GtkBox::spacing", 0,
				  "GtkContainer::border_width", 0,
				  "GtkWidget::parent", window,
				  "GtkWidget::visible", TRUE,
				  NULL);
  vbox		= gtk_widget_new (gtk_vbox_get_type (),
				  "GtkBox::homogeneous", FALSE,
				  "GtkBox::spacing", 5,
				  "GtkContainer::border_width", 5,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_pack_start (GTK_BOX (window_vbox), vbox, FALSE, TRUE, 0);
  any		= gtk_widget_new (gtk_hseparator_get_type (),
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_pack_start (GTK_BOX (window_vbox), any, FALSE, TRUE, 0);
  if (vbox_p)
    {
      *vbox_p	= gtk_widget_new (gtk_vbox_get_type (),
				  "GtkBox::homogeneous", FALSE,
				  "GtkBox::spacing", 5,
				  "GtkContainer::border_width", 5,
				  "GtkWidget::visible", TRUE,
				  NULL);
      gtk_box_pack_start (GTK_BOX (window_vbox), *vbox_p, TRUE, TRUE, 0);
      any	= gtk_widget_new (gtk_hseparator_get_type (),
				  "GtkWidget::visible", TRUE,
				  NULL);
      gtk_box_pack_start (GTK_BOX (window_vbox), any, FALSE, TRUE, 0);
    }
  hbox		= gtk_widget_new (gtk_hbox_get_type (),
				  "GtkBox::homogeneous", TRUE,
				  "GtkBox::spacing", 5,
				  "GtkContainer::border_width", 5,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_box_pack_end (GTK_BOX (window_vbox), hbox, FALSE, TRUE, 0);
  if (action_name)
    button	= gtk_widget_new (gtk_button_get_type (),
				  "GtkButton::label", action_name,
				  "GtkWidget::parent", hbox,
				  "GtkWidget::can_default", TRUE,
				  "GtkWidget::visible", TRUE,
				  "GtkObject::signal::clicked", signal_func, func_data,
				  "GtkObject::object_signal::clicked", gtk_widget_destroy, window,
				  NULL);
  button	= gtk_widget_new (gtk_button_get_type (),
				  "GtkButton::label", "Close",
				  "GtkWidget::parent", hbox,
				  "GtkWidget::can_default", TRUE,
				  "GtkWidget::has_default", TRUE,
				  "GtkWidget::visible", TRUE,
				  "GtkObject::object_signal::clicked", gtk_widget_destroy, window,
				  NULL);
  
  /* (*nullify_pointer) needs to be set to NULL if the window is destroyed
   */
  if (nullify_pointer)
    gtk_signal_connect (GTK_OBJECT (window),
			"destroy",
			GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			nullify_pointer);
  
  /* split message into labels
   * and add them to window
   */
  if (format)
    {
      gchar	*buffer;
      gchar	*message;
      gchar	*cp;
      gchar	*end;

      message = g_strdup (format);
      buffer = g_vsprintf (message, args1_p, args2_p);
      g_free (message);
      
      message = buffer;
      cp = message;
      end = cp + strlen (message);
      while (cp <= end)
	{
	  if (*cp == '\n' || *cp == 0)
	    {
	      GtkWidget	*label;
	      
	      *cp = 0;
	      label = gtk_widget_new (gtk_label_get_type (),
				      "GtkLabel::label", message,
				      "GtkWidget::visible", TRUE,
				      NULL);
	      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 0);
	      message = cp + 1;
	    }
	  cp++;
	}
    }
  
  return window;
}

GtkWidget*
gle_popup_message (GtkWidget	 **nullify_pointer,
		   const gchar	 *window_title,
		   const gchar	 *message_fmt,
		   ...)
{
  GtkWidget *window;
  va_list    args1;
  va_list    args2;
  
  g_return_val_if_fail (message_fmt != NULL, NULL);
  if (!window_title)
    window_title = GLE_MESSAGE_TITLE;

  va_start (args1, message_fmt);
  va_start (args2, message_fmt);
  window = gle_popup_generic (nullify_pointer,
			      window_title,
			      NULL,
			      NULL,
			      NULL,
			      NULL,
			      message_fmt,
			      &args1,
			      &args2);
  va_end (args1);
  va_end (args2);

  gtk_widget_show (window);

  return window;
}

GtkWidget*
gle_popup_dialog (gpointer	  nullify_pointer,
		  const gchar	 *window_title,
		  const gchar	 *action_name,
		  GtkSignalFunc	  signal_func,
		  gpointer	  func_data,
		  const gchar	 *message_fmt,
		  ...)
{
  GtkWidget *window;
  va_list    args1;
  va_list    args2;
  
  g_return_val_if_fail (message_fmt != NULL, NULL);
  g_return_val_if_fail (action_name != NULL, NULL);
  g_return_val_if_fail (signal_func != NULL, NULL);
  if (!window_title)
    window_title = GLE_DIALOG_TITLE;

  va_start (args1, message_fmt);
  va_start (args2, message_fmt);
  window = gle_popup_generic (nullify_pointer,
			      window_title,
			      action_name,
			      signal_func,
			      func_data,
			      NULL,
			      message_fmt,
			      &args1,
			      &args2);
  va_end (args1);
  va_end (args2);

  gtk_widget_show (window);

  return window;
}

static void
gle_list_selection (GtkList	   *list,
		    gpointer	   *user_data_pointer)
{
  g_return_if_fail (list != NULL);
  g_return_if_fail (GTK_IS_LIST (list));
  g_return_if_fail (user_data_pointer != NULL);

  if (list->selection)
    {
      GtkItem *item;
      
      item = list->selection->data;
      *user_data_pointer = gtk_object_get_user_data (GTK_OBJECT (item));
    }
  else
    *user_data_pointer = NULL;
}

static gint
gle_list_button_press (GtkWidget      *list,
		       GdkEventButton *event,
		       GtkWidget      *widget)
{
  g_return_val_if_fail (event != NULL, FALSE);
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_WIDGET (widget), FALSE);

  if (event->type == GDK_2BUTTON_PRESS && event->button == 1)
    {
      gtk_widget_destroy (widget);
      /* gtk_widget_queue_destroy (widget); */
      return TRUE;
    }

  return FALSE;
}

static void
gle_scrolled_window_init_vscroll (GtkWidget	 *scrolled_window,
				  GtkAllocation	 *allocation,
				  gfloat	 *adjust_val)
{
  GtkAdjustment	 *adjustment;
  gfloat	  value;

  g_return_if_fail (scrolled_window != NULL);
  g_return_if_fail (GTK_IS_SCROLLED_WINDOW (scrolled_window));
  g_return_if_fail (allocation != NULL);
  g_return_if_fail (adjust_val != NULL);

  adjustment = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (scrolled_window));
  
  value = CLAMP (((adjustment->upper - (double) adjustment->lower) *
		  (*adjust_val) -
		  adjustment->page_size / 2),
		 adjustment->lower,
		 adjustment->upper - adjustment->page_size);
  
  gtk_adjustment_set_value (adjustment, value);

  gtk_signal_disconnect_by_data (GTK_OBJECT (scrolled_window), adjust_val);

  g_free (adjust_val);
}

GtkWidget*
gle_popup_simple_list (gpointer	      nullify_pointer,
		       const gchar    *window_title,
		       const gchar    *frame_label,
		       GtkSignalFunc  signal_func,
		       gpointer	      func_data,
		       gboolean	      browse,
		       GList	      *item_list,
		       gint	      select_pos,
		       gpointer	      *user_data_pointer,
		       gboolean	      double_click_destroys,
		       const gchar    *message_fmt,
		       ...)
{
  GtkWidget *window;
  GtkWidget *vbox;
  GtkWidget *frame;
  GtkWidget *scrolled_window;
  GtkWidget *list;
  va_list    args1;
  va_list    args2;
  
  g_return_val_if_fail (message_fmt != NULL, NULL);
  g_return_val_if_fail (item_list != NULL, NULL);
  if (!window_title)
    window_title = GLE_DIALOG_TITLE;

  va_start (args1, message_fmt);
  va_start (args2, message_fmt);
  window = gle_popup_generic (nullify_pointer,
			      window_title,
			      NULL,
			      NULL,
			      NULL,
			      &vbox,
			      message_fmt,
			      &args1,
			      &args2);
  va_end (args1);
  va_end (args2);

  /* set the list up
   */
  frame		= gtk_frame_new (frame_label);
  gtk_widget_set (frame,
		  "GtkWidget::parent", vbox,
		  "GtkWidget::visible", TRUE,
		  NULL);
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_ALWAYS);
  gtk_widget_set (scrolled_window,
		  "GtkWidget::parent", frame,
		  "GtkWidget::visible", TRUE,
		  "GtkWidget::width", 200,
		  "GtkWidget::height", 200,
		  NULL);
  list		= gtk_widget_new (gtk_list_get_type (),
				  "GtkWidget::parent", scrolled_window,
				  "GtkWidget::visible", TRUE,
				  NULL);
  gtk_list_set_selection_mode (GTK_LIST (list),
			       browse ? GTK_SELECTION_BROWSE : GTK_SELECTION_SINGLE);
  gtk_list_append_items (GTK_LIST (list), item_list);
  if (user_data_pointer)
    {
      *user_data_pointer = NULL;
      gtk_signal_connect (GTK_OBJECT (list),
			  "selection_changed",
			  GTK_SIGNAL_FUNC (gle_list_selection),
			  user_data_pointer);
    }
  if (signal_func)
    gtk_signal_connect (GTK_OBJECT (list),
			"selection_changed",
			signal_func,
			func_data);
  
  /* position the scrolled window to show the selected item
   */
  if (select_pos >= 0)
    {
      gtk_list_select_item (GTK_LIST (list), select_pos);
      
      if (GTK_LIST (list)->selection)
	{
	  gfloat *adjust_val;
	  
	  adjust_val = g_new (gfloat, 1);
	  
	  select_pos = gtk_list_child_position (GTK_LIST (list),
						GTK_LIST (list)->selection->data);
	  *adjust_val = select_pos / (gdouble) g_list_length (GTK_LIST (list)->children);
	  gtk_signal_connect (GTK_OBJECT (scrolled_window),
			      "size_allocate",
			      GTK_SIGNAL_FUNC (gle_scrolled_window_init_vscroll),
			      adjust_val);
	}
    }
  
  /* destroy the main window on double click
   */
  if (double_click_destroys)
    gtk_signal_connect (GTK_OBJECT (list),
			"button_press_event",
			GTK_SIGNAL_FUNC (gle_list_button_press),
			window);
  
  gtk_widget_show (window);
  
  return window;
}
