/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * This implements at complete ATSC transmitter
 *
 * The input is an MPEG transport stream.
 * The output is 16-bit samples at either 21.52 or 20.0 Msamples/sec
 *   with an IF of 5.75 MHz.
 */

#include <gr_FlowGraph.h>
#include <VrFileSink.h>
#include <VrFileSource.h>
#include <GrTCPSink.h>
#include <GrAtscRandomizer.h>
#include <GrAtscRSEncoder.h>
#include <GrAtscInterleaver.h>
#include <GrAtscTrellisEncoder.h>
#include <GrAtscFieldSyncMux.h>
#include <GrAtscSymbolMapper.h>
#include <GrWeaverModHead.h>
#include <GrWeaverModTail.h>
#include <GrFIRfilterFFF.h>
#include <GrAtscConvert2xTo20.h>
#include <gr_firdes.h>
#include <atsc_types.h>
#include <atsc_consts.h>
#include <unistd.h>


typedef short		output_type;

// INPUT_RATE is in units of MPEG transport packets / second
static const double INPUT_RATE = ATSC_DATA_SEGMENT_RATE*312./313.;

// output_rate is in units of samples / second
static const double OUTPUT_RATE = 2 * ATSC_SYMBOL_RATE;

static const double IF_freq = 5.75e6;		// output freq of tuner module

//static const int NMFTAPS = 93;		// # of matched filter taps
static const int NMFTAPS = 139;			// # of matched filter taps

// freq of hdtv suppressed carrier pilot tone.  
//
// This assumes that the tuner is actually tuned to the middle of the 6
// MHz channel.  The pilot is 0.31 MHz up from the bottom edge of the channel

static const double pilot_freq = IF_freq - 3e6 + 0.31e6;


static const double MIXER_GAIN = 128.0;	// linear value


void
usage (const char *name)
{
  cerr << "usage: " << name
       << " [-s {20 | 2x}] [-g] [-f <filename>] [-o <outfile>] [-p <port_number>] [-l]\n";
  exit (1);
}

int 
main (int argc, char **argv)
{
  try {
    
    char *input_filename  = "mpeg.in";
    char *output_filename = "atsc_tx.txout";
    int	port = 0;
    bool logging_p = false;
    bool output_is_20M_p = true;

    int	ch;
    while ((ch = getopt (argc, argv, "s:f:o:p:lg")) != EOF){
      switch (ch){
      case 's':
	if (strcmp (optarg, "20") == 0)		// 20 MS/s output
	  output_is_20M_p = true;
	else if (strcmp (optarg, "2x") == 0)	// 21.52 MS/s output
	  output_is_20M_p = false;
	else
	  usage (argv[0]);
	break;
	
      case 'f':
	input_filename = optarg;
	break;
    
      case 'o':
	output_filename = optarg;
	break;

      case 'p':
	port = strtol (optarg, 0, 0);
	break;

      case 'l':
	logging_p = true;
	break;
      
      case 'g':
	break;
	
      default:
	usage (argv[0]);
      }
    }

    if (input_filename == 0)
      usage (argv[0]);
    
    if (optind != argc)
      usage (argv[0]);

    // create the modules

    VrFileSource<atsc_mpeg_packet>* source =
      new VrFileSource<atsc_mpeg_packet>(INPUT_RATE, input_filename, 0);

    // These are the basic "grind on the data" modules that don't really
    // have anything to do with digital signal processing
  
    GrAtscRandomizer      	 *randomizer = new GrAtscRandomizer();
    GrAtscRSEncoder       	 *rs_encoder = new GrAtscRSEncoder();
    GrAtscInterleaver     	 *interleaver = new GrAtscInterleaver();
    GrAtscTrellisEncoder  	 *trellis_encoder = new GrAtscTrellisEncoder();
    GrAtscFieldSyncMux		 *field_sync_mux = new GrAtscFieldSyncMux ();

    // These are the digital signal processing modules.
    // We instantiate them below.

    GrAtscSymbolMapper<float>	       *mapper = 0;
    GrWeaverModHead<float,float>       *weaver_head = 0;
    GrWeaverModTail<float,output_type> *weaver_tail = 0;
    GrFIRfilterFFF		       *mfI = 0;
    GrFIRfilterFFF		       *mfQ = 0;
    GrAtscConvert2xTo20		       *interpI = 0;
    GrAtscConvert2xTo20		       *interpQ = 0;

    VrSink<output_type>	               *final_sink = 0;

    // optional logging sinks
  
    VrFileSink<atsc_mpeg_packet_no_sync>    *randomizer_log = 0;
    VrFileSink<atsc_mpeg_packet_rs_encoded> *rs_encoder_log = 0;
    VrFileSink<atsc_mpeg_packet_rs_encoded> *interleaver_log = 0;
    VrFileSink<atsc_data_segment>           *trellis_encoder_log = 0;
    VrFileSink<atsc_data_segment>	    *field_sync_mux_log = 0;
    VrFileSink<float>			    *mapper_log = 0;


    // We've already instantiated:
    //
    //   Read the MPEG data
    //   Randomize it (whiten)
    //   Reed-Solomon encode
    //   Convolutional Interleave
    //   Trellis Encode
    //   FieldSyncMux (insert field syncs as appropriate)
    //

    // ================================================================
    //   Pass it through the symbol mapper
    //   (includes symbol mapping & pilot addition)
    // ================================================================

    mapper = new GrAtscSymbolMapper<float>();

    // ================================================================
    // 			Build Weaver Modulator
    // ================================================================

    // mix with cos/sin Fsym/4 and interpolate by 2
    weaver_head = new GrWeaverModHead<float,float>(2);

    // gain of 2 to make up for the gain of 1/2 of the interpolate by 2
    mfI = new GrFIRfilterFFF (1, gr_firdes::root_raised_cosine (2.0,
								OUTPUT_RATE,
								ATSC_SYMBOL_RATE/2,
								.115, NMFTAPS));

    mfQ = new GrFIRfilterFFF (1, gr_firdes::root_raised_cosine (2.0,
								OUTPUT_RATE,
								ATSC_SYMBOL_RATE/2,
								.115, NMFTAPS));

    // We drop the interpolators in here, instead of at the end,
    // because the bandwidth of interest is much lower here then after
    // the up-conversion to the IF.  This allows us to use interpolators that
    // have many fewer taps.

    interpI = new GrAtscConvert2xTo20 ();	// 21.52 MS/s to 20 MS/s
    interpQ = new GrAtscConvert2xTo20 ();
    
    
    // mix with cos/sin IF_freq, sum components
    weaver_tail = new GrWeaverModTail<float,output_type>(IF_freq, MIXER_GAIN);


    // final sink
    if (port)
      final_sink = new GrTCPSink<output_type>(port);
    else
      final_sink = new VrFileSink<output_type>(output_filename);


    // create empty flow graph
    gr_FlowGraph *fg = gr_FlowGraph::make ();


    /* Hook up the connections among the modules now.  */

    fg->connect (source, randomizer);
    fg->connect (randomizer, rs_encoder);
    fg->connect (rs_encoder, interleaver);
    fg->connect (interleaver, trellis_encoder);
    fg->connect (trellis_encoder, field_sync_mux);
    fg->connect (field_sync_mux, mapper);
    fg->connect (mapper, weaver_head);
    fg->connect (weaver_head, 0, mfI);
    fg->connect (weaver_head, 1, mfQ);

    if (output_is_20M_p){		// splice in the sample rate converters
      fg->connect (mfI, interpI);
      fg->connect (mfQ, interpQ);
      fg->connect (interpI, weaver_tail);
      fg->connect (interpQ, weaver_tail);
    }
    else {
      fg->connect (mfI, weaver_tail);
      fg->connect (mfQ, weaver_tail);
    }

    fg->connect (weaver_tail, final_sink);


    if (logging_p){
      randomizer_log = new VrFileSink<atsc_mpeg_packet_no_sync>("randomizer.txout");
      rs_encoder_log = new VrFileSink<atsc_mpeg_packet_rs_encoded>("rs_encoder.txout");
      interleaver_log = new VrFileSink<atsc_mpeg_packet_rs_encoded>("interleaver.txout");
      trellis_encoder_log = new VrFileSink<atsc_data_segment>("trellis_encoder.txout");
      field_sync_mux_log = new VrFileSink<atsc_data_segment>("field_sync_mux.txout");
      mapper_log = new VrFileSink<float>("mapper.txout");
      
      fg->connect (randomizer,      randomizer_log);
      fg->connect (rs_encoder,      rs_encoder_log);
      fg->connect (interleaver,     interleaver_log);
      fg->connect (trellis_encoder, trellis_encoder_log);
      fg->connect (field_sync_mux,  field_sync_mux_log);
      fg->connect (mapper,	    mapper_log);
    }


    fg->start ();		// start executing flow graph (forks N threads)

    fg->wait ();		// wait (forever)

  } // end try

  catch (std::exception &e){
    cerr << "std library exception: " << e.what () << endl;
    exit (1);
  }
  catch (...) {
    cerr << "unknown exception thrown" << endl;
    exit (1);
  }

  return 0;
}
