/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <atsc_consts.h>
#include <vector>
#include <cmath>
#include <gr_mmse_fir_interpolator.h>


static const double	input_rate = 20e6;
static const double	output_rate = 2 * ATSC_SYMBOL_RATE;

// ----------------------------------------------------------------

class in_thing {
public:
  in_thing (FILE *fp) : fp(fp) {}
  std::vector<float>	contents;

protected:
  FILE			*fp;
};

class in_float : public in_thing {
protected:
  typedef float	element_t;

public:
  in_float (FILE *a_fp) : in_thing (a_fp) {
    element_t	io_sample;
    
    while (fread (&io_sample, sizeof (io_sample), 1, fp) == 1)
      contents.push_back ((float) io_sample);
  }
};

class in_short : public in_thing {
protected:
  typedef short	element_t;

public:
  in_short (FILE *a_fp) : in_thing (a_fp) {
    element_t	io_sample;
    
    while (fread (&io_sample, sizeof (io_sample), 1, fp) == 1)
      contents.push_back ((float) io_sample);
  }
};

// ----------------------------------------------------------------

class out_thing {
public:
  out_thing (FILE *fp) : fp(fp) {}
  virtual bool write_sample (float sample) = 0;

protected:
  FILE	*fp;
};

class out_float : public out_thing {
  typedef float	element_t;

public:
  out_float (FILE *a_fp) : out_thing (a_fp) {}
  virtual bool write_sample (float sample){
    element_t	io_sample = (element_t) sample;
    return (fwrite (&io_sample, sizeof (io_sample), 1, fp) == 1);
  }
};

class out_short : public out_thing {
  typedef short	element_t;

public:
  out_short (FILE *a_fp) : out_thing (a_fp) {}
  virtual bool write_sample (float sample){
    element_t	io_sample = (element_t) rint (sample);
    return (fwrite (&io_sample, sizeof (io_sample), 1, fp) == 1);
  }
};

// ----------------------------------------------------------------

static void
resample (in_thing &in, out_thing &out, double input_rate, double output_rate);

static void
usage (const char *name)
{
  fprintf (stderr, 
   "usage: %s {-f|-F} <infile>  {-o|-O} <outfile>\n", name);
  fprintf (stderr,   "  -f <infile>   : input file contains shorts\n");
  fprintf (stderr,   "  -F <infile>   : input file contains floats\n");
  fprintf (stderr,   "  -o <outfile>  : output file contains shorts\n");
  fprintf (stderr,   "  -O <outfile>  : output file contains floats\n");
  exit (1);
}

int 
main (int argc, char **argv)
{
  int		c;
  char		*input_filename = 0;
  char		*output_filename = 0;
  bool		input_float_p = false;
  bool		output_float_p = false;

  while ((c = getopt (argc, argv, "F:f:O:o:")) != EOF){
    switch (c){

    case 'f':
      if (input_filename != 0){
	fprintf (stderr, "%s: -F and -f are mutually exclusive\n", argv[0]);
	exit (1);
      }
      input_filename = optarg;
      input_float_p = false;
      break;

    case 'F':
      if (input_filename != 0){
	fprintf (stderr, "%s: -F and -f are mutually exclusive\n", argv[0]);
	exit (1);
      }
      input_filename = optarg;
      input_float_p = true;
      break;

    case 'o':
      if (output_filename != 0){
	fprintf (stderr, "%s: -O and -o are mutually exclusive\n", argv[0]);
	exit (1);
      }
      output_filename = optarg;
      output_float_p = false;
      break;

    case 'O':
      if (output_filename != 0){
	fprintf (stderr, "%s: -O and -o are mutually exclusive\n", argv[0]);
	exit (1);
      }
      output_filename = optarg;
      output_float_p = true;
      break;


    default:
      usage (argv[0]);
      break;
    }
  }

  if (optind != argc)
    usage (argv[0]);

  
  
  FILE *in_fp = fopen (input_filename, "rb");
  if (in_fp == 0){
    perror (input_filename);
    exit (1);
  }

  in_thing	*input;
  if (input_float_p)
    input = new in_float (in_fp);
  else
    input = new in_short (in_fp);


  FILE *out_fp = fopen (output_filename, "wb");
  if (out_fp == 0){
    perror (output_filename);
    exit (1);
  }

  out_thing	*output;
  if (output_float_p)
    output = new out_float (out_fp);
  else
    output = new out_short (out_fp);


  // do the work

  resample (*input, *output, input_rate, output_rate);

  // clean up

  fclose (in_fp);
  fclose (out_fp);

  delete input;
  delete output;

  return 0;
}
  
// ----------------------------------------------------------------
//			resample input to output
// ----------------------------------------------------------------

static void
resample (in_thing &in, out_thing &out, double input_rate, double output_rate)
{
  double 	w = input_rate / output_rate;

  // convert vector to array for use by interpolator

  float		*input = new float [in.contents.size ()];
  for (unsigned int i = 0; i < in.contents.size (); i++)
    input[i] = in.contents[i];
  
  gr_mmse_fir_interpolator	intr;
  unsigned 			limit = in.contents.size () - intr.ntaps ();
  unsigned long			int_part = 0;
  double			frac_part = 0.0;

  while (int_part < limit){

    // produce output sample

    float	s;
    s = intr.interpolate (&input[int_part], 1.0 - frac_part);
    out.write_sample (s);

    // fprintf (stdout, "%10ld\t%10.8f\n", int_part, frac_part);
    
    // update "position"
    
    double t = frac_part + w;
    double f = floor (t);
    frac_part = t - f;
    int_part = int_part + (unsigned long) f;
  }

  delete [] input;
}


