/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <GrUsrpSource.h>
#include <stdexcept>
#include <algorithm>
#include <config.h>

#ifdef HAVE_USRP

#include <usrp.h>
#include <usrp_config.h>


static const int MIN_READ_SIZE = 4 * 1024;	// in bytes (must be multiple of 512)
						// guts of user mode usb driver currently does
						// i/o to kernel in max 1 page chunks, hence 4K


GrUsrpSource::GrUsrpSource (int which_board, unsigned int usrp_rx_config,
			    unsigned int adc_clk_div, unsigned int decim_rate)
  : GrSource (sizeof (VrComplex), usrp_rx_config_stream_count (usrp_rx_config))
{
  d_noutputs = usrp_rx_config_stream_count (usrp_rx_config);
  if (d_noutputs != 1 && d_noutputs != 2 && d_noutputs != 4 && d_noutputs != 8){
    fprintf (stderr, "GrUsrpSource: bad usrp_rx_config\n");
    throw std::runtime_error ("bad usrp_rx_config");
  }

  d_usrp = usrp_rx::make (which_board);
  if (d_usrp == 0)
    throw std::runtime_error ("can't open USRP");

  fprintf (stderr, "GrUsrpSource: opened!\n");

  if (!d_usrp->set_adc_clk_div (adc_clk_div))
    throw std::runtime_error ("bad adc_clk_div");

  if (!d_usrp->set_decim_rate (decim_rate))
    throw std::runtime_error ("bad decim_rate");

  // FIXME needs attention for usrp_rx_config


  double our_sampling_freq =
    d_usrp->get_oscillator_freq () / adc_clk_div / decim_rate;
  
  setSamplingFrequency (our_sampling_freq);


  // We want all of our low level reads to be multiples of MIN_READ_SIZE,
  // therefore set output size like this...

  int output_size = MIN_READ_SIZE / (2 * sizeof (short) * d_noutputs);
  fprintf (stderr, "GrUsrpSource: setOutputSize (%d)\n", output_size);
  
  setOutputSize (output_size);
}

GrUsrpSource::~GrUsrpSource ()
{
  delete d_usrp;
}

GrUsrpSource *
GrUsrpSource::make (int which_board, unsigned int usrp_rx_config,
		    unsigned int adc_clk_div, unsigned int decim_rate)
{
  try {
    return new GrUsrpSource (which_board, usrp_rx_config, adc_clk_div, decim_rate);
  }
  catch (...){
    return 0;
  }
}
		  
int 
GrUsrpSource::work2 (VrSampleRange output, void *ao[])
{
  sync (output.index);		// force in-order execution

  if ((int) numberOutputs != d_noutputs){
    throw std::runtime_error ("wrong number of outputs");
  }
  
  int    size = output.size;
  VrComplex **out = (VrComplex **) ao;

  assert ((output.size * 2 * sizeof (short) * d_noutputs) % MIN_READ_SIZE == 0);

  int shorts_per_item = 2 * d_noutputs;		// I + Q for each output
  int bytes_per_item  = shorts_per_item * sizeof (short);

  int base = 0;
  bool overrun;

  while (size > 0){

    static const int BUFMULT = 1;		// could jack up the 1 if useful
    static const int MAX_BYTES = MIN_READ_SIZE * BUFMULT;
    static const int MAX_SHORTS = MAX_BYTES / sizeof (short);
    short buf[MAX_SHORTS];
    int max_items = MAX_SHORTS / shorts_per_item;
    
    int nitems = std::min (size, max_items);	// number of items to read this time through
    int	nbytes = nitems * bytes_per_item; 	// corresponding number of bytes

    int result_nbytes = d_usrp->read (buf, nbytes, &overrun);// read the raw USB data
    if (overrun)
      fprintf (stderr, "GrUsrpSource: overrun\n");

    if (result_nbytes < 0){			// got a problem
      fprintf (stderr, "GrUsrpSource: d_usrp->read problem!\n");
      break;
    }
    
    if (result_nbytes != nbytes){		// not really an error, but unexpected
      fprintf (stderr, "GrUsrpSource: short read.  expected %d, got %d\n",
	       nbytes, result_nbytes);
    }

    int result_nitems = result_nbytes / (shorts_per_item * sizeof (short));

    // now unpack the raw data into their respective VrComplex output streams

    switch (d_noutputs){
    case 1:
      for (int i = 0, j = 0; i < result_nitems; i++){
	out[0][base+i] = VrComplex ((float) buf[j+0], (float) buf[j+1]);
	j += 2;
      }
      break;

    case 2:
      for (int i = 0, j = 0; i < result_nitems; i++){
	out[0][base+i] = VrComplex ((float) buf[j+0], (float) buf[j+1]);
	out[1][base+i] = VrComplex ((float) buf[j+2], (float) buf[j+3]);
	j += 4;
      }
      break;

    case 4:
      for (int i = 0, j = 0; i < result_nitems; i++){
	out[0][base+i] = VrComplex ((float) buf[j+0], (float) buf[j+1]);
	out[1][base+i] = VrComplex ((float) buf[j+2], (float) buf[j+3]);
	out[2][base+i] = VrComplex ((float) buf[j+4], (float) buf[j+5]);
	out[3][base+i] = VrComplex ((float) buf[j+6], (float) buf[j+7]);
	j += 8;
      }
      break;

    case 8:
      for (int i = 0, j = 0; i < result_nitems; i++){
	out[0][base+i] = VrComplex ((float) buf[j+0], (float) buf[j+1]);
	out[1][base+i] = VrComplex ((float) buf[j+2], (float) buf[j+3]);
	out[2][base+i] = VrComplex ((float) buf[j+4], (float) buf[j+5]);
	out[3][base+i] = VrComplex ((float) buf[j+6], (float) buf[j+7]);
	out[4][base+i] = VrComplex ((float) buf[j+8], (float) buf[j+9]);
	out[5][base+i] = VrComplex ((float) buf[j+10], (float) buf[j+11]);
	out[6][base+i] = VrComplex ((float) buf[j+12], (float) buf[j+13]);
	out[7][base+i] = VrComplex ((float) buf[j+14], (float) buf[j+15]);
	j += 16;
      }
      break;

    default:
      assert (0);
    }

    size -= result_nitems;
    base += result_nitems;
  }

  return output.size - size;
}


bool
GrUsrpSource::set_adc_clk_div (unsigned int div)
{
  return d_usrp->set_adc_clk_div (div);
}

bool
GrUsrpSource::set_decim_rate (unsigned int rate)
{
  return d_usrp->set_decim_rate (rate);
}

bool
GrUsrpSource::set_rx_freq (int channel, double freq)
{
  return d_usrp->set_rx_freq (channel, freq);
}

bool
GrUsrpSource::set_ext_clk_div (unsigned int div)
{
  return d_usrp->set_ext_clk_div (div);
}

bool
GrUsrpSource::_set_decim_reg (unsigned int regval)
{
  return d_usrp->_set_decim_reg (regval);
}

// ACCESSORS

double
GrUsrpSource::get_oscillator_freq () const
{
  return d_usrp->get_oscillator_freq ();
}

unsigned int 
GrUsrpSource::get_adc_clk_div () const
{
  return d_usrp->get_adc_clk_div ();
}

unsigned int
GrUsrpSource::get_decim_rate () const
{
  return d_usrp->get_decim_rate ();
}

double
GrUsrpSource::get_rx_freq (int channel) const
{
  return d_usrp->get_rx_freq (channel);
}

unsigned int
GrUsrpSource::get_ext_clk_div () const
{
  return d_usrp->get_ext_clk_div ();
}

#else		

// ----------------------------------------------------------------
// 			stub it all out
// ----------------------------------------------------------------

GrUsrpSource::GrUsrpSource (int which_board, unsigned int usrp_rx_config,
			    unsigned int adc_clk_div, unsigned int decim_rate)
  : GrSource (sizeof (VrComplex), 1)
{
  fprintf (stderr, "GrUsrpSource: stub version.  HAVE_USRP not defined\n");

  double our_sampling_freq =
    get_oscillator_freq () / adc_clk_div / decim_rate;
  
  setSamplingFrequency (our_sampling_freq);
}

GrUsrpSource::~GrUsrpSource ()
{
}

int 
GrUsrpSource::work2 (VrSampleRange output, void *ao[])
{
  return output.size;
}

bool
GrUsrpSource::set_adc_clk_div (unsigned int div)
{
  return false;
}

bool
GrUsrpSource::set_decim_rate (unsigned int rate)
{
  return false;
}

bool
GrUsrpSource::set_rx_freq (int channel, double freq)
{
  return false;
}

bool
GrUsrpSource::set_ext_clk_div (unsigned int div)
{
  return false;
}

bool
GrUsrpSource::_set_decim_reg (unsigned int regval)
{
  return false;
}

// ACCESSORS

double
GrUsrpSource::get_oscillator_freq () const
{
  return 120e6;
}

unsigned int 
GrUsrpSource::get_adc_clk_div () const
{
  return 2;
}

unsigned int
GrUsrpSource::get_decim_rate () const
{
  return 256;
}

double
GrUsrpSource::get_rx_freq (int channel) const
{
  return 0;
}

unsigned int
GrUsrpSource::get_ext_clk_div () const
{
  return 12;
}

#endif
