/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <usrp1_source_s.h>
#include <gr_io_signature.h>
#include <usrp_standard.h>

static const int NUSRP_BYTES_PER_ITEM = 1 * sizeof (short);


usrp1_source_s_sptr
usrp1_make_source_s (int which_board, 
		     unsigned int decim_rate,
		     int nchan,
		     int mux,
		     int mode
		     ) throw (std::runtime_error)
{
  return usrp1_source_s_sptr (new usrp1_source_s (which_board,
						  decim_rate, 
						  nchan,
						  mux,
						  mode));
}


usrp1_source_s::usrp1_source_s (int which_board,
				unsigned int decim_rate,
				int nchan,
				int mux,
				int mode
				) throw (std::runtime_error)
  : usrp1_source_base ("usrp1_source_s",
		       gr_make_io_signature (1, 1, sizeof (short)),
		       which_board, decim_rate, nchan, mux, mode)
{
}

usrp1_source_s::~usrp1_source_s ()
{
  // NOP
}

int
usrp1_source_s::ninput_bytes_reqd_for_noutput_items (int noutput_items)
{
  return noutput_items * NUSRP_BYTES_PER_ITEM;
}

/*
 * Convert interleaved 16-bit I & Q from usrp buffer into a single
 * short output stream.
 */
void
usrp1_source_s::copy_from_usrp_buffer (gr_vector_void_star &output_items,
				       int output_index,
				       int output_items_available,
				       int &output_items_produced,
				       const void *usrp_buffer,
				       int usrp_buffer_length,
				       int &bytes_read)
{
  short *out = &((short *) output_items[0])[output_index];
  short *src = (short *) usrp_buffer;

  int nitems = std::min (output_items_available,
			 usrp_buffer_length / NUSRP_BYTES_PER_ITEM);

  for (int i = 0; i < nitems; i++){
    out[i] = src[i];
  }

  output_items_produced = nitems;
  bytes_read = nitems * NUSRP_BYTES_PER_ITEM;
}
