/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	Author:  Thomas Fruchterman
 *		 John Jozwiak
 *		 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

#include "config.h"
#include <stdio.h>
#include <mbus/api.h>
#include <mbus/keyword.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include  <X11/Intrinsic.h>		/* needed for stuff in SetupBus */
#include "extern.h"
#include "graph.h"
#include "callback.h"
#include "elision.h"
#include "mbus-parse.h"

#if POLL
#include <stropts.h>
#include <poll.h>
#endif

int MBLogLevel = 1;
t_mb_connection conn;

/*-------------------------------------------------------------------------*/
/* Structure for making linked lists out of node lists. */

struct node_link
{
  Vertex *node;
  struct node_link *next;
};

/* ------------------------------------------------------------------------ */
/* Structures for parsing keyword arguments */

static struct keyword_entry_struct NodeKeywords[] =
{
  { ":label", NULL, KEYWORD_COOKED },	                /* 0 */
  { ":type", NULL, KEYWORD_COOKED },                    /* 1 */
  { ":short", NULL, KEYWORD_COOKED },                   /* 2 */
  { ":long", NULL, KEYWORD_COOKED },                    /* 3 */
  { ":add", (void *)KEYWORD_FALSE, KEYWORD_FLAG },      /* 8 */
  { ":update", (void *)KEYWORD_FALSE, KEYWORD_FLAG },   /* 9 */
  { ":remove", (void *)KEYWORD_FALSE, KEYWORD_FLAG },   /* 10 */
  { ":fixed", (void *)KEYWORD_FALSE, KEYWORD_FLAG },    /* 11 */
  { ":xpos", NULL, KEYWORD_COOKED },                    /* 12 */
  { ":ypos", NULL, KEYWORD_COOKED },                    /* 13 */
  { ":data", NULL, KEYWORD_SEXP },
};

static struct keyword_entry_struct EdgeKeywords[] =
{
  { ":label", NULL, KEYWORD_COOKED },
  { ":type", NULL, KEYWORD_COOKED },
  { ":short", NULL, KEYWORD_COOKED },
  { ":long", NULL, KEYWORD_COOKED },
  { ":to-list", NULL, KEYWORD_SEXP },
  { ":from-list", NULL, KEYWORD_SEXP },
  { ":directed", (void *)KEYWORD_FALSE, KEYWORD_FLAG },
  { ":add", (void *)KEYWORD_FALSE, KEYWORD_FLAG },
  { ":update", (void *)KEYWORD_FALSE, KEYWORD_FLAG },
  { ":remove", (void *)KEYWORD_FALSE, KEYWORD_FLAG },
  { ":data", NULL, KEYWORD_SEXP },
};

static struct keyword_entry_struct GraphKeywords[] =
{
  { ":clear", (void *)KEYWORD_FALSE, KEYWORD_FLAG }, /* 0 */
  { ":replace", (void *)KEYWORD_FALSE, KEYWORD_FLAG }, /* 1 */
  { ":create", (void *)KEYWORD_FALSE, KEYWORD_FLAG }, /* 5 */
  { ":title", "Browser", KEYWORD_COOKED }, /* 6 */
  { ":displaytable", NULL, KEYWORD_SEXP },  /* 7 */
  { ":callbacks", NULL, KEYWORD_SEXP },
  { ":inbound_policy", NULL, KEYWORD_SEXP},
  { ":inbound_final", NULL, KEYWORD_SEXP},
  { ":outbound_policy", NULL, KEYWORD_SEXP},
  { ":outbound_final", NULL, KEYWORD_SEXP},
  { ":undirected_policy", NULL, KEYWORD_SEXP},
  { ":undirected_final", NULL, KEYWORD_SEXP},
  { ":label_policy", NULL, KEYWORD_SEXP},
  { ":anti_inbound_policy", NULL, KEYWORD_SEXP},
  { ":anti_outbound_policy", NULL, KEYWORD_SEXP},
  { ":anti_undirected_policy", NULL, KEYWORD_SEXP},
  { ":anti_diff_set_policy", NULL, KEYWORD_SEXP},
  { ":ws", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":active_type", NULL, KEYWORD_COOKED}
};

/*
 *  ConvertNodeList takes a structure of type mb_object and converts
 *  it to a linked list of Vertex *.  This is so that nodes that do
 *  not exist in the graph, but are specified one of the edge's node
 *  lists, are removed from the list, never to be seen again.
*/
struct node_link *
ConvertNodeList(list)
     struct mb_object *list;
{
  struct node_link *head, *entry;
  Vertex *node;

  head = NULL;
  for (; MBconsp(list); list = MBcdr(list))
    if ((node = NodeLookup(MBCstring(MBcar(list)), graph->v))) {
      entry = NEW(struct node_link);
      entry->node = node;
      entry->next = head;
      head = entry;
    }
  return head;
}

#if 0
void
FreeNodeList(list)
     struct node_link *list;
{
  struct node_link *head, *ptr;

  for (ptr = list; ptr != NULL; ) {
    head = ptr;
    ptr = ptr->next;
    free(head);
  }
}
#endif

/*
   ReassignStrVal was created by Doug Bogia on Aug. 26, 1991
   Description:
     This takes the the new data and if it is non-null it will
     free the old data, and then assigns it the new value.
   Entry:
     FillValue-------> This is the value to fill in.  It may have old data
     NewValue--------> This is the new value to assign to the FillValue
   Exit:
     FillValue-------> Will have the new data
     NewValue--------> None.
   Returns:
     None.
*/
void ReassignStrVal (FillValue, NewValue)
  char **          FillValue;
  char *           NewValue;
{
  /* Start of variables section */
  /* End of variables section */
  if (NewValue)
  {
    if (*FillValue)
    {
      free(*FillValue);
    }
    *FillValue = NewValue;
  }
} /* End of ReassignStrVal */

/*
 *  ReassignKeys
*/
void
ReassignKeys(new_keys, old_keys)
     KeyList *new_keys, **old_keys;
{
  KeyList *new_entry, *old_entry, *tmpold, *tmpnew;

  if (*old_keys == NULL) {
    *old_keys = new_keys;
    return;
  }
  tmpnew = new_keys;
  while (new_entry = tmpnew) {
    tmpold = *old_keys;
    while (old_entry = tmpold) {
      if (!strcmp(old_entry->key, new_entry->key)) { /* found a match */
	break;
      }
      tmpold = old_entry->next;
    }
    tmpnew = tmpnew->next;
    if (tmpold == NULL) {
      new_entry->next = *old_keys;
      *old_keys = new_entry;
    }
    else {
      if (old_entry->value != NULL)
	free (old_entry->value);
      old_entry->value = new_entry->value;
      free (new_entry->key);
      free(new_entry);
    }
  }
}

/*
 *  ParseSuperNode will take a message that is sent by the widget server
 *  that contains update information for a given supernode, and place
 *  that information into the right node.  This is used to (possibly) change
 *  the labels and type for the supernode created by elision
*/
int
ParseSuperNode(node, values)
     char *node;
     struct mb_object *values;
{
  Vertex *super_node;

  super_node = NodeLookup(node, graph->v);
  MBparse_keywords(values, NodeKeywords, ARRAY_SIZE(NodeKeywords));
  
  ReassignStrVal (&super_node->values->labels[0],
		  NodeKeywords[NODELABEL].result);
  /* Look at the short label */
  ReassignStrVal (&super_node->values->labels[1],
		  NodeKeywords[NODESHORT].result);
  /* Look at the long label */
  ReassignStrVal (&super_node->values->labels[2],
		  NodeKeywords[NODELONG].result);
  if (NodeKeywords[NODETYPE].result)
  {
    /* Look at the type */
    ReassignStrVal (&super_node->values->type,
		    NodeKeywords[NODETYPE].result);
/*       CheckTables (super_node->values, 0); */
  }
  AssignNodeLabel(super_node);
  return REDRAW;		/* Always make a change */
}

/*
 *  ParseData takes an mb_object that contains key/value pairs, and parses
 *  them.  This data is used in generating message bus callbacks.  It is in
 *  this fasion that the user can define data on a certain node (or edge)
 *  and have this data passed back down the bus.
*/
KeyList *
ParseData(data)
     struct mb_object *data;
{
  struct mb_object *data_item;
  char *key, *value;
  KeyList *new_keys, *new_entry;;

  new_keys = NULL;
  while (data_item = MBcar(data)) {
    new_entry = NEW(KeyList);
    key = MBCstring(MBnth(data_item, 0));
    value = MBCstring(MBnth(data_item, 1));
    new_entry->key = key;
    new_entry->value = value;
    new_entry->next = new_keys;
    new_keys = new_entry;
    data = MBcdr(data);
  }
  return new_keys;
}

int
ParseNodes(nlist)
     struct mb_object *nlist;
{
  struct mb_object *node, *data;
  KeyList *new_keys;
  Vertex *nodeptr;
  char *name;
  int change_made = 0;
  
  for (; MBconsp(nlist); nlist = MBcdr(nlist)) {
    node = MBcar(nlist);
    MBparse_keywords(MBnthcdr(node,1), NodeKeywords, ARRAY_SIZE(NodeKeywords));
    data = NodeKeywords[NODEDATA].result;
    name = MBCstring(MBcar(node));
    if (!name)
    {
#ifdef DEBUG
      fprintf (stderr, "ParseNodes:  Illegal node list\n");
#endif      
      continue;
    }
    if ((nodeptr = NodeLookup(name, graph->v))) {
      if ((t_keyword_flag)NodeKeywords[NODEUPDATE].result == KEYWORD_TRUE) {
	new_keys = ParseData(data);
/*
 *  We have an update.  Substitute any of the keywords values that are present
 *  for the old values of the node
*/
	/* Look at the label */
	ReassignStrVal (&nodeptr->values->labels[0],
			NodeKeywords[NODELABEL].result);
	/* Look at the short label */
	ReassignStrVal (&nodeptr->values->labels[1],
			NodeKeywords[NODESHORT].result);
	/* Look at the long label */
	ReassignStrVal (&nodeptr->values->labels[2],
			NodeKeywords[NODELONG].result);
	if (NodeKeywords[NODETYPE].result)
	{
	  /* Look at the type */
	  ReassignStrVal (&nodeptr->values->type,
			  NodeKeywords[NODETYPE].result);
/* 	  CheckTables (nodeptr->values, 0); */
	}
	ReassignKeys(new_keys, &nodeptr->values->key_list);
	AssignNodeLabel(nodeptr);
	change_made |= REDRAW;
      }
      else if((t_keyword_flag)NodeKeywords[NODEREMOVE].result == KEYWORD_TRUE)
      {
	/* If this node has links to other nodes, then removing it
	 * will affect some edges.  We should reparse the edge positioning
	 * in that case.
	 */
	if (nodeptr->links) change_made |= CHANGED_EDGES;
	DeleteNode(nodeptr);		/* check graph.c */
	change_made |= REDETERMINE; /* Need to redetermine the graph */
      }
      free(name);
    }
    else if ((t_keyword_flag)NodeKeywords[NODEADD].result == KEYWORD_TRUE) {
      new_keys = ParseData(data);
/*
 *  Here, we merely use the keywords and create a new node definition
*/
      nodeptr = MakeVertex(name,
		 (char *)NodeKeywords[NODELABEL].result,
		 (char *)NodeKeywords[NODETYPE].result,
		 (char *)NodeKeywords[NODESHORT].result,
		 (char *)NodeKeywords[NODELONG].result,
		 (((t_keyword_flag)NodeKeywords[NODEFIXED].result ==
		   KEYWORD_TRUE) ? FIXED : FALSE),
		 (char *)NodeKeywords[NODEXPOS].result,
		 (char *)NodeKeywords[NODEYPOS].result,
		 new_keys);
      /* Added a node.  Need to recompute the labels and redetermine */
      AssignNodeLabel(nodeptr);
      change_made |= REDETERMINE; /* New node added */
    }
    else
    {
      free(name);
    }
  }
  return change_made;
}
	
int
ParseEdges(elist)
     struct mb_object *elist;
{
  Edge *edgeptr;
  EdgeList *edge_list;
  struct mb_object *edge, *data;
  struct node_link *to_list, *from_list, *tlist, *flist;
  KeyList *new_keys;
  char *name;
  int change_made = 0;

  edge_list = NULL;
  for (; MBconsp(elist); elist = MBcdr(elist)) {
    edge = MBcar(elist);
    MBparse_keywords(MBnthcdr(edge,1), EdgeKeywords, ARRAY_SIZE(EdgeKeywords));
    data = EdgeKeywords[EDGEDATA].result;
    name = MBCstring(MBcar(edge));
    if (!name)
    {
#ifdef DEBUG
      fprintf (stderr, "ParseEdges:  Illegal edge list\n");
#endif      
      continue;
    }
    if ((edgeptr = EdgeLookup(name, graph->edges))) {
      if ((t_keyword_flag)EdgeKeywords[EDGEUPDATE].result == KEYWORD_TRUE) {
	new_keys = ParseData(data);
/*
 *  We have an update of a node.  Simply substitute the new values
 *  that are present in keywords for the old values.
*/
	/* Look at the label */
	ReassignStrVal (&edgeptr->values->labels[0],
			EdgeKeywords[EDGELABEL].result);
	/* Look at the short label */
	ReassignStrVal (&edgeptr->values->labels[1],
			EdgeKeywords[EDGESHORT].result);
	/* Look at the long label */
	ReassignStrVal (&edgeptr->values->labels[2],
			EdgeKeywords[EDGELONG].result);
	if (EdgeKeywords[EDGETYPE].result)
	{
	  /* Look at the type */
	  ReassignStrVal (&edgeptr->values->type,
			  EdgeKeywords[EDGETYPE].result);
/*	  CheckTables (edgeptr->values, 1); */
	}
	ReassignKeys(new_keys, &edgeptr->values->key_list);
	/* And check whether or not the edge is directed or not. */
	edgeptr->directed = ((t_keyword_flag)
			     EdgeKeywords[EDGEDIRECTED].result==KEYWORD_TRUE
			     ? TRUE : FALSE);
	change_made |= REDRAW;	/* Something got updated */
	/* TO DO:  Need to look at :from-list and :to-list.  If they
	 * are there, we should change to them?  If they are there,
	 * make sure that change_made is set to redetermine the graph too.
	 */
      }
      else if((t_keyword_flag)EdgeKeywords[EDGEREMOVE].result == KEYWORD_TRUE)
      {
	DeleteEdge(edgeptr);	/* check graph.c */
	change_made |= REDETERMINE; /* Something got updated */
      }
      free(name);
    }
    else if ((t_keyword_flag)EdgeKeywords[EDGEADD].result == KEYWORD_TRUE) {
      new_keys = ParseData(data);
/*
 *  Now, check on the from and to list for the edges.  If neither exist,
 *  then bog out on the edge.
*/
      to_list = ConvertNodeList((struct mb_object *)
				EdgeKeywords[EDGETOLIST].result);
      from_list = ConvertNodeList((struct mb_object *)
				  EdgeKeywords[EDGEFROMLIST].result);
      if (!to_list || !from_list) {
#ifdef DEBUG
	fprintf (stderr, "To and from node list must be specified on edge");
	fprintf (stderr, " %s: edge ignored.\n", MBCstring(MBcar(edge)));
#endif
	continue;
      }
      edgeptr = MakeEdge(name,
			 (char *)EdgeKeywords[EDGELABEL].result,
			 (char *)EdgeKeywords[EDGETYPE].result,
			 (char *)EdgeKeywords[EDGESHORT].result,
			 (char *)EdgeKeywords[EDGELONG].result,
			 new_keys);
      AddEdgetoList(&edge_list, edgeptr);
      edgeptr->directed = ((t_keyword_flag)
			   EdgeKeywords[EDGEDIRECTED].result==KEYWORD_TRUE
			   ? TRUE : FALSE);
      for (flist = from_list; flist != NULL; flist = flist->next)
	  for (tlist = to_list; tlist != NULL; tlist = tlist->next)
	      AddLink(flist->node, tlist->node, edgeptr);
      change_made |= (CHANGED_EDGES | REDETERMINE); /* Something got updated */
    }
    else 
    {
      free(name);
    }      
  }
  ReAddEdges(&edge_list);
  return change_made;
}

void
ParseGraph()
{
  extern t_mb_connection conn;
  struct mb_object *x, *message;
  struct mb_object *gdef;
  struct mb_object *s_id, *s_node, *s_edge, *s_common;
  Vertex *node;
  char *s_command, *gid;
  int i, read_ok, change_made;
  struct TableStruct *table_ptr, *table_ptr1;
  struct TableStruct *modified_graphs;
  struct TableStruct *next, *prev;
  /* Code below stolen from message bus code */
#if POLL
  struct pollfd p_fd;
#else /* not POLL, use SELECT */
  fd_set read_fds;
  struct timeval timer;
#endif

  modified_graphs = NULL;

  read_ok = 1;
  while (read_ok)
  {
    /* Check to see if there's data waiting. We do this in order to detect
     * a lost connection to the MBus
     */
#if POLL
    p_fd.fd = MBconnection_fd(conn);
    p_fd.events = POLLIN;
    read_ok = poll(&p_fd, 1, 0) > 0
	&& p_fd.revents & (POLLIN | POLLHUP);
#else
    FD_ZERO(&read_fds);
    FD_SET(MBconnection_fd(conn),&read_fds);
    timer.tv_sec = timer.tv_usec = 0;
    read_ok =
	select(MBconnection_fd(conn)+1, &read_fds, NULL, NULL,
	       &timer) > 0
		   && FD_ISSET(MBconnection_fd(conn),&read_fds);
#endif

    while (NULL != (message = MBNextMessage(conn))) {
      x = MBcdr(MBcdr(message)); /*  skip tag and domain  */
      while (MBconsp(x)) {
	gdef = MBcar(x);
	x = MBcdr(x);

	/*
	 *  get the different parts of the graph by calling mbnth.
	 *  the graph format is (command id (nodes) (edges) :keywords)
	 */
	s_command = MBCstring(MBcar(gdef));
	if (!s_command)
	    break;		/* Not a string */
	
	if (!strcmp(s_command, "graph")) {
	  s_id = MBnth(gdef,1);
	  if (!s_id) continue;	/* Sanity check */
	  s_node = MBnth(gdef,2);
	  s_edge = MBnth(gdef,3);

	  MBparse_keywords(MBnthcdr(gdef,4), GraphKeywords,
			   ARRAY_SIZE(GraphKeywords));
	  gid = MBCstring(s_id);
	  /*
	   *  Check to see if the graph exists in our internal tables or not
	   */
	  graph = GraphLookup(gid, GraphList);
	  /*
	   * If the graph doesn't exist, and the create keyword is nil,
	   * then loop through the rest of the code.
	   */
	  if (graph == NULL)
	  {
	    if ((t_keyword_flag)GraphKeywords[GRAPHCREATE].result == 
		KEYWORD_FALSE)
		continue;
	    else		/* Otherwise create it */
	    {
	      graph = NEW(Graph);
	      table_ptr = NEW(struct TableStruct);
	      table_ptr->id = gid;
	      table_ptr->graph = graph;
	      table_ptr->status = 0; /* Don't really care about this */
	      table_ptr->next = GraphList;
	      GraphList = table_ptr;
	      graph->v = NULL;
	      graph->edges = NULL;
	      graph->title = NULL;
	      graph->active_type = NULL;
	      graph->name_type = 0;
	      graph->have_hyper_edge = FALSE;
	      graph->virtx_min = 0.0;
	      graph->virty_min = 0.0;
	      graph->virtx_max = 1.0;
	      graph->virty_max = 1.0;
	      graph->display_table = NEW(struct DisplayTables);
	      graph->label_policy = ROOT_IS_LABEL;
	      graph->elision_policy = 0;
	      graph->unelision_policy = 0x249;
	      graph->display_table->common = NULL;
	      graph->display_table->nodes = NULL;
	      graph->display_table->edges = NULL;
	      graph->callback_keys = NULL;
	      graph->callback_functions = NULL;
	      graph->super_nodes = NULL;
	      SetXGraph((char *)GraphKeywords[GRAPHTITLE].result);
	      AddGraphtoList(&modified_graphs, graph, REDRAW);
	    }
	  }
/*
 *  Now check for the display table keyword. If the keyword exists, then we are
 *  adding a display table to the graph.
*/
	  if (GraphKeywords[GRAPHDISPLAYTABLE].result != NULL) {
	    if (ParseDisplayTable(GraphKeywords[GRAPHDISPLAYTABLE].result,
				  &(graph->display_table->common),
				  &(graph->display_table->nodes),
				  &(graph->display_table->edges), FALSE,
				  FALSE))
		AddGraphtoList (&modified_graphs, graph, REDRAW|CHANGED_NODES);
	  }
/*
 *  Now, check to see if there are any callbacks that are passed
 *  in this graph definition.  If so, then parse them.
*/
	  if (GraphKeywords[GRAPHCALLBACKS].result != NULL)
	    ParseCallbacks(GraphKeywords[GRAPHCALLBACKS].result,
			   &(graph->callback_keys),
			   &(graph->callback_functions), FALSE);
/*
 *  Now, we need to check out the elision policies.  We will parse out
 *  both the elide and unelide policies.  This gets rather complicated.
 *  Refer to the Programmers manual for more detail on elision.
*/
	  ParseElision(GraphKeywords[INELISION].result,
		       GraphKeywords[INBFINAL].result,
		       GraphKeywords[OUTELISION].result,
		       GraphKeywords[OUTBFINAL].result,
		       GraphKeywords[UNELISION].result,
		       GraphKeywords[UNBFINAL].result,
		       GraphKeywords[LABELPOLICY].result);
	  ParseAntiElision(GraphKeywords[ANTIIN].result,
			   GraphKeywords[ANTIOUT].result,
			   GraphKeywords[ANTIUNDIR].result,
			   GraphKeywords[ANTIDIFF].result);
/*
 *  Check to see if the clear keyword is true.  If it is true, then delete
 *  all of the nodes and edges in the graph, and then parse any new
 *  definition that comes in.
*/
	  if ((t_keyword_flag)GraphKeywords[GRAPHCLEAR].result == KEYWORD_TRUE)
	  {
	    ClearGraph(graph);
	    AddGraphtoList (&modified_graphs, graph, REDRAW);
	  }
/*
 *  Need to check the active type keyword on the graph
*/
	  if ((char *)GraphKeywords[ACTIVETYPE].result != NULL)
	    ReassignStrVal(&(graph->active_type),
			   (char *)GraphKeywords[ACTIVETYPE].result);
	  
	  if (change_made = ParseNodes(s_node))
	      AddGraphtoList(&modified_graphs, graph, change_made);
	  if (change_made = ParseEdges(s_edge))
	      AddGraphtoList(&modified_graphs, graph, change_made);
	      
	  if ((t_keyword_flag)GraphKeywords[GRAPHREPLACE].result == 
	      KEYWORD_TRUE)
	  {
	    RandomizeNodes();
	    AddGraphtoList (&modified_graphs, graph, REDETERMINE);
	  }    
	}
	else if (!strcmp(s_command, "displaytable")) {
/*
 *  get the id of what we are adding the color table too.  The word
 *  "browser" is reserved in this case to mean all graphs running under
 *  the currently running browser.  This gives global type-color tables.
*/
	  s_id = MBnth(gdef,1);
	  if (!s_id) continue;	/* Sanity check */
	  gid = MBCstring(s_id);
	  MBparse_keywords(MBnthcdr(gdef,3), GraphKeywords,
			   ARRAY_SIZE(GraphKeywords));
	  /*
	   *  Check and see if we are adding type tables to individual browser
	   */
	  if (!strcmp(gid, "browser") || !strcmp(gid, "Browser")) {
	    if (ParseDisplayTable(MBnth(gdef,2),
				  &(BrowserDisplays->common),
				  &(BrowserDisplays->nodes),
				  &(BrowserDisplays->edges), FALSE,
				  (t_keyword_flag)GraphKeywords[WS].result ==
				  KEYWORD_TRUE))
		/* If the color table changed */
		for (table_ptr = GraphList; table_ptr; 
		     table_ptr = table_ptr->next)
		    /*
		     * Gotta recompute where the nodes lie (for the case
		     * where a font changes and redraw everything when we
		     * change the global tables.
		     */
		    AddGraphtoList (&modified_graphs, table_ptr->graph,
				    REDRAW|CHANGED_NODES);
	  } else {
	    /*
	     *  Adding tables to a graph....
	     */
	    graph = GraphLookup(gid, GraphList);
	    if (graph != NULL &&
		ParseDisplayTable(MBnth(gdef,2),
				  &(graph->display_table->common),
				  &(graph->display_table->nodes),
				  &(graph->display_table->edges), FALSE,
				  (t_keyword_flag)GraphKeywords[WS].result ==
				  KEYWORD_TRUE))
		/* If the color table changed */
		AddGraphtoList (&modified_graphs, graph, REDRAW|CHANGED_NODES);
	  }
	}
	else if (!strcmp(s_command, "newdisplayentry")) {
/*
 *  parse a message from the widget server.  This is really internal use
 *  type stuff only, and should only be used by the widget server, and
 *  not someone programming this system.
*/
	  s_id = MBnth(gdef,1);
	  if (!s_id) continue;	/* Sanity check */
	  gid = MBCstring(s_id);
	  /*
	   *  Check and see if we are adding type tables to individual browser
	   */
	  if (!strcmp(gid, "browser") || !strcmp(gid, "Browser"))
	    ParseTableEntry(MBnthcdr(gdef,2));
	  else {
	    graph = GraphLookup(gid, GraphList);
	    if (graph != NULL)
	      ParseTableEntry(MBnthcdr(gdef,2));
	  }
	}
	else if (!strcmp(s_command, "callbacks")) {
/*
 *  Now, we want to parse the callback message.  Again, this is similar to
 *  the display tables in that the user can define callbacks either graph
 *  only, or browser wide.  The code here will be very similar.
*/
	  s_id = MBnth(gdef, 1);
	  if (!s_id) continue;	/* Sanity check */
	  gid = MBCstring(s_id);
	  MBparse_keywords(MBnthcdr(gdef,3), GraphKeywords,
			   ARRAY_SIZE(GraphKeywords));
/*
 *  check and see if we are adding callbacks to the browser
*/
	  if (!strcmp(gid, "browser") || !strcmp(gid, "Browser")) {
	    ParseCallbacks(MBnth(gdef,2),
			   &CallbackKeys,
			   &CallbackFunctions, FALSE);
	  }
	  else {
	    graph = GraphLookup(gid, GraphList);
	    if (graph != NULL) {
	      ParseCallbacks(MBnth(gdef,2),
			     &(graph->callback_keys),
			     &(graph->callback_functions), FALSE);
	    }
	  }
	}
/*
 *  The next command should only be sent by the widget server.
*/
	else if (!strncmp(s_command, "SN", 2) &&
		 (t_keyword_flag)GraphKeywords[WS].result == KEYWORD_TRUE)
	{
	  gid = MBCstring(MBnth(gdef,1));
	  graph = GraphLookup(gid, GraphList);
	  if (change_made = ParseSuperNode(s_command, MBnthcdr(gdef, 2)))
	      AddGraphtoList(&modified_graphs, graph, change_made);
	}
	else if (!strcmp(s_command, "shutdown")) 
	{
	  Quit(toplevel_widget, NULL, NULL);
	}
      }
    }
    MBfree(message);		/* Dump the message */
  }

  table_ptr = modified_graphs;

  while (table_ptr)
  {
    graph = table_ptr->graph;
    UpdateGraph(table_ptr->status);
    next = table_ptr->next;
    free (table_ptr);
    table_ptr = next;
  }
}

#ifdef NEVER
void
GetMessage()
{
  extern t_mb_connection conn;
  struct mb_object *x, *message;

  while (NULL != (message = MBNextMessage(conn))) {
    x = MBcdr(MBcdr(message));	/*  skip tag and domain  */
    while (MBconsp(x)) {
      ParseGraph(MBcar(x));
      x = MBcdr(x);
    }
    MBfree(message);		/* Dump the message */
  }
}
#endif

void SetupBus(port, host)
  int port;
  char *host;
{
  char buffer[500];
  extern char *cb_user, *cb_browser_domain, *cb_ui_domain, *cb_base_domain;

  GraphList = NULL;
  conn = MBConnect(host,port);
  if (NULL == conn) {
    fprintf (stderr, "Cannot connect to message-bus.\n");
    exit(1);
  }
  sprintf(buffer, "(id \"Browser for %s\") (accept \".*\" (\"%s.%s.%s.%s\" \"%s.%s.%s\" \"%s.%s\" \"%s.%s.%s\"))",
	  cb_user,
	  cb_browser_domain, cb_user, cb_ui_domain, cb_base_domain,
	  cb_user, cb_ui_domain, cb_base_domain,
	  cb_ui_domain, cb_base_domain,
	  cb_browser_domain, cb_ui_domain, cb_base_domain);
  MBtransmit_Cstring(conn, buffer);
}
