;;; w3-style.el,v --- Emacs-W3 binding style sheet mechanism
;; Author: wmperry
;; Created: 1995/05/29 18:11:32
;; Version: 1.29
;; Keywords: faces, hypermedia

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Copyright (c) 1993, 1994, 1995 by William M. Perry (wmperry@spry.com)
;;;
;;; This file is not part of GNU Emacs, but the same permissions apply.
;;;
;;; GNU Emacs is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; GNU Emacs is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to
;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; A style sheet mechanism for emacs-w3
;;;
;;; This will eventually be able to under DSSSL[-lite] as well as the
;;; experimental W3C mechanism
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fset 'w3-color-values (cond
			((fboundp 'x-color-values) 'x-color-values)
			((fboundp 'color-instance-rgb-components)
			 (function (lambda (x) (color-instance-rgb-components
						(make-color-instance x)))))
			(t (function (lambda (x) (list 0 0 0))))))

(defun w3-blend-colors (start end percentage)
  (interactive "sStart Color:
sEnd Color:
nPercentage: ")
  (setq percentage (max 0 (min percentage 100)))
  (let* ((vals    (w3-color-values start))
	 (red-1   (nth 0 vals))
	 (green-1 (nth 1 vals))
	 (blue-1  (nth 2 vals))
	 (new     (w3-color-values end))
	 (red-2   (abs (/ (* percentage (- red-1   (nth 0 new))) 100)))
	 (green-2 (abs (/ (* percentage (- green-1 (nth 1 new))) 100)))
	 (blue-2  (abs (/ (* percentage (- blue-1  (nth 2 new))) 100))))
    (format "#%04x%04x%04x"
	    (abs (- red-1 red-2))
	    (abs (- green-1 green-2))
	    (abs (- blue-1 blue-2)))))

(defun w3-percentage-from-date (date-1 date-2 length)
  "Return the percentage of LENGTH that has elapsed between DATE-1 and DATE-2
DATE-1 and DATE-2 are lists as returned by `current-time'
LENGTH is in days"
  (let ((secsbetween (+ (lsh (abs (- (nth 0 date-1) (nth 0 date-2))) 16)
			(abs (- (nth 1 date-1) (nth 1 date-2)))))
	(lengthinsecs (* length 24 60 60)))
    (round (* (/ (float secsbetween) (max lengthinsecs 1)) 100))))

(defun w3-parse-dssl-lite (fname &optional string)
  (let ((dest-buf (current-buffer))
	(url-mime-accept-string
	 "Accept: application/stylesheet ; notation=dsssl-lite")
	(sheet nil))
    (save-excursion
      (set-buffer (get-buffer-create
		   (url-generate-new-buffer-name " *style*")))
      (erase-buffer)
      (if fname (url-insert-file-contents fname))
      (goto-char (point-max))
      (if string (insert string))
      (goto-char (point-min))
      (delete-matching-lines "^[ \t]*#") ; Nuke comments
      (delete-matching-lines "^[ \t\r]*$") ; Nuke blank lines
      (goto-char (point-min))
      (insert "(")
      (goto-char (point-max))
      (insert ")")
      (goto-char (point-min))
      (setq sheet (condition-case ()
		      (read (current-buffer))
		    (error nil)))
      ;; Now need to convert the DSSSL-lite flow objects
      ;; into our internal representation
      ;; WORK WORK WORK!
      )))

(if (not (fboundp 'string-to-number))
    (fset 'string-to-number 'string-to-int))

(defun w3-spatial-to-canonical (spec)
  "Convert SPEC (in inches, millimeters, points, or picas) into pixels"
  (let ((num nil)
	(type nil)
	(dim1 (+ 25 (/ (float 4) 10)))
	(dim2 (float 72))
	(retval nil))
    (if (string-match "[^0-9.]+$" spec)
	(setq type (substring spec (match-beginning 0))
	      spec (substring spec 0 (match-beginning 0)))
      (setq type "px"
	    spec spec))
    (setq num (string-to-number spec))
    (cond
     ((url-member type '("pixel" "px" "pix"))
      (setq retval num
	    num nil))
     ((url-member type '("point" "pt"))
      (setq num num))
     ((url-member type '("inch" "in"))
      (setq num (/ num dim2)))
     ((string= type "mm")
      (setq num (* num (/ dim1 dim2))))
     ((string= type "cm")
      (setq num (* num (/ dim1 dim2))))
     )
    (if (not retval)
	(setq retval (* 10 num)))
    retval))      

(defun w3-parse-arena-style-sheet (fname &optional string)
  (let ((dest-buf (current-buffer))
	(url-mime-accept-string
	 (concat
	  "Accept: application/stylesheet ; notation=experimental\r\n"
	  "Accept: application/stylesheet ; notation=w3c-style"))
	(save-pos nil)
	(applies-to nil)		; List of tags to apply style to
	(attrs nil)			; List of name/value pairs
	(tag nil)
	(att nil)
	(val nil)
	(sheet nil))
    (save-excursion
      (set-buffer (get-buffer-create
		   (url-generate-new-buffer-name " *style*")))
      (erase-buffer)
      (if fname (url-insert-file-contents fname))
      (goto-char (point-max))
      (if string (insert string))
      (goto-char (point-min))
      (delete-matching-lines "^[ \t]*#")   ; Nuke comments
      (delete-matching-lines "^[ \t\r]*$") ; Nuke blank lines
      (w3-replace-regexp "^[ \t\r]+" "")   ; Nuke whitespace at beg. of line
      (w3-replace-regexp "[ \t\r]+$" "")   ; Nuke whitespace at end of line
      (w3-replace-regexp "![ \t]*\\([^ \t\r\n]+\\).*" "; priority=\"\\1\"")
      (goto-char (point-min))
      (while (not (eobp))
	(beginning-of-line)
	(setq save-pos (point))
	(skip-chars-forward "^:")
	(downcase-region save-pos (point))
	;; Could use read(), but it would slurp in the ':' as well
	(setq applies-to (url-split (buffer-substring save-pos (point))
				    "[ \t\r\n,&]"))
	(skip-chars-forward " \t:")
	(setq save-pos (point))
	(end-of-line)
	(skip-chars-backward "\r")
	(setq attrs (mm-parse-args save-pos (point) t))
	(skip-chars-forward "\r\n")
	(while applies-to
	  (setq tag (intern (downcase (car (car applies-to))))
		applies-to (cdr applies-to))
	  (let ((loop attrs))
	    (while loop
	      (setq att (car (car loop))
		    val (cdr (car loop))
		    loop (cdr loop))
	      (cond
	       ((string= "align" att)
		(setq val (intern val)))
	       ((or (string= "indent" att)
		    (string-match "^margin" att))
		(setq val (string-to-int val)))
	       (t nil))
	      (let* ((node-1 (assoc tag sheet))
		     (node-2 (and node-1 (assoc att node-1)))
		     (node-3 (assoc (symbol-name tag) sheet))
		     (node-4 (and node-3 (assoc att node-3))))
		(cond
		 ((not node-3)
		  (setq sheet (cons (cons (symbol-name tag)
					  (list (cons att val))) sheet)))
		 ((not node-4)
		  (setcdr node-3 (cons (cons att val) (cdr node-3))))
		 (t
		  (setcdr node-4 val)))
		(cond
		 ((not node-1)
		  (setq sheet (cons (cons tag (list (cons att val))) sheet)))
		 ((not node-2)
		  (setcdr node-1 (cons (cons att val) (cdr node-1))))
		 (t
		  (setcdr node-2 val))))))))
      (set-buffer-modified-p nil)
      (kill-buffer (current-buffer)))
    sheet))

(if (and (not (fboundp 'find-face))
	 (fboundp 'face-list))
    (defun find-face (face)
      (car-safe (memq face (face-list)))))

(defun w3-create-x-font (family style size em)
  (format
   "-*-%s-%s-%s-*-*-*-%s-%s-*-*-*-iso8859-1"
   family
   (if (string-match "bold" style) "bold" "medium")
   (if (string-match "italic" style) "i" "r")
   (if (eq em 'pixels) size "*")
   (if (eq em 'points) size "*")))

(defun w3-generate-stylesheet-faces (sheet)
  (let ((todo (delq nil (mapcar
			 (function (lambda (x) (if (symbolp (car x)) x)))
			 sheet)))
	(cur nil)
	(node nil)
	(fore nil)
	(back nil)
	(font nil)
	(family nil)
	(scale nil)
	(var nil)
	(face-name nil))
    (while todo
      (setq cur (car todo)
	    todo (cdr todo)
	    var (cdr-safe (assoc (car cur) w3-all-faces))
	    node cur)
      (if node
	  (progn
	    (setq fore (downcase (or (cdr-safe (assoc "color.text" node))
				     "none"))
		  back (downcase (or (cdr-safe (assoc "color.background" node))
				     "none"))
		  scale (cdr-safe (assoc "font.size" node))
		  scale (cond
			 ((null scale) "none")
			 ((listp scale) (condition-case ()
					    (int-to-string
					     (eval (read
						    (format "(%c 3 %s)"
							    (car scale)
							    (cdr scale)))))
					  (error 3)))
			 ((stringp scale) (downcase scale)))
		  family (downcase (or (cdr-safe (assoc "font.family" node))
				       "none"))
		  font (downcase (or (cdr-safe (assoc "font.style" node))
				     "none"))
		  font (mapconcat (function (lambda (x)
					      (cond
					       ((= x ? ) "")
					       ((= x ?,) "-")
					       ((= x ?&) "-")
					       (t (char-to-string x)))))
				  font "")
		  font (mapconcat 'identity
				  (sort (mapcar 'car (url-split font "-"))
					'string-lessp)
				  "-")
		  face-name (intern (if (fboundp 'make-face-larger)
					(concat fore "/" back "/" font
						"/" scale)
				      (concat fore "/" back "/" font))))
	    (cond
	     ((and (string= fore "none")
		   (string= back "none")
		   (string= scale "none")
		   (string= font "none"))
	      nil)			; Do nothing - no style directives
	     ((find-face face-name)
	      (setcdr node (cons (cons "face" face-name) (cdr node)))
	      (let ((x (assoc (symbol-name (car node)) w3-current-stylesheet)))
		(if x
		    (setcdr x (cons (cons "face" face-name) (cdr x)))))
	      (and var (set var face-name))) ; face already created
	     (t
	      (setcdr node (cons (cons "face" face-name) (cdr node)))
	      (let ((x (assoc (symbol-name (car node)) w3-current-stylesheet)))
		(if x
		    (setcdr x (cons (cons "face" face-name) (cdr x)))))
	      (make-face face-name)
	      (and var (set var face-name))
	      (if (not (string= fore "none"))
		  (w3-munge-color-fore face-name fore))
	      (if (not (string= back "none"))
		  (w3-munge-color-back face-name back))
	      (if (and (not (string= scale "none"))
		       (fboundp 'make-face-larger))
		  (let ((size (1- (string-to-int scale))))
		    (mapcar (cond
			     ((= size 0) 'identity)
			     ((< size 0) 'make-face-smaller)
			     ((> size 0) 'make-face-larger))
			    (make-list (abs size) face-name))))
	      (if (string= font "none")
		  nil
		(progn
		  (if (string-match "bold" font)
		      (condition-case ()
			  (make-face-bold face-name)
			(error nil)))
		  (if (string-match "italic" font)
		      (condition-case ()
			  (make-face-italic face-name)
			(error nil)))
		  (if (string-match "underline" font)
		      (apply 'set-face-underline-p face-name t)))))))))))

(defun w3-handle-style (&optional args)
  (let ((fname (or (cdr-safe (assoc "href" args))
		   (cdr-safe (assoc "src" args))
		   (cdr-safe (assoc "uri" args))))
	(type (downcase (or (cdr-safe (assoc "notation" args))
			    "experimental")))
	(url-working-buffer " *style*")
	(base (cdr-safe (assoc "base" args)))
	(stylesheet nil)
	(string (cdr-safe (assoc "data" args))))
    (if fname (setq fname (url-expand-file-name fname
						(cdr-safe
						 (assoc base w3-base-alist)))))
    (save-excursion
      (set-buffer (get-buffer-create url-working-buffer))
      (erase-buffer)
      (setq url-be-asynchronous nil)
      (cond
       ((url-member type '("experimental" "arena" "w3c-style"))
	(setq stylesheet (w3-parse-arena-style-sheet fname string)))
       ((string= type "dsssl-lite")
	(setq stylesheet (w3-parse-dsssl-lite fname string)))
       (t
	(message "Unknown stylesheet notation: %s" type))))
    (setq w3-current-stylesheet stylesheet)
    (if (and w3-current-stylesheet (fboundp 'make-face))
	(w3-generate-stylesheet-faces w3-current-stylesheet))))

(provide 'w3-style)
